/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Stefan Eckert <stefan.eckert@open-xchange.com>
 */

define('io.ox/office/text/androidselection',
    ['io.ox/office/tk/utils',
     'io.ox/office/tk/keycodes',
     'io.ox/office/tk/forms',
     'io.ox/office/text/utils/operations',
     'io.ox/office/text/position',
     'io.ox/office/text/dom'
    ], function (Utils, KeyCodes, Forms, Operations, Position, DOM) {

    'use strict';

    // mix-in class AndroidSelection =============================================

    var DUMMYLEN = 20;
    var CURSOR = 'androidcursor';
    var IMEMODE = 'androidime';
    var BLINK = 'androidblink';
    var SELECTOR = '.' + CURSOR + ', .' + IMEMODE;
    var CLASSES = CURSOR + ' ' + IMEMODE;

    var INITTEXT = Utils.repeatString(' ', DUMMYLEN);

    var ENDCHARS = Position.getWordBoundEndChars();

    var FONTATTRS = {
        fontName: 'monospace',
        fontSize: 20//points!
    };
    /**
     *
     * @constructor
     *
     */
    function AndroidSelection(app, rootNode) {

        var self = this;
        var blinkDeferred = null;
        var oldPara = null;
        var oldStart = null;
        var oldEnd = null;

        var wordFrom = null;
        var wordTo = null;
        var word = null;

        var div = $('<div>');
        var textarea = null;
        var singleArray = [0];

        var restoreFN = null;

        var letterWidth = 1;

        // private methods ----------------------------------------------------

        function splitTextSpan(para, pos) {
            singleArray[0] = _.last(pos);

            if (singleArray[0] === 0) {
                var firstSpan = para.children().first();
                if (!firstSpan.is('span') || !firstSpan[0].innerHTML.length) {
                    return firstSpan[0];
                } else {
                    var newSpan = firstSpan.clone(true);
                    fillTextInSpan(newSpan[0], '');
                    para.prepend(newSpan);
                    return newSpan[0];
                }
            }

            var domPos = Position.getDOMPosition(para[0], singleArray);
            if (!domPos) {
                Utils.error('no result', para.children().length, pos, para[0]);
                return null;
            }

            if (domPos.node.nodeValue.length === domPos.offset) {
                return domPos.node.parentElement;
            } else {
                return DOM.splitTextSpan(domPos.node.parentElement, domPos.offset);
            }
        }

        function fillTextInSpan(span, text) {
            span.innerHTML = '';
            span.appendChild(document.createTextNode(text));
        }

        function getWordBoundary(para, pos, beginning, dontAddSpace) {
            var len = pos.length;
            singleArray[0] = pos[len-1];

            var result = _.clone(pos);
            result[len-1] = Position.getWordBoundary(para[0], singleArray, beginning, dontAddSpace)[0];
            return result;
        }

        function changeSelection(event, selectionInfo) {
            var start = selectionInfo.start;
            var end = selectionInfo.end;
            //console.error('changeSelection', start, end);
            var spans = null;
            var node = null;

            if (oldPara) {
                oldPara.removeClass(BLINK);
                spans = oldPara.find(SELECTOR);
                spans.removeClass(CLASSES);
//                spans.each(function() {
//                    CharacterStyles.mergeSiblingTextSpans(this, true);
//                });
                oldPara = null;
            }

            var visibility = 'hidden';

            if (app.getModel().getEditMode()) {

                var selType = window.getSelection().type;
                if (!selType || selType === 'None') {
                    selType = Utils.isSoftKeyboardOpen();
                }

                if (_.isEqual(start, end) && selType) {

                    oldPara = $(Position.getDOMPosition(rootNode, _.initial(start)).node);
                    oldPara.addClass(BLINK);

                    wordFrom = getWordBoundary(oldPara, start, true);
                    wordTo = getWordBoundary(oldPara, start, false, true);
                    if (_.isEqual(wordFrom, wordTo) ||  _.last(wordFrom) >= _.last(wordTo)) {
                        start[start.length-1]--;
                        wordFrom = getWordBoundary(oldPara, start, true);
                        wordTo = getWordBoundary(oldPara, start, false, true);
                        start[start.length-1]++;
                    }

                    if (_.last(wordFrom) < _.last(wordTo)) {
                        splitTextSpan(oldPara, wordFrom);
                        splitTextSpan(oldPara, wordTo);

                        word = '';

                        singleArray[0] = _.last(wordFrom);
                        var spanFrom = Position.getDOMPosition(oldPara[0], singleArray, true).node;

                        var len = 0;
                        var maxLen = _.last(wordTo) - _.last(wordFrom);

                        while(len < maxLen && spanFrom) {
                            word += spanFrom.innerHTML;
                            len = word.length;
                            spanFrom.classList.add(IMEMODE);
                            spanFrom = spanFrom.nextSibling;
                        }
                    } else {
                        wordFrom = _.clone(start);
                        wordTo =  _.clone(start);
                        word = '';
                    }


                    //console.warn('domPos.offset', _.last(start));


                    node = splitTextSpan(oldPara, start);
                    node = $(node);
                    node.addClass(CURSOR);
                    oldStart = start;
                    oldEnd = end;

                    selectTextArea();
                    handleTextAreaPos();
                    visibility = 'visible';
                }
            }
            div.css('visibility', visibility);
        }

        function handleTextAreaPos() {
            var parent = div.parent();
            if (parent && parent.length) {

                var zoom = app.getView().getZoomFactor();
                //console.warn('handleTextAreaPos', oldStart, oldPara.children().length, oldPara);
                var pos = Position.getPixelPositionFromDomPoint (rootNode, [_.last(oldStart)], zoom, oldPara);
                var multi = zoom/100;

//                console.warn('getPixelPositionFromDomPoint', pos, oldPara, _.last(oldStart), zoom);
                div.css({
                    left: (pos.left * multi - 5) + 'px',
                    top: (pos.top * multi + pos.height * multi - 30) + 'px'
                });
            }
        }

        function isPassThrough(event, press) {
            if (event.ctrlKey || event.altKey || event.metaKey) { return true; }
            if (press) {
                return event.keyCode === KeyCodes.ENTER || event.keyCode === KeyCodes.SPACE;
            } else if(event.keyCode === KeyCodes.TAB){
                event.preventDefault();
                return true;
            } else if(event.shiftKey && DOM.isCursorKey(event.keyCode)) {
                restoreFN();
                event.preventDefault();
                return true;
            } else if(event.keyCode === KeyCodes.BACKSPACE) {
                event.preventDefault();
                return true;
            } else {
                return (DOM.isCursorKey(event.keyCode) || event.keyCode === KeyCodes.ENTER || event.keyCode === KeyCodes.DELETE || event.keyCode === KeyCodes.SPACE);
            }
        }

        function passThrough(event) {
            if (isPassThrough(event, true)) {
                handlePassThrough(event);
            }
        }

        function handlePassThrough(event) {
            var docModel = app.getModel();

            if (!event.shiftKey && (event.keyCode === KeyCodes.UP_ARROW || event.keyCode === KeyCodes.DOWN_ARROW)) {
                event.preventDefault();
                if (event.type === 'keydown' && oldPara) {
                    var offset = event.keyCode === KeyCodes.UP_ARROW ? -1 : 1;

                    var cursorSpan = oldPara.find('.' + CURSOR);
                    var w = $(window);

                    var height = cursorSpan.height();
                    offset *= height * 1.5;

                    var pos = cursorSpan.offset();

                    var top = pos.top - w.scrollTop();
                    var left = pos.left + cursorSpan.width() - w.scrollLeft();

                    var newOxo = Position.getOxoPositionFromPixelPosition(rootNode, left, top + height/2 + offset);

                    if (!newOxo) {
                        if (oldStart.length > 2) {
                            return;
                        }
                        newOxo = {
                                start: []
                        };
                        if (event.keyCode === KeyCodes.UP_ARROW) {
                            newOxo.start[0] = Math.max(0, oldStart[0] - 1);
                        } else {
                            newOxo.start[0] = oldStart[0] + 1;
                        }
                        newOxo.start.push(0);
                    }

                    self.setTextSelection(newOxo.start);
                }
            } else if (!event.shiftKey && (event.keyCode === KeyCodes.LEFT_ARROW || event.keyCode === KeyCodes.RIGHT_ARROW)) {
                docModel.handleAndroidTyping();
                event.preventDefault();
                event.preventDefault = $.noop;
                rootNode.trigger(event);
            } else {
                docModel.handleAndroidTyping();
                event.preventDefault = $.noop;
                rootNode.trigger(event);
            }
            return true;
        }

        function changedText(event) {
            var docModel = app.getModel();

            if (isPassThrough(event)) {
                handlePassThrough(event);
            } else {
                docModel.handleAndroidTyping();
                var text = getText();

                if (text.length > DUMMYLEN) {



                    text = text.substring(DUMMYLEN, text.length);

                    if (text === word) {
                        return;
                    }

                    var preAttrs = docModel.getPreselectedAttributes() || undefined;
                    var attrs = preAttrs;
                    var changeTrack = docModel.getChangeTrack();
                    if (changeTrack.isActiveChangeTracking()) {
                        if (!attrs) {
                            attrs = {};
                        } else {
                            attrs = _.copy(attrs, true);
                        }
                        attrs.changes = { 'inserted': changeTrack.getChangeTrackInfo() };
                    }

                    var selection = getCursorIndex() - DUMMYLEN;

                    var undoOperations = [];
                    var redoOperations = [];

                    if (DOM.isImplicitParagraphNode(oldPara[0])) {
                        docModel.sendOperations({ name: Operations.PARA_INSERT, start: _.initial(wordFrom) });
                        oldPara.data('implicit', false);
                    }

                    if (text.length - 1 === word.length && text.substring(0, text.length - 1) === word && !preAttrs) {
                        //optimized for forward typing
                        redoOperations.push({ name: Operations.TEXT_INSERT, start: _.clone(wordTo), text: text.substring(text.length - 1, text.length), attrs: attrs });
                        undoOperations.push({ name: Operations.DELETE, start: _.clone(wordTo), end: Position.increaseLastIndex(wordTo, -1) });

                    } else {

                        var insertOp = { name: Operations.TEXT_INSERT, start: _.clone(wordTo), text: text, attrs: attrs };
                        redoOperations.push(insertOp);


                        var deleteOp = null;

                        if (_.last(wordFrom) !== _.last(wordTo)) {
                            deleteOp = { name: Operations.DELETE, start: _.clone(wordFrom), end: Position.increaseLastIndex(wordTo, -1) };
                            redoOperations.push(deleteOp);
                        }

                        if (deleteOp) {
                            var insStart = Position.increaseLastIndex(wordFrom, text.length);
                            var undoDeleteOp = { name: Operations.TEXT_INSERT, start: insStart, text: word, attrs: preAttrs };
                            undoOperations.push(undoDeleteOp);
                        }

                        var undoInsertOp = { name: Operations.DELETE, start: _.clone(wordFrom), end: Position.increaseLastIndex(wordTo, -1) };
                        undoOperations.push(undoInsertOp);

                    }

                    sendOperations(docModel, changeTrack, undoOperations, redoOperations);

                    var spans;
                    if (_.last(wordFrom) === _.last(oldStart)) {
                        //cursor before beginning of the word
                        spans = oldPara.children('span.' + IMEMODE);
                    } else {
                        spans = oldPara.children('span.' + IMEMODE + ', span.' + CURSOR);
                    }


                    //console.warn('spancount befor', oldPara.children('span.' + IMEMODE).length, oldPara.children('span.' + CURSOR).length);

                    var cursorSpan = oldPara.find('.' + CURSOR);
                    cursorSpan.removeClass(CURSOR);

                    var leftNoIME = _.contains(ENDCHARS, text[selection - 1]);

                    var spanCount = spans.length;

                    if (selection === 1) {
                        //cursor was on the beginning

                        if (!leftNoIME && word.length > text.length) {
                            //backspaced
                            cursorSpan.text(text);
                            cursorSpan.addClass(CURSOR);
                            handlePreselectedAttrs(attrs, cursorSpan);
                        } else if (!cursorSpan.is('span')) {
                            var newSpan = $(spans[0]).clone(true);
                            newSpan.insertAfter(cursorSpan);
                            newSpan.addClass(IMEMODE);
                            newSpan.addClass(CURSOR);
                            fillTextInSpan(newSpan[0], text.substring(0, 1));
                            handlePreselectedAttrs(attrs, newSpan);
                        } else if (leftNoIME) {
                            fillTextInSpan(cursorSpan[0], cursorSpan.text() + text.substring(0, 1));
                            cursorSpan.addClass(CURSOR);
                            handlePreselectedAttrs(attrs, cursorSpan);
                        } else {
                            var newSpan = cursorSpan.clone(true);
                            newSpan.insertAfter(cursorSpan);
                            newSpan.addClass(IMEMODE);
                            newSpan.addClass(CURSOR);
                            fillTextInSpan(newSpan[0], text.substring(0, 1));
                            handlePreselectedAttrs(attrs, newSpan);
                        }

                    } else if (selection === text.length) {
                        //cursor on end
                        _.times(spanCount - 1, function(i) {
                            var el = spans[i];
                            el.parentNode.removeChild(el);
                        });
                        var span = spans[spanCount-1];
                        span.classList.add(CURSOR);
                        fillTextInSpan(span, text);

                        if (leftNoIME) {
                            span.classList.remove(IMEMODE);
                        }
                        handlePreselectedAttrs(attrs, span);
                    } else {
                        _.times(spanCount - 2, function(i) {
                            var el = spans[i];
                            el.parentNode.removeChild(el);
                        });
                        var right = spans[spanCount-1];
                        right.innerText = text.substring(selection, text.length);

                        var left = spans[spanCount-2];
                        left.classList.add(CURSOR);
                        left.innerText = text.substring(0, selection);

                        if (leftNoIME) {
                            left.classList.remove(IMEMODE);
                        }
                        handlePreselectedAttrs(attrs, left, right);
                    }
                    //console.warn('before', leftNoIME, _.last(wordFrom), _.last(wordTo));
                    var pos = _.last(wordFrom) + selection;
                    oldStart[oldStart.length - 1] = pos;
                    oldEnd[oldEnd.length - 1] = pos;

                    if (leftNoIME) {
                        word = text.substring(selection, text.length);
                        wordFrom[wordFrom.length - 1] = pos;
                        wordTo[wordTo.length - 1] = _.last(wordFrom) + word.length;
                        selectTextArea();
                    } else {
                        word = text;
                        wordTo[wordTo.length - 1] = _.last(wordFrom) + text.length;
                    }

                    //console.warn('spancount after', oldPara.children('span.' + IMEMODE).length, oldPara.children('span.' + CURSOR).length);

                    handleTextAreaPos();

                    //console.warn('after ', leftNoIME, _.last(wordFrom), _.last(wordTo));
                    //console.warn('');

                    if (attrs) {
                        docModel.getStyleCollection('paragraph').updateElementFormatting(oldPara);
                    }

                }
            }
        }

        function sendOperations(docModel, changeTrack, undoOperations, redoOperations) {
            if (changeTrack.isActiveChangeTracking()) {
                var info = changeTrack.getChangeTrackInfo();
                _.each(undoOperations, function(op) {
                    addChangeTrackInfo(op, info);
                });
                _.each(redoOperations, function(op) {
                    addChangeTrackInfo(op, info);
                });
            }
            docModel.sendOperations(redoOperations);
            docModel.getUndoManager().addUndo(undoOperations, redoOperations);
        }

        function addChangeTrackInfo(op, info) {
            var attrs = op.attrs;
            if (attrs) {
                if (attrs.changes) {
                    return;
                }
                attrs = _.copy(attrs, true);
            } else {
                attrs = {};
            }


            var name;
            if (op.name === Operations.TEXT_INSERT) {
                name = 'inserted';
            } else {
                name = 'deleted';
            }
            attrs.changes = {};
            attrs.changes[name] = info;

        }

        function handlePreselectedAttrs(attrs) {
            if (attrs) {
                attrs = {
                        attributes: attrs
                };
                for (var i = 1; i < arguments.length; i++) {
                    var span = arguments[i];
                    if (!(span instanceof $)) {
                        span = $(span);
                    }
                    span.data(attrs);
                }
            }
        }

        textarea = $('<textarea>');

        function selectTextArea() {
            textarea.focus();
            textarea.val(INITTEXT + word);
            Forms.setInputSelection(textarea, DUMMYLEN + (_.last(oldStart) - _.last(wordFrom)));

            //workaround for broken initial focus
            if (document.activeElement !== textarea[0]) {
                _.defer(function() {
                    textarea.focus();
                    if (document.activeElement !== textarea[0]) {
                        restoreFN();
                    }
                });
            }
        }

        function getCursorIndex() {
            return textarea[0].selectionStart;
        }

        function getText() {
            return textarea.val();
        }

        function resetTextarea() {
            div.css('visibility', 'hidden');
        }

        function setReadOnly() {
            rootNode.css('-webkit-user-modify', 'read-only');
        }

        function setReadWrite() {
            rootNode.css('-webkit-user-modify', 'read-write');
        }

        var processSelectUpdate = (function () {
            var always = function () {
                var selecttype = window.getSelection().type;
                if (document.activeElement === rootNode[0]) {
                    //listener hangs directly on document, but we only want the events of editdiv
                    if (selecttype !== 'None') {
//                        console.warn('selecttype', selecttype);
                        self.updateSelectionAfterBrowserSelection({ preserveFocus: false, forceTrigger: false });
//                        console.warn('all', self.getStartPosition(), self.getEndPosition());
                        if (selecttype === 'Caret') {
                            setReadOnly();
                        }
                    }
                } else if (document.activeElement === textarea[0]) {
                    if (window.getSelection().anchorNode === div[0]) {
                        //focus really in textarea
                        if (selecttype === 'Range') {
                            if (_.last(wordFrom) < _.last(wordTo)) {
                                restoreFN();
                                self.setTextSelection(wordFrom, wordTo);
                            }
                        }
                        setReadOnly();
                    } else {
                        //focus in document
                        if (selecttype !== 'Range') {
                            setReadOnly();
                        }
                    }
                }
            };
            var notAlways =  _.throttle(always, 250);

            return function(evt) {

                var selecttype = window.getSelection().type;
                if (evt.type === 'selectstart') {
                    if (selecttype === 'Range') {
                        ///User seems to call "select all", but android chrome does not really support that featurem when contenteditable=true and begins with scrolling :(
                        evt.preventDefault();
                        return;
                    }
                    always();
                } else {
                    notAlways();
                }

                if (selecttype === 'Range') {
                    if (Utils.isSoftKeyboardOpen()) {
                        setReadOnly();
                        _.defer(setReadWrite);
                    } else if(rootNode.css('-webkit-user-modify') === 'read-only') {
                        setReadWrite();
                    }
                }
            };
        }());

        //---------init---------

        textarea.on('keyup keydown change', changedText);
        textarea.on('keypress', passThrough);

        self.on('changecursorinfo', changeSelection);


        blinkDeferred = app.repeatDelayed(function () {
            if (oldPara) {
                if (document.activeElement && document.activeElement.tagName === 'INPUT') {
                    oldPara.removeClass(BLINK);
                } else {
                    oldPara.toggleClass(BLINK);
                }
            }
        }, { delay: 500 });

        app.onImportSuccess(function () {
            var view = app.getView();

            restoreFN = _.bind(self.restoreBrowserSelection, self);

            self.restoreBrowserSelection = function() {
                if ($(document.activeElement).is(Forms.BUTTON_SELECTOR)) {
                    //emulate default behavior because menus are not closed
                    Utils.trigger('change:focus', rootNode, document.activeElement, null);
                }
                if (!_.isEqual(self.getStartPosition(), self.getEndPosition())) {
                    //console.warn('restoreFN', self.getStartPosition(), self.getEndPosition());
                    restoreFN();
                }
            };

            var docModel = app.getModel();
            textarea.on('cut copy paste', function(evt) {
                //console.warn('evt', evt.type);
                docModel[evt.type](evt);
            });

            div.css({
                position: 'absolute',
                zIndex: 10,
                top: '30px',
                width: '10px',
                height: '40px',
                overflow: 'hidden'
            });

            letterWidth = docModel.getFontCollection().getTextWidth('A', FONTATTRS);

            textarea.css({
                position: 'absolute',
                width: '10px',
                letterSpacing: -letterWidth + 'px',
                fontFamily: 'monospace',
                fontSize: FONTATTRS.fontSize + 'pt',
                lineHeight: FONTATTRS.fontSize + 'pt',
                height: '30px',
                color: 'transparent',
                overflow: 'hidden',
                background: 'transparent',
                borderColor: 'transparent',
                outline: 'none',
                opacitiy: 0,
                margin: 0,
                padding: 0,
                textIndent: '3px',
                left: 0,
                top: 0
            });

            div.append(textarea);
            rootNode.parent().append(div);

            docModel.listenTo($(document), 'selectstart selectionchange select', processSelectUpdate);

            app.registerWindowResizeHandler(function () {
                if (Utils.isSoftKeyboardOpen() && window.getSelection().type !== 'Range') {
                    var boundRect = docModel.getSelection().getFocusPositionBoundRect();
                    if (boundRect) {
                        app.getView().scrollToPageRectangle(boundRect);
                    }
                }
            } );


            app.registerGlobalEventHandler(window, 'orientationchange', resetTextarea);
            docModel.listenTo(view, 'change:zoom', resetTextarea);



            var grabFocus = view.grabFocus;
            view.grabFocus = function() {
                if (view.isVisible()) {
                    if (Utils.isSoftKeyboardOpen()) {
                        var windowNode = app.getWindow().nodes.outer;
                        var btn = $(windowNode).find('a.button[aria-expanded=true]');
                        if (!btn.length) {
                            //fallback for combo-text-fields
                            btn = $(windowNode).find('.group.dropdown-open a').first();
                        }
                        btn[Forms.DEFAULT_CLICK_TYPE]();

                        if(document.activeElement !== textarea[0]){
                            $(document.activeElement).blur();
                        }
                        selectTextArea();


                    } else {
                        grabFocus.apply(view);
                    }
                }
            };

            textarea.blur(function() {
                if (Utils.isSoftKeyboardOpen() && (!document.activeElement || document.activeElement.tagName !== 'INPUT')) {
                    _.defer(function() {
                        if (document.activeElement.classList.contains('popup-content') || document.activeElement.classList.contains('button')) {
                            textarea.addClass('focus-helper');
                            selectTextArea();
                            _.defer(function() {
                                textarea.removeClass('focus-helper');
                            });
                        }
                    });
                }
            });


            /*
            var lastTouch = null;
            var lastTouchCounter = 0;


            //double tab
            rootNode.on('click', function(evt) {
                if (!lastTouch) {
                    lastTouch = evt.timeStamp;
                    lastTouchCounter = 0;
                }

                var relTime = evt.timeStamp - lastTouch;
                if (relTime < 1500) {
                    lastTouchCounter++;
                    if (lastTouchCounter === 2) {

                        if (_.last(wordFrom) < _.last(wordTo)) {
                            restoreFN();
                            self.setTextSelection(wordFrom, wordTo);

                        }

                        lastTouch = null;
                    }
                } else if (relTime > 3000) {
                    lastTouch = evt.timeStamp;
                    lastTouchCounter = 1;
                } else {
                    lastTouch = null;
                }
            });
            */

        });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            blinkDeferred.abort();
            div.remove();
        });

    } // class AndroidSelection



    // exports ================================================================

    return AndroidSelection;

});
