/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 * @author Edy Haryono <edy.haryono@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/mixin/viewfuncmixin', [
    'io.ox/office/tk/utils',
    'io.ox/office/editframework/utils/color',
    'io.ox/office/editframework/utils/border',
    'io.ox/office/editframework/utils/mixedborder',
    'io.ox/office/drawinglayer/view/imageutil',
    'io.ox/office/spreadsheet/utils/config',
    'io.ox/office/spreadsheet/utils/operations',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/utils/paneutils',
    'io.ox/office/spreadsheet/model/cellcollection',
    'io.ox/office/spreadsheet/view/labels',
    'gettext!io.ox/office/spreadsheet'
], function (Utils, Color, Border, MixedBorder, ImageUtils, Config, Operations, SheetUtils, PaneUtils, CellCollection, Labels, gt) {

    'use strict';

    var // tolerance for optimal column width, in 1/100 mm
        SET_COLUMN_WIDTH_TOLERANCE = 50,

        // tolerance for optimal row height (set explicitly), in 1/100 mm
        SET_ROW_HEIGHT_TOLERANCE = 15,

        // tolerance for optimal row height (updated implicitly), in 1/100 mm
        UPDATE_ROW_HEIGHT_TOLERANCE = 50,

        // all available zoom factors
        ZOOM_FACTORS = [0.5, 0.75, 1, 1.25, 1.5, 1.75, 2, 2.5, 3, 4, 5, 6, 8],

        // default value for a visible border
        DEFAULT_SINGLE_BORDER = { style: 'single', width: Utils.convertLengthToHmm(1, 'px'), color: Color.AUTO },

        // default settings for mixed border attributes
        DEFAULT_BORDER_DATA = {
            borderTop: {},
            borderBottom: {},
            borderLeft: {},
            borderRight: {}
        },

        // default settings for subtotal results
        DEFAULT_SUBTOTAL_DATA = {
            cells: 0,
            numbers: 0,
            sum: 0,
            min: Number.POSITIVE_INFINITY,
            max: Number.NEGATIVE_INFINITY,
            average: Number.NaN

        },

        // default settings for search/replace
        DEFAULT_SEARCH_SETTINGS = {
            query: '', // active search query
            sheets: null, // search data received from server, mapped by sheet index
            index: -1, // current index into the search results
            replace: false // whether search results are replaced currently
        },

        // maps border names to the names of the opposite borders and relative cell position
        OPPOSITE_BORDERS = {
            borderTop:    { borderName: 'borderBottom', cols:  0, rows: -1 },
            borderBottom: { borderName: 'borderTop',    cols:  0, rows:  1 },
            borderLeft:   { borderName: 'borderRight',  cols: -1, rows:  0 },
            borderRight:  { borderName: 'borderLeft',   cols:  1, rows:  0 }
        },

        // maximum size of an image in any direction, in 1/100 mm
        MAX_IMAGE_SIZE = 21000;

    // mix-in class ViewFuncMixin =============================================

    /**
     * Mix-in class for the class SpreadsheetView that provides implementations
     * of complex document functionality and status requests, called from the
     * application controller.
     *
     * @constructor
     *
     * @param {SpreadsheetApplication} app
     *  The spreadsheet application instance.
     */
    function ViewFuncMixin(app) {

        var // self reference (spreadsheet view instance)
            self = this,

            // the spreadsheet model, and other model objects
            docModel = null,
            undoManager = null,
            documentStyles = null,
            fontCollection = null,
            styleCollection = null,
            numberFormatter = null,

            // the active sheet model and index
            activeSheetModel = null,
            activeSheet = -1,

            // collections of the active sheet
            colCollection = null,
            rowCollection = null,
            mergeCollection = null,
            cellCollection = null,
            tableCollection = null,
            drawingCollection = null,

            // the contents and formatting of the active cell, and the selected ranges
            activeCellSettings = null,
            selectionSettings = null,

            // settings for search/replace
            searchSettings = _.copy(DEFAULT_SEARCH_SETTINGS, true);

        // private methods ----------------------------------------------------

        /**
         * Imports the passed settings for the borders of the selection (in the
         * property 'borders'), and the subtotal results (in the property
         * 'subtotals').
         *
         * @param {Object} settings
         *  Settings for the selected ranges, as received from the respective
         *  server request.
         */
        function importSelectionSettings(settings) {

            // clear the old selection settings
            selectionSettings = {};

            // import mixed border attributes
            selectionSettings.borders = _.copy(DEFAULT_BORDER_DATA, true);
            if (_.isObject(settings.borders)) {
                _.extend(selectionSettings.borders, settings.borders);
                // delete inner borders for specific selections
                if (!self.hasMultipleColumnsSelected()) { delete selectionSettings.borders.borderInsideVert; }
                if (!self.hasMultipleRowsSelected()) { delete selectionSettings.borders.borderInsideHor; }
            }

            // import subtotal results
            selectionSettings.subtotals = _.copy(DEFAULT_SUBTOTAL_DATA, true);
            if (_.isObject(settings.subtotals)) {
                _.extend(selectionSettings.subtotals, settings.subtotals);
                selectionSettings.subtotals.average = selectionSettings.subtotals.sum / selectionSettings.subtotals.numbers;
            }

            // import selection lock data
            selectionSettings.locked = settings.locked === true;
        }

        /**
         * Collects several update requests, and sends a single deferred server
         * request.
         *
         * @param {Object} requestProps
         *  The properties to be inserted into the current request data object.
         */
        var requestViewUpdate = (function () {

            var // the data object collecting all grid pane requests
                requestData = null,
                // the last running request
                runningDef = null,
                // the start time of the initial request (without other requests already running)
                startTime = 0,
                // counter for running requests (next free index)
                nextRequestIndex = 0,
                // minimum request index valid for evaluation (ignore requests after switching to another sheet)
                requiredRequestIndex = 0;

            // direct callback: register a new sheet rectangle to be updated in a grid pane
            function registerUpdateRequest(requestProps) {

                // do not request anything in error mode
                if (app.isInQuit() || (app.getState() === 'error')) { return; }

                // initialize request data, if missing or for another sheet
                if (!requestData || (requestData.sheet !== activeSheet)) {
                    requestData = { sheet: activeSheet, locale: Config.LOCALE };
                }

                // extend the array of cell ranges
                if (_.isObject(requestProps.cells)) {
                    requestProps.cells = _.getArray(requestProps.cells);
                    if (_.isArray(requestData.cells)) {
                        requestProps.cells = requestData.cells.concat(requestProps.cells);
                    }
                }

                // insert the passed request data
                _.extend(requestData, requestProps);
            }

            // deferred callback: send single server request and notify all waiting grid panes
            function executeUpdateRequest() {

                var // local reference to the current request dats
                    localRequestData = requestData,
                    // current request index
                    localRequestIndex = nextRequestIndex,
                    // start time of this request
                    localStartTime = _.now(),
                    // the new server request
                    def = null;

                if (!localRequestData || app.isInQuit() || (app.getState() === 'error')) { return; }
                requestData = null;

                // take start time, if no other request is running already
                if (!runningDef) {
                    startTime = _.now();
                }

                // restrict requested ranges to visible area of all current grid panes
                if (_.isArray(localRequestData.cells)) {
                    Utils.error('ViewFuncMixin.executeUpdateRequest(): cell data requested via view update');
                }

                // immediately delete the request data (new updates may be requested in the meantime)
                localRequestIndex = nextRequestIndex;
                nextRequestIndex += 1;
                Utils.info('ViewFuncMixin.executeUpdateRequest(): sending update request #' + (localRequestIndex + 1) + ':', localRequestData);

                // wait for pending actions (bug 30044, must wait for the
                // 'insertSheet' operation before requesting data for the sheet)
                def = runningDef = app.saveChanges().then(function () {
                    return app.sendActionRequest('updateview', localRequestData);
                });

                def.fail(function () {
                    Utils.error('ViewFuncMixin.executeUpdateRequest(): update request #' + (localRequestIndex + 1) + ' failed');
                });

                def.done(function (layoutData) {

                    Utils.info('ViewFuncMixin.executeUpdateRequest(): update response #' + (localRequestIndex + 1) + ' received (' + (_.now() - localStartTime) + 'ms):', layoutData);
                    // check validity of response object
                    if (!_.isObject(layoutData) || !_.isObject(layoutData.sheet)) {
                        Utils.error('ViewFuncMixin.executeUpdateRequest(): missing required layout data');
                        return;
                    }

                    // ignore this response if the active sheet has changed in the meantime
                    if (localRequestIndex < requiredRequestIndex) { return; }

                    // if this update request is triggered by the search function
                    if (_.isObject(localRequestData.search) && !_.isObject(layoutData.found)) {
                        self.yell({
                            type: 'info',
                            message: /*#. Search&Replace: used when the searched text has not been found */ gt('There are no cells matching "%1$s".', localRequestData.search.searchText)
                        });
                    }

                    // store search results
                    if (_.isObject(layoutData.found)) {
                        searchSettings.sheets = _.copy(Utils.getObjectOption(layoutData.found, 'sheets', null), true);
                        // jump to first search result only if this is a new search
                        if (!searchSettings.replace) {
                            self.searchNext('next', searchSettings.query);
                        }
                    }

                    // copy layout data for the current selection
                    if (_.isObject(layoutData.selection)) {
                        importSelectionSettings(layoutData.selection);
                    }

                    // Check if this is the last request. Do not notify listeners about response
                    // data of previous requests, but collect the response data and all change
                    // flags of multiple requests, and notify listeners after last response.
                    if (def === runningDef) {
                        runningDef = null;
                        self.trigger('change:layoutdata');
                    } else if (_.now() - startTime >= 3000) {
                        startTime = _.now();
                        self.trigger('change:layoutdata');
                    }
                });
            }

            // delete the current cached request data when the active sheet changes
            self.on('before:activesheet', function () {
                requestData = null;
                requiredRequestIndex = nextRequestIndex;
            });

            // create and return the debounced method requestViewUpdate()
            return self.createDebouncedMethod(registerUpdateRequest, executeUpdateRequest, { delay: 100, maxDelay: 250 });
        }()); // ViewFuncMixin.requestViewUpdate()

        /**
         * Requests an update of the selection part of the view layout data.
         */
        function requestSelectionUpdate(selection) {
            requestViewUpdate({ selection: { ranges: selection.ranges, active: selection.activeCell } });
        }

        /**
         * Updates the cached data of the active cell (contents, formatting),
         * and the entire selection (border settings, subtotals). To improve
         * performance, updating settings of the entire selection (which is a
         * potentially expensive operation) will be done debounced.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.direct=false]
         *      If set to true, all selection settings will be updated
         *      immediately (without additional debounced update).
         */
        var updateSelectionSettings = (function () {

            var // whether selection update is pending (for option 'direct')
                pendingRangeUpdate = false;

            // update settings of the active cell synchronously for fast GUI feedback
            function updateActiveCell(options) {

                // receive settings of active cell from cell collection
                activeCellSettings = cellCollection.getCellEntry(self.getActiveCell());
                pendingRangeUpdate = true;

                // invoke deferred callback immediately if specified
                if (Utils.getBooleanOption(options, 'direct', false)) {
                    updateSelectedRanges();
                }
            }

            // update settings of the entire selection deferred (expensive cell iterations)
            function updateSelectedRanges() {

                var // the current cell selection
                    selection = self.getSelection(),
                    // all merged ranges covering the selection
                    mergedRanges = null,
                    // object to collect and sort all border types
                    allBorders = {},
                    // whether current selection contains a locked cell
                    selectionLocked = false;

                // selection settings may have been updated immediately (option 'direct')
                if (!pendingRangeUpdate) { return; }
                pendingRangeUpdate = false;

                // If any selected range is not completely covered by the cell collection, or if the
                // selection contains too many cells, request new data from server. Keep old selection
                // settings until that data has been received to prevent flickering icons in the side pane.
                if ((SheetUtils.getCellCountInRanges(selection.ranges) > SheetUtils.MAX_SELECTION_CELL_COUNT) || !cellCollection.containsRanges(selection.ranges)) {
                    selectionSettings.locked = true;
                    requestSelectionUpdate(selection);
                    return;
                }

                // initialize the selection settings object with default data
                importSelectionSettings({});

                // get all merged ranges covering the selection
                mergedRanges = mergeCollection.getMergedRanges(selection.ranges);

                // iterate all ranges
                cellCollection.iterateCellsInRanges(selection.ranges, function (cellData, originalRange) {

                    var // address of the current cell
                        address = cellData.address,
                        // clone the global OPPOSITE_BORDERS for local modification
                        oppositeBorders = _.copy(OPPOSITE_BORDERS, true),
                        // merged range covering the current cell
                        mergedRange = SheetUtils.findFirstRange(mergedRanges, address),
                        // additional columns/rows of a merged cell
                        additionalCols = 0,
                        additionalRows = 0;

                    // special handling for cells that are part of a merged range
                    if (mergedRange) {

                        // skip hidden cells inside merged range
                        if (!_.isEqual(mergedRange.start, address)) { return; }

                        // modify opposite borders for the reference cell of a merged range
                        oppositeBorders.borderRight.cols = SheetUtils.getColCount(mergedRange);
                        oppositeBorders.borderBottom.rows = SheetUtils.getRowCount(mergedRange);
                        additionalCols = oppositeBorders.borderRight.cols - 1;
                        additionalRows = oppositeBorders.borderBottom.rows - 1;
                    }

                    // process all outer borders of the cell
                    _.each(oppositeBorders, function (OPPOSITE_BORDER, borderName) {

                        var // cell data of the preceding/next cell (taking care of merged cells)
                            nextCellData = cellCollection.getCellEntry([address[0] + OPPOSITE_BORDER.cols, address[1] + OPPOSITE_BORDER.rows]),
                            // the border value of the current and adjacent cell
                            thisBorder = cellData.attributes.cell[borderName],
                            nextBorder = nextCellData.attributes.cell[OPPOSITE_BORDER.borderName],
                            // whether the borders are visible
                            thisBorderVisible = Border.isVisibleBorder(thisBorder),
                            nextBorderVisible = Border.isVisibleBorder(nextBorder),
                            // the target border name for the border collection (inner or outer borders)
                            targetBorderName = null,
                            targetBorders = null;

                        // get the target border name for the border collection
                        switch (borderName) {
                        case 'borderLeft':
                            targetBorderName = (address[0] === originalRange.start[0]) ? 'borderLeft' : null;
                            break;
                        case 'borderRight':
                            targetBorderName = (address[0] + additionalCols === originalRange.end[0]) ? 'borderRight' : 'borderInsideVert';
                            break;
                        case 'borderTop':
                            targetBorderName = (address[1] === originalRange.start[1]) ? 'borderTop' : null;
                            break;
                        case 'borderBottom':
                            targetBorderName = (address[1] + additionalRows === originalRange.end[1]) ? 'borderBottom' : 'borderInsideHor';
                            break;
                        }

                        // skip inner border, if already processed by preceding cell
                        if (!_.isString(targetBorderName)) { return; }
                        targetBorders = allBorders[targetBorderName] || (allBorders[targetBorderName] = []);

                        // insert all visible borders into the border collection
                        if (thisBorderVisible) { targetBorders.push(thisBorder); }
                        if (nextBorderVisible) { targetBorders.push(nextBorder); }

                        // ignore invisible border, if the other border is visible
                        if (!thisBorderVisible && !nextBorderVisible) { targetBorders.push(Border.NONE); }
                    });

                    // selection is editable only if all its cells are editable
                    if (!cellData.attributes.cell.unlocked) { selectionLocked = true; }
                });

                // all border information collected and sorted in allBorders
                _.each(allBorders, function (borders, borderName) {
                    selectionSettings.borders[borderName] = MixedBorder.mixBorders(borders);
                });

                // iterate unified ranges to calculate subtotals (no check for result needed, all cells are available)
                cellCollection.iterateCellsInRanges(SheetUtils.getUnifiedRanges(selection.ranges), function (cellEntry) {
                    selectionSettings.subtotals.cells += 1;
                    if (CellCollection.isNumber(cellEntry) && _.isFinite(cellEntry.result)) {
                        selectionSettings.subtotals.numbers += 1;
                        selectionSettings.subtotals.sum += cellEntry.result;
                        selectionSettings.subtotals.min = Math.min(selectionSettings.subtotals.min, cellEntry.result);
                        selectionSettings.subtotals.max = Math.max(selectionSettings.subtotals.max, cellEntry.result);
                    }
                }, { type: 'content' });
                selectionSettings.subtotals.average = selectionSettings.subtotals.sum / selectionSettings.subtotals.numbers;

                // whether selection contains locked cells
                selectionSettings.locked = selectionLocked;

                // notify listeners
                self.trigger('change:layoutdata');
            }

            // create the actual debounced updateSelectionSettings() method
            return self.createDebouncedMethod(updateActiveCell, updateSelectedRanges, { delay: 200 });
        }()); // end of local scope of updateSelectionSettings()

        /**
         * Returns the column/row intervals of the selected ranges, optionally
         * depending on a specific target column/row.
         *
         * @param {Boolean} columns
         *  Whether to return column intervals (true) or row intervals (false).
         *
         * @param {Object} [options]
         *  Additional options. The following options are supported:
         *  @param {Number} [options.target]
         *      If specified, the zero-based index of a specific target column
         *      or row. If the current selection contains entire column/row
         *      ranges covering this column/row, the respective intervals of
         *      these ranges will be returned instead; otherwise a single
         *      interval containing the target column/row will be returned.
         *  @param {Boolean} [options.visible=false]
         *      If set to true, the returned intervals will be reduced to the
         *      column or row intervals covered by the visible panes.
         *
         * @returns {Array}
         *  An array of column/row intervals for the current selection.
         */
        function getSelectedIntervals(columns, options) {

            var // the selected ranges
                ranges = self.getSelectedRanges(),
                // the target column/row
                target = Utils.getIntegerOption(options, 'target'),
                // the resulting intervals
                intervals = null;

            // use entire column/row ranges in selection, if target index has been passed
            if (_.isNumber(target)) {
                // filter full column/row ranges in selection
                ranges = _.filter(ranges, function (range) { return docModel.isFullRange(range, columns); });
                // if the target column/row is not contained in the selection, make an interval for it
                if (!SheetUtils.rangesContainIndex(ranges, target, columns)) {
                    intervals = [{ first: target, last: target }];
                }
            }

            // convert selected ranges to column/row intervals
            if (!intervals) {
                intervals = SheetUtils.getIntervals(ranges, columns);
            }

            // reduce the intervals to the header pane intervals
            if (Utils.getBooleanOption(options, 'visible', false)) {
                intervals = self.getVisibleIntervals(intervals, columns);
            }

            // unify and sort the resulting intervals
            return SheetUtils.getUnifiedIntervals(intervals);
        }

        /**
         * Returns the column/row intervals from the current selection, if it
         * contains hidden columns/rows. If the selected columns/rows are
         * completely visible, the selection consists of a single column/row
         * interval, and that interval is located next to a hidden column/row
         * interval (at either side, preferring leading), that hidden interval
         * will be returned instead.
         *
         * @returns {Array}
         *  The column/row intervals calculated from the current selection,
         *  containing hidden columns/rows. May be an empty array.
         */
        function getSelectedHiddenIntervals(columns) {

            var // the entire selected intervals
                intervals = getSelectedIntervals(columns);

            // safety check, should not happen
            if (intervals.length === 0) { return intervals; }

            var // the column/row collection
                collection = columns ? colCollection : rowCollection,
                // the mixed column/row attributes
                attributes = collection.getMixedAttributes(intervals);

            // if attribute 'visible' is false or null, the intervals contain hidden entries
            if (attributes.visible !== true) { return intervals; }

            // no special behavior for multiple visible intervals
            if (intervals.length > 1) { return []; }

            // try to find preceding hidden interval
            var lastIndex = intervals[0].first - 1;
            if ((lastIndex >= 0) && !collection.isEntryVisible(lastIndex)) {
                var prevEntry = collection.getPrevVisibleEntry(lastIndex);
                return [{ first: prevEntry ? (prevEntry.index + 1) : 0, last: lastIndex }];
            }

            // try to find following hidden interval
            var firstIndex = intervals[0].last + 1;
            if ((firstIndex <= collection.getMaxIndex()) && !collection.isEntryVisible(firstIndex)) {
                var nextEntry = collection.getNextVisibleEntry(firstIndex);
                return [{ first: firstIndex, last: nextEntry ? (nextEntry.index - 1) : collection.getMaxIndex() }];
            }

            // no hidden intervals found
            return [];
        }

        /**
         * Sets the optimal column width in the specified column intervals,
         * based on the content of cells.
         *
         * @param {Object|Array} colIntervals
         *  A single column interval, or an array of column intervals.
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved when the
         *  generated operations have been applied successfully, or that will
         *  be rejected on any error.
         */
        function setOptimalColumnWidth(colIntervals) {

            var // the entire column interval of the right header/grid panes
                rightInterval = self.getHeaderPane('right').getRenderInterval(),
                // resulting intervals with the new column widths
                resultIntervals = [],
                // last entry in the column width cache
                lastEntry = null,
                // default width for empty columns
                defWidth = colCollection.getDefaultAttributes().column.width;

            // returns all cell nodes from top and bottom grid pane in the specified column
            function getCellNodesInCol(col) {

                var // whether to use the left or the right panes
                    colPaneSide = (rightInterval && SheetUtils.intervalContainsIndex(rightInterval, col)) ? 'right' : 'left',
                    // the grid panes to be used to extract the cell nodes
                    topGridPane = self.getGridPane(PaneUtils.getPanePos(colPaneSide, 'top')),
                    bottomGridPane = self.getGridPane(PaneUtils.getPanePos(colPaneSide, 'bottom')),
                    // all existing cell nodes in the top and bottom grid panes
                    cellNodes = $();

                // extract all cell nodes from the top and bottom grid panes
                if (topGridPane.isVisible()) { cellNodes = cellNodes.add(topGridPane.getCellNodesInCol(col)); }
                if (bottomGridPane.isVisible()) { cellNodes = cellNodes.add(bottomGridPane.getCellNodesInCol(col)); }
                return cellNodes;
            }

            // process all visible columns
            colCollection.iterateEntries(colIntervals, function (colEntry) {

                var // process only cells that contain an optimal width attribute, and are not merged horizontally
                    cellNodes = getCellNodesInCol(colEntry.index).filter('[data-optimal-width]').not('[data-col-span]'),
                    // resulting optimal width of the current column
                    colWidth = 0;

                // find the maximum value of all content width attributes
                cellNodes.each(function () {
                    colWidth = Math.max(colWidth, Utils.getElementAttributeAsInteger(this, 'data-optimal-width'));
                });

                // use default column width, if no cells are available
                if (colWidth === 0) { colWidth = defWidth; }

                // do not generate operations, if the custom height was not modified before, and the new width is inside the tolerance
                if (!colEntry.attributes.column.customWidth && (Math.abs(colWidth - colEntry.attributes.column.width) < SET_COLUMN_WIDTH_TOLERANCE)) {
                    return;
                }

                // insert the optimal column width into the cache
                if (lastEntry && (lastEntry.last + 1 === colEntry.index) && (lastEntry.attrs.width === colWidth)) {
                    lastEntry.last += 1;
                } else {
                    lastEntry = { first: colEntry.index, last: colEntry.index, attrs: { width: colWidth, customWidth: false } };
                    resultIntervals.push(lastEntry);
                }
            });

            // create operations for all columns
            return activeSheetModel.setColumnAttributes(resultIntervals);
        }

        /**
         * Sets the optimal row height in the specified row intervals, based on
         * the content of cells.
         *
         * @param {Object|Array} rowIntervals
         *  A single row interval, or an array of row intervals.
         *
         * @param {Object|Array} [updateRanges]
         *  If specified, the addresses of the ranges changed by an operation
         *  which causes updating the automatic row heights implicitly. Rows
         *  with user-defined height (row attribute 'customHeight' set to true)
         *  will not be changed in this case.
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved when the
         *  generated operations have been applied successfully, or that will
         *  be rejected on any error.
         */
        function setOptimalRowHeight(rowIntervals, updateRanges) {

            var // all multi-line DOM cell nodes collected from the grid panes
                gridPaneCellNodes = [],
                // all multi-line DOM cell nodes, mapped by address key
                wrappedCellNodes = {},
                // difference between cell width and text content width
                totalPadding = 2 * activeSheetModel.getEffectiveCellPadding() + 1,
                // tolerance for automatic row height
                heightTolerance = updateRanges ? UPDATE_ROW_HEIGHT_TOLERANCE : SET_ROW_HEIGHT_TOLERANCE,
                // resulting intervals with the new row heights
                resultIntervals = [],
                // last entry in the row height cache
                lastEntry = null;

            // convert passed update ranges to array
            updateRanges = _.isObject(updateRanges) ? _.getArray(updateRanges) : null;

            // find all multi-line cell nodes in all grid panes that are not merged vertically
            self.iterateGridPanes(function (gridPane) {
                gridPaneCellNodes.push(gridPane.findCellNodes('>[data-wrapped]:not([data-row-span])'));
            });

            // map all found multi-line cell nodes by cell address (eliminate duplicates)
            _.each(gridPaneCellNodes, function (cellNodes) {
                cellNodes.each(function () {
                    var cellNode = $(this);
                    wrappedCellNodes[cellNode.attr('data-address')] = cellNode;
                });
            });

            // process all visible rows
            rowCollection.iterateEntries(rowIntervals, function (rowEntry) {

                var // the entire row range
                    rowRange = docModel.makeRowRange(rowEntry.index),
                    // resulting optimal height of the current row
                    rowHeight = documentStyles.getLineHeight(rowEntry.attributes.character);

                // find the maximum height of all existing cells using the cell collection (skip merged ranges over multiple rows)
                cellCollection.iterateCellsInRanges(rowRange, function (cellData, origRange, mergedRange) {

                    // skip all cells covered by merged ranges, or reference cells of merged ranges spanning multiple lines
                    if (mergedRange && (!_.isEqual(cellData.address, mergedRange.start) || (SheetUtils.getRowCount(mergedRange) > 1))) {
                        return;
                    }

                    var // whether the cell is located in the updated area
                        isUpdatedCell = updateRanges && SheetUtils.rangesContainCell(updateRanges, cellData.address),
                        // the DOM node of the cell (multi-line text nodes only)
                        cellNode = wrappedCellNodes[SheetUtils.getCellKey(cellData.address)],
                        // the resulting cell height
                        cellHeight = 0;

                    // temporarily render multi-line cells in updated ranges to get effective content height (cells are not yet rendered in the DOM)
                    if (isUpdatedCell && CellCollection.isWrappedText(cellData)) {

                        var charAttributes = _.clone(cellData.attributes.character);
                        charAttributes.fontSize = activeSheetModel.getEffectiveFontSize(charAttributes.fontSize);

                        // use font collection to get a DOM helper node with prepared font settings
                        cellHeight = fontCollection.getCustomFontMetrics(charAttributes, function (helperNode) {

                            var col = cellData.address[0],
                                cellWidth = colCollection.getIntervalPosition(mergedRange ? SheetUtils.getColInterval(mergedRange) : { first: col, last: col }).size,
                                lineHeight = activeSheetModel.getEffectiveLineHeight(cellData.attributes.character);

                            // prepare the helper node for multi-line text
                            helperNode.css({
                                width: Math.max(2, cellWidth - totalPadding),
                                lineHeight: lineHeight + 'px',
                                whiteSpace: 'pre-wrap',
                                wordWrap: 'break-word'
                            });

                            // insert the text spans
                            helperNode[0].innerHTML = _.map(cellData.display.split(/\n/), function (textLine) {
                                return '<span>' + Utils.escapeHTML(Utils.cleanString(textLine)) + '</span>';
                            }).join('<br>');

                            // return the resulting height in 1/100 mm
                            return activeSheetModel.convertPixelToHmm(helperNode.height());
                        });
                    }

                    // a multi-line cell node exists in the DOM, get its content height
                    else if (!isUpdatedCell && cellNode) {
                        // use height stored at the cell node if existing (e.g. vertically justified cells whose content node is too large)
                        cellHeight = Utils.getElementAttributeAsInteger(cellNode, 'data-optimal-height', 0);
                        // fall-back to the height of the content node (converted to 1/100 mm)
                        if (cellHeight === 0) {
                            cellHeight = activeSheetModel.convertPixelToHmm(cellNode.find('.content').height());
                        }
                    }

                    // calculate cell height from text line height according to the font settings of the cell
                    else {
                        cellHeight = documentStyles.getLineHeight(cellData.attributes.character);
                    }

                    rowHeight = Math.max(rowHeight, cellHeight);

                }, { type: 'existing', merged: true });

                // do not generate operations, if the row height was optimal before, and the new height is inside the tolerance
                if (!rowEntry.attributes.row.customHeight && (Math.abs(rowHeight - rowEntry.attributes.row.height) < heightTolerance)) {
                    return;
                }

                // insert the optimal row height into the cache
                if (lastEntry && (lastEntry.last + 1 === rowEntry.index) && (lastEntry.attrs.height === rowHeight)) {
                    lastEntry.last += 1;
                } else {
                    lastEntry = { first: rowEntry.index, last: rowEntry.index, attrs: { height: rowHeight, customHeight: false } };
                    resultIntervals.push(lastEntry);
                }
            }, { customSize: updateRanges ? false : null }); // do not change custom row height in update mode

            // create operations for all rows
            return activeSheetModel.setRowAttributes(resultIntervals);
        }

        /**
         * Updates the optimal row height in the specified ranges, based on the
         * content of cells.
         *
         * @param {Object|Array} ranges
         *  A single range address, or an array of range addresses.
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved when the
         *  generated operations have been applied successfully, or that will
         *  be rejected on any error.
         */
        function updateOptimalRowHeight(ranges) {

            var // restrict to visible area (performance)
                // TODO: iterate all row intervals, not single rows, important when updating entire column
                rowIntervals = self.getVisibleIntervals(SheetUtils.getRowIntervals(ranges), false);

            // pass modified ranges to indicate update mode (do not modify rows with custom height)
            return setOptimalRowHeight(rowIntervals, ranges);
        }

        /**
         * Fills a cell with the passed value and formatting, and updates the
         * optimal row height of the cell. See method
         * SheetModel.setSingleCellContents() for details.
         */
        function setSingleCellContents(address, value, attributes, options) {
            undoManager.enterUndoGroup(function () {

                var // try to fill the cell
                    result = activeSheetModel.setSingleCellContents(address, value, attributes, options);

                // show warning messages, convert result to boolean
                result = self.yellOnResult(result);

                // set optimal row height on success (bug 36773: return promise to keep undo group open)
                if (result) {
                    return updateOptimalRowHeight({ start: address, end: address });
                }
            });
        }

        /**
         * Fills a cell range with the passed values and formatting, and
         * updates the optimal row height of the affected cells. See method
         * SheetModel.setCellContents() for details.
         */
        function setCellContents(start, contents, options) {
            undoManager.enterUndoGroup(function () {

                var // try to fill the cells
                    result = activeSheetModel.setCellContents(start, contents, options);

                // show warning messages, convert result to boolean
                result = self.yellOnResult(result);

                // set optimal row height on success (bug 36773: return promise to keep undo group open)
                if (result) {
                    return updateOptimalRowHeight({ start: start, end: [start[0], start[1] + contents.length - 1] });
                }
            });
        }

        /**
         * Fills all cell ranges with the same value and formatting, and
         * updates the optimal row height of the affected cells. See method
         * SheetModel.fillCellRanges() for details.
         */
        function fillCellRanges(ranges, value, attributes, options) {
            undoManager.enterUndoGroup(function () {

                var // try to fill the cell ranges
                    result = activeSheetModel.fillCellRanges(ranges, value, attributes, options);

                // show warning messages, convert result to boolean
                result = self.yellOnResult(result);

                // set optimal row height on success (bug 36773: return promise to keep undo group open)
                if (result) { return updateOptimalRowHeight(ranges); }
            });
        }

        /**
         * Clears all cell ranges, and updates the optimal row height of the
         * affected cells. See method SheetModel.clearCellRanges() for details.
         */
        function clearCellRanges(ranges) {
            undoManager.enterUndoGroup(function () {

                var // try to clear the cell ranges
                    result = activeSheetModel.clearCellRanges(ranges);

                // show warning messages, convert result to boolean
                result = self.yellOnResult(result);

                // set optimal row height on success (bug 36773: return promise to keep undo group open)
                if (result) { return updateOptimalRowHeight(ranges); }
            });
        }

        /**
         * Returns the current query string from the search/replace tool bar.
         *
         * @returns {String}
         *  The current query string from the search/replace tool bar.
         */
        function getSearchQuery() {
            var search = app.getWindow().search;
            return _.isObject(search) ? search.query : '';
        }

        /**
         * Clears all cached search results.
         */
        function clearSearchResults() {
            searchSettings = _.copy(DEFAULT_SEARCH_SETTINGS, true);
        }

        /**
         * Selects the specified search result cell, and scrolls to that cell.
         *
         * @param {Number} index
         *  The zero-based index into the array of search result addresses.
         */
        function selectSearchResult(index) {

            var // the addresses of all cells containing the search text
                addresses = _.isObject(searchSettings.sheets) ? searchSettings.sheets[activeSheet] : null,
                // the specified cell address
                address = _.isArray(addresses) ? addresses[index] : null;

            searchSettings.index = index;
            if (docModel.isValidAddress(address)) {
                self.getActiveGridPane().scrollToCell(address);
                self.selectCell(address);
            }
        }

        /**
         * Checks whether the passed search query string matches the cached
         * query string of an ongoing search operation, and returns the cached
         * cell addresses of the search result on success. If the passed string
         * is different from the cached query string, a new search request will
         * be sent to the server automatically.
         *
         * @param {String} query
         *  The search query string to be checked.
         *
         * @returns {Array|Null}
         *  The addresses of all cells containing the query text in the active
         *  sheet; or null, if no search result is available.
         */
        function getSearchResults(query) {

            // if there are no results or if user wants to search a new query, do a search and quit
            if ((searchSettings.query !== query) || !_.isObject(searchSettings.sheets)) {

                // reset current search results before a new search request
                clearSearchResults();

                // do not request results for an empty search string
                // do nothing while sending operations (TODO: really?)
                if ((query.length > 0) && (app.getState() !== 'sending')) {

                    // store search query string for later usage
                    searchSettings.query = query;

                    // the search starts from the active cell (this way, user will always get the closest result)
                    requestViewUpdate({ search: {
                        searchText: query,
                        start: self.getActiveCell(),
                        limit: SheetUtils.MAX_SEARCH_RESULT_COUNT
                    }});
                }
                return null;
            }

            var // the addresses of all cells containing the search text
                addresses = searchSettings.sheets[activeSheet];

            // user may have switched to other sheets after search
            return (_.isArray(addresses) && (addresses.length > 0)) ? addresses : null;
        }

        /**
         * Replaces the text in the active cell, if it is part of the search
         * results, otherwise of the first search result.
         *
         * @param {String} query
         *  The text to be replaced.
         *
         * @param {String} replace
         *  The replacement text.
         */
        function replaceSingle(query, replace) {

            var // the addresses of all cells containing the searched text
                addresses = getSearchResults(query);

            // nothing to do without search results
            if (!addresses) { return; }

            var // address of the active cell
                activeCell = self.getActiveCell(),
                // the array index of the active cell in the search results
                index = Utils.findFirstIndex(addresses, function (elem) { return _.isEqual(elem, activeCell); });

            // select the first search result, if active cell is not a result cell (this
            // updates the 'activeCellSettings' object with the contents of the new cell!)
            index = Math.max(0, index);
            selectSearchResult(index);

            var // the name of the property in the 'activeCellSettings' object to change (formula or display string)
                cellPropName = _.isString(activeCellSettings.formula) ? 'formula' : 'display',
                // the old text before replacement
                oldText = activeCellSettings[cellPropName],
                // the new text after replacement
                // TODO handle number format special replacements like Date, Currency etc.
                newText = oldText.replace(new RegExp(_.escapeRegExp(query), 'gi'), replace);

            // nothing to do, if the text does not change
            if (oldText === newText) { return; }

            // send a document operation to change the cell
            searchSettings.replace = true;
            self.setCellContents(newText, undefined, { parse: true });

            // remove the modified cell from the result array
            addresses.splice(index, 1);

            // show info alert banner, if all cells have been replaced
            if (addresses.length === 0) {
                searchSettings.sheets = null;
                self.yell({ type: 'info', message: gt('Nothing else to replace.') });
            } else {
                // select the next search result (wrap to first result, if the last has been replaced)
                selectSearchResult(index % addresses.length);
            }
        }

        /**
         * Sends a server request to replace all occurrences of the specified
         * text in the active sheet.
         *
         * @param {String} query
         *  The text to be replaced.
         *
         * @param {String} replace
         *  The replacement text.
         */
        function replaceAll(query, replace) {

            var // the index of the active sheet where to replace (cached for deferred done handler)
                sheet = activeSheet,
                // the server request for the 'replace all' action
                request = app.sendActionRequest('replaceAll', {
                    sheet: sheet,
                    searchText: query,
                    replaceText: replace
                });

            // ignore failed requests
            request.fail(function (response) {
                Utils.error('ViewFuncMixin.replaceAll(): replace request failed:', response);
            });

            // show an alert banner containing the number of replaced cells
            request.done(function (result) {

                var // the sheet map containing addresses of all changed cells
                    sheetMap = Utils.getObjectOption(result.changed, 'sheets'),
                    // the addresses of the changed ranges in the active sheet
                    ranges = sheetMap ? Utils.getArrayOption(sheetMap[sheet], 'cells', []) : [],
                    // the number of changed cells
                    cellCount = 0,
                    // the message text for the alert banner
                    message = '';

                // add missing 'end' properties in the returned ranges
                _.each(ranges, function (range) {
                    if (!range.end) { range.end = range.start; }
                });

                // count the cells
                cellCount = SheetUtils.getCellCountInRanges(ranges);

                // show special label, if no cell has changed
                if (cellCount === 0) {
                    message = gt('Nothing to replace.');
                } else {
                    message = gt.format(
                        //#. %1$d is the number of cells replaced with a 'replace all' document action
                        //#, c-format
                        gt.ngettext('Successfully replaced %1$d cell.', 'Successfully replaced %1$d cells.', cellCount),
                        _.noI18n(cellCount)
                    );
                }

                self.yell({ type: 'info', message: message });
            });
        }

        /**
         * Initializes all sheet-dependent class members according to the
         * current active sheet.
         */
        function changeActiveSheetHandler(event, sheet, sheetModel) {

            // store model instance and index of the new active sheet
            activeSheetModel = sheetModel;
            activeSheet = sheet;

            // get collections of the new active sheet
            colCollection = sheetModel.getColCollection();
            rowCollection = sheetModel.getRowCollection();
            mergeCollection = sheetModel.getMergeCollection();
            cellCollection = sheetModel.getCellCollection();
            tableCollection = sheetModel.getTableCollection();
            drawingCollection = sheetModel.getDrawingCollection();
        }

        /**
         * Handles 'docs:update:cells' notifications triggered for any changed
         * cell in the entire document.
         */
        function updateNotificationHandler(changedData) {

            // do nothing, if the active sheet did not change
            if (!(activeSheet in changedData)) { return; }

            // throw away search results if the active sheet is altered in any way
            if (self.isSearchActive() && !searchSettings.replace) {
                clearSearchResults();
            }
        }

        /**
         * Initializes additional document functionality for debug mode.
         */
        function initDebugOperations() {

            function requestUpdate(type) {
                switch (type) {
                case 'view':
                    app.trigger('docs:update:cells', { changed: { sheets: Utils.makeSimpleObject(activeSheet, { cells: [docModel.getSheetRange()] } ) } });
                    break;
                case 'cells':
                    app.trigger('docs:update:cells', { changed: { sheets: Utils.makeSimpleObject(activeSheet, { cells: self.getSelectedRanges() } ) } });
                    break;
                case 'selection':
                    requestSelectionUpdate(self.getSelection());
                    break;
                }
            }

            function insertDrawing() {

                var generator = activeSheetModel.createOperationsGenerator(),
                    index = drawingCollection.getModelCount(),
                    selection = _.extend(self.getSelection(), { drawings: [] });

                _.each(self.getSelectedRanges(), function (range) {
                    var attrs = { drawing: {
                        name: 'Diagram ' + (index + 1),
                        description: 'Very, very, very, very, very, very, very, very, very long description for this diagram.',
                        startCol: range.start[0],
                        startColOffset: 100,
                        startRow: range.start[1],
                        startRowOffset: 100,
                        endCol: range.end[0],
                        endColOffset: 2400,
                        endRow: range.end[1],
                        endRowOffset: 400
                    } };
                    generator.generateDrawingOperation(Operations.INSERT_DRAWING, [index], { type: 'diagram', attrs: attrs });
                    selection.drawings.push([index]);
                    index += 1;
                });

                docModel.applyOperations(generator.getOperations());
                activeSheetModel.setViewAttribute('selection', selection);
            }

            // additional controller items for the debug controls
            app.getController().registerDefinitions({
                'debug/view/update': {
                    parent: 'debug/enabled',
                    set: requestUpdate
                },
                'debug/insert/drawing': {
                    parent: ['debug/enabled', 'document/editable'],
                    set: insertDrawing
                }
            });
        }

        // public methods -----------------------------------------------------

        /**
         * Returns the subtotal results in the current selection.
         *
         * @returns {Object}
         *  The current subtotal results in the current selection, in the
         *  properties 'sum', 'min', 'max', 'numbers', 'cells', and 'average'.
         */
        this.getSubtotals = function () {
            return _.clone(selectionSettings.subtotals);
        };

        /**
         * Returns whether the active sheet is locked. In locked sheets it is
         * not possible to insert, delete, or modify columns and rows, to edit
         * protected cells, or to manipulate drawing objects.
         *
         * @returns {Boolean}
         *  Whether the active sheet is locked.
         */
        this.isSheetLocked = function () {
            return activeSheetModel.isLocked();
        };

        /**
         * Returns whether the active sheet, and the current active cell in the
         * selection is locked.
         *
         * @returns {Boolean}
         *  Whether the active cell in the current selection is locked.
         */
        this.isActiveCellLocked = function () {
            return !activeCellSettings.attributes.cell.unlocked && this.isSheetLocked();
        };

        /**
         * Returns whether the active sheet is locked, and the current
         * selection in the active sheet contains at least one locked cell.
         *
         * @returns {Boolean}
         *  Whether the cells in the current selection are locked.
         */
        this.isSelectionLocked = function () {
            return selectionSettings.locked && this.isSheetLocked();
        };

        /**
         * Returns whether the passed ranges are editable, whereas editable
         * means that no locked cells are found in the ranges, and the sheet
         * where these ranges are, is not protected.
         *
         * @param {Object|Array} ranges
         *  The logical address of a single range, or an array of logical
         *  range addresses whose cells will be iterated.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {String} [options.error='cells:locked']
         *      The error code to be passed to the rejected promise.
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved if the
         *  passed ranges are editable, and rejected with an appropriate error
         *  code (see option 'error' above), if they are not.
         */
        this.areRangesEditable = function (ranges, options) {

            var // the error code for rejecting the promise
                errorCode = Utils.getStringOption(options, 'error', 'cells:locked'),
                // the Deferred object of the AJAX request
                request = null,
                // whether all cells in all ranges are editable (not locked)
                rangesEditable = true,
                // the number of cells visited locally
                visitedCells = 0;

            // return resolved Deferred object and quit if sheet is not locked
            if (!activeSheetModel.isLocked()) { return $.when(); }

            // prevent processing any cell twice
            ranges = SheetUtils.getUnifiedRanges(ranges);

            // visit all cells locally, if they are covered by the cell collection
            if (cellCollection.containsRanges(ranges)) {

                // visit only the existing cells in the collection (performance),
                // but include all cells in all hidden columns and rows
                cellCollection.iterateCellsInRanges(ranges, function (cellData) {
                    visitedCells += 1;
                    if (!cellData.attributes.cell.unlocked) {
                        rangesEditable = false;
                        return Utils.BREAK; // locked cell found: exit loop
                    }
                }, { type: 'existing', hidden: 'all' });

                // if 'rangesEditable' is false, a locked cell has been found already
                if (!rangesEditable) { return $.Deferred().reject(errorCode); }

                // if the ranges contain undefined cells not visited in the iterator above,
                // check the default attributes of empty cells
                if (visitedCells < SheetUtils.getCellCountInRanges(ranges)) {
                    if (!cellCollection.getDefaultAttributes().cell.unlocked) {
                        return $.Deferred().reject(errorCode);
                    }
                }

                // all cells are editable
                return $.when();
            }

            // ranges are not covered by the cell collections, send server request
            // to detect the locked state of the specified cells
            request = app.sendActionRequest('updateview', {
                sheet: activeSheet,
                locale: Config.LOCALE,
                selection: { ranges: ranges, active: ranges[0].start }
            });

            // reject Deferred object if selection contains a locked cell
            return request.then(function (response) {
                return response.selection.locked ? $.Deferred().reject() : null;
            });
        };

        /**
         * Tries to initiate modifying the active sheet. If the document is in
         * read-only mode, or the active sheet is locked, a notification will
         * be shown to the user.
         *
         * @returns {Boolean}
         *  Whether modifying the active sheet is possible.
         */
        this.requireUnlockedActiveSheet = function () {

            // prerequisite is global document edit mode (shows its own warning alert)
            if (!this.requireEditMode()) { return false; }

            // success if the sheet is not locked
            if (!this.isSheetLocked()) { return true; }

            // show warning alert banner for locked sheets
            return this.yellOnResult('sheet:locked');
        };

        /**
         * Tries to initiate modifying the active cell in the active sheet. If
         * the document is in read-only mode, or the active sheet and the
         * active cell are locked, a notification will be shown to the user.
         *
         * @returns {Boolean}
         *  Whether modifying the current active cell is possible.
         */
        this.requireEditableActiveCell = function () {

            // prerequisite is global document edit mode (shows its own warning alert)
            if (!this.requireEditMode()) { return false; }

            // success if the sheet is not locked, or the active cell is not protected
            if (!this.isActiveCellLocked()) { return true; }

            // show warning alert banner for locked active cell
            return this.yellOnResult('cells:locked');
        };

        /**
         * Tries to initiate modifying all cells in the current selection of
         * the active sheet. If the document is in read-only mode, or the
         * active sheet is locked and the selection contains locked cells, a
         * notification will be shown to the user.
         *
         * @returns {Boolean}
         *  Whether modifying the current selection is possible.
         */
        this.requireEditableSelection = function () {

            // prerequisite is global document edit mode (shows its own warning alert)
            if (!this.requireEditMode()) { return false; }

            // success if the sheet is not locked, or the selection does not contain a protected cell
            if (!this.isSelectionLocked()) { return true; }

            // show warning alert banner for locked selection
            return this.yellOnResult('cells:locked');
        };

        // sheet operations ---------------------------------------------------

        /**
         * Returns whether the passed sheet type is supported by this
         * application.
         *
         * @param {String} sheetType
         *  A sheet type identifier, as used in the 'insertSheet' operation.
         *
         * @returns {Boolean}
         *  Whether the passed sheet type is supported by this application.
         */
        this.isSheetTypeSupported = function (sheetType) {
            return (/^(worksheet)$/).test(sheetType);
        };

        /**
         * Inserts a new sheet in the spreadsheet document, and activates it in
         * the view. In case of an error, shows a notification message.
         *
         * @returns {Boolean}
         *  Whether the new sheet has been inserted successfully.
         */
        this.insertSheet = function () {

            var // create an unused sheet name
                sheetName = this.generateUnusedSheetName(),
                // try to insert a new sheet
                result = docModel.insertSheet(activeSheet + 1, sheetName);

            // show error messages, convert result to boolean
            result = this.yellOnResult(result);

            // activate the new sheet
            if (result) { this.activateNextSheet(); }
            return result;
        };

        /**
         * Deletes the active sheet from the spreadsheet document, and
         * activates another sheet in the view. Does nothing, if the document
         * contains a single sheet only.
         *
         * @returns {Boolean}
         *  Whether the sheet has been deleted successfully.
         */
        this.deleteSheet = function () {
            // handlers of the delete events will update active sheet
            return docModel.deleteSheet(activeSheet);
        };

        /**
         * Moves the active sheet in the document to a new position.
         *
         * @param {Number} toSheet
         *  The new zero-based index of the sheet.
         *
         * @returns {Boolean}
         *  Whether moving the sheet was successful.
         */
        this.moveSheet = function (toSheet) {
            return docModel.moveSheet(activeSheet, toSheet);
        };

        /**
         * Copies the active sheet and inserts the copy with the provided sheet
         * name behind the active sheet. In case of an error, shows a warning
         * message.
         *
         * @param {String} sheetName
         *  The name of the new sheet.
         *
         * @returns {Boolean}
         *  Whether the sheet has been copied successfully.
         */
        this.copySheet = function (sheetName) {

            var // try to copy the sheet
                result = docModel.copySheet(activeSheet, activeSheet + 1, sheetName);

            // show error messages, convert result to boolean
            result = this.yellOnResult(result);

            // activate the new sheet
            if (result) { this.activateNextSheet(); }
            return result;
        };

        /**
         * Returns the name of the active sheet.
         *
         * @returns {String}
         *  The name of the active sheet.
         */
        this.getSheetName = function () {
            return docModel.getSheetName(activeSheet);
        };

        /**
         * Renames the active sheet in the spreadsheet document, if possible.
         * In case of an error, shows a notification message.
         *
         * @param {String} sheetName
         *  The new name of the active sheet.
         *
         * @returns {Boolean}
         *  Whether the sheet has been renamed successfully.
         */
        this.setSheetName = function (sheetName) {

            // leave cell edit mode before renaming the sheet (sheet name may be used in formulas)
            if (!this.leaveCellEditMode('auto', { validate: true })) {
                return false;
            }

            var // try to rename the sheet
                result = docModel.setSheetName(activeSheet, sheetName);

            // show error messages, convert result to boolean
            return this.yellOnResult(result);
        };

        /**
         * Returns the merged formatting attributes of the active sheet.
         *
         * @returns {Object}
         *  The merged attribute set of the active sheet.
         */
        this.getSheetAttributes = function () {
            return activeSheetModel.getMergedAttributes();
        };

        /**
         * Changes the formatting attributes of the active sheet.
         *
         * @param {Object} attributes
         *  An (incomplete) attribute set with the new attributes for the
         *  active sheet.
         *
         * @returns {ViewFuncMixin}
         *  A reference to this instance.
         */
        this.setSheetAttributes = function (attributes) {

            // leave cell edit mode before hiding the sheet
            if (_.isObject(attributes.sheet) && (attributes.sheet.visible === false)) {
                if (!this.leaveCellEditMode('auto', { validate: true })) {
                    return this;
                }
            }

            var // the generator for the operation
                generator = activeSheetModel.createOperationsGenerator();

            generator.generateSheetOperation(Operations.SET_SHEET_ATTRIBUTES, { attrs: attributes });
            docModel.applyOperations(generator.getOperations());
            return this;
        };

        /**
         * Makes all hidden sheets in the view visible.
         *
         * @returns {ViewFuncMixin}
         *  A reference to this instance.
         */
        this.showAllSheets = function () {

            var // the generator for the operation
                generator = docModel.createOperationsGenerator(),
                // the attribute set to be inserted into the operations
                properties = { attrs: { sheet: { visible: true } } };

            docModel.iterateSheetModels(function (sheetModel, sheet) {
                if (!sheetModel.getMergedAttributes().sheet.visible) {
                    generator.generateOperation(Operations.SET_SHEET_ATTRIBUTES, _.extend({ sheet: sheet }, properties));
                }
            });
            docModel.applyOperations(generator.getOperations());
            return this;
        };

        // sheet view operations ----------------------------------------------

        /**
         * Returns the values of all view attribute of the active sheet.
         *
         * @returns {Object}
         *  A cloned map containing all view attribute values of the active
         *  sheet.
         */
        this.getSheetViewAttributes = function () {
            return activeSheetModel.getViewAttributes();
        };

        /**
         * Returns the value of a specific view attribute of the active sheet.
         *
         * @param {String} name
         *  The name of the view attribute.
         *
         * @returns {Any}
         *  A clone of the attribute value.
         */
        this.getSheetViewAttribute = function (name) {
            return activeSheetModel.getViewAttribute(name);
        };

        /**
         * Changes specific view attributes of the active sheet.
         *
         * @param {Object} attributes
         *  A map with all view attributes to be changed.
         *
         * @returns {ViewFuncMixin}
         *  A reference to this instance.
         */
        this.setSheetViewAttributes = function (attributes) {
            activeSheetModel.setViewAttributes(attributes);
            return this;
        };

        /**
         * Changes a specific view attribute of the active sheet.
         *
         * @param {String} name
         *  The name of the view attribute.
         *
         * @param {Any} value
         *  The new value for the view attribute.
         *
         * @returns {ViewFuncMixin}
         *  A reference to this instance.
         */
        this.setSheetViewAttribute = function (name, value) {
            activeSheetModel.setViewAttribute(name, value);
            return this;
        };

        /**
         * Returns the current zoom factor of the active sheet.
         *
         * @returns {Number}
         *  The current zoom factor, as floating-point number. The value 1
         *  represents a zoom of 100%.
         */
        this.getZoom = function () {
            return activeSheetModel.getViewAttribute('zoom');
        };

        /**
         * Changes the current zoom factor of all sheets in the document.
         *
         * @param {Number} zoom
         *  The new zoom factor, as floating-point number. The value 1
         *  represents a zoom of 100%.
         *
         * @returns {ViewFuncMixin}
         *  A reference to this instance.
         */
        this.setZoom = function (zoom) {
            activeSheetModel.setViewAttribute('zoom', zoom);
            return this;
        };

        /**
         * Decreases the zoom factor of the active sheet in the document.
         *
         * @returns {ViewFuncMixin}
         *  A reference to this instance.
         */
        this.decZoom = function () {

            var // current zoom factor
                currZoom = this.getZoom(),
                // find last entry in ZOOM_FACTORS with a factor less than current zoom
                prevZoom = Utils.findLast(ZOOM_FACTORS, function (zoom) { return zoom < currZoom; });

            // set new zoom at active sheet
            if (_.isNumber(prevZoom)) { this.setZoom(prevZoom); }
            return this;
        };

        /**
         * Increases the zoom factor of the active sheet in the document.
         *
         * @returns {ViewFuncMixin}
         *  A reference to this instance.
         */
        this.incZoom = function () {

            var // current zoom factor
                currZoom = this.getZoom(),
                // find first entry in ZOOM_FACTORS with a factor greater than current zoom
                nextZoom = _.find(ZOOM_FACTORS, function (zoom) { return zoom > currZoom; });

            // set new zoom at active sheet
            if (_.isNumber(nextZoom)) { this.setZoom(nextZoom); }
            return this;
        };

        /**
         * Generates and sends operations for all view attributes that have
         * been changed while the document was open.
         *
         * @returns {ViewFuncMixin}
         *  A reference to this instance.
         */
        this.sendChangedViewAttributes = function () {

            // Only send changed view attributes when user has edit rights, and has modified
            // the document locally, or if 'important' sheet view settings have been changed.
            // This is needed to prevent creating new versions of the document after selecting
            // cells or scrolling around, but without editing anything.
            if (!docModel.getEditMode()) { return this; }

            var // whether the document is modified, and view settings can be sent
                modified = app.isLocallyModified();

            // detect if any sheet contains 'important' changed view attributes
            if (!modified) {
                docModel.iterateSheetModels(function (sheetModel) {
                    if (sheetModel.hasChangedViewAttributes()) {
                        modified = true;
                        return Utils.BREAK;
                    }
                });
            }

            // nothing to do (e.g. only selection has been changed, which will not be saved then)
            if (!modified) { return this; }

            var // the operations generator
                generator = docModel.createOperationsGenerator();

            // create 'setDocumentAttributes' operation for the active sheet index
            if (activeSheet !== documentStyles.getDocumentAttributes().activeSheet) {
                generator.generateOperation(Operations.SET_DOCUMENT_ATTRIBUTES, { attrs: { document: { activeSheet: activeSheet } } });
            }

            // create operations for the changed view attributes
            docModel.iterateSheetModels(function (sheetModel, sheet) {

                var // the new sheet attributes according to the view settings
                    sheetAttributes = sheetModel.getChangedViewAttributes();

                // create an operation, if any attributes have been changed
                if (!_.isEmpty(sheetAttributes)) {
                    generator.generateOperation(Operations.SET_SHEET_ATTRIBUTES, { sheet: sheet, attrs: { sheet: sheetAttributes } });
                }
            });

            // send and apply the changed view attributes
            docModel.applyOperations(generator.getOperations());
            return this;
        };

        // split settings -----------------------------------------------------

        /**
         * Returns whether the view split mode is enabled for the active sheet,
         * regardless of the specific split mode (dynamic or frozen mode).
         *
         * @returns {Boolean}
         *  Whether the view split mode is enabled for the active sheet.
         */
        this.hasSplit = function () {
            return activeSheetModel.hasSplit();
        };

        /**
         * Returns the current view split mode of the active sheet.
         *
         * @returns {String}
         *  The current view split mode. The empty string indicates that the
         *  view is not split at all. Otherwise, one of the following values
         *  will be returned:
         *  - 'split': Dynamic split mode (movable split lines).
         *  - 'frozen': Frozen split mode (a fixed number of columns and/or
         *      rows are frozen, split lines are not movable). When leaving
         *      frozen split mode, the split lines will be hidden.
         *  - 'frozenSplit': Frozen split mode (same behavior as 'frozen'
         *      mode). When leaving frozen split mode, the split lines remain
         *      visible and become movable (dynamic 'split' mode).
         */
        this.getSplitMode = function () {
            return activeSheetModel.getSplitMode();
        };

        /**
         * Returns whether the view is currently split, and dynamic split mode
         * is activated. Returns also true, if the split view is currently
         * frozen, but will thaw to dynamic split mode.
         *
         * @returns {Boolean}
         *  Whether dynamic split mode is activated (split modes 'split' or
         *  'frozenSplit').
         */
        this.hasDynamicSplit = function () {
            return activeSheetModel.hasDynamicSplit();
        };

        /**
         * Returns whether the view is currently split, and the frozen split
         * mode is activated.
         *
         * @returns {Boolean}
         *  Whether frozen split mode is activated (split modes 'frozen' or
         *  'frozenSplit').
         */
        this.hasFrozenSplit = function () {
            return activeSheetModel.hasFrozenSplit();
        };

        /**
         * Toggles the dynamic view split mode according to the current view
         * split mode, and the current selection. Enabling dynamic split mode
         * in a frozen split view will thaw the frozen panes. Otherwise,
         * enabling the dynamic split mode will split the view at the position
         * left of and above the active cell. Disabling the dynamic split mode
         * will remove the split also if the view is in frozen split mode.
         *
         * @param {Boolean} state
         *  The new state of the dynamic view split mode.
         *
         * @returns {ViewFuncMixin}
         *  A reference to this instance.
         */
        this.setDynamicSplit = function (state) {

            var // whether the view is currently split
                hasSplit = this.hasSplit(),
                // address of the active cell
                address = this.getActiveCell(),
                // the position of the active cell in the sheet
                cellRectangle = null,
                // the visible area of the active grid pane that will be split
                visibleRectangle = null,
                // the new split positions
                splitWidth = 0, splitHeight = 0,
                // additional scroll anchor attributes
                attributes = null;

            // disable dynamic or frozen split mode
            if (!state && hasSplit) {
                activeSheetModel.clearSplit();
            }

            // convert frozen split to dynamic split
            else if (state && this.hasFrozenSplit()) {
                activeSheetModel.setDynamicSplit(activeSheetModel.getSplitWidthHmm(), activeSheetModel.getSplitHeightHmm());
            }

            // set dynamic split at active cell
            else if (state && !hasSplit) {

                // additional scroll anchor attributes
                attributes = {};

                // get position of active cell and visible area of the grid pane
                cellRectangle = this.getCellRectangle(address, { expandMerged: true });
                visibleRectangle = this.getActiveGridPane().getVisibleRectangle();

                // do not split vertically, if the active cell is shown at the left border
                splitWidth = cellRectangle.left - visibleRectangle.left;
                if ((-cellRectangle.width < splitWidth) && (splitWidth < PaneUtils.MIN_PANE_SIZE)) {
                    splitWidth = 0;
                }

                // do not split horizontally, if the active cell is shown at the top border
                splitHeight = cellRectangle.top - visibleRectangle.top;
                if ((-cellRectangle.height < splitHeight) && (splitHeight < PaneUtils.MIN_PANE_SIZE)) {
                    splitHeight = 0;
                }

                // split in the middle of the grid pane, if active cell is outside the visible area
                if (((splitWidth === 0) && (splitHeight === 0)) ||
                    (splitWidth < 0) || (splitWidth > visibleRectangle.width - PaneUtils.MIN_PANE_SIZE) ||
                    (splitHeight < 0) || (splitHeight > visibleRectangle.height - PaneUtils.MIN_PANE_SIZE)
                ) {
                    splitWidth = Math.floor(visibleRectangle.width / 2);
                    splitHeight = Math.floor(visibleRectangle.height / 2);
                    attributes.anchorRight = colCollection.getScrollAnchorByOffset(visibleRectangle.left + splitWidth, { pixel: true });
                    attributes.anchorBottom = rowCollection.getScrollAnchorByOffset(visibleRectangle.top + splitHeight, { pixel: true });
                    attributes.activePane = 'topLeft';
                } else {
                    if (splitWidth > 0) { attributes.anchorRight = { index: address[0], ratio: 0 }; }
                    if (splitHeight > 0) { attributes.anchorBottom = { index: address[1], ratio: 0 }; }
                }

                // activate the dynamic split view
                splitWidth = activeSheetModel.convertPixelToHmm(splitWidth);
                splitHeight = activeSheetModel.convertPixelToHmm(splitHeight);
                activeSheetModel.setDynamicSplit(splitWidth, splitHeight, attributes);
            }

            return this;
        };

        /**
         * Toggles the frozen view split mode according to the current view
         * split mode, and the current selection. Enabling frozen split mode
         * in a dynamic split view will freeze the panes with their current
         * size (split mode 'frozenSplit'). Otherwise, enabling the frozen
         * split mode will split the view at the position left of and above the
         * active cell. Disabling the frozen split mode will either return to
         * dynamic split mode (if split mode was 'frozenSplit'), or will remove
         * the frozen split at all (if split mode was 'frozen').
         *
         * @param {Boolean} state
         *  The new state of the frozen view split mode.
         *
         * @returns {ViewFuncMixin}
         *  A reference to this instance.
         */
        this.setFrozenSplit = function (state) {

            var // whether the view is currently split
                hasSplit = this.hasSplit(),
                // address of the active cell
                address = this.getActiveCell(),
                // the current scroll position
                colAnchor = null, rowAnchor = null,
                // the new frozen column and row interval
                colInterval = null, rowInterval = null;

            // convert frozen split to dynamic split
            if (!state && hasSplit && (this.getSplitMode() === 'frozenSplit')) {
                activeSheetModel.setDynamicSplit(activeSheetModel.getSplitWidthHmm(), activeSheetModel.getSplitHeightHmm());
            }

            // disable split mode completely
            else if (!state && hasSplit) {
                activeSheetModel.clearSplit();
            }

            // convert dynamic split to frozen split
            else if (state && hasSplit && !this.hasFrozenSplit()) {
                activeSheetModel.setFrozenSplit(activeSheetModel.getSplitColInterval(), activeSheetModel.getSplitRowInterval());
            }

            // enable frozen split mode
            else if (state && !hasSplit) {

                // calculate frozen column interval (must result in at least one frozen column)
                colAnchor = activeSheetModel.getViewAttribute('anchorRight');
                colInterval = { first: (colAnchor.ratio < 0.5) ? colAnchor.index : (colAnchor.index + 1), last: address[0] - 1 };
                if (colInterval.first > colInterval.last) { colInterval = null; }

                // calculate frozen row interval (must result in at least one frozen row)
                rowAnchor = activeSheetModel.getViewAttribute('anchorBottom');
                rowInterval = { first: (rowAnchor.ratio < 0.5) ? rowAnchor.index : (rowAnchor.index + 1), last: address[1] - 1 };
                if (rowInterval.first > rowInterval.last) { rowInterval = null; }

                // activate the frozen split view
                activeSheetModel.setFrozenSplit(colInterval, rowInterval);
            }

            return this;
        };

        /**
         * Freezes the passed number of columns and rows in the active sheet,
         * regardless of the current split settings.
         *
         * @param {Number} cols
         *  The number of columns to freeze. The value zero will not freeze any
         *  column.
         *
         * @param {Number} rows
         *  The number of rows to freeze. The value zero will not freeze any
         *  row.
         *
         * @returns {ViewFuncMixin}
         *  A reference to this instance.
         */
        this.setFixedFrozenSplit = function (cols, rows) {

            var // the scroll anchors for the first visible column/row
                colAnchor = activeSheetModel.getViewAttribute(activeSheetModel.hasColSplit() ? 'anchorLeft' : 'anchorRight'),
                rowAnchor = activeSheetModel.getViewAttribute(activeSheetModel.hasRowSplit() ? 'anchorTop' : 'anchorBottom'),
                // the first visible column and row in the split view
                col = ((colAnchor.index < docModel.getMaxCol()) && (colAnchor.ratio > 0.5)) ? (colAnchor.index + 1) : colAnchor.index,
                row = ((rowAnchor.index < docModel.getMaxRow()) && (rowAnchor.ratio > 0.5)) ? (rowAnchor.index + 1) : rowAnchor.index;

            // reduce start indexes to be able to freeze the passed number of columns/rows
            col = Math.max(0, Math.min(col, docModel.getMaxCol() - cols + 1));
            row = Math.max(0, Math.min(row, docModel.getMaxRow() - rows + 1));

            // activate the frozen split view
            activeSheetModel.setFrozenSplit((cols > 0) ? { first: col, last: col + cols - 1 } : null, (rows > 0) ? { first: row, last: row + rows - 1 } : null);

            return this;
        };

        // column operations --------------------------------------------------

        /**
         * Returns whether additional columns can be inserted into the sheet,
         * based on the current selection.
         *
         * @returns {Boolean}
         *  Whether additional columns can be inserted, according to the
         *  current selection and the number of used columns in the sheet.
         */
        this.canInsertColumns = function () {

            var // the column intervals in the current selection
                colIntervals = SheetUtils.getColIntervals(this.getSelectedRanges());

            return activeSheetModel.canInsertColumns(colIntervals);
        };

        /**
         * Inserts new columns into the active sheet, according to the current
         * selection.
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved when the
         *  generated operations have been applied successfully, or that will
         *  be rejected on any error.
         */
        this.insertColumns = function () {

            var // the column intervals in the current selection
                colIntervals = SheetUtils.getColIntervals(this.getSelectedRanges());

            return this.yellOnPromise(activeSheetModel.insertColumns(colIntervals));
        };

        /**
         * Returns whether the selected columns can be deleted from the sheet.
         *
         * @returns {Boolean}
         *  Whether the selected columns can be deleted.
         */
        this.canDeleteColumns = function () {

            var // the column intervals in the current selection
                colIntervals = SheetUtils.getColIntervals(this.getSelectedRanges());

            return activeSheetModel.canDeleteColumns(colIntervals);
        };

        /**
         * Deletes existing columns from the active sheet, according to the
         * current selection.
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved when the
         *  generated operations have been applied successfully, or that will
         *  be rejected on any error.
         */
        this.deleteColumns = function () {

            var // the column intervals in the current selection
                colIntervals = SheetUtils.getColIntervals(this.getSelectedRanges());

            return this.yellOnPromise(activeSheetModel.deleteColumns(colIntervals));
        };

        /**
         * Returns the mixed formatting attributes of all columns in the active
         * sheet covered by the current selection.
         *
         * @returns {Object}
         *  A complete attribute set containing the mixed attributes of all
         *  selected columns. All attributes that could not be resolved
         *  unambiguously will be set to the value null.
         */
        this.getColumnAttributes = function () {

            var // the column intervals in the current selection
                colIntervals = getSelectedIntervals(true);

            return activeSheetModel.getColumnAttributes(colIntervals);
        };

        /**
         * Changes the specified column attributes in the active sheet,
         * according to the current selection.
         *
         * @param {Object} attributes
         *  The (incomplete) column attributes, as simple object (NOT mapped as
         *  a 'column' sub-object).
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved when the
         *  generated operations have been applied successfully, or that will
         *  be rejected on any error.
         */
        this.setColumnAttributes = function (attributes) {

            var // the column intervals in the current selection
                colIntervals = getSelectedIntervals(true);

            _.each(colIntervals, function (interval) { interval.attrs = attributes; });
            return this.yellOnPromise(activeSheetModel.setColumnAttributes(colIntervals));
        };

        /**
         * Returns whether the current selection contains hidden columns, or is
         * located next to hidden columns, that can be made visible.
         *
         * @returns {Boolean}
         *  Whether the current selection contains hidden columns.
         */
        this.canShowColumns = function () {
            return getSelectedHiddenIntervals(true).length > 0;
        };

        /**
         * Make hidden Columns visible
         *  Either all hidden columns in the current selected range(s)
         *  or if only one column is (or multiple single columns are) selected, the previous Column
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved when the
         *  generated operations have been applied successfully, or that will
         *  be rejected on any error.
         */
        this.showColumns = function () {

            var // the hidden column intervals in the current selection
                colIntervals = getSelectedHiddenIntervals(true);

            _.each(colIntervals, function (interval) { interval.attrs = { visible: true }; });
            return this.yellOnPromise(activeSheetModel.setColumnAttributes(colIntervals));
        };

        /**
         * Changes the width of columns in the active sheet, according to the
         * current selection.
         *
         * @param {Number} width
         *  The column width in 1/100 mm. If this value is less than 1, the
         *  columns will be hidden, and their original width will not be
         *  changed.
         *
         * @param {Object} [options]
         *  Additional optional parameters:
         *  @param {Number} [options.target]
         *      The target column to be changed. If specified, the column width
         *      will be set to that column only. If the current selection
         *      contains any ranges covering this column completely, the column
         *      width will be set to all columns contained in these ranges.
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved when the
         *  generated operations have been applied successfully, or that will
         *  be rejected on any error.
         */
        this.setColumnWidth = function (width, options) {

            var // the target column to be modified
                targetCol = Utils.getIntegerOption(options, 'target'),
                // hide columns if passed column width is zero; show hidden columns when width is changed
                attributes = (width === 0) ? { visible: false } : { visible: true, width: width, customWidth: true },
                // the column intervals to be modified
                colIntervals = getSelectedIntervals(true, { target: targetCol });

            _.each(colIntervals, function (interval) { interval.attrs = attributes; });
            return this.yellOnPromise(activeSheetModel.setColumnAttributes(colIntervals));
        };

        /**
         * Sets optimal column width based on the content of cells.
         *
         * @param {Number} [targetCol]
         *  The target column to be changed. If the current selection contains
         *  any ranges covering this column completely, the optimal column
         *  width will be set to all columns contained in these ranges.
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved when the
         *  generated operations have been applied successfully, or that will
         *  be rejected on any error.
         */
        this.setOptimalColumnWidth = function (targetCol) {

            var // the column intervals to be modified (only columns covered by the grid panes)
                colIntervals = getSelectedIntervals(true, { target: targetCol, visible: true });

            return this.yellOnPromise(setOptimalColumnWidth(colIntervals));
        };

        /**
         * Returns the column attributes of the column the active cell is
         * currently located in.
         *
         * @returns {Object}
         *  The column attribute map of the active cell.
         */
        this.getActiveColumnAttributes = function () {

            var // column index of the active cell
                activeCol = this.getActiveCell()[0];

            // return the column attributes only
            return colCollection.getEntry(activeCol).attributes.column;
        };

        // row operations -----------------------------------------------------

        /**
         * Returns whether additional rows scan be inserted into the sheet,
         * based on the current selection.
         *
         * @returns {Boolean}
         *  Whether additional rows can be inserted, according to the current
         *  selection and the number of used rows in the sheet.
         */
        this.canInsertRows = function () {

            var // the row intervals in the current selection
                rowIntervals = SheetUtils.getRowIntervals(this.getSelectedRanges());

            return activeSheetModel.canInsertRows(rowIntervals);
        };

        /**
         * Inserts new rows into the active sheet, according to the current
         * selection.
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved when the
         *  generated operations have been applied successfully, or that will
         *  be rejected on any error.
         */
        this.insertRows = function () {

            var // the row intervals in the current selection
                rowIntervals = SheetUtils.getRowIntervals(this.getSelectedRanges());

            return this.yellOnPromise(activeSheetModel.insertRows(rowIntervals));
        };

        /**
         * Returns whether the selected rows can be deleted from the sheet.
         *
         * @returns {Boolean}
         *  Whether the selected rows can be deleted.
         */
        this.canDeleteRows = function () {

            var // the row intervals in the current selection
                rowIntervals = SheetUtils.getRowIntervals(this.getSelectedRanges());

            return activeSheetModel.canDeleteRows(rowIntervals);
        };

        /**
         * Deletes existing rows from the active sheet, according to the
         * current selection.
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved when the
         *  generated operations have been applied successfully, or that will
         *  be rejected on any error.
         */
        this.deleteRows = function () {

            var // the row intervals in the current selection
                rowIntervals = SheetUtils.getRowIntervals(this.getSelectedRanges());

            return this.yellOnPromise(activeSheetModel.deleteRows(rowIntervals));
        };

        /**
         * Returns the mixed formatting attributes of all rows in the active
         * sheet covered by the current selection.
         *
         * @returns {Object}
         *  A complete attribute set containing the mixed attributes of all
         *  selected rows. All attributes that could not be resolved
         *  unambiguously will be set to the value null.
         */
        this.getRowAttributes = function () {

            var // the row intervals in the current selection
                rowIntervals = getSelectedIntervals(false);

            return activeSheetModel.getRowAttributes(rowIntervals);
        };

        /**
         * Changes the specified row attributes in the active sheet, according
         * to the current selection..
         *
         * @param {Object} attributes
         *  The (incomplete) row attributes, as simple object (NOT mapped as a
         *  'row' sub-object).
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved when the
         *  generated operations have been applied successfully, or that will
         *  be rejected on any error.
         */
        this.setRowAttributes = function (attributes) {

            var // the row intervals in the current selection
                rowIntervals = getSelectedIntervals(false);

            _.each(rowIntervals, function (interval) { interval.attrs = attributes; });
            return this.yellOnPromise(activeSheetModel.setRowAttributes(rowIntervals));
        };

        /**
         * Returns whether the current selection contains hidden rows, or is
         * located next to hidden rows, that can be made visible.
         *
         * @returns {Boolean}
         *  Whether the current selection contains hidden rows.
         */
        this.canShowRows = function () {
            return getSelectedHiddenIntervals(false).length > 0;
        };

        /**
         * Make hidden Rows visible
         *  Either all hidden rows in the current selected range(s)
         *  or if only one column is (or multiple single rows are) selected, the previous Row
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved when the
         *  generated operations have been applied successfully, or that will
         *  be rejected on any error.
         */
        this.showRows = function () {

            var // the hidden row intervals in the current selection
                rowIntervals = getSelectedHiddenIntervals(false);

            _.each(rowIntervals, function (interval) { interval.attrs = { visible: true }; });
            return this.yellOnPromise(activeSheetModel.setRowAttributes(rowIntervals));
        };

        /**
         * Changes the height of rows in the active sheet, according to the
         * current selection.
         *
         * @param {Number} height
         *  The row height in 1/100 mm. If this value is less than 1, the rows
         *  will be hidden, and their original height will not be changed.
         *
         * @param {Object} [options]
         *  Additional optional parameters:
         *  @param {Number} [options.target]
         *      The target row to be changed. If specified, the row height will
         *      be set to that row only. If the current selection contains any
         *      ranges covering this row completely, the row height will be set
         *      to all rows contained in these ranges.
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved when the
         *  generated operations have been applied successfully, or that will
         *  be rejected on any error.
         */
        this.setRowHeight = function (height, options) {

            var // the target row to be modified
                targetRow = Utils.getIntegerOption(options, 'target'),
                // hide rows if passed row height is zero; show hidden rows when height is changed
                attributes = (height === 0) ? { visible: false } : { visible: true, height: height, customHeight: true },
                // the row intervals to be modified
                rowIntervals = getSelectedIntervals(false, { target: targetRow });

            _.each(rowIntervals, function (interval) { interval.attrs = attributes; });
            return this.yellOnPromise(activeSheetModel.setRowAttributes(rowIntervals));
        };

        /**
         * Sets optimal row height based on the content of cells.
         *
         * @param {Number} [targetRow]
         *  The target row to be changed. If the current selection contains any
         *  ranges covering this row completely, the optimal row height will be
         *  set to all rows contained in these ranges.
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved when the
         *  generated operations have been applied successfully, or that will
         *  be rejected on any error.
         */
        this.setOptimalRowHeight = function (targetRow) {

            var // the row intervals to be modified (only rows covered by the grid panes)
                rowIntervals = getSelectedIntervals(false, { target: targetRow, visible: true });

            return this.yellOnPromise(setOptimalRowHeight(rowIntervals));
        };

        /**
         * Returns the row attributes of the row the active cell is currently
         * located in.
         *
         * @returns {Object}
         *  The row attribute map of the active cell.
         */
        this.getActiveRowAttributes = function () {

            var // row index of the active cell
                activeRow = this.getActiveCell()[1];

            // return the row attributes only
            return rowCollection.getEntry(activeRow).attributes.row;
        };

        // cell operations ----------------------------------------------------

        /**
         * Returns the value and formatting of the current active cell.
         *
         * @returns {Object}
         *  The cell value and formatting, in the following properties:
         *  - {String} display
         *      The formatted display string.
         *  - {Null|Number|Boolean|String} result
         *      The typed result value of the cell, or the formula result.
         *  - {String|Null} formula
         *      The formula definition, if the current cell is a formula cell.
         *  - {Object} attributes
         *      All formatting attributes of the cell, mapped by cell attribute
         *      family ('cell' and 'character'), and the identifier of the cell
         *      style sheet in the 'styleId' string property.
         *  - {Object} explicit
         *      The explicit formatting attributes of the cell, mapped by cell
         *      attribute family ('cell' and 'character'), and optionally the
         *      identifier of the cell style sheet.
         */
        this.getCellContents = function () {
            return _.clone(activeCellSettings);
        };

        /**
         * Returns the merged formatting attributes of the current active cell.
         *
         * @returns {Object}
         *  All formatting attributes of the active cell, mapped by cell
         *  attribute family ('cell' and 'character'), and the identifier of
         *  the cell style sheet in the 'styleId' string property. If the cell
         *  in-place edit mode is currently active, the edit attributes added
         *  there will be included too.
         */
        this.getCellAttributes = function () {

            var // the merged attributes of the active cell
                cellAttributes = _.copy(activeCellSettings.attributes, true),
                // additional formatting attributes from cell edit mode
                editAttributes = this.getCellEditAttributes();

            if (_.isObject(editAttributes)) {
                _.extend(cellAttributes.cell, editAttributes.cell);
                _.extend(cellAttributes.character, editAttributes.character);
            }

            return cellAttributes;
        };

        /**
         * Sets the contents of the active cell, and updates the view according
         * to the new cell value and formatting.
         *
         * @param {String} [value]
         *  The new value for the active cell. If omitted, only the cell
         *  formatting will be changed.
         *
         * @param {Object} [attributes]
         *  A cell attribute set, optionally containing the string property
         *  'styleId', and attribute value maps (name/value pairs), keyed by
         *  the supported attribute families 'cell' and 'character'. Omitted
         *  attributes will not be changed. Attributes set to the value null
         *  will be removed from the cell.
         *
         * @param {Object} [options]
         *  Additional optional parameters:
         *  @param {Boolean} [options.parse=false]
         *      If set to true, the 'parse' property will be inserted into the
         *      operation, set to the name of the current UI language.
         *  @param {Boolean} [options.clear=false]
         *      If set to true, all existing explicit attributes will be
         *      removed from the cell while applying the new attributes.
         *  @param {Any} [options.result]
         *      The result of a formula, as calculated by the local formula
         *      interpreter.
         *
         * @returns {ViewFuncMixin}
         *  A reference to this instance.
         */
        this.setCellContents = function (value, attributes, options) {

            // store character attributes while in-place cell edit mode is active
            if (this.isCellEditMode()) {
                return this.setCellEditAttributes(attributes);
            }

            // do nothing if active cell is locked
            if (this.requireEditableActiveCell()) {
                setSingleCellContents(this.getActiveCell(), value, attributes, options);
            }

            return this;
        };

        /**
         * Fills all cell ranges in the current selection with the same value
         * and formatting, and updates the view according to the new cell value
         * and formatting.
         *
         * @param {String} [value]
         *  The new value for all cells in the current selection. If omitted,
         *  only the cell formatting will be changed.
         *
         * @param {Object} [attributes]
         *  A cell attribute set, optionally containing the string property
         *  'styleId', and attribute value maps (name/value pairs), keyed by
         *  the supported attribute families 'cell' and 'character'. Omitted
         *  attributes will not be changed. Attributes set to the value null
         *  will be removed from the cell.
         *
         * @param {Object} [options]
         *  Additional optional parameters:
         *  @param {Boolean} [options.parse=false]
         *      If set to true, the 'parse' property will be inserted into the
         *      operations, set to the name of the current UI language.
         *  @param {Boolean} [options.clear=false]
         *      If set to true, all existing explicit attributes will be
         *      removed from the selection while applying the new attributes.
         *  @param {String} [options.cat]
         *      The number format category the format code contained in the
         *      passed formatting attributes belongs to.
         *
         * @returns {ViewFuncMixin}
         *  A reference to this instance.
         */
        this.fillCellRanges = function (value, attributes, options) {

            // store character attributes while in-place cell edit mode is active
            if (this.isCellEditMode()) {
                this.setCellEditAttributes(attributes);
            }

            // do nothing if current selection contains locked cells
            else if (this.requireEditableSelection()) {
                fillCellRanges(this.getSelectedRanges(), value, attributes, options);
            }

            return this;
        };

        /**
         * Deletes the values and formatting attributes from all cell ranges in
         * the current selection.
         *
         * @returns {ViewFuncMixin}
         *  A reference to this instance.
         */
        this.clearCellRanges = function () {

            // do nothing if current selection contains locked cells
            if (this.requireEditableSelection()) {
                clearCellRanges(this.getSelectedRanges());
            }

            return this;
        };

        /**
         * Changes a single attribute of type 'cell' for all cells in the
         * current selection.
         *
         * @param {String} name
         *  The name of the cell attribute.
         *
         * @param {Any} value
         *  The new value of the cell attribute. If set to the value null, all
         *  attributes set explicitly will be removed from the cells.
         *
         * @returns {ViewFuncMixin}
         *  A reference to this instance.
         */
        this.setCellAttribute = function (name, value) {
            return this.fillCellRanges(undefined, { cell: Utils.makeSimpleObject(name, value) });
        };

        /**
         * Changes a single attribute of type 'character' for all cells in the
         * current selection.
         *
         * @param {String} name
         *  The name of the character attribute.
         *
         * @param {Any} value
         *  The new value of the character attribute. If set to the value null,
         *  the current explicit attribute will be removed from the cells.
         *
         * @returns {ViewFuncMixin}
         *  A reference to this instance.
         */
        this.setCharacterAttribute = function (name, value) {
            return this.fillCellRanges(undefined, { character: Utils.makeSimpleObject(name, value) });
        };

        /**
         * removes all explicit formatting attributes from all cells in the
         * current selection.
         *
         * @returns {ViewFuncMixin}
         *  A reference to this instance.
         */
        this.clearAttributes = function () {
            return this.fillCellRanges(undefined, undefined, { clear: true });
        };

        /**
         * Returns the number format category of the active cell.
         *
         * @returns {String}
         *  The current number format category.
         */
        this.getNumberFormatCategory = function () {
            return activeCellSettings.format.cat;
        };

        /**
         * Sets selected number format category for the current active cell in
         * the current selection, and implicitly set the first format code of
         * the predefined template of this category to the cell.
         *
         * @returns {ViewFuncMixin}
         *  A reference to this instance.
         */
        this.setNumberFormatCategory = function (category) {
            var formatCode = numberFormatter.getCategoryDefaultCode(category);
            if ((category !== 'custom') && _.isString(formatCode)) {
                this.fillCellRanges(undefined, { cell: { numberFormat: numberFormatter.createNumberFormat(formatCode) } }, { cat: category });
            }
            return this;
        };

        /**
         * Returns the format code of the active cell.
         *
         * @returns {String}
         *  The current number format code.
         */
        this.getNumberFormatCode = function () {
            return numberFormatter.resolveFormatCode(activeCellSettings.attributes.cell.numberFormat);
        };

        /**
         * Returns the format code of the active cell.
         *
         * @param {String} formatCode
         *  The number format code to be set to the current selection.
         *
         * @returns {ViewFuncMixin}
         *  A reference to this instance.
         */
        this.setNumberFormatCode = function (formatCode) {
            // do nothing if custom fall-back button is clicked
            if (formatCode !== Labels.CUSTOM_FORMAT_VALUE) {
                this.setCellAttribute('numberFormat', numberFormatter.createNumberFormat(formatCode));
            }
            return this;
        };

        /**
         * Returns the border mode of the current selection.
         *
         * @returns {Object}
         *  The border mode representing the visibility of the borders in all
         *  ranges of the current selection. See 'MixedBorder.getBorderMode()'
         *  for more details.
         */
        this.getBorderMode = function () {
            return MixedBorder.getBorderMode(selectionSettings.borders);
        };

        /**
         * Changes the visibility of the borders in all ranges of the current
         * selection.
         *
         * @param {Object} borderMode
         *  The border mode representing the visibility of the borders in all
         *  ranges of the current selection. See 'MixedBorder.getBorderMode()'
         *  for more details.
         *
         * @returns {ViewFuncMixin}
         *  A reference to this instance.
         */
        this.setBorderMode = function (borderMode) {

            // quit if current selection contains locked cells
            if (this.requireEditableSelection()) {

                var // the border attributes to be sent
                    borderAttributes = MixedBorder.getBorderAttributes(borderMode, selectionSettings.borders, DEFAULT_SINGLE_BORDER),
                    // create and apply the operations
                    result = activeSheetModel.setBorderAttributes(this.getSelectedRanges(), borderAttributes);

                // show warning messages
                this.yellOnResult(result);

                // bug 34021: immediately update all selection settings to prevent flickering UI elements
                updateSelectionSettings({ direct: true });
            }

            return this;
        };

        /**
         * Returns the merged border attributes of the current selection.
         *
         * @returns {Object}
         *  An attribute map containing the merged borders of all available
         *  border attributes ('borderTop', 'borderLeft', etc.) in the current
         *  selection. Each border attribute value contains the properties
         *  'style', 'width', and 'color'. If the border property is equal in
         *  all affected cells, it will be set as property value in the border
         *  object, otherwise the property will be set to the value null.
         */
        this.getBorderAttributes = function () {
            return _.copy(selectionSettings.borders, true);
        };

        /**
         * Changes the specified border properties (line style, line color, or
         * line width) of all visible cell borders in the current selection.
         *
         * @param {Border} border
         *  A border attribute, may be incomplete. All properties contained in
         *  this object will be set for all visible borders in the current
         *  selection. Omitted border properties will not be changed.
         *
         * @returns {ViewFuncMixin}
         *  A reference to this instance.
         */
        this.changeVisibleBorders = function (border) {

            // quit if current selection contains locked cells
            if (this.requireEditableSelection()) {

                var // create and apply the operations
                    result = activeSheetModel.changeVisibleBorders(this.getSelectedRanges(), border);

                // show warning messages
                this.yellOnResult(result);

                // bug 34021: immediately update all selection settings to prevent flickering UI elements
                updateSelectionSettings({ direct: true });
            }

            return this;
        };

        /**
         * Returns whether the format painter is currently active.
         *
         * @returns {Boolean}
         *  Whether the format painter is currently active.
         */
        this.isFormatPainterActive = function () {
            return this.isCustomSelectionMode('painter');
        };

        /**
         * Activates or deactivates the format painter.
         *
         * @param {Boolean} state
         *  Whether to activate or deactivate the format painter.
         *
         * @returns {ViewFuncMixin}
         *  A reference to this instance.
         */
        this.activateFormatPainter = function (state) {

            // deactivate format painter: cancel custom selection mode
            if (!state) {
                this.cancelCustomSelectionMode();
                return this;
            }

            var // the attributes to be pasted into the range that will be selected (deep copy)
                attrs = Utils.extendOptions({ styleId: null }, activeCellSettings.explicit),
                // the promise representing custom selection mode
                promise = null;

            // do not lock editable cells by copying the 'unlocked' attribute (in locked sheets only, as in Excel)
            if (this.isSheetLocked()) {
                attrs.cell = Utils.extendOptions(attrs.cell, { unlocked: true });
            }

            // start custom selection mode (wait for range selection)
            promise = this.enterCustomSelectionMode('painter', {
                selection: this.getActiveCellAsSelection(),
                statusLabel: Labels.FORMAT_PAINTER_LABEL
            });

            // select the target range, bug 35295: check that the cells are not locked
            promise = promise.then(function (selection) {
                self.setCellSelection(selection);
                return self.areRangesEditable(selection.ranges);
            });

            // copy the formatting attributes to the target range
            promise.done(function () {
                fillCellRanges(self.getSelectedRanges(), undefined, attrs, { clear: true });
            });

            // show error alert if copying attributes has failed
            this.yellOnPromise(promise);
            return this;
        };

        /**
         * Fills or clears a cell range according to the position and contents
         * of the cell range currently selected (the 'auto-fill' feature).
         *
         * @param {String} border
         *  Which border of the selected range will be modified by the
         *  operation. Allowed values are 'left', 'right', 'top', or 'bottom'.
         *
         * @param {Number} count
         *  The number of columns/rows to extend the selected range with
         *  (positive values), or to shrink the selected range (negative
         *  values).
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved when the
         *  auto-fill target range is editable, and the auto-fill operation has
         *  been applied. Otherwise, the promise will be rejected.
         */
        this.autoFill = function (border, count) {

            var // all cell ranges selected in the sheet
                ranges = this.getSelectedRanges(),
                // whether to expand/shrink the leading or trailing border
                leading = /^(left|top)$/.test(border),
                // whether to expand/delete columns or rows
                columns = /^(left|right)$/.test(border),
                // the array index in the cell address
                addrIndex = columns ? 0 : 1,
                // the target range for auto-fill
                targetRange = _.copy(ranges[0], true),
                // the promise of the entire auto-fill operations
                promise = null;

            // for safety: do nothing if called in a multi-selection
            if (ranges.length !== 1) {
                // no error code, this should not happen as the GUI does not offer auto-fill
                promise = $.Deferred().reject();
            }

            // fill cells outside the selected range
            else if (count > 0) {

                // build the target range (adjacent to the selected range)
                if (leading) {
                    targetRange.start[addrIndex] = ranges[0].start[addrIndex] - count;
                    targetRange.end[addrIndex] = ranges[0].start[addrIndex] - 1;
                } else {
                    targetRange.start[addrIndex] = ranges[0].end[addrIndex] + 1;
                    targetRange.end[addrIndex] = ranges[0].end[addrIndex] + count;
                }

                // only the target range must be editable (not the source cells)
                promise = this.areRangesEditable(targetRange);

                // apply auto-fill, and update optimal row heights
                promise = promise.then(function () {
                    return undoManager.enterUndoGroup(function () {
                        // apply the 'autoFill' operation
                        activeSheetModel.autoFill(ranges[0], border, count);
                        // update automatic row height in the target range
                        // (bug 36773: return promise to keep undo group open)
                        return updateOptimalRowHeight(targetRange);
                    });
                });

                // expand the selected range
                promise.always(function () {
                    self.changeActiveRange(SheetUtils.getBoundingRange(ranges[0], targetRange));
                });
            }

            // delete cells inside the selected range
            else if (count < 0) {

                // build the target range (part of the selected range, may be the entire selected range)
                if (leading) {
                    targetRange.end[addrIndex] = targetRange.start[addrIndex] - count - 1;
                } else {
                    targetRange.start[addrIndex] = targetRange.end[addrIndex] + count + 1;
                }

                // the target range must be editable (not the entire selected range)
                promise = this.areRangesEditable(targetRange);

                // clear the cell range, and update optimal row heights
                promise = promise.then(function () {
                    return undoManager.enterUndoGroup(function () {
                        // apply the 'clearCellRange' operation
                        activeSheetModel.clearCellRanges(targetRange);
                        // update automatic row height in the deleted range
                        // (bug 36773: return promise to keep undo group open)
                        return updateOptimalRowHeight(targetRange);
                    });
                });

                // update selection
                promise.always(function () {
                    // do not change selection after deleting the entire range
                    if (_.isEqual(ranges[0], targetRange)) { return; }
                    // select the unmodified part of the original range
                    targetRange = _.copy(ranges[0], true);
                    if (leading) {
                        targetRange.start[addrIndex] -= count;
                    } else {
                        targetRange.end[addrIndex] += count;
                    }
                    self.changeActiveRange(targetRange);
                });
            }

            // do nothing if no cells will be filled or cleared
            else {
                promise = $.when();
            }

            return this.yellOnPromise(promise);
        };

        /**
         * Inserts one or more formulas calculating subtotal results into or
         * next to the current selection.
         *
         * @param {String} funcName
         *  The name of the subtotal function to be inserted into the formulas.
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved when the
         *  server request used to query the subtotal formulas has returned.
         */
        this.insertAutoFormula = function (value) {

            var // the current selection
                selection = this.getSelection(),
                // the selected range
                range = selection.ranges[0],
                // whether the range consists of a single column
                singleCol = range.start[0] === range.end[0],
                // whether the range consists of a single row
                singleRow = range.start[1] === range.end[1],
                // the translated upper-case function name
                funcName = app.getLocalizedFunctionName(value);

            // returns the correct formula expression for the specified cell range
            function buildFormula(sourceRange) {
                return '=' + funcName + '(' + SheetUtils.getRangeName(sourceRange) + ')';
            }

            // returns a promise that resolves if all cells in the passed range are editable
            function ensureEditable(editRange) {
                return self.yellOnPromise(self.areRangesEditable(editRange));
            }

            // sends a server request and returns the promise that resolves with the cell content array
            function sendFormulaRequest() {
                Utils.info('ViewFuncMixin.insertAutoFormula(): requesting formulas for range ' + SheetUtils.getRangeName(range));

                var // the request data sent to the server
                    requestData = {
                        func: funcName,
                        sheet: activeSheet,
                        start: range.start,
                        end: range.end,
                        ref: selection.activeCell
                    };

                return app.sendQueryRequest('autoFormula', requestData, {
                    resultFilter: function (data) {
                        return Utils.getArrayOption(data, 'contents', undefined, true);
                    }
                })
                .done(function (response) {
                    Utils.info('ViewFuncMixin.insertAutoFormula(): request succeeded:', response);
                })
                .fail(function (response) {
                    Utils.error('ViewFuncMixin.insertAutoFormula(): request failed:', response);
                });
            }

            // selects the specified range, and keeps the current active cell
            function selectRange(start, end) {
                self.selectRange({ start: start, end: end}, { active: selection.activeCell });
            }

            // multiple selection not supported
            if (selection.ranges.length !== 1) {
                return $.Deferred().reject();
            }

            // single cell is selected
            if (singleCol && singleRow) {

                // query CalcEngine for a formula string and start in-place edit mode
                return sendFormulaRequest().done(function (contents) {
                    self.enterCellEditMode({ text: contents[0].formula });
                });
            }

            // multiple columns and rows are selected: fill formulas into the row below the range
            if (!singleCol && !singleRow) {

                // there must be space below the range
                if (range.end[1] >= docModel.getMaxRow()) { return $.Deferred().reject(); }

                var // fill formulas into the cells below the selected range
                    targetRange = { start: [range.start[0], range.end[1] + 1], end: [range.end[0], range.end[1] + 1] };

                // check that the range is completely unlocked, fill it with the formula
                return ensureEditable(targetRange).done(function () {
                    var sourceRange = { start: range.start, end: [range.start[0], range.end[1]] };
                    fillCellRanges(targetRange, buildFormula(sourceRange), undefined, { ref: targetRange.start, parse: true });
                    selectRange(range.start, targetRange.end);
                });
            }

            // single column or single row (with multiple cells): send server request if range is empty
            if (cellCollection.areRangesBlank(range)) {

                // check cell count first
                if (SheetUtils.getCellCount(range) > SheetUtils.MAX_FILL_CELL_COUNT) {
                    return this.yellOnPromise($.Deferred().reject('cells:overflow'));
                }

                // send server request, fill cells with one operation
                return sendFormulaRequest().done(function (contents) {

                    // convert the contents array according to the 'setCellContents' operation
                    contents = _.map(contents, function (cellData) {
                        var result = { value: cellData.formula };
                        if ('attrs' in cellData) { result.attrs = cellData.attrs; }
                        // put every array element into a single-element array for column selection
                        return singleCol ? [result] : result;
                    });

                    // apply the operation (changing selection not necessary)
                    setCellContents(range.start, singleRow ? [contents] : contents, { parse: true });
                });
            }

            // range contains any values: insert single formula into first empty cell
            return (function () {

                var // the cell address used to find the first empty cell
                    address = _.clone(range.end),
                    // modified array index in a cell address
                    addrIndex = singleRow ? 0 : 1,
                    // maximum index in the cell address (restrict loop to 1000 iterations)
                    maxIndex = Math.min(docModel.getMaxIndex(singleRow), address[addrIndex] + 1000);

                // JSHint requires to defined the function outside the while loop
                function createFormulaCell() {
                    var sourceRange = _.copy(range, true);
                    sourceRange.end[addrIndex] = Math.min(address[addrIndex] - 1, range.end[addrIndex]);
                    setSingleCellContents(address, buildFormula(sourceRange), undefined, { parse: true });
                    selectRange(range.start, address);
                }

                // find the first empty cell, starting at the last cell in the selected range
                while (address[addrIndex] <= maxIndex) {
                    if (cellCollection.isBlankCell(address)) {
                        return ensureEditable({ start: address, end: address }).done(createFormulaCell);
                    }
                    address[addrIndex] += 1;
                }

                return $.Deferred().reject();
            }());
        };

        /**
         * Merges or unmerges all cells of the current selection.
         *
         * @param {String} type
         *  The merge type. Must be one of:
         *  - 'merge': Merges entire ranges.
         *  - 'horizontal': Merges the single rows in all ranges.
         *  - 'vertical': Merges the single columns in all ranges.
         *  - 'unmerge': Removes all merged ranges covered by the selection.
         *  - 'toggle': Will be mapped to 'merge', if the selection does not
         *      cover any merged ranges, otherwise will be mapped to 'unmerge'.
         *
         * @returns {Boolean}
         *  Whether the ranges have been merged/unmerged successfully.
         */
        this.mergeRanges = function (type) {

            var // the selected ranges
                ranges = this.getSelectedRanges(),
                // the result of the operation
                result = null;

            // quit if current selection contains locked cells
            if (!this.requireEditableSelection()) { return false; }

            // toggle merge state if specified
            if (type === 'toggle') {
                type = mergeCollection.rangesOverlapAnyMergedRange(ranges) ? 'unmerge' : 'merge';
            }

            // send the 'mergeCells' operations
            result = activeSheetModel.mergeRanges(ranges, type);
            return this.yellOnResult(result);
        };

        // search operations --------------------------------------------------

        /**
         * Iterates through the current search results.
         *
         * @param {String} direction
         *  The search direction, either 'next', or 'prev'.
         *
         * @param {String} [query]
         *  The current string to be searched in the active sheet, as contained
         *  in the respective GUI element. If this value is different from the
         *  initial query string, a new search request will be sent to the
         *  server. If omitted, the current query string will be used again.
         *
         * @returns {ViewFuncMixin}
         *  A reference to this instance.
         */
        this.searchNext = function (direction, query) {

            var // the addresses of all cells containing the current search text
                addresses = getSearchResults(_.isString(query) ? query : getSearchQuery());

            // scroll to and select the next cell of the search results
            if (addresses) {
                switch (direction) {
                case 'prev':
                    selectSearchResult((searchSettings.index === 0) ? (addresses.length - 1) : (searchSettings.index - 1));
                    break;
                case 'next':
                    selectSearchResult((searchSettings.index + 1) % addresses.length);
                    break;
                }
            }

            return this;
        };

        /**
         * Replaces the value of the current (single) selection with the
         * replacement value.
         *
         * @param {String} query
         *  The current string to be searched in the active sheet, as contained
         *  in the respective GUI element. If this value is different from the
         *  initial query string, a new search request will be sent to the
         *  server.
         *
         * @param {String} replace
         *  The replacement text.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.all=false]
         *      If set to true, all occurrences of the search text in the
         *      active sheet will be replaced.
         *
         * @returns {ViewFuncMixin}
         *  A reference to this instance.
         */
        this.searchReplace = function (query, replace, options) {

            // no replace functionality in locked cells (TODO: replace in unprotected cells?)
            if (!this.requireUnlockedActiveSheet()) { return this; }

            // do nothing and return immediately if search query is empty
            if (query === '') {
                this.yell({ type: 'info', message: gt('Nothing to replace.') });
                return this;
            }

            // send server request to replace all occurrences in the active sheet
            if (Utils.getBooleanOption(options, 'all', false)) {
                replaceAll(query, replace);
            } else {
                replaceSingle(query, replace);
            }
            return this;
        };

        // table/filter operations --------------------------------------------

        /**
         * Returns the model of the table range located at the active cell of
         * the current selection. If the active cell is not located inside a
         * table range, tries to return the model of the anonymous table range
         * representing the auto filter of the active sheet.
         *
         * @returns {TableModel|Null}
         *  The model of the active table range; or null, if no table range has
         *  been found.
         */
        this.getActiveTable = function () {
            return tableCollection.findTable(this.getActiveCell()) || tableCollection.getTable('');
        };

        /**
         * Returns whether the active cell of the current selection is located
         * in a table range with activated filter/sorting. If the active cell
         * is not located in a table range, returns whether the active sheet
         * contains an activated auto filter (regardless of the selection).
         *
         * @returns
         *  Whether a table range with activated filter is selected.
         */
        this.hasTableFilter = function () {
            var tableModel = this.getActiveTable();
            return _.isObject(tableModel) && tableModel.isFilterActive();
        };

        /**
         * Enables or disables the filter in the selected table range of the
         * active sheet.
         *
         * @param {Boolean} state
         *  Whether to activate or deactivate the table filter. If the active
         *  cell of the current selection is not located inside a table range,
         *  the auto filter of the active sheet will be toggled.
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved when the
         *  table filter has been toggled successfully, or that will be
         *  rejected on any error.
         */
        this.toggleTableFilter = function (state) {

            // no filter manipulation in locked sheets
            if (!this.requireUnlockedActiveSheet()) { return $.Deferred().reject(); }

            var // the active table range
                tableModel = self.getActiveTable(),
                // the resulting promise
                promise = null;

            // toggle filter of a real table range (not auto filter)
            if (tableModel && (tableModel.getName() !== '')) {
                promise = activeSheetModel.changeTable(tableModel.getName(), { table: { filtered: state } });
                return this.yellOnPromise(promise);
            }

            // toggle the auto filter (insert table for content range of selected cell)
            return undoManager.enterUndoGroup(function () {

                // remove existing auto filter (regardless of passed state)
                promise = tableModel ? activeSheetModel.deleteTable('') : $.when();

                // create a new table for the auto filter
                if (state) {
                    promise = promise.then(function () {

                        var // the current selection ranges
                            ranges = self.getSelectedRanges(),
                            // the expanded first selected range
                            range = null,
                            // fixed attributes for the auto filter
                            attributes = { table: { headerRow: true, filtered: true } };

                        // auto filter cannot be created on a multi selection
                        if (ranges.length !== 1) {
                            return $.Deferred().reject('autofilter:multiselect');
                        }

                        // expand single cell to content range
                        range = activeSheetModel.getContentRangeForCell(ranges[0]);

                        // bug 36606: restrict range to used area of the sheet (range may become null)
                        range = SheetUtils.getIntersectionRange(range, cellCollection.getUsedRange());

                        // bug 36227: selection must not be entirely blank
                        if (!range || cellCollection.areRangesBlank(range)) {
                            return $.Deferred().reject('autofilter:blank');
                        }

                        // auto filter cannot be created on a multi selection
                        return activeSheetModel.insertTable('', range, attributes);
                    });
                }

                // show warning messages if needed (translate error codes for auto filter)
                return self.yellOnPromise(promise.then(null, function (errorCode) {
                    return errorCode.replace(/^table:/, 'autofilter:');
                }));
            });
        };

        /**
         * Returns whether the active table range can be refreshed (the active
         * sheet must contain an active table, and that table must contain any
         * active filter rules or sort options).
         *
         * @returns {Boolean}
         *  Whether the active table range can be refreshed.
         */
        this.canRefreshTable = function () {
            var tableModel = this.getActiveTable();
            return _.isObject(tableModel) && tableModel.isRefreshable();
        };

        /**
         * Refreshes the active table ranges (reapplies all filter rules and
         * sort options to the current cell contents in the table range).
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved when the
         *  table has been refreshed successfully, or that will be rejected on
         *  any error.
         */
        this.refreshTable = function () {

            var // the active table
                tableModel = this.getActiveTable(),
                // the result promise
                promise = null;

            // no filter manipulation in locked sheets
            if (tableModel && this.requireUnlockedActiveSheet()) {
                promise = activeSheetModel.refreshTable(tableModel.getName());
            } else {
                promise = $.Deferred().reject();
            }

            // show warning messages if needed
            return this.yellOnPromise(promise);
        };

        /**
         * Creates a discrete filter rule for the active filter column in the
         * active sheet (as described by the sheet view property
         * 'activeTableData').
         *
         * @param {Array} [entries]
         *  The values to be filtered for. All other values in the active
         *  filter column not contained in this array will be hidden. If
         *  omitted, the current filter rules will be emoved from the column.
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved when the
         *  filter has been applied successfully, or that will be rejected on
         *  any error.
         */
        this.applyDiscreteFilter = function (entries) {

            var // the active table
                tableData = this.getSheetViewAttribute('activeTableData'),
                // the filter attributes to be set for the column
                attributes = null,
                // the result promise
                promise = null;

            // no filter manipulation in locked sheets
            if (!tableData || !this.requireUnlockedActiveSheet()) { return $.Deferred().reject(); }

            // create discrete filter, or clear filter (for undefined entries)
            attributes = _.isArray(entries) ? { type: 'discrete', entries: entries } : { type: 'none' };

            // generate and apply the operations (including row and drawing operations)
            promise = activeSheetModel.changeTableColumn(tableData.tableModel.getName(), tableData.tableCol, { filter: attributes });

            // show warning messages if needed
            return this.yellOnPromise(promise);
        };

        // drawing operations -------------------------------------------------

        /**
         * Inserts a new drawing object into the active sheet, and selects it
         * afterwards.
         *
         * @param {String} type
         *  The type identifier of the drawing object.
         *
         * @param {Object} attributes
         *  Initial attributes for the drawing object, especially the anchor
         *  attributes specifying the position of the drawing in the sheet.
         *
         * @param {Function} [generatorCallback]
         *  A callback function that will be invoked after the initial
         *  'insertDrawing' operation has been generated. Allows to create
         *  additional operations for the drawing object, before all these
         *  operations will be applied at once. Receives the following
         *  parameters:
         *  (1) {SheetOperationsGenerator} generator
         *      The operations generator (already containing the initial
         *      'insertDrawing' operation).
         *  (2) {Number} sheet
         *      The zero-based index of the active sheet that will contain the
         *      drawing object.
         *  (3) {Array} position
         *      The position of the new drawing object in the sheet.
         *
         * @returns {ViewFuncMixin}
         *  A reference to this instance.
         */
        this.insertDrawing = function (type, attributes, generatorCallback) {

            var // the position of the new drawing object
                position = [drawingCollection.getModelCount()],
                // create the new drawing object in the active sheet
                result = activeSheetModel.insertDrawing(position, type, attributes, generatorCallback);

            // select the new drawing object
            if (result.length === 0) {
                this.selectDrawing(position);
            } else {
                this.removeDrawingSelection();
            }

            return this;
        };

        /**
         * Inserts a new image object into the active sheet, and selects it
         * afterwards.
         *
         * @param {Object} imageDescriptor
         *  The descriptor of the image object, containing the properties 'url'
         *  and 'name'.
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved when the
         *  image object has been created and inserted into the active sheet.
         */
        this.insertImage = function (imageDescriptor) {

            var // the URL of the image
                imageUrl = imageDescriptor.url,
                // create the image DOM node, wait for it to load
                createImagePromise = app.createImageNode(ImageUtils.getFileUrl(app, imageUrl));

            // on success, create the drawing object in the document
            createImagePromise.done(function (imgNode) {

                var // current width of the image, in 1/100 mm
                    width = Utils.convertLengthToHmm(imgNode[0].width, 'px'),
                    // current height of the image, in 1/100 mm
                    height = Utils.convertLengthToHmm(imgNode[0].height, 'px'),
                    // active cell as target position for the image
                    activeCell = self.getActiveCell();

                // restrict size to fixed limit
                if (width > height) {
                    if (width > MAX_IMAGE_SIZE) {
                        height = Math.round(height * MAX_IMAGE_SIZE / width);
                        width = MAX_IMAGE_SIZE;
                    }
                } else {
                    if (height > MAX_IMAGE_SIZE) {
                        width = Math.round(width * MAX_IMAGE_SIZE / height);
                        height = MAX_IMAGE_SIZE;
                    }
                }

                self.insertDrawing('image', {
                    drawing: {
                        anchorType: 'oneCell',
                        startCol: activeCell[0],
                        startRow: activeCell[1],
                        width: width,
                        height: height,
                        name: imageDescriptor.name
                    },
                    image: {
                        imageUrl: imageUrl
                    }
                });

                app.destroyImageNodes(imgNode);
            });

            return createImagePromise;
        };

        /**
         * Deletes all drawings currently selected in the active sheet.
         *
         * @returns {ViewFuncMixin}
         *  A reference to this instance.
         */
        this.deleteDrawings = function () {

            if (this.isSheetLocked()) {
                self.yell({ type: 'info', message: gt('Objects on a protected sheet cannot be deleted.') });
                return this;
            }

            var // the selected drawing objects
                positions = this.getSelectedDrawings();

            // back to cell selection, delete all drawings previously selected
            this.removeDrawingSelection();
            activeSheetModel.deleteDrawings(positions);

            return this;
        };

        /**
         * Setting drawing attributes.
         *
         * @param {Object} [attributes]
         *  A drawing attribute set, keyed by the supported attribute family
         *  'drawing'. Omitted attributes will not be changed. Attributes set
         *  to the value null will be removed from the drawing.
         *
         * @param {Array} [pos]
         *  An array with at least one drawing address. If not specified, the
         *  selected drawings are used.
         *
         * @returns {ViewFuncMixin}
         *  A reference to this instance.
         */
        this.setDrawingAttributes = function (attributes, pos) {

            var // the selected drawing objects
                drawings = pos || this.getSelectedDrawings(),
                // the generator for the operations
                generator = activeSheetModel.createOperationsGenerator();

            // send the 'setDrawingAttributes' operations
            _.each(drawings, function (position) {
                generator.generateDrawingOperation(Operations.SET_DRAWING_ATTRIBUTES, position, { attrs: attributes });
            });
            docModel.applyOperations(generator.getOperations());

            return this;
        };

        // initialization -----------------------------------------------------

        // initialize class members
        app.onInit(function () {

            // resolve reference to document model objects
            docModel = app.getModel();
            undoManager = docModel.getUndoManager();
            documentStyles = docModel.getDocumentStyles();
            fontCollection = documentStyles.getFontCollection();
            styleCollection = documentStyles.getStyleCollection('cell');
            numberFormatter = docModel.getNumberFormatter();

            // initialize selection settings, used until first real update
            importSelectionSettings({});
            activeCellSettings = _.copy(CellCollection.DEFAULT_CELL_DATA, true);
            activeCellSettings.explicit = {};
            activeCellSettings.attributes = styleCollection.getStyleSheetAttributes(styleCollection.getDefaultStyleId());

            // initialize sheet-dependent class members according to the active sheet
            self.on('change:activesheet', changeActiveSheetHandler);

            // refresh selection settings after changing the selection
            self.on('change:selection', function () { updateSelectionSettings(); });

            // update selection settings after any changes in the active sheet (debounced in case of mass operations, e.g. undo)
            self.on('sheet:triggered', self.createDebouncedMethod($.noop, updateSelectionSettings, { delay: 20, maxDelay: 500 }));

            // application notifies changed contents/results of cells
            self.listenTo(app, 'docs:update:cells', updateNotificationHandler);

            // drop search results when closing the search/replace tool pane
            self.listenTo(app.getWindow(), 'search:close', clearSearchResults);

            if (Config.DEBUG) { initDebugOperations(); }
        });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            app = self = docModel = undoManager = null;
            documentStyles = fontCollection = styleCollection = numberFormatter = null;
            activeSheetModel = colCollection = rowCollection = mergeCollection = cellCollection = null;
            tableCollection = drawingCollection = null;
        });

    } // class ViewFuncMixin

    // exports ================================================================

    return ViewFuncMixin;

});
