/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Michael Nimz <michael.nimz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/mixin/sortmixin', [
    'io.ox/office/spreadsheet/utils/config',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/view/popup/sortlayermenu',
    'gettext!io.ox/office/spreadsheet'
], function (Config, SheetUtils, SortLayerMenu, gt) {

    'use strict';

    // mix-in class SortMixin =================================================

    /**
     * Mix-in class for the class SpreadsheetView that provides extensions for
     * sorting selected ranges.
     *
     * @constructor
     *
     * @param {SpreadsheetApplication} app
     *  The spreadsheet application instance.
     */
    function SortMixin(app) {

        var // self reference (the spreadsheet view)
            self = this,

            // custom sort menu
            sortLayerMenu = new SortLayerMenu(app),

            // normal (toggle-)buttons, or custom sort
            type = null,

            // selection
            selection = null,

            // maximum number of allowed columns/rows
            maxColRow = 2000,

            // selected ranges
            ranges = null,

            // first selected range
            range = null,

            // range after expanding
            newRange = null,

            // state of sorting
            sortObj = {
                custom: {
                    state: null,
                    order: 'ascending',
                    colrow: null,
                    direction: 'vertical',
                    hasheadline: false,
                    preventReset: false,
                    preventClose: false,
                    preventDialog: false
                },
                standard: {
                    state: null,
                    order: 'ascending',
                    colrow: null,
                    direction: 'vertical',
                    hasheadline: 'auto',
                    preventReset: false,
                    preventClose: false,
                    preventDialog: false
                }
            };

        // private methods ----------------------------------------------------

        /**
          * Sends the sort-ajax-request to the backend
          */
        function sendSortRequest() {
            var options = sortObj[type],
                arrDirections = ['vertical', 'horizontal'],
                directionIndex = arrDirections.indexOf(options.direction),
                colRowIndex = options.colrow || selection.activeCell[directionIndex];

            app.sendActionRequest('sort', {
                locale:             Config.LOCALE,
                sheet:              self.getActiveSheet(),
                start:              range.start,
                end:                range.end,
                // options
                headers:            options.hasheadline,
                direction:          options.direction,
                // order-level
                settings: [
                    {
                        colRowIndex:    colRowIndex,
                        order:          options.order
                    }
                ]
            });
        }

        /**
          * Expands the selection if needed
          *
          * @return {Boolean}
          */
        function expandSelection() {
            self.setSortOption('preventClose', true, type);
            self.setSortOption('preventReset', true, type);

            // yell, if the new range has too many columns/rows included
            if (SheetUtils.getColCount(newRange) > SheetUtils.MAX_CHANGE_COLS_COUNT) {
                return self.yellOnResult('cols:overflow');
            }
            if (SheetUtils.getRowCount(newRange) > SheetUtils.MAX_CHANGE_ROWS_COUNT) {
                return self.yellOnResult('rows:overflow');
            }

            // select the new range
            self.selectRange(newRange, { active: selection.activeCell });

            // get new/current selection
            selection = self.getSelection();

            // the selected range
            range = selection.ranges[0];

            return true;
        }

        /**
         * checks whether the selection is sortable or not
         *
         * @return {Boolean}
         */
        function possibleSelection() {

            // get current selection
            selection = self.getSelection();

            // the selected range(s)
            ranges = selection.ranges;
            range = selection.ranges[0];

            // get out here, sheet is protected
            if (!self.requireUnlockedActiveSheet()) {
                return false;
            }

            // get out here, if more than one range is selected
            if (ranges.length > 1) {
                self.yellOnResult('sort:multiselect');
                return false;
            }

            // get out here, if selection is blank (nothing to sort)
            if (self.getCellCollection().areRangesBlank(range)) {
                self.yellOnResult('sort:blank');
                return false;
            }

            if ((range.end[0] - range.start[0]) > maxColRow || (range.end[1] - range.start[1]) > maxColRow) {
                self.yell({ type: 'info', message: gt('The command cannot be performed on more than 2000 columns or rows') });
                return false;
            }
            return true;
        }

        /**
         * Check if the selection is expandable or actually must expand.
         */
        function checkSelectionRange() {
            newRange = self.getCellCollection().findContentRange(range);

            // the selection is only within one row/col. Ask the user whether expand selection or not.
            if ((range.start[1] === range.end[1] || range.start[0] === range.end[0]) && !_.isEqual(range.start, range.end) && !_.isEqual(newRange, range)) {
                // ask the user to expand the current selection
                return self.showExpandSelectionForSortingDialog()
                    .done(function(){
                        // expand the selection automatically to current used rectangle
                        expandSelection();
                    })
                    .always(function () {
                        if (type === 'custom') {
                            sortLayerMenu.show();
                        } else {
                            sendSortRequest();
                        }
                    });
            }

            // only one cell is selected. Expand selection automatically (don't ask the user).
            if (_.isEqual(range.start, range.end) && !_.isEqual(newRange, range)) {
                // expand the selection automatically to current used rectangle
                return expandSelection();
            }

            return true;
        }

        // methods ------------------------------------------------------------

        /**
          * Provides the current sort-state for (e.g.) the toggle-button
          */
        this.getSortState = function (sorttype) {
            var type = sorttype || 'standard',
                states = {
                    'ascending': 'descending',
                    'descending': 'ascending'
                };

            if (this.getSortOption('state', type) === null) {
                return 'ascending';
            }

            return states[this.getSortOption('order', type)];
        };

        /**
          * Provides various sort options by sort-type
          *
          * @param {String} key
          *  The index-key of the option which should be provided
          *
          * @param {String} sorttype
          *  The type on which currently be sorted ('default', 'custom')
          */
        this.getSortOption = function (key, sorttype) {
            var type = sorttype || 'default';
            return _.has(sortObj[type], key)?sortObj[type][key]:false;
        };

        /**
          * Set various sort options by sort-type
          *
          * @param {String} key
          *  The index-key of the option which should be set
          *
          * @param {Any} value
          *  The value of the option which should be set
          *
          * @param {String} sorttype
          *  The type on which currently be sorted ('default', 'custom')
          */
        this.setSortOption = function (key, value, sorttype) {
            var type = sorttype || 'standard';
            sortObj[type][key] = value;
            return this;
        };

        /**
          * Resets all sort-options by sort-type
          *
          * @param {String} sorttype
          *  The type on which the reset should be executed
          */
        this.resetSortOptions = function (sorttype) {
            // get out here, if resetting is prevented
            if (_.has(sortObj, sorttype) && this.getSortOption('preventReset', sorttype)) {
                return this.setSortOption('preventReset', false, sorttype);

            // if it not exists, create option-object
            } else if (!_.has(sortObj, sorttype)) {
                sortObj[sorttype] = {};
            }

            var defaults = {
                    custom: {
                        state: null,
                        order: 'ascending',
                        colrow: null,
                        direction: 'vertical',
                        hasheadline: false,
                        preventReset: false,
                        preventClose: false,
                        preventDialog: false
                    },
                    standard: {
                        state: null,
                        order: 'ascending',
                        colrow: null,
                        direction: 'vertical',
                        hasheadline: 'auto',
                        preventReset: false,
                        preventClose: false,
                        preventDialog: false
                    }
                };
            _.each(defaults[sorttype], function(value, key){
                self.setSortOption(key, value, sorttype);
            });

            app.getController().update();
        };


        /**
         * Checks the sort-environment. Decides, whether the selection should expand, or in doubt, asks the user.
         *
         * @returns {jQuery.Promise}
         */
        this.showSortExtensionDialog = function () {
            type = 'custom';
            if (possibleSelection() && checkSelectionRange() === true) {
                sortLayerMenu.show();
            }
        };

        /**
          * Prepares and executes the sorting
          *
          * @param {String} sorttype
          *  The type on which currently be sorted ('standard', 'custom')
          *
          * @param {String} order
          *  The order-method ('ascending', 'descending', ...)
          */
        this.sortCellRange = function (sorttype, order) {
            // type of sorting (standard || custom)
            type = sorttype;

            // set defaults for given type
            switch (type) {
                case 'custom':
                    if (this.getSortState(type) === null) {
                        this.setSortOption('hasheadline', false, type);
                    }
                    this.setSortOption('state', order, type);
                    break;

                default:
                    // if the toggle-button is clicked
                    if (order === 'toggle') {
                        // get last state
                        order = this.getSortState();
                        // and toggle it
                        this.setSortOption('state', order, type);

                    // if sorting is directly selected in DropDown-Menu (asc or desc)
                    } else {
                        // reset (toggle-button-)state
                        this.setSortOption('state', null, type);
                    }

                    // set selected order-direction
                    this.setSortOption('order', order, type);
                    // reset headline-state to automatic detection
                    this.setSortOption('hasheadline', 'auto', type);

                    if (!possibleSelection()) { return $.when(); }

                    if (checkSelectionRange() !== true) { return $.when(); }

                    break;
            }

            // send the sort request to the server
            sendSortRequest();

            if (type === 'custom') {
                self.resetSortOptions('custom');
                sortLayerMenu.hide();
            }
        };

        // initialization -----------------------------------------------------

        // reset the state, if the selection changes (sort settings won't be saved)
        this.on('change:selection', function () {
            self.resetSortOptions('standard');
        });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            sortLayerMenu.destroy();
            app = self = sortLayerMenu = type = selection = ranges = range = newRange = sortObj = null;
        });

    } // class SortMixin

    // exports ================================================================

    return SortMixin;

});
