/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/control/cellstylepicker', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/forms',
    'io.ox/office/editframework/view/control/stylesheetpicker',
    'io.ox/office/spreadsheet/utils/paneutils',
    'gettext!io.ox/office/spreadsheet'
], function (Utils, Forms, StyleSheetPicker, PaneUtils, gt) {

    'use strict';

    // class CellStylePicker ==================================================

    /**
     * A drop-down menu for cell style sheets.
     *
     * @constructor
     *
     * @extends StyleSheetPicker
     */
    function CellStylePicker(app) {

        var // the collection of cell styles in the document
            styleCollection = null;

        // base constructor ---------------------------------------------------

        StyleSheetPicker.call(this, app, 'cell', {
            width: 150,
            icon: 'docs-table-style',
            tooltip: /*#. tool tip: predefined styles for spreadsheet cells */ gt('Cell style'),
            gridColumns: 6,
            i18nModulePath: 'io.ox/office/spreadsheet/resource/cellstylenames',
            sections: ['markup', 'headings', 'themes', 'hidden'],
            smallerVersion: {
                css: { width: 56 },
                hideLabel: true
            }
        });

        // private methods ----------------------------------------------------

        /**
         * Updates the passed button element in the drop-down menu.
         */
        function createItemHandler(event, buttonNode, styleId) {

            var // the merged attributes of the passed cell style sheet
                attributes = styleCollection.getStyleSheetAttributes(styleId),
                // the character attributes
                charAttrs = attributes.character,
                // the cell attributes
                cellAttrs = attributes.cell,
                // the caption element to be formatted
                captionNode = Forms.getCaptionNode(buttonNode),
                // an additional border node as overlay (to prevent different line height in caption)
                borderMarkup = null;

            // TODO: use canvas for complex border styles
            borderMarkup = '<div class="borders" style="';
            borderMarkup += 'background-color:' + styleCollection.getCssColor(cellAttrs.fillColor, 'fill') + ';';
            borderMarkup += 'border-top:' + styleCollection.getCssBorder(cellAttrs.borderTop, { preview: true }) + ';';
            borderMarkup += 'border-bottom:' + styleCollection.getCssBorder(cellAttrs.borderBottom, { preview: true }) + ';';
            borderMarkup += 'border-left:' + styleCollection.getCssBorder(cellAttrs.borderLeft, { preview: true }) + ';';
            borderMarkup += 'border-right:' + styleCollection.getCssBorder(cellAttrs.borderRight, { preview: true }) + ';';
            borderMarkup += '"></div>';
            buttonNode.prepend(borderMarkup);

            // add relevant character and cell formatting
            captionNode.css({
                fontFamily: styleCollection.getCssFontFamily(charAttrs.fontName),
                fontSize: Utils.minMax(Utils.round(10 + (charAttrs.fontSize - 10) / 1.5, 0.1), 6, 22) + 'pt',
                fontWeight: charAttrs.bold ? 'bold' : 'normal',
                fontStyle: charAttrs.italic ? 'italic' : 'normal',
                textDecoration: styleCollection.getCssTextDecoration(charAttrs),
                color: styleCollection.getCssTextColor(charAttrs.color, [cellAttrs.fillColor]),
                textAlign: PaneUtils.getCssTextAlignment({ result: 'a', attributes: attributes })
            });
        }

        // initialization -----------------------------------------------------

        // initialize class members when application is ready
        app.onInit(function () {
            styleCollection = app.getModel().getStyleCollection('cell');
        });

        // add visible formatting of the cell styles to the list item buttons
        this.getMenu().on('create:item', createItemHandler);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            app = styleCollection = null;
        });

    } // class CellStylePicker

    // exports ================================================================

    // derive this class from class StyleSheetPicker
    return StyleSheetPicker.extend({ constructor: CellStylePicker });

});
