/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/control/activesheetgroup', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/forms',
    'io.ox/office/tk/control/radiogroup',
    'io.ox/office/spreadsheet/view/popup/sheetcontextmenu',
    'gettext!io.ox/office/spreadsheet'
], function (Utils, Forms, RadioGroup, SheetContextMenu, gt) {

    'use strict';

    // class ActiveSheetGroup =================================================

    /**
     * The selector for the current active sheet, as radio group (all sheet
     * tabs are visible at the same time).
     *
     * @constructor
     *
     * @extends RadioGroup
     */
    function ActiveSheetGroup(app) {

        var // self reference
            self = this,

            // get application model and view
            docModel = null,
            docView = null,

            // the context menu with additional sheet actions
            contextMenu = null,

            // the scroll button nodes
            prevButton = $(Forms.createButtonMarkup({ icon: 'fa-angle-left', tooltip: gt('Scroll to the previous sheet') })).addClass('scroll-button'),
            nextButton = $(Forms.createButtonMarkup({ icon: 'fa-angle-right', tooltip: gt('Scroll to the next sheet') })).addClass('scroll-button'),

            // index of the first and last sheet tab button currently visible
            firstIndex = 0,
            lastIndex = 0,
            scrollable = false,

            // the sheet tab button currently dragged
            dndButtonNode = null,

            // tracker node shown while dragging a sheet button
            dndTrackerNode = $('<div id="io-ox-office-spreadsheet-sheet-dnd-tracker"><div class="caret-icon"></div></div>'),

            // helper node inside the tracker node containing the sheet name
            dndSheetNode = $('<span class="sheet-name">').appendTo(dndTrackerNode),

            // index of the sheet tab button is currently dragged, and the target button index
            dndStartIndex = -1,
            dndTargetIndex = -1;

        // base constructor ---------------------------------------------------

        RadioGroup.call(this, { classes: 'active-sheet-group', role: 'tablist' });

        // private methods ----------------------------------------------------

        /**
         * Shows all sheet buttons and removes explicit width, and hides the
         * scroll buttons.
         */
        function showAllSheetButtons() {
            var buttonNodes = self.getOptionButtons();
            Forms.showNodes(buttonNodes, true);
            buttonNodes.css('width', '');
            Forms.showNodes(prevButton, false);
            Forms.showNodes(nextButton, false);
            nextButton.css({ position: '', right: '' });
            scrollable = false;
        }

        /**
         * Shows the sheet buttons starting at the start sheet index currently
         * cached.
         */
        function renderSheetButtons() {

            var // the root node of this group
                groupNode = self.getNode(),
                // the inner width of the group (space available for buttons)
                groupWidth = groupNode.width(),
                // all sheet tab buttons
                sheetButtons = self.getOptionButtons(),
                // the width of all sheet buttons
                buttonsWidth = 0,
                // relative start/end position of all sheet buttons
                buttonPositions = null,
                // total width of all buttons, including scroll buttons
                totalWidth = 0,
                // remaining width available in the group node for sheet buttons
                remainingWidth = 0,
                // maximum index of button that can be shown at the left border (without gap to right border)
                maxIndex = 0;

            // do nothing if there are no sheets available (yet)
            if (sheetButtons.length === 0) { return; }

            // initialize all sheet buttons, hide scroll buttons
            showAllSheetButtons();

            // get total width of all buttons (bug 35869: round down, browsers may report slightly oversized width due to text nodes)
            buttonsWidth = Math.floor(sheetButtons.last().offset().left + sheetButtons.last().outerWidth() - sheetButtons.first().offset().left);
            if (buttonsWidth <= groupWidth) { return; }

            // collect position information of all buttons (performance: once for all buttons, before manipulating DOM again)
            var leadingOffset = sheetButtons.first().offset().left;
            buttonPositions = sheetButtons.map(function () {
                var buttonOffset = $(this).offset().left - leadingOffset;
                return {
                    offset: buttonOffset,
                    leadingWidth: buttonOffset + $(this).outerWidth(),
                    trailingWidth: buttonsWidth - buttonOffset
                };
            });

            // do not show scroll buttons, if there is only one visible sheet
            if (sheetButtons.length > 1) {

                // show scroll buttons
                Forms.showNodes(prevButton, true);
                Forms.showNodes(nextButton, true);
                scrollable = true;

                // get total width of all buttons, including scroll buttons, and inner remaining width
                totalWidth = nextButton.offset().left + nextButton.outerWidth() - prevButton.offset().left;
                remainingWidth = groupWidth - (totalWidth - buttonsWidth);

                // hide leading sheet tabs
                maxIndex = Utils.findFirstIndex(buttonPositions, function (position) { return position.trailingWidth <= remainingWidth; }, { sorted: true });
                firstIndex = Utils.minMax(firstIndex, 0, Math.max(0, maxIndex));
                Forms.showNodes(sheetButtons.slice(0, firstIndex), false);

                // adjust button positions relative to first visible button
                var relOffset = buttonPositions[firstIndex].offset;
                _.each(buttonPositions, function (position) {
                    position.offset -= relOffset;
                    position.leadingWidth -= relOffset;
                });

                // hide trailing sheet tabs
                maxIndex = Utils.findFirstIndex(buttonPositions, function (position) { return position.leadingWidth > remainingWidth; }, { sorted: true });
                lastIndex = ((maxIndex < 0) ? sheetButtons.length : maxIndex) - 1;
                Forms.showNodes(sheetButtons.slice(lastIndex + 1), false);

                // enable/disable scroll buttons
                Forms.enableNodes(prevButton, firstIndex > 0);
                Forms.enableNodes(nextButton, lastIndex + 1 < sheetButtons.length);

            } else {
                // single sheet: initialize indexes
                firstIndex = lastIndex = 0;
                totalWidth = buttonsWidth;
                remainingWidth = groupWidth;
            }

            // set fixed position for the right scroll button
            nextButton.css({ position: 'absolute', right: 0 });

            // shrink single visible sheet button, if still oversized
            var sheetButton = sheetButtons.eq(firstIndex);
            if ((firstIndex === lastIndex) && (remainingWidth < sheetButton.outerWidth())) {
                sheetButton.width(remainingWidth - (sheetButton.outerWidth() - sheetButton.width()));
                return;
            }

            // add the first hidden sheet button after the last visible, if there is some space left
            if (lastIndex + 1 === buttonPositions.length) { return; }
            remainingWidth -= buttonPositions[lastIndex + 1].offset;
            if (remainingWidth >= 30) {
                sheetButton = sheetButtons.eq(lastIndex + 1);
                Forms.showNodes(sheetButton, true);
                sheetButton.width(remainingWidth - (sheetButton.outerWidth() - sheetButton.width()));
            }
        }

        /**
         * Called every time the value of the control (the index of the active
         * sheet) has been changed.
         */
        function updateHandler(index) {

            // scroll backwards, if the active sheet is located before the first visible;
            // or scroll forwards, if the active sheet is located after the last visible
            if ((index < firstIndex) || (index > lastIndex)) {
                firstIndex = index;
                renderSheetButtons();
            }
        }

        /**
         * Handles double click events on sheet tab buttons and initiates
         * renaming the sheet.
         */
        function doubleClickHandler() {
            app.getController().executeItem('sheet/rename/dialog');
        }

        /**
         * Scrolls to the left to make at least one preceding sheet tab button
         * visible.
         */
        function scrollToPrev() {

            // do not try to scroll, if all buttons are visible
            if (!scrollable) { return; }

            var // index of the old last visible sheet
                oldLastIndex = lastIndex;

            // move to previous sheets until the index of the last sheet has changed
            while ((firstIndex > 0) && (oldLastIndex === lastIndex)) {
                firstIndex -= 1;
                renderSheetButtons();
            }
        }

        /**
         * Scrolls to the right to make at least one following sheet tab button
         * visible.
         */
        function scrollToNext() {

            // do not try to scroll, if all buttons are visible
            if (!scrollable) { return; }

            var // all sheet tab buttons
                sheetButtons = self.getOptionButtons(),
                // index of the old last visible sheet
                oldLastIndex = lastIndex;

            // move to next sheets until the index of the last sheet has changed
            while ((0 <= lastIndex) && (lastIndex + 1 < sheetButtons.length) && (oldLastIndex === lastIndex)) {
                firstIndex += 1;
                renderSheetButtons();
            }
        }

        /**
         * Handles tracking on the scroll-back button. Stops event propagation,
         * to not interfere with sheet move tracking.
         */
        function prevButtonTrackingHandler(event) {
            scrollToPrev();
            event.stopPropagation();
        }

        /**
         * Handles tracking on the scroll-forward button. Stops event
         * propagation, to not interfere with sheet move tracking.
         */
        function nextButtonTrackingHandler(event) {
            scrollToNext();
            event.stopPropagation();
        }

        /**
         * Handles end of tracking on the scroll buttons. Moves the browser
         * focus back to the application, and stops event propagation (to not
         * interfere with sheet move tracking).
         */
        function buttonTrackingEndHandler(event) {
            event.stopPropagation();
            self.triggerCancel({ sourceEvent: event });
        }

        /**
         * Returns whether the dragged sheet tab button will currently drop to
         * a new position.
         *
         * @returns {Boolean}
         *  Whether the dragged sheet tab button will currently drop to a new
         *  position.
         */
        function isValidDndTargetIndex() {
            // dragging directly behind the start button will not move the sheet
            return (dndTargetIndex < dndStartIndex) || (dndStartIndex + 1 < dndTargetIndex);
        }

        /**
         * Starts drag&drop tracking of a sheet tab button.
         */
        function dndTrackingStartHandler(event) {

            // do nothing in read-only mode
            if (!docModel.getEditMode()) {
                $.cancelTracking();
                return;
            }

            // the sheet tab button clicked for tracking
            dndButtonNode = $(event.target).closest(Forms.OPTION_BUTTON_SELECTOR);

            // remember button index as long as tracking lasts
            dndStartIndex = dndTargetIndex = self.getOptionButtons().index(dndButtonNode);

            // prepare the tracking node shown at the current insertion position
            dndSheetNode.text(Forms.getCaptionText(dndButtonNode));
            $(document.body).append(dndTrackerNode.hide());
            dndButtonNode.addClass('tracking-active');
        }

        /**
         * Updates drag&drop tracking of a sheet tab button.
         */
        function dndTrackingMoveHandler(event) {

            var // all sheet tab buttons
                buttonNodes = self.getOptionButtons(),
                // all visible sheet tab buttons
                visibleNodes = Forms.filterVisibleNodes(buttonNodes),
                // absolute position and size of all visible sheet buttons
                buttonRects = _.map(visibleNodes.get(), Utils.getNodePositionInPage),
                // search rectangle of hovered sheet tab button in array of visible button rectangles
                rectIndex = Utils.findLastIndex(buttonRects, function (data) {
                    return (data.left + data.width / 2) <= event.pageX;
                }, { sorted: true }) + 1;

            // calculate new insertion index (add number of leading hidden buttons)
            dndTargetIndex = buttonNodes.index(visibleNodes[0]) + rectIndex;

            // hide tracking icon on invalid position (next to start button)
            if (!isValidDndTargetIndex()) {
                dndTrackerNode.hide();
                return;
            }

            var // whether to show the tracking position after the last visible button
                trailing = rectIndex >= buttonRects.length,
                // position of the sheet tab button following the tracking position
                buttonRect = buttonRects[rectIndex] || _.last(buttonRects),
                // exact insertion position (center of the tracker node), in pixels
                offsetX = buttonRect.left + (trailing ? buttonRect.width : 0);

            // update position of the tracking icon
            dndTrackerNode.show().css({
                left: Math.round(offsetX - dndTrackerNode.width() / 2),
                top: self.getNode().offset().top - dndTrackerNode.height() + 2
            });
        }

        /**
         * Automatically scrolls the buttons while dragging a sheet tab button.
         */
        function dndTrackingScrollHandler(event) {

            // scroll buttons into the specified direction if possible
            if (event.scrollX < 0) {
                scrollToPrev();
            } else if (event.scrollX > 0) {
                scrollToNext();
            }

            // update position of the tracker node
            dndTrackingMoveHandler(event);
        }

        /**
         * Finalizes drag&drop tracking of a sheet tab button. Moves the sheet
         * associated to the dragged button in the document, and moves the
         * browser focus back to the application.
         */
        function dndTrackingEndHandler() {

            // leave cell edit mode before manipulating the sheet order
            if (isValidDndTargetIndex() && docView.leaveCellEditMode('auto', { validate: true })) {

                // adjust drop position, if target position follows source position
                if (dndStartIndex < dndTargetIndex) { dndTargetIndex -= 1; }

                var // the sheet tab button nodes
                    buttonNodes = self.getOptionButtons(),
                    // source sheet index for document operation
                    fromSheet = Utils.getElementAttributeAsInteger(buttonNodes[dndStartIndex], 'data-index', -1),
                    // target sheet index for document operation
                    toSheet = Utils.getElementAttributeAsInteger(buttonNodes[dndTargetIndex], 'data-index', -1);

                // change sheet order in the document
                docModel.moveSheet(fromSheet, toSheet);
            }

            // remove tracking icon, focus back to document
            dndTrackingCancelHandler();
        }

        /**
         * Cancels drag&drop tracking of a sheet tab button. Moves the browser
         * focus back to the application, without actually moving a sheet.
         */
        function dndTrackingCancelHandler() {
            dndTrackerNode.remove();
            if (dndButtonNode) {
                dndButtonNode.removeClass('tracking-active');
                dndButtonNode = null;
            }
            docView.grabFocus();
        }

        /**
         * Recreates all sheet tabs in this radio group.
         */
        var fillList = this.createDebouncedMethod($.noop, function () {

            // create buttons for all visible sheets
            self.clearOptionButtons();
            docView.iterateVisibleSheets(function (sheetModel, index, sheetName) {
                self.createOptionButton(index, {
                    icon: sheetModel.isLocked() ? 'fa-lock' : null,
                    iconClasses: 'small-icon',
                    label: _.noI18n(sheetName),
                    labelStyle: 'white-space:pre;',
                    tooltip: _.noI18n(sheetName),
                    attributes: { role: 'tab', 'data-index': sheetModel.getIndex() }
                });
            });

            // move trailing scroll button to the end
            self.addChildNodes(nextButton);

            // explicitly notify listeners, needed for reliable layout
            self.trigger('change:sheets');
        });

        // methods ------------------------------------------------------------

        /**
         * Sets this control to automatic width, shows all sheet tab buttons,
         * and hides the scroll buttons.
         *
         * @returns {ActiveSheetGroup}
         *  A reference to this instance.
         */
        this.setFullWidth = function () {
            this.getNode().css('width', '');
            showAllSheetButtons();
            return this;
        };

        /**
         * Sets this control to a fixed width. Performs further initialization
         * according to the available space: shows/hides the scroll buttons,
         * and draws the available sheet tabs according to the last cached
         * scroll position.
         *
         * @param {Number} width
         *  The new outer width of this control, in pixels.
         *
         * @returns {ActiveSheetGroup}
         *  A reference to this instance.
         */
        this.setWidth = function (width) {
            this.getNode().width(width);
            renderSheetButtons();
            return this;
        };

        // initialization -----------------------------------------------------

        // initialization when application is ready (model/view available)
        app.onInit(function () {

            // get application model and view instances
            docModel = app.getModel();
            docView = app.getView();

            // register handler early for fast preview during import
            self.listenTo(docView, 'change:sheets', fillList);

            // rebuild the list of available sheets after it has been changed
            self.listenTo(app.getImportPromise(), 'done', function () {
                fillList();
                self.listenTo(docModel, 'change:sheet:attributes', fillList); // e.g. sheet protection
            });

            // cancel button drag&drop when losing edit rights
            self.listenTo(docModel, 'change:editmode', function (event, editMode) {
                if (!editMode && dndButtonNode) {
                    $.cancelTracking();
                }
            });

            // create the context menu
            contextMenu = new SheetContextMenu(app, self.getNode());
        });

        // the update handler scrolls to the active sheet, after it has been changed
        this.registerUpdateHandler(updateHandler);

        // add scroll buttons to the group node
        this.addChildNodes(prevButton, nextButton);

        // double click shows rename dialog (TODO: in-place editing)
        this.getNode().on('dblclick', Forms.OPTION_BUTTON_SELECTOR, doubleClickHandler);

        // handle tracking events of scroll buttons
        prevButton.enableTracking({ autoRepeat: true })
            .on('tracking:start tracking:repeat', prevButtonTrackingHandler)
            .on('tracking:move', function (event) { event.stopPropagation(); })
            .on('tracking:end tracking:cancel', buttonTrackingEndHandler);
        nextButton.enableTracking({ autoRepeat: true })
            .on('tracking:start tracking:repeat', nextButtonTrackingHandler)
            .on('tracking:move', function (event) { event.stopPropagation(); })
            .on('tracking:end tracking:cancel', buttonTrackingEndHandler);

        // drag sheet buttons to a new position (not on touch devices)
        if (!Modernizr.touch) {
            this.getNode()
                .enableTracking({ selector: Forms.OPTION_BUTTON_SELECTOR, cursor: 'ew-resize', autoScroll: 'horizontal', scrollInterval: 500, borderMargin: -30 })
                .on('tracking:start', dndTrackingStartHandler)
                .on('tracking:move', dndTrackingMoveHandler)
                .on('tracking:scroll', dndTrackingScrollHandler)
                .on('tracking:end', dndTrackingEndHandler)
                .on('tracking:cancel', dndTrackingCancelHandler);
        }

        // destroy all class members on destruction
        this.registerDestructor(function () {
            contextMenu.destroy();
            app = self = docModel = docView = contextMenu = null;
            prevButton = nextButton = dndTrackerNode = dndSheetNode = null;
        });

    } // class ActiveSheetGroup

    // exports ================================================================

    // derive this class from class RadioGroup
    return RadioGroup.extend({ constructor: ActiveSheetGroup });

});
