/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Mario Schroeder <mario.schroeder@open-xchange.com>
 */
define('io.ox/office/spreadsheet/utils/clipboard', [
    'io.ox/office/tk/utils',
    'io.ox/office/editframework/utils/color',
    'io.ox/office/editframework/utils/hyperlinkutils',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/model/formula/tokenarray'
], function (Utils, Color, HyperlinkUtils, SheetUtils, TokenArray) {

    'use strict';


    var // XML name spaces for Office and Excel custom tags
        EXCEL_NAMESPACES = 'xmlns:o="urn:schemas-microsoft-com:office:office" xmlns:x="urn:schemas-microsoft-com:office:excel" xmlns="http://www.w3.org/TR/REC-html40"',

        // the HTML header
        HTML_HEADER = '<head><meta charset="utf-8"></head>',

        // the localized names of the standard number format
        STANDARD_NUMBER_FORMATS = [
            'Standard',
            'General',
            'Estandarra',
            'Skoueriek',
            'Standaard',
            'Yleinen',
            'Bendras',
            '\u041E\u043F\u0448\u0442\u043E',
            '\u0639\u0627\u062F\u06CC',
            'Geral',
            'Padr\xe3o',
            'Estandar'
        ],

        // regex to check for error codes
        ERROR_CODES_REGEX = /#(DIV\/0|N\/A|NAME|NULL|NUM|REF|VALUE)(\!|\?)/i;

    // private static functions ===============================================

    /**
     * Converts a number format code to be used with the mso-number-format style of Excel.
     *
     * @param {String} format
     *  The generic number format code.
     *
     * @returns {String}
     *  The number format code prepared for Excel.
     */
    function convertNumberFormatToExcel(format) {

        // TODO: the token 'standard' may be part of the format code, e.g.: Standard;-Standard;0
        if (format.toLowerCase() === 'standard') {
            return 'General';
        }

        format = Utils.cleanString(format);

        // replace the ampersand with the text &amp; (must be done first!)
        format = format.replace(/&/g, '&amp;');

        if (_.browser.IE) {
            // use ISO Latin-1 code
            format = format
                // replace space or backslash space with the text &#32;
                .replace(/(\\\s|\s)/g, '&#32;')
                // replace backslash double quote with the text &#92;&#34;
                .replace(/\\"/g, '&#92;&#34;')
                // replace backslash apostrophe with the text &#92;&#39;
                .replace(/\\'/g, '&#92;&#39;')
                // replace backslash percent with the text &#92;&#37;
                .replace(/\\%/g, '&#92;&#37;')
                // replace backslash exclamation mark with the text &#92;&#33;
                .replace(/\\!/g, '&#92;&#33;')
                // replace double backslash the text &#92;&#92;
                .replace(/\\\\/g, '&#92;&#92;')
                // replace double quote, Euro symbol, double quote with the text &#92;&#8364; as work around for IE problems with the double quote
                .replace(/"€"/g, '&#92;&#8364;');

        } else {
            // use UTF-8 representation
            format = format
                // replace space or backslash space with the text \\0020;
                .replace(/(\\\s|\s)/g, '\\0020')
                // replace backslash double quote with the text \\005c&#34;
                .replace(/\\"/g, '\\005c&#34;')
                // replace backslash percent with the text \\005c&#37;
                .replace(/\\%/g, '\\005c&#37;')
                // replace backslash exclamation mark with the text \\005c&#33;
                .replace(/\\!/g, '\\005c&#33;')
                // replace double backslash the text \\005c;&#92;
                .replace(/\\\\/g, '\\005c\\005c');
        }

        return format
            // replace the left angle bracket with the text &lt;
            .replace(/</g, '&lt;')
            // replace the right angle bracket with the text &gt;
            .replace(/>/g, '&gt;')
            // replace the double quote character with the text &#34;
            .replace(/"/g, '&#34;')
            // replace the apostrophe with the text &#39; (&apos; is not an HTML entity!)
            .replace(/'/g, '&#39;');
    }

    /**
     * Converts a number format code to be used with the sdnum attribute of Calc.
     *
     * @param {String} format
     *  The generic number format code.
     *
     * @returns {String}
     *  The number format code prepared for Calc.
     */
    function convertNumberFormatToCalc(format) {
        // TODO: the token 'standard' may be part of the format code, e.g.: Standard;-Standard;0
        return (format.toLowerCase() === 'standard') ? 'General' : Utils.escapeHTML(format);
    }

    /**
     * Returns the number format code for the given cell data.
     *
     * @param {Object} cellData
     *  The contents (display string, result, formula) and formatting of the cell.
     *
     * @returns {String|Null}
     *  The number format code.
     */
    function getNumberFormat(numberFormatter, cellData) {
        var numberFormat = cellData && cellData.attributes && cellData.attributes.cell && cellData.attributes.cell.numberFormat;
        return _.isObject(numberFormat) ? numberFormatter.resolveFormatCode(numberFormat) : null;
    }

    /**
     * Returns the number format code for the given cell data to be exported to Excel.
     *
     * @param {Object} cellData
     *  The contents (display string, result, formula) and formatting of the cell.
     *
     * @returns {String|Null}
     *  The number format code.
     */
    function getExcelNumberFormat(numberFormatter, cellData) {
        var formatCode = getNumberFormat(numberFormatter, cellData);
        return formatCode ? convertNumberFormatToExcel(formatCode) : null;
    }

    /**
     * Returns the number format code for the given cell data to be exported to Calc.
     *
     * @param {Object} cellData
     *  The contents (display string, result, formula) and formatting of the cell.
     *
     * @returns {String|undefined}
     *  The number format code.
     */
    function getCalcNumberFormat(numberFormatter, cellData) {
        var formatCode = getNumberFormat(numberFormatter, cellData);
        return formatCode ? convertNumberFormatToCalc(formatCode) : null;
    }

    /**
     * Returns true for the standard number format.
     *
     * @param {String} format
     *  The number format code as String.
     *
     * @returns {Boolean}
     *  Returns true for the standard number format.
     */
    function isStandardNumberFormat(format) {
        return _.any(STANDARD_NUMBER_FORMATS, function (item) {
            return (_.isString(this) && this.indexOf(item.toLowerCase()) > -1);
        }, format.toLowerCase());
    }

    /**
     * Creates a number format cell style object from a Calc number format
     * string.
     *
     * @param {String|Null} format
     *  The number format code as String.
     *
     *  @returns {Object|Null}
     *   The number format object
     */
    function parseCalcNumberFormat(format) {

        if (!_.isString(format)) { return null; }

        if (isStandardNumberFormat(format)) {
            return { id: 0 };
        }

        var numberFormat = format.match(/(?:(\d{4,5}|0);){0,1}(?:(\d{4,5}|0);){0,1}(.*)/),
            language;

        if (_.isArray(numberFormat) && numberFormat[3] && numberFormat[3].length > 0) {
            numberFormat[2] = parseInt(numberFormat[2], 10);
            if (numberFormat[2] > 0) {
                language = numberFormat[2];
            } else {
                numberFormat[1] = parseInt(numberFormat[1], 10);
                if (numberFormat[1] > 0) {
                    language = numberFormat[1];
                } else {
                    language = 1033;
                }
            }

            return { code: numberFormat[3], language: language };
        }

        return null;
    }

    /**
     * Returns the direct child element that matches the given jQuery selector.
     * Is limited to work inside HTML tables, e.g. find a 'strong' as child of a 'td' element.
     *
     * @param {jQuery} element
     *  The jQuery root element to start looking for the child.
     *
     * @returns {jQuery}
     *  The matching child element.
     */
    function findMatchingChild(element, selector) {
        return element.find(selector).filter(function () {
            var parent = $(this).parents('span,a,th,td,tr,tfoot,thead,tbody,colgroup,table').first();
            return parent[0] === element[0];
        });
    }

    /**
     * Returns true if the given jQuery selector matches a direct child element.
     * Is limited to work inside HTML tables, e.g. find a 'strong' as child of a 'td' element.
     *
     * @param {jQuery} element
     *  The jQuery root element to start looking for the child.
     *
     * @returns {Boolean}
     *  Returns true if a child of the given 'element' matches the given 'selector'.
     */
    function hasMatchingChild(element, selector) {
        return findMatchingChild(element, selector).length > 0;
    }

    /**
     * Converts a CSS font size with measurement unit into a HTML font size.
     *
     * @param {String} cssSize
     *  The CSS font size value with its measurement unit to be converted, as String.
     *
     * @returns {Number}
     *  The converted CSS size.
     */
    function convertToFontSize(cssSize) {
        var size = Utils.convertCssLength(cssSize, 'pt');
        if (size > 30) { return 7; }    // 7 -> 36pt
        if (size > 21) { return 6; }    // 6 -> 24pt
        if (size > 16) { return 5; }    // 5 -> 18pt
        if (size > 13) { return 4; }    // 4 -> 14pt
        if (size > 11) { return 3; }    // 3 -> 12pt
        if (size > 9)  { return 2; }    // 2 -> 10pt
        if (size > 0)  { return 1; }    // 1 -> 8pt
        return 0;
    }

    /**
     * Converts a HTML font size String (used as size attribute inside a font element)
     * into a size in point.
     *
     * @param {String} size
     *  The font size to be converted, as String.
     *
     * @returns {Number}
     *  The converted size, in point.
     */
    function convertFontSizeToPoint(size) {
        switch (size) {
        case '1':
            return 8;
        case '2':
            return 10;
        case '3':
            return 12;
        case '4':
            return 14;
        case '5':
            return 18;
        case '6':
            return 24;
        case '7':
            return 36;
        default:
            return 0;
        }
    }

    /**
     * Returns the character styles as CSS style String.
     *
     * @param {Object} cellData
     *  The contents (display string, result, formula) and formatting of the cell.
     *
     * @param {DocumentStyles} documentStyles
     *  The document styles collection of the document.
     *
     *  @returns {String}
     *   The CSS style String.
     */
    function getCssCharacterStyles(cellData, documentStyles) {

        var // attribute map for the 'character' attribute family
            charAttributes = cellData && cellData.attributes && cellData.attributes.character,
            // the fill color cell attribute
            fillColor = cellData && cellData.attributes && cellData.attributes.cell && cellData.attributes.cell.fillColor,
            // the CSS text decoration
            textDecoration = 'none',
            // the character style attribute data
            characterStyle = '';

        if (!charAttributes) { return ''; }

        if (charAttributes.fontName) { characterStyle += 'font-family:' + documentStyles.getCssFontFamily(charAttributes.fontName) + ';'; }
        if (charAttributes.fontSize) { characterStyle += 'font-size:' + charAttributes.fontSize + 'pt;'; }

        characterStyle += 'font-weight:' + ((charAttributes.bold) ? 'bold;' : 'normal;');
        characterStyle += 'font-style:' + ((charAttributes.italic) ? 'italic;' : 'normal;');

        if (charAttributes.underline) { textDecoration = Utils.addToken(textDecoration, 'underline', 'none'); }
        if (charAttributes.strike !== 'none') { textDecoration = Utils.addToken(textDecoration, 'line-through', 'none'); }
        characterStyle += 'text-decoration:' + textDecoration + ';';

        if (charAttributes.color) { characterStyle += 'color:' + documentStyles.getCssTextColor(charAttributes.color, [fillColor]) + ';'; }

        return characterStyle;
    }

    /**
     * Returns the cell styles as CSS style String.
     *
     * @param {Object} cellData
     *  The contents (display string, result, formula) and formatting of the cell.
     *
     * @param {DocumentStyles} documentStyles
     *  The document styles collection of the document.
     *
     *  @returns {String}
     *   The CSS style String.
     */
    function getCssCellStyles(cellData, documentStyles) {

        var // attribute map for the 'cell' attribute family
            cellAttributes = cellData && cellData.attributes && cellData.attributes.cell,
            // the cell style attribute data
            cellStyle = '';

        if (!cellAttributes) { return ''; }

        if (_.contains(['left', 'center', 'right', 'justify'], cellAttributes.alignHor)) { cellStyle += 'text-align:' + cellAttributes.alignHor + ';'; }
        if (_.contains(['top', 'middle', 'bottom', 'justify'], cellAttributes.alignVert)) { cellStyle += 'vertical-align:' + cellAttributes.alignVert + ';'; }

        cellStyle += 'white-space:' + ((cellAttributes.wrapText) ? 'normal;' : 'nowrap;');
        if (cellAttributes.fillColor) { cellStyle += 'background-color:' + documentStyles.getCssColor(cellAttributes.fillColor, 'fill') + ';'; }

        _.each(['Top', 'Bottom', 'Left', 'Right'], function (attrPart) {

            var // the cell style attribute name
                oxAttrName = 'border' + attrPart,
                // the CSS attribute name
                cssAttrName = 'border-' + attrPart.toLowerCase() + ':',
                // the border cell style attribute value
                border = cellAttributes[oxAttrName],
                // the effective CSS attributes
                cssAttrs = documentStyles.getCssBorderAttributes(border);

            cellStyle += cssAttrName;
            cellStyle += ((cssAttrs && cssAttrs.width > 0) ? cssAttrs.style + ' ' + cssAttrs.width + 'px ' + cssAttrs.color : 'none');
            cellStyle += ';';
        });

        return cellStyle;
    }

    /**
     * Returns the character styles as a String of the HTML elements tree.
     *
     * @param {Object} cellData
     *  The contents (display string, result, formula) and formatting of the cell.
     *
     * @param {DocumentStyles} documentStyles
     *  The document styles collection of the document.
     *
     *  @returns {String}
     *   The HTML elements String.
     */
    function getHtmlCharacterStyles(cellData, documentStyles) {

        var // attribute map for the 'character' attribute family
            charAttributes = cellData && cellData.attributes && cellData.attributes.character,
            // the fill color cell attribute
            fillColor = cellData && cellData.attributes && cellData.attributes.cell && cellData.attributes.cell.fillColor,
            // the character style HTML elements
            characterStyle = '',
            // the hyperlink URL
            url;


        if (!charAttributes) { return cellData.display; }

        // create opening tags
        if (charAttributes.bold) { characterStyle += '<b>'; }
        if (charAttributes.italic) { characterStyle += '<i>'; }
        if (charAttributes.underline) { characterStyle += '<u>'; }
        if (charAttributes.strike !== 'none') { characterStyle += '<s>'; }

        url = (charAttributes.url) ? Utils.escapeHTML(HyperlinkUtils.checkForHyperlink(charAttributes.url)) : null;
        if (url && url.length) {
            characterStyle += '<a href="' + url + '">';
        }

        if (charAttributes.fontName || charAttributes.fontSize || charAttributes.color) {
            characterStyle += '<font ';
            // To make it work in Calc we need to use just the font name instead of the entire CSS font family
            if (charAttributes.fontName) { characterStyle += 'face="' + Utils.escapeHTML(charAttributes.fontName) + '" '; }
            // We need to use rgb colors, like #RRGGBB
            if (charAttributes.color) { characterStyle += 'color="' + documentStyles.getCssTextColor(charAttributes.color, [fillColor]) + '" '; }
            // Add the font-size CSS also here to support precise font sizes in Excel
            if (charAttributes.fontSize) { characterStyle += 'size="' + convertToFontSize(charAttributes.fontSize) + '" style="font-size:' + charAttributes.fontSize + 'pt;"'; }
            characterStyle += '>';
        }

        // add cell value
        characterStyle += Utils.escapeHTML(cellData.display);

        // create closing tags
        if (charAttributes.fontName || charAttributes.fontSize || charAttributes.color) { characterStyle += '</font>'; }
        if (url && url.length) { characterStyle += '</a>'; }
        if (charAttributes.strike !== 'none') { characterStyle += '</s>'; }
        if (charAttributes.underline) { characterStyle += '</u>'; }
        if (charAttributes.italic) { characterStyle += '</i>'; }
        if (charAttributes.bold) { characterStyle += '</b>'; }

        return characterStyle;
    }

    /**
     * Returns the cell styles as HTML attributes String.
     *
     * @param {Object} cellData
     *  The contents (display string, result, formula) and formatting of the cell.
     *
     * @param {DocumentStyles} documentStyles
     *  The document styles collection of the document.
     *
     *  @returns {String}
     *   The HTML attributes String.
     */
    function getHtmlCellStyles(cellData, documentStyles) {

        var // attribute map for the 'cell' attribute family
            cellAttributes = cellData && cellData.attributes && cellData.attributes.cell,
            // the cell style attribute data
            cellStyle = '';

        if (!cellAttributes) { return ''; }

        if (_.contains(['left', 'center', 'right', 'justify'], cellAttributes.alignHor)) { cellStyle += ' align="' + cellAttributes.alignHor + '"'; }
        if (_.contains(['top', 'middle', 'bottom'], cellAttributes.alignVert)) { cellStyle += ' valign="' + cellAttributes.alignVert + '"'; }
        if (cellAttributes.fillColor && !Color.isAutoColor(cellAttributes.fillColor)) { cellStyle += ' bgcolor="' + documentStyles.getCssColor(cellAttributes.fillColor, 'fill') + '"'; }

        return cellStyle;
    }

    /**
     * Returns the attribute map for the 'character' attribute family of the HTML cell.
     *
     * @param {jQuery} element
     *  The HTML element.
     *
     *  @returns {Object}
     *   The character attribute map.
     */
    function getCharacterAttributes(element) {

        var // attribute map for the 'character' attribute family
            attrs = { character: {} },
            // the CSS attributes
            fontName, fontSize, fontWeight, fontStyle, textDecoration, color, url,
            // the <font> element, if HTML elements are used instead of CSS
            fontElement = findMatchingChild(element, 'font');


        fontName = fontElement.attr('face') || element.css('font-family');
        if (_.isString(fontName) && fontName.length > 0) {
            fontName = (fontName.split(',')[0]).replace(/["']/g, '');
            attrs.character.fontName = fontName;
        }

        fontSize = convertFontSizeToPoint(fontElement.attr('size')) || Utils.convertCssLength(element.css('font-size'), 'pt');
        if (fontSize > 0) {
            attrs.character.fontSize = fontSize;
        }

        if (hasMatchingChild(element, 'em,i')) {
            attrs.character.italic = true;
        } else {
            fontStyle = element.css('font-style');
            if (_.isString(fontStyle) && fontStyle.length > 0) {
                attrs.character.italic = (fontStyle === 'italic');
            }
        }

        if (hasMatchingChild(element, 'strong,b')) {
            attrs.character.bold = true;
        } else {
            fontWeight = element.css('font-weight');
            if (_.isString(fontWeight) && fontWeight.length > 0) {
                attrs.character.bold = (fontWeight === 'bold' || fontWeight === 'bolder' || fontWeight === '600' || fontWeight === '700' || fontWeight === '800' || fontWeight === '900');
            }
        }

        textDecoration = element.css('text-decoration');

        if (hasMatchingChild(element, 'u')) {
            attrs.character.underline = true;
        } else if (_.isString(textDecoration) && textDecoration.length > 0) {
            attrs.character.underline = (textDecoration.indexOf('underline') >= 0);
        }

        if (hasMatchingChild(element, 's')) {
            attrs.character.strike = 'single';
        } else if (_.isString(textDecoration) && textDecoration.length > 0) {
            attrs.character.strike = ((textDecoration.indexOf('line-through') >= 0) ? 'single' : 'none');
        }

        color = fontElement.attr('color') || element.css('color');
        color = Color.convertCssColorToRgbColor(color);
        if (color) {
            attrs.character.color = color;
        }

        url = findMatchingChild(element, 'a').attr('href');
        if (_.isString(url) && url.length > 0) {
            attrs.character.url = Utils.escapeHTML(url);
        }

        return attrs;
    }

    /**
     * Returns the attribute map for the 'cell' attribute family of the HTML element.
     *
     * @param {jQuery} element
     *  The HTML element.
     *
     *  @returns {Object}
     *   The cell attribute map.
     */
    function getCellAttributes(element) {

        var // attribute map for the 'cell' attribute family
            attrs = { cell: {} },
            // the CSS attributes
            alignHor, alignVert, wrapText, fillColor,
            // the number format
            numberFormat;

        // map css border style to operation border line style
        function mapCssBorderStyle(style) {
            switch (style) {
            case 'solid':
            case 'groove':
            case 'ridge':
            case 'inset':
            case 'outset':
                return 'single';
            case 'double':
                return 'double';
            case 'dotted':
                return 'dotted';
            case 'dashed':
                return 'dashed';
            default:
                return 'none';
            }
        }

        alignHor = element.attr('align') || element.css('text-align');
        // check value and filter prefix (webkit-, moz-, ms-)
        alignHor = _.find(['left', 'center', 'right', 'justify'], function (item) { return (_.isString(this) && this.toLowerCase().indexOf(item) > -1);  }, alignHor);
        if (alignHor) {
            attrs.cell.alignHor = alignHor;
        }

        alignVert = element.attr('valign') || element.css('vertical-align');
        // check value and filter prefix (webkit-, moz-, ms-)
        alignVert = _.find(['top', 'middle', 'bottom', 'justify'], function (item) { return (_.isString(this) && this.toLowerCase().indexOf(item) > -1);  }, alignVert);
        if (alignVert) {
            attrs.cell.alignVert = alignVert;
        }

        wrapText = element.css('white-space');
        if (_.isString(wrapText) && wrapText.length > 0) {
            attrs.cell.wrapText = (wrapText !== 'nowrap') ? true : false;
        }

        fillColor = element.attr('bgcolor') || element.css('background-color');
        fillColor = Color.convertCssColorToRgbColor(fillColor);
        if (fillColor) {
            attrs.cell.fillColor = fillColor;
        }

        _.each(['Top', 'Bottom', 'Left', 'Right'], function (attrPart) {

            var // the element style attribute name
                oxAttrName = 'border' + attrPart,
                // the CSS attribute name
                cssAttrName = 'border-' + attrPart.toLowerCase(),
                // the border style
                borderStyle = mapCssBorderStyle(element.css(cssAttrName + '-style')),
                // the border width
                borderWidth,
                // the border color
                borderColor;

            if (borderStyle !== 'none') {

                if (_.browser.Firefox && element[0]) {
                    // On FF element.css(cssAttrName + '-width') is always 0px, bug #35125
                    borderWidth = element[0].style[oxAttrName + 'Width'] || element.css(cssAttrName + '-width');
                } else {
                    borderWidth = element.css(cssAttrName + '-width');
                }
                borderColor = element.css(cssAttrName + '-color');
                borderColor = Color.convertCssColorToRgbColor(borderColor);

                attrs.cell[oxAttrName] = {
                    style : borderStyle,
                    width : Utils.convertCssLengthToHmm(borderWidth),
                    color : (borderColor ? borderColor : Color.AUTO)
                };
            }

        });

        numberFormat = parseCalcNumberFormat(element.attr('sdnum'));
        if (numberFormat) {
            attrs.cell.numberFormat = numberFormat;
        }

        return attrs;
    }

    // static class Clipboard =================================================

    var Clipboard = {};

    // methods ----------------------------------------------------------------

    /**
     * Creates a unique client clipboard id.
     *
     * @returns {String}
     *  The client clipboard id.
     */
    Clipboard.createClientClipboardId = function () {
        return ox.session + '-' + Date.now();
    };

    /**
     * Returns whether the passed value is an error code.
     *
     * @param {String} value
     *  Any literal value that can be used in formulas.
     *
     * @returns {Boolean}
     *  Whether the passed value is an error code.
     */
    Clipboard.isErrorCode = function (value) {
        return ERROR_CODES_REGEX.test(value);
    };

    /**
     * Creates a HTML table from the active range of the current selection.
     *
     * @param {SpreadsheetApplication} app
     *  The application requesting the HTML table.
     *
     * @param {SheetModel} sheetModel
     *  The sheet model instance containing the cells.
     *
     * @param {Object} range
     *  The range address.
     *
     * @param {String} clientClipboardId
     *  The client clipboard id to identify the copy data when pasting.
     *
     * @returns {String}
     *  The resulting HTML mark-up of the <table> element.
     */
    Clipboard.getHTMLStringFromRange = function (app, sheetModel, range, clientClipboardId) {

        var // the current table row
            row = range.start[1],
            // the style sheet container of the document
            documentStyles = app.getModel().getDocumentStyles(),
            // the number formatter of the document
            numberFormatter = app.getModel().getNumberFormatter(),
            // the formula parser
            tokenArray = new TokenArray(app, sheetModel, { trigger: 'never', grammar: 'ui' }),
            // the table string
            table = '<html ' + EXCEL_NAMESPACES + '>' + HTML_HEADER + '<body><table id="ox-clipboard-data" data-ox-clipboard-id="' + clientClipboardId + '"><tbody><tr>';

        sheetModel.getCellCollection().iterateCellsInRanges(range, function (cellData, origRange, mergedRange) {

            var address = cellData.address,
                isReferenceCell,
                colspan,
                rowspan,
                style,
                excelNumberFormat,
                calcNumberFormat,
                formula;

            // check if the cell is the reference cell of a merged collection
            isReferenceCell = mergedRange && _.isEqual(address, mergedRange.start);

            // check if we need to add a new row to the table
            if (row !== address[1]) {
                table += '</tr><tr>';
                row = address[1];
            }

            // render cell if it's the reference cell or if it's not a merged cell
            if (isReferenceCell || !mergedRange) {

                table += '<td ';

                if (isReferenceCell) {
                    // add cell with colspan and rowspan
                    colspan = mergedRange.end[0] - mergedRange.start[0] + 1;
                    rowspan = mergedRange.end[1] - mergedRange.start[1] + 1;

                    table += (colspan > 1) ? 'colspan="' + colspan + '" ' : '';
                    table += (rowspan > 1) ? 'rowspan="' + rowspan + '" ' : '';
                }

                // add cell and character style attributes
                style = Utils.escapeHTML(getCssCharacterStyles(cellData, documentStyles) + getCssCellStyles(cellData, documentStyles));

                // handle Excel number format
                excelNumberFormat = getExcelNumberFormat(numberFormatter, cellData);
                if (excelNumberFormat) {
                    style += 'mso-number-format:\'' + excelNumberFormat + '\';';
                }

                if (style.length > 0) {
                    table += ' style="' + style + '"';
                }

                // set the cell value type
                if (_.isNumber(cellData.result)) {
                    table += ' x:num="' + cellData.result + '"';
                    table += ' sdval="' + cellData.result + '"';
                } else if (_.isBoolean(cellData.result)) {
                    table += ' x:bool="' + cellData.result + '"';
                } else if (_.isString(cellData.result) && cellData.result[0] === '#') {
                    table += ' x:err="' + Utils.escapeHTML(cellData.result) + '"';
                } else if (_.isString(cellData.result)) {
                    table += ' x:str="' + Utils.escapeHTML(cellData.result) + '"';
                }

                // handle Calc number format
                calcNumberFormat = getCalcNumberFormat(numberFormatter, cellData);
                if (calcNumberFormat) {
                    table += ' sdnum="1033;1033;' + calcNumberFormat + '"';
                }

                // handle formula
                if (_.isString(cellData.formula)) {

                    // relocate all formulas relative to A1
                    tokenArray.parseFormula(cellData.formula);
                    formula = tokenArray.getRelocatedFormula(range.start, [0, 0]);
                    Utils.log('clipboard - formula - range: ' + JSON.stringify(range) + ', original: ' + cellData.formula + ', relocated: ' + formula);

                    if (!Clipboard.isErrorCode(formula)) {
                        table += ' x:fmla="' + Utils.escapeHTML(formula) + '"';
                    }
                }

                // add HTML attributes
                table += getHtmlCellStyles(cellData, documentStyles);

                table += '>';

                // add HTML elements and cell content
                table += getHtmlCharacterStyles(cellData, documentStyles);

                table += '</td>';
            }
        }, { merged: true, ordered: true });

        tokenArray.destroy();
        table += '</tr></tbody></table></body></html>';
        return table;
    };

    /**
     * Creates a plain text representation from the active range of the current
     * selection.
     *
     * @param {SheetModel} sheetModel
     *  The sheet model instance containing the cells.
     *
     * @param {Object} range
     *  The range address.
     *
     * @returns {String}
     *  The resulting plain text string.
     */
    Clipboard.getPlainTextFromRange = function (sheetModel, range) {

        var // the address of the first visible cell
            startAddress = null,
            // the resulting TAB-separated display strings
            result = '';

        sheetModel.getCellCollection().iterateCellsInRanges(range, function (cellData) {
            if (!startAddress) {
                startAddress = cellData.address;
            } else if (cellData.address[0] > startAddress[0]) {
                result += '\t';
            } else if (cellData.address[1] > startAddress[1]) {
                result += '\r\n';
            }
            result += cellData.display;
        }, { ordered: true });

        return result;
    };

    /**
     * Retrieves the client clipboard ID from the HTML data.
     *
     * @param {String|jQuery} html
     *  The HTML data to check.
     *
     * @returns {String|undefined}
     *  The client clipboard ID or undefined.
     */
    Clipboard.getClientClipboardId = function (html) {
        return $(html).find('[id=ox-clipboard-data]').attr('data-ox-clipboard-id') || $(html).filter('[id=ox-clipboard-data]').attr('data-ox-clipboard-id');
    };

    /**
     * Returns true if the HTML contains exactly one table element.
     *
     * @param {String|jQuery} html
     *  The HTML data to check.
     *
     * @returns {Boolean}
     *  Returns true if a html table is found.
     */
    Clipboard.containsHtmlTable = function (html) {
        var table;

        // look for table elements in the jQuery set $(html)
        table = $(html).filter('table');
        if (table.length === 0) {
            // look for table elements that are descendants of the jQuery set $(html)
            table = $(html).find('table');
        }
        // check that the table does not contain nested tables
        return (table.length === 1) && (table.find('table').length === 0);
    };

    /**
     * Parses the HTML from the clipboard data for a table and
     * creates the cell contents and merged cell collection.
     *
     * @param {String|jQuery} html
     *  The HTML input to parse.
     *
     * @returns {Object}
     *  An object with the following attributes:
     *      - {Array} contents
     *          A two-dimensional array containing the values and attribute sets
     *          for the cells.
     *      - {Array} mergeCollection
     *          An array of merged cell ranges.
     */
    Clipboard.parseHTMLData = function (html, documentStyles) {

        // convert HTML mark-up string to DOM tree
        html = $(html);

        var // a two dimensional array containing the cell contents
            contents = [],
            // an array of ranges of merged cells
            mergeCollection = [],
            // the table node
            table,
            // the row nodes
            rows,
            // the number of columns
            colCount,
            // the default character and cell attributes
            defaultAttrs = documentStyles.getDefaultAttributes(['character', 'cell']);

        // Bug 35126: Excel sends CSS rules containing fractional border widths, e.g.
        // 'border-top: .5pt solid ...' which will be translated to 0px borders in Chrome.
        // Only solution is to manually parse and modify the <style> element contained in
        // the pasted HTML data.
        html.find('>style').each(function () {

            // the first and only child node of the <style> element must be a text node
            if ((this.childNodes.length !== 1) || (this.firstChild.nodeType !== 3)) { return; }

            var // the text contents of the <style> node
                cssText = this.firstChild.nodeValue;

            // text must be a (multi-line) HTML comment
            if (!/^\s*<!--(.|[\r\n])+-->\s*$/.test(cssText)) { return; }

            // replace fractional line width in all border specifications with 1px
            cssText = cssText.replace(/([\r\n]\s*\{?(?:border(?:-top|-bottom|-left|-right)?|mso-diagonal-(?:down|up)):\s*)\.\d+pt /g, '$11px ');

            // replace non-standard line style 'hairline' with solid lines
            cssText = cssText.replace(/([\r\n]\s*\{?(?:border(?:-top|-bottom|-left|-right)?|mso-diagonal-(?:down|up)):.*?) hairline /g, '$1 solid ');

            // write the modified CSS rules back to the <style> element
            this.firstChild.nodeValue = cssText;
        });

        table = html.find('table');
        if (table.length === 0) {
            table = html.filter('table');
        }

        // merge in the character and cell attributes of the HTML table
        delete defaultAttrs.cell.numberFormat;
        delete defaultAttrs.character.url;
        documentStyles.extendAttributes(defaultAttrs, getCharacterAttributes(table));
        documentStyles.extendAttributes(defaultAttrs, getCellAttributes(table));

        // make sure that table header, body and footer are in the correct order
        rows = _.union(table.find('thead tr').get(), table.find('tbody tr').get(), table.find('tfoot tr').get());

        colCount = _.max(_.map(rows, function (row) { return $(row).find('td,th').length; }));

        _.each(rows, function (row, rowId) {

            var colId,
                rowData = [],
                cells,
                cell,
                cellId = 0,
                cellData,
                cellValue,
                colspan,
                rowspan,
                isMergedCell,
                rowParent,
                rowAttrs,
                linkChild;

            row = $(row);
            rowParent = row.parents('thead, tbody, tfoot, table').first();

            rowAttrs = getCharacterAttributes(rowParent);
            documentStyles.extendAttributes(rowAttrs, getCellAttributes(rowParent));
            documentStyles.extendAttributes(rowAttrs, getCharacterAttributes(row));
            documentStyles.extendAttributes(rowAttrs, getCellAttributes(row));


            cells = row.find('td,th');

            for (colId = 0; colId < colCount; colId++) {

                isMergedCell = SheetUtils.rangesContainCell(mergeCollection, [colId, rowId]);

                if (isMergedCell) {
                    // the cell is part of a merged collection, but not the reference cell
                    cellData = {};

                } else {
                    // the cell is not merged or it is the reference cell
                    cell = $(cells.get(cellId));
                    colspan = cell.attr('colspan');
                    rowspan = cell.attr('rowspan');

                    if (colspan || rowspan) {
                        // the cell is merged and it's a reference cell
                        colspan = (colspan && colspan.length > 0) ? parseInt(colspan, 10) : 1;
                        rowspan = (rowspan && rowspan.length > 0) ? parseInt(rowspan, 10) : 1;

                        mergeCollection.push({start: [colId, rowId], end: [colId + colspan - 1, rowId + rowspan - 1]});
                    }

                    // look for Calc cell value
                    cellValue = cell.attr('sdval');
                    if (_.isString(cellValue) && cellValue.length > 0) {
                        cellValue = parseFloat(cellValue, 10);
                    }
                    if (!_.isNumber(cellValue) || _.isNaN(cellValue)) {
                        cellValue = cell.text();
                        // if the cell value contains '\n' we need to make sure that we keep only the line feeds
                        // the user wants and not those Excel added by surprise.
                        if ((cellValue.indexOf('\n') > -1) || (cell.find('br').length > 0)) {
                            // remove line feeds from HTML and leave only the <br> elements
                            cell.html(cell.html().replace(/<br\s*\/?>([^\n]*)\n +/gi, '<br>$1').replace(/\n /g, ''));
                            // make '\n' out of the <br> elements
                            cell.find('br').replaceWith('\n');
                            // finally the cell value contains only the user line feeds
                            cellValue = cell.text();
                        }
                    }


                    cellData = {
                        value : cellValue,
                        attrs : _.copy(defaultAttrs, true)
                    };

                    // merge in the character and cell attributes of the HTML row and cell
                    documentStyles.extendAttributes(cellData.attrs, rowAttrs);
                    documentStyles.extendAttributes(cellData.attrs, getCharacterAttributes(cell));
                    documentStyles.extendAttributes(cellData.attrs, getCellAttributes(cell));

                    // merge in the hyperlink's character attributes that are attached to it's child span
                    linkChild = findMatchingChild(cell, 'a').children('span').first();
                    if (linkChild.length > 0) {
                        documentStyles.extendAttributes(cellData.attrs, getCharacterAttributes(linkChild));
                    }

                    cellId++;
                }

                rowData.push(cellData);
            }

            contents.push(rowData);
        });

        return { contents: contents, mergeCollection: mergeCollection };
    };

    /**
     * Parses the plain text from the clipboard data and creates the cell contents.
     * Rows should be separated by line endings ('\r\n', '\r' or '\n')
     * and cells should be separated by tab or semicolon.
     *
     * @param {String} text
     *  The text input to parse.
     *
     * @returns {Object}
     *  An object with the following attributes:
     *      - {Array} contents
     *          A two-dimensional array containing the values and attribute sets
     *          for the cells.
     */
    Clipboard.parseTextData = function (text) {

        var // a two dimensional array containing the cell contents
            contents = [],
            // the row nodes
            rows,
            // the cell separator
            sep;

        if (!_.isString(text)) { return {contents: []}; }

        // remove \r,\n, \t and ; to remove empty rows at the end of the text
        text = text.replace(/[;\s]+$/, '');

        // parse rows according to the line ending type
        rows = (text.indexOf('\r\n') > -1) ? text.split(/\r\n/) : text.split(/[\r\n]/);
        // look for the cell separator
        sep = (text.indexOf('\t') > -1) ? '\t' : ';';

        _.each(rows, function (row) {
            var rowData = [],
                cells;

            cells = row.split(sep);
            _.each(cells, function (cell) {
                rowData.push({ value: Utils.cleanString(cell) });
            });
            contents.push(rowData);
        });

        return {contents: contents};
    };

    // exports ================================================================

    return Clipboard;

});
