/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/tablecollection', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/object/timermixin',
    'io.ox/office/baseframework/model/modelobject',
    'io.ox/office/editframework/model/attributedmodel',
    'io.ox/office/spreadsheet/utils/sheetutils'
], function (Utils, TimerMixin, ModelObject, AttributedModel, SheetUtils) {

    'use strict';

    // class TableColumnModel =================================================

    /**
     * Stores filter and sorting settings for a single column in a table range.
     *
     * @constructor
     *
     * @extends AttributedModel
     *
     * @param {SpreadsheetApplication} app
     *  The application that contains this table range.
     *
     * @param {SheetModel} sheetModel
     *  The sheet model instance containing this table column.
     *
     * @param {SheetModel} tableModel
     *  The table model instance containing this table column.
     *
     * @param {Object} [initAttrs]
     *  The initial table attribute set (style sheet reference, and explicit
     *  table attributes).
     */
    var TableColumnModel = AttributedModel.extend({ constructor: function (app, sheetModel, tableModel, initAttrs) {

        // base constructor ---------------------------------------------------

        AttributedModel.call(this, app, initAttrs, { families: 'filter sort', autoClear: true });

        // protected methods --------------------------------------------------

        /**
         * Creates and returns a cloned instance of this table model for the
         * specified sheet.
         *
         * @internal
         *  Used by the class TableCollection during clone construction. DO NOT
         *  CALL from external code!
         *
         * @param {SheetModel} targetSheetModel
         *  The model instance of the new cloned sheet that will own the clone
         *  returned by this method.
         *
         * @param {TableModel} targetTableModel
         *  The model instance of the new cloned table that will own the clone
         *  returned by this method.
         *
         * @returns {TableColumnModel}
         *  A clone of this table column model, initialized for ownership by
         *  the passed sheet and table models.
         */
        this.clone = function (targetSheetModel, targetTableModel) {
            // construct a new model instance
            return new TableColumnModel(app, targetSheetModel, targetTableModel, this.getExplicitAttributes(true));
        };

        // public methods -----------------------------------------------------

        /**
         * Returns whether this column model contains active filter rules.
         *
         * @returns {Boolean}
         *  Whether this column model contains active filter rules.
         */
        this.isFiltered = function () {
            return this.getMergedAttributes().filter.type !== 'none';
        };

        /**
         * Returns whether this column model contains active sort options.
         *
         * @returns {Boolean}
         *  Whether this column model contains active sort options.
         */
        this.isSorted = function () {
            return false; // TODO
        };

        /**
         * Returns whether this column model contains any filter rules, or sort
         * options, that may require to refresh the table range containing this
         * column.
         *
         * @returns {Boolean}
         *  Whether this column model contains any filter/sorting settings.
         */
        this.isRefreshable = function () {
            return this.isFiltered() || this.isSorted();
        };

        // initialization -----------------------------------------------------

        // destroy all class members
        this.registerDestructor(function () {
            app = sheetModel = tableModel = null;
        });

    }}); // class TableColumnModel

    // class TableModel =======================================================

    /**
     * Stores settings for a single table range in a specific sheet.
     *
     * @constructor
     *
     * @extends AttributedModel
     * @extends TimerMixin
     *
     * @param {SpreadsheetApplication} app
     *  The application that contains this table range.
     *
     * @param {SheetModel} sheetModel
     *  The sheet model instance containing this table.
     *
     * @param {String} tableName
     *  The name of this table.
     *
     * @param {Object} tableRange
     *  The address of the cell range covered by this table.
     *
     * @param {Object} [initAttrs]
     *  The initial table attribute set (style sheet reference, and explicit
     *  table attributes).
     */
    var TableModel = AttributedModel.extend({ constructor: function (app, sheetModel, tableName, tableRange, initAttrs) {

        var // self reference
            self = this,

            // the spreadsheet document model
            docModel = app.getModel(),

            // all existing column models (with active filter or sorting settings), mapped by column index (for fast direct access)
            columnModels = {},

            // default column model for all unmodified columns
            defColumnModel = new TableColumnModel(app, sheetModel, this);

        // base constructors --------------------------------------------------

        AttributedModel.call(this, app, initAttrs, { styleFamily: 'table', autoClear: true });
        TimerMixin.call(this);

        // private methods ----------------------------------------------------

        /**
         * Inserts the passed table column model into this table, and registers
         * a listener for attribute change events that will be forwarded to the
         * listeners of this table.
         */
        function insertColumnModel(tableCol, columnModel) {

            var // the map entry, containing the current table column index as property (may change during runtime,
                // e.g. after inserting/deleting sheet columns, needed for fast access when forwarding events below)
                entry = { tableCol: tableCol, columnModel: columnModel };

            // store the column model as map entry (needed for fast direct access by column index)
            columnModels[tableCol] = entry;

            // listen to attribute change events, forward to own listeners
            self.listenTo(columnModel, 'change:attributes', function (event, newAttrs, oldAttrs) {

                // forward to listeners of this table model (pass current column index, not the
                // original column index passed to the method insertColumnModel(), see comment above)
                self.trigger('change:column', entry.tableCol, newAttrs, oldAttrs);

                // model can be deleted, if it does not contain any active settings
                if (!columnModel.hasExplicitAttributes()) {
                    columnModel.destroy();
                    delete columnModels[entry.tableCol];
                }
            });
        }

        /**
         * Returns the specified table column model, if it exists.
         */
        function getColumnModel(tableCol) {
            var entry = columnModels[tableCol];
            return entry ? entry.columnModel : null;
        }

        /**
         * Creates a missing table column model, and returns the model
         * instance. The passed column index will not be checked for validity.
         */
        function getOrCreateColumnModel(tableCol) {

            var // find an existing column model
                columnModel = getColumnModel(tableCol);

            // create missing column model
            if (!columnModel) {
                columnModel = new TableColumnModel(app, sheetModel, self);
                insertColumnModel(tableCol, columnModel);
            }

            return columnModel;
        }

        /**
         * Updates internal settings after the attributes of this table have
         * been changed.
         */
        function changeAttributesHandler() {

            // silently remove filter rules from all filtered column models, when
            // table is not active anymore (sorting settings remain valid though)
            if (!self.isFilterActive()) {
                _.each(columnModels, function (entry, key) {
                    // remove filter settings from the column
                    entry.columnModel.setAttributes({ filter: { type: 'none' } }, { notify: 'never' });
                    // keep the column model in the array, if any attributes are left
                    if (!entry.columnModel.hasExplicitAttributes()) {
                        entry.columnModel.destroy();
                        delete columnModels[key];
                    }
                });
            }
        }

        /**
         * Creates a filter matcher for the passed entries of a discrete filter
         * rule.
         *
         * @param {String[]} entries
         *  The matching entries of a discrete filter rule.
         *
         * @returns {Function}
         *  A predicate function that takes a cell descriptor object with the
         *  properties 'result' (typed cell result value) and 'display' (the
         *  formatted display string), and returns a Boolean whether the passed
         *  cell matches any of the entries in the string array.
         */
        function createDiscreteFilterMatcher(entries) {

            // no matching entries: all cell values will be filtered
            if (entries.length === 0) {
                return _.constant(false);
            }

            // performance: create a map with all matching display strings
            // as keys, instead of searching in the entry array
            var entryMap = {};
            _.each(entries, function (entry) { entryMap[entry.toLowerCase()] = 1; });

            // create a matcher predicate function that checks the lower-case display string of a cell
            return function discreteFilterMatcher(cellData) {
                // bug 36264: trim space characters but no other whitespace
                return cellData.display.replace(/^ +| +$/g, '').toLowerCase() in entryMap;
            };
        }

        // protected methods --------------------------------------------------

        /**
         * Creates and returns a cloned instance of this table model for the
         * specified sheet.
         *
         * @internal
         *  Used by the class TableCollection during clone construction. DO NOT
         *  CALL from external code!
         *
         * @param {SheetModel} targetModel
         *  The model instance of the new cloned sheet that will own the clone
         *  returned by this method.
         *
         * @returns {TableModel}
         *  A clone of this table model, initialized for ownership by the
         *  passed sheet model.
         */
        this.clone = function (targetModel) {
            // construct a new model instance, pass all own column models as hidden parameter
            return new TableModel(app, targetModel, tableName, tableRange, this.getExplicitAttributes(true), columnModels);
        };

        // public methods -----------------------------------------------------

        /**
         * Returns the name of this table.
         *
         * @returns {String}
         *  The name of this table.
         */
        this.getName = function () {
            return tableName;
        };

        /**
         * Returns whether this table id the special auto-filter range of the
         * sheet.
         *
         * @returns {Boolean}
         *  Whether this table id the special auto-filter range of the sheet.
         */
        this.isAutoFilter = function () {
            return tableName.length === 0;
        };

        /**
         * Returns the address of the cell range covered by this table.
         *
         * @returns {Object}
         *  The address of the cell range covered by this table.
         */
        this.getRange = function () {
            return _.copy(tableRange, true);
        };

        /**
         * Returns whether the passed cell address is covered by this table.
         *
         * @param {Number[]} address
         *  The address of the cell to be checked.
         *
         * @returns {Boolean}
         *  Whether the specified cell is covered by this table.
         */
        this.containsCell = function (address) {
            return SheetUtils.rangeContainsCell(tableRange, address);
        };

        /**
         * Returns whether the passed cell range address overlaps with the cell
         * range of this table.
         *
         * @param {Object} range
         *  The address of the cell range to be checked.
         *
         * @returns {Boolean}
         *  Whether the passed cell range address overlaps with the cell range
         *  of this table.
         */
        this.overlapsRange = function (range) {
            return SheetUtils.rangeOverlapsRange(tableRange, range);
        };

        /**
         * Returns whether this table range contains a header row. Cells in the
         * header row (the top-most row) will neither be filtered nor sorted.
         *
         * @returns {Boolean}
         *  Whether this table range contains a header row.
         */
        this.hasHeaderRow = function () {
            return this.getMergedAttributes().table.headerRow;
        };

        /**
         * Returns whether this table range contains a footer row. Cells in the
         * footer row (the bottom-most row) will neither be filtered nor
         * sorted.
         *
         * @returns {Boolean}
         *  Whether this table range contains a footer row.
         */
        this.hasFooterRow = function () {
            return this.getMergedAttributes().table.footerRow;
        };

        /**
         * Returns whether filtering in this table range is activated. This
         * does not necessarily mean that a row has actually been hidden by the
         * filter rules.
         *
         * @returns {Boolean}
         *  Whether filtering in this table range is activated.
         */
        this.isFilterActive = function () {
            return this.getMergedAttributes().table.filtered;
        };

        /**
         * Returns the address of the data range in this table (the cell range
         * without the header and footer rows). For the special auto-filter
         * table, the data range will be expanded to the bottom to include all
         * existing content cells following the real data range.
         *
         * @returns {Object|Null}
         *  The address of the data range of this table; or null, if no data
         *  rows are available (e.g., if the table consists of a header row
         *  only).
         */
        this.getDataRange = function () {

            var // whether the table contains header and/or footer rows
                hasHeader = this.hasHeaderRow(),
                hasFooter = this.hasFooterRow(),
                // the resulting data range
                dataRange = _.copy(tableRange, true);

            // expand data range of auto-filter to the bottom (before removing
            // the header row from the range, to not invalidate the range address)
            if (!hasFooter && this.isAutoFilter()) {
                dataRange = sheetModel.getCellCollection().findContentRange(dataRange, { directions: 'down' });

                // restrict expansion to the next table range below the auto-filter
                if (tableRange.end[1] < dataRange.end[1]) {
                    var boundRange = { start: [tableRange.start[0], tableRange.end[1] + 1], end: [tableRange.end[0], docModel.getMaxRow()] };
                    _.each(sheetModel.getTableCollection().findTables(boundRange), function (otherTableRange) {
                        tableRange.end[1] = Math.min(tableRange.end[1], otherTableRange.start[1] - 1);
                    });
                }
            }

            // exclude header/footer rows from returned range
            if (hasHeader) { dataRange.start[1] += 1; }
            if (hasFooter) { dataRange.end[1] -= 1; }

            // check existence of data rows
            return (dataRange.start[1] <= dataRange.end[1]) ? dataRange : null;
        };

        /**
         * Returns the relative intervals of all visible rows in the data range
         * of this table.
         *
         * @returns {Array}
         *  The visible row intervals in the data range of this table; or an
         *  empty array, if the table does not contain any data rows. The
         *  properties 'first' and 'last' of each interval object are relative
         *  to the data range of the table (the index 0 refers to the first
         *  data row of this table!).
         */
        this.getVisibleDataRows = function () {

            var // the address of the data range
                dataRange = this.getDataRange(),
                // the absolute row intervals
                rowIntervals = dataRange ? sheetModel.getRowCollection().getVisibleIntervals(SheetUtils.getRowInterval(dataRange)) : [],
                // row index of the first data row
                firstRow = dataRange ? dataRange.start[1] : 0;

            // transform to relative row intervals
            _.each(rowIntervals, function (rowInterval) {
                rowInterval.first -= firstRow;
                rowInterval.last -= firstRow;
            });

            return rowIntervals;
        };

        /**
         * Changes the address of the cell range covered by this table.
         *
         * @param {Object} range
         *  The address of the new cell range.
         *
         * @returns {ValidationModel}
         *  A reference to this instance.
         */
        this.setRange = function (range) {

            if (_.isEqual(range, tableRange)) { return this; }

            // store the new range (use a copy to prevent indirect modification)
            tableRange = _.copy(range, true);

            // delete table columns outside the new range
            var colCount = SheetUtils.getColCount(tableRange);
            _.each(columnModels, function (entry, key) {
                if (entry.tableCol >= colCount) {
                    entry.columnModel.destroy();
                    delete columnModels[key];
                }
            });

            // notify listeners
            this.trigger('change:range', range);
            return this;
        };

        /**
         * Transforms the cell range address of this table, after columns or
         * rows have been inserted into or removed from the sheet.
         *
         * @returns {Boolean}
         *  Whether the transformed range is still valid (not deleted
         *  completely). If false is returned, this collection entry must be
         *  deleted from the table collection.
         */
        this.transformRange = function (interval, insert, columns) {

            var // first column index of the old table range
                startCol = tableRange.start[0];

            // preprocessing when deleting rows in the table range
            if (!insert && !columns) {

                // bug 36226: delete tables whose header row has been deleted, update existence of footer row
                if (this.hasHeaderRow() && SheetUtils.intervalContainsIndex(interval, tableRange.start[1])) {
                    return false;
                }

                // update existence of footer row
                if (this.hasFooterRow() && SheetUtils.intervalContainsIndex(interval, tableRange.end[1])) {
                    this.setAttributes({ table: { footerRow: false } }, { notify: 'never' });
                }
            }

            // transform table range
            tableRange = docModel.transformRange(tableRange, interval, insert, columns);

            // table completely deleted
            if (!tableRange) { return false; }

            // delete or remap all affected column models
            if (columns) {

                var // create a new model map to prevent interfering with existing map entries
                    newColumnModels = {};

                _.each(columnModels, function (entry) {

                    var // transform the absolute column index according to the operation
                        absCol = docModel.transformIndex(startCol + entry.tableCol, interval, insert, true);

                    // delete column model of deleted columns, remap remaining column models
                    if (_.isNumber(absCol)) {
                        entry.tableCol = absCol - tableRange.start[0];
                        newColumnModels[entry.tableCol] = entry;
                    } else {
                        entry.columnModel.destroy();
                    }
                });

                columnModels = newColumnModels;
            }

            return true;
        };

        /**
         * Returns the merged column attributes of the specified table column.
         *
         * @param {Number} tableCol
         *  The zero-based index of the table column, relative to the cell
         *  range covered by this table.
         *
         * @returns {Object|Null}
         *  The merged attribute set of the specified table column; or null, if
         *  the passed table column index is invalid.
         */
        this.getColumnAttributes = function (tableCol) {

            var // the column model (fall-back to default model for unmodified columns)
                columnModel = getColumnModel(tableCol) || defColumnModel;

            return columnModel.getMergedAttributes();
        };

        /**
         * Changes the attributes of the specified table column. Does NOT apply
         * any changed filtering or sorting to the table range.
         *
         * @param {Number} tableCol
         *  The zero-based index of the table column to be modified, relative
         *  to the cell range covered by this table.
         *
         * @param {Object} attributes
         *  The attributes to be set for the specified table column.
         *
         * @returns {Boolean}
         *  Whether the attributes have been set successfully (especially,
         *  whether the passed column index is valid).
         */
        this.setColumnAttributes = function (tableCol, attributes) {

            // check passed column index
            if ((tableCol < 0) || (tableCol >= SheetUtils.getColCount(tableRange))) { return false; }

            getOrCreateColumnModel(tableCol).setAttributes(attributes);
            return true;
        };

        /**
         * Returns the address of the data range of a single column in this
         * table, without the header and footer cell.
         *
         * @param {Number} tableCol
         *  The zero-based index of the column, relative to the cell range
         *  covered by this table.
         *
         * @returns {Object|Null}
         *  The address of the data range in the specified table column; or
         *  null, if no data rows are available (e.g., if the table consists of
         *  a header row only), or if the passed column index is invalid.
         */
        this.getColumnDataRange = function (tableCol) {

            var // the data range of the entire table
                dataRange = this.getDataRange(),
                // the resulting absolute column index
                absCol = tableRange.start[0] + tableCol;

            // check validity of passed column index, and existence of data rows
            if (!dataRange || !SheetUtils.rangeContainsCol(tableRange, absCol)) {
                return null;
            }

            // adjust column indexes in the result range
            dataRange.start[0] = dataRange.end[0] = absCol;
            return dataRange;
        };

        /**
         * Returns the cell contents of the data range of a single column in
         * this table.
         *
         * @param {Number} tableCol
         *  The zero-based index of the column, relative to the cell range
         *  covered by this table.
         *
         * @param {Object|Array} [rowIntervals]
         *  If specified, a single row interval, or an array of row intervals
         *  in the data range of this table to query the cell data for. The
         *  properties 'first' and 'last' of each interval object are relative
         *  to the data range of the table (the index 0 refers to the first
         *  data row of this table!). If omitted, returns the cell contents of
         *  all data cells in the column.
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved with an
         *  compressed array of cell contents in the specified table column.
         *  Each array element is an object with the following properties:
         *  - {String} cellData.display
         *      The display string of the cell result (will be the empty string
         *      for blank cells).
         *  - {Number|String|Boolean|ErrorCode|Null} cellData.result
         *      The typed result value (will be null for blank cells).
         *  - {Number} cellData.count
         *      The number of consecutive cells with equal contents represented
         *      by this array element.
         *  See method CellCollection.queryCellContents() for more details.
         */
        this.queryColumnData = function (tableCol, rowIntervals) {

            var // the address of the data range in the specified column
                dataRange = this.getColumnDataRange(tableCol),
                // the row interval of the entire data range
                boundInterval = dataRange ? SheetUtils.getRowInterval(dataRange) : null,
                // the cell ranges to be passed to the query
                dataRanges = null;

            // reject the request, if no data range is available (e.g., header/footer rows only)
            if (!boundInterval) { return $.when([]); }

            // transform the passed relative intervals to absolute row intervals
            if (_.isObject(rowIntervals)) {
                rowIntervals = _.map(_.getArray(rowIntervals), function (rowInterval) {
                    return {
                        first: rowInterval.first + boundInterval.first,
                        last: rowInterval.last + boundInterval.first
                    };
                });
                rowIntervals = SheetUtils.getIntersectionIntervals(rowIntervals, boundInterval);
            } else {
                rowIntervals = [boundInterval];
            }

            // nothing to do, if the passed relative intervals are outside the data range
            if (rowIntervals.length === 0) { return $.when([]); }

            // build the data range addresses to be queried for
            dataRanges = SheetUtils.makeRangesFromIntervals(dataRange.start[0], rowIntervals);

            // resolve the array of result arrays returned by the cell collection to a single result array
            return sheetModel.getCellCollection()
                .queryCellContents([dataRanges], { hidden: true, compressed: true })
                .then(function (resultsArray) { return resultsArray[0]; });
        };

        /**
         * Returns whether the specified column contains active filter rules.
         *
         * @param {Number} tableCol
         *  The zero-based index of the column, relative to the cell range
         *  covered by this table.
         *
         * @returns {Boolean}
         *  Whether the specified column contains active filter rules.
         */
        this.isColumnFiltered = function (tableCol) {
            var columnModel = getColumnModel(tableCol);
            return _.isObject(columnModel) && columnModel.isFiltered();
        };

        /**
         * Returns whether the specified column contains active sort options.
         *
         * @param {Number} tableCol
         *  The zero-based index of the column, relative to the cell range
         *  covered by this table.
         *
         * @returns {Boolean}
         *  Whether the specified column contains active sort options.
         */
        this.isColumnSorted = function (tableCol) {
            var columnModel = getColumnModel(tableCol);
            return _.isObject(columnModel) && columnModel.isSorted();
        };

        /**
         * Returns whether this table contains any columns with active settings
         * (filter rules or sort options) that can be used to refresh the table
         * range.
         *
         * @returns {Boolean}
         *  Whether this table contains any columns with active settings.
         */
        this.isRefreshable = function () {
            return this.isFilterActive() && _.any(columnModels, function (entry) {
                return entry.columnModel.isRefreshable();
            });
        };

        /**
         * Creates a matcher predicate function according to the filter
         * attributes of the specified table column.
         *
         * @param {Number} tableCol
         *  The zero-based index of the column, relative to the cell range
         *  covered by this table.
         *
         * @returns {Function|Null}
         *  A predicate function that takes a cell descriptor object with the
         *  properties 'result' (typed cell result value) and 'display' (the
         *  formatted display string), and returns a Boolean whether the passed
         *  cell matches the filter rules of the specified column. This method
         *  will return null instead of a matcher function, if the specified
         *  table column does not contain any supported filter rules.
         */
        this.createFilterMatcher = function (tableCol) {

            var // the current filter attributes of the passed column
                filterAttrs = this.getColumnAttributes(tableCol).filter;

            // create a matcher function for discrete filter rule
            if (filterAttrs.type === 'discrete') {
                return createDiscreteFilterMatcher(filterAttrs.entries);
            }

            // unsupported filter type, or no existing filter rule
            return null;
        };

        /**
         * Builds and returns a list of row intervals containing all visible
         * rows not filtered by the current filter settings. For performance
         * reasons (to prevent blocked browser UI and its respective error
         * messages), the implementation of this method may run asynchronously.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Number} [options.excludeTableCol]
         *      The zero-based index of a table column that will be excluded
         *      from filtering (its filter settings will be ignored).
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved with the
         *  visible data row intervals. The properties 'first' and 'last' of
         *  each interval object are relative to the data range of the table
         *  (the index 0 always refers to the first data row of this table!).
         */
        this.queryFilterResult = function (options) {

            var // start with the entire data range
                dataRange = this.getDataRange();

            // no data rows available
            if (!dataRange) { return $.when([]); }

            var // indexes of all filtered columns
                filterCols = [],
                // table column to be excluded from filtering
                excludeTableCol = Utils.getIntegerOption(options, 'excludeTableCol', -1),
                // the entire row interval of the data range (absolute row indexes)
                dataRowInterval = SheetUtils.getRowInterval(dataRange),
                // the remaining visible data intervals for the next column (relative to the data range)
                visibleIntervals = [{ first: 0, last: SheetUtils.getIntervalSize(dataRowInterval) - 1 }],
                // the resulting promise of this method
                promise = null;

            // collect the column indexes of all filtered columns
            if (this.isFilterActive()) {
                _.each(columnModels, function (entry) {
                    if (entry.columnModel.isFiltered() && (entry.tableCol !== excludeTableCol)) {
                        filterCols.push(entry.tableCol);
                    }
                });
            }

            // process the table columns in a background loop to prevent unresponsive browser
            promise = this.iterateArraySliced(filterCols, function (tableCol) {

                // early exit, if no visible data rows are left for filtering
                if (visibleIntervals.length === 0) { return Utils.BREAK; }

                var // the filter matcher for the current column
                    matcher = this.createFilterMatcher(tableCol);

                // ignore columns with unknown/unsupported filter settings
                if (!matcher) { return; }

                // query the cell contents of all remaining visible rows
                return this.queryColumnData(tableCol, visibleIntervals).then(function (cellArray) {

                    var // the resulting row intervals which remain visible
                        newIntervals = [],
                        // index of the current data row
                        dataRow = 0,
                        // index of the current row interval, and offset inside the interval
                        intervalIndex = 0;

                    // pushes a new row interval to the 'newRowIntervals' array (tries to expand last interval)
                    function pushVisibleInterval(cellCount) {

                        var // end position of the new interval (start position is 'dataRow')
                            lastDataRow = dataRow + cellCount - 1,
                            // last pushed interval for expansion
                            lastInterval = _.last(newIntervals);

                        // try to expand the last interval
                        if (lastInterval && (lastInterval.last + 1 === dataRow)) {
                            lastInterval.last = lastDataRow;
                        } else {
                            newIntervals.push({ first: dataRow, last: lastDataRow });
                        }
                    }

                    // simultaneously iterate through the row intervals and compressed result array
                    _.each(cellArray, function (cellData) {

                        var // the number of remaining cell to process for the current cell data object
                            remainingCells = cellData.count,
                            // whether the current cell data results in visible rows (the filter matches)
                            visible = matcher(cellData),
                            // the current row interval
                            visibleInterval = null,
                            // number of cells that can be processed for the current row interval
                            cellCount = 0;

                        // process as many remaining data rows as covered by the current cell data entry
                        while (remainingCells > 0) {

                            // the current row interval
                            visibleInterval = visibleIntervals[intervalIndex];
                            // update 'dataRow' if a new row interval has been started
                            dataRow = Math.max(dataRow, visibleInterval.first);
                            // number of cells that can be processed for the current row interval
                            cellCount = Math.min(remainingCells, visibleInterval.last - dataRow + 1);

                            // if the cell data matches the filter rule, add the row interval to the result
                            if (visible) { pushVisibleInterval(cellCount); }

                            // decrease number of remaining cells for the current cell data object
                            remainingCells -= cellCount;
                            // update data row index
                            dataRow += cellCount;
                            // if end of current row interval has been reached, go to next row interval
                            if (visibleInterval.last < dataRow) { intervalIndex += 1; }
                        }
                    });

                    // use the resulting filtered row intervals for the next filtered column
                    visibleIntervals = newIntervals;
                });
            });

            // return a promise that resolves with the row intervals
            return promise.then(function () { return visibleIntervals; });
        };

        // initialization -----------------------------------------------------

        // update column models after changing table attributes
        this.on('change:attributes', changeAttributesHandler);

        // clone column models passed as hidden argument by the clone() method
        if (_.isObject(arguments[TableModel.length])) {
            _.each(arguments[TableModel.length], function (entry) {
                insertColumnModel(entry.tableCol, entry.columnModel.clone(sheetModel, this));
            }, this);
        }

        // destroy all class members
        this.registerDestructor(function () {
            _.each(columnModels, function (entry) { entry.columnModel.destroy(); });
            defColumnModel.destroy();
            app = docModel = sheetModel = columnModels = defColumnModel = null;
        });

    }}); // class TableModel

    // class TableCollection ==================================================

    /**
     * Stores settings for table ranges in a specific sheet. Table ranges
     * contain filter settings, sorting settings, and specific cell formatting.
     *
     * Triggers the following events:
     * - 'insert:table'
     *      After a new table has been inserted into this collection. Event
     *      handlers receive the following parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {TableModel} tableModel
     *          The model object representing the new table range.
     * - 'change:table'
     *      After an existing table has been changed. Event handlers receive
     *      the following parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {TableModel} tableModel
     *          The model object representing the modified table range.
     *      (3) {String} type
     *          The type of the original event forwarded from the table model.
     *          Can be 'change:attributes', 'change:range', or 'change:column'.
     *      (4) {Any} [...]
     *          Additionally parameters of the original table event.
     * - 'delete:table'
     *      After an existing table has been deleted from this collection.
     *      Event handlers receive the following parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {TableModel} tableModel
     *          The model object representing the deleted table range.
     *
     * @constructor
     *
     * @extends ModelObject
     *
     * @param {SpreadsheetApplication} app
     *  The application that contains this collection instance.
     *
     * @param {SheetModel} sheetModel
     *  The sheet model instance containing this collection.
     */
    function TableCollection(app, sheetModel) {

        var // self reference
            self = this,

            // all collection entries, mapped by table name
            tableModels = {};

        // base constructor ---------------------------------------------------

        ModelObject.call(this, app);

        // private methods ----------------------------------------------------

        /**
         * Inserts the passed table model into this collection, and registers a
         * listener for attribute change events that will be forwarded to the
         * listeners of this collection.
         */
        function insertTableModel(tableModel) {
            var tableName = tableModel.getName().toUpperCase();
            tableModels[tableName] = tableModel;
            self.listenTo(tableModel, 'change:attributes change:range change:column', function (event) {
                self.trigger.apply(self, ['change:table', tableModel, event.type].concat(_.toArray(arguments).slice(1)));
            });
        }

        /**
         * Invokes the specified method on all collection entries.
         *
         * @param {String} methodName
         *  The name of the method to be invoked on each collection entry. If
         *  the method returns the value false, the respective entry will be
         *  removed from the collection.
         *
         * @param {TableModel|Null} excludeModel
         *  A collection entry that will be skipped (its method will not be
         *  invoked).
         *
         * @param {Any} ...
         *  Other parameters that will be passed to the specified method.
         */
        function invokeModelMethod(methodName, excludeModel) {
            var args = _.toArray(arguments).slice(invokeModelMethod.length);
            _.each(tableModels, function (tableModel, tableName) {
                if ((tableModel !== excludeModel) && (tableModel[methodName].apply(tableModel, args) === false)) {
                    self.deleteTable(tableName);
                }
            });
        }

        /**
         * Recalculates the position of all table ranges, after columns or rows
         * have been inserted into or deleted from the sheet.
         */
        function transformRanges(interval, insert, columns) {
            invokeModelMethod('transformRange', null, interval, insert, columns);
        }

        // protected methods --------------------------------------------------

        /**
         * Creates and returns a cloned instance of this collection for the
         * specified sheet.
         *
         * @internal
         *  Used by the class SheetModel during clone construction. DO NOT CALL
         *  from external code!
         *
         * @param {SheetModel} targetModel
         *  The model instance of the new cloned sheet that will own the clone
         *  returned by this method.
         *
         * @returns {TableCollection}
         *  A clone of this collection, initialized for ownership by the passed
         *  sheet model.
         */
        this.clone = function (targetModel) {
            // construct a new collection, pass all own table models as hidden parameter
            return new TableCollection(app, targetModel, tableModels);
        };

        // public methods -----------------------------------------------------

        /**
         * Returns whether the specified table exists in this collection.
         *
         * @param {String} tableName
         *  The name of the table to check for. The empty string addresses the
         *  anonymous table range used to store filter settings for the
         *  standard auto filter of the sheet.
         *
         * @returns {Boolean}
         *  Whether the specified table exists in this collection.
         */
        this.hasTable = function (tableName) {
            return tableName.toUpperCase() in tableModels;
        };

        /**
         * Returns the model of the table with the specified name.
         *
         * @param {String} tableName
         *  The name of the table. The empty string addresses the anonymous
         *  table range used to store filter settings for the standard auto
         *  filter of the sheet.
         *
         * @returns {TableModel|Null}
         *  The model of the table with the specified name; or null, if no
         *  table exists with that name.
         */
        this.getTable = function (tableName) {
            return tableModels[tableName.toUpperCase()] || null;
        };

        /**
         * Invokes the passed iterator function for all table models contained
         * in this collection.
         *
         * @param {Function} iterator
         *  The iterator function invoked for all table models. Receives the
         *  current table model instance as first parameter. If the iterator
         *  returns the Utils.BREAK object, the iteration process will be
         *  stopped immediately.
         *
         * @returns {Utils.BREAK|Undefined}
         *  A reference to the Utils.BREAK object, if the iterator has returned
         *  Utils.BREAK to stop the iteration process, otherwise undefined.
         */
        this.iterateTables = function (iterator) {
            return _.any(tableModels, function (tableModel) {
                return iterator.call(self, tableModel) === Utils.BREAK;
            }) ? Utils.BREAK : undefined;
        };

        /**
         * Returns the model of the table covering the specified cell.
         *
         * @param {Number[]} address
         *  The address of a cell.
         *
         * @returns {TableModel|Null}
         *  The model of the table at the specified position; or null, if the
         *  cell is not part of a table.
         */
        this.findTable = function (address) {
            return _.find(tableModels, function (tableModel) { return tableModel.containsCell(address); }) || null;
        };

        /**
         * Returns the models of all tables overlapping with the specified cell
         * range.
         *
         * @param {Object} range
         *  The address of a cell range.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.active=false]
         *      If set to true, the result returned by this method will be
         *      restricted to tables with activated filters/sorting (see method
         *      TableModel.isFilterActive() for details).
         *
         * @returns {Array}
         *  The models of all tables overlapping with the specified cell range.
         */
        this.findTables = function (range, options) {

            var // whether to only find tables with active filter/sorting
                active = Utils.getBooleanOption(options, 'active', false);

            return _.filter(tableModels, function (tableModel) {
                return tableModel.overlapsRange(range) && (!active || tableModel.isFilterActive());
            });
        };

        /**
         * Creates and inserts a new table into this collection.
         *
         * @param {String} tableName
         *  The name of the table. The empty string addresses the anonymous
         *  table range used to store filter settings for the standard auto
         *  filter of the sheet. The name must not be used yet in this table
         *  collection, otherwise insertion of the table will fail.
         *
         * @param {Array} tableRange
         *  The address of the cell range covered by the new table. Must not
         *  overlap with an existing table, otherwise insertion of the table
         *  will fail.
         *
         * @param {Object} attrs
         *  The initial table attribute set of the new table (style sheet
         *  reference, and explicit table attributes).
         *
         * @returns {Boolean}
         *  Whether the table has been created successfully.
         */
        this.insertTable = function (tableName, tableRange, attrs) {

            // check whether the name exists already
            tableName = tableName.toUpperCase();
            if (tableName in tableModels) { return false; }

            // check whether the cell range covers another table
            if (_.any(tableModels, function (tableModel) { return tableModel.overlapsRange(tableRange); })) { return false; }

            // create and insert the new collection entry
            var tableModel = new TableModel(app, sheetModel, tableName, tableRange, attrs);
            insertTableModel(tableModel);

            // notify listeners
            this.trigger('insert:table', tableModel);
            return true;
        };

        /**
         * Changes the position and/or attributes of an existing table.
         *
         * @param {String} tableName
         *  The name of the table to be modified. The empty string addresses
         *  the anonymous table range used to store filter settings for the
         *  standard auto filter of the sheet.
         *
         * @param {Object|Null} tableRange
         *  The new address of the cell range covered by the table. If set to
         *  null, the original range will not be modified. The new range must
         *  not overlap with another existing table, otherwise modifying the
         *  table will fail.
         *
         * @param {Object|Null} attrs
         *  The changed table attributes (style sheet reference, and explicit
         *  table attributes). Omitted attributes will not be changed. If set
         *  to null, the current table attributes will not be modified at all.
         *
         * @returns {Boolean}
         *  Whether the validation settings have been changed successfully.
         */
        this.changeTable = function (tableName, tableRange, attrs) {

            var // the collection entry to be modified
                tableModel = this.getTable(tableName);

            // validate passed table name
            if (!_.isObject(tableModel)) { return false; }

            // change table range
            if (_.isObject(tableRange)) {

                // check whether the cell range covers another table
                if (_.any(tableModels, function (entry) {
                    return (entry !== tableModel) && entry.overlapsRange(tableRange);
                })) { return false; }

                // set the new ranges at the collection entry
                tableModel.setRange(tableRange);
            }

            // change all other table attributes (change events will be forwarded)
            if (_.isObject(attrs)) {
                tableModel.setAttributes(attrs);
            }

            return true;
        };

        /**
         * Deletes a table from this collection.
         *
         * @param {String} tableName
         *  The name of the table to be deleted. The empty string addresses the
         *  anonymous table range used to store filter settings for the
         *  standard auto filter of the sheet.
         *
         * @returns {Boolean}
         *  Whether the table has been deleted successfully.
         */
        this.deleteTable = function (tableName) {

            var // the collection entry to be deleted
                tableModel = this.getTable(tableName);

            // validate passed table name
            if (!_.isObject(tableModel)) { return false; }

            // remove the collection entry
            delete tableModels[tableName.toUpperCase()];

            // notify listeners, and destroy the model
            this.trigger('delete:table', tableModel);
            tableModel.destroy();
            return true;
        };

        // initialization -----------------------------------------------------

        // update ranges after inserting/deleting columns or rows in the own sheet
        sheetModel.registerTransformationHandler(transformRanges);

        // clone collection entries passed as hidden argument by the clone() method
        if (_.isObject(arguments[TableCollection.length])) {
            _.each(arguments[TableCollection.length], function (tableModel) {
                insertTableModel(tableModel.clone(sheetModel));
            });
        }

        // destroy all class members
        this.registerDestructor(function () {
            _.invoke(tableModels, 'destroy');
            app = sheetModel = self = tableModels = null;
        });

    } // class TableCollection

    // exports ================================================================

    // derive this class from class ModelObject
    return ModelObject.extend({ constructor: TableCollection });

});
