/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/sheetoperationsgenerator', [
    'io.ox/office/tk/utils',
    'io.ox/office/editframework/model/operationsgenerator'
], function (Utils, OperationsGenerator) {

    'use strict';

    // class SheetOperationsGenerator =========================================

    /**
     * An operations generator for a specific sheet. Provides additional
     * methods to generate operations for the passed sheet model.
     *
     * @constructor
     *
     * @extends SpreadsheetOperationsGenerator
     */
    function SheetOperationsGenerator(app, sheet) {

        // base constructor ---------------------------------------------------

        OperationsGenerator.call(this, app);

        // public methods -----------------------------------------------------

        /**
         * Creates and appends a new operation to the operations array, which
         * will contain a property 'sheet' set to the index of the sheet this
         * generator is associated to.
         *
         * @param {String} name
         *  The name of the operation.
         *
         * @param {Object} [properties]
         *  Additional properties that will be stored in the operation.
         *
         * @returns {Object}
         *  The created operation object.
         */
        this.generateSheetOperation = function (name, properties) {
            properties = _.extend({ sheet: sheet }, properties);
            return this.generateOperation(name, properties);
        };

        /**
         * Creates and appends a new operation with 'start' and (optional)
         * 'end' properties to the operations array addressing a column or row
         * interval. If first and last index of the interval are equal, the
         * 'end' property will be omitted in the operation.
         *
         * @param {String} name
         *  The name of the operation.
         *
         * @param {Object} interval
         *  The zero-based column/row indexes of the interval, in the
         *  properties 'first' and 'last'.
         *
         * @param {Object} [properties]
         *  Additional properties that will be stored in the operation.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.complete=false]
         *      If set to true, the property 'end' will be written, if it is
         *      equal to the property 'start'. By default, the property will be
         *      omitted in that case.
         *
         * @returns {Object}
         *  The created operation object.
         */
        this.generateIntervalOperation = function (name, interval, properties, options) {
            // add 'start' and 'end' properties to the operation
            properties = _.extend({ start: interval.first }, properties);
            if ((interval.first < interval.last) || Utils.getBooleanOption(options, 'complete', false)) {
                properties.end = interval.last;
            }
            return this.generateSheetOperation(name, properties);
        };

        /**
         * Creates and appends a new operation with a 'start' property to the
         * operations array addressing a single cell.
         *
         * @param {String} name
         *  The name of the operation.
         *
         * @param {Number[]} address
         *  The address of the cell targeted by the operation.
         *
         * @param {Object} [properties]
         *  Additional properties that will be stored in the operation.
         *
         * @returns {Object}
         *  The created operation object.
         */
        this.generateCellOperation = function (name, address, properties) {
            properties = _.extend({ start: address }, properties);
            return this.generateSheetOperation(name, properties);
        };

        /**
         * Creates and appends a new operation with 'start' and (optional)
         * 'end' properties to the operations array addressing a cell range. If
         * the range addresses a single cell, the 'end' property will be
         * omitted in the operation.
         *
         * @param {String} name
         *  The name of the operation.
         *
         * @param {Object} range
         *  The address of the cell range addressed by the operation.
         *
         * @param {Object} [properties]
         *  Additional properties that will be stored in the operation.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.complete=false]
         *      If set to true, the property 'end' will be written, if it is
         *      equal to the property 'start'. By default, the property will be
         *      omitted in that case.
         *
         * @returns {Object}
         *  The created operation object.
         */
        this.generateRangeOperation = function (name, range, properties, options) {
            // add 'start' and 'end' properties to the operation
            properties = _.extend({ start: range.start }, properties);
            if (!_.isEqual(range.start, range.end) || Utils.getBooleanOption(options, 'complete', false)) {
                properties.end = range.end;
            }
            return this.generateSheetOperation(name, properties);
        };

        /**
         * Creates and appends a new operation with a 'table' property to the
         * operations array addressing a table range.
         *
         * @param {String} name
         *  The name of the operation.
         *
         * @param {String} tableName
         *  The name of a table range. The empty string addresses the anonymous
         *  table range used to store filter settings for the standard auto
         *  filter of the sheet (no 'table' property will be inserted into the
         *  generated operation).
         *
         * @param {Object} [properties]
         *  Additional properties that will be stored in the operation.
         *
         * @returns {Object}
         *  The created operation object.
         */
        this.generateTableOperation = function (name, tableName, properties) {
            if (tableName.length > 0) { properties = _.extend({ table: tableName }, properties); }
            return this.generateSheetOperation(name, properties);
        };

        /**
         * Creates and appends a new operation with a 'start' property to the
         * operations array addressing a drawing object.
         *
         * @param {String} name
         *  The name of the operation.
         *
         * @param {Number[]} position
         *  The position of the drawing object in the sheet.
         *
         * @param {Object} [properties]
         *  Additional properties that will be stored in the operation.
         *
         * @returns {Object}
         *  The created operation object.
         */
        this.generateDrawingOperation = function (name, position, properties) {
            properties = _.extend({ start: [sheet].concat(position) }, properties);
            return this.generateOperation(name, properties);
        };

    } // class SheetOperationsGenerator

    // exports ================================================================

    // derive this class from class OperationsGenerator
    return OperationsGenerator.extend({ constructor: SheetOperationsGenerator });

});
