/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/namecollection', [
    'io.ox/office/tk/utils',
    'io.ox/office/baseframework/model/modelobject',
    'io.ox/office/spreadsheet/model/formula/tokenarray'
], function (Utils, ModelObject, TokenArray) {

    'use strict';

    // private class NameModel ================================================

    /**
     * Stores settings for a single defined name in a specific sheet, or for a
     * global defined name in the document.
     *
     * @constructor
     *
     * @extends ModelObject
     *
     * @param {SpreadsheetApplication} app
     *  The application that contains this defined name.
     *
     * @param {SheetModel} [sheetModel]
     *  The model of the sheet that contains this defined name. If omitted,
     *  this instance is a global document name.
     *
     * @param {String} formula
     *  The formula expression bound to the defined name.
     */
    var NameModel = ModelObject.extend({ constructor: function (app, sheetModel, formula) {

        var // the token array representing the definition of this name
            tokenArray = new TokenArray(app, sheetModel);

        // base constructor ---------------------------------------------------

        ModelObject.call(this, app);

        // protected methods --------------------------------------------------

        /**
         * Creates and returns a cloned instance of this name for the specified
         * sheet.
         *
         * @internal
         *  Used by the class NameCollection during clone construction. DO NOT
         *  CALL from external code!
         *
         * @param {SheetModel} targetModel
         *  The model instance of the new cloned sheet that will own the clone
         *  returned by this method.
         *
         * @returns {NameModel}
         *  A clone of this name model, initialized for ownership by the passed
         *  sheet model.
         */
        this.clone = function (targetModel) {
            // construct a new name model and relocate sheet references
            var nameModel = new NameModel(app, targetModel, tokenArray.getFormula());
            nameModel.getTokenArray().relocateSheet(sheetModel.getIndex(), targetModel.getIndex());
            return nameModel;
        };

        // public methods -----------------------------------------------------

        /**
         * Returns the token array representing the formula definition of this
         * defined name.
         *
         * @returns {TokenArray}
         *  The token array containing the name definition.
         */
        this.getTokenArray = function () {
            return tokenArray;
        };

        /**
         * Changes the formula definition of this defined name.
         *
         * @param {String} newFormula
         *  The new formula expression bound to the defined name.
         *
         * @returns {NameModel}
         *  A reference to this instance.
         */
        this.setDefinition = function (newFormula) {
            tokenArray.parseFormula(newFormula);
            return this;
        };

        // initialization -----------------------------------------------------

        // parse formula definition
        this.setDefinition(formula);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            tokenArray.destroy();
            app = sheetModel = tokenArray = null;
        });

    }}); // class NameModel

    // class NameCollection ===================================================

    /**
     * Collects all global defined names of the spreadsheet document, or the
     * defined names contained in a single sheet of the document.
     *
     * Triggers the following events:
     * - 'insert:name'
     *      After a new defined name has been inserted into the collection.
     *      Event handlers receive the name of the new defined name.
     * - 'change:name'
     *      After the formula definition of a defined name has been modified.
     *      Event handlers receive the name of the changed defined name.
     * - 'delete:name'
     *      After a defined name has been deleted from the collection. Event
     *      handlers receive the name of the deleted defined name.
     *
     * @constructor
     *
     * @extends ModelObject
     *
     * @param {SpreadsheetApplication} app
     *  The application that contains this collection instance.
     *
     * @param {SheetModel} [sheetModel]
     *  The model of the sheet that contains this name collection. If omitted,
     *  the name collection is intended to store global document names.
     */
    function NameCollection(app, sheetModel) {

        var // the definitions of all defined names, mapped by upper-case name
            entries = {};

        // base constructor ---------------------------------------------------

        ModelObject.call(this, app);

        // protected methods --------------------------------------------------

        /**
         * Creates and returns a cloned instance of this name collection for
         * the specified sheet.
         *
         * @internal
         *  Used by the class SheetModel during clone construction. DO NOT CALL
         *  from external code!
         *
         * @param {SheetModel} targetModel
         *  The model instance of the new cloned sheet that will own the clone
         *  returned by this method.
         *
         * @returns {NameCollection}
         *  A clone of this name collection, initialized for ownership by the
         *  passed sheet model.
         */
        this.clone = function (targetModel) {
            // construct a new collection, pass all own names as hidden parameter
            return new NameCollection(app, targetModel, entries);
        };

        // public methods -----------------------------------------------------

        /**
         * Returns the token array representing the formula definition of the
         * specified defined name.
         *
         * @param {String} name
         *  The name of the new defined name whose token array will be
         *  returned.
         *
         * @returns {TokenArray|Null}
         *  The token array containing the name definition of the specified
         *  name, if existing; otherwise null.
         */
        this.getTokenArray = function (name) {
            var entry = entries[name.toUpperCase()];
            return entry ? entry.getTokenArray() : null;
        };

        /**
         * Creates and stores a new defined name, and triggers an 'insert:name'
         * event.
         *
         * @param {String} name
         *  The name of the new defined name.
         *
         * @param {String} formula
         *  The formula expression bound to the defined name.
         *
         * @returns {Boolean}
         *  Whether the defined name has been created successfully.
         */
        this.insertName = function (name, formula) {

            // validate input parameters (name must not exist)
            name = name.toUpperCase();
            if ((name.length === 0) || (name in entries)) {
                return false;
            }

            // create a new collection entry and notify listeners
            entries[name] = new NameModel(app, sheetModel, formula);
            this.trigger('insert:name', name);
            return true;
        };

        /**
         * Changes the definition of an existing defined name, and triggers a
         * 'change:name' event.
         *
         * @param {String} name
         *  The name of the defined name to be changed.
         *
         * @param {String} formula
         *  The new formula expression bound to the defined name.
         *
         * @returns {Boolean}
         *  Whether the defined name has been changed successfully.
         */
        this.changeName = function (name, formula) {

            // validate input parameters (name must exist)
            name = name.toUpperCase();
            if (!(name in entries)) { return false; }

            // change the collection entry and notify listeners
            entries[name].setDefinition(formula);
            this.trigger('change:name', name);
            return true;
        };

        /**
         * Deletes a defined name from this collection, and triggers a
         * 'delete:name' event.
         *
         * @param {String} name
         *  The name of the new defined name.
         *
         * @returns {Boolean}
         *  Whether the defined name has been deleted successfully.
         */
        this.deleteName = function (name) {

            // validate input parameters (name must exist)
            name = name.toUpperCase();
            if (!(name in entries)) { return false; }

            // delete the collection entry and notify listeners
            entries[name].destroy();
            delete entries[name];
            this.trigger('delete:name', name);
            return true;
        };

        // initialization -----------------------------------------------------

        // clone collection entries passed as hidden argument by the clone() method
        if (_.isObject(arguments[NameCollection.length])) {
            _.each(arguments[NameCollection.length], function (sourceEntry, key) {
                entries[key] = sourceEntry.clone(sheetModel);
            });
        }

        // destroy all class members on destruction
        this.registerDestructor(function () {
            _.invoke(entries, 'destroy');
            sheetModel = entries = null;
        });

    } // class NameCollection

    // exports ================================================================

    // derive this class from class ModelObject
    return ModelObject.extend({ constructor: NameCollection });

});
