/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/model', [
    'io.ox/office/tk/utils',
    'io.ox/office/editframework/model/editmodel',
    'io.ox/office/editframework/model/operationsgenerator',
    'io.ox/office/editframework/model/undomanager',
    'io.ox/office/spreadsheet/utils/operations',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/model/documentstyles',
    'io.ox/office/spreadsheet/model/numberformatter',
    'io.ox/office/spreadsheet/model/namecollection',
    'io.ox/office/spreadsheet/model/sheetmodel',
    'io.ox/office/spreadsheet/model/viewsettingsmixin',
    'io.ox/office/spreadsheet/model/formula/compiler',
    'io.ox/office/spreadsheet/model/formula/interpreter'
], function (Utils, EditModel, OperationsGenerator, UndoManager, Operations, SheetUtils, SpreadsheetDocumentStyles, NumberFormatter, NameCollection, SheetModel, ViewSettingsMixin, FormulaCompiler, FormulaInterpreter) {

    'use strict';

    var // definitions for document view attributes
        DOCUMENT_VIEW_ATTRIBUTES = {

            /**
             * Index of the active (visible) sheet in the document.
             */
            activeSheet: {
                def: -1,
                validate: function (sheet) {
                    return ((0 <= sheet) && (sheet < this.getSheetCount())) ? sheet : Utils.BREAK;
                }
            },

            /**
             * An array of active remote clients with selection settings. See
             * method EditApplication.getActiveClients() for details about the
             * contents of this array. The local client must not be part of the
             * array.
             */
            remoteClients: {
                def: [],
                validate: function (clients) { return _.isArray(clients) ? clients : []; }
            }
        },

        // maps sheet event names to different document event names
        SHEET_TO_DOCUMENT_EVENTS = {
            'change:attributes': 'change:sheet:attributes',
            'change:viewattributes': 'change:sheet:viewattributes'
        };

    // class SpreadsheetModel =================================================

    /**
     * Represents a 'workbook', the spreadsheet document model. Implements
     * execution of all supported operations.
     *
     * Triggers the events supported by the base class EditModel, and the
     * following additional events:
     *  - 'insert:sheet:before':
     *      Before a new sheet will be inserted into the document. The event
     *      handler receives the zero-based index of the new sheet it will have
     *      in the collection of all sheets.
     *  - 'insert:sheet:after':
     *      After a new sheet has been inserted into the document. The event
     *      handler receives the zero-based index of the new sheet in the
     *      collection of all sheets, and the new sheet model instance.
     *  - 'delete:sheet:before':
     *      Before a sheet will be removed from the document. The event handler
     *      receives the zero-based index of the sheet in the collection of all
     *      sheets, and the model instance of the sheet to be removed.
     *  - 'delete:sheet:after':
     *      After a sheet has been removed from the document. The event handler
     *      receives the zero-based index of the sheet in the collection of all
     *      sheets before it has been removed.
     *  - 'move:sheet:before':
     *      Before a sheet will be moved to a new position. The event handler
     *      receives the zero-based index of the new position in the collection
     *      of all sheets, the zero-based index of the old position, and the
     *      model instance of the moved sheet.
     *  - 'move:sheet:after':
     *      After a sheet has been moved to a new position. The event handler
     *      receives the zero-based index of the new position in the collection
     *      of all sheets, the zero-based index of the old position, and the
     *      model instance of the moved sheet.
     *  - 'rename:sheet':
     *      After a sheet has been renamed. The event handler receives the
     *      zero-based index of the renamed sheet in the collection of all
     *      sheets, the new name of the sheet, and the sheet mode instance.
     * - 'change:viewattributes'
     *      After at least one view attribute of the document has been changed.
     *      Event handlers receive an incomplete attribute map containing all
     *      changed view attributes with their new values.
     * These events will always be triggered, also during the document import
     * process.
     *
     * @constructor
     *
     * @extends EditModel
     *
     * @param {SpreadsheetApplication} app
     *  The application containing this document model.
     */
    function SpreadsheetModel(app) {

        var // self reference
            self = this,

            // all existing sheets and their names, by zero-based index
            sheetsCollection = [],

            // the undo manager of this document
            undoManager = new UndoManager(app, { remoteUndo: true }),

            // container for all style sheets of all attribute families
            documentStyles = new SpreadsheetDocumentStyles(app),

            // the number formatter for this document
            numberFormatter = new NumberFormatter(app),

            // the formula compiler for this document
            formulaCompiler = new FormulaCompiler(app),

            // the formula interpreter for this document
            formulaInterpreter = new FormulaInterpreter(app),

            // the collection of globally defined names
            nameCollection = new NameCollection(app),

            // handlers to adjust range addresses after inserting/deleting columns/rows
            transformationHandlers = [],

            // the maximum column/row index in a sheet
            maxCol = 0, maxRow = 0;

        // base constructors --------------------------------------------------

        EditModel.call(this, app, OperationsGenerator, undoManager, documentStyles);
        ViewSettingsMixin.call(this, app, DOCUMENT_VIEW_ATTRIBUTES);

        // private methods ----------------------------------------------------

        /**
         * Updates the maximum available column and row index.
         */
        function updateMaxColRow(attributes) {
            maxCol = attributes.cols - 1;
            maxRow = attributes.rows - 1;
        }

        /**
         * Returns the sheet info structure at the specified index.
         *
         * @param {Number|String} sheet
         *  The zero-based index of the sheet, or the case-insensitive sheet
         *  name.
         *
         * @returns {Object|Null}
         *  The info structure of the sheet at the passed index or with the
         *  passed name, containing the sheet name and the reference to the
         *  sheet model instance, or null if the passed index or name is
         *  invalid.
         */
        function getSheetInfo(sheet) {
            sheet = _.isNumber(sheet) ? sheet : _.isString(sheet) ? self.getSheetIndex(sheet) : -1;
            return ((0 <= sheet) && (sheet < sheetsCollection.length)) ? sheetsCollection[sheet] : null;
        }

        /**
         * Returns an error code if the passed sheet name is not valid (empty,
         * or with invalid characters, or already existing in the document).
         *
         * @param {String} sheetName
         *  The desired name of the sheet (case-insensitive).
         *
         * @param {Number} [sheet]
         *  If specified, the zero-based index of the sheet that is allowed to
         *  contain the passed name (used while renaming an existing sheet).
         *
         * @returns {String}
         *  The empty string, if the sheet name is valid; otherwise one of the
         *  following error codes:
         *  - 'sheetname:empty': The passed sheet name is empty.
         *  - 'sheetname:invalid': The passed sheet name contains invalid
         *      characters.
         *  - 'sheetname:used': The passed sheet name is already used in the
         *      document.
         */
        function validateSheetName(sheetName, sheet) {

            var // an existing sheet with the passed name
                sheetModelByName = self.getSheetModel(sheetName),
                // the model of the specified sheet
                sheetModelByIndex = _.isNumber(sheet) ? self.getSheetModel(sheet) : null;

            // name must not be empty
            if (sheetName.length === 0) {
                return 'sheetname:empty';
            }

            // name must not start nor end with an apostrophe (embedded apostrophes
            // are allowed) and must not contain any NPCs or other invalid characters
            if (/^'|[\x00-\x1f\x80-\x9f\[\]*?:\/\\]|'$/.test(sheetName)) {
                return 'sheetname:invalid';
            }

            // name must not be used already for another sheet
            if (_.isObject(sheetModelByName) && (!_.isObject(sheetModelByIndex) || (sheetModelByName !== sheetModelByIndex))) {
                return 'sheetname:used';
            }

            return '';
        }

        /**
         * Returns whether the passed sheet name is valid (non-empty, no
         * invalid characters, and not existing in the document).
         *
         * @param {String} sheetName
         *  The desired name of the sheet (case-insensitive).
         *
         * @param {Number} [sheet]
         *  If specified, the zero-based index of the sheet that is allowed to
         *  contain the passed name (used while renaming an existing sheet).
         *
         * @returns {Boolean}
         *  Whether the sheet name is valid and not used.
         */
        function isValidSheetName(sheetName, sheet) {
            return validateSheetName(sheetName, sheet).length === 0;
        }

        /**
         * Registers own event handlers after a new sheet has been inserted
         * into this document.
         */
        function insertSheetHandler(event, sheet) {

            var // the new sheet model
                sheetModel = self.getSheetModel(sheet);

            // trigger all events of the new sheet at this document model
            sheetModel.on('triggered', function (event2, type) {
                var triggerType = Utils.getStringOption(SHEET_TO_DOCUMENT_EVENTS, type, type);
                self.trigger.apply(self, [triggerType, sheetModel.getIndex()].concat(_.toArray(arguments).slice(2)));
            });

            // invoke all range transformation handlers
            sheetModel.registerTransformationHandler(function (interval, insert, columns) {
                var sheetIndex = sheetModel.getIndex();
                _.each(transformationHandlers, function (handler) {
                    handler.call(self, sheetIndex, interval, insert, columns);
                });
            });
        }

        /**
         * Sends the own selection settings to remote clients.
         */
        function sendActiveSelection() {

            var // index of the active sheet
                activeSheet = self.getActiveSheet(),
                // the model of the active sheet
                sheetModel = self.getSheetModel(activeSheet);

            // do not send anything while settings are invalid
            if (sheetModel) {
                app.updateUserData({ sheet: activeSheet, selection: sheetModel.getViewAttribute('selection') });
            }
        }

        /**
         * Handles 'docs:users' events containing the selections of all remote
         * clients.
         *
         * @param {Array} activeClients
         *  All active clients viewing or editing this document.
         */
        function changeActiveClientsHandler(activeClients) {

            var // filter for remote clients with existing selection data
                remoteClients = _.filter(activeClients, function (client) {
                    return client.remote && _.isObject(client.userData) && _.isNumber(client.userData.sheet) && _.isObject(client.userData.selection);
                });

            // store selections in the document (this triggers a change event which causes rendering)
            self.setViewAttribute('remoteClients', remoteClients);
        }

        /**
         * Tries to extract a cell range address from the passed operation.
         *
         * @param {Object} operation
         *  Any operation.
         *
         * @returns {Object|Null}
         *  The address of a cell range, if the operation contains a 'start'
         *  property, and an optional 'end'property, and if these properties
         *  form a valid range address; otherwise null.
         */
        function getRangeProperty(operation) {
            var range = { start: Utils.getArrayOption(operation, 'start') };
            range.end = Utils.getArrayOption(operation, 'end', _.clone(range.start));
            return self.isValidRange(range) ? range : null;
        }

        /**
         * Tries to extract a list of range addresses from a property of the
         * passed operation.
         *
         * @param {Object} operation
         *  Any operation.
         *
         * @param {String} name
         *  The name of the operation property containing a range list. Each
         *  range object in the array may be incomplete (its 'end' property may
         *  be omitted).
         *
         * @returns {Array|Null}
         *  A complete range list (all ranges contain 'end' properties), if the
         *  operation property is an array with valid range addresses;
         *  otherwise null.
         */
        function getRangesProperty(operation, name) {

            var // the raw property value
                ranges = Utils.getArrayOption(operation, name);

            // ensure that 'ranges' is an array with objects
            if (!_.isArray(ranges) || (ranges.length === 0) || !_.all(ranges, _.isObject)) { return null; }

            // add 'end' properties
            _.each(ranges, function (range) {
                if (!('end' in range)) { range.end = _.clone(range.start); }
            });

            // check validity of all range addresses
            return _.all(ranges, _.bind(self.isValidRange, self)) ? ranges : null;
        }

        // methods ------------------------------------------------------------

        /**
         * Registers a handler function that will be invoked when an operation
         * with the specified name will be applied that addresses a specific
         * collection for defined names (either the global collection, or a
         * collection of a specific sheet in this document).
         *
         * @param {String} name
         *  The name of the operation, as contained in the 'name' attribute of
         *  the operation object.
         *
         * @param {Function} handler
         *  The function that will be invoked for the operation, if it does not
         *  contain a 'sheet' property (global name collection), or a valid
         *  'sheet' property addressing an existing sheet. Will be called in
         *  the context of this document model instance. Receives the following
         *  parameters:
         *  (1) {NameCollection} nameCollection
         *      The name collection instance addressed by the operation.
         *  (2) {Object} operation
         *      The complete operation object.
         *  (3) {Boolean} external
         *      The external state that has been passed to the method
         *      EditModel.applyOperations().
         *  See method EditModel.registerOperationHandler() for more details.
         *
         * @returns {SpreadsheetModel}
         *  A reference to this instance.
         */
        this.registerNameOperationHandler = function (name, handler) {
            this.registerOperationHandler(name, function (operation, external) {
                var sheet = Utils.getIntegerOption(operation, 'sheet'),
                    model = _.isNumber(sheet) ? this.getSheetModel(sheet) : this;
                return _.isObject(model) && handler.call(this, model.getNameCollection(), operation, external);
            });
            return this;
        };

        /**
         * Registers a handler function that will be invoked when an operation
         * with the specified name will be applied that addresses a specific
         * sheet.
         *
         * @param {String} name
         *  The name of the operation, as contained in the 'name' attribute of
         *  the operation object.
         *
         * @param {Function} handler
         *  The function that will be invoked for the operation, if it contains
         *  a valid 'sheet' property addressing an existing sheet. Will be
         *  called in the context of this document model instance. Receives the
         *  following parameters:
         *  (1) {SheetModel} sheetModel
         *      The sheet model instance addressed by the operation.
         *  (2) {Object} operation
         *      The complete operation object.
         *  (3) {Boolean} external
         *      The external state that has been passed to the method
         *      EditModel.applyOperations().
         *  See method EditModel.registerOperationHandler() for more details.
         *
         * @returns {SpreadsheetModel}
         *  A reference to this instance.
         */
        this.registerSheetOperationHandler = function (name, handler) {
            this.registerOperationHandler(name, function (operation, external) {
                var sheetModel = this.getSheetModel(Utils.getIntegerOption(operation, 'sheet', -1));
                return _.isObject(sheetModel) && handler.call(this, sheetModel, operation, external);
            });
            return this;
        };

        /**
         * Registers a handler function that will be invoked when an operation
         * with the specified name will be applied that addresses a specific
         * column/row interval in a specific sheet.
         *
         * @param {String} name
         *  The name of the operation, as contained in the 'name' attribute of
         *  the operation object.
         *
         * @param {String} type
         *  Whether the operation addresses 'columns' or 'rows'.
         *
         * @param {Function} handler
         *  The function that will be invoked for the operation, if it contains
         *  a valid 'sheet' property addressing an existing sheet, a valid
         *  'start' property addressing an existing column or row, and an
         *  optional 'end' property addressing an existing column or row that
         *  follows the start column/row. Will be called in the context of this
         *  document model instance. Receives the following parameters:
         *  (1) {SheetModel} sheetModel
         *      The sheet model instance addressed by the operation.
         *  (2) {Object} interval
         *      The column/row interval, in the properties 'first' and 'last'.
         *  (3) {Object} operation
         *      The complete operation object.
         *  (4) {Boolean} external
         *      The external state that has been passed to the method
         *      EditModel.applyOperations().
         *  See method EditModel.registerOperationHandler() for more details.
         *
         * @returns {SpreadsheetModel}
         *  A reference to this instance.
         */
        this.registerIntervalOperationHandler = function (name, type, handler) {
            this.registerSheetOperationHandler(name, function (sheetModel, operation, external) {
                var start = Utils.getIntegerOption(operation, 'start', -1),
                    end = Utils.getIntegerOption(operation, 'end', start),
                    max = (type === 'columns') ? maxCol : maxRow;
                return (0 <= start) && (start <= end) && (end <= max) &&
                    handler.call(this, sheetModel, { first: start, last: end }, operation, external);
            });
            return this;
        };

        /**
         * Registers a handler function that will be invoked when an operation
         * with the specified name will be applied that addresses a specific
         * single cell in a specific sheet.
         *
         * @param {String} name
         *  The name of the operation, as contained in the 'name' attribute of
         *  the operation object.
         *
         * @param {Function} handler
         *  The function that will be invoked for the operation, if it contains
         *  a valid 'sheet' property addressing an existing sheet, and a valid
         *  'start' property addressing a cell. Will be called in the context
         *  of this document model instance. Receives the following parameters:
         *  (1) {SheetModel} sheetModel
         *      The sheet model instance addressed by the operation.
         *  (2) {Number[]} address
         *      The cell address, from the property 'start'.
         *  (3) {Object} operation
         *      The complete operation object.
         *  (4) {Boolean} external
         *      The external state that has been passed to the method
         *      EditModel.applyOperations().
         *  See method EditModel.registerOperationHandler() for more details.
         *
         * @returns {SpreadsheetModel}
         *  A reference to this instance.
         */
        this.registerCellOperationHandler = function (name, handler) {
            this.registerSheetOperationHandler(name, function (sheetModel, operation, external) {
                var address = Utils.getArrayOption(operation, 'start');
                return self.isValidAddress(address) && handler.call(this, sheetModel, address, operation, external);
            });
            return this;
        };

        /**
         * Registers a handler function that will be invoked when an operation
         * with the specified name will be applied that addresses a specific
         * cell range in a specific sheet.
         *
         * @param {String} name
         *  The name of the operation, as contained in the 'name' attribute of
         *  the operation object.
         *
         * @param {Function} handler
         *  The function that will be invoked for the operation, if it contains
         *  a valid 'sheet' property addressing an existing sheet, a valid
         *  'start' property addressing a start cell, and an optional 'end'
         *  property addressing an end cell that is not located before the
         *  start cell. Will be called in the context of this document model
         *  instance. Receives the following parameters:
         *  (1) {SheetModel} sheetModel
         *      The sheet model instance addressed by the operation.
         *  (2) {Object} range
         *      The range address, in the properties 'start' and 'end'.
         *  (3) {Object} operation
         *      The complete operation object.
         *  (4) {Boolean} external
         *      The external state that has been passed to the method
         *      EditModel.applyOperations().
         *  See method EditModel.registerOperationHandler() for more details.
         *
         * @returns {SpreadsheetModel}
         *  A reference to this instance.
         */
        this.registerRangeOperationHandler = function (name, handler) {
            this.registerSheetOperationHandler(name, function (sheetModel, operation, external) {
                var range = getRangeProperty(operation);
                return _.isObject(range) && handler.call(this, sheetModel, range, operation, external);
            });
            return this;
        };

        /**
         * Registers a handler function that will be invoked when an operation
         * with the specified name will be applied that addresses an entire
         * list of cell ranges in a specific sheet.
         *
         * @param {String} name
         *  The name of the operation, as contained in the 'name' attribute of
         *  the operation object.
         *
         * @param {Function} handler
         *  The function that will be invoked for the operation, if it contains
         *  a valid 'sheet' property addressing an existing sheet, and an array
         *  property 'range' containing valid range addresses (valid 'start'
         *  property addressing a start cell, and an optional 'end' property
         *  addressing an end cell that is not located before the start cell).
         *  Will be called in the context of this document model instance.
         *  Receives the following parameters:
         *  (1) {SheetModel} sheetModel
         *      The sheet model instance addressed by the operation.
         *  (2) {Array} ranges
         *      The range addresses, as array of objects with the properties
         *      'start' and 'end'. Missing 'end' properties in the source
         *      operation have been added already in this array.
         *  (3) {Object} operation
         *      The complete operation object.
         *  (4) {Boolean} external
         *      The external state that has been passed to the method
         *      EditModel.applyOperations().
         *  See method EditModel.registerOperationHandler() for more details.
         *
         * @returns {SpreadsheetModel}
         *  A reference to this instance.
         */
        this.registerRangeListOperationHandler = function (name, handler) {
            this.registerSheetOperationHandler(name, function (sheetModel, operation, external) {
                var ranges = getRangesProperty(operation, 'ranges');
                return _.isArray(ranges) && handler.call(this, sheetModel, ranges, operation, external);
            });
            return this;
        };

        /**
         * Registers a handler function that will be invoked when an operation
         * with the specified name will be applied that addresses a specific
         * table range of a specific sheet in this document.
         *
         * @param {String} name
         *  The name of the operation, as contained in the 'name' attribute of
         *  the operation object.
         *
         * @param {Function} handler
         *  The function that will be invoked for the operation, if it contains
         *  a valid 'sheet' property addressing an existing sheet, and an
         *  optional string property 'table' (missing property will be
         *  interpreted as the anonymous table range for the auto-filter). Will
         *  be called in the context of this document model instance. Receives
         *  the following parameters:
         *  (1) {TableCollection} tableCollection
         *      The table collection instance addressed by the operation.
         *  (1) {String} tableName
         *      The name of the table range, as contained in the operation.
         *      Will be the empty string, if the table name has been omitted in
         *      the operation.
         *  (3) {Object} operation
         *      The complete operation object.
         *  (4) {Boolean} external
         *      The external state that has been passed to the method
         *      EditModel.applyOperations().
         *  See method EditModel.registerOperationHandler() for more details.
         *
         * @returns {SpreadsheetModel}
         *  A reference to this instance.
         */
        this.registerTableOperationHandler = function (name, handler) {
            this.registerSheetOperationHandler(name, function (sheetModel, operation, external) {
                var tableName = ('table' in operation) ? Utils.getStringOption(operation, 'table') : '';
                return _.isString(tableName) && handler.call(this, sheetModel.getTableCollection(), tableName, operation, external);
            });
            return this;
        };

        /**
         * Registers a handler function that will be invoked when an operation
         * with the specified name will be applied that addresses a drawing
         * object.
         *
         * @param {String} name
         *  The name of the operation, as contained in the 'name' attribute of
         *  the operation object.
         *
         * @param {Function} handler
         *  The function that will be invoked for the operation, if it contains
         *  a valid 'start' array property addressing an existing sheet with
         *  its first element. Will be called in the context of this document
         *  model instance. Receives the following parameters:
         *  (1) {SheetModel} sheetModel
         *      The sheet model instance addressed by the operation.
         *  (2) {Number} position
         *      The drawing position (the 'start' operation property without
         *      the first array element which was the sheet index).
         *  (3) {Object} operation
         *      The complete operation object.
         *  (4) {Boolean} external
         *      The external state that has been passed to the method
         *      EditModel.applyOperations().
         *  See method EditModel.registerOperationHandler() for more details.
         *
         * @returns {SpreadsheetModel}
         *  A reference to this instance.
         */
        this.registerDrawingOperationHandler = function (name, handler) {
            this.registerOperationHandler(name, function (operation, external) {
                var start = Utils.getArrayOption(operation, 'start', []),
                    sheetModel = ((start.length >= 2) && _.isNumber(start[0])) ? this.getSheetModel(start[0]) : null;
                return _.isObject(sheetModel) && handler.call(this, sheetModel, start.slice(1), operation, external);
            });
            return this;
        };

        /**
         * Registers an operation with the specified name that will invoke a
         * method of a drawing model instance addressed by the operation.
         *
         * @param {String} name
         *  The name of the operation, as contained in the 'name' attribute of
         *  the operation object.
         *
         * @param {String} methodName
         *  The name of the method that will be invoked at the drawing model
         *  instance. The method receives the entire operation object as first
         *  parameter, and the external state that has been passed to the
         *  method EditModel.applyOperations(). The invoked method must return
         *  a Boolean value indicating whether applying the operation has
         *  succeeded.
         *
         * @param {String} [type]
         *  The type of the drawing object supporting the operation. If
         *  omitted, the operation is supported for all drawing objects.
         *
         * @returns {SpreadsheetModel}
         *  A reference to this instance.
         */
        this.registerSimpleDrawingOperation = function (name, methodName, type) {
            this.registerDrawingOperationHandler(name, function (sheetModel, position, operation, external) {
                var drawingModel = sheetModel.getDrawingCollection().findModel(position, { deep: true, type: type });
                return _.isObject(drawingModel) && _.isFunction(drawingModel[methodName]) && drawingModel[methodName](operation, external);
            });
            return this;
        };

        /**
         * Called from the applications import-fail handler when document
         * import has failed. Inserts an empty sheet, if this model does not
         * contain any sheets. This helps preventing to write if-else
         * statements for empty documents at thousand places in the view code
         * (there will always be an active sheet with selection etc.).
         *
         * @internal
         *  Not intended to be called from any other code than the import
         *  callback handlers of the application.
         *
         * @returns {SpreadsheetModel}
         *  A reference to this instance.
         */
        this.prepareInvalidDocument = function () {
            var sheetModel = null;
            if (sheetsCollection.length === 0) {
                self.trigger('insert:sheet:before', 0);
                sheetModel = new SheetModel(app, 'worksheet');
                sheetsCollection.push({ model: sheetModel, name: 'Sheet1' });
                self.trigger('insert:sheet:after', 0, sheetModel);
            }
            return this;
        };

        /**
         * Returns the number formatter of this document.
         *
         * @returns {NumberFormatter}
         *  The number formatter of this document.
         */
        this.getNumberFormatter = function () {
            return numberFormatter;
        };

        /**
         * Returns the formula compiler of this document.
         *
         * @returns {FormulaCompiler}
         *  The formula compiler of this document.
         */
        this.getFormulaCompiler = function () {
            return formulaCompiler;
        };

        /**
         * Returns the formula interpreter of this document.
         *
         * @returns {FormulaInterpreter}
         *  The formula interpreter of this document.
         */
        this.getFormulaInterpreter = function () {
            return formulaInterpreter;
        };

        /**
         * Returns the collection of all global defined names contained in this
         * document.
         *
         * @returns {NameCollection}
         *  The collection of all defined names in this document.
         */
        this.getNameCollection = function () {
            return nameCollection;
        };

        // sheet range --------------------------------------------------------

        /**
         * Returns the column index of the rightmost cells in a sheet.
         *
         * @returns {Number}
         *  The column index of the rightmost cells in a sheet.
         */
        this.getMaxCol = function () {
            return maxCol;
        };

        /**
         * Returns the row index of the bottom cells in a sheet.
         *
         * @returns {Number}
         *  The row index of the bottom cells in a sheet.
         */
        this.getMaxRow = function () {
            return maxRow;
        };

        /**
         * Returns the maximum column or row index in a sheet.
         *
         * @param {Boolean} columns
         *  Whether to return the maximum column index (true), or the maximum
         *  row index (false).
         *
         * @returns {Number}
         *  The maximum column or row index in a sheet.
         */
        this.getMaxIndex = function (columns) {
            return columns ? maxCol : maxRow;
        };

        /**
         * Returns the range address of the entire area in a sheet, from the
         * top-left cell to the bottom-right cell.
         *
         * @returns {Object}
         *  The range address of an entire sheet.
         */
        this.getSheetRange = function () {
            return { start: [0, 0], end: [maxCol, maxRow] };
        };

        /**
         * Returns whether the passed range address covers one or more entire
         * columns in the active sheet.
         *
         * @param {Object} range
         *  A range address.
         *
         * @returns {Boolean}
         *  Whether the passed range address covers one or more entire columns.
         */
        this.isColRange = function (range) {
            return (range.start[1] === 0) && (range.end[1] === maxRow);
        };

        /**
         * Returns the address of the cell range covering the specified column
         * interval.
         *
         * @param {Object|Number} interval
         *  The column interval, in the zero-based column index properties
         *  'first' and 'last', or a single zero-based column index.
         *
         * @returns {Object}
         *  The address of the column range.
         */
        this.makeColRange = function (interval) {
            return {
                start: [_.isNumber(interval) ? interval : interval.first, 0],
                end: [_.isNumber(interval) ? interval : interval.last, maxRow]
            };
        };

        /**
         * Returns whether the passed range address covers one or more entire
         * rows in the active sheet.
         *
         * @param {Object} range
         *  A range address.
         *
         * @returns {Boolean}
         *  Whether the passed range address covers one or more entire rows.
         */
        this.isRowRange = function (range) {
            return (range.start[0] === 0) && (range.end[0] === maxCol);
        };

        /**
         * Returns the address of the cell range covering the specified row
         * interval.
         *
         * @param {Object|Number} interval
         *  The row interval, in the zero-based row index properties 'first'
         *  and 'last', or a single zero-based row index.
         *
         * @returns {Object}
         *  The address of the row range.
         */
        this.makeRowRange = function (interval) {
            return {
                start: [0, _.isNumber(interval) ? interval : interval.first],
                end: [maxCol, _.isNumber(interval) ? interval : interval.last]
            };
        };

        /**
         * Returns whether the passed range address covers one or more entire
         * columns or rows in the active sheet.
         *
         * @param {Object} range
         *  A range address.
         *
         * @param {Boolean} columns
         *  Whether to check for a full column range (true), or for a full row
         *  range (false).
         *
         * @returns {Boolean}
         *  Whether the passed range address covers one or more entire columns
         *  or rows.
         */
        this.isFullRange = function (range, columns) {
            return columns ? this.isColRange(range) : this.isRowRange(range);
        };

        /**
         * Returns the address of the cell range covering the specified column
         * or row interval.
         *
         * @param {Object|Number} interval
         *  The column or row interval, in the zero-based column index
         *  properties 'first' and 'last', or a single zero-based column or row
         *  index.
         *
         * @param {Boolean} columns
         *  Whether to create a full column range (true), or a full row range
         *  (false).
         *
         * @returns {Object}
         *  The address of the full column or row range.
         */
        this.makeFullRange = function (interval, columns) {
            return columns ? this.makeColRange(interval) : this.makeRowRange(interval);
        };

        /**
         * Returns whether the passed value is a valid cell address.
         *
         * @param {Any} address
         *  Any value that will be checked whether it is a valid cell address.
         *
         * @returns {Boolean}
         *  Whether the passed value is an array with two non-negative integer
         *  elements. The first element (column index) must be less than or
         *  equal to the maximum column index of this document; the second
         *  element (row index) must be less than or equal to the maximum row
         *  index.
         */
        this.isValidAddress = function (address) {
            return _.isArray(address) && (address.length === 2) &&
                _.isFinite(address[0]) && (0 <= address[0]) && (address[0] <= maxCol) &&
                _.isFinite(address[1]) && (0 <= address[1]) && (address[1] <= maxRow);
        };

        /**
         * Returns whether the passed value is a valid range address.
         *
         * @param {Any} range
         *  Any value that will be checked whether it is a valid cell range
         *  address.
         *
         * @returns {Boolean}
         *  Whether the passed value is an object with the properties 'start'
         *  and 'end' containing valid cell addresses. The column indexes and
         *  row indexes must be ordered.
         */
        this.isValidRange = function (range) {
            return _.isObject(range) &&
                this.isValidAddress(range.start) && this.isValidAddress(range.end) &&
                (range.start[0] <= range.end[0]) && (range.start[1] <= range.end[1]);
        };

        /**
         * Returns a new range that has been moved by the specified distance,
         * but always keeping the range inside the sheet limits (see method
         * SpreadsheetModel.getCroppedMovedRange() for an alternative keeping
         * the specified move distance intact).
         *
         * @param {Object} range
         *  The address of the range to be moved.
         *
         * @param {Number} cols
         *  The number of columns the range will be moved. Negative values will
         *  move the range towards the leading border of the sheet, positive
         *  values will move the range towards the trailing border.
         *
         * @param {Number} rows
         *  The number of rows the range will be moved. Negative values will
         *  move the range towards the top border of the sheet, positive values
         *  will move the range towards the bottom border.
         *
         * @returns {Object}
         *  The address of the moved range. If the move distances passed to
         *  this method are too large, the range will only be moved to the
         *  sheet borders.
         */
        this.getBoundedMovedRange = function (range, cols, rows) {

            // reduce move distance to keep the range address valid
            cols = Utils.minMax(cols, -range.start[0], maxCol - range.end[0]);
            rows = Utils.minMax(rows, -range.start[1], maxRow - range.end[1]);

            // return a modified clone of the passed range
            return {
                start: [range.start[0] + cols, range.start[1] + rows],
                end: [range.end[0] + cols, range.end[1] + rows]
            };
        };

        /**
         * Returns a new range that has been moved by the specified distance.
         * If the moved range leaves the sheet area, it will be cropped at the
         * sheet borders (see method SpreadsheetModel.getBoundedMovedRange()
         * for an alternative keeping the original size of the range intact).
         *
         * @param {Object} range
         *  The address of the range to be moved.
         *
         * @param {Number} cols
         *  The number of columns the range will be moved. Negative values will
         *  move the range towards the leading border of the sheet, positive
         *  values will move the range towards the trailing border.
         *
         * @param {Number} rows
         *  The number of rows the range will be moved. Negative values will
         *  move the range towards the top border of the sheet, positive values
         *  will move the range towards the bottom border.
         *
         * @returns {Object|Null}
         *  The address of the moved range. If the specified move distances are
         *  too large, the range will be cropped to the sheet area. If the
         *  range has been moved completely outside the sheet, this method will
         *  return null.
         */
        this.getCroppedMovedRange = function (range, cols, rows) {

            var // the address components of the moved range
                col1 = Math.max(0, range.start[0] + cols),
                row1 = Math.max(0, range.start[1] + rows),
                col2 = Math.min(maxCol, range.end[0] + cols),
                row2 = Math.min(maxRow, range.end[1] + rows);

            // return null, if the resulting range becomes invalid
            return ((col1 <= col2) && (row1 <= row2)) ? { start: [col1, row1], end: [col2, row2] } : null;
        };

        /**
         * Transforms the passed column or row index according to the specified
         * column or row operation. Used to get the effective column/row index
         * after columns or rows have been insert into or deleted from a sheet.
         *
         * @param {Number} sourceIndex
         *  The column/row index to be transformed.
         *
         * @param {Object} operationInterval
         *  The column/row interval of the operation, with the index properties
         *  'first' and 'last'.
         *
         * @param {Boolean} insert
         *  Whether the columns/rows have been inserted into the sheet (true),
         *  or deleted from the sheet (false).
         *
         * @param {Boolean} columns
         *  Whether the passed index is a column index and the passed operation
         *  interval is a column interval (true), or index and interval are for
         *  rows (false).
         *
         * @returns {Number|Null}
         *  The transformed column/row index. If the column/row has been
         *  deleted completely while deleting columns/rows, or has been shifted
         *  outside the sheet completely while inserting columns/rows, null
         *  will be returned instead.
         */
        this.transformIndex = function (sourceIndex, operationInterval, insert, columns) {

            var // the length of the operation interval
                size = SheetUtils.getIntervalSize(operationInterval);

            if (insert) {
                // insert rows/columns (increase the index)
                if (operationInterval.first <= sourceIndex) {
                    sourceIndex += size;
                }
            } else {
                // delete rows/columns (decrease or invalidate the address index)
                if (operationInterval.last < sourceIndex) {
                    sourceIndex -= size;
                } else if (operationInterval.first <= sourceIndex) {
                    sourceIndex = -1;
                }
            }

            return (0 <= sourceIndex) && (sourceIndex <= this.getMaxIndex(columns)) ? sourceIndex : null;
        };

        /**
         * Transforms the passed cell address according to the specified column
         * or row operation. Used to get the effective cell address after
         * columns or rows have been insert into or deleted from a sheet.
         *
         * @param {Number[]} sourceAddress
         *  The address of the cell to be transformed.
         *
         * @param {Object} operationInterval
         *  The column/row interval of the operation, with the index properties
         *  'first' and 'last'.
         *
         * @param {Boolean} insert
         *  Whether the columns/rows have been inserted into the sheet (true),
         *  or deleted from the sheet (false).
         *
         * @param {Boolean} columns
         *  Whether the passed operation interval is a column interval (true),
         *  or a row interval (false).
         *
         * @returns {Number[]|Null}
         *  The address of the transformed cell. If the cell has been deleted
         *  completely while deleting columns/rows, or has been shifted outside
         *  the sheet completely while inserting columns/rows, null will be
         *  returned instead.
         */
        this.transformAddress = function (sourceAddress, operationInterval, insert, columns) {

            var // array index into the cell address
                addrIndex = columns ? 0 : 1,
                // the transformed index
                newIndex = this.transformIndex(sourceAddress[addrIndex], operationInterval, insert, columns),
                // the resulting cell address
                address = null;

            if (_.isNumber(newIndex)) {
                address = _.clone(sourceAddress);
                address[addrIndex] = newIndex;
            }
            return address;
        };

        /**
         * Transforms the passed column/row interval according to the specified
         * operation. Used to get the effective interval after columns or rows
         * have been insert into or deleted from a sheet.
         *
         * @param {Object} sourceInterval
         *  The column/row interval to be transformed, with the index
         *  properties 'first' and 'last'.
         *
         * @param {Object} operationInterval
         *  The column/row interval of the operation, with the index properties
         *  'first' and 'last'.
         *
         * @param {Boolean} insert
         *  Whether the columns/rows have been inserted into the sheet (true),
         *  or deleted from the sheet (false).
         *
         * @param {Boolean} columns
         *  Whether the passed operation interval is a column interval (true),
         *  or a row interval (false). Influences the maximum index allowed in
         *  the transformed interval.
         *
         * @param {Boolean} [expandEnd=false]
         *  If set to true, the end position of the interval will be expanded,
         *  if new columns/rows will be inserted exactly behind the interval.
         *
         * @returns {Object|Null}
         *  The transformed column/row interval. If the interval has been
         *  deleted completely while deleting columns/rows, or has been shifted
         *  outside the sheet completely while inserting columns/rows, null
         *  will be returned instead.
         */
        this.transformInterval = function (sourceInterval, operationInterval, insert, columns, expandEnd) {

            var // the length of the interval
                size = SheetUtils.getIntervalSize(operationInterval),
                // the resulting transformed interval
                targetInterval = _.clone(sourceInterval);

            if (insert) {
                // insert rows/columns (shift, or enlarge the range)
                if (operationInterval.first <= sourceInterval.first) { targetInterval.first += size; }
                if (operationInterval.first <= sourceInterval.last + (expandEnd ? 1 : 0)) { targetInterval.last = Math.min(sourceInterval.last + size, this.getMaxIndex(columns)); }
            } else {
                // delete rows/columns (shift, shrink, or delete the range)
                if (operationInterval.first < sourceInterval.first) { targetInterval.first = Math.max(operationInterval.first, sourceInterval.first - size); }
                if (operationInterval.first <= sourceInterval.last) { targetInterval.last = Math.max(sourceInterval.last - size, operationInterval.first - 1); }
            }

            // delete interval, if resulting start position exceeds end position (happens when deleting
            // the entire interval, or when shifting it outside the sheet while inserting)
            return (targetInterval.first <= targetInterval.last) ? targetInterval : null;
        };

        /**
         * Transforms the passed cell range address according to the specified
         * column or row operation. Used to get the effective range after
         * columns or rows have been insert into or deleted from a sheet.
         *
         * @param {Object} sourceRange
         *  The address of the cell range to be transformed.
         *
         * @param {Object} operationInterval
         *  The column/row interval of the operation, with the index properties
         *  'first' and 'last'.
         *
         * @param {Boolean} insert
         *  Whether the columns/rows have been inserted into the sheet (true),
         *  or deleted from the sheet (false).
         *
         * @param {Boolean} columns
         *  Whether the passed operation interval is a column interval (true),
         *  or a row interval (false).
         *
         * @param {Boolean} [expandEnd=false]
         *  If set to true, the end position of the range will be expanded, if
         *  new columns will be inserted exactly right of the range, or new
         *  rows will be inserted exactly below the range.
         *
         * @returns {Object|Null}
         *  The address of the transformed cell range. If the range has been
         *  deleted completely while deleting columns/rows, or has been shifted
         *  outside the sheet completely while inserting columns/rows, null
         *  will be returned instead.
         */
        this.transformRange = function (sourceRange, operationInterval, insert, columns, expandEnd) {

            var // the source column/row intervals from the range
                colInterval = SheetUtils.getColInterval(sourceRange),
                rowInterval = SheetUtils.getRowInterval(sourceRange),
                // the transformed interval
                resultInterval = this.transformInterval(columns ? colInterval : rowInterval, operationInterval, insert, columns, expandEnd);

            // assign result to the correct interval
            if (columns) { colInterval = resultInterval; } else { rowInterval = resultInterval; }

            // build and return the resulting range
            return _.isObject(resultInterval) ? SheetUtils.makeRangeFromIntervals(colInterval, rowInterval) : null;
        };

        /**
         * Transforms the passed array of column/row intervals according to the
         * specified operation. Used to get the effective intervals after
         * columns or rows have been insert into or deleted from a sheet.
         *
         * @param {Object|Array} sourceIntervals
         *  The column/row interval to be transformed, with the index
         *  properties 'first' and 'last', or an array of column/row intervals.
         *
         * @param {Object} operationInterval
         *  The column/row interval of the operation, with the index properties
         *  'first' and 'last'.
         *
         * @param {Boolean} insert
         *  Whether the columns/rows have been inserted into the sheet (true),
         *  or deleted from the sheet (false).
         *
         * @param {Boolean} columns
         *  Whether the passed operation interval is a column interval (true),
         *  or a row interval (false). Influences the maximum index allowed in
         *  the transformed interval.
         *
         * @param {String} [insertMode='expand']
         *  Specifies how to modify intervals for insert operations (parameter
         *  'insert' set to true). The following modes are supported:
         *  - 'expand' (default): If columns/rows will be inserted into the
         *      middle of an interval (between first and last entry of the
         *      interval), it will be expanded.
         *  - 'expandEnd': Similar to 'expand', but columns/rows inserted
         *      exactly at the end of the interval will cause to expand the
         *      interval too.
         *  - 'split': If columns/rows will be inserted into the middle of an
         *      interval (between first and last entry of the interval), it
         *      will be split into two intervals; the inserted columns/rows
         *      will not be part of the new intervals.
         *
         * @returns {Array}
         *  The transformed column/row intervals. Source intervals that have
         *  been deleted completely while deleting columns/rows, or that have
         *  been shifted outside the sheet completely while inserting
         *  columns/rows, will not be included (the resulting array may even be
         *  empty).
         */
        this.transformIntervals = function (sourceIntervals, operationInterval, insert, columns, insertMode) {

            var // whether to expand intervals at the end when inserting behind the interval
                expandEnd = insertMode === 'expandEnd',
                // whether to split intervals on insertion in the middle
                split = insertMode === 'split',
                // the resulting transformed intervals
                resultIntervals = [];

            // transform all passed intervals
            _.each(_.getArray(sourceIntervals), function (sourceInterval) {

                var // basic transformation of the source interval
                    resultInterval = self.transformInterval(sourceInterval, operationInterval, insert, columns, expandEnd);

                if (resultInterval) {
                    // split expanded intervals if specified
                    if (insert && split && (resultInterval.first < operationInterval.first) && (operationInterval.last < resultInterval.last)) {
                        resultIntervals.push({ first: resultInterval.first, last: operationInterval.first - 1 });
                        resultIntervals.push({ first: operationInterval.last + 1, last: resultInterval.last });
                    } else {
                        resultIntervals.push(resultInterval);
                    }
                }
            });

            return resultIntervals;
        };

        /**
         * Transforms the passed array of cell range addresses according to the
         * specified operation. Used to get the effective range addresses after
         * columns or rows have been insert into or deleted from a sheet.
         *
         * @param {Object|Array} sourceRanges
         *  The cell range address to be transformed, or an array of cell range
         *  addresses.
         *
         * @param {Object} operationInterval
         *  The column/row interval of the operation, with the index properties
         *  'first' and 'last'.
         *
         * @param {Boolean} insert
         *  Whether the columns/rows have been inserted into the sheet (true),
         *  or deleted from the sheet (false).
         *
         * @param {Boolean} columns
         *  Whether the passed operation interval is a column interval (true),
         *  or a row interval (false).
         *
         * @param {String} [insertMode='expand']
         *  Specifies how to modify intervals for insert operations (parameter
         *  'insert' set to true). The following modes are supported:
         *  - 'expand' (default): If columns/rows will be inserted into the
         *      middle of a range (between first and last column/row), it will
         *      be expanded.
         *  - 'expandEnd': Similar to 'expand', but columns/rows inserted
         *      exactly at the end of the range will cause to expand the range
         *      too.
         *  - 'split': If columns/rows will be inserted into the middle of a
         *      range (between first and last column/row), it will be split
         *      into two ranges; the inserted columns/rows will not be part of
         *      the new ranges.
         *
         * @returns {Array}
         *  The transformed cell ranges. Source ranges that have been deleted
         *  completely while deleting columns/rows, or that have been shifted
         *  outside the sheet completely while inserting columns/rows, will not
         *  be included (the resulting array may even be empty).
         */
        this.transformRanges = function (sourceRanges, operationInterval, insert, columns, insertMode) {

            var // the resulting transformed ranges
                resultRanges = [];

            // transform all passed cell ranges
            _.each(_.getArray(sourceRanges), function (sourceRange) {

                var // the source column/row intervals from the range
                    colIntervals = SheetUtils.getColInterval(sourceRange),
                    rowIntervals = SheetUtils.getRowInterval(sourceRange),
                    // the transformed interval
                    resultIntervals = self.transformIntervals(columns ? colIntervals : rowIntervals, operationInterval, insert, columns, insertMode);

                // assign result to the correct interval
                if (columns) { colIntervals = resultIntervals; } else { rowIntervals = resultIntervals; }

                // build and return the resulting range
                resultRanges = resultRanges.concat(SheetUtils.makeRangesFromIntervals(colIntervals, rowIntervals));
            });

            return resultRanges;
        };

        /**
         * Registers a callback function that will be invoked after columns or
         * rows have been inserted into or deleted from any sheet in this
         * document.
         *
         * @param {Function} handler
         *  The callback handler function. Will be invoked in the context of
         *  this model instance. Receives the following parameters:
         *  (1) {Number} sheet
         *      The zero-based sheet index.
         *  (2) {Object} interval
         *      The column/row interval of the insert/delete operation, with
         *      the index properties 'first' and 'last'.
         *  (3) {Boolean} insert
         *      Whether the specified column/row interval has been inserted
         *      into this sheet (true), or deleted from this sheet (false).
         *  (4) {Boolean} columns
         *      Whether the specified interval is a column interval (true), or
         *      a row interval (false).
         *
         * @returns {SpreadsheetModel}
         *  A reference to this instance.
         */
        this.registerTransformationHandler = function (handler) {
            transformationHandlers.push(handler);
            return this;
        };

        /**
         * Unregisters a callback function that has been registered with the
         * method SpreadsheetModel.registerTransformationHandler() before.
         *
         * @param {Function} handler
         *  The callback handler function to be unregistered.
         *
         * @returns {SpreadsheetModel}
         *  A reference to this instance.
         */
        this.unregisterTransformationHandler = function (handler) {
            transformationHandlers = _.without(transformationHandlers, handler);
            return this;
        };

        // sheets -------------------------------------------------------------

        /**
         * Returns the number of sheets in the spreadsheet document.
         *
         * @returns {Number}
         *  The number of sheets in the document.
         */
        this.getSheetCount = function () {
            return sheetsCollection.length;
        };

        /**
         * Returns the names of all sheets in the spreadsheet document.
         *
         * @returns {String[]}
         *  The names of all sheets in the document.
         */
        this.getSheetNames = function () {
            return _.pluck(sheetsCollection, 'name');
        };

        /**
         * Returns whether a sheet at the specified index or with the specified
         * case-insensitive name exists in the spreadsheet document.
         *
         * @param {Number|String} sheet
         *  The zero-based index of the sheet, or the case-insensitive sheet
         *  name.
         *
         * @returns {Boolean}
         *  Whether a sheet exists at the passed index or with the passed name.
         */
        this.hasSheet = function (sheet) {
            return _.isObject(getSheetInfo(sheet));
        };

        /**
         * Returns the exact name of the sheet at the specified index or with
         * the specified case-insensitive name.
         *
         * @param {Number|String} sheet
         *  The zero-based index of the sheet, or the case-insensitive sheet
         *  name.
         *
         * @returns {String|Null}
         *  The name of the sheet at the passed index or with the passed name,
         *  or null if the passed index or name is invalid.
         */
        this.getSheetName = function (sheet) {
            var sheetInfo = getSheetInfo(sheet);
            return _.isObject(sheetInfo) ? sheetInfo.name : null;
        };

        /**
         * Returns the current index of the sheet with the specified name.
         *
         * @param {String} sheetName
         *  The case-insensitive sheet name.
         *
         * @returns {Number}
         *  The current zero-based index of the sheet with the specified name,
         *  or -1 if a sheet with the passed name does not exist.
         */
        this.getSheetIndex = function (sheetName) {
            sheetName = sheetName.toLowerCase();
            return Utils.findFirstIndex(sheetsCollection, function (sheetInfo) {
                return sheetInfo.name.toLowerCase() === sheetName;
            });
        };

        /**
         * Returns the current sheet index of the passed sheet model instance.
         *
         * @param {SheetModel} sheetModel
         *  The sheet model whose index will be returned.
         *
         * @returns {Number}
         *  The current zero-based sheet index of the passed sheet model, or -1
         *  if the passed object is not a valid sheet model.
         */
        this.getSheetIndexOfModel = function (sheetModel) {
            return Utils.findFirstIndex(sheetsCollection, function (sheetInfo) {
                return sheetInfo.model === sheetModel;
            });
        };

        /**
         * Returns the model instance for the sheet at the specified index or
         * with the specified name.
         *
         * @param {Number|String} sheet
         *  The zero-based index of the sheet, or the case-insensitive sheet
         *  name.
         *
         * @returns {SheetModel|Null}
         *  The model instance for the sheet at the passed index or with the
         *  specified name, or null if the passed index or name is invalid.
         */
        this.getSheetModel = function (sheet) {
            var sheetInfo = getSheetInfo(sheet);
            return _.isObject(sheetInfo) ? sheetInfo.model : null;
        };

        /**
         * Invokes the passed iterator function for all sheet models contained
         * in this document.
         *
         * @param {Function} iterator
         *  The iterator function invoked for all sheet models. Receives the
         *  following parameters:
         *  (1) {SheetModel} sheetModel
         *      The current sheet model instance.
         *  (2) {Number} sheet
         *      The zero-based sheet index.
         *  (3) {String} name
         *      The name of the sheet.
         *  If the iterator returns the Utils.BREAK object, the iteration
         *  process will be stopped immediately.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Object} [options.context]
         *      If specified, the iterator will be called with this context
         *      (the symbol 'this' will be bound to the context inside the
         *      iterator function).
         *  @param {Boolean} [options.reverse=false]
         *      If set to true, the sheets will be visited in reversed order.
         *
         * @returns {Utils.BREAK|Undefined}
         *  A reference to the Utils.BREAK object, if the iterator has returned
         *  Utils.BREAK to stop the iteration process, otherwise undefined.
         */
        this.iterateSheetModels = function (iterator, options) {
            return Utils.iterateArray(sheetsCollection, function (sheetInfo, sheet) {
                return iterator.call(this, sheetInfo.model, sheet, sheetInfo.name);
            }, options);
        };

        /**
         * Creates a new sheet in this document.
         *
         * @param {Number} sheet
         *  The zero-based insertion index of the sheet.
         *
         * @param {String} sheetName
         *  The name of the new sheet. Must not be empty. Must not be equal to
         *  the name of any existing sheet.
         *
         * @param {Object} [attrs]
         *  Attribute set containing initial formatting attributes for the new
         *  sheet.
         *
         * @returns {String}
         *  The empty string, if the sheet has been inserted successfully;
         *  otherwise one of the following error codes:
         *  - 'sheetname:empty': The passed sheet name is empty.
         *  - 'sheetname:invalid': The passed sheet name contains invalid
         *      characters.
         *  - 'sheetname:used': The passed sheet name is already used in the
         *      document.
         *  - 'internal': Internal error while applying the operation.
         */
        this.insertSheet = function (sheet, sheetName, attrs) {

            var // the error code for invalid sheet names
                result = '';

            // remove all NPCs from passed sheet name
            sheetName = Utils.cleanString(sheetName);

            // check that the sheet name can be used (prevent internal application error on failing operation)
            result = validateSheetName(sheetName);
            if (result.length > 0) { return result; }

            // generate and apply the operation
            return this.applyOperations({ name: Operations.INSERT_SHEET, sheet: sheet, sheetName: sheetName, attrs: attrs }) ? '' : 'internal';
        };

        /**
         * Removes a sheet from this document.
         *
         * @param {Number} sheet
         *  The zero-based index of the sheet.
         *
         * @returns {Boolean}
         *  Whether deleting the sheet was successful.
         */
        this.deleteSheet = function (sheet) {
            // check that the sheet can be removed (prevent internal application error on failing operation)
            return (this.getSheetCount() > 1) && this.applyOperations({ name: Operations.DELETE_SHEET, sheet: sheet });
        };

        /**
         * Moves a sheet in the spreadsheet document to a new position.
         *
         * @param {Number} fromSheet
         *  The zero-based index of the existing sheet to be moved.
         *
         * @param {Number} toSheet
         *  The new zero-based index of the sheet.
         *
         * @returns {Boolean}
         *  Whether moving the sheet was successful.
         */
        this.moveSheet = function (fromSheet, toSheet) {
            // generate and apply the operation
            return this.applyOperations({ name: Operations.MOVE_SHEET, sheet: fromSheet, to: toSheet });
        };

        /**
         * Creates a copy of an existing sheet in the spreadsheet document.
         *
         * @param {Number} fromSheet
         *  The zero-based index of the existing sheet to be copied.
         *
         * @param {Number} toSheet
         *  The zero-based insertion index of the new sheet.
         *
         * @param {String} sheetName
         *  The name of the new sheet. Must not be empty. Must not be equal to
         *  the name of any other sheet (case-insensitive).
         *
         * @returns {String}
         *  The empty string, if the sheet has been copied successfully;
         *  otherwise one of the following error codes:
         *  - 'sheetname:empty': The passed sheet name is empty.
         *  - 'sheetname:invalid': The passed sheet name contains invalid
         *      characters.
         *  - 'sheetname:used': The passed sheet name is already used in the
         *      document.
         *  - 'internal': Internal error while applying the operation.
         */
        this.copySheet = function (fromSheet, toSheet, sheetName) {

            var // the error code for invalid sheet names
                result = null;

            // remove all NPCs from passed sheet name
            sheetName = Utils.cleanString(sheetName);

            // check that the sheet can be renamed (prevent internal application error on failing operation)
            result = validateSheetName(sheetName);
            if (result.length > 0) { return result; }

            // generate and apply the operation
            return this.applyOperations({ name: Operations.COPY_SHEET, sheet: fromSheet, to: toSheet, sheetName: sheetName }) ? '' : 'internal';
        };

        /**
         * Renames a sheet in the spreadsheet document.
         *
         * @param {Number} sheet
         *  The zero-based index of the sheet to be renamed.
         *
         * @param {String} sheetName
         *  The new name of the sheet. Must not be empty. Must not be equal to
         *  the name of any other sheet (case-insensitive).
         *
         * @returns {String}
         *  The empty string, if the sheet has been renamed successfully;
         *  otherwise one of the following error codes:
         *  - 'sheetname:empty': The passed sheet name is empty.
         *  - 'sheetname:invalid': The passed sheet name contains invalid
         *      characters.
         *  - 'sheetname:used': The passed sheet name is already used in the
         *      document.
         *  - 'internal': Internal error while applying the operation.
         */
        this.setSheetName = function (sheet, sheetName) {

            var // the error code for invalid sheet names
                result = '';

            // remove all NPCs from passed sheet name
            sheetName = Utils.cleanString(sheetName);

            // check that the sheet can be renamed (prevent internal application error on failing operation)
            result = validateSheetName(sheetName, sheet);
            if (result.length > 0) { return result; }

            // do not generate an operation, if passed name matches old name
            if (this.getSheetName(sheet) === sheetName) { return ''; }

            // generate and apply the operation
            return this.applyOperations({ name: Operations.SET_SHEET_NAME, sheet: sheet, sheetName: sheetName }) ? '' : 'internal';
        };

        /**
         * Returns the zero-based index of the active (visible) sheet in this
         * document (the value of the 'activeSheet' view attribute).
         *
         * @returns {Number}
         *  The zero-based index of the active sheet.
         */
        this.getActiveSheet = function () {
            return this.getViewAttribute('activeSheet');
        };

        /**
         * Changes the active (visible) sheet in this document (the value of
         * the 'activeSheet' view attribute).
         *
         * @param {Number} sheet
         *  The zero-based index of the sheet to be activated.
         *
         * @returns {SpreadsheetModel}
         *  A reference to this instance.
         */
        this.setActiveSheet = function (sheet) {
            return this.setViewAttribute('activeSheet', sheet);
        };

        // range contents -----------------------------------------------------

        /**
         * Returns the results and display strings of all cells contained in
         * the passed cell ranges. If any of the sheet cell collections does
         * not cover all passed ranges, the missing data will be requested from
         * the server.
         *
         * @param {Array} rangesArray
         *  An array of arrays (!) of cell range addresses. Each array element
         *  is treated as an independent range list. Each cell range object in
         *  the inner arrays must contain the additional property 'sheet' with
         *  the zero-based sheet index.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.hidden=false]
         *      If set to true, all cells contained in hidden columns or rows
         *      will be included into the result. By default, only visible
         *      cells will be returned.
         *  @param {Boolean} [options.attributes=false]
         *      If set to true, the cell content objects will contain the
         *      merged formatting attributes of the cells.
         *  @param {Boolean} [options.compressed=false]
         *      If set to true, the returned cell arrays will be optimized:
         *      Consecutive cells with equal contents and formatting will be
         *      represented by a single array element with an additional
         *      property 'count'. Especially useful, if large ranges located in
         *      the unused area of the sheet are queried.
         *  @param {Number} [options.maxCount]
         *      If specified, the maximum number of cells per range list that
         *      will be returned in the result, regardless how large the passed
         *      ranges are (the passed ranges will be shortened before the cell
         *      contents will be requested). Example: The first array element
         *      in the parameter 'rangesArray' covers 2000 cells (in several
         *      ranges), the second array element covers 500 cells. With this
         *      option set to the value 1000, the first range list will be
         *      shortened to 1000 cells, the second range list will be resolved
         *      completely. If omitted, and the ranges are too large (as
         *      defined by the constant SheetUtils.MAX_QUERY_CELL_COUNT), the
         *      entire request will be rejected (in compressed mode, the size
         *      of the compressed result array will be checked against that
         *      constant, not the number of cells in the original ranges).
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved with the
         *  cell contents referred by the passed ranges. The result will be an
         *  array of arrays of cell content objects (the length of the outer
         *  result array will be equal to the length of the 'rangesArray'
         *  parameter). Each cell object in the inner arrays will contain the
         *  following properties:
         *  - {String} cellData.display
         *      The display string of the cell result (will be the empty string
         *      for blank cells).
         *  - {Number|String|Boolean|ErrorCode|Null} cellData.result
         *      The typed result value (will be null for blank cells).
         *  - {Object} [cellData.attrs]
         *      The merged formatting attributes of the cell. Will only be set,
         *      if the option 'attributes' has been set to true (see above).
         *  - {Object} [cellData.format]
         *      Additional number format information for the cell format code.
         *      Will only be set, if the option 'attributes' has been set (see
         *      above).
         *  - {Number} [cellData.count]
         *      Always set in compressed mode (see option 'compressed' above).
         *      Contains the number of consecutive cells with equal contents
         *      and formatting represented by this array element.
         *  The cells in the inner arrays will be in order of the range lists.
         *  Cells from a single range will be ordered row-by-row.
         */
        this.queryCellContents = function (rangesArray, options) {

            var // the Deferred object that will be resolved with the cell contents
                def = $.Deferred(),
                // the abortable promise returned by this method
                promise = this.createAbortablePromise(def),
                // whether to include hidden cells
                hidden = Utils.getBooleanOption(options, 'hidden', false),
                // the maximum number of cells in the result
                maxCount = Utils.getIntegerOption(options, 'maxCount', null, 1),
                // the shortened ranges, mapped by sheet index
                rangesMap = {},
                // the promises of all cell collection queries
                allQueries = [],
                // result of the initial sheet-separation loop
                result = null;

            // check sheet index, reduce the ranges to their visible parts
            rangesArray = _.map(rangesArray, function (ranges) {

                // check sheet index contained in the range
                ranges = _.filter(ranges, function (range) {
                    if (_.isNumber(range.sheet) && (0 <= range.sheet) && (range.sheet < self.getSheetCount())) { return true; }
                    Utils.warn('SpreadsheetModel.queryCellContents(): invalid sheet index in range address');
                    return false;
                });

                // reduce the ranges to their visible parts
                if (!hidden) {
                    ranges = _.chain(ranges).map(function (range) {
                        var sheet = range.sheet,
                            visibleRanges = self.getSheetModel(sheet).getVisibleRanges(range);
                        _.each(visibleRanges, function (visibleRange) { visibleRange.sheet = sheet; });
                        return visibleRanges;
                    }).flatten(true).value();
                }

                // shorten the ranges if specified
                if (_.isNumber(maxCount)) {
                    ranges = SheetUtils.shortenRangesByCellCount(ranges, maxCount);
                }

                return ranges;
            });

            // reject the entire request, if the ranges are too large
            if (_.any(rangesArray, function (ranges) { return SheetUtils.getCellCountInRanges(ranges) > SheetUtils.MAX_QUERY_CELL_COUNT; })) {
                def.reject('overflow');
                return promise;
            }

            // build separate arrays of range lists per sheet
            result = Utils.iterateArray(rangesArray, function (ranges, arrayIndex) {
                return Utils.iterateArray(ranges, function (range) {
                    // insert the range into the correct sub-array in the map
                    var mapEntry = rangesMap[range.sheet] || (rangesMap[range.sheet] = _.times(rangesArray.length, function () { return []; }));
                    mapEntry[arrayIndex].push(range);
                });
            });

            // return immediately, if a range contains an invalid sheet index
            if (result === Utils.BREAK) {
                def.reject('invalid');
                return promise;
            }

            // adjust processed options before calling cell collection methods (performance)
            options = _.extend({}, options);
            options.hidden = true; // ranges already reduced to visible parts, do not reduce them again
            delete options.maxCount; // ranges already reduced to maximum count

            // query cell contents from all cell collections, and store all relevant data in the array and map
            _.each(rangesMap, function (sheetRangesArray, sheetKey) {

                var // zero-based index of the current sheet
                    sheet = parseInt(sheetKey, 10),
                    // cell collection instance of the current sheet
                    cellCollection = self.getSheetModel(sheet).getCellCollection();

                // store array index of the promise in the map, to be able to map from sheet index to promise index quickly
                rangesMap[sheetKey] = { sheet: sheet, index: allQueries.length };

                // query all cell contents from the cell collection of the current sheet
                allQueries.push(cellCollection.queryCellContents(sheetRangesArray, options));
            });

            // wait for all sheet promises before building the result
            $.when.apply($.when, allQueries).done(function () {

                var // $.when() passes the result arrays of all sheets in its function parameters
                    queryResultArrays = arguments,
                    // the resulting cell contents of all sheets, in the correct order
                    resultsArray = [];

                // resolve cell data in order of the original ranges
                _.each(rangesArray, function (ranges, arrayIndex) {

                    var // the cell contents for the current range list
                        result = [];

                    // pick all cell results in order of the requested ranges
                    _.each(ranges, function (range) {

                        var // array index of the associated cell collection query (parameter index)
                            queryIndex = rangesMap[range.sheet].index,
                            // the array of cell contents for the sheet
                            sheetResults = queryResultArrays[queryIndex][arrayIndex];

                        // move the appropriate number of cell entries into the result array
                        result = result.concat(sheetResults.splice(0, SheetUtils.getCellCount(range)));
                    });

                    resultsArray.push(result);
                });

                // resolve the Deferred object with the complete result array
                def.resolve(resultsArray);

            }).fail(_.bind(def.reject, def));

            // return an abortable promise
            return promise;
        };

        // operations registration --------------------------------------------

        this.registerOperationHandler(Operations.INSERT_SHEET, function (operation) {

            var // the sheet index passed in the operation
                sheet = Utils.getIntegerOption(operation, 'sheet', -1),
                // the sheet name passed in the operation
                sheetName = Utils.getStringOption(operation, 'sheetName', ''),
                // the sheet type passed in the operation
                sheetType = Utils.getStringOption(operation, 'type', 'worksheet'),
                // the sheet attributes passed in the operation
                attributes = Utils.getObjectOption(operation, 'attrs'),
                // the model of the new sheet
                sheetModel = null;

            // check the sheet index and name (must not exist yet)
            if ((sheet < 0) || (sheet > this.getSheetCount()) || !isValidSheetName(sheetName)) {
                return false;
            }

            // check the sheet type
            if (!/^(worksheet|chartsheet|macrosheet|dialogsheet)$/.test(sheetType)) {
                return false;
            }

            // create and insert the new sheet
            self.trigger('insert:sheet:before', sheet);
            sheetModel = new SheetModel(app, sheetType, attributes);
            sheetsCollection.splice(sheet, 0, { model: sheetModel, name: sheetName });
            self.trigger('insert:sheet:after', sheet, sheetModel);
            return true;
        });

        this.registerSheetOperationHandler(Operations.DELETE_SHEET, function (sheetModel, operation) {

            // there must remain at least one sheet in the document
            if (this.getSheetCount() === 1) {
                return false;
            }

            // finally, remove the sheet from the document model
            this.trigger('delete:sheet:before', operation.sheet, sheetModel);
            sheetModel.destroy();
            sheetsCollection.splice(operation.sheet, 1);
            this.trigger('delete:sheet:after', operation.sheet);
            return true;
        });

        this.registerSheetOperationHandler(Operations.MOVE_SHEET, function (sheetModel, operation) {

            var // the target position of the sheet
                to = Utils.getIntegerOption(operation, 'to', -1);

            if ((to < 0) || (to > this.getSheetCount())) {
                return false;
            }

            var sheetInfo = sheetsCollection[operation.sheet];
            self.trigger('move:sheet:before', operation.sheet, to, sheetInfo.model);
            sheetsCollection.splice(operation.sheet, 1);
            sheetsCollection.splice(to, 0, sheetInfo);
            self.trigger('move:sheet:after', operation.sheet, to, sheetInfo.model);
            return true;

        });

        this.registerSheetOperationHandler(Operations.COPY_SHEET, function (sheetModel, operation) {

            var // the target position for the new sheet
                to = Utils.getIntegerOption(operation, 'to', -1),
                // the new sheet name passed in the operation
                sheetName = Utils.getStringOption(operation, 'sheetName', '');

            // the sheet must exist, check name (no existing sheet must contain this name)
            if ((to < 0) || (to > this.getSheetCount()) || !isValidSheetName(sheetName)) {
                return false;
            }

            // clone the sheet model, and insert the new model into the collection
            self.trigger('insert:sheet:before', to);
            // first, create the collection entry (this allows to receive the sheet name from this document while cloning the sheet)
            sheetsCollection.splice(to, 0, { model: null, name: sheetName });
            sheetsCollection[to].model = sheetModel = sheetModel.clone(to);
            self.trigger('insert:sheet:after', to, sheetModel);

            return true;
        });

        this.registerSheetOperationHandler(Operations.SET_SHEET_NAME, function (sheetModel, operation) {

            var // the sheet object
                sheetInfo = getSheetInfo(operation.sheet),
                // the sheet name passed in the operation
                sheetName = Utils.getStringOption(operation, 'sheetName', '');

            // the sheet must exist, check name (no other sheet must contain this name)
            if (!_.isObject(sheetInfo) || !isValidSheetName(sheetName, operation.sheet)) {
                return false;
            }

            // no no events, if passed name matches old name
            if (sheetInfo.name !== sheetName) {
                sheetInfo.name = sheetName;
                this.trigger('rename:sheet', operation.sheet, sheetName);
            }

            return true;
        });

        this.registerSheetOperationHandler(Operations.SET_SHEET_ATTRIBUTES, function (sheetModel, operation) {
            var attributes = Utils.getObjectOption(operation, 'attrs');
            if (!_.isObject(attributes)) { return false; }
            sheetModel.setAttributes(attributes);
            return true;
        });

        this.registerNameOperationHandler(Operations.INSERT_NAME, function (opNameCollection, operation) {
            var name = Utils.getStringOption(operation, 'exprName', ''),
                formula = Utils.getStringOption(operation, 'formula', '');
            return opNameCollection.insertName(name, formula);
        });

        this.registerNameOperationHandler(Operations.CHANGE_NAME, function (opNameCollection, operation) {
            var name = Utils.getStringOption(operation, 'exprName', ''),
                formula = Utils.getStringOption(operation, 'formula', '');
            return opNameCollection.changeName(name, formula);
        });

        this.registerNameOperationHandler(Operations.DELETE_NAME, function (opNameCollection, operation) {
            var name = Utils.getStringOption(operation, 'exprName', '');
            return opNameCollection.deleteName(name);
        });

        this.registerTableOperationHandler(Operations.INSERT_TABLE, function (tableCollection, tableName, operation) {
            var range = getRangeProperty(operation),
                attributes = Utils.getObjectOption(operation, 'attrs');
            return _.isObject(range) && tableCollection.insertTable(tableName, range, attributes);
        });

        this.registerTableOperationHandler(Operations.CHANGE_TABLE, function (tableCollection, tableName, operation) {
            var range = getRangeProperty(operation), // optional range address
                attributes = Utils.getObjectOption(operation, 'attrs');
            return tableCollection.changeTable(tableName, range, attributes);
        });

        this.registerTableOperationHandler(Operations.DELETE_TABLE, function (tableCollection, tableName) {
            return tableCollection.deleteTable(tableName);
        });

        this.registerTableOperationHandler(Operations.CHANGE_TABLE_COLUMN, function (tableCollection, tableName, operation) {
            var tableModel = tableCollection.getTable(tableName),
                tableCol = Utils.getIntegerOption(operation, 'col', -1),
                attributes = Utils.getObjectOption(operation, 'attrs');
            return _.isObject(tableModel) && _.isObject(attributes) && tableModel.setColumnAttributes(tableCol, attributes);
        });

        this.registerRangeListOperationHandler(Operations.INSERT_VALIDATION, function (sheetModel, ranges, operation) {
            var index = Utils.getIntegerOption(operation, 'index');
            return sheetModel.getValidationCollection().insertValidation(ranges, operation, index);
        });

        this.registerSheetOperationHandler(Operations.CHANGE_VALIDATION, function (sheetModel, operation) {
            var index = Utils.getIntegerOption(operation, 'index', -1),
                ranges = null;
            // resolve optional 'ranges' property
            if ('ranges' in operation) {
                ranges = getRangesProperty(operation, 'ranges');
                if (!_.isArray(ranges)) { return false; }
            }
            return sheetModel.getValidationCollection().changeValidation(index, ranges, operation);
        });

        this.registerSheetOperationHandler(Operations.DELETE_VALIDATION, function (sheetModel, operation) {
            var index = Utils.getIntegerOption(operation, 'index', -1);
            return sheetModel.getValidationCollection().deleteValidation(index);
        });

        this.registerIntervalOperationHandler(Operations.INSERT_COLUMNS, 'columns', function (sheetModel, interval) {
            sheetModel.getColCollection().insertEntries(interval);
            return true;
        });

        this.registerIntervalOperationHandler(Operations.DELETE_COLUMNS, 'columns', function (sheetModel, interval) {
            sheetModel.getColCollection().deleteEntries(interval);
            return true;
        });

        this.registerIntervalOperationHandler(Operations.SET_COLUMN_ATTRIBUTES, 'columns', function (sheetModel, interval, operation) {
            var attributes = Utils.getObjectOption(operation, 'attrs');
            if (!_.isObject(attributes)) { return false; }
            sheetModel.getColCollection().setAttributes(interval, attributes, { rangeBorders: operation.rangeBorders, visibleBorders: operation.visibleBorders });
            return true;
        });

        this.registerIntervalOperationHandler(Operations.INSERT_ROWS, 'rows', function (sheetModel, interval) {
            sheetModel.getRowCollection().insertEntries(interval);
            return true;
        });

        this.registerIntervalOperationHandler(Operations.DELETE_ROWS, 'rows', function (sheetModel, interval) {
            sheetModel.getRowCollection().deleteEntries(interval);
            return true;
        });

        this.registerIntervalOperationHandler(Operations.SET_ROW_ATTRIBUTES, 'rows', function (sheetModel, interval, operation) {
            var attributes = Utils.getObjectOption(operation, 'attrs');
            if (!_.isObject(attributes)) { return false; }
            sheetModel.getRowCollection().setAttributes(interval, attributes, { rangeBorders: operation.rangeBorders, visibleBorders: operation.visibleBorders });
            return true;
        });

        this.registerRangeOperationHandler(Operations.INSERT_CELLS, function (/*sheetModel, range*/) {
            // TODO
            return false;
        });

        this.registerRangeOperationHandler(Operations.DELETE_CELLS, function (/*sheetModel, range*/) {
            // TODO
            return false;
        });

        this.registerRangeOperationHandler(Operations.MERGE_CELLS, function (sheetModel, range, operation) {
            var type = Utils.getStringOption(operation, 'type', 'merge');
            sheetModel.getMergeCollection().mergeRange(range, type);
            return true;
        });

        this.registerCellOperationHandler(Operations.SET_CELL_CONTENTS, function (sheetModel, start, operation) {
            return sheetModel.getCellCollection().setCellContents(start, operation);
        });

        this.registerRangeOperationHandler(Operations.FILL_CELL_RANGE, function (sheetModel, range, operation) {
            return sheetModel.getCellCollection().fillCellRange(range, operation);
        });

        this.registerRangeOperationHandler(Operations.CLEAR_CELL_RANGE, function (sheetModel, range, operation) {
            return sheetModel.getCellCollection().clearCellRange(range, operation);
        });

        this.registerRangeOperationHandler(Operations.AUTO_FILL, function (sheetModel, range, operation) {
            return sheetModel.getCellCollection().autoFill(range, operation);
        });

        this.registerDrawingOperationHandler(Operations.INSERT_DRAWING, function (sheetModel, position, operation) {
            var type = Utils.getStringOption(operation, 'type', ''),
                attributes = Utils.getObjectOption(operation, 'attrs');
            return sheetModel.getDrawingCollection().insertModel(position, type, attributes);
        });

        this.registerDrawingOperationHandler(Operations.DELETE_DRAWING, function (sheetModel, position) {
            return sheetModel.getDrawingCollection().deleteModel(position);
        });

        this.registerDrawingOperationHandler(Operations.SET_DRAWING_ATTRIBUTES, function (sheetModel, position, operation) {
            var attributes = Utils.getObjectOption(operation, 'attrs'),
                drawingModel = sheetModel.getDrawingCollection().findModel(position, { deep: true });
            if (!_.isObject(attributes) || !_.isObject(drawingModel)) { return false; }
            drawingModel.setAttributes(attributes);
            return true;
        });

        this.registerSimpleDrawingOperation(Operations.INSERT_CHART_DATASERIES, 'insertDataSeries', 'chart');
        this.registerSimpleDrawingOperation(Operations.DELETE_CHART_DATASERIES, 'deleteDataSeries', 'chart');
        this.registerSimpleDrawingOperation(Operations.SET_CHART_DATASERIES_ATTRIBUTES, 'setChartDataSeriesAttributes', 'chart');
        this.registerSimpleDrawingOperation(Operations.SET_CHART_AXIS_ATTRIBUTES, 'setAxisAttributes', 'chart');
        this.registerSimpleDrawingOperation(Operations.SET_CHART_GRIDLINE_ATTRIBUTES, 'setGridlineAttributes', 'chart');
        this.registerSimpleDrawingOperation(Operations.SET_CHART_TITLE_ATTRIBUTES, 'setTitleAttributes', 'chart');
        this.registerSimpleDrawingOperation(Operations.SET_CHART_LEGEND_ATTRIBUTES, 'setLegendAttributes', 'chart');

        // initialization -----------------------------------------------------

        // set the column/row count, listen to document attribute changes
        updateMaxColRow(documentStyles.getDocumentAttributes());
        documentStyles.on('change:attributes', function (event, attributes) {
            updateMaxColRow(attributes);
        });

        // register own event handlers after inserting a new sheet
        this.on('insert:sheet:after', insertSheetHandler);

        // send changed selection to remote clients after activating another sheet
        this.on('change:viewattributes', function (event, attributes) {
            if ('activeSheet' in attributes) {
                sendActiveSelection();
            }
        });

        // send changed selection in the active sheet to remote clients
        this.on('change:sheet:viewattributes', function (event, sheet, attributes) {
            if (('selection' in attributes) && (sheet === self.getActiveSheet())) {
                sendActiveSelection();
            }
        });

        // application notifies changed data of remote clients
        this.listenTo(app, 'docs:users', changeActiveClientsHandler);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            _.chain(sheetsCollection).pluck('model').invoke('destroy');
            nameCollection.destroy();
            formulaInterpreter.destroy();
            formulaCompiler.destroy();
            numberFormatter.destroy();
            // base class owns undo manager and document styles
            sheetsCollection = undoManager = documentStyles = null;
            numberFormatter = formulaCompiler = formulaInterpreter = nameCollection = transformationHandlers = null;
        });

    } // class SpreadsheetModel

    // exports ================================================================

    // derive this class from class EditModel
    return EditModel.extend({ constructor: SpreadsheetModel });

});
