/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/formula/tokenutils', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/logger',
    'io.ox/office/spreadsheet/utils/sheetutils'
], function (Utils, Logger, SheetUtils) {

    'use strict';

    // static private functions ===============================================

    /**
     * Creates a special error code for internal usage in the formula engine.
     */
    function makeInternalErrorCode(code) {
        var errorCode = SheetUtils.makeErrorCode('[#' + code + ']');
        errorCode.internal = code;
        return errorCode;
    }

    // static class TokenUtils ================================================

    /**
     * The static class TokenUtils is a console logger bound to the URL hash
     * flag 'spreadsheet:log-formulas', that logs detailed information about
     * the formula parser, compiler, and interpreter.
     */
    var TokenUtils = new Logger({ enable: 'spreadsheet:log-formulas', prefix: 'FMLA' });

    // constants --------------------------------------------------------------

    /**
     * Distance between the number 1 and the smallest number greater than 1
     * (the value of 2^-52). Provided as non-standard property Number.EPSILON
     * by Chrome and Firefox, but not supported by IE10 and IE11.
     *
     * @constant
     */
    TokenUtils.EPSILON = Math.pow(2, -52);

    /**
     * The smallest positive normalized floating-point number (browsers may
     * support smaller numbers though, which are stored with a denormalized
     * mantissa).
     *
     * @constant
     */
    TokenUtils.MIN_NUMBER = Math.pow(2, -1022);

    /**
     * Maximum number of function parameters (CalcEngine limitation).
     *
     * @constant
     */
    TokenUtils.MAX_PARAM_COUNT = 254;

    /**
     * Maximum number of rows supported in matrix literals.
     *
     * @constant
     */
    TokenUtils.MAX_MATRIX_ROW_COUNT = 16;

    /**
     * Maximum number of columns supported in matrix literals.
     *
     * @constant
     */
    TokenUtils.MAX_MATRIX_COL_COUNT = 16;

    /**
     * Maximum number of references in a reference list literal.
     *
     * @constant
     */
    TokenUtils.MAX_REF_LIST_SIZE = 1024;

    /**
     * Maximum number of cells in cell references visited in iterator loops.
     *
     * @constant
     */
    TokenUtils.MAX_CELL_ITERATION_COUNT = 2000;

    /**
     * Special error code literal to transport the 'unsupported' state.
     */
    TokenUtils.UNSUPPORTED_ERROR = makeInternalErrorCode('unsupported');

    /**
     * Special error code literal to transport the 'circular reference' result.
     */
    TokenUtils.CIRCULAR_ERROR = makeInternalErrorCode('circular');

    // public methods ---------------------------------------------------------

    /**
     * Compares the passed numbers, and returns a signed indicator number.
     *
     * @param {Number} number1
     *  The first number for comparison.
     *
     * @param {Number} number2
     *  The second number for comparison.
     *
     * @returns {Number}
     *  The number -1, if number1 is less than number2; or the number 1, if
     *  number1 is greater than number2; or the number 0, if both numbers are
     *  equal. If the difference of the numbers is less than
     *  TokenUtils.MIN_NUMBER (e.g., due to rounding errors), the numbers are
     *  also considered to be equal. If any of the numbers is NaN or infinite,
     *  the result will be NaN.
     */
    TokenUtils.compareNumbers = function (number1, number2) {
        var diff = number1 - number2, abs = Math.abs(diff);
        return !isFinite(diff) ? Number.NaN : (abs < TokenUtils.MIN_NUMBER) ? 0 : (diff / abs);
    };

    /**
     * Compares the passed strings lexicographically, and returns a signed
     * indicator number.
     *
     * @param {Boolean} string1
     *  The first string for comparison.
     *
     * @param {Boolean} string2
     *  The second string for comparison.
     *
     * @param {Boolean} [caseSens=false]
     *  If set to true, the strings will be compared case-sensitively. By
     *  default, the case of the strings will be ignored.
     *
     * @returns {Number}
     *  The number -1, if bool1 is FALSE, and bool2 is TRUE (bool1 is 'less
     *  than' bool2); or the number 1, if bool1 is TRUE, and bool2 is FALSE
     *  (bool1 is 'greater than' bool2); otherwise the number 0 (the Boolean
     *  values are equal).
     */
    TokenUtils.compareStrings = function (string1, string2, caseSens) {
        if (!caseSens) {
            string1 = string1.toUpperCase();
            string2 = string2.toUpperCase();
        }
        return (string1 < string2) ? -1 : (string1 > string2) ? 1 : 0;
    };

    /**
     * Compares the passed Boolean values, and returns a signed indicator
     * number.
     *
     * @param {Boolean} bool1
     *  The first Boolean value for comparison.
     *
     * @param {Boolean} bool2
     *  The second Boolean value for comparison.
     *
     * @returns {Number}
     *  The number -1, if bool1 is FALSE, and bool2 is TRUE (bool1 is 'less
     *  than' bool2); or the number 1, if bool1 is TRUE, and bool2 is FALSE
     *  (bool1 is 'greater than' bool2); otherwise the number 0 (the Boolean
     *  values are equal).
     */
    TokenUtils.compareBooleans = function (bool1, bool2) {
        // subtraction operator converts Booleans to numbers implicitly
        return bool1 - bool2;
    };

    /**
     * Compares the internal indexes of built-in error codes, and returns a
     * signed indicator number.
     *
     * @param {ErrorCode} errorCode1
     *  The first error code for comparison.
     *
     * @param {ErrorCode} errorCode2
     *  The second error code for comparison.
     *
     * @returns {Number}
     *  The value NaN, if any of the passed error codes does not represent a
     *  built-in error (the property 'num' of the error code is not finite);
     *  the number -1, if the index of the first error code is less than the
     *  index of the second error code; the number 1, if the index of the first
     *  error code is greater than the index of the second error code, or the
     *  number 0, if both error codes have equal indexes.
     */
    TokenUtils.compareErrorCodes = function (errorCode1, errorCode2) {
        return TokenUtils.compareNumbers(errorCode1.num, errorCode2.num);
    };

    /**
     * Compares the passed values, and returns a signed indicator number.
     *
     * @param {Number|String|Boolean|ErrorCode|Null} value1
     *  The first value for comparison.
     *
     * @param {Number|String|Boolean|ErrorCode|Null} value2
     *  The second value for comparison.
     *
     * @param {Boolean} [caseSens=false]
     *  If set to true, strings will be compared case-sensitively. By default,
     *  the case of the strings will be ignored.
     *
     * @returns {Number}
     *  The number -1, if value1 is considered less than value2; or the number
     *  1, if value1 is considered greater than value2; otherwise the number 0
     *  (the values are equal). Blank cells (special value null) are always
     *  less than numbers, numbers are always less than strings, strings are
     *  always less than Boolean values, and Boolean values are always less
     *  than error codes. Values of the same type are compared as described for
     *  the methods TokenUtils.compareNumbers(), TokenUtils.compareStrings(),
     *  TokenUtils.compareBooleans(), and TokenUtils.compareErrorCodes().
     */
    TokenUtils.compareValues = function (value1, value2, caseSens) {

        // returns a numeric identifier according to the value type
        function getTypeId(value) {
            return _.isNumber(value) ? 1 : _.isString(value) ? 2 : _.isBoolean(value) ? 3 : SheetUtils.isErrorCode(value) ? 4 : 0;
        }

        var type1 = getTypeId(value1),
            type2 = getTypeId(value2);

        // different types: ignore the actual values
        if (type1 !== type2) {
            return TokenUtils.compareNumbers(type1, type2);
        }

        // compare values of equal types
        switch (type1) {
        case 1:
            return TokenUtils.compareNumbers(value1, value2);
        case 2:
            return TokenUtils.compareStrings(value1, value2, caseSens);
        case 3:
            return TokenUtils.compareBooleans(value1, value2);
        case 4:
            return TokenUtils.compareErrorCodes(value1, value2);
        }

        // both values are null
        return 0;
    };

    /**
     * Returns the string representation of the passed literal value for debug
     * logging.
     *
     * @param {Number|Date|String|Boolean|Complex|ErrorCode|Null}
     *  The value to be converted to the string representation.
     *
     * @returns {String}
     *  The string representation of the value.
     */
    TokenUtils.valueToString = function (value) {
        return _.isString(value) ? ('"' + value.replace('"', '\\"') + '"') :
            (value instanceof Date) ? value.toISOString() :
            _.isNull(value) ? '[empty]' :
            _.isBoolean(value) ? value.toString().toUpperCase() :
            value.toString();
    };

    /**
     * Writes all passed tokens to the browser console.
     *
     * @param {String} message
     *  A message string written before the formula tokens.
     *
     * @param {Array} tokens
     *  An array of formula tokens to be written to the browser console.
     */
    TokenUtils.logTokens = TokenUtils.isLoggingActive() ? function (message, tokens) {
        TokenUtils.log(message + ': ' + tokens.join(' '));
    } : $.noop;

    // exports ================================================================

    return TokenUtils;

});
