/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/formula/matrix', [
    'io.ox/office/tk/utils',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/model/formula/tokenutils'
], function (Utils, SheetUtils, TokenUtils) {

    'use strict';

    var // shortcut for the map of error code literals
        ErrorCodes = SheetUtils.ErrorCodes;

    // class Matrix ===========================================================

    /**
     * Representation of a matrix literal: a two-dimensional array of constant
     * values (numbers, strings, Boolean values, and error codes).
     *
     * @constructor
     *
     * @param {Number|Array} rows
     *  The number of rows contained in the matrix. Must be a positive integer.
     *  Alternatively, the constructor can be called with a two-dimensional
     *  JavaScript array of values which will become the contents of this
     *  matrix instance. The array MUST contain at least one inner array. All
     *  inner arrays MUST contain at least one value, and all MUST contain the
     *  same number of values.
     *
     * @param {Number} cols
     *  The number of columns contained in the matrix. Must be a positive
     *  integer.
     *
     * @param {Number|String|Boolean|ErrorCode|Null|Function} value
     *  The value to be inserted into the matrix elements. If a function has
     *  been passed, it will be invoked for every matrix element, receiving the
     *  zero-based row and column index (in this order!), and its return value
     *  will become the value of the matrix element.
     */
    function Matrix(rows, cols, value) {

        var array = null;

        // public methods -----------------------------------------------------

        /**
         * Returns the number of rows contained in this matrix.
         *
         * @returns {Number}
         *  The number of rows contained in this matrix.
         */
        this.getRowCount = function () {
            return rows;
        };

        /**
         * Returns the number of columns contained in this matrix.
         *
         * @returns {Number}
         *  The number of columns contained in this matrix.
         */
        this.getColCount = function () {
            return cols;
        };

        /**
         * Returns the number of elements contained in this matrix.
         *
         * @returns {Number}
         *  The number of elements contained in this matrix.
         */
        this.getElementCount = function () {
            return rows * cols;
        };

        /**
         * Returns the internal two-dimensional JavaScript array containing all
         * elements of this matrix.
         *
         * @returns {Array}
         *  The internal two-dimensional JavaScript array of this matrix.
         */
        this.getArray = function () {
            return array;
        };

        /**
         * Returns the value of the specified element in this matrix.
         *
         * @param {Number} row
         *  The zero-based row index of the element. If the matrix consists of
         *  a single row only, this index will be ignored, and the element from
         *  the existing row will be used instead (matrix auto-expansion).
         *
         * @param {Number} col
         *  The zero-based column index of the element. If the matrix consists
         *  of a single column only, this index will be ignored, and the
         *  element from the existing column will be used instead (matrix
         *  auto-expansion).
         *
         * @returns {Number|String|Boolean|ErrorCode|Null}
         *  The value of the matrix element.
         *
         * @throws {ErrorCode}
         *  The #N/A error code, if the passed index is located outside the
         *  matrix (except for auto-expansion, see above).
         */
        this.getElement = function (row, col) {
            // auto-expansion for single row or column vector
            if (rows === 1) { row = 0; }
            if (cols === 1) { col = 0; }
            if ((row < rows) && (col < cols)) { return array[row][col]; }
            throw ErrorCodes.NA;
        };

        /**
         * Changes the value of the specified element in this matrix.
         *
         * @param {Number} row
         *  The zero-based row index of the element.
         *
         * @param {Number} col
         *  The zero-based column index of the element.
         *
         * @param {Number|String|Boolean|ErrorCode|Null} elem
         *  The new value for the matrix element.
         *
         * @returns {Matrix}
         *  A reference to this instance.
         */
        this.setElement = function (row, col, elem) {
            if ((row < 0) || (row >= rows) || (col < 0) || (col >= cols)) {
                Utils.error('Matrix.setElement(): element position out of bounds');
                throw 'fatal';
            }
            array[row][col] = elem;
            return this;
        };

        /**
         * Calls the passed iterator function for all elements of this matrix.
         *
         * @param {Function} iterator
         *  The iterator function invoked for all elements in the matrix.
         *  Receives the following parameters:
         *  (1) {Number|String|Boolean|ErrorCode|Null} element
         *      The current matrix element.
         *  (2) {Number} row
         *      The zero-based row index of the matrix element.
         *  (3) {Number} col
         *      The zero-based column index of the matrix element.
         *  If the iterator returns the Utils.BREAK object, the iteration
         *  process will be stopped immediately.
         *
         * @param {Object} [context]
         *  If specified, the iterator will be called with this context.
         *
         * @returns {Utils.BREAK|Undefined}
         *  A reference to the Utils.BREAK object, if the iterator has returned
         *  Utils.BREAK to stop the iteration process, otherwise undefined.
         */
        this.iterateElements = function (iterator, context) {
            return Utils.iterateArray(array, function (elems, row) {
                return Utils.iterateArray(elems, function (elem, col) {
                    return iterator.call(context, elem, row, col);
                });
            });
        };

        /**
         * Calls the passed iterator function for all elements of this matrix,
         * and sets the return values as new element values.
         *
         * @param {Function} iterator
         *  The iterator function invoked for all elements in the matrix.
         *  Receives the following parameters:
         *  (1) {Number|String|Boolean|ErrorCode|Null} element
         *      The current matrix element.
         *  (2) {Number} row
         *      The zero-based row index of the matrix element.
         *  (3) {Number} col
         *      The zero-based column index of the matrix element.
         *  Must return the new value for the element.
         *
         * @param {Object} [context]
         *  If specified, the iterator will be called with this context.
         *
         * @returns {Matrix}
         *  A reference to this instance.
         */
        this.transformElements = function (iterator, context) {
            this.iterateElements(function (elem, row, col) {
                array[row][col] = iterator.call(context, elem, row, col);
            });
            return this;
        };

        /**
         * Returns the string representation of this matrix for debug logging.
         */
        this.toString = function () {
            return '{' + _.map(array, function (elems) {
                return _.map(elems, TokenUtils.valueToString).join(';');
            }).join('|') + '}';
        };

        // initialization -----------------------------------------------------

        if (_.isArray(rows)) {

            // an array has been passed, use it directly (but initialize constructor
            // parameters which are used in various methods of this class)
            array = rows;
            rows = array.length;
            cols = array[0].length;

        } else {

            // convert passed constant value to a function
            if (!_.isFunction(value)) { value = _.constant(value); }

            // create the two-dimensional array
            array = _.times(rows, function (row) {
                return _.times(cols, function (col) {
                    return value(row, col);
                });
            });
        }

    } // class Matrix

    // exports ================================================================

    return Matrix;

});
