/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/app/controller', [
    'io.ox/office/tk/utils',
    'io.ox/office/editframework/utils/border',
    'io.ox/office/editframework/utils/mixedborder',
    'io.ox/office/editframework/app/editcontroller',
    'io.ox/office/spreadsheet/utils/config',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/utils/paneutils',
    'io.ox/office/spreadsheet/app/drawingcontroller'
], function (Utils, Border, MixedBorder, EditController, Config, SheetUtils, PaneUtils, DrawingController) {

    'use strict';

    // class SpreadsheetController ============================================

    /**
     * The controller of a OX Spreadsheet application.
     *
     * @constructor
     *
     * @extends EditController
     */
    function SpreadsheetController(app) {

        var // self reference
            self = this,

            // the spreadsheet model
            docModel = null,

            // the spreadsheet view
            docView = null,

            // all the little controller items
            items = {

                // document ---------------------------------------------------

                'document/search/nextResult': {
                    // enabled in read-only mode
                    set: function () { docView.searchNext('next'); },
                    shortcut: [{ keyCode: 'G', ctrlOrMeta: true }, { keyCode: 'F3' }],
                    preserveFocus: true // do not return focus to document
                },

                'document/search/previousResult': {
                    // enabled in read-only mode
                    set: function () { docView.searchNext('prev'); },
                    shortcut: [{ keyCode: 'G', ctrlOrMeta: true, shift: true }, { keyCode: 'F3', shift: true }],
                    preserveFocus: true // do not return focus to document
                },

                'document/search/replaceAll': {
                    parent: 'document/editable',
                    set: function (options) { docView.searchReplace(options.query, options.replace, { all: true }); },
                    preserveFocus: true // do not return focus to document
                },

                'document/search/replaceSelected': {
                    parent: 'document/editable',
                    set: function (options) { docView.searchReplace(options.query, options.replace); },
                    preserveFocus: true // do not return focus to document
                },

                // selection mode ---------------------------------------------

                // enabled if the document is editable, and no drawing objects are selected
                'document/editable/cell': {
                    parent: 'document/editable',
                    enable: function () { return !docView.hasDrawingSelection(); }
                },

                // view -------------------------------------------------------

                // enabled if the in-place cell edit mode is NOT active
                'view/cell/editmode/off': {
                    parent: 'app/valid',
                    enable: function () { return !docView.isCellEditMode(); }
                },

                'view/statuspane/show': {
                    parent: 'app/valid',
                    get: function () { return docView.getStatusPane().isVisible(); },
                    set: function (state) { docView.getStatusPane().toggle(state); }
                },

                // index of the active sheet in the array of visible sheets (not model sheet index!)
                'view/sheet/active': {
                    parent: 'app/valid',
                    get: function () { return docView.getActiveSheet({ visible: true }); },
                    set: function (index) { docView.setActiveSheet(index, { visible: true }); }
                },

                'view/sheet/prev': {
                    parent: 'app/valid',
                    enable: function () { return docView.getActiveSheet({ visible: true }) > 0; },
                    set: function () { docView.activatePreviousSheet(); },
                    shortcut: { keyCode: 'PAGE_UP', ctrlOrMeta: true, alt: true }
                },

                'view/sheet/next': {
                    parent: 'app/valid',
                    enable: function () { return docView.getActiveSheet({ visible: true }) + 1 < docView.getVisibleSheetCount(); },
                    set: function () { docView.activateNextSheet(); },
                    shortcut: { keyCode: 'PAGE_DOWN', ctrlOrMeta: true, alt: true }
                },

                'view/subtotals/type': {
                    parent: 'app/valid',
                    enable: function () {
                        var subtotals = docView.getSubtotals();
                        return (subtotals.cells > 1) && (this.getValue() in subtotals);
                    },
                    get: function () { return app.getUserSettingsValue('subtotalType', 'sum'); },
                    set: function (subtotalType) { app.setUserSettingsValue('subtotalType', subtotalType); }
                },

                'view/status': {
                    parent: 'app/valid',
                    enable: function () { return _.isString(docView.getStatusLabel()); },
                    get: function () { return docView.getStatusLabel(); }
                },

                'view/formula': {
                    parent: 'view/cell/editmode/off',
                    enable: function () { return docView.isSingleCellSelection() && !docView.hasDrawingSelection() && _.isString(this.getValue()); },
                    get: function () { return docView.getCellContents().formula; }
                },

                'view/grid/show': {
                    parent: 'app/valid',
                    get: function () { return docView.getSheetViewAttribute('showGrid'); },
                    set: function (state) { docView.setSheetViewAttribute('showGrid', state); }
                },

                'view/split/dynamic': {
                    parent: 'view/cell/editmode/off',
                    get: function () { return docView.hasDynamicSplit(); },
                    set: function (state) { docView.setDynamicSplit(state); }
                },

                'view/split/frozen': {
                    parent: 'view/cell/editmode/off',
                    get: function () { return docView.hasFrozenSplit(); },
                    set: function (value) { docView.setFrozenSplit(value); }
                },

                'view/split/frozen/fixed': {
                    parent: 'view/cell/editmode/off',
                    set: function (value) { docView.setFixedFrozenSplit(value.cols, value.rows); }
                },

                'view/zoom': {
                    parent: 'app/valid',
                    get: function () { return docView.getZoom(); },
                    set: function (zoom) { docView.setZoom(zoom); }
                },

                'view/zoom/dec': {
                    parent: 'view/zoom',
                    enable: function (zoom) { return zoom > SheetUtils.MIN_ZOOM; },
                    set: function () { docView.decZoom(); }
                },

                'view/zoom/inc': {
                    parent: 'view/zoom',
                    enable: function (zoom) { return zoom < SheetUtils.MAX_ZOOM; },
                    set: function () { docView.incZoom(); }
                },

                // sheet operations -------------------------------------------

                'sheet/unlocked': {
                    parent: 'document/editable',
                    enable: function () { return !docView.isSheetLocked(); }
                },

                'sheet/attributes': {
                    parent: 'document/editable',
                    get: function () { return docView.getSheetAttributes().sheet; }
                },

                'sheet/visible': {
                    parent: 'sheet/attributes',
                    enable: function () { return docView.getVisibleSheetCount() > 1; },
                    get: function (attributes) { return attributes.visible; },
                    set: function (visible) { docView.setSheetAttributes({ sheet: { visible: visible } }); }
                },

                'sheet/showall': {
                    parent: 'document/editable',
                    enable: function () { return docView.getHiddenSheetCount() > 0; },
                    set: function () { docView.showAllSheets(); }
                },

                'sheet/insert': {
                    parent: 'document/editable',
                    enable: function () { return docModel.getSheetCount() < Config.MAX_SHEET_COUNT; },
                    set: function () { docView.insertSheet(); }
                },

                'sheet/delete': {
                    parent: 'document/editable',
                    enable: function () { return docView.getVisibleSheetCount() > 1; },
                    set: function () { docView.deleteSheet(); }
                },

                'sheet/name': {
                    parent: 'document/editable',
                    get: function () { return docView.getSheetName(); },
                    set: function (sheetName) { docView.setSheetName(sheetName); }
                },

                'sheet/rename/dialog': {
                    parent: 'sheet/name',
                    set: function () { return docView.showRenameSheetDialog(); }
                },

                'sheet/copy/dialog': {
                    parent: 'sheet/insert',
                    set: function () { return docView.showCopySheetDialog(); }
                },

                'sheet/reorder/dialog': {
                    parent: 'document/editable',
                    enable: function () { return docView.getVisibleSheetCount() > 1; },
                    set: function () { return docView.showReorderSheetsDialog(); }
                },

                'sheet/formulas/refresh': {
                    parent: 'view/cell/editmode/off',
                    set: function () { return docView.getCellCollection().refreshAllFormulaCells(); },
                    shortcut: { keyCode: 'F9', shift: true, ctrlOrMeta: true }
                },

                // parent item for various sheet-level operations (active sheet), enabled if:
                // - document is in edit mode
                // - cell selection (no drawing selection)
                // - the active sheet is unlocked
                // - in-place cell edit mode is not active
                'sheet/operation': {
                    parent: ['document/editable/cell', 'sheet/unlocked', 'view/cell/editmode/off']
                },

                // table operations -------------------------------------------

                // toggles the table filter, if a table is selected in the active
                // sheet, otherwise toggles the auto filter of the active sheet
                'table/filter': {
                    parent: 'sheet/operation',
                    get: function () { return docView.hasTableFilter(); },
                    set: function (state) { return docView.toggleTableFilter(state); }
                },

                // refreshes the filter and sorting of the selected table range
                'table/refresh': {
                    parent: 'sheet/operation',
                    enable: function () { return docView.canRefreshTable(); },
                    set: function () { return docView.refreshTable(); }
                },

                // parent item for table column operations on-the-fly: requires an
                // 'active' table column stored in the sheet view attribute
                // 'activeTableData' (e.g. via pop-up menu), but no selected table range
                'table/column/active': {
                    parent: 'sheet/operation',
                    enable: function () { return _.isObject(this.getValue()); },
                    get: function () { return docView.getSheetViewAttribute('activeTableData'); }
                },

                // applies a discrete filter rule for the active table column
                // (array of strings, or undefined to remove the filter rule)
                'table/column/filter/discrete': {
                    parent: 'table/column/active',
                    set: function (entries) { return docView.applyDiscreteFilter(entries); }
                },

                // column operations ------------------------------------------

                'column/insert': {
                    parent: 'sheet/operation',
                    enable: function () { return docView.canInsertColumns(); },
                    set: function () { return docView.insertColumns(); }
                },

                'column/delete': {
                    parent: 'sheet/operation',
                    enable: function () { return docView.canDeleteColumns(); },
                    set: function () { return docView.deleteColumns(); }
                },

                // returns the mixed column attributes
                'column/attributes': {
                    parent: 'sheet/operation',
                    get: function () { return docView.getColumnAttributes(); }
                },

                'column/hide': {
                    parent: 'column/attributes',
                    // enabled when any visible columns are selected
                    enable: function (attributes) { return attributes.visible !== false; },
                    set: function () { return docView.setColumnAttributes({ visible: false }); }
                },

                'column/show': {
                    parent: 'sheet/operation',
                    enable: function () { return docView.canShowColumns(); },
                    set: function () { return docView.showColumns(); }
                },

                'column/width/optimal': {
                    parent: 'column/hide', // only enabled when any visible columns are selected
                    set: function () { return docView.setOptimalColumnWidth(); }
                },

                'column/attributes/active': {
                    parent: 'sheet/operation',
                    get: function () { return docView.getActiveColumnAttributes(); }
                },

                'column/width/active': {
                    parent: 'column/attributes/active',
                    get: function (attributes) { return attributes.visible ? attributes.width : 0; },
                    set: function (value) { return docView.setColumnWidth(value); }
                },

                // row operations ---------------------------------------------

                'row/insert': {
                    parent: 'sheet/operation',
                    enable: function () { return docView.canInsertRows(); },
                    set: function () { return docView.insertRows(); }
                },

                'row/delete': {
                    parent: 'sheet/operation',
                    enable: function () { return docView.canDeleteRows(); },
                    set: function () { return docView.deleteRows(); }
                },

                // returns the mixed row attributes
                'row/attributes': {
                    parent: 'sheet/operation',
                    get: function () { return docView.getRowAttributes(); }
                },

                'row/hide': {
                    parent: 'row/attributes',
                    // enabled when any visible rows are selected
                    enable: function (attributes) { return attributes.visible !== false; },
                    set: function () { return docView.setRowAttributes({ visible: false }); }
                },

                'row/show': {
                    parent: 'sheet/operation',
                    enable: function () { return docView.canShowRows(); },
                    set: function () { return docView.showRows(); }
                },

                'row/height/optimal': {
                    parent: 'row/hide', // only enabled when any visible rows are selected
                    set: function () { return docView.setOptimalRowHeight(); }
                },

                'row/attributes/active': {
                    parent: 'sheet/operation',
                    get: function () { return docView.getActiveRowAttributes(); }
                },

                'row/height/active': {
                    parent: 'row/attributes/active',
                    get: function (attributes) { return attributes.visible ? attributes.height : 0; },
                    set: function (value) { return docView.setRowHeight(value); }
                },

                // cell operations --------------------------------------------

                'cell/operation': {
                    parent: ['document/editable/cell', 'view/cell/editmode/off']
                },

                'cell/merge': {
                    parent: 'cell/operation',
                    enable: function () { return docView.hasAnyRangeSelected(); },
                    get: function () { return docView.hasMergedRangeSelected(); },
                    set: function (type) { docView.mergeRanges(type); }
                },

                'cell/autoformula': {
                    parent: 'cell/operation',
                    enable: function () { return docView.getSelectedRanges().length === 1; },
                    set: function (funcName) { docView.insertAutoFormula(funcName); }
                },

                'cell/painter': {
                    parent: 'cell/operation',
                    get: function () { return docView.isFormatPainterActive(); },
                    set: function (state) { docView.activateFormatPainter(state); }
                },

                'cell/sort/expanddialog': {
                    parent: ['view/cell/editmode/off'],
                    set: function () { docView.showSortExtensionDialog(); }
                },

                'cell/sort': {
                    parent: ['view/cell/editmode/off'],
                    get: function () { return docView.getSortState(); },
                    set: function (value) { docView.sortCellRange('standard', value); }
                },

                'cell/sort/custom': {
                    parent: ['view/cell/editmode/off'],
                    get: function () { return docView.getSortState('custom'); },
                    set: function (value) { docView.sortCellRange('custom', value, 'custom'); }
                },

                'cell/sort/clear': {
                    parent: ['view/cell/editmode/off'],
                    set: function () { docView.resetSortOptions('custom'); }
                },

                'cell/sort/hasheadline': {
                    parent: ['view/cell/editmode/off'],
                    get: function () { return docView.getSortOption('hasheadline', 'custom'); },
                    set: function (value) { docView.setSortOption('hasheadline', value, 'custom'); }
                },

                'cell/sort/casesensitive': {
                    parent: ['view/cell/editmode/off'],
                    get: function () { return docView.getSortOption('casesensitive', 'custom'); },
                    set: function (value) { docView.setSortOption('casesensitive', value, 'custom'); }
                },

                'cell/sort/direction': {
                    parent: ['view/cell/editmode/off'],
                    get: function () { return docView.getSortOption('direction', 'custom'); },
                    set: function (value) { docView.setSortOption('direction', value, 'custom'); }
                },

                'cell/sort/order': {
                    parent: ['view/cell/editmode/off'],
                    get: function () { return docView.getSortOption('order', 'custom'); },
                    set: function (value) { docView.setSortOption('order', value, 'custom'); }
                },

                'cell/sort/colrow': {
                    parent: ['view/cell/editmode/off'],
                    get: function () { return docView.getSortOption('colrow', 'custom'); },
                    set: function (value) { docView.setSortOption('colrow', value, 'custom'); }
                },

                // cell attributes --------------------------------------------

                'cell/attributes': {
                    parent: 'document/editable/cell',
                    get: function () { return docView.getCellAttributes(); }
                },

                'cell/stylesheet': {
                    parent: ['cell/attributes', 'view/cell/editmode/off'],
                    get: function (attributes) { return attributes.styleId; },
                    set: function (styleId) { docView.fillCellRanges(undefined, { styleId: styleId }); }
                },

                'cell/reset': {
                    parent: ['cell/attributes', 'view/cell/editmode/off'],
                    set: function () { docView.clearAttributes(); }
                },

                // cell attributes (family 'cell') ----------------------------

                'cell/attributes/cell': {
                    parent: ['cell/attributes', 'view/cell/editmode/off'],
                    get: function (attributes) { return attributes.cell; }
                },

                'cell/fillcolor': {
                    parent: 'cell/attributes/cell',
                    get: function (attributes) { return attributes.fillColor; },
                    set: function (color) { docView.setCellAttribute('fillColor', color); }
                },

                'cell/linebreak': {
                    parent: 'cell/attributes/cell',
                    get: function (attributes) { return attributes.wrapText; },
                    set: function (wrap) { docView.setCellAttribute('wrapText', wrap); }
                },

                'cell/alignhor': {
                    parent: 'cell/attributes/cell',
                    get: function (attributes) { return attributes.alignHor; },
                    set: function (alignment) { docView.setCellAttribute('alignHor', alignment); }
                },

                'cell/alignvert': {
                    parent: 'cell/attributes/cell',
                    get: function (attributes) { return attributes.alignVert; },
                    set: function (alignment) { docView.setCellAttribute('alignVert', alignment); }
                },

                // border mode, as Boolean map with position keys (left, top, ...)
                'cell/border/mode': {
                    parent: ['document/editable/cell', 'view/cell/editmode/off'],
                    get: function () { return docView.getBorderMode(); },
                    set: function (borderMode) { docView.setBorderMode(borderMode); }
                },

                // all mixed border attributes of the selection
                'cell/border/attributes': {
                    parent: 'document/editable/cell',
                    get: function () { return docView.getBorderAttributes(); }
                },

                // enabled if at least one border in the selection is visible
                'cell/border/visible': {
                    parent: 'cell/border/attributes',
                    enable: function (borderAttributes) { return _.any(borderAttributes, MixedBorder.isVisibleBorder); }
                },

                // a single mixed border for all borders in the selection
                'cell/border/value': {
                    parent: ['cell/border/visible', 'view/cell/editmode/off'],
                    get: function (borderAttributes) { return MixedBorder.mixBorders(_.values(borderAttributes)); }
                },

                'cell/border/style': {
                    parent: 'cell/border/value',
                    get: function (border) { return border.style; },
                    set: function (value) { docView.changeVisibleBorders({ style: value }); }
                },

                // width of the border lines, in points (rounded to 1/2 points)
                'cell/border/width': {
                    parent: 'cell/border/value',
                    get: function (border) { return !MixedBorder.isVisibleBorder(border) ? 0 : _.isNumber(border.width) ? Utils.convertHmmToLength(border.width, 'pt', 0.5) : null; },
                    set: function (value) { docView.changeVisibleBorders({ width: Utils.convertLengthToHmm(value, 'pt') }); }
                },

                'cell/border/color': {
                    parent: 'cell/border/value',
                    get: function (border) { return border.color; },
                    set: function (value) { docView.changeVisibleBorders({ color: value }); }
                },

                // combined style and width as enumeration as used by Excel
                'cell/border/style/preset': {
                    parent: 'cell/border/value',
                    get: function (border) { return PaneUtils.getPresetStyleForBorder(border); },
                    set: function (style) { docView.changeVisibleBorders(PaneUtils.getBorderForPresetStyle(style)); }
                },

                'cell/numberformat/category': {
                    parent: ['document/editable/cell', 'view/cell/editmode/off'],
                    get: function () { return docView.getNumberFormatCategory(); },
                    set: function (category) { docView.setNumberFormatCategory(category); },
                    shortcut: [
                        { keyCode: '1', shift: true, ctrlOrMeta: true, value: 'number' },
                        { keyCode: '2', shift: true, ctrlOrMeta: true, value: 'scientific' },
                        { keyCode: '3', shift: true, ctrlOrMeta: true, value: 'date' },
                        { keyCode: '4', shift: true, ctrlOrMeta: true, value: 'currency' },
                        { keyCode: '5', shift: true, ctrlOrMeta: true, value: 'percent' },
                        { keyCode: '6', shift: true, ctrlOrMeta: true, value: 'standard' }
                    ]
                },

                'cell/numberformat/code': {
                    parent: 'cell/numberformat/category',
                    enable: function (category) { return !/^(standard|custom)$/.test(category); },
                    get: function () { return docView.getNumberFormatCode(); },
                    set: function (formatCode) { docView.setNumberFormatCode(formatCode); }
                },

                // cell attributes (family 'character') -----------------------

                // also enabled in in-place cell edit mode
                'cell/attributes/character': {
                    parent: 'cell/attributes',
                    get: function (attributes) { return attributes.character; }
                },

                'character/bold': {
                    parent: 'cell/attributes/character',
                    get: function (attributes) { return attributes.bold; },
                    set: function (state) { docView.setCharacterAttribute('bold', state); },
                    shortcut: { keyCode: 'B', ctrlOrMeta: true, value: function (state) { return !state; } }
                },

                'character/italic': {
                    parent: 'cell/attributes/character',
                    get: function (attributes) { return attributes.italic; },
                    set: function (state) { docView.setCharacterAttribute('italic', state); },
                    shortcut: { keyCode: 'I', ctrlOrMeta: true, value: function (state) { return !state; } }
                },

                'character/underline': {
                    parent: 'cell/attributes/character',
                    get: function (attributes) { return attributes.underline; },
                    set: function (state) { docView.setCharacterAttribute('underline', state); },
                    shortcut: { keyCode: 'U', ctrlOrMeta: true, value: function (state) { return !state; } }
                },

                'character/strike': {
                    parent: 'cell/attributes/character',
                    get: function (attributes) { return _.isString(attributes.strike) ? (attributes.strike !== 'none') : null; },
                    set: function (state) { docView.setCharacterAttribute('strike', state ? 'single' : 'none'); }
                },

                'character/color': {
                    parent: 'cell/attributes/character',
                    get: function (attributes) { return attributes.color; },
                    set: function (color) { docView.setCharacterAttribute('color', color); }
                },

                'character/fontname': {
                    parent: 'cell/attributes/character',
                    get: function (attributes) { return attributes.fontName; },
                    set: function (fontName) { docView.setCharacterAttribute('fontName', fontName); }
                },

                'character/fontsize': {
                    parent: 'cell/attributes/character',
                    get: function (attributes) { return attributes.fontSize; },
                    set: function (fontSize) { docView.setCharacterAttribute('fontSize', fontSize); }
                },

                'character/url': {
                    parent: 'cell/attributes/character',
                    get: function (attributes) { return attributes.url; },
                    set: function (url) { docView.setCharacterAttribute('url', url); }
                },

                'character/hyperlink/dialog': {
                    parent: ['character/url', 'view/cell/editmode/off'],
                    set: function () { return docView.editHyperlink(); }
                },

                'character/hyperlink/remove': {
                    parent: ['character/url', 'view/cell/editmode/off'],
                    set: function () { docView.removeHyperlink(); }
                }

            };

        // base constructor ---------------------------------------------------

        EditController.call(this, app);
        DrawingController.call(this, app);

        // initialization -----------------------------------------------------

        // register item definitions
        this.registerDefinitions(items);

        // initialization after construction
        app.onInit(function () {

            // model and view are not available at construction time
            docModel = app.getModel();
            docView = app.getView();

            // update GUI after changed selection, and after receiving view updates
            self.listenTo(docView, 'change:sheet:viewattributes change:layoutdata celledit:enter celledit:change celledit:leave', function () { self.update(); });
        });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            app = self = docModel = docView = items = null;
        });

    } // class SpreadsheetController

    // exports ================================================================

    // derive this class from class EditController
    return EditController.extend({ constructor: SpreadsheetController });

});
