/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */
define('io.ox/office/portal/mainhelper',
    ['io.ox/core/date',
     'io.ox/core/capabilities',
     'io.ox/core/commons',
     'io.ox/core/notifications',
     'io.ox/office/portal/collection/documentcollection',
     'io.ox/office/portal/collection/templatecollection',
     'io.ox/office/portal/view/documentview',
     'io.ox/office/portal/view/templateview',
     'io.ox/office/portal/view/toolbarview',
     'io.ox/office/tk/io',
     'io.ox/office/tk/utils',
     'settings!io.ox/office',
     'gettext!io.ox/office/portal',
     'less!io.ox/office/portal/style'
    ], function (CoreDate, Capabilities, Commons, Notifications, DocumentCollection, TemplateCollection, DocumentView, TemplateView, ToolbarView, IO, Utils, Settings, gt) {

    'use strict';

    // static class MainHelper ================================================

    var // the exported MainHelper class
        MainHelper = {};

    // methods ----------------------------------------------------------------

    MainHelper.createLauncher = function (appBaseName, appTitle) {

        var // the module name of the portal application
            PORTAL_MODULE_NAME = 'io.ox/office/portal/' + appBaseName,
            // the module name of the edit application
            EDIT_MODULE_NAME = 'io.ox/office/' + appBaseName,
            // the application instance
            app = ox.ui.createApp({ name: PORTAL_MODULE_NAME, title: appTitle });

        app.setLauncher(function (options) {

            // create main window object
            var portalWindow = ox.ui.createWindow({
                name: PORTAL_MODULE_NAME,
                title: appTitle,
                chromeless: true
            });

            // the bootstrap of the OX Documents app
            function initOXDocuments() {
                // prepare app skeleton. Maybe this could be the AppView in the future.
                var rootNode = portalWindow.nodes.main,
                    windowBodyNode = portalWindow.nodes.body,
                    recentsNode = $('<div>').addClass('office-portal-recents').append($('<h2>').text(gt('Recent Documents'))),
                    recentsList = $('<div>').addClass('office-portal-recents-list f6-target'),
                    templatesNode = $('<div>').addClass('office-portal-templates').append($('<h2>').text(gt('New from Template'))),
                    // the document collection for the recent documents list
                    documentCollection = new DocumentCollection(),
                    // the template collection for the listed templates
                    templateCollection = new TemplateCollection(),
                    // whether this is an Android device and unsupported Chrome version or other browser => reduce functionality
                    reducedFunctionality = (_.browser.Android) && (appTitle === 'Text') && !Utils.supportedChromeVersionOnAndroidForText();

                // prefetch application source code
                IO.prefetchModuleSource(EDIT_MODULE_NAME);

                rootNode.empty().append(recentsNode.append(recentsList), templatesNode);

                // disable templates in Text App on Androids
                if (reducedFunctionality) {
                    recentsNode.addClass('reduced-recents-functionality');
                    rootNode.find('.office-portal-templates').remove();
                }

                // create application views
                new ToolbarView({ el: windowBodyNode, model: app});
                var documentView = new DocumentView({collection: documentCollection, el: recentsList, model: app}),
                    templateView = new TemplateView({collection: templateCollection, el: templatesNode, model: app});

                // try to get recent files and templates
                portalWindow.busy();
                IO.sendRequest(IO.FILTER_MODULE_NAME, { action: 'gettemplateandrecentlist', type: appBaseName })
                .always(function () {
                    portalWindow.idle();
                }).done(function (data) {
                    // populate the document and template collection with data from server
                    documentCollection.setData(data.recents || []);
                    templateCollection.setData(data.templates || []);
                    // render views with fresh data
                    documentView.render(appBaseName);
                    templateView.render(appBaseName);
                }).fail(appFail);

            }

            // yells if something went wrong...
            function appFail(result) {
                var errorMsg = (result && result.error) ? result.error + ':' : '';
                errorMsg += gt('Application may not work as expected until this problem is solved.');
                Notifications.yell('error', errorMsg);
            }

            // refreshes Documents app only on visibility
            function refresh() {
                if (!portalWindow.state.visible) { return; }
                initOXDocuments();
            }

            // save settings to the app object
            app.settings = Settings;

            portalWindow.addClass('io-ox-office-portal-main');

            // set it as the application window
            app.setWindow(portalWindow);

            // show portal window
            portalWindow.show();

            // Reinit app on portal window show. This is needed for this use case:
            // User opens a document from the recents list -> loses app focus,
            // and close the document -> gained focus to this app without an app 'resume' event).
            portalWindow.on('show', initOXDocuments);

            // refresh Documents portal on OX global refresh event
            ox.on('refresh^', refresh);

            // init OX Documents in folder support addition success.
            Commons.addFolderSupport(app, null, 'infostore', options.folder)
                .always(function always() {
                    initOXDocuments();
                })
                .fail(appFail);
        });

        return { getApp: app.getInstance };
    };

    // exports ================================================================

    return MainHelper;

});
