/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define('io.ox/office/portal/action/toolbaractionsimpl',
    ['gettext!io.ox/office/portal',
         'io.ox/office/tk/utils',
        'io.ox/core/folder/api',
        'io.ox/core/extensions',
        'io.ox/core/extPatterns/links',
        'io.ox/backbone/mini-views/dropdown',
        'io.ox/office/portal/portalutils',
        'io.ox/core/capabilities'
    ], function (gt, Utils, FolderAPI, Ext, Links, Dropdown, PortalUtils) {

        'use strict';

        /**
         * Creates all extensions points and bind actions for the portal toolbar.
         *
         * @constructor
         *
         * @param {Object} options
         *  @param {String} [options.appBaseName]
         *  @param {String} [options.newDocText]
         *  @param {String} [options.openDocText]
         *
         */
        var ToolbarActionsImpl = function (options) {

            var // the name of the application, used for filtering files in file picker
                appBaseName =  Utils.getStringOption(options, 'appBaseName'),
                // identifier for the appBaseName toolbar
                toolbarPointId = 'io.ox/office/portal/toolbar' + appBaseName,
                // deferred is set when "newdoc" & "loaddoc" is called, so we can allow only one new call at once
                newDeferred = null, loadDeferred = null;

            this.toolbarPoint = Ext.point(toolbarPointId);

            // create extension point for the 'new appBaseName' link
            Ext.point(toolbarPointId + '/new').extend(new Links.Link({
                id: 'new_' + appBaseName,
                index: 100,
                prio: 'hi',
                label: Utils.getStringOption(options, 'newDocText'),
                cssClasses: 'new-document',
                ref: toolbarPointId + '/actions/new/' + appBaseName
            }));

            // add link to the toolbar point
            this.toolbarPoint.extend({
                id: 'new-' + appBaseName + 'document',
                index: 100,
                draw: function (baton) {
                    Ext.point(toolbarPointId + '/new').invoke('draw', this, baton);
                }
            });

            //add open document link to  the toolbar point
            this.toolbarPoint.extend(new Links.Link({
                id: 'open-' + appBaseName + 'document',
                index: 200,
                prio: 'hi',
                label: Utils.getStringOption(options, 'openDocText'),
                cssClasses: 'open-document',
                ref: 'io.ox/office/portal/open' + appBaseName
            }));

            // define action that will be fired on the 'new appBaseName', 'new text' and 'open document' link
            new Links.Action(toolbarPointId + '/actions/new/' + appBaseName, {
                capabilities: appBaseName,
                action: function (baton) {
                    if (newDeferred && newDeferred.state() === 'pending') {
                        Utils.warn('cant run new document, because old call is still pending');
                        return;
                    }
                    newDeferred = ox.launch('io.ox/office/' + appBaseName + '/main', { action: 'new', folderId: baton.app.folder.get() });
                }
            });

            new Links.Action('io.ox/office/portal/open' + appBaseName, {
                action: function () {
                    require(['io.ox/files/filepicker']).done(function (Picker) {
                        new Picker({
                            // prefix for custom ext. point
                            point: 'io.ox/office/portal/toolbar/open',
                            // filter files of disabled applications (capabilities)
                            filter: function (file) {
                                return appBaseName === PortalUtils.getAppBaseName(file.filename);
                            },
                            primaryButtonText: gt('Open'),
                            cancelButtonText: gt('Cancel'),
                            header: gt('Open document'),
                            multiselect: false,
                            tree: {
                                // filter deleted files folder
                                filter: function (parent, model) {
                                    if (FolderAPI.is('trash', model.toJSON())) { return false; }
                                }
                            }
                        })
                        .done(function (files) {
                            // get single file in array (multiselect false)
                            if (files && files[0]) {
                                var selectedFile = files[0],
                                    moduleName = PortalUtils.getModuleName(selectedFile.filename);
                                if (loadDeferred && loadDeferred.state() === 'pending') {
                                    Utils.warn('cant run load document, because old call is still pending');
                                    return;
                                }
                                loadDeferred = ox.launch(moduleName + '/main', { action: 'load', file: selectedFile });
                            }
                        });
                    });
                }
            });

        };

        return ToolbarActionsImpl;

    });
