/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Edy Haryono <edy.haryono@open-xchange.com>
 */
define('io.ox/office/portal/action/templateactions',
    ['io.ox/core/extensions',
        'io.ox/files/api',
        'io.ox/core/extPatterns/links',
        'io.ox/core/extPatterns/actions',
        'io.ox/office/portal/portalutils',
        'io.ox/office/tk/io',
        'io.ox/office/tk/keycodes',
        'gettext!io.ox/office/portal'
    ], function (Ext, FilesAPI, Links, CoreActions, PortalUtils, IO, KeyCodes, gt) {

        'use strict';

        var TemplateExtensions = function () {


            var // extension point id
                templateExtension = 'io.ox/portal/office/templates',
                // template thumbnail width in px
                thumbnailWidth = 120;

            /*
             * Function returning the URL to the global templates using
             * the file descriptor for the creation of the URL.
             *
             * @param {Object} data
             *  The file descriptor object.
             *
             * @param {Object} [options]
             *  Additional optional options.
             *  @param {Boolean} [options.isGlobal=true]
             *      If set to true, the url for global templates is created,
             *      otherwise for user templates.
             *
             * @returns {String}
             *  The url string for the template.
             */
            var createTemplateURL = function (data, options) {

                var // whether the template files are local or global
                    isGlobal = PortalUtils.getBooleanOption(options, 'isGlobal', true),
                    // the constructed url
                    url = isGlobal ? ox.abs + ox.apiRoot + '/oxodocumentfilter?action=templatepreview&id=' + data.id +
                        '&folder=' + data.folder_id + '&format=preview_image&delivery=view&scaleType=contains' :
                        FilesAPI.getUrl(data, 'preview');

                return url += '&width=' + thumbnailWidth + '&dummy=' + data.last_modified;
            };

            /*
             * Helper function to generate the draw-functions for the
             * different extension points for creating new documents from
             * global or user templates.
             *
             * @param {itemClass} String
             *  A class name specific for the application and for global or
             *  user templates.
             *
             * @param {Object} [options]
             *  Additional optional options.
             *  @param {Boolean} [options.isGlobal=true]
             *      If set to true, the url for global templates is created,
             *      otherwise for user templates.
             *
             * @returns {Function}
             *  The draw function for the extension.
             */
            var getDrawFunction = function (itemClass, options) {

                // baton : a template file descriptor
                return function (baton) {

                    var templateItem = $('<a>').addClass('template-item ' + itemClass)
                        .attr({
                            'data-toggle': 'tooltip',
                            'data-placement': 'top',
                            'title': _.noI18n(baton.data.filename),
                            'tabindex' : 0
                        })
                        .tooltip(),
                        templatePreview = $('<div>').addClass('template-preview').css({
                            backgroundImage: 'url(' + createTemplateURL(baton.data, options) + ')'
                        }),
                        templateName = $('<p>').addClass('template-name').text(_.noI18n(PortalUtils.removeFileExtension(baton.data.filename)));

                    // add new blank template
                    templateItem.append(templatePreview, templateName);

                    // Get file descriptor of the newly created file as result, and open it with an office app.
                    templateItem.click(function () {
                        CoreActions.invoke(templateExtension + '/action/open', this, baton.data);
                    });

                    templateItem.on('keydown', {grid: true}, PortalUtils.keydownHandler);

                    this.append(templateItem);
                };

            };

            /*
             * Helper function to generate the draw-functions for the
             * different extension points for creating new documents from
             * blank templates.
             *
             * @param {fileName} String
             *  The localized name for the empty template visible in the
             *  application.
             *
             * @param {invokeId} String
             *  The unique ID used to invoke the application with a click on the
             *  blank template.
             *
             * @returns {Function}
             *  The draw function for the blank template extension.
             */
            var getBlankDrawFunction = function (fileName, invokeId) {

                return function (baton) {

                    var // the name displayed for the empty document
                        templateItem = $('<a>').addClass('template-item')
                        .attr({
                            'data-toggle': 'tooltip',
                            'data-placement': 'top',
                            'title': _.noI18n(fileName),
                            'tabindex': 0
                        })
                        .tooltip(),
                        templatePreview = $('<div>').addClass('new-blank template-preview')
                            .append($('<i>').addClass('fa fa-plus')),
                        templateName = $('<p>').addClass('template-name').text(_.noI18n(fileName));

                    // add new blank template
                    templateItem.append(templatePreview, templateName);

                    templateItem.click(function () {
                        CoreActions.invoke(invokeId, this, baton);
                    });

                    templateItem.on('keydown', PortalUtils.keydownHandler);

                    this.append(templateItem);
                };

            };

            // create extension points
            this.text = Ext.point(templateExtension + '/text');
            this.textuser = Ext.point(templateExtension + '/text/templateuser');
            this.textglobal = Ext.point(templateExtension + '/text/templateglobal');
            this.spreadsheet = Ext.point(templateExtension + '/spreadsheet');
            this.spreadsheetuser = Ext.point(templateExtension + '/spreadsheet/templateuser');
            this.spreadsheetglobal = Ext.point(templateExtension + '/spreadsheet/templateglobal');

            // action to create a blank text document
            new Links.Action(templateExtension + '/action/new/text', {
                action: function (baton) {
                    ox.launch('io.ox/office/text/main', { action: 'new', folderId: baton.app.folder.get() });
                }
            });

            // action to create a blank spreadsheet document
            new Links.Action(templateExtension + '/action/new/spreadsheet', {
                action: function (baton) {
                    ox.launch('io.ox/office/spreadsheet/main', { action: 'new', folderId: baton.app.folder.get() });
                }
            });

            // action to create from an existing template
            new Links.Action(templateExtension + '/action/open', {
                action: function (baton) {
                    var appModuleName = PortalUtils.getModuleName(baton.data.filename);
                    ox.launch(appModuleName + '/main', { action: 'convert', template: true, folderId: baton.data.folder_id, templateFile: baton.data, initial_filename: gt('unnamed') });
                }
            });

            // define extension for creating text document with a blank template
            this.text.extend({
                id: 'text_template_blank',
                draw: getBlankDrawFunction(gt('Blank text document'), templateExtension + '/action/new/text')
            });

            // define extension for creating text documents with global templates
            this.textglobal.extend({
                id: 'text_global_template',
                draw: getDrawFunction('template-text-global', { isGlobal: true })
            });

            // define extension for creating text documents with user templates
            this.textuser.extend({
                id: 'text_user_template',
                draw: getDrawFunction('template-text-local', { isGlobal: false })
            });

            // define extension for creating spreadsheet with a blank template
            this.spreadsheet.extend({
                id: 'spreadsheet_template_blank',
                draw: getBlankDrawFunction(gt('Blank spreadsheet'), templateExtension + '/action/new/spreadsheet')
            });

            // define extension for creating spreadsheet with global templates
            this.spreadsheetglobal.extend({
                id: 'spreadsheet_global_template',
                draw: getDrawFunction('template-spreadsheet-global', { isGlobal: true })
            });

            // define extension for creating spreadsheet with user templates
            this.spreadsheetuser.extend({
                id: 'spreadsheet_user_template',
                draw: getDrawFunction('template-spreadsheet-local', { isGlobal: false })
            });

        };

        return new TemplateExtensions();

    });
