/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/editframework/view/control/userbadge', [
    'io.ox/contacts/api',
    'io.ox/office/tk/utils',
    'io.ox/office/tk/forms',
    'io.ox/office/tk/control/picture',
    'io.ox/office/editframework/utils/editconfig',
    'io.ox/office/editframework/view/editlabels',
    'gettext!io.ox/office/editframework'
], function (ContactsAPI, Utils, Forms, Picture, Config, Labels, gt) {

    'use strict';

    var // the picture size of profile photos
        PICTURE_SIZE = 32;

    // class UserBadge ========================================================

    /**
     * A badge displaying properties of an OX user, consisting the user's
     * profile picture, display name, and active status.
     *
     * @constructor
     *
     * @extends Picture
     *
     * @param {Object} client
     *  The client descriptor of the user, containing identifiers and user
     *  name. See method EditApplication.getActiveClients() for details.
     */
    function UserBadge(app, client, initOptions) {

        var // self reference
            self = this,

            // the current client data
            clientData = _.clone(client),

            // the link containing the user name
            userNameNode = $('<a class="user-name halo-link">'),

            // the status node at the right border
            userStatusNode = $('<div class="user-status">'),

            // the color box in the status node
            userColorNode = $('<div class="user-color">').appendTo(userStatusNode);

        // base constructor ---------------------------------------------------

        Picture.call(this, Labels.DEFAULT_PROFILE_PICTURE, Utils.extendOptions({
            tooltip: _.noI18n(client.userName),
            alt: client.userName
        }, initOptions, {
            pictureWidth: PICTURE_SIZE,
            pictureHeight: PICTURE_SIZE,
            backgroundMode: true
        }));

        // private methods ----------------------------------------------------

        /**
         * Updates the visual appearance of this badge (user name, profile
         * picture, and state of edit rights).
         */
        function refreshElements() {

            var // the user name to be inserted into the elements
                userName = _.noI18n(clientData.userName),
                // size of the profile picture, double picture resolution for retina displays
                pictureSize = PICTURE_SIZE * (Utils.RETINA ? 2 : 1),
                // parameters needed to resolve the picture
                pictureParams = { internal_userid: clientData.userId, width: pictureSize, height: pictureSize, scaleType: 'cover' },
                // request the profile picture URL
                pictureUrl = ContactsAPI.pictureHalo(null, pictureParams),
                // the promise for a temporary image node
                imagePromise = app.createImageNode(pictureUrl);

            // check the image size (if the image is 1x1 pixel, the user does not have a profile picture uploaded yet)
            imagePromise = imagePromise.then(function (imgNode) {
                if (imgNode[0].width === 1) { return $.Deferred().reject(); }
            });

            // set the picture URL on success (use listenTo() in case the image resolves after destroying this control)
            self.listenTo(imagePromise, 'done', function () { self.setSource(pictureUrl); });
            self.listenTo(imagePromise, 'fail', function () { self.setSource(Labels.DEFAULT_PROFILE_PICTURE); });

            // add current user name and user identifier to link
            userNameNode.text(_.noI18n(userName)).data('internal_userid', clientData.userId);
            self.setToolTip(_.noI18n(userName));
            self.getImageNode().attr('alt', userName);

            // add color for remote users
            userStatusNode.toggleClass('user-' + clientData.colorIndex, Config.SHOW_REMOTE_SELECTIONS && clientData.remote);

            // mark user with editing rights with an appropriate icon
            userColorNode.empty().removeAttr('title');
            if (clientData.editor) {
                userColorNode.append(Forms.createIconMarkup(Labels.EDIT_ICON));
                userColorNode.attr('title', clientData.remote ?
                    //#. shown as tooltip in a list of all users working on the same document
                    gt('This user has edit rights for the document.') :
                    //#. shown as tooltip in a list of all users working on the same document
                    gt('You have edit rights for the document.')
                );
            }
        }

        // public methods -----------------------------------------------------

        /**
         * Updates the visual appearance of this badge according to the passed
         * client descriptor.
         *
         * @returns {UserBadge}
         *  A reference to this instance.
         */
        this.updateClient = function (newClient) {
            clientData = _.clone(newClient);
            refreshElements();
            return this;
        };

        // initialization -----------------------------------------------------

        // initialization of the root node of this badge
        this.getNode().addClass('user-badge').attr('role', 'dialog');

        // insert the additional nodes into the group
        this.addChildNodes($('<div class="user-name">').append(userNameNode), userStatusNode);

        // initialize visual appearance (user name, profile picture, etc.)
        refreshElements();

        // destroy all class members
        this.registerDestructor(function () {
            app = client = initOptions = self = clientData = null;
            userNameNode = userStatusNode = userColorNode = null;
        });

    } // class UserBadge

    // exports ================================================================

    // derive this class from class Picture
    return Picture.extend({ constructor: UserBadge });

});
