/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/editframework/view/control/tablesizepicker', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/keycodes',
    'io.ox/office/tk/forms',
    'io.ox/office/tk/canvaswrapper',
    'io.ox/office/tk/popup/basemenu',
    'io.ox/office/tk/popup/tooltip',
    'io.ox/office/tk/control/button',
    'io.ox/office/tk/control/menumixin',
    'gettext!io.ox/office/editframework'
], function (Utils, KeyCodes, Forms, CanvasWrapper, BaseMenu, ToolTip, Button, MenuMixin, gt) {

    'use strict';

    var // width of a single grid cell, in pixels
        CELL_WIDTH = Modernizr.touch ? 27 : 18,

        // height of a single grid cell, in pixels
        CELL_HEIGHT = Modernizr.touch ? 24 : 16;

    // class TableSizePicker ==================================================

    /**
     * A drop-down button and a drop-down menu containing a resizable grid
     * allowing to select a specific size.
     *
     * @constructor
     *
     * @extends Button
     * @extends MenuMixin
     *
     * @param {Object} [initOptions]
     *  Optional parameters. Supports all options that are supported by the
     *  base class Button. Additionally, the following options are supported:
     *  @param {Integer} [initOptions.maxCols=15]
     *      The maximal number of columns to be displayed in the control.
     *  @param {Integer} [initOptions.maxRows=15]
     *      The maximal number of rows to be displayed in the control.
     */
    function TableSizePicker(app, initOptions) {

        var // self reference
            self = this,

            // minimum size allowed to choose
            MIN_SIZE = { width: 1, height: 1 },
            // maximum size allowed to choose (defaulting to 15 columns and rows)
            MAX_SIZE = { width: Utils.getIntegerOption(initOptions, 'maxCols', 15, 1), height: Utils.getIntegerOption(initOptions, 'maxRows', 15, 1) },
            // minimum size visible even if selected size is smaller
            MIN_VIEW_SIZE = { width: Math.min(MAX_SIZE.width, 6), height: Math.min(MAX_SIZE.height, 4) },

            // maximum grid size, according to options, and screen size
            maxGridSize = _.clone(MAX_SIZE),

            // current grid size
            gridSize = _.clone(MIN_SIZE),

            // the drop-down menu instance (must be created after Button base constructor!)
            menu = null,

            // the button control to be inserted into the drop-down menu
            // since 'aria-label' would override 'aria-describedby' we set the title attribute only
            gridButtonNode = $(Forms.createButtonMarkup({ attributes: { title: gt('Table size'), role: 'menuitem' } })),

            // canvas element for rendering the grid lines
            canvasWrapper = new CanvasWrapper(),

            // the tooltip node showing the current grid size
            sizeToolTip = null,

            // whether to grow to the left or right or to the top or bottom
            growTop = false, growLeft = false,

            // current position of mouse pointer (page coordinates)
            mousePos = null;

        // base constructors --------------------------------------------------

        Button.call(this, Utils.extendOptions({
            icon: 'docs-table-insert',
            tooltip: gt('Insert table'),
            attributes: { 'aria-haspopup': true, 'aria-expanded': false }
        }, initOptions));
        menu = new BaseMenu({
            classes: 'table-size-picker',
            anchor: this.getNode(),
            prepareLayoutHandler: prepareLayoutHandler
        });
        MenuMixin.call(this, menu, { button: this.getButtonNode() });

        // private methods ----------------------------------------------------

        /**
         * Handles 'mousemove' events in the open drop-down menu element.
         */
        function gridMouseMoveHandler(event) {
            mousePos = { x: event.pageX, y: event.pageY };
            menu.refreshImmediately();
            return false;
        }

        /**
         * Handles keyboard events in the open drop-down menu element.
         */
        function gridKeyDownHandler(event) {

            // update the grid size according to the pressed cursor key
            switch (event.keyCode) {
            case KeyCodes.LEFT_ARROW:
                gridSize.width += (growLeft ? 1 : -1);
                break;
            case KeyCodes.UP_ARROW:
                gridSize.height += (growTop ? 1 : -1);
                break;
            case KeyCodes.RIGHT_ARROW:
                gridSize.width += (growLeft ? -1 : 1);
                break;
            case KeyCodes.DOWN_ARROW:
                gridSize.height += (growTop ? -1 : 1);
                break;
            default:
                return;
            }

            // clear last cached mouse position when using cursor keys, repaint the grid
            mousePos = null;
            menu.refreshImmediately();
            return false;
        }

        /**
         * Unbind the global event listener registered at the document.
         */
        function unbindGlobalEventHandlers() {
            $(document).off('mousemove', gridMouseMoveHandler);
        }

        /**
         * Updates the size of the table grid according to the available space
         * around this form control.
         */
        function prepareLayoutHandler(anchorPosition, availableSizes) {

            // decide where to grow the grid to
            growTop = (availableSizes.bottom.height < 200) && (availableSizes.top.height > availableSizes.bottom.height);
            growLeft = availableSizes.left.width > availableSizes.right.width;

            // calculate maximum allowed size of the grid according to available space in the window
            maxGridSize.width = Math.min(MAX_SIZE.width, Math.floor((anchorPosition.width + (growLeft ? availableSizes.left.width : availableSizes.right.width)) / CELL_WIDTH));
            maxGridSize.height = Math.min(MAX_SIZE.height, Math.floor((growTop ? availableSizes.top.height : availableSizes.bottom.height) / CELL_HEIGHT));

            // update grid size, if a mouse position is available
            if (mousePos) {
                (function () {

                    var // position/size of the grid
                        gridPosition = Utils.getNodePositionInPage(canvasWrapper.getNode()),
                        // mouse position relative to grid origin
                        mouseRelX = growLeft ? (gridPosition.left + gridPosition.width - mousePos.x) : (mousePos.x - gridPosition.left),
                        mouseRelY = growTop ? (gridPosition.top + gridPosition.height - mousePos.y) : (mousePos.y - gridPosition.top);

                    // enlarge grid if the last column/row is covered more than 80% of its width/height
                    gridSize = { width: Math.floor(mouseRelX / CELL_WIDTH + 1.2), height: Math.floor(mouseRelY / CELL_HEIGHT + 1.2) };
                }());
            }

            // validate grid size according to current limits
            gridSize.width = Utils.minMax(gridSize.width, MIN_SIZE.width, maxGridSize.width);
            gridSize.height = Utils.minMax(gridSize.height, MIN_SIZE.height, maxGridSize.height);

            // render the grid
            canvasWrapper.initialize({
                width: Math.max(gridSize.width, MIN_VIEW_SIZE.width) * CELL_WIDTH + 1,
                height: Math.max(gridSize.height, MIN_VIEW_SIZE.height) * CELL_HEIGHT + 1
            });
            canvasWrapper.render(function (context, width, height) {

                var // the pixel size of the highlighted area
                    highlightWidth = gridSize.width * CELL_WIDTH + 1,
                    highlightHeight = gridSize.height * CELL_HEIGHT + 1,
                    // a path object containing all line segments of the grid
                    path = context.createPath();

                // render the filled area
                context.setLineStyle(null).setFillStyle({ style: '#c8c8c8' });
                context.drawRect(growLeft ? (width - highlightWidth) : 0, growTop ? (height - highlightHeight) : 0, highlightWidth, highlightHeight);

                // render the lines
                for (var x = 0; x <= width; x += CELL_WIDTH) { path.pushLine(x, 0, x, height); }
                for (var y = 0; y <= height; y += CELL_HEIGHT) { path.pushLine(0, y, width, y); }
                context.setLineStyle({ style: 'black', width: 1 });
                context.setGlobalAlpha(0.1);
                context.translate(0.5, 0.5);
                context.drawPath(path);
            });

            // update the tooltip
            sizeToolTip.setText(_.noI18n(gt.format(
                //#. Tooltip for an 'Insert table' toolbar control
                //#. %1$d is the number of columns currently selected with the mouse
                //#, c-format
                gt.ngettext('%1$d column', '%1$d columns', gridSize.width),
                _.noI18n(gridSize.width)
            ) + ' \xd7 ' + gt.format(
                //#. Tooltip for an 'Insert table' toolbar control
                //#. %1$d is the number of rows currently selected with the mouse
                //#, c-format
                gt.ngettext('%1$d row', '%1$d rows', gridSize.height),
                _.noI18n(gridSize.height)
            )));

            // return precalculated position and alignment to the anchor
            return { anchorBorder: growTop ? 'top' : 'bottom', anchorAlign: growLeft ? 'trailing' : 'leading' };
        }

        /**
         * Dynamically calculates the position and alignment of the grid size
         * tooltip according to the growing direction of the pop-up node.
         */
        function toolTipPrepareLayoutHandler() {
            var anchorBorder = growTop ? 'top bottom' : 'bottom top',
                anchorAlign = growLeft ? 'trailing' : 'leading';
            if (Modernizr.touch) {
                anchorBorder = (growLeft ? 'left' : 'right') + ' ' + anchorBorder;
                anchorAlign = (growTop ? 'trailing' : 'leading') + ' ' + anchorAlign;
            }
            return { anchorBorder: anchorBorder, anchorAlign: anchorAlign };
        }

        /**
         * Handles 'popup:beforeshow' events and initializes the drop-down
         * grid. Registers a 'mouseenter' handler at the drop-down menu that
         * starts a 'mousemove' listener when the mouse will hover the grid
         * element the first time.
         */
        function popupBeforeShowHandler() {

            // unbind all running global event handlers
            unbindGlobalEventHandlers();

            // initialize and render the grid node
            gridSize = _.clone(MIN_SIZE);
            mousePos = null;

            // wait for mouse to enter the grid before listening globally to mousemove events
            gridButtonNode.off('mouseenter').one('mouseenter', function () {
                $(document).on('mousemove', gridMouseMoveHandler);
            });
        }

        /**
         * Handles 'popup:show' events, shows the tooltip node.
         */
        function popupShowHandler() {
            sizeToolTip.show();
        }

        /**
         * Handles 'popup:hide' events.
         */
        function popupHideHandler() {
            sizeToolTip.hide();
            unbindGlobalEventHandlers();
        }

        // initialization -----------------------------------------------------

        // embed the button control in the drop-down menu
        gridButtonNode.append(canvasWrapper.getNode());
        menu.appendContentNodes(gridButtonNode);

        // create the tooltip that shows the table size currently selected
        sizeToolTip = new ToolTip({ anchor: menu.getNode(), prepareLayoutHandler: toolTipPrepareLayoutHandler });

        // ARIA additions
        sizeToolTip.getNode().attr({ id: sizeToolTip.getUid(), 'aria-live': 'assertive', 'aria-relevant': 'additions', 'aria-atomic': true, role: 'tooltip' });
        gridButtonNode.attr('aria-describedby', sizeToolTip.getUid());

        // button click handler (convert ENTER, SPACE, and TAB keys to click events)
        Forms.setButtonKeyHandler(gridButtonNode, { tab: true });
        gridButtonNode.on('click', function (event) {
            // bug 31170: restrict to 'click' events that originate from keyboard shortcuts
            // (prevent double trigger from 'tracking:end' and 'click')
            if (_.isNumber(event.keyCode)) {
                self.triggerChange(gridSize, { sourceEvent: event });
            }
        });

        // register event handler for tracking (bug 28583, needed for touch devices)
        gridButtonNode.on('tracking:end', function (event) {
            self.triggerChange(gridSize, { sourceEvent: event });
        });

        // register event handlers
        menu.on({ 'popup:beforeshow': popupBeforeShowHandler, 'popup:show': popupShowHandler, 'popup:hide': popupHideHandler });

        // inclusive "start" for the first touch on mobile
        gridButtonNode.on('tracking:start tracking:move', gridMouseMoveHandler);
        menu.getContentNode().on('keydown', gridKeyDownHandler);

        // bug 28583: enable tracking for touch devices
        gridButtonNode.enableTracking();

        // destroy all class members on destruction
        this.registerDestructor(function () {
            unbindGlobalEventHandlers();
            canvasWrapper.destroy();
            sizeToolTip.destroy();
            gridButtonNode.remove();
            app = self = menu = gridButtonNode = canvasWrapper = sizeToolTip = null;
        });

    } // class TableSizePicker

    // exports ================================================================

    // derive this class from class Button
    return Button.extend({ constructor: TableSizePicker });

});
