/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/editframework/view/control/stylesheetpicker', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/io',
    'io.ox/office/tk/control/radiolist'
], function (Utils, IO, RadioList) {

    'use strict';

    // class StyleSheetPicker =================================================

    /**
     * A drop-down list control used to select a style sheet from a list. The
     * drop-down list entries will visualize the formatting attributes of the
     * style sheet if possible.
     *
     * @constructor
     *
     * @extends RadioList
     *
     * @param {EditApplication} app
     *  The application containing this control group.
     *
     * @param {String} family
     *  The attribute family of the style sheets visualized by this control.
     *
     * @param {Object} [initOptions]
     *  Additional options passed to the RadioList constructor. Supports all
     *  options of the RadioList class, and the following additional options:
     *  @param {String|String[]} [initOptions.previewFamilies]
     *      The attribute families used to get the formatting options of the
     *      list items representing the style sheets. If omitted, only
     *      attributes of the family specified by the 'family' parameter will
     *      be used.
     *  @param {String[]} [initOptions.sections]
     *      An array specifying the order of all predefined section
     *      identifiers.
     *  @param {String} [initOptions.i18nModulePath]
     *      The module path to the localized JSON object containing the
     *      configuration of translated style sheet names. Each property of the
     *      map must be a string with the English style sheet name as key.
     *      These keys may contain regular expressions which must be placed in
     *      capturing groups (in parentheses). The values of the map are the
     *      translated style sheet names. The placeholders '$1', '$2', etc.
     *      will match the capturing groups defined in the key patterns.
     */
    function StyleSheetPicker(app, family, initOptions) {

        var // self reference
            self = this,

            // the model instance
            docModel = null,

            // the style collection
            styleCollection = null,

            // the path to the translated style sheet names
            modulePath = Utils.getStringOption(initOptions, 'i18nModulePath', ''),

            // the configuration for translated style sheet names
            translationDatabase = {},

            // the cache with translated style sheet names
            translationCache = {};

        // base constructor ---------------------------------------------------

        RadioList.call(this, Utils.extendOptions({
            itemDesign: 'grid',
            sortItems: true,
            updateCaptionMode: 'none'
        }, initOptions));

        // private methods ----------------------------------------------------

        /**
         * Returns the translation of the passed style sheet name for the
         * current locale. Additionally, the passed style name will be adjusted
         * further for GUI display. All words will be capitalized, automatic
         * line breaks before numbers will be prevented by converting the
         * preceding space characters to NBSP characters.
         *
         * @param {String} styleName
         *  The original (English) style sheet name.
         *
         * @returns {String}
         *  The translated and adjusted style sheet name. If no translation is
         *  available, the passed style sheet name will be adjusted and
         *  returned.
         */
        function translateStyleName(styleName) {

            var // the lower-case style name, used as database key
                styleKey = styleName.toLowerCase(),
                // the translated style name
                translatedName = null;

            // first, check the cache of the translation database
            if (styleKey in translationCache) { return translationCache[styleKey]; }

            // try to find a database entry for the base style name
            _.any(translationDatabase, function (value, key) {

                var // the matches for the pattern of the current entry
                    matches = new RegExp('^' + key + '$', 'i').exec(styleName);

                // continue with next database entry, if key does not match
                if (!_.isArray(matches)) { return; }

                // matching entry found, translate it and replace the placeholders
                translatedName = value;
                for (var index = matches.length - 1; index > 0; index -= 1) {
                    translatedName = translatedName.replace(new RegExp('\\$' + index, 'g'), matches[index]);
                }

                // exit the _.any() loop
                return true;
            });

            // adjust the resulting style name for GUI display
            translatedName = (translatedName || Utils.capitalizeWords(styleName)).replace(/ (\d)/g, '\xa0$1');

            // put the translated name into the cache
            return (translationCache[styleKey] = _.noI18n(translatedName));
        }

        /**
         * Writes the translated name of the specified style sheet into the
         * drop-down button.
         */
        function updateHandler(styleId) {
            var styleName = styleId ? styleCollection.getName(styleId) : null;
            self.setLabel(styleName ? translateStyleName(styleName) : '');
        }

        /**
         * Creates a new button in the drop-down menu for the specified style
         * sheet.
         */
        function createOptionButton(styleName, styleId) {

            var // the section identifier for the style sheet
                category = styleCollection.getUICategory(styleId) || '',
                // sorting priority
                priority = Math.floor(styleCollection.getUIPriority(styleId)),
                // the sort index stored at the button for lexicographical sorting
                sortIndex = String((priority < 0) ? (priority + 0x7FFFFFFF) : priority);

            // translate and adjust style name
            styleName = translateStyleName(styleName);

            // build a sorting index usable for lexicographical comparison:
            // 1 digit for priority sign, 10 digits positive priority,
            // followed by lower-case translated style sheet name
            sortIndex = ('000000000' + sortIndex).substr(-10);
            sortIndex = ((priority < 0) ? '0' : '1') + sortIndex + styleName.toLowerCase();

            // create the list item, pass sorting index
            self.createOptionButton(styleId, { section: category, label: styleName, tooltip: styleName, sortIndex: sortIndex });
        }

        /**
         * Fills the drop-down list with all known style names, and adds
         * preview CSS formatting to the list items.
         */
        var fillList = this.createDebouncedMethod($.noop, function () {
            // bug 33737, 33747: restore browser focus
            self.getMenu().guardFocusedListItem(function () {
                self.clearMenu();
                // configure the sections
                _.each(Utils.getArrayOption(initOptions, 'sections'), function (sectionId) {
                    self.createMenuSection(sectionId);
                });
                // insert the style sheets
                _.each(styleCollection.getStyleSheetNames(), createOptionButton);
            });
        }, { delay: 1000 });

        /**
         * Handles inserted or modified style sheets in the style collection.
         */
        function insertStyleSheetHandler(event, styleId) {
            // bug 33737, 33747: restore browser focus
            self.getMenu().guardFocusedListItem(function () {
                self.deleteOptionButton(styleId);
                createOptionButton(styleCollection.getName(styleId), styleId);
            });
        }

        /**
         * Handles deleted style sheets in the style collection.
         */
        function deleteStyleSheetHandler(event, styleId) {
            self.deleteOptionButton(styleId);
        }

        // initialization -----------------------------------------------------

        app.onInit(function () {
            docModel = app.getModel();
            styleCollection = docModel.getStyleCollection(family);
        });

        // add CSS class to menu button and drop-down list for special formatting
        self.getNode().addClass('style-picker family-' + family);
        self.getMenuNode().addClass('app-' + app.getDocumentType() + ' style-picker family-' + family);

        // update translated style name in drop-down button
        this.registerUpdateHandler(updateHandler);

        // add all known style sheets after import and changed style sheets
        this.listenTo(app.getImportPromise(), 'done', function () {
            self.executeDelayed(fillList, 1000);
            self.listenTo(styleCollection, 'insert:stylesheet', insertStyleSheetHandler);
            self.listenTo(styleCollection, 'delete:stylesheet', deleteStyleSheetHandler);
            self.listenTo(styleCollection, 'modify:stylesheet', insertStyleSheetHandler);
            // also reinitialize the preview if theme settings have been changed
            self.listenTo(docModel.getThemeCollection(), 'triggered', fillList);
        });

        // load the translated style names if specified
        if (modulePath.length > 0) {
            this.listenTo(IO.loadResource(modulePath), 'done', function (data) {
                translationDatabase = data;
                translationCache = {};
                if (app.isImportSucceeded()) {
                    fillList();
                }
            });
        }

        // destroy all class members on destruction
        this.registerDestructor(function () {
            app = initOptions = self = docModel = styleCollection = null;
        });

    } // class StyleSheetPicker

    // exports ================================================================

    // derive this class from class RadioList
    return RadioList.extend({ constructor: StyleSheetPicker });

});
