/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/editframework/view/control/colorpicker', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/control/radiolist',
    'io.ox/office/editframework/utils/color',
    'gettext!io.ox/office/editframework'
], function (Utils, RadioList, Color, gt) {

    'use strict';

    var // predefined color definitions
        BUILTIN_COLOR_DEFINITIONS = [
            { name: 'dark-red',    label: gt('Dark red'),    color: { type: 'rgb', value: 'C00000' } },
            { name: 'red',         label: gt('Red'),         color: { type: 'rgb', value: 'FF0000' } },
            { name: 'orange',      label: gt('Orange'),      color: { type: 'rgb', value: 'FFC000' } },
            { name: 'yellow',      label: gt('Yellow'),      color: { type: 'rgb', value: 'FFFF00' } },
            { name: 'light-green', label: gt('Light green'), color: { type: 'rgb', value: '92D050' } },
            { name: 'green',       label: gt('Green'),       color: { type: 'rgb', value: '00B050' } },
            { name: 'light-blue',  label: gt('Light blue'),  color: { type: 'rgb', value: '00B0F0' } },
            { name: 'blue',        label: gt('Blue'),        color: { type: 'rgb', value: '0070C0' } },
            { name: 'dark-blue',   label: gt('Dark blue'),   color: { type: 'rgb', value: '002060' } },
            { name: 'purple',      label: gt('Purple'),      color: { type: 'rgb', value: '7030A0' } }
        ],

        // definitions for the theme color table (in order as shown in the GUI color picker)
        // transformation table: 0 = pure color; negative values = 'shade'; positive values = 'tint'
        SCHEME_COLOR_DEFINITIONS = [
            { name: 'background1', label: getBackgroundColorName(1), transformations: [    0, -242, -217, -191, -166, -128 ] },
            { name: 'text1',       label: getTextColorName(1),       transformations: [  128,  166,  191,  217,  242,    0 ] },
            { name: 'background2', label: getBackgroundColorName(2), transformations: [    0, -230, -191, -128,  -64,  -26 ] },
            { name: 'text2',       label: getTextColorName(2),       transformations: [   51,  102,  153,    0, -191, -128 ] },
            { name: 'accent1',     label: getAccentColorName(1),     transformations: [   51,  102,  153,    0, -191, -128 ] },
            { name: 'accent2',     label: getAccentColorName(2),     transformations: [   51,  102,  153,    0, -191, -128 ] },
            { name: 'accent3',     label: getAccentColorName(3),     transformations: [   51,  102,  153,    0, -191, -128 ] },
            { name: 'accent4',     label: getAccentColorName(4),     transformations: [   51,  102,  153,    0, -191, -128 ] },
            { name: 'accent5',     label: getAccentColorName(5),     transformations: [   51,  102,  153,    0, -191, -128 ] },
            { name: 'accent6',     label: getAccentColorName(6),     transformations: [   51,  102,  153,    0, -191, -128 ] }
        ];

    // private global functions ===============================================

    /**
     * Returns the localized name of a text color in a color scheme.
     */
    function getTextColorName(index) {
        var label =
            //#. The name of a text color in a color scheme (a color scheme consists
            //#. of two text colors, two background colors, and six accent colors).
            //#. Example result: "Text 1", "Text 2"
            //#. %1$d is the index of the text color
            //#, c-format
            gt('Text %1$d', _.noI18n(index));
        return label;
    }

    /**
     * Returns the localized name of a background color in a color scheme.
     */
    function getBackgroundColorName(index) {
        var label =
            //#. The name of a background color in a color scheme (a color scheme consists
            //#. of two text colors, two background colors, and six accent colors).
            //#. Example result: "Background 1", "Background 2"
            //#. %1$d is the index of the background color
            //#, c-format
            gt('Background %1$d', _.noI18n(index));
        return label;
    }

    /**
     * Returns the localized name of an accented color in a color scheme.
     */
    function getAccentColorName(index) {
        var label =
            //#. The name of an accent color in a color scheme (a color scheme consists
            //#. of two text colors, two background colors, and six accent colors).
            //#. Example result: "Accent 1", "Accent 2"
            //#. %1$d is the index of the accent color
            //#, c-format
            gt('Accent %1$d', _.noI18n(index));
        return label;
    }

    // class ColorPicker ======================================================

    /**
     * Creates a control with a drop-down menu used to choose a color from a
     * set of color items. Shows a selection of standard colors, and a table of
     * shaded scheme colors from the current document theme.
     *
     * @constructor
     *
     * @extends RadioList
     *
     * @param {EditApplication} app
     *  The application containing this control group.
     *
     * @param {String|Object} autoColor
     *  Additional information needed to resolve the automatic color. Can be a
     *  string for predefined automatic colors, or an explicit color object
     *  with a value other than the automatic color. See method
     *  Color.getCssColor() for more details.
     *
     * @param {Object} [initOptions]
     *  Optional parameters. Supports all options of the RadioList base class.
     */
    function ColorPicker(app, autoColor, initOptions) {

        var // self reference
            self = this,

            // the color box in the drop-down button
            colorBox = $('<div>').addClass('color-preview-box'),

            // the style sheet containers of the document
            documentStyles = app.getModel().getDocumentStyles(),

            // show the 'auto-color' entry in the drop down
            showAutoColor = Utils.getBooleanOption(initOptions, 'showAutoColor', true);

        // base constructor ---------------------------------------------------

        RadioList.call(this, Utils.extendOptions({
            itemDesign: 'grid',
            gridColumns: 10,
            updateCaptionMode: 'none',
            itemMatcher: sameColors,
            dropDownVersion: { label: Utils.getStringOption(initOptions, 'tooltip') }
        }, initOptions));

        // private methods ----------------------------------------------------

        /**
         * Converts the passed color value to a CSS color string.
         */
        function setBackgroundColor(boxNode, color) {
            var cssColor = _.isObject(color) ? documentStyles.getCssColor(color, autoColor) : 'transparent';
            boxNode.css('background-color', cssColor);
        }

        /**
         * Initializes a new color item button.
         */
        function createItemHandler(event, buttonNode, color) {
            var boxNode = $('<span class="color-button">');
            setBackgroundColor(boxNode, color);
            buttonNode.prepend(boxNode);
        }

        /**
         * Updates the color box in the drop-down menu button.
         */
        function updateHandler(color) {
            setBackgroundColor(colorBox, color);
        }

        /**
         * Returns whether the passed color objects are considered being equal.
         */
        function sameColors(color1, color2) {

            // quick check: same reference
            if (color1 === color2) { return true; }

            // both colors must be objects
            if (!_.isObject(color1) || !_.isObject(color2)) { return false; }

            var color1Value = _.isString(color1.value) ? color1.value.toLowerCase() : null,
                color2Value = _.isString(color2.value) ? color2.value.toLowerCase() : null;

            // returns whether type and value are considered being equal
            function sameTypeAndValue() {

                // exactly the same type and value
                if ((color1.type === color2.type) && ((color1.type === 'auto') || (color1Value === color2Value))) { return true; }

                // scheme colors: match 'textN' and 'darkN', as well as 'backgroundN' and 'lightN'
                // TODO: really always, also in dark color schemes where background would be dark?
                return (color1.type === 'scheme') && (color2.type === 'scheme') && (
                    ((color1Value === 'text1') && (color2Value === 'dark1')) ||
                    ((color1Value === 'dark1') && (color2Value === 'text1')) ||
                    ((color1Value === 'text2') && (color2Value === 'dark2')) ||
                    ((color1Value === 'dark2') && (color2Value === 'text2')) ||
                    ((color1Value === 'background1') && (color2Value === 'light1')) ||
                    ((color1Value === 'light1') && (color2Value === 'background1')) ||
                    ((color1Value === 'background2') && (color2Value === 'light2')) ||
                    ((color1Value === 'light2') && (color2Value === 'background2')));
            }

            // returns whether transformations are considered being equal
            function sameTransformations() {

                // transformations may be missing in both colors
                if (_.isUndefined(color1.transformations) && _.isUndefined(color2.transformations)) { return true; }

                // arrays must be equally sized, and transformations must be similar enough (bug 30085)
                return _.isArray(color1.transformations) && _.isArray(color2.transformations) &&
                    (color1.transformations.length === color2.transformations.length) &&
                    _.all(color1.transformations, function (tf1, index) {
                        var tf2 = color2.transformations[index];
                        return _.isString(tf1.type) && (tf1.type === tf2.type) &&
                            _.isNumber(tf1.value) && _.isNumber(tf2.value) &&
                            // may not be completely equal (bug 30085)
                            (Math.abs(tf1.value - tf2.value) <= 256);
                    });
            }

            // compare colors directly
            if (sameTypeAndValue() && sameTransformations()) {
                return true;
            }

            // special treatment for fall-back values in ODF mode
            return app.isODF() &&
                ((color1.type === 'scheme') && _.isString(color1.fallbackValue) && (color1.fallbackValue.toLowerCase() === color2Value)) ||
                ((color2.type === 'scheme') && _.isString(color2.fallbackValue) && (color2.fallbackValue.toLowerCase() === color1Value));
        }

        /**
         * Creates the option buttons for one table row of scheme colors.
         */
        function fillSchemeColorRow(rowIndex) {
            _.each(SCHEME_COLOR_DEFINITIONS, function (definition) {

                var // the encoded transformation
                    encoded = definition.transformations[rowIndex],
                    // decoded tint/shade value
                    tint = encoded > 0,
                    value = Math.round(Math.abs(encoded) / 255 * 100000),
                    // the theme color object
                    color = { type: 'scheme', value: definition.name },
                    // the data value added at the button
                    dataValue = definition.name,
                    // description of the color (tool tip)
                    label = definition.label,
                    // shade/tint value as integral percentage
                    percent = (100 - Math.round(value / 1000)),
                    // fall-back value for format without theme support
                    fallbackValue = null;

                if (value !== 0) {
                    Color.appendTransformation(color, tint ? 'tint' : 'shade', value);
                    dataValue += (tint ? '-lighter' : '-darker') + percent;
                    label = tint ?
                        //#. The full name of a light theme color (a base color lightened by a specific percentage value)
                        //#. Example result: "Green, lighter 20%"
                        //#. %1$s is the name of the base color
                        //#. %2$d is the percentage value, followed by a literal percent sign
                        //#, c-format
                        gt('%1$s, lighter %2$d%', label, _.noI18n(percent)) :
                        //#. The full name of a dark theme color (a base color darkened by a specific percentage value)
                        //#. Example result: "Green, darker 20%"
                        //#. %1$s is the name of the base color
                        //#. %2$d is the percentage value, followed by a literal percent sign
                        //#, c-format
                        gt('%1$s, darker %2$d%', label, _.noI18n(percent));
                }

                // store the resulting RGB color in the color value object
                if (Color.isThemeColor(color)) {
                    fallbackValue = documentStyles.getCssColor(color, 'fill');
                    if (fallbackValue[0] === '#') {
                        color.fallbackValue = fallbackValue.slice(1);
                    } else {
                        Utils.warn('ColorPicker.fillSchemeColorRow(): unexpected CSS fallback color: ', fallbackValue);
                    }
                }

                self.createOptionButton(color, { section: 'theme', tooltip: label, dataValue: dataValue });
            });
        }

        /**
         * Returns the button label for the automatic color.
         */
        function getAutoColorLabel() {
            return Color.isTransparentColor(Color.AUTO, autoColor) ?
                /*#. no fill color, transparent */ gt('No color') :
                /*#. automatic text color (white on dark backgrounds, otherwise black) */ gt('Automatic color');
        }

        /**
         * Inserts all available colors into the drop-down menu.
         */
        function initializeColorTable() {

            self.clearMenu();

            // add automatic color
            if (showAutoColor) {
                self.createMenuSection('auto', { gridColumns: 1 })
                    .createOptionButton(Color.AUTO, { section: 'auto', label: getAutoColorLabel(), dataValue: 'auto' });
            }

            // add scheme colors
            if (documentStyles.hasSchemeColors()) {
                self.createMenuSection('theme', { label: gt('Theme Colors') });
                _.times(SCHEME_COLOR_DEFINITIONS[0].transformations.length, fillSchemeColorRow);
            }

            // add predefined colors
            self.createMenuSection('standard', { label: gt('Standard Colors') });
            _.each(BUILTIN_COLOR_DEFINITIONS, function (definition) {
                self.createOptionButton(definition.color, { section: 'standard', tooltip: definition.label, dataValue: definition.name });
            });
        }

        /**
         * Debounced version of the method initializeColorTable().
         */
        var initializeColorTableDebounced = this.createDebouncedMethod($.noop, initializeColorTable, { delay: 50 });

        // public methods -----------------------------------------------------

        /**
         * Changes the information needed to resolve the automatic color, and
         * repaints the menu list item for the automatic color.
         *
         * @param {String|Object} newAutoColor
         *   Can be a string for predefined automatic colors, or an explicit
         *   color object with a value other than the automatic color. See
         *   method Color.getCssColor() for more details.
         *
         * @returns {ColorPicker}
         *  A reference to this instance.
         */
        this.setAutoColor = function (newAutoColor) {
            if (!_.isEqual(autoColor, newAutoColor)) {
                autoColor = newAutoColor;
                setBackgroundColor(this.getMenu().findItemNodes(Color.AUTO).find('.color-button'), Color.AUTO);
            }
            return this;
        };

        // initialization -----------------------------------------------------

        // add marker class for additional CSS formatting
        this.getMenuNode().addClass('color-picker');

        // add the color preview box to the menu button
        this.getMenuButton().append(colorBox);

        // register an update handler that updates the color box
        this.registerUpdateHandler(updateHandler);

        // register a handler that inserts a color box into each list item
        this.getMenu().on('create:item', createItemHandler);

        // lazy initialization of the color table
        this.getMenu().one('popup:beforeshow', initializeColorTable);

        // refresh color buttons in the drop-down menu after theme has changed
        this.listenTo(app.getImportPromise(), 'done', function () {
            self.listenTo(app.getModel().getThemeCollection(), 'triggered', initializeColorTableDebounced);
        });

        // destroy all members on destruction
        this.registerDestructor(function () {
            self = colorBox = documentStyles = null;
        });

    } // class ColorPicker

    // exports ================================================================

    // derive this class from class RadioList
    return RadioList.extend({ constructor: ColorPicker });

});
