/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/editframework/view/control/borderstylepicker', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/forms',
    'io.ox/office/tk/canvaswrapper',
    'io.ox/office/tk/control/radiolist',
    'io.ox/office/editframework/utils/border',
    'io.ox/office/editframework/view/editlabels'
], function (Utils, Forms, CanvasWrapper, RadioList, Border, Labels) {

    'use strict';

    // private global functions ===============================================

    /**
     * Creates a bitmap for the specified border line style, and returns its
     * data URL.
     *
     * @param {Number} width
     *  The width of the generated bitmap, in pixels.
     *
     * @param {Number} height
     *  The height of the generated bitmap, in pixels.
     *
     * @param {Object} borderStyle
     *  The style descriptor of a border line, with the following properties:
     *  - {String} borderStyle.style
     *      The effective line style: one of 'solid', 'dashed', 'dotted',
     *      'dashDot', or 'dashDotDot'.
     *  - {Number} borderStyle.width
     *      Width of a (single) line in the border style, in pixels. The value
     *      0 will be interpreted as hair line.
     *  - {String} borderStyle.color
     *      The line color, as CSS color value.
     *  - {Number} [borderStyle.count=1]
     *      The number of parallel lines shown in the border.
     *
     * @returns {String}
     *  The data URL of the generated bitmap.
     */
    var getBorderStyleBitmapUrl = (function () {

        var // the canvas element used to generate the bitmaps
            canvasWrapper = new CanvasWrapper(),

            // maps unique bitmap keys values to the data URLs
            bitmapUrls = {};

        // the getBorderStyleBitmapUrl() method to be returned from the local scope
        function getBorderStyleBitmapUrl(width, height, borderStyle) {

            var // number of lines
                lineCount = borderStyle.count || 1,
                // unique key of the generated bitmap
                bitmapKey = width + ',' + height + ',' + borderStyle.style + ',' + borderStyle.width + ',' + borderStyle.color + ',' + lineCount;

            // return data URL of a bitmap already created
            if (bitmapKey in bitmapUrls) {
                return bitmapUrls[bitmapKey];
            }

            // initialize the canvas with the passed bitmap size (implicitly clears the canvas)
            canvasWrapper.initialize({ width: width, height: height });

            if (borderStyle.style !== 'none') {
                // create the bitmap
                canvasWrapper.render(function (context) {

                    var // the path with all line segments (of double lines)
                        path = context.createPath(),
                        // effective line width
                        lineWidth = Math.max(1, borderStyle.width),
                        // dash pattern
                        pattern = Border.getBorderPattern(borderStyle.style, lineWidth),
                        // Y offset of the first line
                        y = Math.floor(height / 2) - lineWidth * (lineCount - 1) - (lineWidth % 2) / 2;

                    context.setLineStyle({ style: borderStyle.color, width: lineWidth, pattern: pattern });
                    if (borderStyle.width < 1) { context.setGlobalAlpha(0.4); }
                    _.times(lineCount, function () {
                        path.pushLine(0, y, width, y);
                        y += 2 * lineWidth;
                    });
                    context.drawPath(path);
                });
            }

            // convert the bitmap to a data URL
            return (bitmapUrls[bitmapKey] = canvasWrapper.getDataURL());
        }

        return getBorderStyleBitmapUrl;
    }());

    // class BorderStylePicker ================================================

    /**
     * A generic drop-down list control for border styles.
     *
     * @constructor
     *
     * @extends RadioList
     *
     * @param {Array} listEntries
     *  An array of descriptors for the entries of the drop-down list. Each
     *  descriptor is an object with the following properties:
     *  - {Any} value
     *      The value associated to the list item.
     *  - {String} label
     *      The text label for the list item.
     *  - {Object} style
     *      The style properties for the generated preview icon, with the
     *      following properties:
     *      - {String} style.style
     *          The line style: one of 'solid', 'dashed', 'dotted', 'dashDot',
     *          or 'dashDotDot'.
     *      - {Number} style.width
     *          Width of a (single) line, in pixels. The value 0 will be
     *          interpreted as hair line.
     *      - {Number} [style.count=1]
     *          Number of parallel lines.
     *
     * @param {Object} [initOptions]
     *  Optional parameters. Supports all options supported by the base class
     *  RadioList.
     */
    function BorderStylePicker(listEntries, initOptions) {

        var // the icon-like element in the drop-down button
            styleBoxNode = null,

            // maps control values to border style descriptors
            borderStyles = {};

        // base constructor ---------------------------------------------------

        initOptions = Utils.extendOptions({
            icon: 'docs-line-style',
            tooltip: Labels.BORDER_STYLE_LABEL,
            updateCaptionMode: 'none'
        }, initOptions);

        RadioList.call(this, Utils.extendOptions({
            dropDownVersion: { label: Utils.getStringOption(initOptions, 'tooltip') }
        }, initOptions));

        // private methods ----------------------------------------------------

        /**
         * Returns the data URL of the preview bitmap for the passed value.
         */
        function getBitmapUrl(value) {
            var borderStyle = (value && (value in borderStyles)) ? borderStyles[value] : { style: 'solid', width: 1, color: '#333333' };
            return getBorderStyleBitmapUrl(32, 18, borderStyle);
        }

        /**
         * Creates an image element showing a preview border line of the
         * specified control value.
         */
        function createStyleBox(value) {
            return $('<img class="border-style-box" src="' + getBitmapUrl(value) + '">');
        }

        /**
         * Initializes a new list item according to the control value.
         */
        function createItemHandler(event, buttonNode, value) {
            Forms.getCaptionNode(buttonNode).prepend(createStyleBox(value));
        }

        // initialization -----------------------------------------------------

        // register a handler that inserts a border box into each list item
        this.getMenu().on('create:item', createItemHandler);

        // create the bitmaps for the icons, and insert the entries into the drop-down list
        _.each(listEntries, function (entry) {
            borderStyles[entry.value] = _.extend({ color: '#333333' }, entry.icon);
            this.createOptionButton(entry.value, { label: entry.label });
        }, this);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            initOptions = styleBoxNode = null;
        });

    } // class BorderStylePicker

    // exports ================================================================

    // derive this class from class RadioList
    return RadioList.extend({ constructor: BorderStylePicker });

});
