/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/editframework/app/editcontroller',
    ['io.ox/office/tk/utils',
     'io.ox/office/baseframework/app/basecontroller',
     'io.ox/office/editframework/utils/editconfig'
    ], function (Utils, BaseController, Config) {

    'use strict';

    // class EditController ===================================================

    /**
     * The base class for controller classes of all OX Documents allowing to
     * edit a document.
     *
     * @constructor
     *
     * @extends BaseController
     *
     * @param {EditApplication} app
     *  The OX Documents application that has created this controller instance.
     *
     * @param {Object} [initOptions]
     *  Optional parameters. Supports all options supported by the base class
     *  BaseController.
     */
    function EditController(app, initOptions) {

        var // self reference
            self = this,

            // the model instance of the passed application
            model = null,

            // the view instance of the passed application
            view = null,

            // all the little controller items
            items = {

                // application and view ---------------------------------------

                // enabled unless the application is in 'internal error' state
                'app/valid': {
                    parent: 'app/imported',
                    enable: function () { return app.getState() !== 'error'; }
                },

                // toggle visibility of the tool bars
                'view/toolbars/show': {
                    // always enabled (also in read-only mode, and in application error state)
                    get: function () { return view.getToolPane().isVisible(); },
                    set: function (state) { view.getToolPane().toggle(state); }
                },

                // the identifier of the active tool bar
                'view/toolbars/tab': {
                    // always enabled (also in read-only mode, and in application error state)
                    get: function () { return view.getToolBarTabs().getActiveTabId(); },
                    set: function (tabId) { view.getToolBarTabs().activateTab(tabId); },
                    focusTarget: function (sourceType) {
                        // when activating via ENTER key, move focus into tool pane
                        return (sourceType === 'keyboard') ? view.getToolPane() : null;
                    }
                },

                'view/searchpane': {
                    enable: 'app/valid',
                    get: function () { return view.getSearchPane().isVisible(); },
                    set: function (state) { view.getSearchPane().toggle(state); },
                    focusTarget: function () { return view.getSearchPane(); },
                    // shortcuts always enable the search pane (no toggling)
                    shortcut: [{ keyCode: 'F', ctrlOrMeta: true, value: true }, { keyCode: 'F3', value: true }]
                },

                // document ---------------------------------------------------

                'document/acquireedit': {
                    parent: 'app/valid',
                    enable: function () { return app.isAcquireEditRightsEnabled(); },
                    set: function () { app.acquireEditRights(); }
                },

                'document/reload': {
                    parent: 'app/imported',
                    enable: function () { return app.getState() === 'error'; },
                    set: function () { app.reloadDocument(); }
                },

                // to be used as parent item for all items that require edit mode
                'document/editable': {
                    parent: 'app/valid',
                    enable: function () { return model.getEditMode(); }
                },
                
                'document/renameenabled': {
                    parent: 'document/editable',
                    enable: function () { return !Config.RENAME_DISABLED; }
                },

                'document/fullname': {
                    get: function () { return app.getFullFileName(); }
                },

                'document/rename': {
                    parent: ['document/editable', 'app/bundled', 'document/renameenabled'],
                    get: function () { return app.getShortFileName(); },
                    set: function (fileName) { return app.rename(fileName); }
                },

                'document/rename/dialog': {
                    parent: 'document/rename',
                    set: function () { return view.showRenameDialog(); }
                },

                'document/undo': {
                    parent: 'document/editable',
                    enable: function () { return view.isUndoAvailable(); },
                    set: function () { return view.undo(); },
                    shortcut: [
                       // bug 33077: restrict to application pane, to not interfere with text field's
                       // native undo/redo and automatic item execution after losing focus
                        { keyCode: 'Z', ctrlOrMeta: true, selector: '.app-pane' },
                        { keyCode: 'BACKSPACE', alt: true, selector: '.app-pane' }
                    ]
                },

                'document/redo': {
                    parent: 'document/editable',
                    enable: function () { return view.isRedoAvailable(); },
                    set: function () { return view.redo(); },
                    shortcut: [
                        // bug 33077: restrict to application pane, to not interfere with text field's
                        // native undo/redo and automatic item execution after losing focus
                        { keyCode: 'Y', ctrlOrMeta: true, selector: '.app-pane' },
                        { keyCode: 'BACKSPACE', shift: true, alt: true, selector: '.app-pane' }
                    ]
                },

                'document/saveas/dialog': {
                    parent: 'app/valid',
                    set: function (type) { return view.showSaveAsDialog(type); }
                },

                'document/autosave': {
                    parent: 'app/valid',
                    get: function () { return app.isAutoSaveEnabled(); },
                    set: function (state) { app.toggleAutoSave(state); }
                },

                'document/users': {
                    parent: 'app/valid',
                    enable : function () { return app.getActiveClients().length > 1; },
                    get : function () { return view.getCollaboratorPopup().isVisible(); },
                    set : function (state) { view.toggleCollaboratorPopup(state); }
                },

                // debug mode -------------------------------------------------

                'debug/enabled': {
                    parent: 'app/valid',
                    enable: function () { return Config.DEBUG; }
                }
            };

        // base constructor ---------------------------------------------------

        BaseController.call(this, app, initOptions);

        // initialization -----------------------------------------------------

        // register item definitions
        this.registerDefinitions(items);

        // initialization after construction (model not available at construction time)
        app.onInit(function () {
            model = app.getModel();
            view = app.getView();
        });

        // update controller after operations, or changed state of edit mode
        app.onImport(function () {
            self.listenTo(app, 'docs:state docs:users', function () { self.update(); });
            self.listenTo(model, 'operations:after change:editmode', function () { self.update(); });
            self.listenTo(model.getUndoManager(), 'change:count', function () { self.update(); });
        });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            app = initOptions = self = model = view = items = null;
        });

    } // class EditController

    // exports ================================================================

    // derive this class from class BaseController
    return BaseController.extend({ constructor: EditController });

});
