/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/drawinglayer/view/drawingframe',
    ['io.ox/core/date',
     'io.ox/office/tk/utils',
     'io.ox/office/tk/forms',
     'io.ox/office/tk/io',
     'io.ox/office/tk/canvaswrapper',
     'io.ox/office/editframework/utils/attributeutils',
     'io.ox/office/drawinglayer/view/drawinglabels',
     'io.ox/office/drawinglayer/lib/canvasjs.min',
     'io.ox/office/editframework/utils/border',
     'less!io.ox/office/drawinglayer/view/drawingstyle'
    ], function (CoreDate, Utils, Forms, IO, CanvasWrapper, AttributeUtils, Labels, CanvasJS, Border) {

    'use strict';

    var // the CSS class name of the drawing content node
        CONTENT_CLASS = 'content',

        // the CSS class used to crop images
        CROPPING_CLASS = 'cropping-frame',

        // the CSS class name of the selection root node
        SELECTION_CLASS = 'selection',

        // the CSS class name for active tracking mode
        TRACKING_CLASS = 'tracking-active',

        CULTURE_INFO = null;

    // private global functions ===============================================

    /**
     * Returns the selection root node of the specified drawing frame.
     *
     * @param {HTMLElement|jQuery} drawingFrame
     *  The root node of the drawing frame. If this object is a jQuery
     *  collection, uses the first DOM node it contains.
     *
     * @returns {jQuery}
     *  The selection root node of the specified drawing frame. Will be an
     *  empty collection, if the drawing frame is not selected currently.
     */
    function getSelectionNode(drawingFrame) {
        return $(drawingFrame).first().children('.' + SELECTION_CLASS);
    }

    /**
     * Calculates the offset and size of the bitmap in an image frame for one
     * dimension (either horizontally or vertically), according to the passed
     * cropping settings.
     *
     * @param {Number} frameSize
     *  Visible size (with or height) of the drawing frame containing the image
     *  bitmap, in 1/100 of millimeters.
     *
     * @param {Number} leadingCrop
     *  The leading cropping value (left/top).
     *
     * @param {Number} trailingCrop
     *  The trailing cropping value (right/bottom).
     *
     * @returns {Object}
     *  An object containing 'offset' and 'size' CSS attributes specifying how
     *  to format the bitmap node (in pixels with 'px' unit name).
     */
    function calculateBitmapSettings(app, frameSize, leadingCrop, trailingCrop) {

        // ODF: absolute cropping length (in 1/100mm), OOXML: percentage
        if (app.isODF()) {
            leadingCrop /= frameSize;
            trailingCrop /= frameSize;
        } else {
            leadingCrop /= 100;
            trailingCrop /= 100;
        }

        var // sum of leading and trailing cropping (must not exceed a specific amount)
            totalCrop = leadingCrop + trailingCrop,
            // resulting settings for the bitmap
            size = 0, offset = 0;

        // do not crop away more than 90% of the bitmap
        if (totalCrop > 0.9) {
            leadingCrop *= (0.9 / totalCrop);
            trailingCrop *= (0.9 / totalCrop);
        }

        // bitmap size and offset, according to object size and cropping
        size = frameSize / (1 - leadingCrop - trailingCrop);
        offset = size * leadingCrop;

        // convert to CSS pixels
        return {
            offset: Utils.convertHmmToCssLength(-offset, 'px', 1),
            size: Utils.convertHmmToCssLength(size, 'px', 1)
        };
    }

    // static class DrawingFrame ==============================================

    /**
     * Contains common static helper functions to display and handle drawing
     * frames in any document.
     */
    var DrawingFrame = {};

    // constants --------------------------------------------------------------

    /**
     * The CSS class used to mark drawing frame nodes.
     *
     * @constant
     */
    DrawingFrame.NODE_CLASS = 'drawing';

    /**
     * A jQuery selector that matches nodes representing a drawing frame.
     *
     * @constant
     */
    DrawingFrame.NODE_SELECTOR = '.' + DrawingFrame.NODE_CLASS;

    /**
     * A jQuery selector that matches nodes representing a content element.
     *
     * @constant
     */
    DrawingFrame.CONTENT_SELECTOR = 'div.' + CONTENT_CLASS;

    /**
     * The CSS class used to mark drawing content nodes as place holder.
     *
     * @constant
     */
    DrawingFrame.PLACEHOLDER_CLASS = 'placeholder';

    /**
     * A jQuery selector that matches nodes representing a place holder element
     * of type 'div'.
     *
     * @constant
     */
    DrawingFrame.PLACEHOLDER_SELECTOR = 'div.' + DrawingFrame.PLACEHOLDER_CLASS;

    /**
     * The CSS class used to mark drawing content nodes as container for further drawings.
     *
     * @constant
     */
    DrawingFrame.GROUPCONTENT_CLASS = 'groupcontent';

    /**
     * A jQuery selector that matches nodes representing a group content element
     * of type 'div'.
     *
     * @constant
     */
    DrawingFrame.GROUPCONTENT_SELECTOR = 'div.' + DrawingFrame.GROUPCONTENT_CLASS;

    /**
     * The CSS class used to mark drawing content nodes inside text frames
     * for automatic resizing of height.
     *
     * @constant
     */
    DrawingFrame.AUTORESIZEHEIGHT_CLASS = 'autoresizeheight';

    /**
     * A jQuery selector that matches nodes representing a content node inside a
     * test frame that automatically resizes its height.
     *
     * @constant
     */
    DrawingFrame.AUTORESIZEHEIGHT_SELECTOR = 'div.' + DrawingFrame.AUTORESIZEHEIGHT_CLASS;

    /**
     * The CSS class used to mark drawing content nodes inside text frames
     * for automatic resizing of height.
     *
     * @constant
     */
    DrawingFrame.ODFTEXTFRAME_CLASS = 'odftextframe';

    /**
     * A jQuery selector that matches nodes representing a text frame node that
     * are 'classical' text frames in odf format.
     *
     * @constant
     */
    DrawingFrame.ODFTEXTFRAME_SELECTOR = 'div.' + DrawingFrame.ODFTEXTFRAME_CLASS;

    /**
     * The name of the jQuery data attribute that stores the unique node
     * identifier of a drawing frame node.
     *
     * @constant
     */
    DrawingFrame.DATA_UID = 'uid';

    /**
     * The name of the jQuery data attribute that stores the object type of a
     * drawing frame node.
     *
     * @constant
     */
    DrawingFrame.DATA_TYPE = 'type';

    /**
     * The name of the jQuery data attribute that stores the drawing model
     * instance for a drawing frame node.
     *
     * @constant
     */
    DrawingFrame.DATA_MODEL = 'model';

    /**
     * A jQuery selector that matches elements representing a text frame element
     * inside a drawing frame of type shape.
     */
    DrawingFrame.TEXTFRAME_NODE_SELECTOR = 'div.textframe';

    /**
     * The CSS class used to mark text frame content nodes inside a
     * drawing node.
     *
     * @constant
     */
    DrawingFrame.TEXTFRAMECONTENT_NODE_CLASS = 'textframecontent';

    /**
     * A jQuery selector that matches nodes representing a text frame content
     * element inside a drawing node.
     *
     * @constant
     */
    DrawingFrame.TEXTFRAMECONTENT_NODE_SELECTOR = 'div.' + DrawingFrame.TEXTFRAMECONTENT_NODE_CLASS;

    /**
     * The CSS class used to mark drawing nodes inside a group container.
     *
     * @constant
     */
    DrawingFrame.GROUPED_NODE_CLASS = 'grouped';

    /**
     * A jQuery selector that matches nodes representing a grouped element.
     *
     * @constant
     */
    DrawingFrame.GROUPED_NODE_SELECTOR = '.' + DrawingFrame.GROUPED_NODE_CLASS;

    /**
     * The CSS class used to mark border nodes of a drawing.
     *
     * @constant
     */
    DrawingFrame.BORDER_CLASS = 'border';

    /**
     * The HTML element used to mark the border of a drawing.
     *
     * @constant
     */
    DrawingFrame.BORDER_NODE = 'canvas';

    /**
     * A jQuery selector that matches nodes representing a border element.
     *
     * @constant
     */
    DrawingFrame.BORDER_NODE_SELECTOR = DrawingFrame.BORDER_NODE + '.' + DrawingFrame.BORDER_CLASS;

    /**
     * The CSS class used to mark unsupported drawing nodes.
     *
     * @constant
     */
    DrawingFrame.UNSUPPORTED_CLASS = 'unsupported';

    /**
     * A jQuery selector that matches nodes representing unsupported drawings.
     *
     * @constant
     */
    DrawingFrame.UNSUPPORTED_SELECTOR = '.' + DrawingFrame.UNSUPPORTED_CLASS;

    // methods ----------------------------------------------------------------

    /**
     * Returns a new drawing frame node.
     *
     * @param {String} type
     *  The type of the drawing.
     *
     * @param {DrawingModel|String} model
     *  The drawing model instance this drawing frame is based on.
     *  Alternatively (for applications not supporting drawing models), this
     *  value can be a string with the type of the drawing object.
     *
     * @returns {jQuery}
     *  A new drawing frame with empty content node, as jQuery object.
     */
    DrawingFrame.createDrawingFrame = function (model) {

        var // whether the drawing frame is a drawing group
            isGroup = (_.isString(model) && model === 'group') || (_.isObject(model) && model.getType() === 'group');

        return $('<div>', { contenteditable: (isGroup && !_.browser.IE) ? true : false })
            .addClass(DrawingFrame.NODE_CLASS)
            .data(DrawingFrame.DATA_UID, 'frame' + _.uniqueId())
            .data(DrawingFrame.DATA_TYPE, _.isObject(model) ? model.getType() : model)
            .data(DrawingFrame.DATA_MODEL, _.isObject(model) ? model : null)
            .on('dragstart', false)
            .append($('<div>').addClass(CONTENT_CLASS).addClass(isGroup ? DrawingFrame.GROUPCONTENT_CLASS : ''));
    };

    /**
     * Returns whether the passed node is the root node of a drawing frame.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is the root node of a drawing frame.
     */
    DrawingFrame.isDrawingFrame = function (node) {
        return $(node).is(DrawingFrame.NODE_SELECTOR);
    };

    /**
     * Returns the type of the specified drawing frame.
     *
     * @param {HTMLElement|jQuery} drawingFrame
     *  The root node of the drawing frame. If this object is a jQuery
     *  collection, uses the first DOM node it contains.
     *
     * @returns {String}
     *  The type of the specified drawing frame, as specified while the frame
     *  has been created.
     */
    DrawingFrame.getDrawingType = function (drawingFrame) {
        return $(drawingFrame).first().data(DrawingFrame.DATA_TYPE);
    };

    /**
     * Returns whether the passed node is a content node inside the drawing frame.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a content node inside a drawing frame.
     */
    DrawingFrame.isDrawingContentNode = function (node) {
        return $(node).is(DrawingFrame.CONTENT_SELECTOR);
    };

    /**
     * Returns whether the passed node is a content node inside the drawing frame
     * that has no content.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a content node inside a drawing frame that has
     *  no content.
     */
    DrawingFrame.isEmptyDrawingContentNode = function (node) {
        return $(node).is(DrawingFrame.CONTENT_SELECTOR) && $(node).children().not(DrawingFrame.BORDER_NODE_SELECTOR).length === 0;
    };

    /**
     * Returns whether the passed node is a placeholder node.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a place holder node (of a drawing frame).
     */
    DrawingFrame.isPlaceHolderNode = function (node) {
        return $(node).is(DrawingFrame.PLACEHOLDER_SELECTOR);
    };

    /**
     * Returns whether the passed node is a drawing frame, that is of type 'shape'.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a drawing frame of type 'shape'.
     */
    DrawingFrame.isShapeDrawingFrame = function (node) {
        return DrawingFrame.isDrawingFrame(node) && DrawingFrame.getDrawingType(node) === 'shape';
    };

    /**
     * Returns whether the passed node is a drawing frame, that is of type 'shape' AND
     * that is used as text frame.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a drawing frame of type 'shape' and it is used
     *  as text frame.
     */
    DrawingFrame.isTextFrameShapeDrawingFrame = function (node) {
        return DrawingFrame.isShapeDrawingFrame(node) && DrawingFrame.getTextFrameNode(node).length > 0;
    };

    /**
     * Returns whether the passed node is a drawing frame, that is of type 'group'.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a drawing frame of type 'group'.
     */
    DrawingFrame.isGroupDrawingFrame = function (node) {
        return DrawingFrame.isDrawingFrame(node) && DrawingFrame.getDrawingType(node) === 'group';
    };

    /**
     * Returns whether the passed node is a drawing frame that is grouped inside a
     * group.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a grouped drawing frame.
     */
    DrawingFrame.isGroupedDrawingFrame = function (node) {
        return DrawingFrame.isDrawingFrame(node) && $(node).is(DrawingFrame.GROUPED_NODE_SELECTOR);
    };

    /**
     * Returns whether the passed node is a drawing frame, that can be moved only
     * in the region near the border. In its center, it can contain text frames.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a drawing frame, that cannot be moved by clicking
     *  in its center.
     */
    DrawingFrame.isOnlyBorderMoveableDrawing = function (node) {
        return DrawingFrame.isTextFrameShapeDrawingFrame(node) || DrawingFrame.isGroupDrawingFrame(node);
    };

    /**
     * Returns whether the passed node is a drawing frame, that is of type 'shape'
     * and that contains text content. Furthermore the drawing is NOT marked as
     * 'classical' odf text frame, so that there is only reduced text functionality
     * available.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a drawing frame, that is of type 'shape'
     *  and that contains text content. Furthermore the drawing is NOT marked as
     *  'classical' odf text frame, so that there is only reduced text functionality
     *  available.
     */
    DrawingFrame.isReducedOdfTextframeNode = function (node) {
        return DrawingFrame.isTextFrameShapeDrawingFrame(node) && !$(node).is(DrawingFrame.ODFTEXTFRAME_SELECTOR);
    };

    /**
     * Returns whether the passed node is a drawing frame, that is of type 'shape'
     * and that contains text content. Furthermore the drawing is marked as
     * 'classical' odf text frame, so that there is no text functionality
     * available.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a drawing frame, that is of type 'shape'
     *  and that contains text content. Furthermore the drawing is marked as
     *  'classical' odf text frame, so that there is no reduced text functionality
     *  available.
     */
    DrawingFrame.isFullOdfTextframeNode = function (node) {
        return DrawingFrame.isTextFrameShapeDrawingFrame(node) && $(node).is(DrawingFrame.ODFTEXTFRAME_SELECTOR);
    };

    /**
     * Returns the number of drawing frames inside a specified drawing group. Only
     * the direct children are counted, not the children in sub groups.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Number|Null}
     *  The number of drawing frames inside a specified drawing group
     */
    DrawingFrame.getGroupDrawingCount = function (node) {
        if (!DrawingFrame.isGroupDrawingFrame(node)) { return null; }
        return $(node).children().first().children().length;
    };

    /**
     * Returns the child drawing frame of a drawing group at a specified
     * position. Or null, if it cannot be determined.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @param {Number} [number]
     *  The zero based number of the child to be returned.
     *
     * @returns {Node|Null}
     *  The child drawing frame of a drawing group at a specified position.
     *  Or null, if it cannot be determined.
     */
    DrawingFrame.getGroupDrawingChildren = function (node, number) {
        if (!DrawingFrame.isGroupDrawingFrame(node) || !_.isNumber(number)) { return null; }
        return $(node).children().first()[0].childNodes[number];
    };

    /**
     * Returns all drawing frames inside a specified drawing group. This does
     * not only contain direct children, but also the children of further groups
     * inside the group. If the option 'onlyDirectChildren' is set to true,
     * only the direct children are returned.
     * Returns null, if the specified node is not a group drawing node.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @param {Object} [options]
     *  A map with options to control the selection. The following options are
     *  supported:
     *  @param {Boolean} [options.onlyDirectChildren=false]
     *      If set to true, only those drawing nodes are returned, that are
     *      direct children of the group.
     *
     * @returns {Node|Null}
     *  The child drawing frames of a drawing group at a specified position.
     *  Or null, if it cannot be determined.
     */
    DrawingFrame.getAllGroupDrawingChildren = function (node, options) {

        if (!DrawingFrame.isGroupDrawingFrame(node)) { return null; }

        if (Utils.getBooleanOption(options, 'onlyDirectChildren', false)) {
            return $(node).children().children(DrawingFrame.NODE_SELECTOR);
        } else {
            return $(node).find(DrawingFrame.NODE_SELECTOR);
        }
    };

    /**
     * Returns the closest drawing node of a given node (for example a canvas.border node).
     * If this cannot be found, null is returned.
     *
     * @param {HTMLDivElement|jQuery} node
     *  If this object is a jQuery collection, uses
     *  the first DOM node it contains.
     *
     * @returns {jQuery|Null}
     *  The drawing group node, or null, if it cannot be determined.
     */
    DrawingFrame.getDrawingNode = function (node) {

        var // the group content node is the direct child of the group node itself
            drawingNode = $(node).closest(DrawingFrame.NODE_SELECTOR);

        return drawingNode ? drawingNode : null;
    };

    /**
     * Returns the closest drawing node of a given node that is a text frame shape
     * drawing node. If this cannot be found, null is returned.
     *
     * @param {HTMLDivElement|jQuery} node
     *  If this object is a jQuery collection, uses the first DOM node it contains.
     *
     * @returns {jQuery|Null}
     *  The text frame drawing node, or null, if it cannot be determined.
     */
    DrawingFrame.getClosestTextFrameDrawingNode = function (node) {

        var // the closest drawing node that is ancestor of the specified node
            drawingNode = DrawingFrame.getDrawingNode(node);

        if (!drawingNode) { return null; }

        return DrawingFrame.isTextFrameShapeDrawingFrame(drawingNode) ? drawingNode : null;
    };

    /**
     * Returns the closest drawing group node of a drawing node that is grouped.
     * If this cannot be found, null is returned.
     *
     * @param {HTMLDivElement|jQuery} drawingNode
     *  The grouped drawing node. If this object is a jQuery collection, uses
     *  the first DOM node it contains.
     *
     * @param {Object} [options]
     *  A map with options to determine the group node. The following options are
     *  supported:
     *  @param {Boolean} [options.farthest=false]
     *      If set to true, that group node is returned, that is the highest in
     *      the dom. This is necessary, because groups can be included into groups.
     *      Otherwise the closest group node is returned.
     *  @param {Node} [options.rootNode]
     *      If the option 'farthest' is set to true, the root node must be determined,
     *      until which the ancestors will be searched.
     *
     * @returns {jQuery|Null}
     *  The closest or farthest drawing group node, or null, if it cannot be determined.
     */
    DrawingFrame.getGroupNode = function (node, options) {

        var // the group content node is the direct child of the group node itself
            groupContentNode = null,
            // the root node, until which the ancestors are searched. This is only
            // used, if the option 'farthest' is set to true
            rootNode = null;

        if (Utils.getBooleanOption(options, 'farthest', false)) {
            rootNode = Utils.getOption(options, 'rootNode');
            if (rootNode) { groupContentNode = $(Utils.findFarthest(rootNode, node, DrawingFrame.GROUPCONTENT_SELECTOR)); }
        } else {
            groupContentNode = $(node).closest(DrawingFrame.GROUPCONTENT_SELECTOR);
        }

        return (groupContentNode && groupContentNode.length) > 0 ? groupContentNode.parent() : null;
    };

    /**
     * Returns the container node of a drawing frame, that contains all
     * top-level content nodes (paragraphs and tables).
     *
     * @param {HTMLDivElement|jQuery} drawingNode
     *  The drawing frame DOM node. If this object is a jQuery collection, uses
     *  the first DOM node it contains.
     *
     * @returns {jQuery}
     *  The container DOM node from the passed table cell that contains all
     *  top-level content nodes (paragraphs and tables).
     */
    DrawingFrame.getTextFrameNode = function (drawingNode) {
        return $(drawingNode).find('> * > ' + DrawingFrame.TEXTFRAME_NODE_SELECTOR);
    };

    /**
     * Returns whether the passed node is a text frame <div> element, that is
     * used inside drawing frames
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a div textframe element.
     */
    DrawingFrame.isTextFrameNode = function (node) {
        return $(node).is(DrawingFrame.TEXTFRAME_NODE_SELECTOR);
    };

    /**
     * Returns whether the passed node is a drawing node, that is a text frame, that
     * automatically resizes its height. In this case the content node contains a
     * specific class.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is an automatically resizing text frame drawing node.
     */
    DrawingFrame.isAutoResizeHeightDrawingFrame = function (node) {
        // checking if the content node inside the text frame drawing has class 'autoresizeheight'
        return DrawingFrame.getContentNode(node).is(DrawingFrame.AUTORESIZEHEIGHT_SELECTOR);
    };

    /**
     * Returns whether the passed node is a drawing node, that is a text frame, that
     * has a fixed height.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a text frame drawing node with fixed height.
     */
    DrawingFrame.isFixedHeightDrawingFrame = function (node) {
        // checking if the content node inside the text frame drawing has NOT the class 'autoresizeheight'
        return DrawingFrame.isTextFrameShapeDrawingFrame(node) && !DrawingFrame.getContentNode(node).is(DrawingFrame.AUTORESIZEHEIGHT_SELECTOR);
    };

    /**
     * Returns whether the passed node is a text frame <div> element, that is
     * used inside drawing frames as container for additional drawings.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a div groupcontent element.
     */
    DrawingFrame.isGroupContentNode = function (node) {
        return $(node).is(DrawingFrame.GROUPCONTENT_SELECTOR);
    };

    /**
     * Returns whether the passed node is a unsupported drawing.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a unsupported drawing element.
     */
    DrawingFrame.isUnsupportedDrawing = function (drawingFrame) {
        return $(drawingFrame).hasClass(DrawingFrame.UNSUPPORTED_CLASS);
    };

    /**
     * Returns the model of the specified drawing frame.
     *
     * @param {HTMLElement|jQuery} drawingFrame
     *  The root node of the drawing frame. If this object is a jQuery
     *  collection, uses the first DOM node it contains.
     *
     * @returns {Object|Null}
     *  The model object of the specified drawing frame, as specified while the
     *  frame has been created.
     */
    DrawingFrame.getModel = function (drawingFrame) {
        return $(drawingFrame).first().data(DrawingFrame.DATA_MODEL);
    };

    /**
     * Returns the unique identifier of the specified drawing frame.
     *
     * @param {HTMLElement|jQuery} drawingFrame
     *  The root node of the drawing frame. If this object is a jQuery
     *  collection, uses the first DOM node it contains.
     *
     * @returns {String}
     *  The unique identifier of the specified drawing frame, as specified
     *  while the frame has been created.
     */
    DrawingFrame.getUid = function (drawingFrame) {
        return $(drawingFrame).first().data(DrawingFrame.DATA_UID);
    };

    /**
     * Returns the root content node of the specified drawing frame containing
     * all type specific contents of the drawing.
     *
     * @param {HTMLElement|jQuery} drawingFrame
     *  The root node of the drawing frame. If this object is a jQuery
     *  collection, uses the first DOM node it contains.
     *
     * @returns {jQuery}
     *  The root content node of the specified drawing frame.
     */
    DrawingFrame.getContentNode = function (drawingFrame) {
        return $(drawingFrame).first().children('.' + CONTENT_CLASS);
    };

    /**
     * Appends a new border canvas element to the DOM and returns it.
     *
     * @param {HTMLElement|jQuery} drawingFrame
     *  The root node of the drawing frame. If this object is a jQuery
     *  collection, uses the first DOM node it contains.
     *
     * @returns {jQuery}
     *  A new CanvasWrapper element for the given drawigFrame.
     */
    DrawingFrame.addBorderNode = function (drawingFrame, canvas) {
        var contentNode = DrawingFrame.getContentNode(drawingFrame);

        contentNode.prepend(canvas);
        return DrawingFrame;
    };

    /**
     * Returns the border node (canvas) of the specified drawing frame
     *
     * @param {HTMLElement|jQuery} drawingFrame
     *  The root node of the drawing frame. If this object is a jQuery
     *  collection, uses the first DOM node it contains.
     *
     * @returns {jQuery}
     *  The border node of the specified drawing frame.
     */
    DrawingFrame.getBorderNode = function (drawingFrame) {
        return $(drawingFrame).find(DrawingFrame.BORDER_NODE_SELECTOR);
    };

    /**
     * Checks, whether the passed node contains a border element
     *
     * @param {HTMLElement|jQuery} drawingFrame
     *  The root node of the drawing frame. If this object is a jQuery
     *  collection, uses the first DOM node it contains.
     *
     * @returns {Boolean}
     *  Whether the passed node contains a border element (canvas).
     *
     */
    DrawingFrame.borderNodeExists = function (drawingFrame) {
        return (DrawingFrame.getBorderNode(drawingFrame).length > 0);
    };

    /**
     * Clears and returns the root content node of the specified drawing frame.
     *
     * @param {HTMLElement|jQuery} drawingFrame
     *  The root node of the drawing frame. If this object is a jQuery
     *  collection, uses the first DOM node it contains.
     *
     * @returns {jQuery}
     *  The empty root content node of the specified drawing frame.
     */
    DrawingFrame.getAndClearContentNode = function (drawingFrame) {
        return DrawingFrame.getContentNode(drawingFrame).empty().removeClass(DrawingFrame.PLACEHOLDER_CLASS).attr('style', '');
    };

    /**
     * Prepares a specified drawing frame for text insertion. This requires
     * that the content node inside drawing frame is cleared and then a new
     * text frame child is appended. This new text frame element is returned
     * and can be used as parent for paragraphs, tables, ... .
     * Only drawingFrames of type 'shape' allow this conversion.
     *
     * @param {HTMLElement|jQuery} drawingFrame
     *  The root node of the drawing frame. If this object is a jQuery
     *  collection, uses the first DOM node it contains.
     *
     * @returns {jQuery}
     *  The new text frame 'div' element, that can be used as container for
     *  the text content.
     */
    DrawingFrame.prepareDrawingFrameForTextInsertion = function (drawingFrame) {

        // only drawing frames of type 'shape' can be converted
        if (!DrawingFrame.isShapeDrawingFrame(drawingFrame)) { return $(); }

        var // the content node of the drawing
            contentNode = DrawingFrame.getAndClearContentNode(drawingFrame),
            // the text frame node inside the content node, parent of paragraphs and tables
            // -> required for setting cursor type, distance to content node and click event
            textFrameNode = $('<div>').addClass('textframe').attr('contenteditable', true);

        // making content node visible by assigning a border via class 'textframecontent'
        contentNode.addClass(DrawingFrame.TEXTFRAMECONTENT_NODE_CLASS).append(textFrameNode);
        // in IE the drawing node itself must have contenteditable 'false', otherwise the
        // grabbers will be visible. Other browsers have to use the value 'true'.
        if (!_.browser.IE) { $(drawingFrame).attr('contenteditable', true); }

        // returning the new created text frame node
        return textFrameNode;
    };

    /**
     * Inserts replacement layout nodes for unsupported drawing types. Inserts
     * the passed name and description of the drawing as text.
     *
     * @param {HTMLElement|jQuery} drawingFrame
     *  The root node of the drawing frame. If this object is a jQuery
     *  collection, uses the first DOM node it contains.
     *
     * @param {String} [name]
     *  The name of the drawing frame.
     *
     * @param {String} [description]
     *  The description text for the drawing frame.
     */
    DrawingFrame.insertReplacementNodes = function (drawingFrame, name, description) {

        var // the type of the drawing frame
            type = DrawingFrame.getDrawingType(drawingFrame),
            // the empty content node
            contentNode = DrawingFrame.getAndClearContentNode(drawingFrame),
            // the inner width and height available in the content node
            innerWidth = Math.max(0, drawingFrame.outerWidth() - 2),
            innerHeight = Math.max(0, drawingFrame.outerHeight() - 2),
            // the vertical padding in the content node
            verticalPadding = Utils.minMax(Math.min(innerWidth, innerHeight) / 24, 1, 6),
            // the font size of the picture icon
            pictureIconSize = Utils.minMax(Math.min(innerWidth - 16, innerHeight - 28), 8, 72),
            // the base font size of the text
            fontSize = Utils.minMax(Math.min(innerWidth, innerHeight) / 4, 9, 13);

        // set border width at the content node, insert the picture icon
        contentNode
            .addClass(DrawingFrame.PLACEHOLDER_CLASS)
            .css({
                padding: Math.floor(verticalPadding) + 'px ' + Math.floor(verticalPadding * 2) + 'px',
                fontSize: fontSize + 'px'
            })
            .append(
                $('<div>')
                    .addClass('abs background-icon')
                    .css('line-height', innerHeight + 'px')
                    .append(Forms.createIconMarkup(Labels.getDrawingTypeIcon(type), { style: 'font-size:' + pictureIconSize + 'px' })),
                $('<p>').text(_.noI18n(name) || Labels.getDrawingTypeLabel(type))
            );

        // insert description if there is a reasonable amount of space available
        if ((innerWidth >= 20) && (innerHeight >= 20)) {
            contentNode.append($('<p>').text(_.noI18n(description)));
        }
    };

    // formatting -------------------------------------------------------------

    /**
     * Updates the CSS formatting of the passed drawing frame, according to the
     * passed generic formatting attributes.
     *
     * @param {EditApplication} app
     *  The application instance containing the drawing frame.
     *
     * @param {HTMLElement|jQuery} drawingFrame
     *  The root node of the drawing frame to be updated. If this object is a
     *  jQuery collection, uses the first DOM node it contains.
     *
     * @param {Object} mergedAttributes
     *  The drawing attribute set (a map of attribute maps as name/value pairs,
     *  keyed by the 'drawing' attribute family), containing the effective
     *  attribute values merged from style sheets and explicit attributes.
     */
    DrawingFrame.updateFormatting = function (app, drawingFrame, mergedAttributes) {

        var // the drawing attributes of the passed attribute set
            drawingAttributes = mergedAttributes.drawing,
            // value of the 'transform' CSS attribute, e.g. for flipped drawings
            transformations = '',
            // type of the drawing object: 'image', ...
            type = DrawingFrame.getDrawingType(drawingFrame),
            // the content node inside the drawing frame
            contentNode = DrawingFrame.getContentNode(drawingFrame),
            // the replacement data for unsupported drawings
            replacementData = Utils.getStringOption(drawingAttributes, 'replacementData', ''),
            // rendering result (false to fall-back to placeholder graphic)
            rendered = false;

        // forehanded, remove class which marks this drawing as unsupported
        drawingFrame.removeClass(DrawingFrame.UNSUPPORTED_CLASS);

        // helper function to get all node attributes (TODO: This needs to be valid for all apps)
        function getAllDrawingAttributes(node, family) {

            var // the style collection of the document
                documentStyles = app.getModel().getDocumentStyles(),
                // the drawing style collection
                styleCollection = documentStyles ? documentStyles.getStyleCollection('drawing') : null,
                // the element specific attributes
                elementAttributes = styleCollection ? styleCollection.getElementAttributes(node) : null,
                // the family specific attributes of the element
                familyAttributes = elementAttributes ? elementAttributes[family] : null;

            return familyAttributes;
        }

        // creates an image node in the content node with the specified source URL
        function createImageNodeFromUrl(srcUrl) {
            return app.createImageNode(srcUrl, { timeout: 15000 }).done(function (imgNode) {
                var cropNode = $('<div class="' + CROPPING_CLASS + '">').append(imgNode);
                DrawingFrame.getAndClearContentNode(drawingFrame).append(cropNode);
            });
        }

        // creates an image node or an SVG image according to the passed image data
        function createImageNodeFromData(imageData) {
            if (/^data:/.test(imageData)) {
                return createImageNodeFromUrl(imageData);
            }
            if (/^<svg/.test(imageData)) {
                DrawingFrame.getAndClearContentNode(drawingFrame)[0].innerHTML = imageData;
                return $.when();
            }
            Utils.warn('DrawingFrame.createImageNodeFromData() - unsupported source data for the image object');
            return $.Deferred().reject();
        }

        // updates all formatting for unsupported objects
        function updateUnsupportedFormatting() {
            if (replacementData.length === 0) {
                // insert default replacement nodes, if no replacement data is available
                DrawingFrame.insertReplacementNodes(drawingFrame, drawingAttributes.name, drawingAttributes.description);
            } else if (!drawingFrame.data('has-replacement')) {
                // replacement data MUST NOT change at runtime
                createImageNodeFromData(replacementData);
                drawingFrame.data('has-replacement', true);
            }
            // add class to mark this drawing as a unsupported drawing.
            drawingFrame.addClass(DrawingFrame.UNSUPPORTED_CLASS);
        }

        // updating the size attributes at an image inside a drawing (also supported inside a drawing group)
        function setImageCssSizeAttributes(imageParent, drawingWidth, drawingHeight, drawingAttrs, imageAttrs) {

            var // IE requires to swap cropping settings if image is flipped
                swapHorCrop = _.browser.IE && drawingAttrs.fliph,
                swapVertCrop = _.browser.IE && drawingAttrs.flipv,
                // effective cropping values (checking for 0, because explicit attributes might be used)
                cropLeft = swapHorCrop ? (imageAttrs.cropRight || 0) : (imageAttrs.cropLeft || 0),
                cropRight = swapHorCrop ? (imageAttrs.cropLeft || 0) : (imageAttrs.cropRight || 0),
                cropTop = swapVertCrop ? (imageAttrs.cropBottom || 0) : (imageAttrs.cropTop || 0),
                cropBottom = swapVertCrop ? (imageAttrs.cropTop || 0) : (imageAttrs.cropBottom || 0),
                // horizontal offset/size of cropped bitmaps, as CSS attributes
                horizontalSettings = calculateBitmapSettings(app, drawingWidth, cropLeft, cropRight),
                // vertical offset/size of cropped bitmaps, as CSS attributes
                verticalSettings = calculateBitmapSettings(app, drawingHeight, cropTop, cropBottom);

            imageParent.find('img').css({
                left: horizontalSettings.offset,
                top: verticalSettings.offset,
                width: horizontalSettings.size,
                height: verticalSettings.size
            });
        }

        // updates all formatting for image objects
        function updateImageFormatting() {

            var // special attributes for images
                imageAttributes = mergedAttributes.image;

            // updates attributes of the image node after loading
            // -> but not for grouped drawings. Those drawings are handled by the group container
            function updateImageAttributes(options) {

                var // current width of the drawing frame, in 1/100 mm
                    drawingWidth = Utils.convertLengthToHmm(drawingFrame.width(), 'px'),
                    // current height of the drawing frame, in 1/100 mm
                    drawingHeight = Utils.convertLengthToHmm(drawingFrame.height(), 'px'),
                    // whether the call of this function is asynchronous
                    isAsync = Utils.getBooleanOption(options, 'isAsync', true);

                if ((drawingWidth > 0) && (drawingHeight > 0)) {

                    if (DrawingFrame.isGroupedDrawingFrame(drawingFrame)) {
                        // For grouped drawings, it is useless to set the image attributes directly,
                        // because these attributes are dependent from the group itself.
                        // -> after the image is loaded (deferred!), the image attributes need to be set again
                        // to be correct for the complete group.
                        // The call of resizeDrawingsInGroup() is only required after loading the document for
                        // grouped drawings when loading happens without fast load and without
                        // local storage.
                        if (isAsync) { resizeDrawingsInGroup(DrawingFrame.getGroupNode(drawingFrame)); }
                    } else {
                        setImageCssSizeAttributes(contentNode, drawingWidth, drawingHeight, drawingAttributes, imageAttributes);
                    }
                    // remove class which marks this drawing as unsupported
                    drawingFrame.removeClass(DrawingFrame.UNSUPPORTED_CLASS);
                    // update the border
                    updateLineFormatting();
                }
            }

            // create the image or SVG node (considered to be constant at runtime)
            if (contentNode.find('>.' + CROPPING_CLASS).children().length === 0) {
                // start with replacement graphics (image data will be loaded asynchronously)
                updateUnsupportedFormatting();
                if (imageAttributes.imageData.length > 0) {
                    createImageNodeFromData(imageAttributes.imageData).done(updateImageAttributes);
                } else if (imageAttributes.imageUrl.length > 0) {
                    // convert relative URL to an application-specific absolute URL
                    createImageNodeFromUrl(/:\/\//.test(imageAttributes.imageUrl) ? imageAttributes.imageUrl : app.getServerModuleUrl(IO.FILTER_MODULE_NAME, { action: 'getfile', get_filename: imageAttributes.imageUrl })).done(updateImageAttributes);
                } else {
                    Utils.warn('DrawingFrame.updateFormatting() - missing source data for the image object');
                }
            } else {
                // update attributes of an existing image element
                updateImageAttributes({ isAsync: false });
            }

            // image successfully rendered
            return true;
        }

        // updates chart formatting
        function updateChartFormatting() {

            var // the drawing object model
                model = DrawingFrame.getModel(drawingFrame),
                // the chart rendering engine (lazy initialization, see below)
                renderer = drawingFrame.data('chart-renderer'),
                // unique DOM element identifier for the renderer
                chartId = null,
                drawingNode = null;

            // drawing object model must exist for chart objects
            if (!model) { return false; }

            // lazy creation of the renderer
            if (!renderer) {
                chartId = 'io-ox-documents-chart-' + DrawingFrame.getUid(drawingFrame);
                contentNode.addClass('chartholder');

                // localization for CanvasJS
                if (!CULTURE_INFO) {
                    CULTURE_INFO = {
                        decimalSeparator: app.getDecimalSeparator(),
                        digitGroupSeparator: app.getGroupSeparator(),
                        shortDays: CoreDate.locale.daysShort,
                        days: CoreDate.locale.days,
                        months: CoreDate.locale.months,
                        shortMonths: CoreDate.locale.monthsShort
                    };
                    CanvasJS.addCultureInfo('en', CULTURE_INFO);
                }

                drawingNode = $('<div>');
                drawingNode.addClass('chartnode');
                drawingNode.attr('id', chartId);
                contentNode.append(drawingNode);

                // really important call, because of a bug in canvasJS
                model.resetData();

                renderer = new CanvasJS.Chart(chartId, model.getModelData());
                drawingFrame.data('chart-renderer', renderer);
                model.firstInit();
            }

            // render the chart object
            model.updateRenderInfo();
            try {
                renderer.render();
            } catch (ex) {
                Utils.exception(ex, 'chart rendering error');
            }

            // background color of model data is already prepared for css
            contentNode.css('backgroundColor', model.getModelData().cssBackgroundColor);

            // chart successfully rendered
            return true;
        }

        /**
         * Updates the fill formatting of the drawing frame.
         *
         * @param {jQuery} [node]
         *  An optional jQuery node, to which the css properties are
         *  assigned. If it is not set, the content node inside the
         *  drawing frame is used. Setting this node, is only required
         *  for drawing groups, where all children need to be modified.
         */
        function updateFillFormatting() {
            var // the CSS properties for the drawing frame
                cssProps = {},
                // the node, whose properties will be set
                cssNode = contentNode,
                // the fill attributes
                fillAttrs = mergedAttributes.fill,
                // the style collection of the document
                documentStyles = app.getModel().getDocumentStyles();

            // calculate the CSS fill attributes
            switch (fillAttrs.type) {
            case 'none':
                // clear everything: color and bitmaps
                cssProps.background = '';
                break;
            case 'solid':
                // use 'background' compound attribute to clear bitmaps
                cssProps.background = documentStyles.getCssColor(fillAttrs.color, 'fill');
                break;
            default:
                Utils.warn('DrawingFrame.updateFillFormatting(): unknown fill type "' + fillAttrs.type + '"');
            }

            // apply the fill attributes
            cssNode.css(cssProps);
        }

        /**
         * Updates the border formatting of the drawing frame.
         *
         * @param {jQuery} [node]
         *  An optional jQuery node, to which the css properties are
         *  assigned. If it is not set, the content node inside the
         *  drawing frame is used. Setting this node, is only required
         *  for drawing groups, where all children need to be modified.
         */
        function updateLineFormatting(node) {

            var // gets the correct node to work on (parent() used at grouped drawings)
                currentNode = (node) ? DrawingFrame.getDrawingNode(node): drawingFrame,
                // the line attributes
                lineAttrs = (node) ? AttributeUtils.getExplicitAttributes(currentNode).line : mergedAttributes.line;

            // Explicit attributes are not sufficient, if default line attributes are used (36892)
            if (node && (!lineAttrs || !lineAttrs.type)) { lineAttrs = getAllDrawingAttributes(node, 'line'); }

            if (lineAttrs && lineAttrs.type) {
                // calculate the border properties
                switch (lineAttrs.type) {
                case 'none':
                    // clear everything
                    DrawingFrame.clearBorder(currentNode);
                    break;
                case 'solid':
                    DrawingFrame.drawBorder(app, currentNode, lineAttrs);
                    break;
                default:
                    Utils.warn('DrawingFrame.updateLineFormatting(): unknown line type "' + lineAttrs.type + '"');
                }
            }
        }

        // updates the shape attributes (distance between border and text in text frames)
        function updateShapeFormatting() {

            var // the CSS properties for the drawing frame
                cssProps = {},
                // the fill attributes
                shapeAttrs = mergedAttributes.shape,
                // the half border width in hmm (same value for all borders)
                borderWidth = 0.5 * mergedAttributes.line.width;

            cssProps.paddingLeft = Utils.convertHmmToLength(shapeAttrs.paddingLeft + borderWidth, 'px', 1);
            cssProps.paddingRight = Utils.convertHmmToLength(shapeAttrs.paddingRight + borderWidth, 'px', 1);
            cssProps.paddingTop = Utils.convertHmmToLength(shapeAttrs.paddingTop + borderWidth, 'px', 1);
            cssProps.paddingBottom = Utils.convertHmmToLength(shapeAttrs.paddingBottom + borderWidth, 'px', 1);

            // apply the fill attributes
            contentNode.css(cssProps);

            // setting the value for automatic vertical resizing of the shape.
            if (shapeAttrs.autoResizeHeight) {
                contentNode.addClass(DrawingFrame.AUTORESIZEHEIGHT_CLASS);
            } else {
                contentNode.removeClass(DrawingFrame.AUTORESIZEHEIGHT_CLASS);
            }

        }

        // updates the class at the drawing, that is used to find standard odt text frames (and not shapes with text)
        function updateTextFrameState() {

            var // the explicit drawing attributes (not using merge attributes, because they always contain a style id)
                attrs = AttributeUtils.getExplicitAttributes(drawingFrame);

            if (attrs && attrs.styleId && _.isString(attrs.styleId)) {
                drawingFrame.addClass(DrawingFrame.ODFTEXTFRAME_CLASS);
            } else {
                drawingFrame.removeClass(DrawingFrame.ODFTEXTFRAME_CLASS);
            }

        }

        // updates the size and position of all children of a drawing group
        function resizeDrawingsInGroup(drawing) {

            var // the direct drawing children of the drawing group
                allChildren = DrawingFrame.getAllGroupDrawingChildren(drawing, { onlyDirectChildren: true }),
                // the width of the drawing group element in 1/100 mm
                groupWidth = Utils.convertLengthToHmm(drawing.children().first().width(), 'px'),
                // the height of the drawing group element in  1/100 mm
                groupHeight = Utils.convertLengthToHmm(drawing.children().first().height(), 'px'),
                // the maximum width of the children
                maxWidth = 0,
                // the maximum height of the children
                maxHeight = 0,
                // the horizontal stretch factor for the children
                horzFactor = 1,
                // the vertical stretch factor for the children
                vertFactor = 1;

            // updates the size and position of one drawing child inside a drawing group
            function handleDrawingInsideDrawingGroup(drawing, attrs, horzFactor, vertFactor) {

                var // the attributes at the drawing property
                    drawingAttrs = attrs.drawing,
                    // the required css attributes
                    cssAttrs = {},
                    // an optional horizontal resize factor
                    horzResizeFactor = horzFactor || 1,
                    // an optional vertical resize factor
                    vertResizeFactor = vertFactor || 1,
                    // type of the drawing object inside the group: 'image', ...
                    type = DrawingFrame.getDrawingType(drawing);

                // updates attributes of the image node of a drawing frame inside a drawing group container
                function updateImageAttributesInGroup() {

                    var // current width of the image, in 1/100 mm
                        drawingWidth = Utils.roundDown(drawingAttrs.width * horzResizeFactor, 1),
                        // current height of the image, in 1/100 mm
                        drawingHeight = Utils.roundDown(drawingAttrs.height * vertResizeFactor, 1);

                    setImageCssSizeAttributes(drawing.children().first(), drawingWidth, drawingHeight, drawingAttrs, attrs.image);
                }

                // set the CSS attributes for the drawing
                if (_.isNumber(drawingAttrs.left)) { cssAttrs.left = Utils.convertHmmToCssLength(drawingAttrs.left * horzResizeFactor, 'px', 1); }
                if (_.isNumber(drawingAttrs.top)) { cssAttrs.top = Utils.convertHmmToCssLength(drawingAttrs.top * vertResizeFactor, 'px', 1); }
                if (_.isNumber(drawingAttrs.width)) { cssAttrs.width = Utils.convertHmmToCssLength(drawingAttrs.width * horzResizeFactor, 'px', 1); }
                if (_.isNumber(drawingAttrs.height)) { cssAttrs.height = Utils.convertHmmToCssLength(drawingAttrs.height * vertResizeFactor, 'px', 1); }

                drawing.css(cssAttrs);

                // updating the size of the images inside the drawing
                if (type === 'image') { updateImageAttributesInGroup(drawing); }

                // update border
                if (type !== 'group') { updateLineFormatting(drawing); }
            }

            // iterating over all drawing children of the group
            if (allChildren.length > 0) {

                // comparing width and height of the group with all values of the children
                // -> the aim is, that all children fit optimal into the group
                // -> this needs to be set after loading and after resizing

                // getting the maximum value of the children for 'left + width' and 'top + height')
                _.each(allChildren, function (drawing) {

                    var // the explicit drawing attributes
                        drawingAttrs = AttributeUtils.getExplicitAttributes(drawing),
                        // the horizontal expansion of one drawing
                        currentHorz = 0,
                        // the vertical expansion of one drawing
                        currentVert = 0;

                    if (drawingAttrs && drawingAttrs.drawing) {
                        if (_.isNumber(drawingAttrs.drawing.left)) { currentHorz = drawingAttrs.drawing.left; }
                        if (_.isNumber(drawingAttrs.drawing.width)) { currentHorz += drawingAttrs.drawing.width; }
                        if (_.isNumber(drawingAttrs.drawing.top)) { currentVert = drawingAttrs.drawing.top; }
                        if (_.isNumber(drawingAttrs.drawing.height)) { currentVert += drawingAttrs.drawing.height; }

                        if (currentHorz > maxWidth) { maxWidth = currentHorz; }
                        if (currentVert > maxHeight) { maxHeight = currentVert; }
                    }
                });

                if (maxWidth > 0 || maxHeight > 0) {

                    // now checking values for maxWidth and maxHeight
                    horzFactor = Utils.roundDown(groupWidth / maxWidth, 0.01);
                    vertFactor = Utils.roundDown(groupHeight / maxHeight, 0.01);

                    // getting the maximum value of the children for 'left + width' and 'top + height')
                    _.each(allChildren, function (drawing) {
                        handleDrawingInsideDrawingGroup($(drawing), AttributeUtils.getExplicitAttributes(drawing), horzFactor, vertFactor);
                    });
                }
            }
        }

        // convert to single jQuery object
        drawingFrame = $(drawingFrame).first();

        // set text attribute to support tooltip for the drawing
        Forms.setToolTip(drawingFrame, drawingAttributes.name);

        // vertically or horizontally flipped drawings
        if (drawingAttributes.flipH && drawingAttributes.flipV) {
            transformations = 'scale(-1)';
        } else if (drawingAttributes.flipH) {
            transformations = 'scaleX(-1)';
        } else if (drawingAttributes.flipV) {
            transformations = 'scaleY(-1)';
        }

        // always set the CSS attribute, to be able to remove existing transformations
        Utils.setCssAttributeWithPrefixes(contentNode, 'transform', transformations);

        // update drawing frame specific to the drawing type
        switch (type) {
        case 'image':
            updateLineFormatting();
            rendered = updateImageFormatting();
            break;
        case 'ole':
            rendered = updateImageFormatting();
            break;
        case 'chart':
            rendered = updateChartFormatting();
            break;
        case 'shape':
            // updateShapeFormatting must be invoked first, so that the correct
            // dimensions can be set before border and/or background will be painted
            updateShapeFormatting();
            updateFillFormatting();
            updateLineFormatting();
            if (app.isODF()) { updateTextFrameState(); }
            rendered = DrawingFrame.isTextFrameShapeDrawingFrame(drawingFrame);
            if (!_.browser.IE && rendered) { $(drawingFrame).attr('contenteditable', true); }  // TODO: Not good here, but required for fast load (36246)
            break;
        case 'group':
            // the group needs to take care of the size of all children
            resizeDrawingsInGroup(drawingFrame);
            if (!_.browser.IE) { $(drawingFrame).attr('contenteditable', true); }  // TODO: Not good here, but required for fast load
            rendered = true;
            break;
        }

        // during loading of document, an additional resize of drawings in groups might be required
        // -> during working with document, operations are only generated for the group itself.
        if (DrawingFrame.isGroupedDrawingFrame(drawingFrame)) {
            resizeDrawingsInGroup(DrawingFrame.getGroupNode(drawingFrame));
        }

        // add replacement in case the drawing type is not supported, or if rendering was not successful
        if (!rendered) {
            updateUnsupportedFormatting();
        }

    };

    // draws the border to the canvas
    DrawingFrame.drawBorder = function (app, currentNode, lineAttrs) {
        var // the border node inside the drawing frame
            borderNode = DrawingFrame.getBorderNode(currentNode),
            // the canvas wrapper of the border node
            borderCanvas = null,
            // the style collection of the document
            documentStyles = app.getModel().getDocumentStyles(),
            // border width
            lineWidth = Utils.convertHmmToLength(lineAttrs.width, 'px', 1),
            // line width to calculate the border correct
            calcLineWidth = 0,
            // dimensions of the canvas element
            cWidth = 0,
            cHeight = 0,
            // position of the canvas element
            cTop = 0,
            cLeft = 0;

        if (borderNode.length === 0) {
            borderCanvas = new CanvasWrapper({classes: DrawingFrame.BORDER_CLASS});
            borderNode = borderCanvas.getNode();
            DrawingFrame.addBorderNode(currentNode, borderNode);
        } else {
            borderCanvas = new CanvasWrapper({classes: DrawingFrame.BORDER_CLASS, canvasElement: borderNode});
        }

        // lineWidth cannot be "0" (min. "1")
        lineWidth = (lineWidth === 0) ? 1 : lineWidth;

        // calculate the dimension of the canvas
        calcLineWidth = (lineWidth % 2 === 0) ? lineWidth : (lineWidth+1);

        // calculate the position of the line in the canvas
        // special behavior for text frames. Their border covers their boundary
        cWidth  = (DrawingFrame.isTextFrameShapeDrawingFrame(currentNode)) ? (currentNode.first().width() + calcLineWidth) : (currentNode.first().width() + (lineWidth*2));
        cHeight = (DrawingFrame.isTextFrameShapeDrawingFrame(currentNode)) ? (currentNode.first().height() + calcLineWidth) : (currentNode.first().height() + (lineWidth*2));
        cTop    = (DrawingFrame.isTextFrameShapeDrawingFrame(currentNode)) ? '-'+Math.ceil(lineWidth / 2)+'px' : '-'+lineWidth+'px';
        cLeft   = (DrawingFrame.isTextFrameShapeDrawingFrame(currentNode)) ? '-'+Math.ceil(lineWidth / 2)+'px' : '-'+lineWidth+'px';

        // set dimension of the canvas element
        borderCanvas.initialize({
            top: 0,
            left: 0,
            width:  cWidth,
            height: cHeight
        });
        // set position of the canvas element
        borderCanvas.getNode().css({
            top: cTop,
            left: cLeft
        });

        borderCanvas.render(function (context, width, height) {
            var color = documentStyles.getCssColor(lineAttrs.color, 'line'),
                pattern = Border.getBorderPattern(lineAttrs.style, lineWidth);

            context.setLineStyle({ style: color, width: lineWidth, pattern: pattern });
            context.drawRect( (lineWidth / 2), (lineWidth / 2), (width - lineWidth), (height - lineWidth));
        });
    };

    // clears the canvas
    DrawingFrame.clearBorder = function (currentNode) {
        var // the border node inside the drawing frame
            borderNode = DrawingFrame.getBorderNode(currentNode),
            // the canvas wrapper of the border node
            borderCanvas = null;

        if (borderNode.length > 0) {
            borderCanvas = new CanvasWrapper({classes: DrawingFrame.BORDER_CLASS, canvasElement: borderNode});
        }

        if (!_.isNull(borderCanvas)) {
            borderCanvas.initialize({
                top: 0,
                left: 0,
                width:  borderNode.width(),
                height: borderNode.height()
            });
            borderCanvas.clear();
        }
    };

    // selection --------------------------------------------------------------

    /**
     * Returns whether the specified drawing frame is currently selected.
     *
     * @param {HTMLElement|jQuery} drawingFrame
     *  The root node of the drawing frame. If this object is a jQuery
     *  collection, uses the first DOM node it contains.
     *
     * @returns {Boolean}
     *  Whether the specified drawing frame is selected.
     */
    DrawingFrame.isSelected = function (drawingFrame) {
        return Forms.isSelectedNode(drawingFrame);
    };

    /**
     * Creates or updates additional nodes displayed while a drawing frame is
     * selected.
     *
     * @param {HTMLElement|jQuery} drawingFrame
     *  The root node of the drawing frame. If this object is a jQuery
     *  collection, uses the first DOM node it contains.
     *
     * @param {Object} [options]
     *  A map with options to control the appearance of the selection. The
     *  following options are supported:
     *  @param {Boolean} [options.movable=false]
     *      If set to true, mouse pointer will be changed to a movable pointer
     *      while the mouse hovers the drawing.
     *  @param {Boolean} [options.resizable=false]
     *      If set to true, mouse pointer will be changed to an appropriate
     *      resize pointer while the mouse hovers a resize handle.
     *
     * @returns {jQuery}
     *  The selection root node contained in the drawing frame, as jQuery
     *  object.
     */
    DrawingFrame.drawSelection = function (drawingFrame, options) {

        var // HTML mark-up for the selection
            markup = null;

        // create new selection nodes if missing
        if (!DrawingFrame.isSelected(drawingFrame)) {

            // create a new selection node
            markup = '<div class="' + SELECTION_CLASS + '">';

            // create the border nodes
            markup += '<div class="borders">';
            _.each('tblr', function (pos) {
                markup += '<div data-pos="' + pos + '"></div>';
            });
            markup += '</div>';

            // create resizer handles
            markup += '<div class="resizers">';
            _.each(['t', 'b', 'l', 'r', 'tl', 'tr', 'bl', 'br'], function (pos) {
                markup += '<div data-pos="' + pos + '"></div>';
            });
            markup += '</div>';

            // create the tracker node
            markup += '<div class="tracker"></div>';

            // close the selection node
            markup += '</div>';

            $(drawingFrame).first().append(markup);
        }

        // update state classes at the drawing frame
        $(drawingFrame).first()
            .addClass(Forms.SELECTED_CLASS)
            .toggleClass('movable', Utils.getBooleanOption(options, 'movable', false))
            .toggleClass('resizable', Utils.getBooleanOption(options, 'resizable', false));

        return getSelectionNode(drawingFrame);
    };

    /**
     * Removes the selection node from the specified drawing frame.
     *
     * @param {HTMLElement|jQuery} drawingFrame
     *  The root node of the drawing frame. If this object is a jQuery
     *  collection, uses the first DOM node it contains.
     *
     * @param {Object} [options]
     *  A map with options to control the appearance of the selection. The
     *  following options are supported:
     *  @param {Boolean} [options.movable=false]
     *      If set to true, mouse pointer will be changed to a movable pointer
     *      while the mouse hovers the drawing.
     *  @param {Boolean} [options.resizable=false]
     *      If set to true, mouse pointer will be changed to an appropriate
     *      resize pointer while the mouse hovers a resize handle.
     *
     * @returns {jQuery}
     *  The selection root node contained in the drawing frame, as jQuery
     *  object.
     */
    DrawingFrame.clearSelection = function (drawingFrame) {
        $(drawingFrame).first().removeClass(Forms.SELECTED_CLASS + ' movable resizable');
        getSelectionNode(drawingFrame).remove();
    };

    /**
     * Sets or resets the CSS class representing the active tracking mode for
     * the specified drawing frames.
     *
     * @param {HTMLElement|jQuery} drawingFrames
     *  The root node of a single drawing frame, or multiple drawing frames in
     *  a jQuery collection. This method changes the tracking mode of all
     *  drawing frames at once.
     *
     * @param {Boolean} active
     *  Whether tracking mode is active.
     */
    DrawingFrame.toggleTracking = function (drawingFrames, active) {
        $(drawingFrames).find('>.' + SELECTION_CLASS).toggleClass(TRACKING_CLASS, active === true);
    };

    /**
     * Returns the tracking direction stored in a resizer handle node of a
     * drawing frame selection box.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {String|Null}
     *  If the passed node is a resizer handle of a drawing frame selection
     *  box, returns its direction identifier (one of 'l', 'r', 't', 'b', 'tl',
     *  'tr', 'bl', or 'br'). Otherwise, returns null.
     */
    DrawingFrame.getResizerHandleType = function (node) {
        return $(node).is(DrawingFrame.NODE_SELECTOR + '>.' + SELECTION_CLASS + '>.resizers>[data-pos]') ? $(node).attr('data-pos') : null;
    };

    // exports ================================================================

    return DrawingFrame;

});
