/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/drawinglayer/model/drawingcollection', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/object/triggerobject',
    'io.ox/office/tk/object/timermixin',
    'io.ox/office/drawinglayer/model/drawingmodel'
], function (Utils, TriggerObject, TimerMixin, DrawingModel) {

    'use strict';

    // class DrawingCollection ================================================

    /**
     * A collection containing models of drawing objects.
     *
     * Triggers the following events:
     * - 'insert:drawing'
     *      After a new drawing model has been inserted into this collection.
     *      Event handlers receive the new drawing model, and the position of
     *      the drawing object.
     * - 'delete:drawing'
     *      After a drawing model has been removed from this collection. Event
     *      handlers receive the removed drawing model, and the last position
     *      of the drawing object.
     * - 'change:drawing'
     *      After some contents of a drawing model in this collection have been
     *      changed (dependent on the type of the drawing object), including
     *      the formatting attributes of the drawing object. Event handlers
     *      receive the following parameters:
     *      (1) {jQuery.Event} event
     *          The JQuery event object.
     *      (2) {DrawingModel} drawingModel
     *          The drawing model that caused the change event.
     *      (3) {Number[]} position
     *          The current position of the drawing object.
     *      (4) {String} type
     *          The exact type of the change event. The possible values are
     *          dependent on the type of the drawing object. Will be set to
     *          'attributes', if the formatting attributes of the drawing
     *          objects have been changed.
     *      (5) {Any} [...]
     *          Additional data dependent on the type of the change event. if
     *          the change event type is 'attributes', the fifth parameter
     *          contains the new merged attribute set, and the sixth parameter
     *          contains the old merged attribute set.
     *
     * @constructor
     *
     * @extends TriggerObject
     * @extends TimerMixin
     *
     * @param {EditApplication} app
     *  The application instance containing this drawing collection.
     *
     * @param {Object} initOptions
     *  Optional parameters:
     *  @param {Function} initOptions.clientCreateModelHandler
     *      A factory function that constructs drawing model instances for the
     *      specified drawing type. Receives the type of the drawing model to
     *      be created, as string; and optionally the initial formatting
     *      attributes for the drawing model. Must return a new instance of a
     *      drawing model according to the passed type. May return null for
     *      unsupported drawing types.
     *  @param {Function} initOptions.clientResolveModelHandler
     *      A function that receives an arbitrary document position, and must
     *      return the model of a drawing object at that position, if
     *      available; otherwise null.
     *  @param {Function} initOptions.clientResolvePositionHandler
     *      A function that receives an arbitrary drawing model, and must
     *      return its document position, if available; otherwise null.
     *  @param {Function} initOptions.clientRegisterModelHandler
     *      A function that receives a new drawing model that has just been
     *      inserted into this collection, and the document position of the new
     *      drawing model. The function has to perform client-side registration
     *      of the drawing model according to the document position, and
     *      returns whether registration of the drawing model at the passed
     *      position was successful.
     *  @param {Function} [initOptions.clientSetAttributesHandler]
     *      A function that will be called every time the formatting attributes
     *      of a drawing model contained in this collection will be changed by
     *      using its DrawingModel.setAttributes() method. Receives the drawing
     *      model instance.
     */
    function DrawingCollection(app, initOptions) {

        var // self reference
            self = this,

            // the client factory function for drawing model instances
            clientCreateModelHandler = Utils.getFunctionOption(initOptions, 'clientCreateModelHandler', $.noop),

            // the client resolves document positions to models
            clientResolveModelHandler = Utils.getFunctionOption(initOptions, 'clientResolveModelHandler', $.noop),

            // the client resolves models to document positions
            clientResolvePositionHandler = Utils.getFunctionOption(initOptions, 'clientResolvePositionHandler', $.noop),

            // the client has to register new drawing models
            clientRegisterModelHandler = Utils.getFunctionOption(initOptions, 'clientRegisterModelHandler', $.noop),

            // the client can hook into the setAttributes operation and can update other attributes
            clientSetAttributesHandler = Utils.getFunctionOption(initOptions, 'clientSetAttributesHandler'),

            // all drawing models, mapped by internal unique identifier
            drawingModels = {},

            // cached number of drawing models in this collection (faster than counting map properties)
            modelCount = 0;

        // base constructors --------------------------------------------------

        TriggerObject.call(this);
        TimerMixin.call(this);

        // private methods ----------------------------------------------------

        /**
         * Returns the drawing model at the specified document position
         * contained in this drawing collection.
         *
         * @param {Number[]} position
         *  The document position of the drawing model.
         */
        function getModel(position) {

            var // let client resolve the drawing model
                model = clientResolveModelHandler.call(self, position);

            // check that the model is part of this collection
            if (model && !(model.getUid() in drawingModels)) {
                Utils.warn('DrawingCollection.getModel() - unknown drawing model');
                return null;
            }

            return model;
        }

        /**
         * Tries to find the drawing collection embedded in a drawing object
         * inside this collection.
         *
         * @param {Number[]} position
         *  The document position of the drawing model.
         *
         * @returns {Object|Null}
         *  A result object containing the following properties:
         *  - {DrawingCollection} collection
         *      The drawing collection of a drawing object inside this
         *      collection.
         *  - {Number[]} position
         *      The remaining document position relative to the drawing
         *      collection.
         * If no drawing object with an own embedded collection has been found,
         * the value null will be returned.
         */
        function findEmbeddedCollection(position) {

            var // an existing drawing model
                model = null,
                // the collection embedded in the drawing model
                collection = null;

            for (var length = 1; length < position.length; length += 1) {
                if ((model = getModel(position.slice(0, length)))) {
                    collection = model.getCollection();
                    return collection ? { collection: collection, position: position.slice(length) } : null;
                }
            }

            return null;
        }

        // methods ------------------------------------------------------------

        /**
         * Returns the number of drawings inserted directly into this drawing
         * collection (excluding the drawing objects embedded in other drawing
         * objects).
         *
         * @returns {Number}
         *  The number of drawing objects contained in this collection.
         */
        this.getModelCount = function () {
            return modelCount;
        };

        /**
         * Tries to find the drawing model at the passed document position.
         *
         * @param {Number[]} position
         *  The document position of the drawing model. May specify a position
         *  inside another drawing object.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.deep=false]
         *      If set to true, searches for drawing objects embedded in other
         *      drawing objects.
         *  @param {String} [options.type]
         *      If specified, the drawing model must be of the specified type.
         *      If a drawing model with another type exists at the specified
         *      document position, this method returns null.
         *
         * @returns {DrawingModel|Null}
         *  The drawing model at the specified document position; or null, if
         *  no drawing model has been found.
         */
        this.findModel = function (position, options) {

            var // whether to search deeply in embedded collections
                deep = Utils.getBooleanOption(options, 'deep', false),
                // whether to restrict to a specific type
                type = Utils.getStringOption(options, 'type'),
                // embedded collection addressed by the passed position
                embedInfo = null,
                // the drawing model
                model = null;

            // existing drawing object with embedded collection: search drawing model there
            if (deep && (embedInfo = findEmbeddedCollection(position))) {
                return embedInfo.collection.findModel(embedInfo.position, options);
            }

            // find the drawing model at the specified position directly
            model = getModel(position);

            // check the type of the drawing model if specified
            return (model && (!_.isString(type) || (type === model.getType()))) ? model : null;
        };

        /**
         * Returns the document position of the specified drawing model.
         *
         * @param {DrawingModel} model
         *  The drawing model whose position will be obtained.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.deep=false]
         *      If set to true, also supports drawing objects embedded in other
         *      drawing objects, and resolves the complete position of the
         *      embedded drawing object. Otherwise, the passed drawing object
         *      must be contained directly in this collection.
         *
         * @returns {Number[]|Null}
         *  The position of the drawing object, if it is contained in this
         *  collection; otherwise null.
         */
        this.getModelPosition = function (model/*, options*/) {

            var // whether to support embedded drawing models
                /*deep = Utils.getBooleanOption(options, 'deep', false),*/
                // position of the drawing model
                position = null;

            // TODO: support for deep option
            position = clientResolvePositionHandler.call(self, model);
            return position;
        };

        /**
         * Invokes the passed iterator function for all drawing models in this
         * collection, in no specific order.
         *
         * @param {Function} iterator
         *  The iterator function called for all drawing models. Receives the
         *  current drawing model as first parameter. If the iterator returns
         *  the Utils.BREAK object, the iteration process will be stopped
         *  immediately.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Object} [options.context]
         *      If specified, the iterator will be called with this context
         *      (the symbol 'this' will be bound to the context inside the
         *      iterator function).
         *
         * @returns {Utils.BREAK|Undefined}
         *  A reference to the Utils.BREAK object, if the iterator has returned
         *  Utils.BREAK to stop the iteration process, otherwise undefined.
         */
        this.iterateModels = function (iterator, options) {

            var // the calling context for the iterator function
                context = Utils.getOption(options, 'context');

            return _.any(drawingModels, function (model) {
                return iterator.call(context, model) === Utils.BREAK;
            }) ? Utils.BREAK : undefined;
        };

        /**
         * Stores the passed model of a new drawing object into this drawing
         * collection.
         *
         * @param {Number[]} position
         *  The document position of the new drawing model. May specify a
         *  position inside another drawing object (if the array without the
         *  last element points to an existing drawing object). If that drawing
         *  model supports embedding drawing objects, the new drawing model
         *  will be inserted into the drawing collection of that drawing object
         *  instead into this collection.
         *
         * @param {DrawingModel} model
         *  The model instance of the new drawing object. This collection (or
         *  the drawing collection of an embedded drawing object) takes
         *  ownership of this model!
         *
         * @returns {Boolean}
         *  Whether the new drawing model has been inserted successfully.
         */
        this.insertModelInstance = function (position, model) {

            var // embedded collection addressed by the passed position
                embedInfo = findEmbeddedCollection(position);

            // existing drawing object with embedded collection found: create new drawing model there
            if (embedInfo) {
                return embedInfo.collection.insertModelInstance(embedInfo.position, model);
            }

            // store the new drawing model
            drawingModels[model.getUid()] = model;
            modelCount += 1;

            // register the new drawing model at the client application
            if (!clientRegisterModelHandler.call(this, model, position)) { return false; }

            // notify all listeners
            this.trigger('insert:drawing', model, position);

            // listen to attribute changes of the drawing model
            model.on('change:attributes', function (event, newAttributes, oldAttributes) {
                // call the client hook to preprocess the formatting attributes
                if (_.isFunction(clientSetAttributesHandler)) {
                    clientSetAttributesHandler.call(this, model);
                    newAttributes = model.getMergedAttributes();
                }
                self.trigger('change:drawing', model, self.getModelPosition(model), 'attributes', newAttributes, oldAttributes);
            });

            // forward generic change events from the drawing model
            model.on('change:drawing', function () {
                self.trigger('change:drawing', model, self.getModelPosition(model), _.toArray(arguments).slice(1));
            });

            return true;
        };

        /**
         * Creates and stores the model of a new drawing object based on the
         * passed settings.
         *
         * @param {Number[]} position
         *  The document position of the new drawing model. May specify a
         *  position inside another drawing object (if the array without the
         *  last element points to an existing drawing object). If that drawing
         *  model supports embedding drawing objects, the new drawing model
         *  will be inserted into the drawing collection of that drawing object
         *  instead into this collection.
         *
         * @param {String} type
         *  The type of the drawing object to be created.
         *
         * @param {Object} [attributes]
         *  An attribute set with initial formatting attributes for the drawing
         *  object.
         *
         * @returns {Boolean}
         *  Whether the new drawing model has been created successfully.
         */
        this.insertModel = function (position, type, attributes) {

            var // embedded collection addressed by the passed position
                embedInfo = findEmbeddedCollection(position),
                // the drawing model constructed by the factory function
                model = null;

            // existing drawing object with embedded collection found: create new drawing model there
            if (embedInfo) {
                return embedInfo.collection.insertModel(embedInfo.position, type, attributes);
            }

            // create and store the new drawing model
            model = clientCreateModelHandler.call(this, type, attributes);
            return (model instanceof DrawingModel) && this.insertModelInstance(position, model);
        };

        /**
         * Removes the model of a drawing object located at the passed document
         * position.
         *
         * @param {Number[]} position
         *  The document position of the drawing model. May specify a position
         *  inside another drawing object (if the array without the last
         *  element points to an existing drawing object).
         *
         * @returns {Boolean}
         *  Whether the drawing model has been removed successfully.
         */
        this.deleteModel = function (position) {

            var // embedded collection addressed by the passed position
                embedInfo = findEmbeddedCollection(position),
                // the drawing model
                model = null;

            // existing drawing object with embedded collection found: create new drawing model there
            if (embedInfo) {
                return embedInfo.collection.deleteModel(embedInfo.position);
            }

            // find the drawing model at the specified position
            if (!(model = this.findModel(position))) { return false; }

            // notify all listeners
            this.trigger('delete:drawing', model, position);

            // remove and destroy the drawing model
            delete drawingModels[model.getUid()];
            model.destroy();
            modelCount -= 1;

            return true;
        };

        // initialization -----------------------------------------------------

        // destroy all class members on destruction
        this.registerDestructor(function () {
            _.invoke(drawingModels, 'destroy');
            drawingModels = null;
        });

    } // class DrawingCollection

    // exports ================================================================

    // derive this class from class TriggerObject
    return TriggerObject.extend({ constructor: DrawingCollection });

});
