/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/baseframework/model/modelobject', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/object/triggerobject'
], function (Utils, TriggerObject) {

    'use strict';

    // class ModelObject ======================================================

    /**
     * An abstract model object that allows to trigger change events to event
     * listeners. As long as the application still imports the document, this
     * instance will not trigger any events (performance optimization).
     *
     * @constructor
     *
     * @extends TriggerObject
     *
     * @param {EditApplication} app
     *  The application instance containing this attributed model object.
     *
     * @param {Object} [initOptions]
     *  Optional parameters:
     *  @param {String} [initOptions.trigger='runtime']
     *      The behavior of this instance when the trigger() method has been
     *      called to trigger a new event. The following modes are supported:
     *      - 'runtime' (default):
     *          Events are triggered during runtime of the application, but not
     *          when importing the document (performance optimization).
     *      - 'always':
     *          Events are always triggered, also during document import.
     *      - 'never':
     *          Events will never be triggered (the method trigger() of this
     *          instance is empty). Nevertheless, this instance provides the
     *          full event API (the on(), one(), off(), and trigger() methods).
     *          Useful if a few sub classes in a class hierarchy chose to be
     *          permanently silent although the classes derive indirectly from
     *          this class.
     */
    function ModelObject(app, initOptions) {

        var // the trigger mode
            triggerMode = Utils.getStringOption(initOptions, 'trigger', 'runtime');

        // base constructor ---------------------------------------------------

        TriggerObject.call(this);

        // initialization -----------------------------------------------------

        // set dummy event methods for silent objects
        if (triggerMode === 'never') {
            this.on = this.one = this.off = this.trigger = Utils.NOOP;
        }

        // disable triggering until import is finished
        else if ((triggerMode !== 'always') && !app.isImportFinished()) {

            // overwrite the trigger() method with a method that triggers
            // events only if the document has been imported already
            this.trigger = (function (trigger) {
                this.listenTo(app.getImportPromise(), 'always', _.bind(function () {
                    this.trigger = trigger; // back to original method
                }, this));
                return Utils.NOOP;
            }.call(this, this.trigger));
        }

    } // class ModelObject

    // exports ================================================================

    // derive this class from class TriggerObject
    return TriggerObject.extend({ constructor: ModelObject });

});
