/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/formula/interpreter', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/object/baseobject',
    'io.ox/office/baseframework/model/modelobject',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/model/formula/tokenutils',
    'io.ox/office/spreadsheet/model/formula/complex',
    'io.ox/office/spreadsheet/model/formula/matrix',
    'io.ox/office/spreadsheet/model/formula/reference',
    'io.ox/office/spreadsheet/model/formula/operators'
], function (Utils, BaseObject, ModelObject, SheetUtils, TokenUtils, Complex, Matrix, Reference, Operators) {

    'use strict';

    var // shortcut for the map of error code literals
        ErrorCodes = SheetUtils.ErrorCodes,

        // maximum time available for evaluating a single formula, in milliseconds
        MAX_EVAL_TIME = 1000;

    // private global functions ===============================================

    /**
     * Throws a fatal exception, and prints the passed error message to the
     * browser console.
     */
    function fatalError(msg) {
        Utils.error(msg);
        throw 'fatal';
    }

    /**
     * Throws the passed value, if it is an error code literal. Otherwise,
     * the passed value will be returned.
     *
     * @param {Number|String|Boolean|ErrorCode|Null} value
     *  Any literal value used in formulas.
     *
     * @returns {Number|String|Boolean|Null}
     *  The passed value, if it is not an error code.
     *
     * @throws {ErrorCode}
     *  The passed value, if it is an error code.
     */
    function checkValueForErrorCode(value) {
        if (SheetUtils.isErrorCode(value)) { throw value; }
        return value;
    }

    /**
     * Throws the first error code literal found in the passed constant matrix.
     * If no error code has been found, the passed matrix will be returned.
     *
     * @param {Matrix} matrix
     *  An matrix literal containing any values used in formulas.
     *
     * @returns {Matrix}
     *  The passed matrix, if it does not contain an error code.
     *
     * @throws {ErrorCode}
     *  The first error code contained in the passed matrix.
     */
    function checkMatrixForErrorCode(matrix) {
        matrix.iterateElements(checkValueForErrorCode);
        return matrix;
    }

    /**
     * Checks that all elements in the passed matrix literal pass the specified
     * truth test. If any matrix element fails, the #VALUE! error code will be
     * thrown. If any matrix element is an error code literal, it will be
     * thrown instead. If all matrix elements are valid, the passed matrix will
     * be returned.
     *
     * @param {Matrix} matrix
     *  A matrix literal containing any values used in formulas.
     *
     * @param {Function} callback
     *  The predicate callback function invoked for every matrix element.
     *
     * @returns {Matrix}
     *  The passed matrix, if all its elements are valid.
     *
     * @throws {ErrorCode}
     *  The first error code contained in the passed matrix, or the #VALUE!
     *  error code, if a matrix element does not pass the truth test.
     */
    function checkMatrixForDataType(matrix, callback) {
        matrix.iterateElements(function (elem) {
            checkValueForErrorCode(elem);
            if (!callback(elem)) { throw ErrorCodes.VALUE; }
        });
        return matrix;
    }

    /**
     * Returns whether the passed value has the specified base type.
     */
    function isValueOfType(value, type) {
        switch (type) {
        case 'val':
            return _.isNumber(value) || _.isString(value) || _.isBoolean(value) || (value instanceof Date) || (value instanceof Complex) || SheetUtils.isErrorCode(value);
        case 'mat':
            return value instanceof Matrix;
        case 'ref':
            return value instanceof Reference;
        }
        fatalError('isValueOfType(): unknown type identifier: "' + type + '"');
    }

    /**
     * Returns whether the passed matrix size is inside the limits supported by
     * the formula interpreter.
     */
    function isValidMatrixSize(rows, cols) {
        return (0 < rows) && (rows <= TokenUtils.MAX_MATRIX_ROW_COUNT) && (0 < cols) && (cols <= TokenUtils.MAX_MATRIX_COL_COUNT);
    }

    // class FormulaContext ===================================================

    /**
     * An instance of this class serves as calling context object for the
     * implementations of operators and functions, so that the symbol 'this'
     * inside these implementations provides various useful helper methods.
     *
     * @constructor
     *
     * @param {SpreadsheetApplication} app
     *  The spreadsheet application instance.
     *
     * @param {Tokenizer} tokenizer
     *  The tokenizer instance needed for specific conversion tasks.
     *
     * @param {Number} refSheet
     *  The index of the sheet the interpreted formula is related to. Used to
     *  resolve reference tokens without sheet reference.
     *
     * @param {Number[]} refAddress
     *  The address of the reference cell in the passed sheet the formula
     *  is related to. Used to resolve relative references in defined names
     *  used by the interpreted formula.
     */
    function FormulaContext(app, tokenizer, refSheet, refAddress) {

        var // self reference
            self = this,

            // the document model
            docModel = app.getModel(),

            // the number formatter
            numberFormatter = docModel.getNumberFormatter(),

            // maximum length of a string, maximum valid string index (one-based)
            MAX_STRING_LEN = app.isODF() ? 65535 : 32767;

        // private methods ----------------------------------------------------

        /**
         * Visits all single values contained in the passed operand, matrix, or
         * reference (invokes the iterator with the result of the value
         * converter).
         */
        function iterateOperandValues(operand, converter, iterator, options) {

            var // whether to visit the empty function parameter
                emptyParam = Utils.getBooleanOption(options, 'emptyParam', false),
                // whether to visit empty cells in references
                emptyCell = Utils.getBooleanOption(options, 'emptyCell', false),
                // whether to pass error codes to the iterator
                acceptErrors = Utils.getBooleanOption(options, 'acceptErrors', false),
                // whether to visit cells in references in order
                ordered = Utils.getBooleanOption(options, 'ordered', false),
                // performance: do not visit more than a specific number of cells in cell references
                count = 0;

            // invokes the iterator for a single value
            function invokeIterator(value, col, row) {
                // throw error codes, unless error codes will be accepted
                if (!acceptErrors) { checkValueForErrorCode(value); }
                // convert the passed value (may throw an error code)
                value = converter.call(self, value);
                // check maximum number of visited values (for performance)
                count += 1;
                if (count > TokenUtils.MAX_CELL_ITERATION_COUNT) { throw TokenUtils.UNSUPPORTED_ERROR; }
                // invoke the iterator callback function
                iterator.call(self, value, col, row);
            }

            // visits a single value operand (skip empty parameters unless specified otherwise)
            function iterateValue(value) {
                if (emptyParam || !_.isNull(value)) {
                    invokeIterator(value, 0, 0);
                }
            }

            // visits all elements of a matrix
            function iterateMatrix(matrix) {
                matrix.iterateElements(function (element, row, col) {
                    invokeIterator(element, col, row);
                });
            }

            // visits all filled cells in the passed range
            function iterateCellRange(range) {

                var // start position of the iteration range
                    col = range.start[0], row = range.start[1];

                Utils.iterateRange(range.sheet1, range.sheet2 + 1, function (sheet) {

                    var // the sheet model containing the cell
                        sheetModel = docModel.getSheetModel(sheet);

                    // safety check: #REF! error if passed sheet index is invalid
                    if (!sheetModel) { throw ErrorCodes.REF; }

                    // iterate all content cells in the cell collection (skip empty cells)
                    sheetModel.getCellCollection().iterateCellsInRanges(range, function (cellData) {
                        invokeIterator(cellData.result, cellData.address[0] - col, cellData.address[1] - row);
                    }, { type: emptyCell ? 'all' : 'content', hidden: 'all', ordered: ordered });
                });
            }

            // visit all filled cells in a cell reference
            function iterateReference(ref) {
                if (Utils.getBooleanOption(options, 'complexRef', false)) {
                    ref.iterateValueRanges(refSheet, refAddress, iterateCellRange);
                } else {
                    iterateCellRange(ref.getValueRange(refSheet, refAddress));
                }
            }

            // passed operand can also be a matrix or a reference
            if (operand instanceof Matrix) {
                iterateMatrix(operand);
            } else if (operand instanceof Reference) {
                iterateReference(operand);
            } else if (operand instanceof Operand) {
                if (operand.isMatrix()) {
                    iterateMatrix(operand.getMatrix());
                } else if (operand.isReference()) {
                    iterateReference(operand.getReference());
                } else {
                    iterateValue(operand.getValue());
                }
            } else {
                fatalError('FormulaContext.iterateOperandValues(): invalid operand type');
            }
        }

        // public methods -----------------------------------------------------

        /**
         * Returns the application instance containing the interpreted formula.
         *
         * @returns {SpreadsheetApplication}
         *  The application instance containing the interpreted formula.
         */
        this.getApp = function () {
            return app;
        };

        /**
         * Returns whether the edited document is an Office-Open-XML file.
         *
         * @returns {Boolean}
         *  Whether the edited document is an Office-Open-XML file.
         */
        this.isOOXML = function () {
            return app.isOOXML();
        };

        /**
         * Returns whether the edited document is an OpenDocument file.
         *
         * @returns {Boolean}
         *  Whether the edited document is an OpenDocument file.
         */
        this.isODF = function () {
            return app.isODF();
        };

        /**
         * Returns the document model containing the interpreted formula.
         *
         * @returns {SpreadsheetModel}
         *  The document model containing the interpreted formula.
         */
        this.getModel = function () {
            return docModel;
        };

        /**
         * Returns the number formatter of the document model.
         *
         * @returns {NumberFormatter}
         *  The number formatter of the document model.
         */
        this.getNumberFormatter = function () {
            return numberFormatter;
        };

        /**
         * Returns the formula tokenizer needed for specific conversion tasks.
         *
         * @returns {Tokenizer}
         *  The formula tokenizer needed for specific conversion tasks.
         */
        this.getTokenizer = function () {
            return tokenizer;
        };

        /**
         * Converts the passed value to be stored in an operand. A date will be
         * converted to a floating-point number, if possible. If the value is
         * INF or NaN, the #NUM! error code literal will be returned instead.
         * If the number is too close to zero (using a denormalized binary
         * representation internally), zero will be returned. If a string is
         * too long, a #VALUE! error code will be returned. Complex numbers
         * will be converted to their string representation, unless one of the
         * coefficients is INF or NaN which will result in the #NUM! error too.
         *
         * @param {Number|Date|String|Boolean|Complex|ErrorCode|Null} value
         *  The value to be validated.
         *
         * @returns {Number|String|Boolean|Null|ErrorCode}
         *  The validated value, ready to be inserted into a formula operand.
         */
        this.validateValue = function (value) {

            // convert date object to floating-point number
            if (value instanceof Date) {
                value = numberFormatter.convertDateToNumber(value);
                return _.isNumber(value) ? value : ErrorCodes.NUM;
            }

            // convert complex numbers to strings
            if (value instanceof Complex) {
                // convert INF or NaN to #NUM! error
                if (!_.isFinite(value.real) || !_.isFinite(value.imag)) { return ErrorCodes.NUM; }
                // convert very small (denormalized) numbers to zero
                if (Math.abs(value.real) < TokenUtils.MIN_NUMBER) { value.real = 0; }
                if (Math.abs(value.imag) < TokenUtils.MIN_NUMBER) { value.imag = 0; }
                // convert to string
                value = this.convertComplexToString(value);
            }

            // validate floating-point numbers
            if (_.isNumber(value)) {
                // convert INF or NaN to #NUM! error
                if (!_.isFinite(value)) { return ErrorCodes.NUM; }
                // convert very small (denormalized) numbers to zero
                return (Math.abs(value) < TokenUtils.MIN_NUMBER) ? 0 : value;
            }

            // validate strings
            if (_.isString(value)) {
                // check maximum length of string (depends on file format)
                return (value.length <= MAX_STRING_LEN) ? value : ErrorCodes.VALUE;
            }

            // no conversion for other values
            return value;
        };

        /**
         * Checks the passed string length used in string functions. Throws the
         * #VALUE! error code, if the index is less than zero, or larger than
         * the maximum length of string results supported by the formula
         * engine.
         *
         * @param {Number} length
         *  The string length to be checked.
         *
         * @throws {ErrorCode}
         *  The #VALUE! error code, if the passed string length is invalid.
         */
        this.checkStringLength = function (length) {
            if ((length < 0) || (length > MAX_STRING_LEN)) { throw ErrorCodes.VALUE; }
        };

        /**
         * Checks the passed character index used in string functions. Throws
         * the #VALUE! error code, if the index is less than one, or larger
         * than the maximum length of string results supported by the formula
         * engine.
         *
         * @param {Number} index
         *  The character index to be checked.
         *
         * @throws {ErrorCode}
         *  The #VALUE! error code, if the passed character index is invalid.
         */
        this.checkStringIndex = function (index) {
            if ((index < 1) || (index > MAX_STRING_LEN)) { throw ErrorCodes.VALUE; }
        };

        /**
         * Checks the passed complex numbers for their imaginary unit. Throws
         * the #VALUE! error code, if the imaginary units of the numbers are
         * different.
         *
         * @param {Complex} complex1
         *  The first complex number to be checked.
         *
         * @param {Complex} complex2
         *  The second complex number to be checked.
         *
         * @throws {ErrorCode}
         *  The #VALUE! error code, if the imaginary units of the passed
         *  complex numbers are different.
         */
        this.checkComplexUnits = function (complex1, complex2) {
            if (_.isString(complex1.unit) && _.isString(complex2.unit) && (complex1.unit !== complex2.unit)) {
                throw ErrorCodes.VALUE;
            }
        };

        /**
         * Returns the index of the reference sheet the interpreted formula is
         * located in.
         *
         * @returns {Number}
         *  The zero-based index of the reference sheet.
         */
        this.getRefSheet = function () {
            return refSheet;
        };

        /**
         * Returns the address of the reference cell the interpreted formula is
         * located in.
         *
         * @returns {Number[]}
         *  The address of the reference cell.
         */
        this.getRefAddress = function () {
            return refAddress;
        };

        /**
         * Returns the current result value of the specified cell.
         *
         * @param {Number} sheet
         *  The zero-based index of the sheet.
         *
         * @param {Number[]} address
         *  The address of the cell in the specified sheet.
         *
         * @returns {Number|String|Boolean|ErrorCode|Null}
         *  The current value of the specified cell.
         *
         * @throws {ErrorCode}
         *  The #REF! error code, if the passed sheet index is invalid.
         */
        this.getCellValue = function (sheet, address) {

            var // the sheet model containing the cell
                sheetModel = docModel.getSheetModel(sheet);

            // safety check: #REF! error if passed sheet index is invalid
            if (!sheetModel) { throw ErrorCodes.REF; }

            // return typed value from cell collection
            return sheetModel.getCellCollection().getCellResult(address);
        };

        /**
         * Returns the formula string of the specified cell.
         *
         * @param {Number} sheet
         *  The zero-based index of the sheet.
         *
         * @param {Number[]} address
         *  The address of the cell in the specified sheet.
         *
         * @returns {String|Null}
         *  The current formula string of the specified cell; or null, if the
         *  cell does not contain a formula.
         *
         * @throws {ErrorCode}
         *  The #REF! error code, if the passed sheet index is invalid.
         */
        this.getCellFormula = function (sheet, address) {

            var // the sheet model containing the cell
                sheetModel = docModel.getSheetModel(sheet);

            // safety check: #REF! error if passed sheet index is invalid
            if (!sheetModel) { throw ErrorCodes.REF; }

            // get formula string from cell collection
            return sheetModel.getCellCollection().getCellFormula(address);
        };

        /**
         * Converts the passed floating-point number to a string formatted with
         * the 'General' number format using the appropriate maximum number of
         * digits.
         *
         * @param {Number} number
         *  The floating-point number to be converted to a string.
         *
         * @returns {String}
         *  The converted floating-point number.
         */
        this.convertNumberToString = function (number) {
            return numberFormatter.formatStandardNumber(number, SheetUtils.MAX_LENGTH_STANDARD_FORMULA);
        };

        /**
         * Converts the passed text to a floating-point number.
         *
         * @param {String} text
         *  The string to be converted to a floating-point number.
         *
         * @returns {Number}
         *  The floating-point number represented by the passed string; or NaN,
         *  if the string cannot be parsed to a number.
         */
        this.convertStringToNumber = function (text) {
            return numberFormatter.convertStringToNumber(text);
        };

        /**
         * Converts the passed complex number to a string. The real and
         * imaginary coefficients will be formatted with the 'General' number
         * format using the appropriate maximum number of digits.
         *
         * @param {Complex} complex
         *  The complex number to be converted to a string.
         *
         * @returns {String}
         *  The converted complex number.
         */
        this.convertComplexToString = function (complex) {

            var // the real coefficient, as string
                real = this.convertNumberToString(complex.real),
                // the imaginary coefficient, as string
                imag = this.convertNumberToString(complex.imag),
                // fall-back to unit 'i'
                unit = complex.unit || 'i';

            // leave out imaginary part if zero
            if (complex.imag === 0) { return real; }

            // leave out real part if missing; do not add single '1' for imaginary part
            return ((complex.real === 0) ? '' : real) + (((complex.real !== 0) && (complex.imag > 0)) ? '+' : '') + (/^-?1$/.test(imag) ? '' : imag) + unit;
        };

        /**
         * Converts the passed text to a complex number.
         *
         * @param {String} text
         *  The string to be converted to a complex number.
         *
         * @returns {Complex|Null}
         *  The complex number represented by the passed string; or null, if
         *  the string cannot be parsed to a complex number.
         */
        this.convertStringToComplex = function (text) {

            var // the matches of a regular expression
                matches = null,
                // the parse result for a floating-point number from formatter
                parseResult = null,
                // the parsed real coefficient
                real = 0;

            // do not accept empty strings
            if (text.length === 0) { return null; }

            // string may be a single imaginary unit without coefficients: i, +i, -i (same for j)
            if ((matches = /^([-+]?)([ij])$/.exec(text))) {
                return new Complex(0, (matches[1] === '-') ? -1 : 1, matches[2]);
            }

            // pull leading floating-point number from the string
            if (!(parseResult = numberFormatter.parseNumberFromString(text, { sign: true }))) {
                return null;
            }

            // check for simple floating-point number without imaginary coefficient: a, +a, -a
            real = parseResult.number;
            text = parseResult.remaining;
            if (text === '') {
                return new Complex(real, 0);
            }

            // check for imaginary number without real coefficient: bi, +bi, -bi
            if ((text === 'i') || (text === 'j')) {
                return new Complex(0, real, text);
            }

            // check for following imaginary unit without coefficients, but with sign: a+i, a-i
            if ((matches = /^([-+])([ij])$/.exec(text))) {
                return new Complex(real, (matches[1] === '-') ? -1 : 1, matches[2]);
            }

            // pull trailing floating-point number from the string: a+bi, a-bi (sign is required here, something like 'abi' is not valid)
            if (!(parseResult = numberFormatter.parseNumberFromString(text, { sign: true })) || (parseResult.sign.length === 0)) {
                return null;
            }

            // remaining text must be the imaginary unit
            text = parseResult.remaining;
            if ((text === 'i') || (text === 'j')) {
                return new Complex(real, parseResult.number, text);
            }

            return null;
        };

        /**
         * Tries to convert the passed literal value to a floating-point
         * number.
         *
         * @param {Number|Date|String|Boolean|ErrorCode|Null} value
         *  A literal value to be converted to a number. Strings that represent
         *  a valid number (containing the decimal separator of the current GUI
         *  language) will be converted to the number. The Boolean value FALSE
         *  will be converted to 0, the Boolean value TRUE will be converted to
         *  1. The special value null (representing an empty cell) will be
         *  converted to 0.
         *
         * @returns {Number}
         *  The floating-point number, if available.
         *
         * @throws {ErrorCode}
         *  If the passed value represents a literal error code, it will be
         *  thrown. If the value cannot be converted to a floating-point
         *  number, a #VALUE! error will be thrown.
         */
        this.convertToNumber = function (value) {

            // immediately throw literal error code
            if (SheetUtils.isErrorCode(value)) { throw value; }

            // convert strings and Booleans to numbers
            if (_.isString(value)) {
                value = numberFormatter.convertStringToNumber(value);
            } else if (_.isBoolean(value)) {
                value = value ? 1 : 0;
            } else if (value instanceof Date) {
                // value may become null which will cause the #VALUE! error below
                value = numberFormatter.convertDateToNumber(value);
            } else if (_.isNull(value)) {
                value = 0;
            }

            // the resulting value must be a finite floating-point number
            if (_.isNumber(value) && _.isFinite(value)) { return value; }
            throw ErrorCodes.VALUE;
        };

        /**
         * Tries to convert the passed literal value to a Date object.
         *
         * @param {Number|Date|String|Boolean|ErrorCode|Null} value
         *  A literal value to be converted to a number. Strings that represent
         *  a valid number (containing the decimal separator of the current GUI
         *  language) will be converted to a date representing that number. The
         *  Boolean value FALSE will be converted to the null date, the Boolean
         *  value TRUE will be converted to the day following the null date.
         *  The special value null (representing an empty cell) will be
         *  converted to 0.
         *
         * @returns {Date}
         *  The date, if available.
         *
         * @throws {ErrorCode}
         *  If the passed value represents a literal error code, it will be
         *  thrown. If the value cannot be converted to a date value, a #VALUE!
         *  error will be thrown.
         */
        this.convertToDate = function (value) {

            // immediately throw literal error code
            if (SheetUtils.isErrorCode(value)) { throw value; }

            if (!(value instanceof Date)) {
                // convert the value to a number (failing conversion will throw)
                value = this.convertToNumber(value);
                // convert the number to a date object
                value = numberFormatter.convertNumberToDate(value);
            }

            // the resulting value must be a Date object
            if (value instanceof Date) { return value; }
            throw ErrorCodes.VALUE;
        };

        /**
         * Tries to convert the passed literal value to a string.
         *
         * @param {Number|Date|String|Boolean|ErrorCode|Null} value
         *  A literal value to be converted to a string. Numbers will be
         *  converted to decimal or scientific notation (containing the decimal
         *  separator of the current GUI language), Boolean values will be
         *  converted to their translated text representation (formulas will
         *  always be calculated using the current GUI language). The special
         *  value null (representing an empty cell) will be converted to the
         *  empty string.
         *
         * @returns {String}
         *  The string representation of the value, if available.
         *
         * @throws {ErrorCode}
         *  If the passed value represents a literal error code, it will be
         *  thrown. All other values will be converted to strings.
         */
        this.convertToString = function (value) {

            // immediately throw literal error code
            if (SheetUtils.isErrorCode(value)) { throw value; }

            // treat date objects as plain unformatted numbers
            if (value instanceof Date) {
                value = this.convertToNumber(value);
            }

            // convert numbers and Booleans to strings
            if (_.isNumber(value) && _.isFinite(value)) {
                value = this.convertNumberToString(value);
            } else if (_.isBoolean(value)) {
                value = app.getBooleanLiteral(value); // always translated
            } else if (_.isNull(value)) {
                value = '';
            }

            // the resulting value must be a string
            if (_.isString(value)) { return value; }
            throw ErrorCodes.VALUE;
        };

        /**
         * Tries to convert the passed value to a Boolean value.
         *
         * @param {Number|Date|String|Boolean|ErrorCode|Null} value
         *  A literal value to be converted to a Boolean. Floating-point
         *  numbers will be converted to TRUE if not zero, otherwise FALSE.
         *  Strings containing the exact translated name of the TRUE or FALSE
         *  values (case-insensitive) will be converted to the respective
         *  Boolean value (formulas will always be calculated using the current
         *  GUI language). The special value null (representing an empty cell)
         *  will be converted to FALSE.
         *
         * @returns {Boolean}
         *  The Boolean value, if available.
         *
         * @throws {ErrorCode}
         *  If the passed value represents a literal error code, it will be
         *  thrown. If the value cannot be converted to a Boolean, a #VALUE!
         *  error will be thrown.
         */
        this.convertToBoolean = function (value) {

            // immediately throw literal error code
            if (SheetUtils.isErrorCode(value)) { throw value; }

            // treat date objects as plain unformatted numbers
            if (value instanceof Date) {
                value = this.convertToNumber(value);
            }

            // convert numbers and strings to Booleans
            if (_.isNumber(value) && _.isFinite(value)) {
                value = Math.abs(value) > TokenUtils.MIN_NUMBER;
            } else if (_.isString(value)) {
                value = value.toUpperCase();
                if (value === app.getBooleanLiteral(true)) { // always translated
                    value = true;
                } else if (value === app.getBooleanLiteral(false)) {
                    value = false;
                }
            } else if (_.isNull(value)) {
                value = false;
            }

            // the resulting value must be a Boolean
            if (_.isBoolean(value)) { return value; }
            throw ErrorCodes.VALUE;
        };

        /**
         * Tries to convert the passed literal value to a complex number.
         *
         * @param {Number|Date|String|Boolean|ErrorCode|Null} value
         *  A literal value to be converted to a complex number. Floating-point
         *  numbers will be converted to complex numbers with the imaginary
         *  coefficient set to zero. Strings must represent a valid complex
         *  number (optional leading signed real coefficient, optional trailing
         *  signed imaginary coefficient, followed by the lower-case character
         *  'i' or 'j'). The coefficients must containing the decimal separator
         *  of the current GUI language. The special value null (representing
         *  an empty cell) will be converted to the complex number 0.
         *
         * @returns {Complex}
         *  The complex number, if available.
         *
         * @throws {ErrorCode}
         *  If the passed value represents a literal error code, it will be
         *  thrown. Boolean values will cause to throw a #VALUE! error. If the
         *  string cannot be converted to a complex number, a #NUM! error will
         *  be thrown (not the #VALUE! error code as thrown by the other
         *  conversion methods).
         */
        this.convertToComplex = function (value) {

            // immediately throw literal error code
            if (SheetUtils.isErrorCode(value)) { throw value; }

            // Boolean values result in #VALUE! (but not invalid strings)
            if (_.isBoolean(value)) { throw ErrorCodes.VALUE; }

            // treat date objects as plain unformatted numbers
            if (value instanceof Date) {
                value = this.convertToNumber(value);
            }

            // convert numbers and strings to complex numbers
            if (_.isNumber(value) && _.isFinite(value)) {
                value = new Complex(value, 0);
            } else if (_.isString(value)) {
                value = this.convertStringToComplex(value);
            } else if (_.isNull(value)) {
                // empty cells are treated as complex number 0 (in difference to empty strings)
                value = new Complex(0, 0);
            }

            // the resulting value must be a complex number
            if (value instanceof Complex) { return value; }
            // invalid strings will result in #NUM! instead of #VALUE!
            throw ErrorCodes.NUM;
        };

        /**
         * Visits all values contained in the passed operand. Provides a single
         * constant of a literal operand, all elements of a matrix operand, and
         * the cell contents referred by a reference operand.
         *
         * @param {Any} operand
         *  The operand value (single value of any supported data type,
         *  including null and error codes; or instances of the classes Matrix
         *  or Reference) whose content values will be visited.
         *
         * @param {Function} iterator
         *  The iterator callback function invoked for each value contained in
         *  the operand. Will be called in the context of this instance.
         *  Receives the value as first parameter.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.emptyParam=false]
         *      If set to true, and the operand represents the empty function
         *      parameter, the iterator callback function will be invoked with
         *      the value null. By default, empty parameters will be skipped.
         *  @param {Boolean} [options.acceptErrors=false]
         *      If set to true, the iterator callback function will be invoked
         *      for error code literals too. By default, error codes will be
         *      thrown immediately.
         *  @param {Boolean} [options.complexRef=false]
         *      If set to true, a reference operand may be complex. Complex
         *      references may contain multiple cell ranges, and each of the
         *      cell ranges may refer to multiple sheets. By default, a
         *      reference must refer to a single range in a single sheet.
         *  @param {Boolean} [options.ordered=false]
         *      If set to true, the cells in each cell range will be visited in
         *      order (row-by-row from top to bottom, and left-to-right inside
         *      each of the rows).
         *
         * @returns {FormulaContext}
         *  A reference to this instance.
         */
        this.iterateValues = function (operand, iterator, options) {
            iterateOperandValues(operand, _.identity, iterator, options);
            return this;
        };

        /**
         * Visits all floating-point numbers contained in the passed operand.
         * Provides a single constant of a literal operand, all elements of a
         * matrix operand, and the cell contents referred by a reference
         * operand. In difference to simple value parameters where strings and
         * Boolean values contained in matrixes and cell ranges will always be
         * converted to numbers if possible, this method implements different
         * conversion strategies as required by the respective functions.
         *
         * @param {Number|Date|String|Boolean|Null|ErrorCode|Matrix|Reference} operand
         *  The operand value (single value, matrix, or reference) whose
         *  content values will be visited.
         *
         * @param {Function} iterator
         *  The iterator callback function invoked for each floating-point
         *  number contained in the operand. Will be called in the context of
         *  this instance. Receives the number as first parameter.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {String} [options.valMode='convert']
         *      Conversion strategy for operands of type value:
         *      - 'convert': Strings will be converted to numbers, if possible.
         *          Boolean values will be converted to numbers. Strings that
         *          represent a Boolean value result in a #VALUE! error code
         *          though.
         *      - 'rconvert' (restricted conversion): Strings will be converted
         *          to numbers, if possible. Boolean values and strings that
         *          represent a Boolean value result in a #VALUE! error code
         *          though.
         *      - 'exact': Accepts numbers only. All strings and Boolean values
         *          result in a #VALUE! error code.
         *      - 'skip': All strings and all Boolean values will be skipped
         *          silently, also strings that could be converted to numbers.
         *      - 'zero': All strings will be treated as the number zero, also
         *          strings that could be converted to numbers. Boolean values
         *          will be converted to numbers.
         *  @param {String} [options.matMode='convert']
         *      Conversion strategy for the elements in matrixes. See option
         *      'valMode' for details.
         *  @param {String} [options.refMode='convert']
         *      Conversion strategy for the contents of cells referenced by the
         *      operand. See option 'valMode' for details.
         *  @param {Boolean} [options.emptyParam=false]
         *      If set to true, and the operand represents the empty function
         *      parameter, the iterator callback function will be invoked with
         *      the value zero. By default, empty parameters will be skipped.
         *  @param {Boolean} [options.complexRef=false]
         *      If set to true, a reference operand may be complex. Complex
         *      references may contain multiple cell ranges, and each of the
         *      cell ranges may refer to multiple sheets. By default, a
         *      reference must refer to a single range in a single sheet.
         *  @param {Boolean} [options.ordered=false]
         *      If set to true, the cells in each cell range will be visited in
         *      order (row-by-row from top to bottom, and left-to-right inside
         *      each of the rows).
         *
         * @returns {FormulaContext}
         *  A reference to this instance.
         */
        this.iterateNumbers = function (operand, iterator, options) {

            var // conversion strategy
                convertMode = Utils.getStringOption(options, operand.getType() + 'Mode', 'convert');

            iterateOperandValues(operand, function (value) {
                switch (convertMode) {
                case 'convert':
                    // convert value to number (may still fail for strings)
                    return this.convertToNumber(value);
                case 'rconvert':
                    // throw #VALUE! error code on Booleans
                    if (_.isBoolean(value)) { throw ErrorCodes.VALUE; }
                    // convert value to number (may still fail for strings)
                    return this.convertToNumber(value);
                case 'exact':
                    // throw #VALUE! error code on Booleans and strings
                    if (_.isString(value) || _.isBoolean(value)) { throw ErrorCodes.VALUE; }
                    // convert dates to numbers
                    return this.convertToNumber(value);
                case 'skip':
                    // skip strings and Boolean values
                    return _.isNumber(value) ? value : null;
                case 'zero':
                    // replace all strings with zero
                    return _.isString(value) ? 0 : this.convertToNumber(value);
                default:
                    fatalError('FormulaContext.iterateNumbers(): unknown conversion mode: "' + convertMode + '"');
                }
            }, iterator, Utils.extendOptions(options, { acceptErrors: false }));

            return this;
        };

        /**
         * Visits all Boolean values contained in the passed operand. Provides
         * a single constant of a literal operand, all elements of a matrix
         * operand, and the cell contents referred by a reference operand. In
         * difference to simple value parameters where numbers and strings
         * contained in matrixes and cell ranges will always be converted to
         * Boolean values if possible, this method implements different
         * conversion strategies as required by the respective functions.
         *
         * @param {Number|Date|String|Boolean|Null|ErrorCode|Matrix|Reference} operand
         *  The operand value (single value, matrix, or reference) whose
         *  content values will be visited.
         *
         * @param {Function} iterator
         *  The iterator callback function invoked for each Boolean value
         *  contained in the operand. Will be called in the context of this
         *  instance. Receives the Boolean value as first parameter.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {String} [options.valMode='convert']
         *      Conversion strategy for operands of type value:
         *      - 'convert': Strings will be converted to Boolean values, if
         *          possible. Numbers will be converted to Boolean values.
         *          Strings that represent a number result in a #VALUE! error
         *          code though.
         *      - 'skip': All strings will be skipped silently, also strings
         *          that could be converted to Boolean values. Numbers will be
         *          converted though.
         *  @param {String} [options.matMode='convert']
         *      Conversion strategy for the elements in matrixes. See option
         *      'valMode' for details.
         *  @param {String} [options.refMode='convert']
         *      Conversion strategy for the contents of cells referenced by the
         *      operand. See option 'valMode' for details.
         *  @param {Boolean} [options.emptyParam=false]
         *      If set to true, and the operand represents the empty function
         *      parameter, the iterator callback function will be invoked with
         *      the value FALSE. By default, empty parameters will be skipped.
         *  @param {Boolean} [options.complexRef=false]
         *      If set to true, a reference operand may be complex. Complex
         *      references may contain multiple cell ranges, and each of the
         *      cell ranges may refer to multiple sheets. By default, a
         *      reference must refer to a single range in a single sheet.
         *  @param {Boolean} [options.ordered=false]
         *      If set to true, the cells in each cell range will be visited in
         *      order (row-by-row from top to bottom, and left-to-right inside
         *      each of the rows).
         *
         * @returns {FormulaContext}
         *  A reference to this instance.
         */
        this.iterateBooleans = function (operand, iterator, options) {

            var // conversion strategy
                convertMode = Utils.getStringOption(options, operand.getType() + 'Mode', 'convert');

            iterateOperandValues(operand, function (value) {
                switch (convertMode) {
                case 'convert':
                    // convert value to Boolean (may still fail for strings)
                    return this.convertToBoolean(value);
                case 'skip':
                    // skip strings (but not numbers)
                    return _.isString(value) ? null : this.convertToBoolean(value);
                default:
                    fatalError('FormulaContext.iterateBooleans(): unknown conversion mode: "' + convertMode + '"');
                }
            }, iterator, Utils.extendOptions(options, { acceptErrors: false }));

            return this;
        };

        /**
         * Reduces all numbers (and other values convertible to numbers) of all
         * function parameters to the result of a spreadsheet function.
         *
         * @param {Array} operands
         *  The operands containing numbers to be aggregated. The elements of
         *  the array can be instances of the classes Operand, Reference, or
         *  Matrix; or may be any literal values used in formulas.
         *
         * @param {Number} initial
         *  The initial intermediate result, passed to the first invocation of
         *  the aggregation callback function.
         *
         * @param {Function} aggregate
         *  The aggregation callback function. Receives the following
         *  parameters:
         *  (1) {Number} result
         *      The current intermediate result. On first invocation, this is
         *      the number passed in the 'initial' parameter.
         *  (2) {Number} number
         *      The new number to be combined with the intermediate result.
         *  Must return the new intermediate result which will be passed to the
         *  next invocation of this callback function, or may throw an error
         *  code. Will be called in the context of this instance.
         *
         * @param {Function} finalize
         *  A callback function invoked at the end to convert the intermediate
         *  result (returned by the last invocation of the aggregation callback
         *  function) to the actual function result. Receives the following
         *  parameters:
         *  (1) {Number} result
         *      The last intermediate result to be converted to the result of
         *      the implemented function.
         *  (2) {Number} count
         *      The count of all visited numbers.
         *  Must return the final numeric result of the spreadsheet function,
         *  or an error code object. Alternatively, may throw an error code.
         *  Will be called in the context of this instance.
         *
         * @param {Object} iteratorOptions
         *  Parameters passed to the number iterator used internally (see
         *  method FormulaContext.iterateNumbers() for details).
         *
         * @returns {Number}
         *  The result of the spreadsheet function.
         *
         * @throws {ErrorCode}
         *  An error code if calculating the function result has failed.
         */
        this.aggregateNumbers = function (operands, initial, aggregate, finalize, iteratorOptions) {

            var // the intermediate result
                result = initial,
                // the count of all visited numbers in all operands
                count = 0;

            // process all function parameters
            _.each(operands, function (operand) {
                this.iterateNumbers(operand, function (number) {
                    result = aggregate.call(this, result, number);
                    count += 1;
                }, iteratorOptions);
            }, this);

            // resolve the final result, throw error codes
            result = finalize.call(this, result, count);
            if (_.isNumber(result)) { return result; }
            if (SheetUtils.isErrorCode(result)) { throw result; }
            fatalError('FormulaContext.aggregateNumbers(): invalid aggregation result');
        };

        /**
         * Reduces all Boolean values (and other values convertible to Boolean
         * values) of all function parameters to the result of a spreadsheet
         * function.
         *
         * @param {Array} operands
         *  The operands containing Boolean values to be aggregated. The
         *  elements of the array can be instances of the classes Operand,
         *  Reference, or Matrix; or may be any literal values used in
         *  formulas.
         *
         * @param {Boolean} initial
         *  The initial intermediate result, passed to the first invocation of
         *  the aggregation callback function.
         *
         * @param {Function} aggregate
         *  The aggregation callback function. Receives the following
         *  parameters:
         *  (1) {Boolean} result
         *      The current intermediate result. On first invocation, this is
         *      the Boolean value passed in the 'initial' parameter.
         *  (2) {Boolean} bool
         *      The new Boolean value to be combined with the intermediate
         *      result.
         *  Must return the new intermediate result which will be passed to the
         *  next invocation of this callback function, or may throw an error
         *  code. Will be called in the context of this instance.
         *
         * @param {Function} finalize
         *  A callback function invoked at the end to convert the intermediate
         *  result (returned by the last invocation of the aggregation callback
         *  function) to the actual function result. Receives the following
         *  parameters:
         *  (1) {Boolean} result
         *      The last intermediate result to be converted to the result of
         *      the implemented function.
         *  (2) {Number} count
         *      The count of all visited Boolean values.
         *  Must return the final Boolean result of the spreadsheet function,
         *  or an error code object. Alternatively, may throw an error code.
         *  Will be called in the context of this instance.
         *
         * @param {Object} iteratorOptions
         *  Parameters passed to the Boolean value iterator used internally
         *  (see method FormulaContext.iterateBooleans() for details).
         *
         * @returns {Boolean}
         *  The result of the spreadsheet function.
         *
         * @throws {ErrorCode}
         *  An error code if calculating the function result has failed.
         */
        this.aggregateBooleans = function (operands, initial, aggregate, finalize, iteratorOptions) {

            var // the intermediate result
                result = initial,
                // the count of all visited Boolean values in all operands
                count = 0;

            // process all function parameters
            _.each(operands, function (operand) {
                this.iterateBooleans(operand, function (bool) {
                    result = aggregate.call(this, result, bool);
                    count += 1;
                }, iteratorOptions);
            }, this);

            // resolve the final result, throw error codes
            result = finalize.call(this, result, count);
            if (_.isBoolean(result)) { return result; }
            if (SheetUtils.isErrorCode(result)) { throw result; }
            fatalError('FormulaContext.aggregateBooleans(): invalid aggregation result');
        };

        /**
         * Converts the passed filter pattern to a case-insensitive regular
         * expression object.
         *
         * @param {String} pattern
         *  The filter pattern. An asterisk character in the string matches any
         *  sequence of characters (also the empty sequence), and a question
         *  mark matches an arbitrary single character. If an asterisk or
         *  question mark is preceded by a tilde character, they lose their
         *  special meaning and will be matched literally.
         *
         * @param {Boolean} [complete=false]
         *  If set to true, the created regular expression will only match
         *  complete strings. By default, any substring of a string will be
         *  matched.
         *
         * @returns {RegExp}
         *  A case-insensitive regular expression that represents the passed
         *  filter pattern.
         */
        this.convertPatternToRegExp = function (pattern, complete) {

            // convert passed search text to regular expression pattern
            pattern = _.escapeRegExp(pattern)
                // replace all occurrences of asterisk (without leading tilde) to '.*' RE pattern (e.g. 'a\*b' => 'a.*b')
                .replace(/(^|[^~])\\\*/g, '$1.*')
                // replace all occurrences of question mark (without leading tilde) to '.' RE pattern (e.g. 'a\?b' => 'a.b')
                .replace(/(^|[^~])\\\?/g, '$1.')
                // remove tilde characters before (already escaped) asterisks and question marks (e.g. 'a~\*b' => 'a\*b')
                .replace(/~(\\[*?])/g, '$1');

            // optionally, let the regular expression only match complete strings
            if (complete) { pattern = '^' + pattern + '$'; }

            // create the case-insensitive regular expression
            return new RegExp(pattern, 'i');
        };

        /**
         * Creates a filter matcher predicate function for the specified filter
         * criterion.
         *
         * @param {Number|String|Boolean|ErrorCode|Null} criterion
         *  The filter criterion. Numbers, Boolean values and error codes will
         *  match the corresponding values, and will match strings that can be
         *  converted to these values. The value null represents a reference to
         *  an empty cell, and will match the number zero only (but not the
         *  empty string, and not empty cells). Strings may contain leading
         *  comparison operators, or may specify filter patterns with asterisk
         *  characters or question marks. Otherwise, string criteria will be
         *  converted to numbers, Boolean values, or error codes, if possible,
         *  and will match the corresponding values.
         *
         * @returns {Function}
         *  A predicate function that takes a single parameter (a value of any
         *  type that can occur in a formula), and returns a Boolean whether
         *  the passed value matches the filter criterion.
         */
        this.createFilterMatcher = function (criterion) {

            var // the number formatter of the document
                numberFormatter = this.getNumberFormatter(),
                // the filter operator (empty string for criterion without operator)
                op = null,
                // the filter operand
                comp = null,
                // whether to test for equality (needs some special handling)
                equality = null;

            // parse the filter operator and the filter operand from a string criterion
            if (_.isString(criterion)) {
                var matches = /^(<=|>=|<>|<|>|=)?(.*)$/.exec(criterion);
                op = _.isString(matches[1]) ? matches[1] : '';
                // leading apostrophe in operand counts as regular character
                comp = numberFormatter.parseString(matches[2], { keepApos: true });
            } else {
                // all other data types cause filtering by equality
                op = '';
                comp = criterion;
            }

            // testing for equality converts strings in source data to other data types
            equality = (op === '') || (op === '=');

            // empty filter operand (empty parameter, or reference to empty cell) matches zero
            // numbers (and strings that can be parsed to zero number) in source data, but not
            // empty strings, FALSE, or empty cells
            if (_.isNull(comp)) {
                return function zeroMatcher(value) {
                    return (_.isString(value) ? numberFormatter.parseString(value) : value) === 0;
                };
             }

            // special handling for the empty string as filter operand
            if (comp === '') {
                switch (op) {

                // literal empty string without filter operator matches empty strings AND empty cells
                case '':
                    return function emptyMatcher(value) { return (value === '') || _.isNull(value); };

                // single equality operator without operand matches empty cells, but not empty strings
                case '=':
                    return _.isNull;

                // single inequality operator without operand matches anything but empty cells
                case '<>':
                    return function notEmptyMatcher(value) { return !_.isNull(value); };

                // all other comparison operators without operand do not match anything
                default:
                    return _.constant(false);
                }
            }

            // equality or inequality with pattern matching
            if (_.isString(comp) && /[*?]/.test(comp) && (equality || (op === '<>'))) {

                var // the regular expression for the pattern (match complete strings only)
                    regExp = this.convertPatternToRegExp(comp, true);

                // create the pattern matcher predicate function (pattern filters work on strings only)
                return function patternMatcher(value) {
                    return equality === (_.isString(value) && regExp.test(value));
                };
            }

            // strings are always matched case-insensitively
            if (_.isString(comp)) { comp = comp.toUpperCase(); }

            // create the comparison matcher predicate function (pattern filters work on strings only)
            return function comparisonMatcher(value) {

                var // the relation between the value and the filter operand (signed integer)
                    rel = Number.NaN;

                // equality operator (but not inequality operator) converts strings in source data to other data types
                if (equality && _.isString(value)) {
                    value = numberFormatter.parseString(value);
                }

                // compare the passed value with the filter operand (NaN means incompatible types)
                if (_.isNumber(comp) && _.isNumber(value)) {
                    rel = TokenUtils.compareNumbers(value, comp);
                } else if (_.isString(comp) && _.isString(value)) {
                    rel = TokenUtils.compareStrings(value, comp);
                } else if (_.isBoolean(comp) && _.isBoolean(value)) {
                    rel = TokenUtils.compareBooleans(value, comp);
                } else if (SheetUtils.isErrorCode(comp) && SheetUtils.isErrorCode(value)) {
                    rel = TokenUtils.compareErrorCodes(value, comp);
                }

                // return whether the value matches, according to the filter operand
                switch (op) {
                case '':
                case '=':
                    return isFinite(rel) && (rel === 0);
                case '<>':
                    return !isFinite(rel) || (rel !== 0); // incompatible types match the inequality operator
                case '<':
                    return isFinite(rel) && (rel < 0);
                case '<=':
                    return isFinite(rel) && (rel <= 0);
                case '>':
                    return isFinite(rel) && (rel > 0);
                case '>=':
                    return isFinite(rel) && (rel >= 0);
                }

                fatalError('FormulaContext.comparisonMatcher(): invalid operator');
            };
        };

        /**
         * Reduces all values that match a specific filter criterion to the
         * result of a spreadsheet function.
         *
         * @param {Reference} sourceRef
         *  A reference with the address of the cell range containing the
         *  source data to be filtered for.
         *
         * @param {Number|String|Boolean|ErrorCode|Null} criterion
         *  The filter criterion. See FormulaContext.createFilterMatcher()
         *  method for more details.
         *
         * @param {Reference|Null} dataRef
         *  A reference to the data cells that will be aggregated to the result
         *  of the spreadsheet function. If missing, uses the values in the
         *  filtered source reference.
         *
         * @param {Number} initial
         *  The initial intermediate result, passed to the first invocation of
         *  the aggregation callback function.
         *
         * @param {Function} aggregate
         *  The aggregation callback function. Receives the following
         *  parameters:
         *  (1) {Number} result
         *      The current intermediate result. On first invocation, this is
         *      the number passed in the 'initial' parameter.
         *  (2) {Number} number
         *      The new number to be combined with the intermediate result.
         *  Must return the new intermediate result which will be passed to the
         *  next invocation of this callback function, or may throw an error
         *  code. Will be called in the context of this instance.
         *
         * @param {Function} finalize
         *  A callback function invoked at the end to convert the intermediate
         *  result (returned by the last invocation of the aggregation callback
         *  function) to the actual function result. Receives the following
         *  parameters:
         *  (1) {Number} result
         *      The last intermediate result to be converted to the result of
         *      the implemented function.
         *  (2) {Number} numberCount
         *      The count of all numbers the aggregation callback function has
         *      been invoked for.
         *  (3) {Number} matchingCount
         *      The count of all matching values in the source range (also
         *      non-numeric matching cells).
         *  Must return the final numeric result of the spreadsheet function,
         *  or an error code object. Alternatively, may throw an error code.
         *  Will be called in the context of this instance.
         *
         * @returns {Number}
         *  The result of the spreadsheet function.
         *
         * @throws {ErrorCode}
         *  An error code if calculating the function result has failed.
         */
        this.aggregateFiltered = function (sourceRef, criterion, dataRef, initial, aggregate, finalize) {

            var // value matcher predicate function for the source data
                matcher = this.createFilterMatcher(criterion),
                // the start address of the data range
                dataRange = (dataRef instanceof Reference) ? dataRef.getValueRange(refSheet, refAddress) : null,
                // the intermediate result
                result = initial,
                // the count of all numbers
                numberCount = 0,
                // the count of all matching values in the source range
                matchingCount = 0;

            // process all values in the passed operand (also empty cells in cell references)
            this.iterateValues(sourceRef, function (value, col, row) {

                // skip non-matching cells completely
                if (!matcher(value)) { return; }

                var // the target address in the data range (invalid addresses will be ignored silently)
                    address = dataRange ? [dataRange.start[0] + col, dataRange.start[1] + row] : null,
                    // pick the value from the data cell to be aggregated, or use the source value
                    dataValue = address ? this.getCellValue(dataRange.sheet1, address) : value;

                // aggregate numbers only, without automatic conversion of strings
                if (_.isNumber(dataValue)) {
                    result = aggregate.call(this, result, dataValue);
                    numberCount += 1;
                }

                // count all matching cells regardless of their type, or the type of the corresponding data cell
                matchingCount += 1;

            }, {
                emptyParam: true, // empty parameters count as zero
                emptyCell: true, // empty cells in references are matched by the empty string
                acceptErrors: true, // error codes can be matched by the filter criterion
                complexRef: false // do not accept multi-range and multi-sheet references
            });

            // resolve the final result
            return finalize.call(this, result, numberCount, matchingCount);
        };

    } // class FormulaContext

    // class ValueOperandMixin ================================================

    /**
     * Mix-in for the class Operand for constant values.
     *
     * @constructor
     *
     * @param {FormulaContext} context
     *  The context object providing a connection to the document model.
     *
     * @param {Number|Date|String|Boolean|Null|ErrorCode} value
     *  The value carried by the operand.
     */
    function ValueOperandMixin(context, value) {

        // public methods -----------------------------------------------------

        /**
         * Returns the operand type.
         *
         * @returns {String}
         *  The text 'val' representing single literal values.
         */
        this.getType = function () {
            return 'val';
        };

        /**
         * Returns the value contained in this operand.
         *
         * @returns {Number|String|Boolean|ErrorCode}
         *  The literal value contained in this operand.
         */
        this.getValue = function () {
            return value;
        };

        /**
         * Packs the value of this operand into a 1x1 matrix.
         *
         * @returns {Matrix}
         *  A constant 1x1 matrix with the value of this operand.
         */
        this.getMatrix = function () {
            return new Matrix(1, 1, value);
        };

        /**
         * Returns the unresolved cell reference contained in this operand.
         * Constant values and constant matrixes will immediately result in a
         * #VALUE! error.
         *
         * @throws {ErrorCode}
         *  The error code #VALUE!.
         */
        this.getReference = function () {
            throw ErrorCodes.VALUE;
        };

        /**
         * Returns the string representation of the operand for debug logging.
         */
        this.toString = function () {
            return TokenUtils.valueToString(value);
        };

        // initialization -----------------------------------------------------

        // validate the passed value
        value = context.validateValue(value);

    } // ValueOperandMixin

    // class MatrixOperandMixin ===============================================

    /**
     * Mix-in for the class Operand for matrix constants.
     *
     * @constructor
     *
     * @param {FormulaContext} context
     *  The context object providing a connection to the document model.
     *
     * @param {Matrix} matrix
     *  The matrix carried by the operand.
     */
    function MatrixOperandMixin(context, matrix) {

        // public methods -----------------------------------------------------

        /**
         * Returns the operand type.
         *
         * @returns {String}
         *  The text 'mat' representing a constant matrix literal.
         */
        this.getType = function () {
            return 'mat';
        };

        /**
         * Returns the top-left element of the matrix contained in this
         * operand.
         *
         * @returns {Number|String|Boolean|ErrorCode|Null}
         *  The specified matrix element.
         */
        this.getValue = function (row, col) {
            return matrix.getElement(row || 0, col || 0);
        };

        /**
         * Returns the constant matrix contained in this operand.
         *
         * @returns {Matrix}
         *  The matrix contained in this operand.
         */
        this.getMatrix = function () {
            return matrix;
        };

        /**
         * Returns the unresolved cell reference contained in this operand.
         * Constant values and constant matrixes will immediately result in a
         * #VALUE! error.
         *
         * @throws {ErrorCode}
         *  The error code #VALUE!.
         */
        this.getReference = function () {
            throw ErrorCodes.VALUE;
        };

        /**
         * Returns the string representation of the operand for debug logging.
         */
        this.toString = function () {
            return matrix.toString();
        };

        // initialization -----------------------------------------------------

        // validate all numbers in the matrix
        matrix.transformElements(_.bind(context.validateValue, context));

    } // MatrixOperandMixin

    // class ReferenceOperandMixin ============================================

    /**
     * Mix-in for the class Operand for references.
     *
     * @constructor
     *
     * @param {FormulaContext} context
     *  The context object providing a connection to the document model.
     *
     * @param {Number} refSheet
     *  The zero-based index of the reference sheet.
     *
     * @param {Number[]} refAddress
     *  The address of the reference cell.
     *
     * @param {Reference} ref
     *  The reference carried by the operand.
     */
    function ReferenceOperandMixin(context, refSheet, refAddress, ref) {

        var // the document model
            docModel = context.getModel();

        // public methods -----------------------------------------------------

        /**
         * Returns the operand type.
         *
         * @returns {String}
         *  The text 'ref' representing an unresolved reference.
         */
        this.getType = function () {
            return 'ref';
        };

        /**
         * Returns the content value of the cell that is related to the passed
         * reference cell.
         *
         * @returns {Number|String|Boolean|ErrorCode|Null}
         *  A single literal value, if available. The value null represents an
         *  empty cell, or a cell that has not been stored locally yet.
         *
         * @throws {ErrorCode}
         *  - CIRCULAR_ERROR, if the operand results in a circular reference.
         *  - The #VALUE! error code, if the reference contains more than one
         *      cell range, or if the cell range cannot be resolved to a single
         *      value according to the reference cell.
         */
        this.getValue = function () {

            var // the single validated range structure
                range = ref.getValueRange(refSheet, refAddress);

            // pick single cell
            if (_.isEqual(range.start, range.end)) {
                return context.getCellValue(range.sheet1, range.start);
            }

            // pick matching cell from left or right (row interval must cover the reference cell)
            if ((range.end[0] < refAddress[0]) || (refAddress[0] < range.start[0])) {
                if (range.start[0] !== range.end[0]) { throw ErrorCodes.VALUE; }
                if (!SheetUtils.rangeContainsRow(range, refAddress[1])) { throw ErrorCodes.VALUE; }
                return context.getCellValue(range.sheet1, [range.start[0], refAddress[1]]);
            }

            // pick matching cell from above or below (column interval must cover the reference cell)
            if ((range.end[1] < refAddress[1]) || (refAddress[1] < range.start[1])) {
                if (range.start[1] !== range.end[1]) { throw ErrorCodes.VALUE; }
                if (!SheetUtils.rangeContainsCol(range, refAddress[0])) { throw ErrorCodes.VALUE; }
                return context.getCellValue(range.sheet1, [refAddress[0], range.start[1]]);
            }

            // range is not left of, right of, above, or below the reference cell
            throw ErrorCodes.VALUE;
        };

        /**
         * Returns a two-dimensional matrix with the contents of the referenced
         * cells. The operand must consist of a single reference, and the range
         * must not exceed specific limits in order to prevent performance
         * problems.
         *
         * @returns {Matrix}
         *  A constant matrix, if available.
         *
         * @throws {ErrorCode}
         *  - UNSUPPORTED_ERROR, if the resulting matrix would be too large.
         *  - CIRCULAR_ERROR, if the operand results in a circular reference.
         *  - The #VALUE! error code, if the reference contains more than one
         *      cell range.
         */
        this.getMatrix = function () {

            var // the single validated range structure
                range = ref.getValueRange(refSheet, refAddress),
                // size of the matrix
                rows = SheetUtils.getRowCount(range),
                cols = SheetUtils.getColCount(range),
                // the sheet model containing the cell
                sheetModel = docModel.getSheetModel(range.sheet1),
                // the resulting matrix
                matrix = null;

            // safety check: #REF! error if passed sheet index is invalid
            if (!sheetModel) { throw ErrorCodes.REF; }

            // restrict matrix size
            if (!isValidMatrixSize(rows, cols)) {
                throw TokenUtils.UNSUPPORTED_ERROR;
            }

            // build an empty matrix
            matrix = new Matrix(rows, cols, null);

            // fill existing cell values
            sheetModel.getCellCollection().iterateCellsInRanges(range, function (cellData) {
                matrix.setElement(cellData.address[1] - range.start[1], cellData.address[0] - range.start[0], cellData.result);
            }, { type: 'content', hidden: 'all' });

            return matrix;
        };

        /**
         * Returns the unresolved cell reference contained in this operand.
         *
         * @returns {Reference}
         *  The unresolved reference contained in this operand.
         */
        this.getReference = function () {
            return ref;
        };

        /**
         * Returns the string representation of the operand for debug logging.
         */
        this.toString = function () {
            return ref.toString();
        };

    } // ReferenceOperandMixin

    // class Operand ==========================================================

    /**
     * Represents a single operand stored in the operand stack of a formula
     * token interpreter.
     *
     * @constructor
     *
     * @param {FormulaContext} context
     *  The context object providing a connection to the document model.
     *
     * @param {Number} refSheet
     *  The zero-based index of the reference sheet.
     *
     * @param {Number[]} refAddress
     *  The address of the reference cell.
     *
     * @param {Number|Date|String|Boolean|Null|ErrorCode|Matrix|Reference} value
     *  The value to be stored as operand. Can be a constant value (numbers,
     *  strings, Boolean values), a Date object (will be converted to a
     *  floating-point number immediately), an error code literal (instance of
     *  the class ErrorCode), a matrix literal (instance of the class Matrix),
     *  or an unresolved reference (instance of the class Reference).
     */
    function Operand(context, refSheet, refAddress, value) {

        // public methods -----------------------------------------------------

        /**
         * Returns whether this operand contains a single value (type 'val').
         *
         * @returns {Boolean}
         *  Whether this operand contains a single value (type 'val').
         */
        this.isValue = function () {
            return this.getType() === 'val';
        };

        /**
         * Returns whether this operand contains a matrix (type 'mat').
         *
         * @returns {Boolean}
         *  Whether this operand contains a constant matrix (type 'mat').
         */
        this.isMatrix = function () {
            return this.getType() === 'mat';
        };

        /**
         * Returns whether this operand contains a reference (type 'ref').
         *
         * @returns {Boolean}
         *  Whether this operand contains an unresolved reference (type 'ref').
         */
        this.isReference = function () {
            return this.getType() === 'ref';
        };

        /**
         * Returns whether this operand is empty (e.g. the second operand in
         * the formula =SUM(1,,2) is empty).
         *
         * @returns {Boolean}
         *  Whether this operand is empty.
         */
        this.isEmpty = function () {
            return this.isValue() && _.isNull(this.getValue());
        };

        // initialization -----------------------------------------------------

        // instance of Reference passed: validate reference
        if (value instanceof Reference) {
            if (value.getLength() === 0) {
                value = ErrorCodes.NULL;
            } else if (value.getLength() > TokenUtils.MAX_REF_LIST_SIZE) {
                value = TokenUtils.UNSUPPORTED_ERROR;
            } else {
                ReferenceOperandMixin.call(this, context, refSheet, refAddress, value);
                return;
            }
        }

        // matrix passed: validate size
        if (value instanceof Matrix) {
            if (!isValidMatrixSize(value.getRowCount(), value.getColCount())) {
                value = TokenUtils.UNSUPPORTED_ERROR;
            } else {
                MatrixOperandMixin.call(this, context, value);
                return;
            }
        }

        // otherwise: simple value (as last check, preceding code may have
        // changed the passed value to an error code)
        ValueOperandMixin.call(this, context, value);

    } // class Operand

    // class OperandsMixin ====================================================

    /**
     * A mix-in class for an instance of FormulaContext. Adds more methods
     * providing access to the existing operands in a function call.
     *
     * @constructor
     *
     * @extends FormulaContext
     *
     * @param {Array} operands
     *  The original operands processed by the current operator or function.
     */
    function OperandsMixin(operands) {

        // public methods -----------------------------------------------------

        /**
         * Returns the number of existing operands passed to the processed
         * operator or function.
         *
         * @returns {Number}
         *  The number of existing operands.
         */
        this.getOperandCount = function () {
            return operands.length;
        };

        /**
         * Returns the specified operand passed to the processed operator or
         * function.
         *
         * @param {Number} index
         *  The zero-based index of an operand.
         *
         * @returns {Operand}
         *  The specified operand, if existing.
         *
         * @throws {ErrorCode}
         *  The #VALUE! error code, if the operand does not exist.
         */
        this.getOperand = function (index) {
            if ((0 <= index) && (index < operands.length)) { return operands[index]; }
            throw ErrorCodes.VALUE;
        };

        /**
         * Returns the specified operands passed to the processed operator or
         * function as array.
         *
         * @param {Number} index
         *  The zero-based index of the first operand to be returned.
         *
         * @returns {Array}
         *  The specified operand and all its successors, if existing.
         *
         * @throws {ErrorCode}
         *  The #VALUE! error code, if the specified operand does not exist.
         */
        this.getOperands = function (index) {
            if ((0 <= index) && (index < operands.length)) { return operands.slice(index); }
            throw ErrorCodes.VALUE;
        };

        /**
         * Returns whether the specified operand is missing (the optional
         * parameters at the end of a function; e.g. the third parameter in the
         * formula =SUM(1,2) is missing).
         *
         * @param {Number} index
         *  The zero-based index of an operand.
         *
         * @returns {Boolean}
         *  Whether the specified operand is missing. Returns false for
         *  existing but empty operands, e.g. in the formula =SUM(1,,2).
         */
        this.isMissingOperand = function (index) {
            return index >= operands.length;
        };

        /**
         * Returns whether the specified operand exists, but is empty (e.g. the
         * second operand in the formula =SUM(1,,2) is empty).
         *
         * @param {Number} index
         *  The zero-based index of an operand.
         *
         * @returns {Boolean}
         *  Whether the specified operand exists but is empty. Returns false
         *  for missing operands (optional parameters at the end of the
         *  function). Returns false for references to empty cells too.
         */
        this.isEmptyOperand = function (index) {
            var operand = operands[index];
            return _.isObject(operand) && operand.isEmpty();
        };

        /**
         * Returns whether the specified operand is missing (the optional
         * parameters at the end of a function; e.g. the third parameter in the
         * formula =SUM(1,2) is missing), or empty (e.g. the second operand in
         * the formula =SUM(1,,2) is empty).
         *
         * @param {Number} index
         *  The zero-based index of an operand.
         *
         * @returns {Boolean}
         *  Whether the specified operand is missing or empty.
         */
        this.isMissingOrEmptyOperand = function (index) {
            return this.isMissingOperand(index) || this.isEmptyOperand(index);
        };

        /**
         * Visits the existing operands of the processed operator or function.
         *
         * @param {Number} start
         *  Specifies the zero-based index of the first operand to be visited.
         *
         * @param {Function} iterator
         *  The iterator callback function that will be invoked for each
         *  operand. Receives the following parameters:
         *  (1) {Operand} operand
         *      The original operand instance.
         *  (2) {Number} index
         *      The zero-based index of the operand.
         *  The iterator will be called in the context of this instance.
         *
         * @returns {OperandsMixin}
         *  A reference to this instance.
         */
        this.iterateOperands = function (start, iterator) {
            _.each(operands.slice(start), function (operand, index) {
                iterator.call(this, operand, start + index);
            }, this);
            return this;
        };

    } // class OperandsMixin

    // class TokenResolver ====================================================

    /**
     * Resolves a compiled formula token array to the result of the formula.
     * This implementation has been moved outside the Interpreter class to be
     * able to recursively resolve multiple formulas (e.g. defined names).
     *
     * @constructor
     *
     * @param {SpreadsheetApplication} app
     *  The spreadsheet application instance.
     *
     * @param {Tokenizer} tokenizer
     *  The tokenizer instance needed for specific conversion tasks.
     *
     * @param {Number} refSheet
     *  The index of the sheet the interpreted formula is related to. Used to
     *  resolve reference tokens without sheet reference.
     *
     * @param {Number[]} refAddress
     *  The address of the reference cell in the passed sheet the formula
     *  is related to. Used to resolve relative references in defined names
     *  used by the interpreted formula.
     *
     * @param {Array} compilerTokens
     *  An array of token descriptors, in prefix notation, as returned by the
     *  method Compiler.compileTokens().
     *
     * @param {Object} [initOptions]
     *  Optional parameters:
     *  @param {Boolean} [initOptions.relocate=false]
     *      If set to true, all relative references contained in the passed
     *      token array will be interpreted with relocated position. The
     *      address contained in the parameter 'refAddress' will be used as
     *      relocation distance. Used when interpreting defined names that
     *      always contain cell references relative to cell position A1.
     */
    function TokenResolver(app, tokenizer, refSheet, refAddress, compilerTokens, initOptions) {

        var // the calling context for the operator resolver callback function
            context = new FormulaContext(app, tokenizer, refSheet, refAddress),

            // all operators/functions of the current file format
            descriptors = Operators.getDescriptors(app.getFileFormat()),

            // whether to relocate reference tokens with relative address components
            relocate = Utils.getBooleanOption(initOptions, 'relocate', false),

            // current array index into the compiler token array
            tokenIndex = 0,

            // start time of formula evaluation
            t0 = 0;

        // private methods ----------------------------------------------------

        /**
         * Resolves the passed token array of a defined name to its result.
         *
         * @param {String} contextType
         *  The context type that influences how to resolve values, matrixes,
         *  and references. See methods FormulaInterpreter.interpretTokens()
         *  for details.
         *
         * @param {TokenArray} [tokenArray]
         *  The token array received from the token of a defined name. If
         *  missing (unknown name in a formula), this method returns the #NAME?
         *  error code.
         *
         * @returns {Any}
         *  The result value of the token array (may be values, matrixes, or
         *  references).
         */
        function resolveDefinedName(contextType, tokenArray) {

            var // the interpreter result object
                result = null;

            // unknown names result in the #NAME? error code
            if (!tokenArray) { return ErrorCodes.NAME; }

            // resolve the formula in the defined name (relative to the reference address)
            result = tokenArray.interpretFormula(contextType, refSheet, refAddress, { relocate: true });

            // return UNSUPPORTED_ERROR on any error (TODO: evaluate error/warning code?)
            return (result.type === 'result') ? result.value : TokenUtils.UNSUPPORTED_ERROR;
        }

        /**
         * Checks that the passed return type, and the context type of the
         * formula or subexpression match. Throws a 'reference' error, if the
         * context type is 'ref', and the passed type is different from 'ref'
         * or 'any'. Excel rejects such formulas, and even announces a broken
         * file format if such formulas are written into the file. Examples of
         * such invalid formulas are:
         *  =A1:1 (combined number with range operator)
         *  =OFFSET(1,1,1) (passed number to a reference-only parameter)
         *
         * It is possible to use operands and functions with return type
         * 'any' in reference context, e.g.
         *  =A1:name (return type of defined names is 'any')
         *  =A1:IF(TRUE,A1,B1) (return type of IF is 'any').
         * If the defined name or function returns a non-reference value, the
         * formula simply results in the #VALUE! error.
         *
         * @param {String} type
         *  The return type of an operator or function, or the type of a single
         *  operand to be checked. Must be one of:
         *  - 'val': A single value (numbers, strings, Boolean values, or error
         *      codes),
         *  - 'mat': Constant matrixes (matrix literals, or functions returning
         *      matrixes),
         *  - 'ref': Cell references (literal references, reference operators,
         *      functions returning a reference, or the literal #REF! error
         *      code),
         *  - 'any': Any of the above (defined names, a few functions returning
         *      any data type).
         *
         * @param {String} contextType
         *  The context type that influences how to resolve values, matrixes,
         *  and references. See method FormulaInterpreter.interpretTokens() for
         *  details. If the context type is 'ref', only the return types 'ref'
         *  and 'any' will be accepted. All other return types will cause to
         *  throw an internal 'reference' exception that marks the formula
         *  structure to be ill-formed (see description above).
         *
         * @throws {String}
         *  The special error code 'reference', if the passed return type is
         *  not supported in reference context.
         */
        function checkContextType(type, contextType) {
            // context type 'ref': accept 'ref' and 'any' only
            if ((contextType === 'ref') && (type !== 'ref') && (type !== 'any')) {
                throw 'reference';
            }
        }

        /**
         * Creates an instance of the Operand class for the passed value.
         */
        function makeOperand(value) {
            return new Operand(context, refSheet, refAddress, value);
        }

        /**
         * Creates an operand object for the passed parser token.
         *
         * @param {Object} compilerToken
         *  A compiler token representing an operand (literal values, constant
         *  matrixes, cell references, defined names).
         *
         * @param {String} contextType
         *  The context type that influences how to resolve values, matrixes,
         *  and references. See methods FormulaInterpreter.interpretTokens()
         *  for details.
         *
         * @returns {Operand}
         *  An operand instance representing the passed token.
         *
         * @throws {ErrorCode}
         *  The special error code UNSUPPORTED_ERROR, if the type of the passed
         *  token is not supported.
         */
        function makeTokenOperand(compilerToken, contextType) {

            var // the parser token representing the operand value
                token = compilerToken.token,
                // sheet range and cell range of a reference token
                sheetRange = null, range = null,
                // the token type
                type = null,
                // the value to be inserted into the operand
                value = null;

            TokenUtils.log('> resolving operand token: ' + token);

            switch (token.getType()) {
            case 'lit': // literal value
                value = token.getValue();
                type = ErrorCodes.REF.equals(value) ? 'ref' : 'val';
                break;
            case 'mat': // matrix literals
                value = token.getMatrix();
                type = 'mat';
                break;
            case 'ref': // cell range references
                sheetRange = token.getSheetRange();
                range = relocate ? token.getRelocatedRange([0, 0], refAddress, true) : token.getRange();
                value = (sheetRange && range) ? new Reference(_.extend({}, sheetRange, range)) : ErrorCodes.REF;
                type = 'ref';
                break;
            case 'name': // defined names
                value = resolveDefinedName(contextType, token.getTokenArray());
                type = 'any';
                break;
            default:
                fatalError('TokenResolver.makeTokenOperand(): unknown token type: "' + token.getType() + '"');
            }

            // check context type (will throw 'reference' error)
            checkContextType(type, contextType);
            return makeOperand(value);
        }

        /**
         * Processes the passed operator or function call. Calculates the
         * required number of operands from the following compiler tokens, and
         * calculates the result of the operator or function.
         *
         * @param {Object} compilerToken
         *  A compiler token representing an operator or function.
         *
         * @param {String} contextType
         *  The context type that influences how to resolve values, matrixes,
         *  and references. See methods FormulaInterpreter.interpretTokens()
         *  for details.
         *
         * @returns {Operand}
         *  An operand instance representing the result of the operator or
         *  function call.
         *
         * @throws {String}
         *  The error code 'missing', if there are not enough tokens available
         *  in the compiled token array.
         */
        function processOperator(compilerToken, contextType) {

            var // the parser token representing the operator or function
                token = compilerToken.token,
                // the operator/function specification
                descriptor = null,
                // minimum/maximum number of parameters supported by the operator
                minParams = 0,
                maxParams = 0,
                // length of repeated parameter sequences
                repeatParams = 0,
                // the expanded type signature
                signature = null,
                // the operands required by the operator or function
                operands = null,
                // matrix size for repeated processing of value operators
                repeatRows = 0,
                repeatCols = 0,
                // the resulting operand
                result = null;

            TokenUtils.log('> processing operator ' + token + ' with ' + compilerToken.params + ' parameters');

            // returns the type of the specified parameter from the signature
            function getParamType(index) {
                if (descriptor.signature.length <= index) {
                    var repeat = _.isNumber(descriptor.repeatCount) ? descriptor.repeatCount : 1;
                    index = descriptor.signature.length - repeat + ((index - descriptor.signature.length) % repeat);
                }
                return descriptor.signature[index];
            }

            // convert operand according to the expected type from the signature
            function resolveParam(operand, paramType, row, col) {
                switch (paramType) {
                case 'any':
                    return operand;
                case 'val':
                    return checkValueForErrorCode(operand.getValue(row, col));
                case 'val:num':
                    return context.convertToNumber(operand.getValue(row, col));
                case 'val:int':
                    return Math.floor(context.convertToNumber(operand.getValue(row, col)));
                case 'val:date':
                    return context.convertToDate(operand.getValue(row, col));
                case 'val:str':
                    return context.convertToString(operand.getValue(row, col));
                case 'val:bool':
                    return context.convertToBoolean(operand.getValue(row, col));
                case 'val:comp':
                    return context.convertToComplex(operand.getValue(row, col));
                case 'val:any':
                    return operand.getValue(row, col);
                case 'mat':
                    return checkMatrixForErrorCode(operand.getMatrix());
                case 'mat:num':
                    // no conversion from other data types in matrix parameters!
                    return checkMatrixForDataType(operand.getMatrix(), _.isNumber);
                case 'mat:str':
                    // no conversion from other data types in matrix parameters!
                    return checkMatrixForDataType(operand.getMatrix(), _.isString);
                case 'mat:bool':
                    // no conversion from other data types in matrix parameters!
                    return checkMatrixForDataType(operand.getMatrix(), _.isBoolean);
                case 'mat:any':
                    return operand.getMatrix();
                case 'ref':
                    return operand.getReference();
                }
                fatalError('TokenResolver.resolveParam(): unknown parameter type: "' + paramType + '"');
            }

            // resolves the operands according to the type signature
            function resolveOperator(row, col) {

                var // the parameter values to be passed to the operator resolver callback
                    params = null,
                    // the resulting operand
                    result = null;

                try {
                    // resolves the operand values according to the signature
                    params = _.map(operands, function (operand, index) {
                        return resolveParam(operand, signature[index].paramType, row, col);
                    });

                    // invoke the resolver callback function of the operator
                    TokenUtils.logTokens('  resolving with parameters', params);
                    result = descriptor.resolve.apply(context, params);

                    // compare result with the specified return type
                    switch (descriptor.type) {
                    case 'val':
                    case 'mat':
                    case 'ref':
                        if (!isValueOfType(result, descriptor.type)) {
                            fatalError('TokenResolver.resolveOperator(): operator result type does not match operator return type');
                        }
                        break;
                    case 'any':
                        if (!isValueOfType(result, 'val') && !isValueOfType(result, 'mat') && !isValueOfType(result, 'ref') && !(result instanceof Operand)) {
                            fatalError('TokenResolver.resolveOperator(): missing operator result');
                        }
                        break;
                    }

                } catch (error) {
                    if (SheetUtils.isErrorCode(error)) {
                        // operator may throw literal error codes
                        result = error;
                    } else {
                        // re-throw other error codes, or internal JavaScript exceptions
                        throw error;
                    }
                }

                // check elapsed time, return with UNSUPPORTED_ERROR if formula is too complex
                if (_.now() - t0 > MAX_EVAL_TIME) {
                    result = TokenUtils.UNSUPPORTED_ERROR;
                }

                return result;
            }

            // resolve token to operator or function descriptor
            if (token.isType('op')) {
                descriptor = descriptors.OPERATORS[token.getText()];
            } else if (token.isType('func') && token.isSheetFunction()) {
                descriptor = descriptors.FUNCTIONS[token.getValue().toUpperCase()];
            }

            // unknown function (external/macro function calls)
            if (!descriptor) {
                return makeOperand(ErrorCodes.NAME);
            }

            // check return type of the operator or function
            if (!/^(val|mat|ref|any)$/.test(descriptor.type)) {
                fatalError('TokenResolver.processOperator(): invalid return type: "' + descriptor.type + '"');
            }

            // check context type (will throw 'reference' error)
            checkContextType(descriptor.type, contextType);

            // get minimum number of parameters
            if (_.isNumber(descriptor.minParams) && (0 <= descriptor.minParams)) {
                minParams = descriptor.minParams;
            } else {
                fatalError('TokenResolver.processOperator(): invalid value for minParams: ' + descriptor.minParams);
            }

            // get maximum number of parameters, and length of repeated sequences
            if (!('maxParams' in descriptor)) {
                repeatParams = _.isNumber(descriptor.repeatParams) ? descriptor.repeatParams : 1;
                if ((repeatParams < 1) || (minParams < repeatParams)) {
                    fatalError('TokenResolver.processOperator(): invalid value for repeatParams: ' + descriptor.repeatParams);
                }
                maxParams = minParams + Utils.roundDown(TokenUtils.MAX_PARAM_COUNT - minParams, repeatParams);
            } else if (_.isNumber(descriptor.maxParams) && (minParams <= descriptor.maxParams)) {
                maxParams = descriptor.maxParams;
                repeatParams = 1;
            } else {
                fatalError('TokenResolver.processOperator(): invalid value for maxParams: ' + descriptor.maxParams);
            }

            // check operand count
            if (compilerToken.params < minParams) { throw 'missing'; }
            if (compilerToken.params > maxParams) { throw 'unexpected'; }

            // repeated parameter sequences must be complete
            if (((compilerToken.params - minParams) % repeatParams) !== 0) { throw 'missing'; }

            // check missing implementation
            if (!_.isFunction(descriptor.resolve)) {
                TokenUtils.warn('TokenResolver.processOperator(): unsupported built-in function "' + token.getText() + '"');
                return makeOperand(TokenUtils.UNSUPPORTED_ERROR);
            }

            // check operator signature
            if ((compilerToken.params > 0) && (descriptor.signature.length === 0)) {
                fatalError('TokenResolver.processOperator(): missing operator signature');
            }

            // build the complete type signature
            signature = _.times(compilerToken.params, function (index) {

                var // get operand type from signature in the descriptor
                    paramType = getParamType(index),
                    // the base parameter type used as context type for the parameter
                    baseType = paramType.split(':')[0];

                // return type signature entry
                return { paramType: paramType, baseType: baseType };
            });

            // build the array of operands
            operands = _.map(signature, function (typeData) {

                var // the resulting context type for the operand
                    opContextType = typeData.baseType;

                // if context type for this operator is matrix, and the current parameter is of type value,
                // and this operator returns a value, pass the matrix context type through to the operand
                // (resolve nested cell references to matrixes instead of single values)
                if ((contextType === 'mat') && (typeData.baseType === 'val') && (descriptor.type === 'val')) {
                    opContextType = 'mat';
                }

                return getNextOperand(opContextType);
            });

            // special handling for all operands of operators returning values
            if (descriptor.type === 'val') {
                _.each(operands, function (operand, index) {

                    // convert or preprocess operands, if operator expects a value type
                    if (signature[index].baseType === 'val') {

                        // If outer operator context is matrix, convert cell references to matrixes.
                        // Example: In the formula =MMULT(A1:B2;ABS(A1:B2)), the function MMULT passes
                        // context type matrix to its operands which causes ABS to resolve the cell
                        // reference as matrix instead of value.
                        if ((contextType === 'mat') && operand.isReference()) {
                            operand = operands[index] = makeOperand(operand.getMatrix());
                        }

                        // Check for matrixes in value operands. Resolve operator repeatedly to a matrix,
                        // if it returns values. Example: The plus operator in the formula =SUM({1;2|3;4}+1)
                        // will convert the number 1 to the matrix {1;1|1;1}.
                        if (operand.isMatrix()) {
                            var matrix = operand.getMatrix();
                            repeatRows = Math.max(repeatRows, matrix.getRowCount());
                            repeatCols = Math.max(repeatCols, matrix.getColCount());
                        }
                    }
                });
            }

            // extend the calling context for the operator resolver with the operands
            OperandsMixin.call(context, operands);

            // resolve as matrix, if any parameter with value signature type is a matrix
            if (repeatRows * repeatCols > 0) {

                // restrict to supported matrix size
                if (!isValidMatrixSize(repeatRows, repeatCols)) {
                    return makeOperand(TokenUtils.UNSUPPORTED_ERROR);
                }

                // build a matrix as result value
                TokenUtils.info('  process as ' + repeatRows + 'x' + repeatCols + ' matrix');
                result = new Matrix(repeatRows, repeatCols, resolveOperator);

            } else {
                result = resolveOperator();
            }

            // return result as operand instance
            return (result instanceof Operand) ? result : makeOperand(result);
        }

        /**
         * Returns the result of the next compiler token. If it is an operand,
         * it will be returned; if it is an operator or function call, its
         * result will be calculated and returned. If the operands of the
         * operator or functions are also operators or functions, their results
         * will be calculated recursively.
         *
         * @param {String} contextType
         *  The context type that influences how to resolve values, matrixes,
         *  and references. See methods FormulaInterpreter.interpretTokens()
         *  for details.
         *
         * @returns {Operand}
         *  The result of the next compiler token.
         */
        function getNextOperand(contextType) {

            var // the compiler token top be processed
                compilerToken = compilerTokens[tokenIndex],
                // the resulting operand
                result = null;

            // double-check the token (compiler should not create invalid token arrays)
            if (!compilerToken) {
                fatalError('TokenResolver.getNextOperand(): missing compiler token');
            }

            // move to next unprocessed token
            tokenIndex += 1;

            // process token by type
            switch (compilerToken.type) {
            case 'val':
                // return operands directly
                result = makeTokenOperand(compilerToken, contextType);
                break;
            case 'op':
                // process operators recursively
                result = processOperator(compilerToken, contextType);
                break;
            default:
                fatalError('TokenResolver.getNextOperand(): unknown token type: "' + compilerToken.type + '"');
            }

            TokenUtils.log('< result of ' + compilerToken + ': ' + result);
            return result;
        }

        // public methods -----------------------------------------------------

        /**
         * Calculates the result of the formula represented by the token array
         * passed in the constructor.
         *
         * @param {String} contextType
         *  The context type that influences how to resolve values, matrixes,
         *  and references. See method FormulaInterpreter.interpretTokens()
         *  for details.
         *
         * @returns {Operand}
         *  The result of the formula, as unresolved operand object.
         */
        this.getResult = function (contextType) {
            t0 = _.now();
            return getNextOperand(contextType);
        };

    } // class TokenResolver

    // class Interpreter ======================================================

    /**
     * Calculates the result for an array of token descriptors that has been
     * compiled to prefix notation (a.k.a. Polish Notation).
     *
     * @constructor
     *
     * @extends ModelObject
     *
     * @param {SpreadsheetApplication} app
     *  The spreadsheet application instance.
     */
    function Interpreter(app) {

        // base constructor ---------------------------------------------------

        ModelObject.call(this, app);

        // public methods -----------------------------------------------------

        /**
         * Calculates the result of the formula represented by the passed
         * compiled token array.
         *
         * @param {String} contextType
         *  The context type that influences how to resolve values, matrixes,
         *  and references. The following context types are supported:
         *  - 'val': A single value is expected, e.g. in simple cell formulas,
         *      or operators and functions working with simple values (plus
         *      operator, ABS).
         *  - 'mat': A constant matrix of values is expected, e.g. in matrix
         *      formulas, or functions working on entire matrixes (MMULT).
         *  - 'ref': An unresolved cell reference is expected, e.g. reference
         *      operators (list, intersection, range), or functions calculating
         *      with range addresses instead of the cell contents (OFFSET).
         *  - 'any': Accepts any of the types mentioned above, with minimum
         *      conversion, e.g. the result of defined names, functions passing
         *      one of their original parameters through (IF, CHOOSE), or
         *      functions that iterate over available values in matrixes and
         *      cell references (SUM, PRODUCT).
         *
         * @param {Array} compilerTokens
         *  An array of token descriptors, in prefix notation, as returned by
         *  the method Compiler.compileTokens().
         *
         * @param {Tokenizer} tokenizer
         *  The tokenizer instance needed for specific conversion tasks.
         *
         * @param {Number} refSheet
         *  The index of the sheet the interpreted formula is related to.
         *
         * @param {Number[]} refAddress
         *  The address of the reference cell in the passed sheet the formula
         *  is related to.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.relocate=false]
         *      If set to true, all relative references contained in the passed
         *      token array will be interpreted with relocated position. The
         *      address contained in the parameter 'refAddress' will be used as
         *      relocation distance. Used when interpreting defined names that
         *      always contain cell references relative to cell position A1.
         *
         * @returns {Object}
         *  The result descriptor of the formula interpreter, with the
         *  following properties:
         *  - {String} type
         *      The result type:
         *      - 'result': A valid result has been calculated. The property
         *          'value' contains the formula result.
         *      - 'warn': The formula result has not been calculated, but the
         *          formula structure is valid. The property 'value' specifies
         *          the warning: 'circular' for circular references, or
         *          'unsupported', if an unsupported feature was found in the
         *          formula, or 'internal', if any internal error has occurred.
         *      - 'error': The formula structure is invalid. The property
         *          'value' contains an identifier for the reason: 'missing',
         *          if something is missing, e.g. a function was called with
         *          less arguments than required; or 'unexpected' if something
         *          was not expected, e.g. a function was called with too many
         *          arguments.
         *  - {Any} value
         *      The result of the formula, or an error code, if calculation has
         *      failed (see property 'type' above).
         */
        this.interpretTokens = function (contextType, compilerTokens, tokenizer, refSheet, refAddress, options) {

            var // the implementation
                resolver = new TokenResolver(app, tokenizer, refSheet, refAddress, compilerTokens, options),
                // the resulting operand
                result = null;

            try {

                // calculate the formula, pull formula result operand
                TokenUtils.takeTime('Interpreter.interpretTokens(): contextType=' + contextType + ' refSheet=' + refSheet + ' refAddress=' + SheetUtils.getCellName(refAddress), function () {
                    result = resolver.getResult(contextType);
                });

                // 'any' context: resolve to actual type of the resulting operand
                contextType = (contextType === 'any') ? result.getType() : contextType;

                // resolve result operand to result value according to context type
                switch (contextType) {
                case 'val':
                    result = result.getValue();
                    // in value context, empty values (reference to empty cell) become zero
                    if (_.isNull(result)) { result = 0; }
                    break;
                case 'mat':
                    result = result.getMatrix();
                    break;
                case 'ref':
                    result = result.getReference();
                    break;
                default:
                    fatalError('FormulaInterpreter.interpretTokens(): unknown context type: "' + contextType + '"');
                }

                // throw error code literal (special handling for internal errors in catch clause)
                if (SheetUtils.isErrorCode(result)) { throw result; }

                // convert to valid result object
                result = { type: 'result', value: result };

            } catch (error) {

                // operator may throw literal error codes
                if (SheetUtils.isErrorCode(error)) {

                    // 'unsupported' state or circular references: return empty result with warning property
                    result = _.isString(error.internal) ?
                        { type: 'warn', value: error.internal } :
                        { type: 'result', value: error };
                }

                // 'fatal' state for internal runtime errors: do not return error (this would state an
                // invalid formula structure), but return empty result
                else if (error === 'fatal') {
                    result = { type: 'warn', value: 'internal' };
                }

                // other internal errors (must be strings): invalid formula structure
                else if (_.isString(error)) {
                    TokenUtils.warn('  error: ' + error);
                    result = { type: 'error', value: error };
                }

                // re-throw internal JS exceptions
                else {
                    throw error;
                }
            }

            TokenUtils.info('result: type=' + result.type + ', value=' + TokenUtils.valueToString(result.value));
            return result;
        };

        // initialization -----------------------------------------------------

        // destroy all class members on destruction
        this.registerDestructor(function () {
            app = null;
        });

    } // class Interpreter

    // exports ================================================================

    // derive this class from class ModelObject
    return ModelObject.extend({ constructor: Interpreter });

});
