/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/cellattributesmodel', [
    'io.ox/office/tk/utils',
    'io.ox/office/editframework/utils/border',
    'io.ox/office/editframework/utils/attributeutils',
    'io.ox/office/editframework/model/attributedmodel'
], function (Utils, Border, AttributeUtils, AttributedModel) {

    'use strict';

    // class CellAttributesModel ==============================================

    /**
     * Base class for objects containing cell attributes. Provides a
     * specialized implementation of the method 'setAttributes()' that is able
     * to update single border properties of visible border attributes.
     *
     * @constructor
     *
     * @extends AttributedModel
     *
     * @param {SpreadsheetApplication} app
     *  The application instance containing this cell model object.
     *
     * @param {Object} [initAttributes]
     *  An attribute set with initial formatting attributes for the cell.
     *
     * @param {Object} [initOptions]
     *  Optional parameters. Supports all options also supported by the base
     *  class AttributedModel. The following options will be set to fixed
     *  values and cannot be changed:
     *  - initOptions.silent will be set to true (no change events).
     *  - initOptions.styleFamily will be set to 'cell'.
     */
    function CellAttributesModel(app, initAttributes, initOptions) {

        // base constructor ---------------------------------------------------

        // do not trigger any events (collection entries are potential mass objects)
        AttributedModel.call(this, app, initAttributes, Utils.extendOptions(initOptions, {
            trigger: 'never',
            styleFamily: 'cell',
            changeStyleHandler: changeStyleHandler
        }));

        // private methods -----------------------------------------------------

        /**
         * Callback function invoked when the cell style reference of this
         * instance has been changed. Handles explicit formatting attributes to
         * be cleared according to the apply flags of the cell style sheet.
         */
        function changeStyleHandler(explicitAttributeSet, styleAttributeSet) {

            var // the global document style container
                documentStyles = app.getModel().getDocumentStyles();

            // add null attributes according to the 'apply' property in the attribute definitions
            _.each(styleAttributeSet, function (styleAttrs, family) {
                if (family !== 'apply') {
                    _.each(documentStyles.getAttributeDefinitions(family), function (definition, name) {
                        if (definition.apply && styleAttributeSet.apply[definition.apply]) {
                            AttributeUtils.insertAttribute(explicitAttributeSet, family, name, null, true);
                        }
                    });
                }
            });
        }

        // public methods -----------------------------------------------------

        /**
         * Changes and/or removes specific explicit formatting attributes, or
         * the style sheet reference of this model object. In addition to the
         * base class method 'AttributedModel.setAttributes()', this method
         * adds special treatment for border attributes (see options below).
         *
         * @param {Object} attributeSet
         *  An (incomplete) attribute set.
         *
         * @param {Object} [options]
         *  Optional parameters. Additionally to the options supported by the
         *  method AttributedModel.setAttributes(), the following options are
         *  supported:
         *  @param {Boolean} [options.innerLeft=false]
         *      If set to true, the border attribute 'borderInsideVert' will be
         *      used for the left border of the cell, instead of the attribute
         *      'borderLeft'.
         *  @param {Boolean} [options.innerRight=false]
         *      If set to true, the border attribute 'borderInsideVert' will be
         *      used for the right border of the cell, instead of the attribute
         *      'borderRight'.
         *  @param {Boolean} [options.innerTop=false]
         *      If set to true, the border attribute 'borderInsideHor' will be
         *      used for the top border of the cell, instead of the attribute
         *      'borderTop'.
         *  @param {Boolean} [options.innerBottom=false]
         *      If set to true, the border attribute 'borderInsideHor' will be
         *      used for the bottom border of the cell, instead of the
         *      attribute 'borderBottom'.
         *  @param {Boolean} [options.visibleBorders=false]
         *      If set to true, border attributes may be incomplete. Specified
         *      border properties (line style, line width, line color) will be
         *      applied for visible borders only.
         *
         * @returns {Boolean}
         *  Whether any attributes have actually been changed.
         */
        this.setCellAttributes = function (attributeSet, options) {

            var // the current merged cell attributes
                mergedCellAttributes = this.getMergedAttributes().cell;

            // copies the inner border attribute to an outer border attribute
            function copyBorderAttribute(cellAttributes, optionName, innerBorderName, outerBorderName) {
                if (Utils.getBooleanOption(options, optionName, false)) {
                    if (innerBorderName in cellAttributes) {
                        cellAttributes[outerBorderName] = cellAttributes[innerBorderName];
                    } else {
                        delete cellAttributes[outerBorderName];
                    }
                }
            }

            // updates a visible border attribute with the passed incomplete border (in 'visibleBorders' mode)
            function updateVisibleBorderAttribute(cellAttributes, borderName) {
                if (_.isObject(cellAttributes[borderName]) && Border.isVisibleBorder(mergedCellAttributes[borderName])) {
                    cellAttributes[borderName] = _.extend({}, mergedCellAttributes[borderName], cellAttributes[borderName]);
                    if (!Border.isVisibleBorder(cellAttributes[borderName])) {
                        cellAttributes[borderName] = Border.NONE;
                    }
                } else {
                    delete cellAttributes[borderName];
                }
            }

            // process border attributes
            if (_.isObject(attributeSet.cell)) {

                // update the border attributes in a deep copy of the passed attribute set
                attributeSet = _.clone(attributeSet);
                attributeSet.cell = _.copy(attributeSet.cell, true);

                // move inner border attributes to outer border attributes if specified in the options
                copyBorderAttribute(attributeSet.cell, 'innerLeft',   'borderInsideVert', 'borderLeft');
                copyBorderAttribute(attributeSet.cell, 'innerRight',  'borderInsideVert', 'borderRight');
                copyBorderAttribute(attributeSet.cell, 'innerTop',    'borderInsideHor',  'borderTop');
                copyBorderAttribute(attributeSet.cell, 'innerBottom', 'borderInsideHor',  'borderBottom');

                // cells cannot contain inner border attributes
                delete attributeSet.cell.borderInsideVert;
                delete attributeSet.cell.borderInsideHor;

                // update existing borders in 'visibleBorders' mode
                if (Utils.getBooleanOption(options, 'visibleBorders', false)) {
                    updateVisibleBorderAttribute(attributeSet.cell, 'borderLeft');
                    updateVisibleBorderAttribute(attributeSet.cell, 'borderRight');
                    updateVisibleBorderAttribute(attributeSet.cell, 'borderTop');
                    updateVisibleBorderAttribute(attributeSet.cell, 'borderBottom');
                }
            }

            // apply the attributes
            return this.setAttributes(attributeSet, options);
        };

    } // class CellAttributesModel

    // exports ================================================================

    // derive this class from class AttributedModel
    return AttributedModel.extend({ constructor: CellAttributesModel });

});
