/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/preview/view/toppane',
    ['io.ox/office/tk/utils',
     'io.ox/office/tk/forms',
     'io.ox/office/baseframework/view/toolpane',
     'io.ox/office/baseframework/view/toolbar',
     'io.ox/office/preview/view/labels',
     'io.ox/office/preview/view/controls',
     'gettext!io.ox/office/preview'
    ], function (Utils, Forms, ToolPane, ToolBar, Labels, Controls, gt) {

    'use strict';

    // class TopPane ==========================================================

    /**
     * Represents the top-level pane, containing global
     * controls such as the Close, Edit button.
     *
     * The TopPane controls are constructed as if there is only one page available,
     * each other control will be enabled after updating the page count via
     * updatePageCount function.
     *
     * @constructor
     *
     * @extends ToolPane
     *
     * @param {EditApplication} app
     *  The application containing this tool bar.
     */
    function TopPane(app) {

        var pageCount = null,

            leftToolBar = new ToolBar(app),
            centerToolBar = new ToolBar(app),
            rightToolBar = new ToolBar(app),

            // sizeableControls is an array of following object
            // {control: singlePageHidden: on{from:, to: }, off{from:, to:}}
            sizeableControls = [],

            prio1long = { singlePageDisabled: true, on: {from: 1, to: 1},  off: {from: 2, to: 10}},
            prio1short= { singlePageHidden: true,   on: {from: 2, to: 10}, off: {from: 1, to: 1}},
            prio2long = {                           on: {from: 1, to: 2},  off: {from: 3, to: 10}},
            prio2short= {                           on: {from: 3, to: 10}, off: {from: 1, to: 2}},
            prio3long = { singlePageDisabled: true, on: {from: 1, to: 3},  off: {from: 4, to: 10}},
            prio3short= { singlePageHidden: true,   on: {from: 4, to: 10}, off: {from: 1, to: 3}},

            labelMenuShort1 = null,
            labelMenuShort2 = null,
            textFieldShort = null,
            labelMenuLong2 = null,
            textFieldLong = null,

            actionsMenuGroupShort = null,

            viewMenuGroup = null,

            // Controller items in this toppane, that should not be shown in the standalone mode.
            // This is a workaround since the 'visibleKey' behaviour is overriden by functions of this class.
            standaloneHiddenItems = ['document/sendmail', 'document/edit'];

        // base constructor ---------------------------------------------------

        ToolPane.call(this, app, { position: 'top', classes: 'top-pane' });

        // private methods ----------------------------------------------------

        function paneLayoutHandler() {
            if(!pageCount) {
                return;
            }
            for(var prio=1; prio<5; prio++) {
                for(var i = 0; i < sizeableControls.length; i++) {
                    var options = sizeableControls[i];
                    if((pageCount===1)&&options.singlePageHidden) {
                        sizeableControls[i].control.hide();
                    }
                    else if((pageCount===1)&&typeof options.singlePageDisabled==='boolean') {
                        sizeableControls[i].control.enable(!options.singlePageDisabled);
                    }
                    else {
                        if(options.on) {
                            var from = options.on.from,
                                to = options.on.to;
                            if(from&&prio>=from) {
                                if(!to||to>=prio && !options.standaloneHidden) {
                                    options.control.show();
                                }
                            }
                        }
                        if(options.off) {
                            var from = options.off.from,
                                to = options.off.to;
                            if(from&&prio>=from) {
                                if(!to||to>=prio) {
                                    options.control.hide();
                                }
                            }
                        }
                    }
                }
                // check if the size now is sufficient
                var paneWidth = this.getNode().width(),
                    left = leftToolBar.getNode().outerWidth(),
                    center = centerToolBar.getNode().outerWidth(),
                    right = rightToolBar.getNode().outerWidth();
                if(paneWidth>(left+right+center)) {
                    return;
                }
            }
        }

        function createSidePaneControls(group) {
            if(!Modernizr.touch) {
                addGroup(group, null, createSeparatorGroup(), { inline: false }, { singlePageHidden: true });
                addGroup(group, null, createHeaderLabelGroup(Labels.OPTIONS_LABEL), { inline: false }, { singlePageHidden: true });
                addGroup(group, 'view/sidepane', new Controls.CheckBox({ label: gt('Show side panel') }), null, { singlePageHidden: true });
            }
            return group;
        }

        /**
         * @param {Object} destination
         *  the destination object where the control is added
         *
         * @param {String} key
         *  the controller key bound to the control
         *
         * @param {Object} control
         *  the source control that has to be added at the destination
         *
         * @param {Object} [controlOptions]
         *  specifies the options that are to be used when inserting the control into the destination
         *
         * @param {Object} [sizingOptions]
         *  Control sizing options:
         *  {singlePageHidden:... on{from:..., to:... }, off{from:..., to:...}}
         *      ('singlePageHidden', 'on' and 'off' are optional)
         */
        function addGroup(destination, key, control, controlOptions, sizingOptions) {
            if (sizingOptions && sizingOptions.singlePageHidden) {
                control.hide();
            }
            destination.addGroup(key, control, controlOptions);
            if (sizingOptions) {
                var v = {control: control, standaloneHidden: _.contains(standaloneHiddenItems, key)};
                for (var k in sizingOptions) {
                    v[k] = sizingOptions[k];
                }
                sizeableControls.push(v);
            }
        }

        // creating a header label by exchanging class 'label' to 'header-label'
        function createHeaderLabelGroup(label) {
            var labelGroup = new Controls.Label({ label: label, classes: 'header-label' });
            labelGroup.getLabelNode().attr('class', 'header-label');
            return labelGroup;
        }

        function createSeparatorGroup() {
            return new Controls.Group({ classes: 'separator-line' });
        }

        // public methods -----------------------------------------------------

        this.updatePageCount = function(_pageCount) {

            pageCount = _pageCount;
            // the label for the 'Go to page' menu entry
            // the leading and trailing part of the 'Go to page' label
            var goToLabelParts = _.noI18n.fix(
                    //#. an interactive menu entry for the current page number in a document
                    //#. "##" is a placeholder for the current page number (can be modified by the user) - do not change
                    //#. %1$d is the total count of pages in the document
                    //#, c-format
                    gt('Go to page ## of %1$d', _.noI18n(pageCount)).split('##').map(Utils.trimString));

            if(goToLabelParts) {
                if(goToLabelParts[0]) {
                    labelMenuShort1.setLabel(goToLabelParts[0]);
                }
                if(goToLabelParts[1]) {
                    labelMenuShort2.setLabel(goToLabelParts[1]);
                    labelMenuLong2.setLabel(goToLabelParts[1]);
                }
            }
            var validator = new Controls.TextField.NumberValidator({ min: 1, max: pageCount });
            textFieldShort.setValidator(validator);
            textFieldLong.setValidator(validator);
            if(pageCount>1) {
                for(var i = 0; i < sizeableControls.length; i++) {
                    if(sizeableControls[i].singlePageHidden && !sizeableControls[i].standaloneHidden) {
                        sizeableControls[i].control.show();
                    }
                }
            }
        };

        // initialization -----------------------------------------------------

        // the 'View' drop-down menu

        viewMenuGroup = new Controls.CompoundButton(app, { label: Labels.VIEW_LABEL, anchorPadding: -3, classes: 'link-style' })
            .addGroup(null, createHeaderLabelGroup(Labels.ZOOM_LABEL), { inline: false })
            .addGroup('zoom/dec', new Controls.Button(Labels.ZOOMOUT_BUTTON_OPTIONS), { inline: false, sticky: true })
            .addGroup('zoom/inc', new Controls.Button(Labels.ZOOMIN_BUTTON_OPTIONS), { inline: true, sticky: true })
            .addGroup('zoom/percentage', new Controls.PercentageLabel(), { inline: true })
            .addGroup(null, createSeparatorGroup())
            .addGroup('zoom/width', new Controls.CheckBox({ label: Labels.ZOOM_SCREEN_WIDTH_LABEL }))
            .addGroup('zoom/page', new Controls.CheckBox({ label: Labels.ZOOM_SCREEN_SIZE_LABEL }));
        createSidePaneControls(viewMenuGroup);
        addGroup(viewMenuGroup, null, createSeparatorGroup(), { inline: false }, prio3short);
        addGroup(viewMenuGroup, null, createHeaderLabelGroup(gt.pgettext('menu-title', 'Pages')), { inline: false }, prio3short);
        addGroup(viewMenuGroup, 'pages/first', new Controls.Button(Labels.FIRST_PAGE_BUTTON_OPTIONS), { inline: false }, prio3short);
        addGroup(viewMenuGroup, 'pages/previous', new Controls.Button(Labels.PREV_PAGE_BUTTON_OPTIONS), { inline: true, sticky: true }, prio3short);
        addGroup(viewMenuGroup, 'pages/next', new Controls.Button(Labels.NEXT_PAGE_BUTTON_OPTIONS), { inline: true, sticky: true }, prio3short);
        addGroup(viewMenuGroup, 'pages/last', new Controls.Button(Labels.LAST_PAGE_BUTTON_OPTIONS), { inline: true }, prio3short);
        addGroup(viewMenuGroup, null, createSeparatorGroup(), { inline: false }, prio1short);
        addGroup(viewMenuGroup, 'pages/current', labelMenuShort1 = new Controls.Label(), { inline: true }, prio1short);
        addGroup(viewMenuGroup, 'pages/current', textFieldShort = new Controls.PageNumberField(), { inline: true }, prio1short);
        addGroup(viewMenuGroup, 'pages/current', labelMenuShort2 = new Controls.Label(), { inline: true }, prio1short);

        //
        // Left ToolBar
        //
        // create the 'actions' drop-down menu
        actionsMenuGroupShort = new Controls.CompoundButton(app, { label: Labels.ACTIONS_LABEL, anchorPadding: -3, classes: 'link-style' })
            .addGroup('document/download', new Controls.Button({ icon: Labels.DOWNLOAD_ICON, label: Labels.DOWNLOAD_LABEL }))
            .addGroup('document/print', new Controls.Button({ icon: Labels.PRINT_ICON, label: Labels.PRINT_LABEL }))
            .addGroup('document/sendmail', new Controls.Button({ icon: Labels.SEND_MAIL_ICON, label: Labels.SEND_MAIL_LABEL }), { visibleKey: 'app/bundled' })
            .addGroup('document/edit', new Controls.Button({ icon: Labels.EDIT_ICON, label: Labels.EDIT_LABEL }), { visibleKey: 'document/edit' });

        addGroup(leftToolBar, null, actionsMenuGroupShort, null, prio2short);
        addGroup(leftToolBar, 'document/download', new Controls.Button({ icon: Labels.DOWNLOAD_ICON, tooltip: Labels.DOWNLOAD_LABEL, classes: 'link-style' }), null, prio2long);
        addGroup(leftToolBar, 'document/print', new Controls.Button({ icon: Labels.PRINT_ICON, tooltip: Labels.PRINT_LABEL, classes: 'link-style' }), null, prio2long);
        addGroup(leftToolBar, 'document/sendmail', new Controls.Button({ icon: Labels.SEND_MAIL_ICON, tooltip: Labels.SEND_MAIL_LABEL, classes: 'link-style' }), { visibleKey: 'app/bundled' }, prio2long);
        addGroup(leftToolBar, 'document/edit', new Controls.Button({ icon: Labels.EDIT_ICON, tooltip: Labels.EDIT_LABEL, classes: 'link-style' }), { visibleKey : 'document/edit' }, prio2long);

        this.addViewComponent(leftToolBar, { targetArea: 'leading' });

        //
        // center ToolBar
        //
        addGroup(centerToolBar, 'pages/first', new Controls.Button(Utils.extendOptions(Labels.FIRST_PAGE_BUTTON_OPTIONS, { classes: 'link-style' })), null, prio3long);
        addGroup(centerToolBar, 'pages/previous', new Controls.Button(Utils.extendOptions(Labels.PREV_PAGE_BUTTON_OPTIONS, { classes: 'link-style' })), null, prio3long);
        addGroup(centerToolBar, 'pages/current', textFieldLong = new Controls.PageNumberField(), null, prio1long);
        addGroup(centerToolBar, 'pages/current', labelMenuLong2 = new Controls.Label({ classes: 'link-style' }), null, prio1long);
        addGroup(centerToolBar, 'pages/next', new Controls.Button(Utils.extendOptions(Labels.NEXT_PAGE_BUTTON_OPTIONS, { classes: 'link-style' })), null, prio3long);
        addGroup(centerToolBar, 'pages/last', new Controls.Button(Utils.extendOptions(Labels.LAST_PAGE_BUTTON_OPTIONS, { classes: 'link-style' })), null, prio3long);
        this.addViewComponent(centerToolBar, { targetArea: 'center', classes: 'centerToolPane' });

        //
        // right ToolBar
        //
        rightToolBar
            .addGroup(null, viewMenuGroup)
            .addGap()
            .addGroup('app/quit', new Controls.Button(Utils.extendOptions(Labels.QUIT_BUTTON_OPTIONS, { classes: 'link-style' })), { visibleKey : 'app/bundled' });
        this.addViewComponent(rightToolBar, { targetArea: 'trailing' });

        // update view pane after it has been resized, or contents have been changed
        this.on('pane:layout', paneLayoutHandler);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            app = actionsMenuGroupShort = null;
        });
    } // class TopPane

    // exports ================================================================

    // derive this class from class ToolPane
    return ToolPane.extend({ constructor: TopPane });
});
