/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/baseframework/view/component', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/forms',
    'io.ox/office/tk/object/timermixin',
    'io.ox/office/baseframework/model/modelobject'
], function (Utils, Forms, TimerMixin, ModelObject) {

    'use strict';

    // class Component ========================================================

    /**
     * Base class for view components containing instances of Group objects
     * (controls or groups of controls).
     *
     * Instances of this class trigger the following events:
     * - 'component:beforeshow'
     *      Before the view component will be shown.
     * - 'component:show'
     *      After the view component has been shown.
     * - 'component:beforehide'
     *      Before the view component will be hidden.
     * - 'component:hide'
     *      After the view component has been hidden.
     * - 'component:resize'
     *      After the size of the root node has been changed. The event handler
     *      receives the new width and height of the root node, in pixels.
     * - 'component:layout'
     *      After any group in this view component has changed its visibility
     *      state or size, even if the size of the root node did not change.
     * - 'component:focus'
     *      After a control group in this component has been focused, by
     *      initially focusing any of its focusable child nodes.
     * - 'component:blur'
     *      After a control group in this component has lost the browser focus,
     *      after focusing any other DOM node outside the group.
     *
     * @constructor
     *
     * @extends ModelObject
     * @extends TimerMixin
     *
     * @param {BaseApplication} app
     *  The application containing this view component instance.
     *
     * @param {Object} [initOptions]
     *  Optional parameters:
     *  @param {String} [initOptions.classes]
     *      Additional CSS classes that will be set at the root DOM node of
     *      this view component.
     *  @param {Boolean} [initOptions.landmark=false]
     *      If set to true, the view component will be inserted into
     *      the chain of focusable landmark nodes reachable with specific
     *      global keyboard shortcuts (usually the F6 key with platform
     *      dependent modifier keys).
     *  @param {Boolean} [initOptions.hoverEffect=false]
     *      If set to true, all control groups in this view component will be
     *      displayed half-transparent as long as the mouse does not hover the
     *      view component. Has no effect, if the current device is a touch
     *      device.
     */
    function Component(app, initOptions) {

        var // self reference
            self = this,

            // the container element representing the pane
            rootNode = Utils.createContainerNode('view-component', initOptions),

            // the last cached size of the root node, used to detect layout changes
            nodeSize = null,

            // all control groups, as plain array
            groups = [],

            // all control groups, as arrays mapped by key
            groupsByKey = {},

            // all control groups with bound visibility, as arrays mapped by key
            visibleGroupsByKey = {},

            // handler called to insert a new group into this view component
            groupInserters = [],

            // whether the view component is set to visible (regardless of effective visibility)
            visible = true,

            // whether a 'component:resize' event is currently triggering (prevent recursion)
            isResizeTriggering = false,

            // whether a 'component:layout' event is currently triggering (prevent recursion)
            isLayoutTriggering = false;

        // base constructors --------------------------------------------------

        ModelObject.call(this, app);
        TimerMixin.call(this);

        // private methods ----------------------------------------------------

        /**
         * Removes the passed object from the array in-place.
         */
        function removeFromArray(array, object) {
            var index = _.indexOf(array, object);
            if (index >= 0) { array.splice(index, 1); }
        }

        /**
         * Inserts the passed group into the map under a specific group key.
         */
        function addGroupToMap(map, key, group) {
            (map[key] || (map[key] = [])).push(group);
        }

        /**
         * Removes the passed group from the map.
         */
        function removeGroupFromMap(map, group) {
            _.each(map, function (array) {
                removeFromArray(array, group);
            });
        }

        /**
         * Updates the effective visibility of the root node, according to the
         * own visible state, and the visibility of all control groups.
         *
         * @returns {Boolean}
         *  Whether the component has changed from hidden to visible state.
         */
        function updateNodeVisibility() {

            var // whether the component is visible and contains at least one visible group
                makeVisible = visible && _.any(groups, function (group) { return group.isVisible(); });

            // update node visibility and notify listeners, if really changed
            if (self.isVisible() !== makeVisible) {
                self.trigger(makeVisible ? 'component:beforeshow' : 'component:beforehide');
                Forms.showNodes(rootNode, makeVisible);
                self.trigger(makeVisible ? 'component:show' : 'component:hide');
                return makeVisible;
            }
            return false;
        }

        /**
         * Checks the current outer size of the root node of this view
         * component. Triggers a 'component:resize' event if the component is
         * visible and its size has changed. Does nothing if the component is
         * currently hidden.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.updateLayout=false]
         *      If set to true, updates the layout of all control groups.
         *  @param {Boolean} [options.triggerLayout=false]
         *      If set to true, forces to trigger a 'component:layout' event.
         *      This event will be triggered anyway if the size of the root
         *      node has changed, or if the option 'updateLayout' has been set.
         */
        function updateNodeSize(options) {

            // do nothing if the component is hidden, or is inside a hidden container node
            if (!self.isVisible() || !Forms.isDeeplyVisibleNode(rootNode)) { return; }

            var // whether to update the view components
                updateLayout = Utils.getBooleanOption(options, 'updateLayout', false),
                // whether to trigger a 'pane:layout' event
                triggerLayout = updateLayout || Utils.getBooleanOption(options, 'triggerLayout', false);

            // update layout of all control groups if specified
            if (updateLayout) { _.invoke(groups, 'layout'); }

            var // current size of the DOM root node
                newSize = { width: rootNode.outerWidth(), height: rootNode.outerHeight() };

            // notify listeners if size has changed
            if (!_.isEqual(nodeSize, newSize)) {
                nodeSize = newSize;
                triggerLayout = true; // always trigger 'component:layout' event after resize
                if (!isResizeTriggering) {
                    isResizeTriggering = true;
                    self.trigger('component:resize', nodeSize.width, nodeSize.height);
                    isResizeTriggering = false;
                }
            }

            // notify listeners that a component has changed
            if (triggerLayout) {
                isLayoutTriggering = true;
                self.trigger('component:layout');
                isLayoutTriggering = false;
            }
        }

        /**
         * Updates visibility and size of the root node of this view component.
         *
         * @param {Boolean} [updateLayout=false]
         *  If set to true, updates the layout of all control groups. The
         *  control groups will be updated automatically, if the view component
         *  has changed from hidden to visible state.
         */
        function updateNode(updateLayout) {
            var becameVisible = updateNodeVisibility();
            updateNodeSize({ updateLayout: updateLayout || becameVisible });
        }

        /**
         * Handles all groups in this view component whose node size has
         * changed (shown, hidden, resized). Triggers a 'component:resize'
         * event to all listeners, if the size of the component has been
         * changed due to the changed group. Afterwards, a 'component:layout'
         * event will always be triggered.
         */
        var groupResizeHandler = (function () {
            var triggerLayout = false;
            return self.createDebouncedMethod(function (event) {
                if (!isLayoutTriggering) { triggerLayout = true; }
                if (event.type !== 'group:resize') { updateNodeVisibility(); }
            }, function () {
                var options = { triggerLayout: triggerLayout };
                triggerLayout = false;
                updateNodeSize(options);
            });
        }());

        /**
         * Handles focus events from all registered group instances. Updates
         * the CSS marker class at the root node of this view component, and
         * forwards the event to all listeners.
         */
        function groupFocusHandler(event) {

            var focused = event.type === 'group:focus',
                type = focused ? 'component:focus' : 'component:blur';

            rootNode.toggleClass(Forms.FOCUSED_CLASS, focused);
            self.trigger(type);
        }

        /**
         * Handler for application controller 'change:items' events.
         */
        function controllerChangeHandler(event, changedItems) {
            _.each(changedItems, function (itemState, key) {
                if (key in visibleGroupsByKey) {
                    // show/hide all controls bound to the item
                    _.invoke(visibleGroupsByKey[key], 'toggle', itemState.enabled);
                }
                if (key in groupsByKey) {
                    // enable/disable all controls bound to the item
                    _.invoke(groupsByKey[key], 'enable', itemState.enabled);
                    // change value of all controls bound to the item
                    _.invoke(groupsByKey[key], 'setValue', itemState.value);
                }
            });
        }

        // protected methods --------------------------------------------------

        /**
         * Registers a callback function that implements inserting the root DOM
         * node of a new control group into this view component. The last
         * registered callback function will be invoked first when adding a new
         * group. If it returns the Boolean value false, the preceding inserter
         * callback functions will be invoked as long as they return false too.
         * If all inserter callback functions have returned false, or if no
         * callback function has been registered at all, the root node of the
         * group will be appended to the root node of this component.
         *
         * @attention
         *  Intended to be used by the internal implementations of sub classes.
         *  DO NOT CALL from external code!
         *
         * @param {Function} inserter
         *  The callback function called for every control group inserted into
         *  this view component by calling the method Component.addGroup().
         *  Receives the reference to the new control group instance as first
         *  parameter, and the options passed to that method as second
         *  parameter. Will be called in the context of this instance. May
         *  return false to indicate that the group has not been inserted yet.
         *
         * @returns {Component}
         *  A reference to this instance.
         */
        this.registerGroupInserter = function (inserter) {
            groupInserters.unshift(inserter);
            return this;
        };

        /**
         * Removes a callback function that has been registered with the method
         * Component.registerGroupInserter(), to be able to register a custom
         * node inserter temporarily.
         *
         * @attention
         *  Intended to be used by the internal implementations of sub classes.
         *  DO NOT CALL from external code!
         *
         * @param {Function} inserter
         *  The callback function that has been registered by calling the
         *  method Component.registerGroupInserter().
         *
         * @returns {Component}
         *  A reference to this instance.
         */
        this.unregisterGroupInserter = function (inserter) {
            removeFromArray(groupInserters, inserter);
            return this;
        };

        // methods ------------------------------------------------------------

        /**
         * Returns the root element containing this view component as jQuery
         * object.
         */
        this.getNode = function () {
            return rootNode;
        };

        /**
         * Returns the optional parameters that have been passed to the
         * 'initOptions' parameter of the class constructor.
         *
         * @returns {Object|Undefined}
         *  The optional parameters if existing.
         */
        this.getOptions = function () {
            return initOptions;
        };

        /**
         * Checks the current outer size of the root node of this view
         * component. Triggers a 'component:resize' event if the component is
         * visible and its size has changed. Does nothing if the component is
         * currently hidden.
         *
         * @returns {Component}
         *  A reference to this instance.
         */
        this.layout = function () {
            updateNode(true);
            return this;
        };

        /**
         * Returns whether this view component is visible.
         */
        this.isVisible = function () {
            return Forms.isVisibleNode(rootNode);
        };

        /**
         * Displays this view component, if it is currently hidden.
         *
         * @returns {Component}
         *  A reference to this view component.
         */
        this.show = function () {
            return this.toggle(true);
        };

        /**
         * Hides this view component, if it is currently visible.
         *
         * @returns {Component}
         *  A reference to this view component.
         */
        this.hide = function () {
            return this.toggle(false);
        };

        /**
         * Toggles the visibility of this view component.
         *
         * @param {Boolean} [state]
         *  If specified, shows or hides the view component depending on the
         *  boolean value. If omitted, toggles the current visibility of the
         *  view component.
         *
         * @returns {Component}
         *  A reference to this view component.
         */
        this.toggle = function (state) {

            // resolve missing parameter for toggling visibility
            visible = (state === true) || ((state !== false) && !visible);

            // update layout, this sets the effective visibility
            updateNode();
            return this;
        };

        /**
         * Adds the passed control group to this view component and binds it to
         * the specified controller item.
         *
         * @param {String|Null} key
         *  The key of a controller item associated to the control group. The
         *  enabled state and the current value of the group will be bound to
         *  the state and value of the controller item. If the group triggers a
         *  'group:change' event, the controller item will be executed. The key
         *  can be set to the value null to insert an unbound group into this
         *  component.
         *
         * @param {Group} group
         *  The control group object to be inserted.
         *
         * @param {Object} [options]
         *  Optional parameters, that will be passed to the registered group
         *  inserter callback functions. The following standard options are
         *  supported:
         *  @param {String} [options.visibleKey]
         *      If specified, the visibility of the control group will be bound
         *      to the enabled state of the controller item with the key passed
         *      in this option. If the item gets disabled, the group will be
         *      hidden automatically, and vice versa.
         *  @param {HTMLElement|jQuery|Object|Function} [options.focusTarget]
         *      A DOM node that will receive the browser focus after the
         *      inserted group has been activated, an object that provides a
         *      grabFocus() method, or a callback function that returns a focus
         *      target dynamically. Default focus target is the document view.
         *
         * @returns {Component}
         *  A reference to this view component.
         */
        this.addGroup = function (key, group, options) {

            var // the key of a controller item bound to the visibility of the group
                visibleKey = Utils.getStringOption(options, 'visibleKey'),
                // target focus node after activating the group
                focusTarget = Utils.getOption(options, 'focusTarget');

            // builds a focus options object to be passed to the controller
            function getFocusOptions(event) {
                var options = { changeEvent: event, preserveFocus: event.preserveFocus === true };
                if ('sourceType' in event) { options.sourceType = event.sourceType; }
                if (!_.isUndefined(focusTarget)) { options.focusTarget = focusTarget; }
                return options;
            }

            // returns browser focus to the application
            function groupCancelHandler(event) {
                app.getController().grabFocus(getFocusOptions(event));
                app.getController().update();
            }

            // remember the group object
            groups.push(group);

            // insert the group into this view component
            if (_.all(groupInserters, function (inserter) { return inserter.call(self, group, options) === false; })) {
                rootNode.append(group.getNode());
            }

            // handle and forward group events
            this.listenTo(group, 'group:cancel', groupCancelHandler);
            this.listenTo(group, 'group:show group:hide group:resize', groupResizeHandler);
            this.listenTo(group, 'group:focus group:blur', groupFocusHandler);

            // bind group to controller item
            if (_.isString(key)) {

                // store group reference in internal map
                addGroupToMap(groupsByKey, key, group);

                // execute controller item for 'group:change' events
                this.listenTo(group, 'group:change', function (event, value) {
                	var executeOptions = {
                        changeEvent: event
                    };
                    if (event.sourceType) { executeOptions.sourceType = event.sourceType; }
                    if (event.preserveFocus) { executeOptions.preserveFocus = event.preserveFocus; }
                    if (focusTarget) { executeOptions.focusTarget = focusTarget; }

                    app.getController().executeItem(key, value, executeOptions);
                });

                // set the key as DOM data attribute at the root node of the group
                group.getNode().attr('data-key', key);
            }

            // bind group visibility to controller item
            if (_.isString(visibleKey)) {

                // store group reference in internal map
                addGroupToMap(visibleGroupsByKey, visibleKey, group);
            }

            // update internal layout (component visibility and size)
            updateNode();
            return this;
        };

        /**
         * Removes the specified group from this view component, if it has been
         * registered before with the method Component.addGroup(). The group
         * will be unregistered and removed, but it will NOT be destroyed.
         * Caller MUST take ownership of the passed group.
         *
         * @param {Group} group
         *  The control group object to be removed.
         *
         * @returns {Component}
         *  A reference to this instance.
         */
        this.removeGroup = function (group) {

            // unregister all event handlers for the group
            this.stopListeningTo(group);

            // remove group from internal containers
            removeFromArray(groups, group);
            removeGroupFromMap(groupsByKey, group);
            removeGroupFromMap(visibleGroupsByKey, group);

            // remove group root node from DOM
            group.getNode().detach();

            // update internal layout (component visibility and size)
            updateNode();
            return this;
        };

        /**
         * Visits all control groups registered at this view component.
         *
         * @param {Function} iterator
         *  The iterator called for each control group. Receives the reference
         *  to the control group, and its insertion index. If the iterator
         *  returns the Utils.BREAK object, the iteration process will be
         *  stopped immediately.
         *
         * @param {Object} [context]
         *  If specified, the iterator will be called with this context (the
         *  symbol 'this' will be bound to the context inside the iterator
         *  function).
         *
         * @returns {Utils.BREAK|Undefined}
         *  A reference to the Utils.BREAK object, if the iterator has returned
         *  Utils.BREAK to stop the iteration process, otherwise undefined.
         */
        this.iterateGroups = function (iterator, context) {
            return _.any(groups, function (group, index) {
                return iterator.call(context, group, index) === Utils.BREAK;
            }) ? Utils.BREAK : undefined;
        };

        /**
         * Returns whether this view component contains visible groups.
         *
         * @returns {Boolean}
         *  Whether this view component contains visible groups.
         */
        this.hasVisibleGroups = function () {
            return _.any(groups, function (group) { return group.isVisible(); });
        };

        /**
         * Returns whether this view component contains the control group that
         * is currently focused. Searches in all registered group objects.
         * Returns also true, if a DOM element is focused that is related to a
         * control group in this view component, but is not a child of the
         * group's root node, e.g. a pop-up menu.
         *
         * @returns {Boolean}
         *  Whether a control group of this view component currently owns the
         *  browser focus.
         */
        this.hasFocus = function () {
            return _.any(groups, function (group) { return group.hasFocus(); });
        };

        /**
         * Sets the focus to the first enabled control group object in this
         * view component, unless the component already contains the browser
         * focus.
         *
         * @returns {Component}
         *  A reference to this view component.
         */
        this.grabFocus = function () {
            if (!this.hasFocus()) { Forms.grabFocus(rootNode); }
            return this;
        };

        // initialization -----------------------------------------------------

        // update all registered groups after change events of the application controller
        app.onInit(function () {
            self.listenTo(app.getController(), 'change:items', controllerChangeHandler);
        });

        // initially hidden until the first visible group will be inserted
        Forms.showNodes(rootNode, false);

        // marker for touch devices
        rootNode.toggleClass('touch', Modernizr.touch);

        // whether the component will be a land mark for F6 focus traveling
        rootNode.toggleClass('f6-target', Utils.getBooleanOption(initOptions, 'landmark', false));

        // hover effect for groups embedded in the view component
        rootNode.toggleClass('hover-effect', Utils.getBooleanOption(initOptions, 'hoverEffect', false));

        // log focus and layout events
        Utils.FocusLogger.logEvent(this, 'component:focus component:blur');
        Utils.LayoutLogger.logEvent(this, 'component:beforeshow component:show component:beforehide component:hide component:layout');
        Utils.LayoutLogger.logEvent(this, 'component:resize', 'width', 'height');

        // destroy all class members on destruction
        this.registerDestructor(function () {
            _.invoke(groups, 'destroy');
            rootNode.remove();
            app = initOptions = self = rootNode = null;
            groups = groupsByKey = visibleGroupsByKey = groupInserters = null;
        });

    } // class Component

    // exports ================================================================

    // derive this class from class ModelObject
    return ModelObject.extend({ constructor: Component });

});
