/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/tk/forms',
    ['io.ox/office/tk/config',
     'io.ox/office/tk/utils',
     'io.ox/office/tk/keycodes'
    ], function (Config, Utils, KeyCodes) {

    'use strict';

    var // CSS class name for caption elements
        CAPTION_CLASS = 'caption',

        // CSS selector for caption elements in control nodes
        CAPTION_SELECTOR = '>.' + CAPTION_CLASS,

        // CSS selector for icon elements in control captions
        ICON_SELECTOR = '>i',

        // CSS selector for text spans in control captions
        SPAN_SELECTOR = '>span';

    // private global functions ===============================================

    /**
     * Converts the passed camel-case name to a lower-case name with dashes as
     * separators.
     */
    function convertName(name) {
        return name.replace(/([A-Z])/g, '-$1').toLowerCase();
    }

    /**
     * Converts the passed string or object to the mark-up of a 'style' element
     * attribute.
     *
     * @param {String|Object} [style]
     *  The value for the 'style' element attribute. Can be a string that will
     *  be inserted into the 'style' attribute as is, or can be an object that
     *  maps CSS properties to their values. The property names can be given in
     *  camel-case notation. If omitted, this method returns an empty string.
     *
     * @returns {String}
     *  The string ' style="[value]"' (with leading space character), where
     *  [value] is the value passed to this method; or an empty string, if no
     *  valid value has been passed.
     */
    function createStyleMarkup(style) {

        var styleMarkup = '';

        if (_.isObject(style)) {
            _.each(style, function (value, name) {
                styleMarkup += convertName(name) + ':' + value + ';';
            });
        } else if (_.isString(style)) {
            styleMarkup += style;
        }

        if (styleMarkup.length > 0) {
            styleMarkup = ' style="' + styleMarkup + '"';
        }

        return styleMarkup;
    }

    /**
     * Converts the passed string to the mark-up of a 'title' element
     * attribute, together with the according 'aria-label' attribute.
     */
    function createToolTipMarkup(tooltip) {
        if (!_.isString(tooltip) || (tooltip.length === 0)) { return ''; }
        tooltip = Utils.escapeHTML(tooltip);
        return ' title="' + tooltip + '" aria-label="' + tooltip + '"';
    }

    // static class Forms =====================================================

    var // the exported Forms class
        Forms = {};

    // constants --------------------------------------------------------------

    /**
     * CSS class for hidden DOM elements.
     */
    Forms.HIDDEN_CLASS = 'hidden';

    /**
     * CSS selector for hidden DOM elements.
     */
    Forms.HIDDEN_SELECTOR = '.' + Forms.HIDDEN_CLASS;

    /**
     * CSS selector for visible DOM elements.
     */
    Forms.VISIBLE_SELECTOR = ':not(' + Forms.HIDDEN_SELECTOR + ')';

    /**
     * CSS class for disabled DOM elements.
     *
     * @constant
     */
    Forms.DISABLED_CLASS = 'disabled';

    /**
     * CSS selector for disabled DOM elements.
     *
     * @constant
     */
    Forms.DISABLED_SELECTOR = '.' + Forms.DISABLED_CLASS;

    /**
     * CSS selector for enabled DOM elements.
     *
     * @constant
     */
    Forms.ENABLED_SELECTOR = ':not(' + Forms.DISABLED_SELECTOR + ')';

    /**
     * CSS class for selected (active) DOM elements.
     *
     * @constant
     */
    Forms.SELECTED_CLASS = 'selected';

    /**
     * CSS selector for selected (active) DOM elements.
     *
     * @constant
     */
    Forms.SELECTED_SELECTOR = '.' + Forms.SELECTED_CLASS;

    /**
     * CSS class for buttons with ambiguous value.
     *
     * @constant
     */
    Forms.AMBIGUOUS_CLASS = 'ambiguous';

    /**
     * CSS class for DOM container elements that contain the browser focus.
     *
     * @constant
     */
    Forms.FOCUSED_CLASS = 'focused';

    /**
     * CSS selector for DOM elements that can receive the keyboard focus. By
     * convention, all focusable DOM elements MUST have a non-negative value in
     * their 'tabindex' element attribute (also elements which are natively
     * focusable, such as <input> elements of type 'text').
     *
     * @constant
     */
    Forms.FOCUSABLE_SELECTOR = '[tabindex]:not([tabindex^="-"]):visible';

    /**
     * CSS class for button elements.
     *
     * @constant
     */
    Forms.BUTTON_CLASS = 'button';

    /**
     * CSS selector for button elements.
     *
     * @constant
     */
    Forms.BUTTON_SELECTOR = '.' + Forms.BUTTON_CLASS;

    // generic helpers --------------------------------------------------------

    /**
     * Creates and returns the HTML mark-up for a DOM element.
     *
     * @param {String} nodeName
     *  The tag name of the DOM element to be created.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Object} [options.attributes]
     *      The element attributes to be inserted into the DOM element. The
     *      attribute names can be given in camel-case notation. MUST NOT
     *      contain the attribute 'style' if the option 'style' (see below) is
     *      in use. MUST NOT contain the attribute 'title' if the option
     *      'tooltip' (see below) is in use.
     *  @param {String|Object} [options.style]
     *      Additional CSS style attributes for the control element. Will be
     *      inserted into the 'style' attribute of the element. Can be a string
     *      that will be inserted into the 'style' attribute as is, or can be
     *      an object that maps CSS properties to their values. The property
     *      names can be given in camel-case notation.
     *  @param {String} [options.tooltip]
     *      Tool tip text shown when the mouse hovers the DOM element. If
     *      omitted, the element will not show a tool tip.
     *  @param {String} [options.content='']
     *      The HTML mark-up of the DOM element's contents.
     *
     * @returns {String}
     *  The HTML mark-up of the DOM element.
     */
    Forms.createElementMarkup = function (nodeName, options) {

        var // the HTML mark-up
            markup = '<' + nodeName;

        // add the attributes
        _.each(Utils.getObjectOption(options, 'attributes'), function (value, name) {
            markup += ' ' + convertName(name) + '="' + Utils.escapeHTML(String(value)) + '"';
        });

        // add CSS styles and tool tip
        markup += createStyleMarkup(Utils.getOption(options, 'style'));
        markup += createToolTipMarkup(Utils.getOption(options, 'tooltip'));

        // add inner HTML mark-up and close the element
        markup += '>' + Utils.getStringOption(options, 'content', '') + '</' + nodeName + '>';
        return markup;
    };

    /**
     * Adds a tool tip to the specified elements.
     *
     * @param {HTMLElement|jQuery} nodes
     *  The elements to be manipulated. If this object is a jQuery collection,
     *  sets the tool tip for all nodes it contains.
     *
     * @param {String} [tooltip]
     *  Tool tip text shown when the mouse hovers one of the nodes. If omitted,
     *  the nodes will not show a tool tip anymore.
     */
    Forms.setToolTip = function (nodes, tooltip) {
        if (_.isString(tooltip) && tooltip) {
            $(nodes).attr({ title: tooltip, 'aria-label': tooltip });
        } else {
            $(nodes).removeAttr('title aria-label');
        }
    };

    /**
     * Returns whether the specified DOM element is in visible state. If node
     * contains the CSS class Forms.HIDDEN_CLASS, it is considered being
     * hidden.
     *
     * @attention
     *  Checks the existence of the CSS class Forms.HIDDEN_CLASS only; does NOT
     *  check the effective visibility of the DOM node according to its CSS
     *  formatting attributes.
     *
     * @param {HTMLElement|jQuery} node
     *  The DOM element. If this object is a jQuery collection, uses the first
     *  node it contains.
     *
     * @returns {Boolean}
     *  Whether the element is visible.
     */
    Forms.isVisibleNode = function (node) {
        return $(node).first().is(Forms.VISIBLE_SELECTOR);
    };

    /**
     * Returns whether the specified DOM element and all its ancestors are in
     * visible state. If there is any ancestor containing the CSS class
     * Forms.HIDDEN_CLASS, or the node is not part of the document DOM, it is
     * considered being hidden.
     *
     * @attention
     *  Checks the existence of the CSS class Forms.HIDDEN_CLASS only; does NOT
     *  check the effective visibility of the DOM node according to its CSS
     *  formatting attributes.
     *
     * @param {HTMLElement|jQuery} node
     *  The DOM element. If this object is a jQuery collection, uses the first
     *  node it contains.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Boolean} [options.deep=false]
     *      If set to true, all ancestors of the passed element will be checked
     *      for visibility too.
     *
     * @returns {Boolean}
     *  Whether the element is visible.
     */
    Forms.isDeeplyVisibleNode = function (node) {
        node = $(node).first();
        return Utils.containsNode(document.body, node[0]) && (node.closest(Forms.HIDDEN_SELECTOR).length === 0);
    };

    /**
     * Filters the visible elements from the passed jQuery collection.
     *
     * @attention
     *  Checks the existence of the CSS class Forms.HIDDEN_CLASS only; does NOT
     *  check the effective visibility of the DOM nodes according to their CSS
     *  formatting attributes.
     *
     * @param {HTMLElement|jQuery} nodes
     *  The elements to be filtered.
     *
     * @returns {jQuery}
     *  A jQuery collection with all visible elements.
     */
    Forms.filterVisibleNodes = function (nodes) {
        return $(nodes).filter(Forms.VISIBLE_SELECTOR);
    };

    /**
     * Filters the hidden elements from the passed jQuery collection.
     *
     * @attention
     *  Checks the existence of the CSS class Forms.HIDDEN_CLASS only; does NOT
     *  check the effective visibility of the DOM nodes according to their CSS
     *  formatting attributes.
     *
     * @param {HTMLElement|jQuery} nodes
     *  The elements to be filtered.
     *
     * @returns {jQuery}
     *  A jQuery collection with all hidden elements.
     */
    Forms.filterHiddenNodes = function (nodes) {
        return $(nodes).not(Forms.VISIBLE_SELECTOR);
    };

    /**
     * Shows or hides the specified DOM elements.
     *
     * @attention
     *  Modifies the existence of the CSS class Forms.HIDDEN_CLASS only; does
     *  NOT modify the 'display' or any other CSS properties of the DOM node.
     *
     * @param {HTMLElement|jQuery} nodes
     *  The DOM elements to be manipulated. If this object is a jQuery
     *  collection, uses all nodes it contains.
     *
     * @param {Boolean} state
     *  Whether to show (true) or hide (false) the DOM elements.
     */
    Forms.showNodes = function (nodes, state) {
        nodes = $(nodes);
        nodes.toggleClass(Forms.HIDDEN_CLASS, !state);
        if (state) { nodes.removeAttr('aria-hidden'); } else { nodes.attr('aria-hidden', true); }
    };

    /**
     * Returns whether the specified DOM element is in enabled state.
     *
     * @param {HTMLElement|jQuery} node
     *  The DOM element. If this object is a jQuery collection, uses the first
     *  node it contains.
     *
     * @returns {Boolean}
     *  Whether the element is enabled.
     */
    Forms.isEnabledNode = function (node) {
        return $(node).first().is(Forms.ENABLED_SELECTOR);
    };

    /**
     * Filters the enabled elements from the passed jQuery collection.
     *
     * @param {HTMLElement|jQuery} nodes
     *  The elements to be filtered.
     *
     * @returns {jQuery}
     *  A jQuery collection with all enabled elements.
     */
    Forms.filterEnabledNodes = function (nodes) {
        return $(nodes).filter(Forms.ENABLED_SELECTOR);
    };

    /**
     * Filters the disabled elements from the passed jQuery collection.
     *
     * @param {HTMLElement|jQuery} nodes
     *  The elements to be filtered.
     *
     * @returns {jQuery}
     *  A jQuery collection with all disabled elements.
     */
    Forms.filterDisabledNodes = function (nodes) {
        return $(nodes).not(Forms.ENABLED_SELECTOR);
    };

    /**
     * Enables or disables the specified DOM elements.
     *
     * @param {HTMLElement|jQuery} nodes
     *  The DOM elements to be manipulated. If this object is a jQuery
     *  collection, uses all nodes it contains.
     *
     * @param {Boolean} state
     *  Whether to enable (true) or disable (false) the DOM elements.
     */
    Forms.enableNodes = function (nodes, state) {
        nodes = $(nodes);
        nodes.toggleClass(Forms.DISABLED_CLASS, !state);
        if (state) { nodes.removeAttr('aria-disabled'); } else { nodes.attr('aria-disabled', true); }
    };

    /**
     * Returns whether the specified DOM element is in selected state.
     *
     * @param {HTMLElement|jQuery} node
     *  The DOM element. If this object is a jQuery collection, uses the first
     *  node it contains.
     *
     * @returns {Boolean}
     *  Whether the element is selected.
     */
    Forms.isSelectedNode = function (node) {
        return $(node).first().is(Forms.SELECTED_SELECTOR);
    };

    /**
     * Filters the selected elements from the passed jQuery collection.
     *
     * @param {HTMLElement|jQuery} nodes
     *  The elements to be filtered.
     *
     * @returns {jQuery}
     *  A jQuery collection with all selected elements.
     */
    Forms.filterSelectedNodes = function (nodes) {
        return $(nodes).filter(Forms.SELECTED_SELECTOR);
    };

    /**
     * Filters the unselected elements from the passed jQuery collection.
     *
     * @param {HTMLElement|jQuery} nodes
     *  The elements to be filtered.
     *
     * @returns {jQuery}
     *  A jQuery collection with all unselected elements.
     */
    Forms.filterUnselectedNodes = function (nodes) {
        return $(nodes).not(Forms.SELECTED_SELECTOR);
    };

    /**
     * Selects or deselects the specified DOM elements.
     *
     * @param {HTMLElement|jQuery} nodes
     *  The DOM elements to be manipulated. If this object is a jQuery
     *  collection, uses all nodes it contains.
     *
     * @param {Boolean} state
     *  Whether to select (true) or deselect (false) the DOM elements.
     */
    Forms.selectNodes = function (nodes, state) {
        $(nodes).toggleClass(Forms.SELECTED_CLASS, state);
    };

    /**
     * Triggers a 'click' event at the target node of the passed keyboard
     * event. The new click event will contain additional information about the
     * original keyboard event:
     * - {Event} originalEvent
     *      The original keyboard DOM event.
     * - {Number} keyCode
     *      The key code of the original event.
     * - {Boolean} shiftKey
     *      The state of the SHIFT key.
     * - {Boolean} altKey
     *      The state of the ALT key.
     * - {Boolean} ctrlKey
     *      The state of the CTRL key.
     * - {Boolean} metaKey
     *      The state of the META key.
     *
     * @param {jQuery.Event}
     *  The original event, as jQuery event object.
     */
    Forms.triggerClickForKey = function (event) {
        $(event.target).trigger(new $.Event('click', {
            originalEvent: event.originalEvent,
            keyCode: event.keyCode,
            shiftKey: event.shiftKey,
            altKey: event.altKey,
            ctrlKey: event.ctrlKey,
            metaKey: event.metaKey
        }));
    };

    // browser focus ----------------------------------------------------------

    /**
     * Returns all focusable control elements contained in the specified DOM
     * elements.
     *
     * @param {HTMLElement|jQuery} nodes
     *  The DOM elements to be searched. If this object is a jQuery collection,
     *  uses all nodes it contains.
     *
     * @returns {jQuery}
     *  A jQuery collection with all focusable form controls contained in the
     *  passed elements.
     */
    Forms.findFocusableNodes = function (nodes) {
        return $(nodes).find(Forms.FOCUSABLE_SELECTOR);
    };

    /**
     * Sets the browser focus to the first focusable control element contained
     * in the specified DOM element.
     *
     * @param {HTMLElement|jQuery} nodes
     *  The DOM element to be focused. If this object is a jQuery collection,
     *  searches all nodes it contains for a focusable form control.
     */
    Forms.grabFocus = function (nodes) {
        Forms.findFocusableNodes(nodes).first().focus();
    };

    /**
     * Enables focus navigation with cursor keys in the specified DOM element.
     *
     * @param {HTMLElement|jQuery} rootNode
     *  A DOM element whose descendant focusable form controls can be focused
     *  with the cursor keys.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Boolean} [options.homeEnd=false]
     *      If set to true, the HOME key will focus the first, and the END key
     *      will focus the last available form control.
     */
    Forms.enableCursorFocusNavigation = function (rootNode, options) {

        var // whether to use HOME/END keys
            homeEnd = Utils.getBooleanOption(options, 'homeEnd', false),
            // anchor node for vertical movement
            anchorVertNode = null,
            // anchor node for horizontal movement
            anchorHorNode = null;

        function setFocus(focusNode, resetAnchor) {
            focusNode.focus();
            if (resetAnchor || !anchorVertNode) { anchorVertNode = focusNode; }
            if (resetAnchor || !anchorHorNode) { anchorHorNode = focusNode; }
        }

        function moveFocus(focusNode, direction) {

            var // all focusable nodes contained in the root node
                focusableNodes = Forms.findFocusableNodes(rootNode),
                // the index of the focused node
                focusIndex = focusableNodes.index(focusNode),
                // whether to move forward
                forward = /^(right|down)$/.test(direction),
                // whether to move up/down
                vertical = /^(up|down)$/.test(direction),
                // position and size of the focused node
                focusPosition = null;

            // any other node was focused (e.g. the root node itself): start with first/last available node
            if (focusIndex < 0) {
                setFocus(forward ? focusableNodes.first() : focusableNodes.last(), true);
                return;
            }

            // resolve the positions and sizes of all focus nodes
            focusableNodes = _.map(focusableNodes.get(), function (node) {
                var position = Utils.getNodePositionInPage(node);
                return _.extend(position, {
                    node: node,
                    centerX: position.left + position.width / 2,
                    centerY: position.top + position.height / 2
                });
            });

            // remember position of current focus node
            focusPosition = focusableNodes[focusIndex];

            // filter out all nodes that are not reachable in the specified direction
            focusableNodes = _.filter(focusableNodes, function (nodePosition) {
                switch (direction) {
                case 'left':
                    return nodePosition.left + nodePosition.width <= focusPosition.left;
                case 'right':
                    return nodePosition.right + nodePosition.width <= focusPosition.right;
                case 'up':
                    return nodePosition.top + nodePosition.height <= focusPosition.top;
                case 'down':
                    return nodePosition.bottom + nodePosition.height <= focusPosition.bottom;
                }
            });

            // return early, if no focusable nodes left
            if (focusableNodes.length === 0) { return; }

            // insert the centered anchor position into the focus position, if available
            (function () {
                var anchorNode = vertical ? anchorVertNode : anchorHorNode,
                    anchorPosition = anchorNode ? Utils.getNodePositionInPage(anchorNode) : null;
                if (anchorPosition) {
                    if (vertical) {
                        focusPosition.centerX = anchorPosition.left + anchorPosition.width / 2;
                    } else {
                        focusPosition.centerY = anchorPosition.top + anchorPosition.height / 2;
                    }
                }
            }());

            // sort nodes by distance to current focus node
            focusableNodes = _.sortBy(focusableNodes, function (nodePosition) {
                return Math.pow(nodePosition.centerX - focusPosition.centerX, 2) + Math.pow(nodePosition.centerY - focusPosition.centerY, 2);
            });

            // clear the old anchor node for the opposite direction
            if (vertical) { anchorHorNode = null; } else { anchorVertNode = null; }

            // focus the nearest node
            setFocus(focusableNodes[0].node);
        }

        function keyDownHandler(event) {

            // ignore all keyboard events with any modifier keys (only plain cursor keys supported)
            if (KeyCodes.hasModifierKeys(event)) { return; }

            // LEFT/RIGHT cursor keys (not in text fields)
            if (!$(event.target).is('input,textarea')) {
                switch (event.keyCode) {
                case KeyCodes.LEFT_ARROW:
                    moveFocus(event.target, 'left');
                    return false;
                case KeyCodes.RIGHT_ARROW:
                    moveFocus(event.target, 'right');
                    return false;
                }
            }

            // UP/DOWN keys (always)
            switch (event.keyCode) {
            case KeyCodes.UP_ARROW:
                moveFocus(event.target, 'up');
                return false;
            case KeyCodes.DOWN_ARROW:
                moveFocus(event.target, 'down');
                return false;
            }

            // HOME/END keys (if configured)
            if (homeEnd) {
                switch (event.keyCode) {
                case KeyCodes.HOME:
                    setFocus(Forms.findFocusableNodes(rootNode).first(), true);
                    return false;
                case KeyCodes.END:
                    setFocus(Forms.findFocusableNodes(rootNode).last(), true);
                    return false;
                }
            }
        }

        $(rootNode).first().on('keydown', keyDownHandler);
    };

    // control captions -------------------------------------------------------

    /**
     * Returns the complete CSS class names of the specified icon. For icons
     * from the AweSome font (icon name starting with 'fa-'), the class 'fa'
     * will be added. For OX Document icons (icon name starting with 'docs-'),
     * additional classes for the current locale will be added, and the class
     * 'retina' if the current device has a retina display.
     *
     * @param {String} icon
     *  The CSS class name of the icon.
     *
     * @returns {String}
     *  The complete CSS classes for the icon element.
     */
    Forms.getIconClasses = (function () {

        var // the CSS classes added to localized bitmap icons
            docsIconClasses = ' lc-' + Config.LOCALE;

        if ((Config.LANGUAGE !== '') && (Config.COUNTRY !== '')) {
            docsIconClasses += ' lc-' + Config.LANGUAGE;
        }

        if (Utils.RETINA) {
            docsIconClasses += ' retina';
        }

        return function (icon) {
            return (/^fa-/).test(icon) ? ('fa ' + icon) : (/^docs-/).test(icon) ? (icon + docsIconClasses) : icon;
        };
    }());

    /**
     * Creates the HTML mark-up of an <i> DOM element representing an icon.
     *
     * @param {String} icon
     *  The CSS class name of the icon.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {String} [options.classes]
     *      The names of all CSS classes to be added to the icon element, as
     *      space separated string.
     *  @param {String|Object} [options.style]
     *      Additional CSS style attributes for the icon. Will be inserted into
     *      the 'style' attribute of the <i> element. Can be a string that will
     *      be inserted into the 'style' attribute as is, or can be an object
     *      that maps CSS properties to their values. The property names can be
     *      given in camel-case notation.
     *
     * @returns {String}
     *  The HTML mark-up of the icon element, as string.
     */
    Forms.createIconMarkup = function (icon, options) {

        var // the CSS class names to be added to the node
            classes = Utils.getStringOption(options, 'classes', '');

        if (classes.length) { classes = ' ' + classes; }
        classes = Forms.getIconClasses(icon) + classes;

        return Forms.createElementMarkup('i', {
            attributes: { 'class': classes, 'aria-hidden': 'true' },
            style: Utils.getOption(options, 'style')
        });
    };

    /**
     * Creates the HTML mark-up of a <span> DOM element representing the text
     * label for a control.
     *
     * @param {String} text
     *  The label text.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {String} [options.classes]
     *      The names of all CSS classes to be added to the <span> element, as
     *      space separated string.
     *  @param {String|Object} [options.style]
     *      Additional CSS style attributes for the label. Will be inserted
     *      into the 'style' attribute of the <span> element. Can be a string
     *      that will be inserted into the 'style' attribute as is, or can be
     *      an object that maps CSS properties to their values. The property
     *      names can be given in camel-case notation.
     *
     * @returns {String}
     *  The HTML mark-up of the label element, as string.
     */
    Forms.createSpanMarkup = function (text, options) {

        var // the CSS class names to be added to the node
            classes = Utils.getStringOption(options, 'classes'),
            // the element attributes
            attributes = classes ? { 'class': classes } : null;

        return Forms.createElementMarkup('span', {
            attributes: attributes,
            style: Utils.getOption(options, 'style'),
            content: Utils.escapeHTML(text)
        });
    };

    /**
     * Creates the HTML mark-up of an <img> DOM element.
     *
     * @param {String} source
     *  The source URL of the image.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {String} [options.classes]
     *      The names of all CSS classes to be added to the <img> element, as
     *      space separated string.
     *  @param {String} [options.alt]
     *      Alternate text of the image element.
     *
     * @returns {String}
     *  The HTML mark-up of the image element, as string.
     *
     */
    Forms.createImageMarkup = function (source, options) {

        if (!source) { Utils.error('Forms.createImageMarkup(): image source URL missing'); }

        var // the CSS class names to be added to the node
            classes = Utils.getStringOption(options, 'classes'),
            // alternate text for the image element
            altText = Utils.getStringOption(options, 'alt', null),
            // custom picture height
            pictureHeight = Utils.getIntegerOption(options, 'pictureHeight', 48),
            // custom picture width
            pictureWidth = Utils.getIntegerOption(options, 'pictureWidth', 48),
            // the element attributes
            attributes = { src: source, 'class': classes ? classes : null, width: pictureWidth, height: pictureHeight, alt: altText };

        return Forms.createElementMarkup('img', {
            attributes: attributes
        });
    };

    /**
     * Creates the HTML mark-up for icons and text labels for a form control.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {String} [options.icon]
     *      The CSS class name of the icon. If omitted, no icon will be shown.
     *  @param {String|Object} [options.iconClasses]
     *      The names of all CSS classes to be added to the icon element, as
     *      space separated string.
     *  @param {String|Object} [options.iconStyle]
     *      Additional CSS style attributes for the icon. Will be inserted into
     *      the 'style' attribute of the <i> element. Can be a string that will
     *      be inserted into the 'style' attribute as is, or can be an object
     *      that maps CSS properties to their values. The property names can be
     *      given in camel-case notation.
     *  @param {String} [options.label]
     *      The text label. If omitted, no text will be shown.
     *  @param {String|Object} [options.labelClasses]
     *      The names of all CSS classes to be added to the <span> element, as
     *      space separated string.
     *  @param {String|Object} [options.labelStyle]
     *      Additional CSS style attributes for the label. Will be inserted
     *      into the 'style' attribute of the <span> element. Can be a string
     *      that will be inserted into the 'style' attribute as is, or can be
     *      an object that maps CSS properties to their values. The property
     *      names can be given in camel-case notation.
     *  @param {String} [options.iconPos='leading']
     *      The position of the icon relative to the text label. Possible
     *      values are:
     *      - 'leading': The icon will be located in front of the label.
     *      - 'trailing': The icon will be located after the label.
     *
     * @returns {String}
     *  The HTML mark-up of the complete caption nodes, as string.
     */
    Forms.createCaptionMarkup = function (options) {

        var // the icon mark-up
            icon = Utils.getStringOption(options, 'icon', ''),
            // the label mark-up
            label = Utils.getStringOption(options, 'label', ''),
            // the position of the icon
            iconPos = Utils.getStringOption(options, 'iconPos', 'leading');

        // create the icon mark-up
        if (icon) {
            icon = Forms.createIconMarkup(icon, {
                classes: Utils.getOption(options, 'iconClasses'),
                style: Utils.getOption(options, 'iconStyle')
            });
        }

        // create the label mark-up
        if (label) {
            label = Forms.createSpanMarkup(label, {
                classes: Utils.getOption(options, 'labelClasses'),
                style: Utils.getOption(options, 'labelStyle')
            });
        }

        // concatenate icon and label mark-up in the correct order
        return '<span class="' + CAPTION_CLASS + '">' + ((iconPos === 'trailing') ? (label + icon) : (icon + label)) + '</span>';
    };

    /**
     * Returns the caption container node of the passed control element.
     *
     * @param {HTMLElement|jQuery} controlNode
     *  The control element. If this object is a jQuery collection, uses the
     *  first node it contains.
     *
     * @returns {jQuery}
     *  The caption container node of the control element.
     */
    Forms.getCaptionNode = function (controlNode) {
        return $(controlNode).first().find(CAPTION_SELECTOR);
    };

    /**
     * Sets or changes the icon in the captions of the passed control elements.
     *
     * @param {HTMLElement|jQuery} controlNodes
     *  The control elements. If this object is a jQuery collection, modifies
     *  all nodes it contains.
     *
     * @param {String} icon
     *  The CSS class name of the new icon.
     *
     * @param {Object} [options]
     *  Optional parameters. Supports all options supported by the method
     *  Forms.createIconMarkup(). Additionally, the following options are
     *  supported:
     *  @param {Object} [options.pos='auto']
     *      The position of the icon relative to the text label of the control
     *      element. Possible values are:
     *      - 'auto': Replaces an existing icon at its current position. If no
     *          icon exists, the new icon will be inserted in front of the text
     *          label (if existing).
     *      - 'leading': The new icon will be inserted in front of the label.
     *      - 'trailing': The new icon will be inserted after the label.
     */
    Forms.setCaptionIcon = function (controlNodes, icon, options) {

        var iconMarkup = Forms.createIconMarkup(icon, options),
            iconPos = Utils.getStringOption(options, 'pos', 'auto');

        $(controlNodes).each(function () {

            var captionNode = $(this).find(CAPTION_SELECTOR),
                iconNode = captionNode.find(ICON_SELECTOR);

            if ((iconPos === 'auto') && (iconNode.length > 0)) {
                iconNode.after(iconMarkup);
            } else if (iconPos === 'trailing') {
                captionNode.append(iconMarkup);
            } else {
                captionNode.prepend(iconMarkup);
            }

            iconNode.remove();
        });
    };

    /**
     * Removes the icon from the captions of the passed control elements.
     *
     * @param {HTMLElement|jQuery} controlNodes
     *  The control elements. If this object is a jQuery collection, modifies
     *  all nodes it contains.
     */
    Forms.removeCaptionIcon = function (controlNodes) {
        $(controlNodes).find(CAPTION_SELECTOR + ICON_SELECTOR).remove();
    };

    /**
     * Returns the current label text of the passed control element.
     *
     * @param {HTMLElement|jQuery} controlNode
     *  The control element. If this object is a jQuery collection, uses the
     *  first node it contains.
     *
     * @returns {String}
     *  The current text label.
     */
    Forms.getCaptionText = function (controlNode) {
        return $(controlNode).first().find(CAPTION_SELECTOR + SPAN_SELECTOR).text();
    };

    /**
     * Changes the label texts in the captions of the passed control elements.
     *
     * @param {HTMLElement|jQuery} controlNodes
     *  The control elements. If this object is a jQuery collection, modifies
     *  all nodes it contains.
     *
     * @param {String} text
     *  The new label text.
     *
     * @param {Object} [options]
     *  Optional parameters. Supports all options supported by the method
     *  Forms.createSpanMarkup(). Additionally, the following options are
     *  supported:
     *  @param {Object} [options.pos='auto']
     *      The position of the label relative to the icon of the control
     *      element. Possible values are:
     *      - 'auto': Replaces an existing label at its current position. If no
     *          label exists, the new label will be inserted after the icon (if
     *          existing).
     *      - 'leading': The new label will be inserted in front of the icon.
     *      - 'trailing': The new label will be inserted after the icon.
     */
    Forms.setCaptionText = function (controlNodes, text, options) {

        var spanMarkup = Forms.createSpanMarkup(text, options),
            spanPos = Utils.getStringOption(options, 'pos', 'auto');

        $(controlNodes).each(function () {

            var captionNode = $(this).find(CAPTION_SELECTOR),
                spanNode = captionNode.find(SPAN_SELECTOR);

            if ((spanPos === 'auto') && (spanNode.length > 0)) {
                spanNode.after(spanMarkup);
            } else if (spanPos === 'leading') {
                captionNode.prepend(spanMarkup);
            } else {
                captionNode.append(spanMarkup);
            }

            spanNode.remove();
        });
    };

    /**
     * Removes the text label from the captions of the passed control elements.
     *
     * @param {HTMLElement|jQuery} controlNodes
     *  The control elements. If this object is a jQuery collection, modifies
     *  all nodes it contains.
     */
    Forms.removeCaptionText = function (controlNodes) {
        $(controlNodes).find(CAPTION_SELECTOR + SPAN_SELECTOR).remove();
    };

    /**
     * Changes the captions of the passed control elements.
     *
     * @param {HTMLElement|jQuery} controlNodes
     *  The control elements. If this object is a jQuery collection, modifies
     *  all nodes it contains.
     *
     * @param {Object} [options]
     *  Optional parameters. Supports all options supported by the method
     *  Forms.createCaptionMarkup(). Additionally, the following options are
     *  supported:
     *  @param {Object} [options.pos='auto']
     *      The position of the caption relative to the other content nodes of
     *      the control element. Possible values are:
     *      - 'auto': Replaces an existing caption at its current position. If
     *          no caption exists, the new caption will be inserted in front of
     *          the other content nodes.
     *      - 'leading': The new caption will be inserted in front of the other
     *          content nodes.
     *      - 'trailing': The new caption will be inserted after the other
     *          content nodes.
     */
    Forms.setCaption = function (controlNodes, options) {

        var captionMarkup = Forms.createCaptionMarkup(options),
            captionPos = Utils.getStringOption(options, 'pos', 'auto');

        $(controlNodes).each(function () {

            var captionNode = $(this).find(CAPTION_SELECTOR);

            if ((captionPos === 'auto') && (captionNode.length > 0)) {
                captionNode.after(captionMarkup);
            } else if (captionPos === 'trailing') {
                $(this).append(captionMarkup);
            } else {
                $(this).prepend(captionMarkup);
            }

            captionNode.remove();
        });
    };

    // drop-down carets -------------------------------------------------------

    /**
     * Creates the HTML mark-up of a DOM element containing a drop-down caret
     * symbol intended to be inserted into a button control element.
     *
     * @param {String} direction
     *  The direction for the caret symbol. Can be one of 'up', 'down', 'left',
     *  or 'right'.
     *
     * @returns {String}
     *  The HTML mark-up of the caret node, as string.
     */
    Forms.createCaretMarkup = function (direction) {
        return '<span class="caret-node">' + Forms.createIconMarkup('fa-caret-' + direction) + '</span>';
    };

    // label elements ---------------------------------------------------------

    /**
     * Creates and returns the HTML mark-up for a label element.
     *
     * @param {Object} [options]
     *  Optional parameters. Supports all options supported by the methods
     *  Forms.createElementMarkup() and Forms.setControlCaption(). Additional
     *  HTML content specified with the option 'content' will be appended to
     *  the label caption.
     *
     * @returns {String}
     *  The HTML mark-up of the label element.
     */
    Forms.createLabelMarkup = function (options) {

        var captionMarkup = Forms.createCaptionMarkup(options),
            contentMarkup = Utils.getStringOption(options, 'content', '');

        // A11Y: use <div> instead of <label> to fulfill WCAG 2.0 Success Criteria - labels must reference an input via the 'for' attribute.
        return Forms.createElementMarkup('div', Utils.extendOptions(options, { content: captionMarkup + contentMarkup, attributes: { 'class': 'label' } }));
    };

    // button elements --------------------------------------------------------

    /**
     * Creates and returns the HTML mark-up for a button element.
     *
     * @param {Object} [options]
     *  Optional parameters. Supports all options supported by the methods
     *  Forms.createElementMarkup() and Forms.setControlCaption(). Additional
     *  HTML content specified with the option 'content' will be appended to
     *  the button caption. Additionally, the following options are supported:
     *  @param {Boolean} [options.focusable=true]
     *      If set to false, the 'tabindex' attribute of the button element
     *      will be set to the value -1 in order to remove the button from the
     *      keyboard focus chain.
     *
     * @returns {String}
     *  The HTML mark-up of the button element.
     */
    Forms.createButtonMarkup = function (options) {

        var attributes = Utils.getObjectOption(options, 'attributes'),
            classes = Utils.getStringOption(attributes, 'class', ''),
            focusable = Utils.getBooleanOption(options, 'focusable', true),
            captionMarkup = Forms.createCaptionMarkup(options),
            contentMarkup = Utils.getStringOption(options, 'content', '');

        // add more element attributes, return the mark-up
        attributes = _.extend({ tabindex: focusable ? 1 : -1, role: 'button' }, attributes);
        if (!focusable) { attributes = _.extend({ 'aria-hidden': 'true' }, attributes); }
        attributes['class'] = Forms.BUTTON_CLASS;
        if (classes.length > 0) { attributes['class'] += ' ' + classes; }
        return Forms.createElementMarkup('a', Utils.extendOptions(options, { attributes: attributes, content: captionMarkup + contentMarkup }));
    };

    /**
     * Returns the value stored in the 'value' data attribute of the specified
     * button element. If the stored value is a function, calls that function
     * (with the button node as first parameter) and returns its result.
     *
     * @param {HTMLElement|jQuery} buttonNode
     *  The button element. If this object is a jQuery collection, uses the
     *  first node it contains.
     *
     * @returns {Any}
     *  The value stored at in the button element.
     */
    Forms.getButtonValue = function (buttonNode) {
        var value = $(buttonNode).first().data('value');
        return _.isFunction(value) ? value($(buttonNode)) : value;
    };

    /**
     * Stores the passed value in the 'value' data attribute of the specified
     * button elements.
     *
     * @param {HTMLElement|jQuery} buttonNodes
     *  The button elements. If this object is a jQuery collection, modifies
     *  all nodes it contains.
     *
     * @param {Any} value
     *  A value, object, or function that will be copied to the 'value' data
     *  attribute of the button element (see method Forms.getButtonValue() for
     *  details).
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {String} [options.dataValue]
     *      A string value that will be inserted into the 'data-value'
     *      attribute of the button elements. If omitted, the JSON string
     *      representation of the passed value will be used instead (all
     *      double-quote characters will be removed from the string though),
     *      unless the passed value is a function.
     */
    Forms.setButtonValue = function (buttonNodes, value, options) {

        // set the passed value as jQuery data attribute (may be objects or functions)
        buttonNodes = $(buttonNodes).data('value', value);

        // set the alternative data value as plain element attribute
        var dataValue = Utils.getStringOption(options, 'dataValue');
        if (!_.isString(dataValue) && !_.isFunction(value) && !_.isUndefined(value)) {
            dataValue = JSON.stringify(value).replace(/"/g, '');
        }
        if (dataValue) {
            buttonNodes.attr('data-value', dataValue);
        } else {
            buttonNodes.removeAttr('data-value');
        }
    };

    /**
     * Selects or deselects the passed button elements, or sets them into the
     * mixed state in case the value represented by the buttons is ambiguous.
     *
     * @param {HTMLElement|jQuery} buttonNodes
     *  The button elements to be manipulated. If this object is a jQuery
     *  collection, uses all nodes it contains.
     *
     * @param {Boolean|Null} state
     *  If set to a Boolean value, selects or deselects all button elements
     *  according to the value. If set to null, sets the button elements to the
     *  mixed (ambiguous) state.
     */
    Forms.selectButtonNodes = function (buttonNodes, state) {
        var selected = state === true;
        Forms.selectNodes(buttonNodes, selected);
        $(buttonNodes).toggleClass(Forms.AMBIGUOUS_CLASS, _.isNull(state)).attr({'aria-pressed': selected, 'aria-checked': selected, 'aria-selected': selected });
    };

    /**
     * Filters the passed button elements by comparing to a specific value.
     *
     * @param {HTMLElement|jQuery} buttonNodes
     *  The button elements to be filtered. If this object is a jQuery
     *  collection, uses all nodes it contains.
     *
     * @param {Any} value
     *  The button value to be filtered for.
     *
     * @param {Function} [options.matcher=_.isEqual]
     *  A comparison function that returns whether the actual button values
     *  should be included in the result. Receives the passed value as first
     *  parameter, and the value of the current button element as second
     *  parameter. The function must return the Boolean value true in order to
     *  include the respective button element in the result. If omitted, uses
     *  _.isEqual() which compares arrays and objects deeply.
     *
     * @returns {jQuery}
     *  A jQuery collection with all button nodes whose value is equal to the
     *  passed value.
     */
    Forms.filterButtonNodes = function (buttonNodes, value, matcher) {
        if (!_.isFunction(matcher)) { matcher = _.isEqual; }
        return $(buttonNodes).filter(function () {
            return matcher(value, Forms.getButtonValue(this)) === true;
        });
    };

    /**
     * Filters the passed button elements by a specific value.
     *
     * @param {HTMLElement|jQuery} buttonNodes
     *  The button elements to be filtered. If this object is a jQuery
     *  collection, uses all nodes it contains.
     *
     * @param {Any} value
     *  The button value to be filtered for.
     *
     * @param {Function} [options.matcher=_.isEqual]
     *  A comparison function that returns whether the actual button values
     *  should be selected, deselected, or skipped. Receives the passed value
     *  as first parameter, and the value of the current button element as
     *  second parameter. The function must return the Boolean value true to
     *  select the respective button element, the Boolean value false to
     *  deselect the button element, or any other value to leave the button
     *  element unmodified. If omitted, uses _.isEqual() which compares arrays
     *  and objects deeply, and does not skip any button element.
     */
    Forms.selectMatchingButtonNodes = function (buttonNodes, value, matcher) {
        if (!_.isFunction(matcher)) { matcher = _.isEqual; }
        $(buttonNodes).each(function () {
            var state = matcher(value, Forms.getButtonValue(this));
            if (_.isBoolean(state)) { Forms.selectButtonNodes(this, state); }
        });
    };

    /**
     * Adds a keyboard event handler to the passed button elements, which
     * listens to the ENTER and SPACE keys, and triggers 'click' events at the
     * target node of the received event. The event object will contain the
     * additional property 'keyCode' set to the correct key code. Bubbling of
     * the original keyboard events ('keydown', 'keypress', and 'keyup') will
     * be suppressed, as well as the default browser action. The 'click' event
     * will be triggered after receiving the 'keyup' event.
     *
     * @param {HTMLElement|jQuery} nodes
     *  The nodes to bind the event handles to. May contain button elements, or
     *  any other container elements with descendant button elements.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Boolean} [options.tab=false]
     *      If set to true, 'click' events will also be triggered for the TAB
     *      key and the SHIFT+TAB key combination (no other modifier keys must
     *      be pressed).
     */
    Forms.setButtonKeyHandler = function (nodes, options) {

        var // whether to listen to the TAB key
            tab = Utils.getBooleanOption(options, 'tab', false);

        // Bug 28528: ENTER key must be handled explicitly, <a> elements without
        // 'href' attribute do not trigger click events. The 'href' attribute
        // has been removed from the buttons to prevent useless tooltips with the
        // link address. Additionally, SPACE key must always be handled manually.

        // handler for the 'keyup' events
        function keyHandler(event) {

            // ENTER and SPACE key: wait for keyup event
            if ((event.keyCode === KeyCodes.ENTER) || (event.keyCode === KeyCodes.SPACE)) {
                if (event.type === 'keyup') {
                    Forms.triggerClickForKey(event);
                }
                return false; // stop propagation of ALL keyboard events
            }

            if (tab && (event.type === 'keydown') && KeyCodes.matchKeyCode(event, 'TAB', { shift: null })) {
                Forms.triggerClickForKey(event);
                // let all TAB key events bubble up for focus navigation
            }
        }

        // directly bind the event handler to all button nodes
        $(nodes).filter(Forms.BUTTON_SELECTOR).on('keydown keypress keyup', keyHandler);
        // bind the event handler to all descendant button nodes
        $(nodes).not(Forms.BUTTON_SELECTOR).on('keydown keypress keyup', Forms.BUTTON_SELECTOR, keyHandler);
    };

    // text input elements ----------------------------------------------------

    /**
     * Creates and returns the HTML mark-up for an <input> element.
     *
     * @param {Object} [options]
     *  Optional parameters. Supports all options supported by the method
     *  Forms.createElementMarkup(). Additionally, the following options are
     *  supported:
     *  @param {String} [options.placeholder='']
     *      A place holder text that will be shown in an empty text field.
     *  @param {String} [options.keyboard='text']
     *      Specifies which virtual keyboard should occur on touch devices.
     *      Supported types are 'text' (default) for a generic text field with
     *      alphanumeric keyboard, 'number' for floating-point numbers with
     *      numeric keyboard, 'url' for an alphanumeric keyboard with additions
     *      for entering URLs, or 'email' for an alphanumeric keyboard with
     *      additions for entering e-mail addresses.
     *  @param {Number} [options.maxLength]
     *      If specified and a positive number, sets the 'maxLength' attribute
     *      of the text field, used to restrict the maximum length of the text
     *      that can be inserted into the text field.
     *
     * @returns {jQuery}
     *  A jQuery object containing the new text field element.
     */
    Forms.createInputMarkup = function (options) {

        var type = Modernizr.touch ? Utils.getStringOption(options, 'keyboard', 'text') : 'text',
            attributes = Utils.getObjectOption(options, 'attributes', {}),
            placeHolder = Utils.getStringOption(options, 'placeholder'),
            maxLength = Utils.getIntegerOption(options, 'maxLength');

        // add more element attributes, return the mark-up
        attributes = _.extend({ type: type, tabindex: 1 }, attributes);
        if (_.isString(placeHolder) && (placeHolder.length > 0)) { attributes.placeholder = placeHolder; }
        if (_.isNumber(maxLength) && (maxLength > 0)) { attributes.maxlength = maxLength; }
        return Forms.createElementMarkup('input', Utils.extendOptions(options, { attributes: attributes, content: null }));
    };

    /**
     * Creates and returns the HTML mark-up for a <textarea> element.
     *
     * @param {Object} [options]
     *  Optional parameters. Supports all options supported by the method
     *  Forms.createElementMarkup(). Additionally, the following options are
     *  supported:
     *  @param {String} [options.placeholder='']
     *      A place holder text that will be shown in an empty text area.
     *  @param {Number} [options.maxLength]
     *      If specified and a positive number, sets the 'maxLength' attribute
     *      of the text area, used to restrict the maximum length of the text
     *      that can be inserted into the text area.
     *
     * @returns {jQuery}
     *  A jQuery object containing the new text area element.
     */
    Forms.createTextAreaMarkup = function (options) {

        var attributes = Utils.getObjectOption(options, 'attributes', {}),
            placeHolder = Utils.getStringOption(options, 'placeholder'),
            maxLength = Utils.getIntegerOption(options, 'maxLength');

        // add more element attributes, return the mark-up
        attributes = _.extend({ tabindex: 1 }, attributes);
        if (_.isString(placeHolder) && (placeHolder.length > 0)) { attributes.placeholder = placeHolder; }
        if (_.isNumber(maxLength) && (maxLength > 0)) { attributes.maxlength = maxLength; }
        return Forms.createElementMarkup('textarea', Utils.extendOptions(options, { attributes: attributes, content: null }));
    };

    /**
     * Returns the current selection in the passed text input control.
     *
     * @param {HTMLElement|jQuery} inputNode
     *  A text input element (an HTML <input> or <textarea> element). If this
     *  object is a jQuery collection, used the first DOM node it contains.
     *
     * @returns {Object}
     *  An object with the properties 'start' and 'end' containing the start
     *  and end character offset of the selection in the input control.
     */
    Forms.getInputSelection = function (inputNode) {
        inputNode = Utils.getDomNode(inputNode);
        try {
            return { start: inputNode.selectionStart, end: inputNode.selectionEnd };
        } catch (ex) {
            // Bug 31286: Mobile Chrome: input elements of type 'number' do not support selection
            return { start: inputNode.value.length, end: inputNode.value.length };
        }
    };

    /**
     * Changes the current selection in the passed text input control.
     *
     * @param {HTMLElement|jQuery} inputNode
     *  A text input element (an HTML <input> or <textarea> element). If this
     *  object is a jQuery collection, used the first DOM node it contains.
     *
     * @param {Number} start
     *  The start character offset of the new selection in the input control.
     *
     * @param {Number} [end=start]
     *  The end character offset of the new selection in the input control. If
     *  omitted, sets a text cursor according to the passed start position.
     */
    Forms.setInputSelection = function (inputNode, start, end) {
        inputNode = Utils.getDomNode(inputNode);
        try {
            inputNode.setSelectionRange(start, _.isNumber(end) ? end : start);
        } catch (ex) {
            // Bug 31286: Mobile Chrome: input elements of type 'number' do not support selection
        }
    };

    /**
     * Replaces the current selection of the text input control with the
     * specified text, and places a simple text cursor behind the new text.
     *
     * @param {HTMLElement|jQuery} inputNode
     *  A text input element (an HTML <input> or <textarea> element). If this
     *  object is a jQuery collection, used the first DOM node it contains.
     *
     * @param {String} [text='']
     *  The text used to replace the selected text in the input control. If
     *  omitted, the selection will simply be deleted.
     */
    Forms.replaceTextInInputSelection = function (inputNode, text) {

        var node = Utils.getDomNode(inputNode),
            start = node.selectionStart;

        text = _.isString(text) ? text : '';
        node.value = Utils.replaceSubString(node.value, start, node.selectionEnd, text);
        node.setSelectionRange(start + text.length, start + text.length);
    };

    // exports ================================================================

    return Forms;

});
