/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/tk/control/label',
    ['io.ox/office/tk/utils',
     'io.ox/office/tk/forms',
     'io.ox/office/tk/control/group',
     'io.ox/office/tk/control/captionmixin',
     'io.ox/office/tk/control/widthmixin'
    ], function (Utils, Forms, Group, CaptionMixin, WidthMixin) {

    'use strict';

    // class Label ============================================================

    /**
     * Creates a label control listening to update requests. By registering an
     * update handler that modifies the caption it is even possible to update
     * the label dynamically with the method 'Group.setValue()' based on any
     * arbitrary value.
     *
     * @constructor
     *
     * @extends Group
     * @extends CaptionMixin
     * @extends WidthMixin
     *
     * @param {Object} [initOptions]
     *  Optional parameters. Supports all options of the Group base class
     *  constructor, all label formatting options supported by the method
     *  Forms.createLabelMarkup(), and all options of the mix-in class
     *  WidthMixin. Additionally, the following options are supported:
     *  @param {Object} [initOptions.smallerVersion]
     *      If specified, the label will known, how it should act, if there
     *      will be not enough free place to show normal view
     */
    function Label(initOptions) {

        var // self reference
            self = this,

            // the <label> element
            labelNode = $(Forms.createLabelMarkup(initOptions)),

            // smaller version of label (to fit in smaller resolutions)
            smallerVersion = Utils.getOption(initOptions, 'smallerVersion', false),

            // saves the (old) styles in case of switching to small version
            savedStyle = null;

        // base constructor ---------------------------------------------------

        Group.call(this, initOptions);
        CaptionMixin.call(this, labelNode);
        WidthMixin.call(this, labelNode, initOptions);

        // private methods ----------------------------------------------------

        /**
         * Activates/Deactivates a smaller version of the label
         *
         * @param {Boolean} value
         *  decides wether the label should be small or wide
         */
        function smallLabel(value){
            // show the small version
            if (value === true) {
                // set new css if exists
                if (_.isObject(smallerVersion.css)) {
                    savedStyle = labelNode.attr('style');
                    labelNode.removeAttr('style');
                    labelNode.css(smallerVersion.css);
                }

                // hide text of label (if set)
                if (smallerVersion.hideLabel) { labelNode.addClass('hideLabel'); }

                // hide complete label
                if (smallerVersion.hide) { self.getNode().addClass('hidden'); }

            // show the default
            } else {
                // re-set the old (saved) styles
                if (!_.isNull(savedStyle)) {
                    labelNode.removeAttr('style');
                    labelNode.attr('style', savedStyle);
                    savedStyle = null;
                }

                // show text of label (if set)
                if (smallerVersion.hideLabel) { labelNode.removeClass('hideLabel'); }

                // hide complete label
                if (smallerVersion.hide) { self.getNode().removeClass('hidden'); }
            }
        }



        // public methods -----------------------------------------------------

        /**
         * Returns the DOM label element, as jQuery object.
         *
         * @returns {jQuery}
         *  The DOM label element, as jQuery object.
         */
        this.getLabelNode = function () {
            return labelNode;
        };

        /**
         * Overwrites the base-methods (group.js) to
         * activate/deactivate the small version of the button
         */
        this.activateSmallVersion = function(){
            if (_.isObject(smallerVersion)) {
                smallLabel(true);
            }
        };
        this.deactivateSmallVersion = function(){
            if (_.isObject(smallerVersion)) {
                smallLabel(false);
            }
        };

        // initialization -----------------------------------------------------

        this.getNode().addClass('label-group');

        // insert the label into this group
        this.addChildNodes(labelNode);

    } // class Label

    // exports ================================================================

    // derive this class from class Group
    return Group.extend({ constructor: Label });

});
