/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define('io.ox/office/text/changeTrack',
    ['io.ox/office/tk/utils',
     'io.ox/office/tk/dialogs',
     'io.ox/office/tk/object/triggerobject',
     'io.ox/office/drawinglayer/view/drawingframe',
     'io.ox/office/editframework/utils/attributeutils',
     'io.ox/office/text/dom',
     'io.ox/office/text/operations',
     'io.ox/office/text/position',
     'io.ox/office/text/table',
     'gettext!io.ox/office/text'
    ], function (Utils, Dialogs, TriggerObject, DrawingFrame, AttributeUtils, DOM, Operations, Position, Table, gt) {

    'use strict';

    // private static functions ===============================================

    var // the allowed changeTrack types (order is relevant)
        CHANGE_TRACK_TYPES = ['inserted', 'removed', 'modified'],
        // the allowed changeTrack information saved at nodes
        CHANGE_TRACK_INFOS = ['author', 'date', 'uid'],
        // clip board operations, that will be expanded with change tracking information (explicitly not setAttributes)
        CHANGETRACK_CLIPBOARD_OPERATIONS = [Operations.TEXT_INSERT,
                                            Operations.DRAWING_INSERT,
                                            Operations.FIELD_INSERT,
                                            Operations.TAB_INSERT,
                                            Operations.HARDBREAK_INSERT,
                                            Operations.PARA_INSERT,
                                            Operations.ROWS_INSERT,
                                            Operations.TABLE_INSERT],
        // whether the resizing of a drawing needs to be change tracked
        CHANGE_TRACK_SUPPORTS_DRAWING_RESIZING = false,
        // whether the assignment of a position to a drawing needs to be change tracked
        CHANGE_TRACK_SUPPORTS_DRAWING_POSITION = false;

    // class ChangeTrack ========================================================

    /**
     * An instance of this class represents a change track handler in the edited document.
     *
     * @constructor
     *
     * @extends TriggerObject
     *
     * @param {TextApplication} app
     *  The application instance.
     *
     * @param {HTMLElement|jQuery} rootNode
     *  The root node of the document. If this object is a jQuery collection,
     *  uses the first node it contains.
     *
     * @param {TextModel} model
     *  The text model instance.
     */
    function ChangeTrack(app, rootNode, model) {

        var // self reference
            self = this,
            // a list of all change track authors
            listOfAuthors = [],
            // the current changeTrack date
            changeTrackInfoDate = null,
            // the local user id of the current user
            userid = null,
            // the promise for the repeated date update process
            dateUpdatePromise = null,
            // the period of time, that an update of change track time happens repeatedly (in ms).
            updateDateDelay = 30000,
            // the change track selection object
            // -> activeNode: HTMLElement, the one currently node used by the iterator
            // -> activeNodeType: String, the one currently used node type, one of 'inserted', 'removed', 'modified'
            // -> activeNodeTypesOrdered: String[], ordered list of strings, corresponding to priority of node types
            // -> selectedNodes: HTMLElement[], sorted list of HTMLElements of the group belonging to 'activeNode' and 'activeNodeType
            // -> allChangeTrackNodes: jQuery, optional, cache for performance reasons, contains all change track elements in the document
            changeTrackSelection = null,
            // a collector for additional clipboard operations
            clipboardCollector = [],
            // css class for showing change track nodes
            highlightClassname = 'change-track-highlight',
            // whether the handler for the change track side bar are active or not
            sideBarHandlerActive = false,
            // the node for the change track side bar
            sideBarNode = null,
            // the node containing the scroll bar
            scrollNode = null,
            // the node with the page content
            pageContentNode = null,
            // collector for all change tracked nodes (required only for the side bar)
            allNodesSideBarCollector = null,
            // a collector for all markers in the change track side bar
            allSideBarMarker = {};


        // base constructor ---------------------------------------------------

        TriggerObject.call(this);

        // private methods ----------------------------------------------------

        /**
         * Returns all tracking nodes inside the selection. If the selection
         * has no range, return only the currently selected node.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.minimalSelection=false]
         *      If set to true and the selection is a range, the iteration
         *      stops after finding the first change track node. If the
         *      selection is not a range, this parameter will be ignored.
         *  @param {Boolean} [options.markLastParagraph=false]
         *      If set to true and the selection is a range, the final paragraph
         *      of the selection will be marked, if it is not completely part of
         *      the selection. In this case the 'inserted' and 'removed' change
         *      tracks must be resolved, but not the 'modified' change track.
         *  @param {Boolean} [options.split=false]
         *      If set to true the spans that are only partially covered by
         *      the selection, are split during the iteration. As a result
         *      all nodes in the collector are covered completely.
         *
         * @returns {jQuery}
         *  A list of nodes in a jQuery element. The list is sorted corresponding
         *  to the appearance of the nodes in the DOM.
         */
        function getTrackingNodesInSelection(options) {

            var // the tracking nodes in the current selection
                trackingNodes = $(),
                // whether it is sufficient to find only one change tracking node
                minimalSelection = Utils.getBooleanOption(options, 'minimalSelection', false),
                // whether the final paragraph of a selection shall be marked in that way, that
                // the 'modified' change track attribute shall not be resolved. This is the case
                // if this final paragraph is not completely included into the selection.
                markLastParagraph = Utils.getBooleanOption(options, 'markLastParagraph', false),
                // whether spans that are only partially covered by the selection, need to be split
                split = Utils.getBooleanOption(options, 'split', false),
                // a container with all paragraphs. This is necessary to check, if the last paragraph
                // shall be added to the selection
                paragraphsCollector = [];

            // all tracking nodes that are parents, grandparents, ... need to be added
            // to the list of all tracking nodes. Otherwise only text nodes are collected.
            function collectParagraphs(paraNode) {
                if (_.contains(paragraphsCollector, paraNode)) { return; }  // paragraph already checked
                paragraphsCollector.push(paraNode); // collecting each paragraph only once
            }

            // checking, if the final paragraph needs to be added to the change track collector.
            // This is necessary, if it was 'inserted' or 'removed', and is not completely included
            // into the selection.
            function checkFinalParagraph() {

                if (paragraphsCollector.length === 0) { return; }

                var // the final paragraph
                    lastParagraph = paragraphsCollector[paragraphsCollector.length - 1],
                    // an array with the tracking nodes (used to find and insert the paragraph node
                    trackingNodeArray = trackingNodes.get(),
                    // the change tracked children of the last paragraph
                    paraChildren = null,
                    // the first change tracked child of the last paragraph
                    firstChild = null,
                    // the position of the first child in the list of change tracked nodes
                    firstChildPosition = null;

                // is the last paragraph a change tracked node ('inserted' or 'removed' that is not
                // already part of the collected tracking nodes?
                // -> searching from end, because it probably needs to be added to the end
                if (lastParagraph && (DOM.isChangeTrackInsertNode(lastParagraph) || DOM.isChangeTrackRemoveNode(lastParagraph)) && (trackingNodeArray.lastIndexOf(lastParagraph) === -1)) {
                    // the paragraph must be inserted into the list of tracking nodes
                    // -> at a position before its children or at the end, if none of its children
                    // are part of the list
                    paraChildren = getAllChangeTrackNodes(lastParagraph);

                    if (paraChildren.length > 0) {
                        firstChild = paraChildren[0];
                        firstChildPosition = trackingNodeArray.lastIndexOf(firstChild);
                        if (firstChildPosition === -1) {
                            trackingNodes = trackingNodes.add(lastParagraph);  // Simply adding to the end
                        } else {
                            trackingNodeArray.splice(firstChildPosition, 0, lastParagraph);  // inserting last paragraph before its first child
                            trackingNodes = $(trackingNodeArray); // converting back to jQuery list
                        }

                    } else {
                        trackingNodes = trackingNodes.add(lastParagraph);  // Simply adding to the end
                    }

                    // setting the 'skipModify' data attribute, so that 'modify' will not be resolved/rejected
                    if (markLastParagraph) { $(lastParagraph).data('skipModify', true); }
                }
            }

            if (model.getSelection().hasRange()) {

                // iterating over all nodes in the selection (using option split
                // to split partially covered spans). Using 'shortestPath: true' it is possible to check,
                // if complete paragraphs or tables are part of the selection. In this case the change
                // tracked children can be found with the 'find' function.
                model.getSelection().iterateNodes(function (node) {

                    var // a collector for all change tracked children of a content node
                        changeTrackChildren = null,
                        // the parent node of a text span
                        parentNode = null;

                    // in text component nodes the span has the change track attribute, not the
                    // text component itself. But collector must contain the text component.
                    function getValidChangeTrackNode(node) {
                        return DOM.isTextComponentNode(node.parentNode) ? node.parentNode : node;
                    }

                    // collecting all paragraphs in a collector
                    if (DOM.isTextSpan(node)) {
                        parentNode = Utils.getDomNode(node).parentNode;
                        collectParagraphs(parentNode);
                    }

                    if (DOM.isChangeTrackNode(node)) {
                        trackingNodes = trackingNodes.add(node);
                    }

                    if (minimalSelection && trackingNodes.length > 0) { return Utils.BREAK; }  // exit as early as possible

                    // collecting all children inside the content node
                    if (DOM.isContentNode(node)) {
                        changeTrackChildren = getAllChangeTrackNodes(node);

                        if (changeTrackChildren.length > 0) {
                            _.each(changeTrackChildren, function (node) {
                                trackingNodes = trackingNodes.add(getValidChangeTrackNode(node));
                            });
                        }
                    }

                    if (minimalSelection && trackingNodes.length > 0) { return Utils.BREAK; }  // exit as early as possible

                }, null, { split: split, shortestPath: true });

                // special handling for the final paragraph, that needs to be inserted into the tracking
                // node collection, if it was 'inserted' or 'removed' AND the previous paragraph is also
                // added to the paragraph collector.
                checkFinalParagraph();

                // TODO: grouping of cells to columns? In which scenario is this necessary? Tables are always
                // completely part of the selection or not. Otherwise only paragraphs are completely evaluated.

            } else {
                trackingNodes = findChangeTrackNodeAtCursorPosition();
            }

            return trackingNodes;
        }

        /**
         * Returns the currently selected tracking node at the cursor position or at the
         * start position of a selection range. If there is no change tracking element, an
         * empty jQuery list is returned.
         *
         * @returns {jQuery}
         *  A list of selected nodes in a jQuery element.
         */
        function findChangeTrackNodeAtCursorPosition() {

            var // the node info for the start position of the current selection
                nodeInfo = Position.getDOMPosition(rootNode, model.getSelection().getStartPosition()),
                // the node at the cursor position
                currentNode = nodeInfo ? nodeInfo.node : null,
                // the offset at the cursor position
                offset = nodeInfo ? nodeInfo.offset : null,
                // a helper node for empty text spans
                emptySpan = null,
                // the currently selected tracking node
                trackingNode = null;

            // fast exit, if the node could not be determined
            if (!currentNode) { return; }

            // special handling for drawings
            if (model.isDrawingSelected()) { return model.getSelection().getSelectedDrawing(); }

            // special handling for empty text nodes between inline nodes, for example div elements
            if ((currentNode.nodeType === 3) && currentNode.parentNode && DOM.isEmptySpan(currentNode.parentNode)) {
                emptySpan = currentNode.parentNode;
                // checking for previous or following
                if (emptySpan.previousSibling && DOM.isChangeTrackNode(emptySpan.previousSibling)) {
                    trackingNode = $(emptySpan.previousSibling);
                } else if (emptySpan.nextSibling && DOM.isChangeTrackNode(emptySpan.nextSibling)) {
                    trackingNode = $(emptySpan.nextSibling);
                }
            }

            trackingNode = trackingNode || $(currentNode).closest(DOM.CHANGETRACK_NODE_SELECTOR);  // standard process to find change track node

            if ((trackingNode.length === 0) && (offset === 0) && currentNode.parentNode.previousSibling && DOM.isChangeTrackNode(currentNode.parentNode.previousSibling)) {
                // this happens behind a div.inline component
                trackingNode = $(currentNode.parentNode.previousSibling);
            }

            if ((trackingNode.length === 0) && (DOM.isEmptySpan(currentNode.parentNode) || ((currentNode.nodeType === 3) && (currentNode.length === offset))) && DOM.isChangeTrackNode(currentNode.parentNode.nextSibling)) {
                // this happens before a change tracked component, div.inline or text span (special handling for empty spans between two tabs)
                trackingNode = $(currentNode.parentNode.nextSibling);
            }

            return trackingNode;
        }

        /**
         * Returns for a given list of nodes an expanded list, that contains
         * all the given nodes and additionally all nodes affected by the
         * change track process.
         * Using a jQuery element as collection, keeps all nodes unique.
         *
         * @param {HTMLElement[]|jQuery} nodes
         *  A list of all nodes, for that the adjacent nodes shall be detected
         *
         * @param {String} [type]
         *  The type of resolving ('inserted', 'removed' or 'modified'). I not specified,
         *  all types will be evaluated.
         *
         * @returns {jQuery}
         *  A list of nodes in a jQuery element.
         */
        function groupTrackingNodes(nodes, type) {

            var // the jQuery collector for all grouped nodes
                allNodes = $(),
                // whether all types shall be resolved (if 'type' is not defined)
                allResolveTypes = type ? false : true,
                // a list with all resolve types
                resolveTypeList = allResolveTypes ? CHANGE_TRACK_TYPES : [type];

            _.each(nodes, function (node) {
                _.each(resolveTypeList, function (oneType) {
                    var combinedNodes = combineAdjacentNodes(node, { type: oneType });
                    allNodes = allNodes.add(combinedNodes);
                });
            });

            return allNodes;
        }

        /**
         * Starting the process for updating the change track info date
         * regularly. This is needed for performance reasons, so that
         * 'self.updateChangeTrackInfoDate()' is not called for every
         * operation.
         */
        function startDateUpdate() {
            if (!dateUpdatePromise) {
                self.updateChangeTrackInfoDate();  // updating once immediately
                dateUpdatePromise = app.repeatDelayed(self.updateChangeTrackInfoDate, { delay: updateDateDelay });
            }
        }

        /**
         * Stopping the process for updating the change track info date
         * regularly. This is needed for performance reasons, so that
         * 'self.updateChangeTrackInfoDate()' is not called for every
         * operation.
         */
        function stopDateUpdate() {
            if (dateUpdatePromise) {
                dateUpdatePromise.abort();
                dateUpdatePromise = null;
            }
        }

        /**
         * Collecting all change tracking nodes in the document
         *
         * @param {HTMLElement|jQuery} [node]
         *  The node, in which the change track nodes are searched. If
         *  not specified, the rootNode is used.
         *  If this object is a jQuery collection, uses the first node
         *  it contains.
         *
         * @returns {jQuery}
         *  A list of change track nodes in the document.
         */
        function getAllChangeTrackNodes(node) {

            var // the node, in which the change track nodes are searched
                searchNode = node || rootNode;

            return $(searchNode).find(DOM.CHANGETRACK_NODE_SELECTOR);
        }

        /**
         * Checking, whether the specified change track selection is a column
         * selection. This is the case if the change track type is 'inserted'
         * and 'removed' and only table cell elements are part of the selection.
         *
         * @param {Object} selection
         *  The change track selection that will be investigated.
         *
         * @returns {Boolean}
         *  Whether the specified change track selection is a column selection.
         */
        function checkColumnSelection(selection) {

            var // a helper node
                noCellNode = null;

            // checking node type
            if (selection.activeNodeType === 'modified') { return false; }

            // checking the selected nodes: A column selection must have 2 cells at least
            if ((!selection.selectedNodes) || (selection.selectedNodes.length < 2)) { return false; }

            // checking, that all members of selected nodes are cells
            noCellNode = _.find(selection.selectedNodes, function (oneNode) {
                return !DOM.isTableCellNode(oneNode);
            });

            if (noCellNode) { return false; }

            return true;
        }

        /**
         * Calculating an internal user id, that is saved at the node as attribute
         * and saved in the xml file. The base for this id is the app server id.
         *
         * @param {Number} id
         *  The user id used by the app suite server.
         *
         * @returns {Number}
         *  The user id, that is saved in xml file and saved as node attribute.
         */
        function calculateUserId(id) {
            return _.isNumber(id) ? (id + 123) * 10 - 123 : null;
        }

        /**
         * Calculating the app suite user id from the saved and transported user id.
         *
         * @param {Number} id
         *  The user id, that is saved in xml file and saved as node attribute.
         *
         * @returns {Number}
         *  The user id used by the app suite server.
         */
        function resolveUserId(id) {
            return _.isNumber(id) ? ((id + 123) / 10 - 123) : null;
        }

        /**
         * Setting a change track selection. The new change track selection must be
         * given to this function. The text selection is set to the first logical
         * position of the active node of the change track selection. So it must
         * not be possible, that a text selection is visible next to a change track
         * selection.
         * Before setting the new change track selection, an old change track
         * selection is cleared.
         *
         * @param {Object} selection
         *  The change track selection that will be set.
         *
         * @param {Object} [options]
         *  A map with additional options controlling the change track selection.
         *  The following options are supported:
         *  @param {Boolean} [options.makeVisible=true]
         *      Whether the change track selection shall be made visible immediately.
         *  @param {Boolean} [keepChangeTrackPopup=false]
         *      Whether this change of change track selection was triggered by the change
         *      track pop up. In this case the resulting change of text selection must
         *      not lead to an update of the change track pop up.
         */
        function setChangeTrackSelection(selection, options) {

            var // whether the change track selection shall be made visible
                makeVisible = Utils.getBooleanOption(options, 'makeVisible', true),
                // whether the change track pop up triggered this change of selection
                keepChangeTrackPopup = Utils.getBooleanOption(options, 'keepChangeTrackPopup', false),
                // whether this change track selection is a column selection (in this case the text selection must be a cursor selection)
                isColumnSelection = checkColumnSelection(selection),
                // the logical start position of the change track selection
                textStartPosition = Position.getTextLevelPositionInNode(_.first(selection.selectedNodes), rootNode),
                // the logical end position of the change track selection
                textEndPosition = isColumnSelection ? null : Position.getTextLevelPositionInNode(_.last(selection.selectedNodes), rootNode, { lastPosition: true });

            // removing color of existing selection
            self.clearChangeTrackSelection();

            // assigning the new selection
            changeTrackSelection = selection;

            // highlight change track nodes
            if (makeVisible) {
                _.each(changeTrackSelection.selectedNodes, function (node) {
                    // highlight table nodes only
                    if (DOM.isTableNode(node) || DOM.isTableRowNode(node) || DOM.isTableCellNode(node)) { $(node).addClass(highlightClassname); }
                });
            }

            // adapting the text selection to the change track selection
            model.getSelection().setTextSelection(textStartPosition, textEndPosition,  { keepChangeTrackPopup: keepChangeTrackPopup });
        }

        /**
         * Finding the next or following change tracking node from the current text selection (not
         * change track selection!).
         *
         * @param {jQuery} allNodes
         *  A jQuery list with all change track nodes in the document.
         *
         * @param {Object} [options]
         *  A map with additional options controlling the search of the change track node.
         *  The following options are supported:
         *  @param {Boolean} [options.next=true]
         *      Whether the following or the previous change track node shall be found.
         *  @param {Boolean} [options.useDocumentBorder=false]
         *      Whether the change track node shall be found starting the search from the beginning
         *      (for next = true) or the end (for next = false) of the document.
         *
         * @returns {HTMLElement|Null}
         *  The next or the previous change track node in relation to the current cursor position.
         *  If this element is a span inside an in-line component, the in-line component itself
         *  must be returned.
         */
        function getNextChangeTrackNode(allNodes, options) {

            var // whether the following valid change track shall be found or the previous
                next = Utils.getBooleanOption(options, 'next', true),
                // whether the search shall be started at the beginning or end of the document
                useDocumentBorder = Utils.getBooleanOption(options, 'useDocumentBorder', false),
                // start searching at the current start position of the selection
                name = next ? 'findNextNode' : 'findPreviousNode',
                // the start position from which the neighboring change track node shall be searched
                startPosition = useDocumentBorder ? (next ? model.getSelection().getFirstDocumentPosition() : model.getSelection().getLastDocumentPosition()) : model.getSelection().getStartPosition(),
                // the node currently selected by the text cursor (or at the beginning of an arbitrary selection range)
                node = Position.getDOMPosition(rootNode, startPosition).node,
                // the change tracked node for the selection (must not be a span inside an in-line component
                nextNode = null;

            if (!node) { return null; }

            // replacing text nodes by its parent text span
            if (node.nodeType === 3) {
                node = node.parentNode;
            }

            nextNode = Utils[name](rootNode, node, DOM.CHANGETRACK_NODE_SELECTOR, null);

            // the next node for the selection must not be the span inside an in-line component. In this
            // case it must be the in-line component itself.
            if (nextNode && nextNode.parentNode && DOM.isTextComponentNode(nextNode.parentNode)) {
                nextNode = nextNode.parentNode;
            }

            return nextNode;
        }

        /**
         * Creating and returning an ordered list of change track types for a specified node.
         *
         * @param {Node|jQuery|Null} node
         *  The DOM node to be checked. If this object is a jQuery collection, uses
         *  the first DOM node it contains. If missing or null, returns false.
         *
         * @returns {Any[]}
         *  The ordered list of change track types for the given node. This can be
         *  an array of strings are an empty array.
         */
        function getOrderedChangeTrackTypes(node) {

            var // the different change track node types ordered by priority
                nodeTypeOrder = [];

            // Info: If a change track is 'inserted' and 'removed', the removal has to happen
            // AFTER the insertion. Therefore removal is always the more relevant tracking
            // type, because it happened after the insertion.

            if (DOM.isChangeTrackRemoveNode(node)) {
                nodeTypeOrder.push('removed');
            }

            if (DOM.isChangeTrackInsertNode(node)) {
                nodeTypeOrder.push('inserted');
            }

            if (DOM.isChangeTrackModifyNode(node)) {
                nodeTypeOrder.push('modified');
            }

            return nodeTypeOrder;
        }

        /**
         * Creating and returning a new selection object describing a change track selection.
         * This object requires the elements 'activeNode', 'activeNodeType', 'activeNodeTypesOrdered',
         * 'selectedNodes' and optionally 'allChangeTrackNodes'.
         *
         * @param {HTMLElement} activeNode
         *  The HTMLElement of the change track selection. Must be the first element of all elements
         *  in the selection.
         *
         * @param {String} activeNodeType
         *  The change track type of the active node. Must be one of the values defined in
         *  CHANGE_TRACK_TYPES.
         *
         * @param {String[]} activeNodeTypesOrdered
         *  An ordered list of strings of node types defined in CHANGE_TRACK_TYPES. The order
         *  corresponds to the priority of the change tracking types for the active node.
         *
         * @param {HTMLElement[]} selectedNodes
         *  An ordered list of all HTML elements, that are combined in the current change track
         *  selection.
         *
         * @param {jQuery} [allChangeTrackNodes]
         *  An optional cache that contains all change track elements in the document. This is
         *  saved in the selection for performance reasons.
         *
         * @returns {Object}
         *  An object, describing the change track selection.
         */
        function createChangeTrackSelection(activeNode, activeNodeType, activeNodeTypesOrdered, selectedNodes, allChangeTrackNodes) {

            var // the new change track selection object
                selection = {};

            selection.activeNode = activeNode;
            selection.activeNodeType = activeNodeType;
            selection.activeNodeTypesOrdered = activeNodeTypesOrdered;
            selection.selectedNodes = selectedNodes;
            selection.allChangeTrackNodes = allChangeTrackNodes;  // this can be 'undefined'

            return selection;
        }

        /**
         * Finding a valid change track selection. This can start with an existing
         * change track selection or without such a selection at any text selection.
         *
         * @param {Object} [options]
         *  A map with additional options controlling the search of a valid change
         *  track selection.
         *  The following options are supported:
         *  @param {Boolean} [options.next=true]
         *      Whether the following or the previous change track selection node shall be found.
         *
         * @returns {Object|Null}
         *  The next or the previous change track selection or Null, if no valid change track selection
         *  can be determined.
         */
        function iterateNextValidChangeTrackSelection(options) {

            var // whether the following valid change track shall be found or the previous
                next = Utils.getBooleanOption(options, 'next', true),
                // whether the iteration shall be continued
                continueIteration = true,
                // whether a valid change track selection was found
                foundValidChangeTrackSelection = false,
                // the active node of a change track selection
                activeNode = null,
                // an ordered list of change track types for the active node
                activeNodeTypesOrdered = null,
                // the change track type for the active node
                activeNodeType = null,
                // the ordered array of selected nodes neighboring the active node and its change track type
                selectedNodes = null,
                // the list of all change track nodes in a jQuery list
                allChangeTrackNodes = null,
                // a helper object to create a change track selection
                localChangeTrackSelection = null;

            // helper function to find next or previous change track node type in ordered list of change track type.
            // If there is no previous or following change track type in the list, null is returned.
            function getNextNodeType() {

                var index = null,
                    newIndex = null;

                if (next && (_.last(activeNodeTypesOrdered) === activeNodeType)) { return null; }

                if (!next && (_.first(activeNodeTypesOrdered) === activeNodeType)) { return null; }

                index = _.indexOf(activeNodeTypesOrdered, activeNodeType);

                newIndex = next ? (index + 1) : (index - 1);

                return activeNodeTypesOrdered[newIndex];
            }

            // helper function to find next or previous change track node in jQuery list of all change track nodes.
            // If there is no previous or following change track node in the jQuery list, null is returned.
            function getNextChangeTrackingNodeInListOfChangeTrackingNodes() {

                var // the index of the node inside the jQuery collection of all change track nodes
                    index = null,
                    // the modified new index
                    newIndex = null,
                    // a helper node
                    checkNode = activeNode;

                // receiving the node from list of all nodes at specified index
                // -> for text component nodes the 'div' element must be returned, not the span inside
                function getNodeAtIndex(index) {

                    if (DOM.isTextComponentNode(Utils.getDomNode(allChangeTrackNodes[index]).parentNode)) {
                        return Utils.getDomNode(allChangeTrackNodes[index]).parentNode;
                    } else {
                        return allChangeTrackNodes[index];
                    }

                }

                // the active node can be an in-line component, but the list of all change tracks contains
                // the span inside the in-line components.
                if (DOM.isTextComponentNode(checkNode)) { checkNode = Utils.getDomNode(checkNode).firstChild; }

                // searching the current node inside the list of all change track nodes.
                index = allChangeTrackNodes.index(checkNode);

                if (next && (index === allChangeTrackNodes.length - 1)) { return getNodeAtIndex(0); }

                if (!next && (index === 0)) { return getNodeAtIndex(allChangeTrackNodes.length - 1); }

                newIndex = next ? (index + 1) : (index - 1);

                return getNodeAtIndex(newIndex);
            }

            if (changeTrackSelection) {

                // there is an existing change track selection -> using this selection as start point for new selection
                if (! changeTrackSelection.allChangeTrackNodes) { changeTrackSelection.allChangeTrackNodes = getAllChangeTrackNodes(); }

                allChangeTrackNodes = changeTrackSelection.allChangeTrackNodes;
                activeNode = changeTrackSelection.activeNode;
                activeNodeType = changeTrackSelection.activeNodeType;
                activeNodeTypesOrdered = changeTrackSelection.activeNodeTypesOrdered;

                // first iteration step for existing change track selection
                activeNodeType = getNextNodeType();
                if (!activeNodeType) {
                    // finding the next change tracking node
                    activeNode = getNextChangeTrackingNodeInListOfChangeTrackingNodes();
                    if (!activeNode) { return null; }
                    activeNodeTypesOrdered = getOrderedChangeTrackTypes(activeNode);
                    activeNodeType = next ? _.first(activeNodeTypesOrdered) : _.last(activeNodeTypesOrdered);
                }

            } else {
                // There is no existing change track selection
                // -> collecting all change tracking nodes in the document
                allChangeTrackNodes = getAllChangeTrackNodes();
                // there is no existing change track selection -> finding the next or previous change track node
                activeNode = getNextChangeTrackNode(allChangeTrackNodes, {next: next});
                if (!activeNode) {
                    // there is the chance to find a change track element, starting from beginning or end of document
                    activeNode = getNextChangeTrackNode(allChangeTrackNodes, {next: next, useDocumentBorder: true });
                    if (!activeNode) { return null; }
                }
                activeNodeTypesOrdered = getOrderedChangeTrackTypes(activeNode);
                activeNodeType = next ? _.first(activeNodeTypesOrdered) : _.last(activeNodeTypesOrdered);
            }

            // 1. iterating over all change tracking nodes
            while (activeNode && continueIteration) {

                // 2. iterating over all activeNodesTypesOrdered
                while (activeNodeType && continueIteration) {

                    if (DOM.isFloatingNode(activeNode)) {
                        // floated nodes (drawings) are never grouped with their neighbors
                        selectedNodes = [activeNode];
                    } else {
                        // finding all adjacent elements for the specified active node and the change track type.
                        // Using 'findTableChildren', so that search in combineAdjacentNodes happens not from span to
                        // paragraph to cell to row to table, but in the opposite direction.
                        selectedNodes = combineAdjacentNodes(activeNode, { type: activeNodeType, findTableChildren: true });
                    }

                    // is the active node type the first element of the selected nodes?
                    if (selectedNodes && (selectedNodes[0] === activeNode)) {
                        continueIteration = false;
                        foundValidChangeTrackSelection = true;
                    }

                    if (!foundValidChangeTrackSelection) {
                        // changing to the next/previous change tracking type
                        activeNodeType = getNextNodeType();
                    }
                }

                if (!foundValidChangeTrackSelection) {
                    // finding the next change tracking node
                    activeNode = getNextChangeTrackingNodeInListOfChangeTrackingNodes();
                    if (!activeNode) { return null; }
                    activeNodeTypesOrdered = getOrderedChangeTrackTypes(activeNode);
                    activeNodeType = next ? _.first(activeNodeTypesOrdered) : _.last(activeNodeTypesOrdered);
                }
            }

            if (foundValidChangeTrackSelection) {
                localChangeTrackSelection = createChangeTrackSelection(activeNode, activeNodeType, activeNodeTypesOrdered, selectedNodes, allChangeTrackNodes);
            }

            return localChangeTrackSelection;
        }

        /**
         * Selecting all nodes with the same tracking type that are direct neighbors.
         *
         * @param {{HTMLElement|jQuery}} trackingNode
         *  Element that will be evaluated for change tracking.
         *
         * @param {Object} [options]
         *  A map with additional options controlling the preparation of the span.
         *  The following options are supported:
         *  @param {String} [options.type='']
         *   The type of resolving, must be member of CHANGE_TRACK_TYPES.
         *  @param {Boolean} [options.findTableChildren=false]
         *   Whether the group shall be created from table to its children and not
         *   vice versa. Typically the grouping process is based to search from table
         *   cells to table rows to tables. But if a table node is given to this
         *   function as trackingNode and the rows or cells of the same group need
         *   to be defined, 'findTableChildren' must be set to true.
         *
         * @returns {HTMLElement[]}
         *  All nodes with the same tracking type that are direct neighbors within
         *  one array.
         */
        function combineAdjacentNodes(trackingNode, options) {

            var // all adjacent nodes with specified type
                allNodes = [],
                // the tracking type of combined nodes
                type = Utils.getStringOption(options, 'type', '').toLowerCase(),
                // the author, whose tracking nodes will be combined
                author = null,
                // the currently investigated node
                currentNode = Utils.getDomNode(trackingNode),
                // whether the tracking node is a table cell node
                isTableCellTrackingNode = DOM.isTableCellNode(trackingNode),
                // whether a table needs to find affected children
                findTableChildren = Utils.getBooleanOption(options, 'findTableChildren', false);

            // helper function to find a previous or following change track node, that
            // has the same resolveType and the same author (checked by the validator).
            function neighboringChangeTrackNode(node, validator, options) {

                var // the relevant change track node
                    neighborNode = null,
                    // the parent of the current node and its neighbor node
                    parentNode = null, neighborParentNode = null, checkNode = null,
                    // the child of a specified paragraph node
                    childNode = null,
                    // the currently investigated paragraph, cell, row, or table node
                    paraNode = null, cellNode = null, rowNode = null, tableNode = null,
                    // whether the next or previous node shall be found
                    next = Utils.getBooleanOption(options, 'next', true),
                    // whether neighbors with same modification shall be searched
                    modified = Utils.getBooleanOption(options, 'modified', false),
                    // the name of the dom function to be called
                    finder = next ? 'nextSibling' : 'previousSibling';

                if (DOM.isTableNode(node)) {
                    neighborNode = null;  // no additional node need to be collected
                    // -> is it possible, that rows and cells need to be collected, too?
                    // -> neighboring tables will not be combined (like neighboring paragraphs)
                } else if (DOM.isTableRowNode(node)) {

                    rowNode = node;

                    if (rowNode[finder]) {
                        rowNode = rowNode[finder];

                        // skipping page break rows
                        while (rowNode && DOM.isTablePageBreakRowNode(rowNode)) {
                            rowNode = rowNode[finder] || null;
                        }

                        if (rowNode && DOM.isTableRowNode(rowNode) && validator.call(this, rowNode)) {
                            neighborNode = rowNode;
                        }
                    } else {
                        if (next) {
                            // checking also the complete table element, if there is no following row
                            tableNode = rowNode.parentNode.parentNode;

                            if (tableNode && DOM.isTableNode(tableNode) && validator.call(this, tableNode)) {
                                neighborNode = tableNode;
                            }
                        }
                    }

                } else if (DOM.isTableCellNode(node)) {

                    cellNode = node;

                    // 'modified': Finding all single modified cells, also the table (ignoring rows)
                    // 'inserted'/'deleted': All cells in column are found in another process

                    if (!modified) {
                        neighborNode = null;  // accepting insert/delete of one column at a time
                        // -> neighboring columns will not be combined (like neighboring paragraphs)
                        // -> 'inserted' and 'deleted' can be set at cells only for complete columns.
                    } else {

                        // simple case for modifications:
                        // -> accepting/rejecting all modifications in the table in one step
                        // -> this includes modifications in cells and the table itself,
                        // but not rows (example: modified table style)

                        if (cellNode[finder]) {
                            cellNode = cellNode[finder];

                            if (cellNode && DOM.isTableCellNode(cellNode) && validator.call(this, cellNode)) {
                                neighborNode = cellNode;
                            }
                        } else {
                            // no neighboring cell found -> checking cells in next row (but ignoring the row itself)
                            rowNode = cellNode.parentNode;
                            if (rowNode && rowNode[finder]) {
                                rowNode = rowNode[finder];
                                if (rowNode && DOM.isTableRowNode(rowNode)) {
                                    cellNode = next ? rowNode.firstChild : rowNode.lastChild;
                                    if (cellNode && DOM.isTableCellNode(cellNode) && validator.call(this, cellNode)) {
                                        neighborNode = cellNode;
                                    }
                                }
                            } else {
                                if (next) {
                                    // checking also the complete table element, if there is no following row
                                    tableNode = rowNode.parentNode.parentNode;
                                    if (tableNode && DOM.isTableNode(tableNode) && validator.call(this, tableNode)) {
                                        neighborNode = tableNode;
                                    }
                                }
                            }
                        }
                    }

                } else if (DOM.isParagraphNode(node)) {

                    if (modified) {
                        paraNode = node;
                    } else {
                        paraNode = next ? node : node[finder]; // using previous paragraph, when searching to the beginning
                    }

                    // skipping page break nodes
                    while (paraNode && DOM.isPageBreakNode(paraNode)) {
                        paraNode = paraNode[finder] || null;
                    }

                    if (paraNode && DOM.isParagraphNode(paraNode)) {
                        if (modified) {
                            // modifications can be grouped on paragraph level, independent from the content inside the paragraph
                            paraNode = paraNode[finder];
                            if (paraNode && DOM.isParagraphNode(paraNode) && validator.call(this, paraNode)) {
                                neighborNode = paraNode;  // improvement: checking type of modification
                            }
                        } else {
                            childNode = next ? DOM.findFirstPortionSpan(paraNode) : DOM.findLastPortionSpan(paraNode);
                            if (childNode && (DOM.isTextSpan(childNode) || DOM.isTextComponentNode(childNode)) && validator.call(this, childNode)) {
                                neighborNode = childNode;
                            } else if (Position.getParagraphNodeLength(paraNode) === 0) {
                                // checking neighboring paragraph, if it is empty
                                if (next) {
                                    paraNode = paraNode[finder];
                                    // skipping page break nodes
                                    while (paraNode && DOM.isPageBreakNode(paraNode)) {
                                        paraNode = paraNode[finder] || null;
                                    }
                                }

                                if (paraNode && DOM.isParagraphNode(paraNode) && validator.call(this, paraNode)) {
                                    neighborNode = paraNode;  // using the current paragraph
                                }
                            }
                        }
                    }

                } else {  // text level components

                    checkNode = node[finder];  // getting the sibling

                    while (checkNode && !DOM.isInlineComponentNode(checkNode) && !DOM.isTextSpan(checkNode)) {  // but handling only text component siblings
                        checkNode = checkNode[finder] || null;
                    }

                    if (checkNode) {
                        // checking a direct sibling, but ignoring nodes of specified type
                        if (validator.call(this, checkNode)) {
                            neighborNode = checkNode;  // improvement for 'modified': checking type of modification
                        }
                    } else {
                        // checking if a paragraph parent node can be used temporary (only valid for text level nodes)
                        // For modified attributes, no switch from text to para and vice versa
                        if ((!modified) && (DOM.isTextSpan(node) || DOM.isInlineComponentNode(node))) {

                            parentNode = node.parentNode;

                            if (parentNode) {
                                if (!next) {
                                    // searching to the beginning -> adding current paragraph parent
                                    if (DOM.isParagraphNode(parentNode) && validator.call(this, parentNode)) {
                                        neighborNode = parentNode;  // using the current paragraph
                                    }
                                } else {
                                    // searching to the end -> adding the following paragraph parent
                                    neighborParentNode = parentNode[finder];

                                    // skipping page break nodes
                                    while (neighborParentNode && DOM.isPageBreakNode(neighborParentNode)) {
                                        neighborParentNode = neighborParentNode[finder] || null;
                                    }

                                    if (DOM.isParagraphNode(parentNode) && neighborParentNode && DOM.isParagraphNode(neighborParentNode) && validator.call(this, neighborParentNode)) {
                                        neighborNode = neighborParentNode;  // using the previous or following paragraph
                                    }
                                }
                            }
                        }
                    }
                }

                return neighborNode;
            }

            // helper function to collect all neighboring nodes with
            function collectNeighbours(node, validator, options) {

                var // whether the modified neighbors shall be search
                    modified = Utils.getBooleanOption(options, 'modified', false),
                    // the currently investigated neighbor node
                    current = neighboringChangeTrackNode(node, validator, { next: false, modified: modified });

                while (current) {
                    allNodes.unshift(current);
                    current = neighboringChangeTrackNode(current, validator, { next: false, modified: modified });
                }

                current = neighboringChangeTrackNode(node, validator, { next: true, modified: modified });

                while (current) {
                    allNodes.push(current);
                    current = neighboringChangeTrackNode(current, validator, { next: true, modified: modified });
                }

            }

            // helper function to remove empty text spans, if they are at the beginning
            // or at the end of the nodes collection. This can be more than one on each
            // side, for example caused by floated drawings
            function removeEmptyNodesAtBorders(allNodes) {

                var // the node to be checked, if it can be removed from selection
                    checkNode = allNodes[allNodes.length - 1];

                while (DOM.isEmptySpan(checkNode) && !DOM.isChangeTrackNode(checkNode)) {
                    allNodes.pop();
                    checkNode = allNodes[allNodes.length - 1];
                }

                checkNode = allNodes[0];
                while (DOM.isEmptySpan(checkNode) && !DOM.isChangeTrackNode(checkNode)) {
                    allNodes.shift();
                    checkNode = allNodes[0];
                }

            }

            // special handling required for table columns (cells that are inserted or removed)
            if (isTableCellTrackingNode &&
                (((type === 'inserted') && DOM.isChangeTrackInsertNode(currentNode)) || ((type === 'removed') && DOM.isChangeTrackRemoveNode(currentNode)))) {
                // -> collecting all cells of the column and adding it to allNodes
                // -> no further collecting required
                return Table.getAllCellsNodesInColumn(rootNode, currentNode);
            }

            // special handling for table nodes, if this is a 'up-down-selection' (caused by next and forward button)
            // The process in 'collectNeighbours' is a 'down-up' from table cells to rows to tables.
            if (DOM.isTableNode(currentNode) && findTableChildren) {

                // a modified table must be grouped with the modified cells
                if ((type === 'modified') && DOM.isChangeTrackModifyNode(currentNode)) {
                    _.each($(currentNode).find('> tbody > tr > td[data-change-track-modified="true"]'), function (node) {
                        allNodes.push(node);
                    });
                } else if ((type === 'inserted') && DOM.isChangeTrackInsertNode(currentNode)) { // an inserted table must be grouped with all rows
                    _.each($(currentNode).find('> tbody > tr[data-change-track-inserted="true"]'), function (node) {
                        allNodes.push(node);
                    });
                }

                // filtering by author
                if (type) {
                    author = self.getChangeTrackInfoFromNode(currentNode, 'author', type);
                    allNodes = _.filter(allNodes, function (oneNode) {
                        return self.nodeHasChangeTrackAuthor(oneNode, type, author);
                    });
                }

                allNodes.push(currentNode);

                return allNodes;
            }

            allNodes.push(currentNode);
            author = self.getChangeTrackInfoFromNode(currentNode, 'author', type);

            if ((DOM.isChangeTrackRemoveNode(currentNode)) && (type === 'removed')) {
                collectNeighbours(currentNode, function (node) {
                    return (DOM.isChangeTrackRemoveNode(node) && self.nodeHasChangeTrackAuthor(node, 'removed', author)) || DOM.isEmptySpan(node);
                });
            } else if ((DOM.isChangeTrackInsertNode(currentNode)) && (type === 'inserted')) {
                collectNeighbours(currentNode, function (node) {
                    return (DOM.isChangeTrackInsertNode(node) && self.nodeHasChangeTrackAuthor(node, 'inserted', author)) || DOM.isEmptySpan(node);
                });
            } else if (DOM.isChangeTrackModifyNode(currentNode) && (type === 'modified')) {
                collectNeighbours(currentNode, function (node) {
                    return (DOM.isChangeTrackModifyNode(node) && self.nodeHasChangeTrackAuthor(node, 'modified', author)) || DOM.isEmptySpan(node);
                }, { modified: true });
            }

            // empty spans are only allowed between tracking nodes, if they are not tracked
            removeEmptyNodesAtBorders(allNodes);

            return allNodes;
        }

        /**
         * Checking whether inside a specified paragraph all elements are change tracked
         * and after resolving all change tracks, the paragraph will be empty.
         *
         * The paragraph will be empty, if:
         * - all children from paragraph node are in the specified list of resolve nodes
         *   and are of type 'inserted' and accepted is false.
         *   or
         * - all children from paragraph node are in the specified list of resolve nodes
         *   and are of type 'removed' and accepted is true.
         *
         * @param {HTMLElement|jQuery} node
         *  The paragraph element whose children will be investigated.
         *
         * @param {} resolveNodes
         *  The list of all nodes, that are resolved in this specific process.
         *
         * @param {Boolean} accepted
         *  Whether the change tracks shall be accepted or rejected.
         *
         * @returns {Boolean}
         *  Whether all sub nodes of the paragraph are removed during accepting
         *  or rejecting the change tracks.
         */
        function paragraphEmptyAfterResolving(node, resolveNodes, accepted) {

            var // the html dom node
                domNode = Utils.getDomNode(node),
                // the child, that is not correctly change tracked
                childNode = null,
                // the change track type that needs to be checked
                type = accepted ? 'removed' : 'inserted';

            // check, if the node is a paragraph
            if (!DOM.isParagraphNode(domNode)) { return false; }

            // iterating over all children. Trying to find one node, that is
            // not an empty text span and that is not of the specific type or
            // that is not in the list of all nodes that are resolved in this process.
            childNode = _.find($(domNode).children(), function (child) {
                return !DOM.isEmptySpan(child) && (!self.nodeHasSpecifiedChangeTrackType(child, type) || !_.contains(resolveNodes, child));
            });

            return childNode ?  false : true;
        }

        // methods ------------------------------------------------------------

        /**
         * Whether the assignment of a new position (in-line, left floated, ...) to a drawing
         * needs to be change tracked. Currently this is not supported in OOXML format.
         * Therefore the change track information will  not be available in MS Word and will
         * be lost after reloading a document into OX Text. In this case, it will lead to an
         * error, if the change will be rejected in OX Text.
         */
        this.ctSupportsDrawingPosition = function () {
            return CHANGE_TRACK_SUPPORTS_DRAWING_POSITION;
        };

        /**
         * Whether the resizing of a drawing in OX Text needs to be change tracked. Currently
         * this is not supported in OOXML format. Therefore the change track information will
         * not be available in MS Word and will be lost after reloading a document into OX Text.
         * In this case, it will lead to an error, if the change will be rejected in OX Text.
         */
        this.ctSupportsDrawingResizing = function () {
            return CHANGE_TRACK_SUPPORTS_DRAWING_RESIZING;
        };

        /**
         * Updates the current date for change track operations on a regular basis.
         * This is needed for performance reasons, so that the function
         * 'Utils.getMSFormatIsoDateString' is not called for every operation.
         */
        this.updateChangeTrackInfoDate = function () {
            changeTrackInfoDate = Utils.getMSFormatIsoDateString({'useSeconds': false});
        };

        /**
         * Returns, whether change tracking is activated on the document.
         *
         * @returns {Boolean}
         *  Whether change tracking is activated locally.
         */
        this.isActiveChangeTracking = function () {
            if (app.isODF()) { return false; } // in odf files change tracking is not supported yet
            return model.getDocumentStyles().getDocumentAttributes().changeTracking || false;
        };

        /**
         * Sets the state for change trackting to 'true' or 'false'.
         * If set to true, change tracking is activated on the document.
         *
         * @param {Boolean} state
         *  The state for the change tracking. 'true' enables change tracking,
         *  'false' disables' change tracking.
         */
        this.setActiveChangeTracking = function (state) {
            if (typeof state !== 'boolean') { return; }
            model.applyOperations({ name: Operations.SET_DOCUMENT_ATTRIBUTES, attrs: { document: { changeTracking: state } } });
            // activating the change tracking timer update
            if (state) {
                startDateUpdate();
            } else {
                stopDateUpdate();
            }
        };

        /**
         *  List of change track authors in document
         */
        this.defineListOfChangeTrackAuthors = function () {
            listOfAuthors = model.getDocumentStyles().getDocumentAttributes().changeTrackAuthors;
        };

        /**
         * Returning the current author of the application as string
         * with full display name.
         *
         * @returns {String}
         *  The author string for the change track info object.
         */
        this.getChangeTrackInfoAuthor = function () {
            return app.getClientDisplayName();
        };

        /**
         * Returning the current date as string in ISO date format.
         * This function calls 'Utils.getMSFormatIsoDateString()' to simulate
         * the MS behavior, that the local time is send a UTC time. The
         * ending 'Z' in the string represents UTC time, but MS always
         * uses local time with ending 'Z'.
         *
         * @returns {String}
         *  The date string for the change track info object.
         */
        this.getChangeTrackInfoDate = function () {
            // Performance: changeTrackInfoDate is stored in a variable, needs only to be updated every 30s
            if (!dateUpdatePromise && self.isActiveChangeTracking()) { startDateUpdate(); }
            return changeTrackInfoDate ? changeTrackInfoDate : Utils.getMSFormatIsoDateString({'useSeconds': false});
        };

        /**
         * Returning an object used for additional change track information with author
         * and date of tracked change.
         *
         * @returns {Object}
         *  The change track info object containing author and date.
         */
        this.getChangeTrackInfo = function () {
            if (!dateUpdatePromise && self.isActiveChangeTracking()) { startDateUpdate(); }
            if (!userid) { userid = calculateUserId(ox.user_id); }
            return { author: self.getChangeTrackInfoAuthor(), date: self.getChangeTrackInfoDate(), uid: userid.toString() };
        };

        /**
         * Receiving the author, date and uid of a change track node for a specified type.
         *
         * @param {Node|jQuery|Null} node
         *  The DOM node to be checked. If this object is a jQuery collection, uses
         *  the first DOM node it contains. If missing or null, returns false.
         *
         * @param {String} info
         *  The info that is searched from the node. Allowed values are listed in
         *  'CHANGE_TRACK_INFOS'. 'author', 'date' and 'uid' are currently supported.
         *
         * @param {String} type
         *  The type for which the author is searched. Allowed values are listed in
         *  'CHANGE_TRACK_TYPES'.
         *
         * @returns {Number|String|Null}
         *  The requested information. This can be the 'author', the 'date' or null,
         *  if the information could not be found.
         */
        this.getChangeTrackInfoFromNode = function (node, info, type) {

            if (!node || !_.contains(CHANGE_TRACK_TYPES, type) || !_.contains(CHANGE_TRACK_INFOS, info)) { return null; }

            if (DOM.isTextComponentNode(node)) { node = Utils.getDomNode(node).firstChild; }

            var // the author or date of the change tracked node
                value = null,
                // the explicit attributes set at the specified node
                explicitAttributes = AttributeUtils.getExplicitAttributes(node);

            if (explicitAttributes && explicitAttributes.changes && explicitAttributes.changes[type]) {
                value = explicitAttributes.changes[type][info];
                if (info === 'uid') { value = resolveUserId(parseInt(value)); }
            }

            return value;
        };

        /**
         * Checking whether a specified node is a change tracked 'inserted' node, that was inserted
         * by the currently editing user.
         *
         * @param {Node|jQuery|Null} node
         *  The DOM node to be checked. If this object is a jQuery collection, uses
         *  the first DOM node it contains. If missing or null, returns false.
         *
         * @returns {Boolean}
         *  Whether a specified node is a change tracked 'inserted' node, that was inserted
         *  by the currently editing user.
         */
        this.isInsertNodeByCurrentAuthor = function (node) {
            return node && DOM.isChangeTrackInsertNode(node) && (self.getChangeTrackInfoFromNode(node, 'author', 'inserted') === self.getChangeTrackInfoAuthor());
        };

        /**
         * Checking whether all specified nodes in the nodes collector are change tracked
         * 'inserted', and this was inserted by the currently editing user.
         *
         * @param {Node[]|jQuery|Null} nodes
         *  The DOM nodes to be checked. This can be an array of nodes or a jQuery collection.
         *  If missing or empty, returns false.
         *
         * @returns {Boolean}
         *  Whether all specified nodes are change tracked 'inserted', and this was inserted
         *  by the currently editing user.
         */
        this.allAreInsertedNodesByCurrentAuthor = function (nodes) {

            var // whether all specified nodes in the collector were inserted by the current author
                allInserted = true;

            if ((!nodes) || (nodes.length === 0)) { return false; }

            Utils.iterateArray(nodes, function (node) {
                if (!self.isInsertNodeByCurrentAuthor(node)) {
                    allInserted = false;
                    return Utils.BREAK;
                }
            });

            return allInserted;
        };

        /**
         * Checking whether a specified node has a specified change track type.
         *
         * @param {Node|jQuery|Null} node
         *  The DOM node to be checked. If this object is a jQuery collection, uses
         *  the first DOM node it contains. If missing or null, returns false.
         *
         * @param {String} type
         *  The type for which the node is checked. Allowed values are listed in
         *  'CHANGE_TRACK_TYPES'.
         *
         * @returns {Boolean}
         *  Whether the specified node has the specified change track type.
         */
        this.nodeHasSpecifiedChangeTrackType = function (node, type) {
            return node &&
                   ((type === 'inserted' && DOM.isChangeTrackInsertNode(node)) ||
                    (type === 'removed' && DOM.isChangeTrackRemoveNode(node)) ||
                    (type === 'modified' && DOM.isChangeTrackModifyNode(node)));
        };

        /**
         * Checking whether a specified node has a specified author for a specified
         * change track type.
         *
         * @param {Node|jQuery|Null} node
         *  The DOM node to be checked. If this object is a jQuery collection, uses
         *  the first DOM node it contains. If missing or null, returns false.
         *
         * @param {String} type
         *  The type for which the author is searched. Allowed values are listed in
         *  'CHANGE_TRACK_TYPES'.
         *
         * @param {String} author
         *  The specified author, who is checked to be the change track owner.
         *
         * @returns {Boolean}
         *  Whether the specified node has the specified author.
         */
        this.nodeHasChangeTrackAuthor = function (node, type, author) {
            return node && self.nodeHasSpecifiedChangeTrackType(node, type) && (self.getChangeTrackInfoFromNode(node, 'author', type) === author);
        };

        /**
         * During pasting content into the document, if change tracking is NOT
         * activated, one additional setAttributes operations is required,
         * directly following the first splitParagraph operation. This
         * setAttributes operation removes change tracking information, from
         * the split paragraph.
         *
         * @param {Object[]} operations
         *  The list of all operations that will be applied during pasting.
         */
        this.removeChangeTrackInfoAfterSplitInPaste = function (operations) {

            var // a counter for the operations
                counter = 0,
                // the first split operation in the operations array
                splitOperation = null;

            // helper function to create a setAttributes operation
            // to remove change tracks family.
            function generateChangeTrackSetAttributesOperation(operation) {
                var paraPos = _.clone(operation.start);
                paraPos.pop();
                paraPos[paraPos.length - 1]++;
                return { name: Operations.SET_ATTRIBUTES, start: paraPos, attrs: { changes: { inserted: null, removed: null, modified: null }}};
            }

            // finding the first splitParagraph operation
            splitOperation = _.find(operations, function (operation) {
                counter++;
                return (Operations.PARA_SPLIT === operation.name);
            });

            // Adding a setAttributes operation behind the first splitParagraph operation
            if (splitOperation) {
                operations.splice(counter, 0, generateChangeTrackSetAttributesOperation(splitOperation));
            }
        };

        /**
         * During pasting content into the document, if change tracking is activated,
         * the change track information need to be added to the operations.
         * Additionally some additional setAttributes operations are required,
         * directly following the splitParagraph operations.
         *
         * @param {Object[]} operations
         *  The list of all operations that will be applied during pasting.
         */
        this.handleChangeTrackingDuringPaste = function (operations) {

            var // the information object for the change tracking
                changeTrackInfo = self.getChangeTrackInfo();

            _(operations).map(function (operation) {
                if (!operation) { return null; }
                self.addChangeTrackInfoToOperation(operation, changeTrackInfo, { clipboard: true });
                return operation;
            });

            // order the new setAttributes operations into the correct positions
            // in the operations collector
            if (clipboardCollector.length) {
                _.each(operations, function (operation, index) {
                    if (Operations.PARA_SPLIT === operation.name) {
                        var newOperation = clipboardCollector.shift();
                        operations.splice(index + 1, 0, newOperation);
                    }
                });
                // operations = operations.concat(self.getClipboardCollector());
                clipboardCollector = [];
            }
        };

        /**
         * Adding change tracking information (change track type object with author and
         * date) to a given operation. The operation is modified directly, no copy is
         * created. This functionality is currently implemented for clipboard operations.
         *
         * @param {Object} operation
         *  The operation that will be expanded for the change track information.
         *
         * @param {Object} [info]
         *  The change tracking information object containing the author and the date.
         *  This parameter is optional. If it is not set, this object is generated.
         *
         * @param {Object} [options]
         *  A map with additional options controlling the addition of change tracking
         *  information.
         *  The following options are supported:
         *  @param {Boolean} [options.clipboard=false]
         *      Whether this function was called from the clip board API.
         */
        this.addChangeTrackInfoToOperation = function (operation, info, options) {

            if (!operation) { return null; }
            if (!info) { info = self.getChangeTrackInfo(); }

            var // the key for the changes object
                changesKey = null,
                // whether this function was triggered from clip board API
                isClipboard = Utils.getBooleanOption(options, 'clipboard', false);

            // helper function to generate and collect additional setAttributes
            // operations after splitParagraph operations.
            function generateChangeTrackSetAttributesOperation() {
                var paraPos = _.clone(operation.start);
                paraPos.pop();
                paraPos[paraPos.length - 1]++;
                clipboardCollector.push({ name: Operations.SET_ATTRIBUTES, start: paraPos, attrs: { changes: { inserted: info }}});
            }

            // Info: Operations.SET_ATTRIBUTES does not need to be change tracked from clip board operations
            // because during pasting, attributes can also be assigned to inserted content. It is even better
            // not to change track setAttributes (with 'modified' key), because otherwise the user needs to
            // resolve two change tracks.

            if (isClipboard) {
                if (_.contains(CHANGETRACK_CLIPBOARD_OPERATIONS, operation.name)) {
                    changesKey = 'inserted';
                } else if (Operations.PARA_SPLIT === operation.name) {
                    // After splitParagraph an additional setAttributes operations is required,
                    // so that the new paragraph is marked as inserted.
                    generateChangeTrackSetAttributesOperation();
                }
            }

            // Idea: In the future this function can be the central point, where change track information is
            // added to any operation. Not only for clip board generated operations. In this case, the
            // changesKey must be a parameter or it must be determined dynamically.

            if (changesKey) {
                operation.attrs = operation.attrs || {};
                operation.attrs.changes = operation.attrs.changes || {};
                operation.attrs.changes[changesKey] = info;
            }
        };

        /**
         * Shows the change track group of a user selection by setting
         * temporary background colors to all change tracking nodes,
         * which belongs to the current change track node of the selection.
         */
        this.showChangeTrackGroup = function () {

            // do nothing with range selections (explicitely also not with drawing selections)
            if (model.getSelection().hasRange()) { return; } // && !model.getSelection().isDrawingFrameSelection()) { return; }

            var // the node currently selected by the text cursor (or at the beginning of an arbitrary selection range)
                node = findChangeTrackNodeAtCursorPosition();

            // quit if a change track node is not found
            if (node.length === 0) { return; }

            var nodeRelevantType = this.getRelevantTrackingTypeByNode(node),
                nodeGroup = groupTrackingNodes(node, nodeRelevantType),
                activeNode = nodeGroup[0],
                activeNodeTypesOrdered = getOrderedChangeTrackTypes(activeNode),
                localChangeTrackSelection = createChangeTrackSelection(activeNode, nodeRelevantType, activeNodeTypesOrdered, nodeGroup);

            if (localChangeTrackSelection) {
                // setting the new change track selection. Because this process also modifies the text
                // selection, this could repaint the visible change track pop up again. To avoid this
                // the flag 'keepChangeTrackPopup' must be transported to the 'change' and 'selection'
                // events, so that the change track pop up is not
                setChangeTrackSelection(localChangeTrackSelection, { keepChangeTrackPopup: true });
            }

        };

        /**
         *  Clears the current change track selection and remove all of their highlights
         */
        this.clearChangeTrackSelection = function () {
            if (!changeTrackSelection) { return; }
            $(changeTrackSelection.selectedNodes).removeClass(highlightClassname);
            changeTrackSelection = null;
        };

        /**
         * Receiving the relevant tracking information of a specified node.
         * Relevant means, that 'inserted' or 'removed' are more important
         * than 'modified'.
         *
         * @param {HTMLElement|jQuery} oneNode
         *  The element whose tracking information is investigated.
         *
         * @returns {Object}
         *  The relevant tracking info of a given node, or null, if the
         *  selection contains no tracking node. The relevant info is an
         *  object containing the properties 'type', 'author' and 'date'.
         */
        this.getRelevantTrackingInfoByNode = function (oneNode) {

            var // the explicit attributes of the node
                explicitAttributes = null,
                // the relevant type of the node
                allTypes = null, type = null,
                // the type of the node affected by change track
                nodeType = null,
                // the author, date and user id of the modified type
                author = null, date = null, uid = null,
                // the jQuerified node
                localNode = oneNode ? $(oneNode) : null;

            // helper function to determine the type of the node
            // that is displayed to the user:
            // Allowed values: text, paragraph, table cell, table row, table
            function getTypeOfNode(node) {

                var description = '';

                if (DOM.isTextSpan(node) || DOM.isTextComponentNode(node)) {
                    description = 'text';
                } else if (DOM.isParagraphNode(node)) {
                    description = 'paragraph';
                } else if (DOM.isTableNode(node)) {
                    description = 'table';
                } else if (DOM.isTableRowNode(node)) {
                    description = 'table row';
                } else if (DOM.isTableCellNode(node)) {
                    description = 'table cell';
                } else if (DOM.isDrawingFrame(node)) {
                    description = 'drawing';
                }

                return description;
            }

            if (localNode && localNode.length > 0) {

                if (DOM.isTextComponentNode(localNode)) { localNode = $(Utils.getDomNode(localNode).firstChild); }

                explicitAttributes = AttributeUtils.getExplicitAttributes(localNode);

                if (explicitAttributes && explicitAttributes.changes) {

                    // finding the relevant change track type
                    allTypes = getOrderedChangeTrackTypes(localNode);
                    type = allTypes.length > 0 ? allTypes[0] : null;

                    if (type) {
                        author = explicitAttributes.changes[type].author;
                        date = explicitAttributes.changes[type].date;
                        uid = resolveUserId(parseInt(explicitAttributes.changes[type].uid));

                        // also returning the node type, that will be affected by this change track:
                        // text, paragraph, table cell, table row, table
                        nodeType = getTypeOfNode(localNode);
                    }
                }

            }

            return type ? { type: type, author: author, date: date, uid: uid, nodeType: nodeType } : null;
        };

        /**
         * Receiving the relevant tracking information of the current selection.
         * If the selection has a range, null is returned. Relevant means,
         * that 'inserted' or 'removed' are more important than 'modified'.
         *
         * @returns {Object}
         *  The relevant tracking info of a given node, or null, if the
         *  selection contains no tracking node. The relevant info is an
         *  object containing the properties 'type', 'author' and 'date'.
         */
        this.getRelevantTrackingInfo = function () {

            // no relevant type, if there is a selection range
            // but allowing selection of inline elements (tab, drawing, ...)

            // -> checking inline div elements or (floated) drawings
            if (model.getSelection().hasRange() && !model.getSelection().isInlineComponentSelection() && !model.getSelection().isDrawingFrameSelection()) { return null; }

            var node = findChangeTrackNodeAtCursorPosition();

            if (node.length === 0) { return null; }

            return self.getRelevantTrackingInfoByNode(node);
        };

        /**
         * Receiving the relevant tracking information of a specified node.
         * Relevant tracking type means, that 'inserted' or 'removed' are
         * more important than 'modified'.
         *
         * @param {HTMLElement|jQuery} node
         *  The element whose tracking type is investigated.
         *
         * @returns {String}
         *  The relevant tracking type of a given node, or null, if the
         *  selection contains no tracking node.
         */
        this.getRelevantTrackingTypeByNode = function (node) {
            var info = self.getRelevantTrackingInfoByNode(node);
            return info ? info.type : null;
        };

        /**
         * Receiving the relevant tracking type of the current selection.
         * If the selection has a range, null is returned. Relevant means,
         * that 'inserted' or 'removed' are more important than 'modified'.
         *
         * @returns {String}
         *  The relevant tracking type of a given node, or null, if the
         *  selection contains no tracking node.
         */
        this.getRelevantTrackingType = function () {
            var info = self.getRelevantTrackingInfo();
            return info ? info.type : null;
        };

        /**
         * Checking, if inside an array of operations at least one operation
         * contains change tracking information.
         *
         * @param {Array} operations
         *  The operations to be checked.
         *
         * @returns {Boolean}
         *  Whether at least one operation in the specified array of operations
         *  contains change tracking information.
         */
        this.hasChangeTrackOperation = function (operations) {
            return _.filter(operations, self.isChangeTrackOperation).length > 0;
        };

        /**
         * Checking, whether a specified operation contains change tracking information.
         *
         * @param {Object} operation
         *  The operation to be checked.
         *
         * @returns {Boolean}
         *  Whether the specified operation contains change tracking information.
         */
        this.isChangeTrackOperation = function (operation) {
            return (operation && operation.attrs && operation.attrs.changes &&
                    ((operation.attrs.changes.inserted && operation.attrs.changes.inserted.author) ||
                     (operation.attrs.changes.removed && operation.attrs.changes.removed.author) ||
                     (operation.attrs.changes.modified && operation.attrs.changes.modified.author)));
        };

        /**
         * Saving the state of the change track side bar handler. This is used for
         * performance reasons. Registering handler is only necessary, if they are not
         * already registered. The same for de-registering.
         *
         * @param {Boolean} state
         *  The state of the change track side bar handler.
         */
        this.setSideBarHandlerActive = function (state) {
            sideBarHandlerActive = state;
        };

        /**
         * Returning information about the state of the change track side bar handler.
         *
         * @returns {Boolean}
         *  Whether the handler for the sidebar are activated or not.
         */
        this.isSideBarHandlerActive = function () {
            return sideBarHandlerActive;
        };

        /**
         * Resolving the change tracking for all text spans or only for one
         * selected text span.
         *
         * @param {{HTMLElement|jQuery}} [trackingNodes]
         *  Optional html element that will be evaluated for change tracking.
         *  If specified, the selection or the parameter 'all' are ignored.
         *
         * @param {Object} [options]
         *  A map with additional options controlling the preparation of the span.
         *  The following options are supported:
         *  @param {String} [options.type='']
         *      The type of resolving ('inserted', 'removed' or 'modified').
         *  @param {Boolean} [options.accepted=true]
         *      If set to true, the change tracking will be resolved as 'accepted'.
         *      Otherwise it will be resolved as 'rejected'.
         *  @param {Boolean} [options.all=false]
         *      If set to true, all text span with change tracking attribute
         *      will be resolved. Otherwise only the currently selected text
         *      span.
         */
        this.resolveChangeTracking = function (trackingNodes, options) {

            var // the operations generator
                generator = model.createOperationsGenerator(),
                // whether all change tracks shall be accepted or rejected
                all = Utils.getBooleanOption(options, 'all', false),
                // saving the start position of an existing range
                savedStartPosition = null,
                // whether a delete operation was used for resolving change tracking
                contentDeleted = false,
                // whether the change shall be accepted or rejected
                accepted = Utils.getBooleanOption(options, 'accepted', true),
                // the type of resolving ('inserted', 'removed' or 'modified')
                resolveType = Utils.getStringOption(options, 'type', ''),
                // a collector for all tracking nodes
                trackingNodesCollector = null,
                // a deferred object for applying the actions
                operationsDef = null,
                // a timer for the progress bar
                progressBarTimer = null;

            // restoring the old character attribute values
            function receiveOldAttributes(currentAttrs, oldAttrs) {

                var // using old attributes, if available
                    attrs = {},
                    // the attributes, that are currently defined
                    key = null;

                currentAttrs = currentAttrs || {};

                attrs =  _.copy(oldAttrs, true);  // using all old attributes

                // and setting all new attributes, that are not defined in the old attributes, to null.
                for (key in currentAttrs) {
                    if (!(key in oldAttrs)) {
                        attrs[key] = null;
                    }
                }

                return attrs;
            }

            // setting tracking, that will be resolved
            if (changeTrackSelection && changeTrackSelection.selectedNodes && !all) {
                // using selected nodes and resolve type from current change track selection
                trackingNodes = changeTrackSelection.selectedNodes;
                resolveType = changeTrackSelection.activeNodeType;
            } else if (trackingNodes && !all) {
                trackingNodes = $(trackingNodes);
            } else {
                if (all) {
                    // collecting all change track nodes in the document (using from change tracking selection, if available)
                    trackingNodes = (changeTrackSelection && changeTrackSelection.allChangeTrackNodes) ? changeTrackSelection.allChangeTrackNodes : getAllChangeTrackNodes();
                } else {
                    // using the current selection, to resolve the change tracking
                    // -> this can be a selection with or without range.
                    // -> splitting nodes, if they are not completely inside the selection and the selection has a range
                    trackingNodesCollector = getTrackingNodesInSelection({ split: model.getSelection().hasRange(), markLastParagraph: true });

                    if (trackingNodesCollector.length === 0) {
                        // no change tracking nodes were found in the selection
                        return;
                    } else {
                        // change tracking nodes were found in the selection
                        if (model.getSelection().hasRange()) {

                            // simply using the collected (and already split) nodes
                            // from the selection with range.
                            trackingNodes = trackingNodesCollector;

                            if (model.getSelection().isDrawingFrameSelection()) {
                                // special handling for drawing selections, where the popup box is displayed
                                // and not all change tracks shall be resolved but only the specified type
                                if (! resolveType) { resolveType = self.getRelevantTrackingTypeByNode(trackingNodesCollector); }

                            } else {

                                // after determining the affected nodes, the process of 'all' can be used
                                // to resolve the change tracks. This is not limited to the relevant
                                // resolve type, but includes all resolve types. So inside a given arbitrary
                                // selection, all affected nodes (after grouping maybe even more) need to
                                // be resolved with all resolve types.
                                all = true;
                            }

                        } else {
                            // if resolve type is not specified, the relevant type need to be defined
                            if (! resolveType) { resolveType = self.getRelevantTrackingTypeByNode(trackingNodesCollector); }
                            // -> finding all relevant tracking nodes (keeping each node unique)
                            if (resolveType) {
                                trackingNodes = groupTrackingNodes(trackingNodesCollector, resolveType);
                            }
                        }
                    }
                }
            }

            if (trackingNodes && trackingNodes.length > 0) {

                // iterating over all tracking nodes
                _.each(trackingNodes, function (trackingNode, nodeIndex) {

                    var // the logical start and end position of the currently evaluated node (must always be inside one paragraph)
                        changeTrackingRange = Position.getNodeStartAndEndPosition(trackingNode, rootNode, { fullLogicalPosition: true }),
                        // a neighbor node of the tracking node
                        neighborNode = null,
                        // the following node of the tracking node
                        followingNode = null,
                        // the explicit node attributes
                        explicitAttributes = null,
                        // the original attributes
                        oldCharAttrs = null, oldParaAttrs = null, oldCellAttrs = null, oldRowAttrs = null, oldTableAttrs = null, oldStyleId = null, oldDrawingAttrs = null,
                        // the old attributes to be restored
                        restoredAttrs = null,
                        // whether the node is a text span, a text component (tab, hard break, ...) or a drawing
                        isTextSpan = DOM.isTextSpan(trackingNode) || DOM.isTextComponentNode(trackingNode) || DrawingFrame.isDrawingFrame(trackingNode),
                        // whether the node is a paragraph
                        isParagraph = DOM.isParagraphNode(trackingNode),
                        // whether the node is a table
                        isTable = DOM.isTableNode(trackingNode),
                        // whether the node is a table row
                        isTableRow = DOM.isTableRowNode(trackingNode),
                        // whether the node is a table cell
                        isTableCell = DOM.isTableCellNode(trackingNode),
                        // helper variables for column handling
                        tableNode = null, tableGrid = null, gridRange = null, tablePos = null,
                        // whether this is the last node of the trackingNodes collector
                        isLastNode = (nodeIndex === trackingNodes.length - 1),
                        // whether a page break needs to be removed from table
                        removePageBreak = false;

                    // saving start position of the first node
                    savedStartPosition = savedStartPosition || _.clone(changeTrackingRange.start);

                    // creating operation(s) required to accept or reject the change track
                    if ((DOM.isChangeTrackInsertNode(trackingNode)) && ((resolveType === 'inserted') || all)) {
                        if (accepted) {
                            generator.generateOperation(Operations.SET_ATTRIBUTES, { start: changeTrackingRange.start, end: changeTrackingRange.end, attrs: { changes: { inserted: null }}});
                        } else {
                            if (isTextSpan) {
                                generator.generateOperation(Operations.DELETE, { start: changeTrackingRange.start, end: changeTrackingRange.end });
                                // the 'inserted' is no longer valid at a completely removed node
                                // Info: The order is reversed -> 1. step: Setting attributes, 2. step: Removing all content
                                // Setting attributes, before deleting node. This is (only) necessary if an empty text node remains (for example last node in paragraph)
                                generator.generateOperation(Operations.SET_ATTRIBUTES, { start: changeTrackingRange.start, end: changeTrackingRange.end, attrs: { changes: { inserted: null }}});
                                contentDeleted = true;
                            } else if (isParagraph) {
                                neighborNode = DOM.getAllowedNeighboringNode(trackingNode, { next: false });
                                followingNode = DOM.getAllowedNeighboringNode(trackingNode, { next: true });
                                // checking, if a following table contains page break attribute
                                if (DOM.isTableNode(followingNode) && DOM.isTableWithManualPageBreak(followingNode)) { removePageBreak = true; }

                                // merging the paragraph with its previous sibling, if available
                                if (neighborNode && DOM.isParagraphNode(neighborNode)) {
                                    generator.generateOperation(Operations.PARA_MERGE, { start: Position.increaseLastIndex(changeTrackingRange.start, -1) });
                                    contentDeleted = true;
                                } else if (!neighborNode || DOM.isTableNode(neighborNode)) {
                                    // the paragraph is marked with 'inserted' (maybe even between two tables)
                                    // -> NOT moving text into previous or next table (like MS Office)
                                    // -> delete paragraph, if it is empty or contains only content marked as inserted
                                    // -> merge tables, if they are directly following
                                    if ((Position.getParagraphNodeLength(trackingNode) === 0) || (paragraphEmptyAfterResolving(trackingNode, trackingNodes, accepted))) {
                                        // taking care of operation order: Later reverting of operations -> first delete paragraph,
                                        // then merge tables (if possible)
                                        if (DOM.isTableNode(followingNode) && Table.mergeableTables(neighborNode, followingNode)) {
                                            generator.generateOperation(Operations.TABLE_MERGE, { start: Position.getOxoPosition(rootNode, neighborNode) });
                                        }
                                        // deleting an empty paragraph (or a paragraph containing content completely marked as 'inserted'
                                        generator.generateOperation(Operations.DELETE, { start: changeTrackingRange.start });
                                        contentDeleted = true;
                                    }
                                }
                                // removing page break attribute from following table, if required (calculating position before deleting table)
                                if (removePageBreak) {
                                    generator.generateOperation(Operations.SET_ATTRIBUTES, { attrs: { paragraph: { pageBreakBefore: false }}, start: Position.getFirstPositionInParagraph(rootNode, Position.getOxoPosition(rootNode, followingNode)).slice(0, -1)});
                                }
                                // removing change track attribute from paragraph
                                generator.generateOperation(Operations.SET_ATTRIBUTES, { start: changeTrackingRange.start, end: changeTrackingRange.end, attrs: { changes: { inserted: null }}});
                            } else if (isTableCell) {
                                // rejecting for all cells in the column
                                generator.generateOperation(Operations.DELETE, { start: changeTrackingRange.start });
                                if (isLastNode) {
                                    // Setting new table grid attribute to table
                                    tableNode = $(trackingNode).closest(DOM.TABLE_NODE_SELECTOR);
                                    tableGrid = _.clone(model.getStyleCollection('table').getElementAttributes(tableNode).table.tableGrid);
                                    gridRange = Table.getGridColumnRangeOfCell(trackingNode);
                                    tablePos = Position.getOxoPosition(rootNode, tableNode, 0);
                                    tableGrid.splice(gridRange.start, gridRange.end - gridRange.start + 1);  // removing column(s) in tableGrid (automatically updated in table node)
                                    generator.generateOperation(Operations.SET_ATTRIBUTES, { attrs: { table: { tableGrid: tableGrid } }, start: _.clone(tablePos) });
                                    contentDeleted = true;
                                }
                            } else if (isTable || isTableRow) {

                                // deleting of table needs to be checked and generated before deleting of rows
                                // -> handling the problem, that after reload, table no longer is marked as 'inserted'
                                if (isTableRow && ($(trackingNode).prev().length === 0)) {
                                    tableNode = $(trackingNode).closest(DOM.TABLE_NODE_SELECTOR);
                                    if (tableNode && !DOM.isChangeTrackInsertNode(tableNode)) {
                                        // is there a next sibling, that is not marked as inserted?
                                        // -> comparing the number of next inserted siblings + 1 with the number of rows in the table
                                        if (((($(trackingNode).nextAll(DOM.CHANGETRACK_INSERTED_ROW_SELECTOR)).length) + 1) === DOM.getTableRows(tableNode).length) {
                                            tablePos = Position.getOxoPosition(rootNode, tableNode, 0);
                                            generator.generateOperation(Operations.DELETE, { start: tablePos });
                                        }
                                    }
                                }

                                generator.generateOperation(Operations.DELETE, { start: changeTrackingRange.start });
                                contentDeleted = true;
                            }
                        }
                    }

                    if ((DOM.isChangeTrackRemoveNode(trackingNode)) && ((resolveType === 'removed') || all)) {
                        if (accepted) {
                            if (isTextSpan) {
                                generator.generateOperation(Operations.DELETE, { start: changeTrackingRange.start, end: changeTrackingRange.end });
                                contentDeleted = true;
                            } else if (isParagraph) {
                                neighborNode = DOM.getAllowedNeighboringNode(trackingNode, { next: false });
                                followingNode = DOM.getAllowedNeighboringNode(trackingNode, { next: true });
                                // checking, if the following table contains page break attribute
                                if (DOM.isTableNode(followingNode) && DOM.isTableWithManualPageBreak(followingNode)) { removePageBreak = true; }

                                // merging the paragraph with its next sibling, if available
                                if (neighborNode && DOM.isParagraphNode(neighborNode)) {
                                    // merging previous paragraph with marked paragraph
                                    generator.generateOperation(Operations.PARA_MERGE, { start: Position.increaseLastIndex(changeTrackingRange.start, -1) });
                                    contentDeleted = true;
                                } else if (!neighborNode || DOM.isTableNode(neighborNode)) {
                                    // the paragraph is marked with 'deleted' (maybe even between two tables)
                                    // -> delete paragraph, if it is empty or contains only content marked as inserted
                                    // -> merge tables, if they are directly following
                                    if ((Position.getParagraphNodeLength(trackingNode) === 0) || (paragraphEmptyAfterResolving(trackingNode, trackingNodes, accepted))) {
                                        // taking care of operation order: Later reverting of operations -> first delete paragraph,
                                        // then merge tables (if possible)
                                        if (DOM.isTableNode(followingNode) && Table.mergeableTables(neighborNode, followingNode)) {
                                            // Checking, if the following table is also change tracked(?)
                                            generator.generateOperation(Operations.TABLE_MERGE, { start: Position.getOxoPosition(rootNode, neighborNode) });
                                        }
                                        // deleting an empty paragraph (or a paragraph containing content completely marked as 'inserted'
                                        generator.generateOperation(Operations.DELETE, { start: changeTrackingRange.start });
                                        contentDeleted = true;
                                    }
                                }
                                // removing page break attribute from second table, if required (calculating position before deleting table)
                                if (removePageBreak) {
                                    generator.generateOperation(Operations.SET_ATTRIBUTES, { attrs: { paragraph: { pageBreakBefore: false }}, start: Position.getFirstPositionInParagraph(rootNode, Position.getOxoPosition(rootNode, followingNode)).slice(0, -1)});
                                }
                                // removing change track attribute from paragraph
                                generator.generateOperation(Operations.SET_ATTRIBUTES, { start: changeTrackingRange.start, end: changeTrackingRange.end, attrs: { changes: { removed: null }}});
                            } else if (isTableCell) {
                                // accepting for all cells in the column (only one cell determines complete column)
                                generator.generateOperation(Operations.DELETE, { start: changeTrackingRange.start });
                                // Setting new table grid attribute to table
                                if (isLastNode) {
                                    tableNode = $(trackingNode).closest(DOM.TABLE_NODE_SELECTOR);
                                    tableGrid = _.clone(model.getStyleCollection('table').getElementAttributes(tableNode).table.tableGrid);
                                    gridRange = Table.getGridColumnRangeOfCell(trackingNode);
                                    tablePos = Position.getOxoPosition(rootNode, tableNode, 0);
                                    tableGrid.splice(gridRange.start, gridRange.end - gridRange.start + 1);  // removing column(s) in tableGrid (automatically updated in table node)
                                    generator.generateOperation(Operations.SET_ATTRIBUTES, { attrs: { table: { tableGrid: tableGrid } }, start: _.clone(tablePos) });
                                    contentDeleted = true;
                                }
                            } else if (isTable || isTableRow) {

                                // deleting of table needs to be checked and generated before deleting of rows
                                // -> handling the problem, that after reload, table no longer is marked as 'removed'
                                if (isTableRow && ($(trackingNode).prev().length === 0)) {
                                    tableNode = $(trackingNode).closest(DOM.TABLE_NODE_SELECTOR);
                                    if (tableNode && !DOM.isChangeTrackRemoveNode(tableNode)) {
                                        // is there a next sibling, that is not marked as removed?
                                        // -> comparing the number of next removed siblings + 1 with the number of rows in the table
                                        if (((($(trackingNode).nextAll(DOM.CHANGETRACK_REMOVED_ROW_SELECTOR)).length) + 1) === DOM.getTableRows(tableNode).length) {
                                            tablePos = Position.getOxoPosition(rootNode, tableNode, 0);
                                            generator.generateOperation(Operations.DELETE, { start: tablePos });
                                        }
                                    }
                                }

                                generator.generateOperation(Operations.DELETE, { start: changeTrackingRange.start });
                                contentDeleted = true;
                            }
                        } else {
                            generator.generateOperation(Operations.SET_ATTRIBUTES, { start: changeTrackingRange.start, end: changeTrackingRange.end, attrs: { changes: { removed: null }}});
                        }
                    }

                    if ((DOM.isChangeTrackModifyNode(trackingNode)) && ((resolveType === 'modified') || (all && !$(trackingNode).data('skipModify')))) {

                        if (accepted) {
                            generator.generateOperation(Operations.SET_ATTRIBUTES, { start: changeTrackingRange.start, end: changeTrackingRange.end, attrs: { changes: { modified: null }}});
                        } else {
                            // Restoring the old attribute state
                            explicitAttributes = AttributeUtils.getExplicitAttributes(trackingNode);

                            // comparing 'character' with 'changes.modified.attrs.character' and 'paragraph' with 'changes.modified.attrs.paragraph'
                            oldCharAttrs = receiveOldAttributes(explicitAttributes.character, (explicitAttributes.changes.modified.attrs && explicitAttributes.changes.modified.attrs.character) || {});
                            oldParaAttrs = receiveOldAttributes(explicitAttributes.paragraph, (explicitAttributes.changes.modified.attrs && explicitAttributes.changes.modified.attrs.paragraph) || {});
                            oldCellAttrs = receiveOldAttributes(explicitAttributes.cell, (explicitAttributes.changes.modified.attrs && explicitAttributes.changes.modified.attrs.cell) || {});
                            oldRowAttrs = receiveOldAttributes(explicitAttributes.row, (explicitAttributes.changes.modified.attrs && explicitAttributes.changes.modified.attrs.row) || {});
                            oldTableAttrs = receiveOldAttributes(explicitAttributes.table, (explicitAttributes.changes.modified.attrs && explicitAttributes.changes.modified.attrs.table) || {});
                            oldDrawingAttrs = receiveOldAttributes(explicitAttributes.drawing, (explicitAttributes.changes.modified.attrs && explicitAttributes.changes.modified.attrs.drawing) || {});
                            oldStyleId = (explicitAttributes.changes.modified.attrs && explicitAttributes.changes.modified.attrs.styleId) || null;
                            restoredAttrs = { character: oldCharAttrs, paragraph: oldParaAttrs, cell: oldCellAttrs, row: oldRowAttrs, table: oldTableAttrs, drawing: oldDrawingAttrs, changes: { modified: null }};
                            if (oldStyleId) { restoredAttrs.styleId = oldStyleId; }
                            else if (explicitAttributes.styleId) { restoredAttrs.styleId = null; }
                            generator.generateOperation(Operations.SET_ATTRIBUTES, { start: changeTrackingRange.start, end: changeTrackingRange.end, attrs: restoredAttrs });
                        }
                    }

                    // removing data attribute 'skipModify'
                    if ($(trackingNode).data('skipModify')) { $(trackingNode).removeData('skipModify'); }

                });

                // operations MUST be executed in reverse order to preserve the positions
                generator.reverseOperations();

                // invalidate an existing change track selection
                // -> clearing selection before (!) applying operations, so that
                // the highlighting attributes are not handled inside the operations
                if (changeTrackSelection) { self.clearChangeTrackSelection(); }

                // applying the operations
                model.setGUITriggeredOperation(true);

                // fire apply actions asynchronously
                operationsDef = model.applyActions({ operations : generator.getOperations()}, { async : true });

                // Wait for a second before showing the progress bar (entering busy mode).
                // Users applying minimal or 'normal' amount of change tracks will not see this progress bar at all.
                progressBarTimer = app.executeDelayed(function () {
                    app.getView().enterBusy({
                        initHandler: function (header, footer) {
                            footer.append(
                                $('<div>').addClass('change-track-progress').append(
                                    $('<div>').addClass('alert alert-warning').append(
                                        $('<div>').text(gt('Applying changes, please wait...'))
                                    )
                                )
                            );
                        },
                        cancelHandler: function () {
                            if (operationsDef) {
                                operationsDef.abort();
                                app.getView().leaveBusy();
                                window.clearTimeout(progressBarTimer);
                            }
                        }
                    });
                }, { delay: 1000 });

                // handle the result of change track operations
                operationsDef
                    .progress(function (progress) {
                        // update the progress bar according to progress of the operations promise
                        app.getView().updateBusyProgress(progress);
                    })
                    .done(function () {
                        // abort and reset existing progress bar timer
                        progressBarTimer.abort();
                        window.clearTimeout(progressBarTimer);
                        // close progress bar and leave busy mode
                        app.getView().leaveBusy();
                    })
                    .always(function () {
                        model.setGUITriggeredOperation(false);
                    });

                // if content was deleted and there was a range, the new selection must be a text cursor at start position
                if (savedStartPosition && contentDeleted) { model.getSelection().setTextSelection(savedStartPosition); }

            }
        };

        /**
         * Updating the change track attributes for a given node.
         *
         * @param {jQuery} node
         *  The node whose character attributes shall be updated, as
         *  jQuery object.
         *
         * @param {Object} mergedAttributes
         *  A map of attribute maps (name/value pairs), keyed by attribute
         *  family, containing the effective attribute values merged from style
         *  sheets and explicit attributes.
         */
        this.updateChangeTrackAttributes = function (node, mergedAttributes) {

            var // the explicit attributes of the specified node
                nodeAttrs = null;

            // helper function to remove all change track attributes
            // from a specified node
            function removeAllChangeTrackAttributesFromNode(oneNode) {
                oneNode.removeAttr('data-change-track-modified');
                oneNode.removeAttr('data-change-track-inserted');
                oneNode.removeAttr('data-change-track-removed');
                oneNode.removeAttr('data-change-track-modified-author');
                oneNode.removeAttr('data-change-track-inserted-author');
                oneNode.removeAttr('data-change-track-removed-author');
            }

            // Setting and deleting changes attributes, but without using attribute from parent!
            if (_.isObject(mergedAttributes.changes)) {

                nodeAttrs = AttributeUtils.getExplicitAttributes(node); // -> no inheritance

                if (_.isObject(nodeAttrs.changes) && (!DOM.isEmptySpan(node) || DOM.isTabSpan(node))) {  // not for empty text spans

                    if (_.isObject(nodeAttrs.changes.modified)) {
                        node.attr('data-change-track-modified', true);  // necessary for visualization of modifications
                        if (nodeAttrs.changes.modified.author) {
                            if (_.indexOf(listOfAuthors, nodeAttrs.changes.modified.author) > -1) {
                                node.attr('data-change-track-modified-author', (_.indexOf(listOfAuthors, nodeAttrs.changes.modified.author) % 8));
                            } else {
                                node.attr('data-change-track-inserted-author', (listOfAuthors.push(nodeAttrs.changes.modified.author) - 1) % 8);
                            }
                        }
                    } else {
                        node.removeAttr('data-change-track-modified');
                        node.removeAttr('data-change-track-modified-author');
                    }

                    if (_.isObject(nodeAttrs.changes.inserted)) {
                        node.attr('data-change-track-inserted', true);  // necessary for visualization of modifications
                        if (nodeAttrs.changes.inserted.author) {
                            if (_.indexOf(listOfAuthors, nodeAttrs.changes.inserted.author) > -1) {
                                node.attr('data-change-track-inserted-author', (_.indexOf(listOfAuthors, nodeAttrs.changes.inserted.author) % 8));
                            } else {
                                node.attr('data-change-track-inserted-author', (listOfAuthors.push(nodeAttrs.changes.inserted.author) - 1) % 8);
                            }
                        }
                    } else {
                        node.removeAttr('data-change-track-inserted');
                        node.removeAttr('data-change-track-inserted-author');
                    }

                    if (_.isObject(nodeAttrs.changes.removed)) {
                        node.attr('data-change-track-removed', true);  // necessary for visualization of modifications
                        if (nodeAttrs.changes.removed.author) {
                            if (_.indexOf(listOfAuthors, nodeAttrs.changes.removed.author) > -1) {
                                node.attr('data-change-track-removed-author', (_.indexOf(listOfAuthors, nodeAttrs.changes.removed.author) % 8));
                            } else {
                                node.attr('data-change-track-inserted-author', (listOfAuthors.push(nodeAttrs.changes.removed.author) - 1) % 8);
                            }
                        }
                    } else {
                        node.removeAttr('data-change-track-removed');
                        node.removeAttr('data-change-track-removed-author');
                    }
                } else {
                    removeAllChangeTrackAttributesFromNode(node);
                }

            } else {
                removeAllChangeTrackAttributesFromNode(node);
            }
        };

        /**
         * Receiving the existing explicit attributes for a given node.
         * They are returned for a change track modification operation.
         *
         * @param {jQuery} node
         *  The node whose character attributes have been changed, as
         *  jQuery object.
         *
         * @returns {Object}
         *  An object with all existing explicit node attributes, taking
         *  special care of the change tracking attributes.
         */
        this.getOldNodeAttributes = function (node) {

            var // an object with the old node attributes
                oldAttrs = {};

            // Expanding operation for change tracking with old explicit attributes
            oldAttrs.attrs = AttributeUtils.getExplicitAttributes(node);

            // special handling for multiple modifications
            if (_.isObject(oldAttrs.attrs.changes) && _.isObject(oldAttrs.attrs.changes.modified) && _.isObject(oldAttrs.attrs.changes.modified.attrs)) {
                // Never overwriting existing changes.modified.attrs, because there might be multiple attribute modifications
                oldAttrs = oldAttrs.attrs.changes.modified;  // -> reusing the already registered old attributes
            }

            // old attributes must not contain the information about the changes
            if (oldAttrs && _.isObject(oldAttrs.attrs.changes)) {
                delete oldAttrs.attrs.changes;
            }

            return oldAttrs;
        };

        /**
         * Setting the change track selection, for example after pressing
         * the 'Next' or 'Previous' button in the change track GUI.
         *
         * @param {Object} [options]
         *  A map with additional options controlling the change track selection.
         *  The following options are supported:
         *  @param {String} [options.next=true]
         *      Whether the following (next is true) or the previous change track
         *      node shall be selected.
         */
        this.getNextChangeTrack = function (options) {

            var // whether the following valid change track shall be found or the previous
                next = Utils.getBooleanOption(options, 'next', true),
                // the next valid selection
                nextSelection = iterateNextValidChangeTrackSelection({ next: next });

            if (nextSelection) {
                setChangeTrackSelection(nextSelection);  // setting CT selection and making it visible
            } else {
                app.getView().yell({ type: 'info', message: gt('There are no tracked changes in your document.') });
            }

        };

        /**
         * Returning the change track selection.
         *
         * @returns {Object|Null}
         *  An object describing the change track selection or Null, if no change track selection
         *  exists.
         */
        this.getChangeTrackSelection = function () {
            return changeTrackSelection;
        };

        /**
         * Returns whether the current selection contains at least one change track element.
         *
         * @returns {Boolean}
         *  Whether the current selection contains at least one change track element.
         */
        this.isOnChangeTrackNode = function () {
            // if there were no changeTracked elements, get out here (for performance)
            if (getAllChangeTrackNodes(rootNode).length === 0) { return false; }

            // if there were changeTracked elements and the hole document is selected, get out here (for performance)
            if (model.getSelection().isAllSelected() && getAllChangeTrackNodes(rootNode).length > 0) { return true; }

            var // receiving at least one change track node from the current selection
                changeTrackNodes = getTrackingNodesInSelection({ minimalSelection: true });

            return (changeTrackNodes && changeTrackNodes.length > 0);
        };

        /**
         * Setting, removing or updating the side bar and the markers for the change tracking.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.invalidate=true]
         *      Whether an existing side bar needs to be invalidated.
         *
         * @returns {Boolean}
         *  Whether there is at least one active change track node in the document.
         */
        this.updateSideBar = function (options) {

            var // the first child of the page content node
                firstPageContentNodeChild = null,
                // the top offset of the page content node
                pageContentNodeTopOffset = 0,
                // the top margin of the first child of the content node
                firstChildTopOffset = 0,
                // the minimum and maximum offset, until which the change track divs are created
                minOffset = 0, maxOffset = 0,
                // whether the current marker in the side pane need to be invalidated
                invalidate = Utils.getBooleanOption(options, 'invalidate', true),
                // the current zoom factor of the document
                zoomFactor = 0;

            // creating one marker in the change track side bar
            // for a specified change track node
            function createSideBarMarker(node) {

                var // the vertical offset of the specific node
                    topOffset = Utils.round($(node).offset().top, 1),
                    // the height and offset of the change track marker
                    markerHeight = 0, markerOffset = 0,
                    // a key for each marker
                    markerKey = null,
                    // whether the node is a table cell
                    isCellNode = DOM.isTableCellNode(node);

                // Performance: Only insert divs, if the value for offset().top is '> minOffset' and
                // smaller than the height of the scrollNode multiplied with 1.5

                if (topOffset > maxOffset) { return Utils.BREAK; }

                if (topOffset > minOffset) {

                    markerHeight = $(node).is('div.p') ? $(node.firstChild).css('line-height') : (isCellNode ? $(node).outerHeight() : $(node).height());
                    markerOffset = Utils.round(((topOffset + firstChildTopOffset - pageContentNodeTopOffset) / zoomFactor), 1);
                    markerKey = markerOffset + '_' + markerHeight;

                    // finally inserting the marker for the change tracked element, if it is not already inserted
                    if (!allSideBarMarker[markerKey]) {
                        sideBarNode.append($('<div>').addClass('ctdiv').height(markerHeight).css('top', markerOffset));
                        allSideBarMarker[markerKey] = 1;
                    }
                }

            }

            // not supporting small devices
            if (Utils.SMALL_DEVICE) { return false; }

            // setting scroll node, if not already done
            if (!scrollNode) { scrollNode = app.getView().getContentRootNode(); }

            // setting page content node, if not already done
            if (!pageContentNode) { pageContentNode = rootNode.children(DOM.PAGECONTENT_NODE_SELECTOR); }

            // invalidating an existing side bar
            if (sideBarNode && invalidate) {
                sideBarNode.remove();
                sideBarNode = null;
                allSideBarMarker = {};
            }

            // collecting all nodes with change track information (not necessary after scrolling)
            if (!allNodesSideBarCollector || invalidate) {
                allNodesSideBarCollector = getAllChangeTrackNodes(); // Very fast call, but avoid if possible
            }

            // drawing the side bar, if there is at least one change track
            if (allNodesSideBarCollector.length > 0) {

                // setting the zoom factor
                zoomFactor = app.getView().getZoomFactor() / 100;

                // the top offset of the content node
                pageContentNodeTopOffset = Utils.round(pageContentNode.offset().top, 1);

                // if the first paragraph in the document has a top margin, this also needs to be added
                // into the calculation. Otherwise all markers are shifted by this top margin.
                firstPageContentNodeChild = Utils.getDomNode(pageContentNode).firstChild;
                if (firstPageContentNodeChild && $(firstPageContentNodeChild).css('margin-top')) {
                    firstChildTopOffset = Utils.round((Utils.convertCssLength($(firstPageContentNodeChild).css('margin-top'), 'px', 1) * zoomFactor), 1);
                }

                // calculating the minimum and maximum offset in that region, for which the
                // side bar marker will be created (for performance reasons)
                maxOffset = 1.5 * (scrollNode.height() + scrollNode.offset().top);  // one and a half screen downwards
                minOffset = -0.3 * maxOffset; // a half screen upwards

                // creating and appending side bar, if it does not exist,
                // and if there is at least one change track
                if (sideBarNode === null) {
                    sideBarNode = $('<div>').addClass('ctsidebar')
                        .css('top',  Utils.convertCssLength(rootNode.css('padding-top'), 'px', 1))
                        .css('left', Utils.round(0.5 * Utils.convertCssLength(rootNode.css('padding-left'), 'px', 1), 1))
                        .height(pageContentNode.height());

                    rootNode.append(sideBarNode);
                }

                // iterating over all collected change tracked nodes
                Utils.iterateArray(allNodesSideBarCollector, function (ctNode) {

                    if (DOM.isTableWithPageBreakRowNode(ctNode)) {
                        // do not mark complete table, because of problem with row inserted for
                        // page break -> mark all rows except the page break row
                        _.each(DOM.getTableRows(ctNode), createSideBarMarker);
                    } else {
                        createSideBarMarker(ctNode);
                    }

                });

            }

            // if there are no change tracks, inform listeners about this
            if (allNodesSideBarCollector.length === 0) {
                model.trigger('changeTrack:stateInfo', { state: false });
            }

            return allNodesSideBarCollector.length > 0;

        };

        // initialization -----------------------------------------------------

    } // class ChangeTrack

    // export =================================================================

    // derive this class from class TriggerObject
    return TriggerObject.extend({ constructor: ChangeTrack });

});
