/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 * @author Mario Schroeder <mario.schroeder@open-xchange.com>
 * @author Edy Haryono <edy.haryono@open-xchange.com>
 * @author Marko Benigar <marko.benigar@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/view',
    ['io.ox/office/tk/utils',
     'io.ox/office/tk/keycodes',
     'io.ox/office/tk/forms',
     'io.ox/office/editframework/utils/color',
     'io.ox/office/editframework/utils/border',
     'io.ox/office/editframework/utils/mixedborder',
     'io.ox/office/editframework/view/editview',
     'io.ox/office/spreadsheet/utils/config',
     'io.ox/office/spreadsheet/utils/sheetutils',
     'io.ox/office/spreadsheet/utils/paneutils',
     'io.ox/office/spreadsheet/utils/clipboard',
     'io.ox/office/spreadsheet/model/operations',
     'io.ox/office/spreadsheet/model/cellcollection',
     'io.ox/office/spreadsheet/view/statuspane',
     'io.ox/office/spreadsheet/view/gridpane',
     'io.ox/office/spreadsheet/view/headerpane',
     'io.ox/office/spreadsheet/view/cornerpane',
     'io.ox/office/spreadsheet/view/controls',
     'io.ox/office/spreadsheet/view/render/renderutils',
     'io.ox/office/spreadsheet/view/render/rendercache',
     'io.ox/office/spreadsheet/view/mixin/selectionmixin',
     'io.ox/office/spreadsheet/view/mixin/celleditmixin',
     'io.ox/office/spreadsheet/view/mixin/dialogsmixin',
     'io.ox/office/spreadsheet/view/mixin/hyperlinkmixin',
     'gettext!io.ox/office/spreadsheet',
     'less!io.ox/office/spreadsheet/view/style'
    ], function (Utils, KeyCodes, Forms, Color, Border, MixedBorder, EditView, Config, SheetUtils, PaneUtils, Clipboard, Operations, CellCollection, StatusPane, GridPane, HeaderPane, CornerPane, Controls, RenderUtils, RenderCache, SelectionMixin, CellEditMixin, DialogsMixin, HyperlinkMixin, gt) {

    'use strict';

    var // class name shortcuts
        Button = Controls.Button,
        CheckBox = Controls.CheckBox,
        Label = Controls.Label,
        UnitField = Controls.UnitField,
        ComponentMenuButton = Controls.ComponentMenuButton,
        ComponentMenuCheckBox = Controls.ComponentMenuCheckBox,

        // warning message text for maximum number of columns/rows that can be modified
        MAX_CHANGE_COLS_WARNING = gt('It is not possible to modify more than %1$d columns at the same time.', _.noI18n(SheetUtils.MAX_CHANGE_COLS_COUNT)),
        MAX_CHANGE_ROWS_WARNING = gt('It is not possible to modify more than %1$d rows at the same time.', _.noI18n(SheetUtils.MAX_CHANGE_ROWS_COUNT)),

        // warning message text for maximum number of cells that can be filled
        MAX_CHANGE_CELLS_WARNING = gt('It is not possible to modify more than %1$d cells at the same time.', _.noI18n(SheetUtils.MAX_FILL_CELL_COUNT)),

        // warning message text for protected cells
        PROTECTED_CELLS_WARNING = gt('Protected cells cannot be modified.'),
        // warning message text for maximum number of cells that can be filled

        // all available zoom factors
        ZOOM_FACTORS = [0.5, 0.75, 1, 1.25, 1.5, 1.75, 2, 2.5, 3, 4, 5, 6, 8],

        // the size of the freeze separator nodes, in pixels
        FREEZE_SIZE = 1,

        // the size of the split separator nodes, in pixels
        SPLIT_SIZE = 2,

        // the highlighted inner size of the split tracking node, in pixels
        TRACKING_SIZE = 4,

        // the additional margin of the split tracking nodes, in pixels
        TRACKING_MARGIN = 3,

        // the position offset of tracking nodes compared to split lines
        TRACKING_OFFSET = (TRACKING_SIZE - SPLIT_SIZE) / 2 + TRACKING_MARGIN,

        // the minimum size of grid/header panes, in pixels
        MIN_PANE_SIZE = 10,

        // tolerance for optimal column width, in 1/100 mm
        SET_COLUMN_WIDTH_TOLERANCE = 50,

        // tolerance for optimal row height (set explicitly), in 1/100 mm
        SET_ROW_HEIGHT_TOLERANCE = 15,

        // tolerance for optimal row height (updated implicitly), in 1/100 mm
        UPDATE_ROW_HEIGHT_TOLERANCE = 50,

        // default settings for a pane side
        DEFAULT_PANE_SIDE_SETTINGS = { visible: true, offset: 0, size: 0, frozen: false, showOppositeScroll: false, hiddenSize: 0 },

        // default value for a visible border
        DEFAULT_SINGLE_BORDER = { style: 'single', width: Utils.convertLengthToHmm(1, 'px'), color: Color.AUTO },

        // default layout data for mixed border attributes
        DEFAULT_BORDER_DATA = {
            borderTop: {},
            borderBottom: {},
            borderLeft: {},
            borderRight: {}
        },

        // default layout data for subtotal results
        DEFAULT_SUBTOTAL_DATA = {
            cells: 0,
            numbers: 0,
            sum: 0,
            min: Number.POSITIVE_INFINITY,
            max: Number.NEGATIVE_INFINITY,
            average: Number.NaN
        },

        // maps border names to the names of the opposite borders and relative cell position
        OPPOSITE_BORDERS = {
            borderTop:    { borderName: 'borderBottom', cols:  0, rows: -1 },
            borderBottom: { borderName: 'borderTop',    cols:  0, rows:  1 },
            borderLeft:   { borderName: 'borderRight',  cols: -1, rows:  0 },
            borderRight:  { borderName: 'borderLeft',   cols:  1, rows:  0 }
        },

        // default options for unit fields (column width and row height)
        DEFAULT_UNIT_FIELD_OPTIONS = {
            width: 70,
            min: 0,
            precision: 100,
            smallStep: 100,
            largeStep: 500,
            roundStep: true
        },

        // the maximum number of returned search result hit
        MAX_SEARCH_RESULT_COUNT = 100,

        // delay time for destruction of unused rendering caches
        RENDER_CACHE_DESTROY_DELAY = (Utils.COMPACT_DEVICE ? 2 : 10) * 60000,

        // clipboard client and server id to identify the copy data when pasting.
        clipboardId = { client: null, server: null };

    // class SpreadsheetView ==================================================

    /**
     * Represents the entire view of a spreadsheet document. Contains the view
     * panes of the sheet currently shown (the 'active sheet'), which contain
     * the scrollable cell grids (there will be several view panes, if the
     * sheet view is split or frozen); and the selections of all existing
     * sheets.
     *
     * Additionally to the events triggered by the base class EditView, an
     * instance of this class triggers events if global contents of the
     * document, or the contents of the current active sheet have been changed.
     * The following events will be triggered:
     * - 'before:activesheet'
     *      Before another sheet will be activated. Event handlers receive the
     *      zero-based index of the old active sheet in the document.
     * - 'change:activesheet'
     *      After another sheet has been activated. Event handlers receive the
     *      zero-based index of the new active sheet in the document, and the
     *      sheet model instance of the new active sheet.
     * - 'change:sheets'
     *      After the collection of all sheets in the document has been
     *      changed (after inserting, deleting, moving, renaming, hiding a
     *      visible, or showing a hidden sheet).
     * - 'change:layoutdata'
     *      After the selection layout data has been updated, either directly
     *      or from the response data of a server view update.
     * - 'change:sheet:attributes'
     *      After the formatting attributes of the active sheet have been
     *      changed.
     * - 'change:sheet:viewattributes'
     *      After the view attributes of the active sheet have been changed.
     *      Event handlers receive an incomplete attribute map containing all
     *      changed view attributes of the active sheet with their new values.
     * - 'change:selection'
     *      Will be triggered directly after the 'change:sheet:viewattributes'
     *      event caused by a changed selection, for convenience. Event
     *      handlers receive the new selection.
     * - 'celledit:enter', 'celledit:leave'
     *      After entering or leaving the cell in-place edit mode.
     * - 'celledit:change'
     *      After changing text while cell in-place edit mode is active.
     * - 'celledit:reject'
     *      After attempting to enter or to leave the cell in-place edit mode
     *      has been rejected, e.g. when trying to edit a protected cell, or
     *      when trying to commit an invalid/incomplete formula expression.
     * - 'change:usedarea'
     *      The number of used columns or rows in the active sheet has been
     *      changed. See class 'CellCollection' for details.
     * - 'change:cells'
     *      The cell collection in the active sheet has been changed. See class
     *      'CellCollection' for details.
     * - 'insert:columns', 'delete:columns', 'change:columns'
     *      The column collection in the active sheet has been changed. See
     *      class 'ColRowCollection' for details.
     * - 'insert:rows', 'delete:rows', 'change:rows'
     *      The row collection in the active sheet has been changed. See class
     *      'ColRowCollection' for details.
     * - 'insert:merged', 'delete:merged'
     *      The merge collection in the active sheet has been changed. See
     *      class 'MergeCollection' for details.
     * - 'insert:drawing', 'delete:drawing', 'change:drawing'
     *      The collection of drawing objects in the active sheet has been
     *      changed. See class 'DrawingCollection' for details.
     *
     * @constructor
     *
     * @extends EditView
     *
     * @param {SpreadsheetApplication} app
     *  The application instance containing this spreadsheet view.
     */
    function SpreadsheetView(app) {

        var // self reference
            self = this,

            // the scrollable root DOM node containing the headers and panes
            rootNode = $('<div class="abs pane-root">'),

            // the top-left corner pane
            cornerPane = null,

            // settings for all pane sides, mapped by pane side identifier
            paneSideSettings = {},

            // the row/column header panes, mapped by position keys
            headerPanes = {},

            // the grid panes, mapped by position keys
            gridPanes = {},

            // the status pane with the sheet tabs
            statusPane = null,

            // rendering caches, mapped by UIDs of the sheet models
            renderCaches = {},

            // the split line separating the left and right panes
            colSplitLineNode = $('<div class="split-line columns">'),

            // the split line separating the top and bottom panes
            rowSplitLineNode = $('<div class="split-line rows">'),

            // the split tracking node separating left and right panes
            colSplitTrackingNode = $('<div class="abs split-tracking columns" style="width:' + TRACKING_SIZE + 'px;padding:0 ' + TRACKING_MARGIN + 'px;" tabindex="-1">'),

            // the split tracking node separating top and bottom panes
            rowSplitTrackingNode = $('<div class="abs split-tracking rows" style="height:' + TRACKING_SIZE + 'px;padding:' + TRACKING_MARGIN + 'px 0;" tabindex="-1">'),

            // the split tracking node covering the intersection of the other tracking points
            centerSplitTrackingNode = $('<div class="abs split-tracking columns rows" style="width:' + TRACKING_SIZE + 'px;height:' + TRACKING_SIZE + 'px;padding:' + TRACKING_MARGIN + 'px;" tabindex="-1">'),

            // all split tracking nodes, as jQuery collection
            allSplitTrackingNodes = colSplitTrackingNode.add(rowSplitTrackingNode).add(centerSplitTrackingNode),

            // tracking overlay nodes for column/row resizing
            resizeOverlayNode = $('<div class="abs resize-tracking"><div class="abs leading"></div><div class="abs trailing"></div></div>'),

            // the collections of visible and hidden sheets
            visibleSheets = [],
            hiddenSheets = [],

            // the spreadsheet document model, and the style sheet containers
            model = null,
            documentStyles = null,
            fontCollection = null,
            styleCollection = null,
            numberFormatter = null,

            // the active sheet model
            activeSheetModel = null,
            activeSheet = -1,
            activeRenderCache = null,

            // collections of the active sheet
            cellCollection = null,
            colCollection = null,
            rowCollection = null,
            mergeCollection = null,
            drawingCollection = null,

            // the contents and formatting of the active cell, and the selected ranges
            activeCellSettings = null,
            selectionSettings = null,

            // the current status text label
            statusText = null,

            // search results
            searchLayoutData = {},

            activeSearchQuery = '',

            // default current selected search occurrence index in the iteration
            searchResultPointer = -1,

            replaceAction = false,

            // formatting attributes used by format painter
            formatPainterAttrs = null;

        // base constructor ---------------------------------------------------

        EditView.call(this, app, {
            initHandler: initHandler,
            initDebugHandler: initDebugHandler,
            initGuiHandler: initGuiHandler,
            initDebugGuiHandler: initDebugGuiHandler,
            grabFocusHandler: grabFocusHandler,
            overlayMargin: { left: 8, right: Utils.SCROLLBAR_WIDTH + 8, top: 20, bottom: Utils.SCROLLBAR_HEIGHT }
        });

        SelectionMixin.call(this, app, rootNode);
        CellEditMixin.call(this, app, rootNode);
        DialogsMixin.call(this, app, rootNode);
        HyperlinkMixin.call(this, app);

        // private methods ----------------------------------------------------

        /**
         * Imports the passed settings for the borders of the selection (in the
         * property 'borders'), and the subtotal results (in the property
         * 'subtotals').
         *
         * @param {Object} settings
         *  Settings for the selected ranges, as received from the respective
         *  server request.
         */
        function importSelectionSettings(settings) {

            // clear the old selection settings
            selectionSettings = {};

            // import mixed border attributes
            selectionSettings.borders = _.copy(DEFAULT_BORDER_DATA, true);
            if (_.isObject(settings.borders)) {
                _.extend(selectionSettings.borders, settings.borders);
                // delete inner borders for specific selections
                if (!self.hasMultipleColumnsSelected()) { delete selectionSettings.borders.borderInsideVert; }
                if (!self.hasMultipleRowsSelected()) { delete selectionSettings.borders.borderInsideHor; }
            }

            // import subtotal results
            selectionSettings.subtotals = _.copy(DEFAULT_SUBTOTAL_DATA, true);
            if (_.isObject(settings.subtotals)) {
                _.extend(selectionSettings.subtotals, settings.subtotals);
                selectionSettings.subtotals.average = selectionSettings.subtotals.sum / selectionSettings.subtotals.numbers;
            }

            // import selection lock data
            selectionSettings.locked = settings.locked === true;
        }

        /**
         * Requests an update of the selection part of the view layout data.
         */
        function requestSelectionUpdate(selection) {
            self.requestUpdate({ selection: { ranges: selection.ranges, active: selection.activeCell } });
        }

        /**
         * Updates the cached data of the active cell (contents, formatting),
         * and the entire selection (border settings, subtotals). To improve
         * performance, updating settings of the entire selection (which is a
         * potentially expensive operation) will be done debounced.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.direct=false]
         *      If set to true, all selection settings will be updated
         *      immediately (without additional debounced update).
         */
        var updateSelectionSettings = (function () {

            var // whether selection update is pending (for option 'direct')
                pendingRangeUpdate = false;

            // update settings of the active cell synchronously for fast GUI feedback
            function updateActiveCell(options) {

                // receive settings of active cell from cell collection
                activeCellSettings = cellCollection.getCellEntry(self.getActiveCell());
                pendingRangeUpdate = true;

                // invoke deferred callback immediately if specified
                if (Utils.getBooleanOption(options, 'direct', false)) {
                    updateSelectedRanges();
                }
            }

            // update settings of the entire selection deferred (expensive cell iterations)
            function updateSelectedRanges() {

                var // the current cell selection
                    selection = self.getSelection(),
                    // all merged ranges covering the selection
                    mergedRanges = null,
                    // object to collect and sort all border types
                    allBorders = {},
                    // whether current selection contains a locked cell
                    selectionLocked = false;

                // selection settings may have been updated immediately (option 'direct')
                if (!pendingRangeUpdate) { return; }
                pendingRangeUpdate = false;

                // If any selected range is not completely covered by the cell collection, or if the
                // selection contains too many cells, request new data from server. Keep old selection
                // settings until that data has been received to prevent flickering icons in the side pane.
                if ((Utils.getSum(selection.ranges, SheetUtils.getCellCount) > SheetUtils.MAX_SELECTION_CELL_COUNT) || !cellCollection.containsRanges(selection.ranges)) {
                    selectionSettings.locked = true;
                    requestSelectionUpdate(selection);
                    return;
                }

                // initialize the selection settings object with default data
                importSelectionSettings({});

                // get all merged ranges covering the selection
                mergedRanges = mergeCollection.getMergedRanges(selection.ranges);

                // iterate all ranges
                cellCollection.iterateCellsInRanges(selection.ranges, function (cellData, originalRange) {

                    var // address of the current cell
                        address = cellData.address,
                        // cloning the global OPPOSITE_BORDERS for local modifications
                        oppositeBorders = _.copy(OPPOSITE_BORDERS, true),
                        // merged range covering the current cell
                        mergedRange = SheetUtils.findFirstRange(mergedRanges, address),
                        // additional columns/rows of a merged cell
                        additionalCols = 0,
                        additionalRows = 0;

                    // special handling for cells that are part of a merged range
                    if (mergedRange) {

                        // skip hidden cells inside merged range
                        if (!_.isEqual(mergedRange.start, address)) { return; }

                        // modify opposite borders for the reference cell of a merged range
                        oppositeBorders.borderRight.cols = SheetUtils.getColCount(mergedRange);
                        oppositeBorders.borderBottom.rows = SheetUtils.getRowCount(mergedRange);
                        additionalCols = oppositeBorders.borderRight.cols - 1;
                        additionalRows = oppositeBorders.borderBottom.rows - 1;
                    }

                    // process all outer borders of the cell
                    _.each(oppositeBorders, function (OPPOSITE_BORDER, borderName) {

                        var // cell data of the preceding/next cell (taking care of merged cells)
                            nextCellData = cellCollection.getCellEntry([address[0] + OPPOSITE_BORDER.cols, address[1] + OPPOSITE_BORDER.rows]),
                            // the border value of the current and adjacent cell
                            thisBorder = cellData.attributes.cell[borderName],
                            nextBorder = nextCellData.attributes.cell[OPPOSITE_BORDER.borderName],
                            // whether the borders are visible
                            thisBorderVisible = Border.isVisibleBorder(thisBorder),
                            nextBorderVisible = Border.isVisibleBorder(nextBorder),
                            // the target border name for the border collection (inner or outer borders)
                            targetBorderName = null,
                            targetBorders = null;

                        // get the target border name for the border collection
                        switch (borderName) {
                        case 'borderLeft':
                            targetBorderName = (address[0] === originalRange.start[0]) ? 'borderLeft' : null;
                            break;
                        case 'borderRight':
                            targetBorderName = (address[0] + additionalCols === originalRange.end[0]) ? 'borderRight' : 'borderInsideVert';
                            break;
                        case 'borderTop':
                            targetBorderName = (address[1] === originalRange.start[1]) ? 'borderTop' : null;
                            break;
                        case 'borderBottom':
                            targetBorderName = (address[1] + additionalRows === originalRange.end[1]) ? 'borderBottom' : 'borderInsideHor';
                            break;
                        }

                        // skip inner border, if already processed by preceding cell
                        if (!_.isString(targetBorderName)) { return; }
                        targetBorders = allBorders[targetBorderName] || (allBorders[targetBorderName] = []);

                        // insert all visible borders into the border collection
                        if (thisBorderVisible) { targetBorders.push(thisBorder); }
                        if (nextBorderVisible) { targetBorders.push(nextBorder); }

                        // ignore invisible border, if the other border is visible
                        if (!thisBorderVisible && !nextBorderVisible) { targetBorders.push(Border.NONE); }
                    });

                    // selection is editable only if all its cells are editable
                    if (!cellData.attributes.cell.unlocked) { selectionLocked = true; }
                });

                // all border information collected and sorted in allBorders
                _.each(allBorders, function (borders, borderName) {
                    selectionSettings.borders[borderName] = MixedBorder.mixBorders(borders);
                });

                // iterate unified ranges to calculate subtotals (no check for result needed, all cells are available)
                cellCollection.iterateCellsInRanges(SheetUtils.getUnifiedRanges(selection.ranges), function (cellEntry) {
                    selectionSettings.subtotals.cells += 1;
                    if (CellCollection.isNumber(cellEntry) && _.isFinite(cellEntry.result)) {
                        selectionSettings.subtotals.numbers += 1;
                        selectionSettings.subtotals.sum += cellEntry.result;
                        selectionSettings.subtotals.min = Math.min(selectionSettings.subtotals.min, cellEntry.result);
                        selectionSettings.subtotals.max = Math.max(selectionSettings.subtotals.max, cellEntry.result);
                    }
                }, { type: 'content' });
                selectionSettings.subtotals.average = selectionSettings.subtotals.sum / selectionSettings.subtotals.numbers;

                // whether selection contains locked cells
                selectionSettings.locked = selectionLocked;

                // notify listeners
                self.trigger('change:layoutdata');
            }

            // create the actual debounced updateSelectionSettings() method
            return app.createDebouncedMethod(updateActiveCell, updateSelectedRanges, { delay: 200 });
        }()); // end of local scope of updateSelectionSettings()

        /**
         * Destroys the rendering cache associated to the specified sheet.
         *
         * @param {String} uid
         *  The unique identifier of a sheet model.
         */
        function destroyRenderCache(uid) {
            if (uid in renderCaches) {
                RenderUtils.log('SpreadsheetView.destroyRenderCache(): deleting cache for uid=' + uid);

                // abort auto-destruction timer
                if (renderCaches[uid].timer) {
                    renderCaches[uid].timer.abort();
                }

                // destroy the cache instance, remove the descriptor from the map
                renderCaches[uid].cache.destroy();
                delete renderCaches[uid];
            }
        }

        /**
         * Triggers a 'before:activesheet' event in preparation to change the
         * current active sheet.
         */
        function prepareSetActiveSheet() {

            // do nothing if no sheet is currently active (initially after import)
            if (!activeSheetModel) { return; }

            var // the unique identifier of the active sheet
                uid = activeSheetModel.getUid();

            // leave cell edit mode before switching sheets; TODO: do not leave if active sheet just moves
            self.leaveCellEditMode('auto');

            // create a timer for auto-destruction of the rendering cache
            renderCaches[uid].timer = app.executeDelayed(function () {
                destroyRenderCache(uid);
            }, { delay: RENDER_CACHE_DESTROY_DELAY });

            // notify listeners
            self.trigger('before:activesheet', activeSheet);
        }

        /**
         * Activates the specified sheet, and triggers a 'change:activesheet'
         * event with the new sheet index.
         *
         * @param {Number} sheet
         *  The zero-based index of the new active sheet.
         */
        function setActiveSheet(sheet) {

            // get model instance and collections of the new active sheet
            activeSheetModel = model.getSheetModel(sheet);
            cellCollection = activeSheetModel.getCellCollection();
            colCollection = activeSheetModel.getColCollection();
            rowCollection = activeSheetModel.getRowCollection();
            mergeCollection = activeSheetModel.getMergeCollection();
            drawingCollection = activeSheetModel.getDrawingCollection();
            activeSheet = sheet;

            var // the unique identifier of the active sheet
                uid = activeSheetModel.getUid(),
                // descriptor for the rendering cache
                cacheData = renderCaches[uid];

            // abort the auto-destruction timer of an existing cache
            if (cacheData && cacheData.timer) {
                cacheData.timer.abort();
                cacheData.timer = null;
            }

            // initialize a new rendering cache
            else if (!cacheData) {
                RenderUtils.log('SpreadsheetView.setActiveSheet(): creating cache for uid=' + uid);

                // create a new cache, initialize the descriptor object for the map
                renderCaches[uid] = cacheData = {
                    cache: new RenderCache(app, activeSheetModel),
                    timer: null
                };

                // forward all events of the rendering cache to the view
                cacheData.cache.on('triggered', function () {
                    if (cacheData.cache === activeRenderCache) {
                        self.trigger.apply(self, _.toArray(arguments).slice(1));
                    }
                });
            }

            // store reference to the active rendering cache
            activeRenderCache = cacheData.cache;

            // notify listeners
            self.trigger('change:activesheet', sheet, activeSheetModel);
            self.trigger('change:sheet:viewattributes', activeSheetModel.getViewAttributes());

            // refresh the layout of the header and grid panes
            initializePanes();

            // refresh selection settings for e.g. updating subtotal list
            updateSelectionSettings();
        }

        /**
         * Updates the collection of visible sheets.
         */
        function updateVisibleSheets() {

            visibleSheets = [];
            hiddenSheets = [];

            // collect all supported and visible sheets
            model.iterateSheetModels(function (sheetModel, sheet, name) {

                // ignore unsupported sheet types
                if (!self.isSheetTypeSupported(sheetModel.getType())) { return; }

                // collect visible and hidden sheets, count hidden sheets
                if (sheetModel.getMergedAttributes().sheet.visible) {
                    visibleSheets.push({ sheet: sheet, name: name });
                } else {
                    hiddenSheets.push({ sheet: sheet, name: name });
                }
            });
        }

        /**
         * Returns the array index of the specified sheet in the collection of
         * all visible sheets.
         */
        function getVisibleIndex(sheet) {
            var arrayIndex = _.sortedIndex(visibleSheets, { sheet: sheet }, 'sheet');
            return ((0 <= arrayIndex) && (arrayIndex < visibleSheets.length) && (visibleSheets[arrayIndex].sheet === sheet)) ? arrayIndex : -1;
        }

        /**
         * Returns the intersection of the passed column/row intervals with the
         * visible intervals of the header panes in the specified direction.
         *
         * @param {Object|Array} intervals
         *  A single column/row interval, or an array of intervals.
         *
         * @param {Boolean} columns
         *  Whether to return the column intervals of the left and right header
         *  panes (true), or the row intervals of the top and bottom header
         *  panes (false).
         *
         * @returns {Array}
         *  The parts of the passed column/row intervals which are covered by
         *  the respective visible header panes.
         */
        function getVisibleIntervals(intervals, columns) {

            var // the resulting intervals covered by the header panes
                resultIntervals = [];

            // pick the intervals of all visible header panes in the correct orientation
            _.each(headerPanes, function (headerPane, paneSide) {
                if ((PaneUtils.isColumnSide(paneSide) === columns) && headerPane.isVisible()) {
                    resultIntervals = resultIntervals.concat(SheetUtils.getIntersectionIntervals(intervals, headerPane.getRenderInterval()));
                }
            });

            // unify intervals in case the header panes cover the same sheet area
            return SheetUtils.getUnifiedIntervals(resultIntervals);
        }

        /**
         * Returns the column/row intervals of the selected ranges, optionally
         * depending on a specific target column/row.
         *
         * @param {Boolean} columns
         *  Whether to return column intervals (true) or row intervals (false).
         *
         * @param {Object} [options]
         *  Additional options. The following options are supported:
         *  @param {Number} [options.target]
         *      If specified, the zero-based index of a specific target column
         *      or row. If the current selection contains entire column/row
         *      ranges covering this column/row, the respective intervals of
         *      these ranges will be returned instead; otherwise a single
         *      interval containing the target column/row will be returned.
         *  @param {Boolean} [options.visible=false]
         *      If set to true, the returned intervals will be reduced to the
         *      column or row intervals covered by the visible panes.
         *
         * @returns {Array}
         *  An array of column/row intervals for the current selection.
         */
        function getSelectedIntervals(columns, options) {

            var // the selected ranges
                ranges = self.getSelectedRanges(),
                // the target column/row
                target = Utils.getIntegerOption(options, 'target'),
                // the resulting intervals
                intervals = null;

            // use entire column/row ranges in selection, if target index has been passed
            if (_.isNumber(target)) {
                // filter full column/row ranges in selection
                ranges = _.filter(ranges, function (range) { return model.isFullRange(range, columns); });
                // if the target column/row is not contained in the selection, make an interval for it
                if (!SheetUtils.rangesContainIndex(ranges, target, columns)) {
                    intervals = [{ first: target, last: target }];
                }
            }

            // convert selected ranges to column/row intervals
            if (!intervals) {
                intervals = SheetUtils.getIntervals(ranges, columns);
            }

            // reduce the intervals to the header pane intervals
            if (Utils.getBooleanOption(options, 'visible', false)) {
                intervals = getVisibleIntervals(intervals, columns);
            }

            return intervals;
        }

        /**
         * Sets the optimal column width in the specified column intervals,
         * based on the content of cells.
         *
         * @param {Object|Array} colIntervals
         *  A single column interval, or an array of column intervals.
         */
        function setOptimalColumnWidth(colIntervals) {

            var // the entire column interval of the right header/grid panes
                rightInterval = headerPanes.right.getRenderInterval(),
                // resulting intervals with the new column widths
                resultIntervals = [],
                // last entry in the column width cache
                lastEntry = null,
                // default width for empty columns
                defWidth = colCollection.getDefaultAttributes().column.width;

            // returns all cell nodes from top and bottom grid pane in the specified column
            function getCellNodesInCol(col) {

                var // whether to use the left or the right panes
                    colPaneSide = (rightInterval && SheetUtils.intervalContainsIndex(rightInterval, col)) ? 'right' : 'left',
                    // the grid panes to be used to extract the cell nodes
                    topGridPane = self.getGridPane(PaneUtils.getPanePos(colPaneSide, 'top')),
                    bottomGridPane = self.getGridPane(PaneUtils.getPanePos(colPaneSide, 'bottom')),
                    // all existing cell nodes in the top and bottom grid panes
                    cellNodes = $();

                // extract all cell nodes from the top and bottom grid panes
                if (topGridPane.isVisible()) { cellNodes = cellNodes.add(topGridPane.getCellNodesInCol(col)); }
                if (bottomGridPane.isVisible()) { cellNodes = cellNodes.add(bottomGridPane.getCellNodesInCol(col)); }
                return cellNodes;
            }

            // process all visible columns
            colCollection.iterateEntries(colIntervals, function (colEntry) {

                var // process only cells that contain an optimal width attribute, and are not merged horizontally
                    cellNodes = getCellNodesInCol(colEntry.index).filter('[data-optimal-width]').not('[data-col-span]'),
                    // resulting optimal width of the current column
                    colWidth = 0;

                // find the maximum value of all content width attributes
                cellNodes.each(function () {
                    colWidth = Math.max(colWidth, Utils.getElementAttributeAsInteger(this, 'data-optimal-width'));
                });

                // use default column width, if no cells are available
                if (colWidth === 0) { colWidth = defWidth; }

                // do not generate operations, if the custom height was not modified before, and the new width is inside the tolerance
                if (!colEntry.attributes.column.customWidth && (Math.abs(colWidth - colEntry.attributes.column.width) < SET_COLUMN_WIDTH_TOLERANCE)) {
                    return;
                }

                // insert the optimal column width into the cache
                if (lastEntry && (lastEntry.last + 1 === colEntry.index) && (lastEntry.width === colWidth)) {
                    lastEntry.last += 1;
                } else {
                    lastEntry = { first: colEntry.index, last: colEntry.index, width: colWidth };
                    resultIntervals.push(lastEntry);
                }
            });

            // create operations for all columns
            model.getUndoManager().enterUndoGroup(function () {
                _.each(resultIntervals, function (resultInterval) {
                    activeSheetModel.setColumnWidth(resultInterval, resultInterval.width, { custom: false });
                });
            });
        }

        /**
         * Sets the optimal row height in the specified row intervals, based on
         * the content of cells.
         *
         * @param {Object|Array} rowIntervals
         *  A single row interval, or an array of row intervals.
         *
         * @param {Object|Array} [updateRanges]
         *  If specified, the addresses of the ranges changed by an operation
         *  which causes updating the automatic row heights implicitly. Rows
         *  with user-defined height (row attribute 'customHeight' set to true)
         *  will not be changed in this case.
         */
        function setOptimalRowHeight(rowIntervals, updateRanges) {

            var // all multi-line DOM cell nodes, mapped by address key
                wrappedCellNodes = {},
                // difference between cell width and text content width
                totalPadding = 2 * activeSheetModel.getEffectiveCellPadding() + 1,
                // tolerance for automatic row height
                heightTolerance = updateRanges ? UPDATE_ROW_HEIGHT_TOLERANCE : SET_ROW_HEIGHT_TOLERANCE,
                // resulting intervals with the new row heights
                resultIntervals = [],
                // last entry in the row height cache
                lastEntry = null;

            // convert passed update ranges to array
            updateRanges = _.isObject(updateRanges) ? _.getArray(updateRanges) : null;

            // find all multi-line cell nodes that are not merged vertically, and map them by cell address
            _.chain(gridPanes).invoke('findCellNodes', '>[data-wrapped]:not([data-row-span])').each(function (paneCellNodes) {
                paneCellNodes.each(function () {
                    var cellNode = $(this);
                    wrappedCellNodes[cellNode.attr('data-address')] = cellNode;
                });
            });

            // process all visible rows
            rowCollection.iterateEntries(rowIntervals, function (rowEntry) {

                var // the entire row range
                    rowRange = model.makeRowRange(rowEntry.index),
                    // resulting optimal height of the current row
                    rowHeight = documentStyles.getLineHeight(rowEntry.attributes.character);

                // find the maximum height of all existing cells using the cell collection (skip merged ranges over multiple rows)
                cellCollection.iterateCellsInRanges(rowRange, function (cellData, origRange, mergedRange) {

                    // skip all cells covered by merged ranges, or reference cells of merged ranges spanning multiple lines
                    if (mergedRange && (!_.isEqual(cellData.address, mergedRange.start) || (SheetUtils.getRowCount(mergedRange) > 1))) {
                        return;
                    }

                    var // whether the cell is located in the updated area
                        isUpdatedCell = updateRanges && SheetUtils.rangesContainCell(updateRanges, cellData.address),
                        // the DOM node of the cell (multi-line text nodes only)
                        cellNode = wrappedCellNodes[SheetUtils.getCellKey(cellData.address)],
                        // the resulting cell height
                        cellHeight = 0;

                    // temporarily render multi-line cells in updated ranges to get effective content height (cells are not yet rendered in the DOM)
                    if (isUpdatedCell && CellCollection.isWrappedText(cellData)) {

                        var charAttributes = _.clone(cellData.attributes.character);
                        charAttributes.fontSize = activeSheetModel.getEffectiveFontSize(charAttributes.fontSize);

                        // use font collection to get a DOM helper node with prepared font settings
                        cellHeight = fontCollection.getCustomFontMetrics(charAttributes, function (helperNode) {

                            var col = cellData.address[0],
                                cellWidth = colCollection.getIntervalPosition(mergedRange ? SheetUtils.getColInterval(mergedRange) : { first: col, last: col }).size,
                                lineHeight = activeSheetModel.getEffectiveLineHeight(cellData.attributes.character);

                            // prepare the helper node for multi-line text
                            helperNode.css({
                                width: Math.max(2, cellWidth - totalPadding),
                                lineHeight: lineHeight + 'px',
                                whiteSpace: 'pre-wrap',
                                wordWrap: 'break-word'
                            });

                            // insert the text spans
                            helperNode[0].innerHTML = _.map(cellData.display.split(/\n/), function (textLine) {
                                return '<span>' + Utils.escapeHTML(Utils.cleanString(textLine)) + '</span>';
                            }).join('<br>');

                            // return the resulting height in 1/100 mm
                            return activeSheetModel.convertPixelToHmm(helperNode.height());
                        });
                    }

                    // a multi-line cell node exists in the DOM, get its content height
                    else if (!isUpdatedCell && cellNode) {
                        // use height stored at the cell node if existing (e.g. vertically justified cells whose content node is too large)
                        cellHeight = Utils.getElementAttributeAsInteger(cellNode, 'data-optimal-height', 0);
                        // fall-back to the height of the content node (converted to 1/100 mm)
                        if (cellHeight === 0) {
                            cellHeight = activeSheetModel.convertPixelToHmm(cellNode.find('.content').height());
                        }
                    }

                    // calculate cell height from text line height according to the font settings of the cell
                    else {
                        cellHeight = documentStyles.getLineHeight(cellData.attributes.character);
                    }

                    rowHeight = Math.max(rowHeight, cellHeight);

                }, { type: 'existing', merged: true });

                // do not generate operations, if the row height was optimal before, and the new height is inside the tolerance
                if (!rowEntry.attributes.row.customHeight && (Math.abs(rowHeight - rowEntry.attributes.row.height) < heightTolerance)) {
                    return;
                }

                // insert the optimal row height into the cache
                if (lastEntry && (lastEntry.last + 1 === rowEntry.index) && (lastEntry.height === rowHeight)) {
                    lastEntry.last += 1;
                } else {
                    lastEntry = { first: rowEntry.index, last: rowEntry.index, height: rowHeight };
                    resultIntervals.push(lastEntry);
                }
            }, { customSize: updateRanges ? false : null }); // do not change custom row height in update mode

            // create operations for all rows
            model.getUndoManager().enterUndoGroup(function () {
                _.each(resultIntervals, function (resultInterval) {
                    activeSheetModel.setRowHeight(resultInterval, resultInterval.height, { custom: false });
                });
            });
        }

        /**
         * Updates the optimal row height in the specified ranges, based on the
         * content of cells.
         *
         * @param {Object|Array} ranges
         *  A single range address, or an array of range addresses.
         */
        function updateOptimalRowHeight(ranges) {

            var // restrict to visible area (performance)
                // TODO: iterate all row intervals, not single rows, important when updating entire column
                rowIntervals = getVisibleIntervals(SheetUtils.getRowIntervals(ranges), false);

            // pass modified ranges to indicate update mode (do not modify rows with custom height)
            setOptimalRowHeight(rowIntervals, ranges);
        }

        /**
         * Fills a cell with the passed value and formatting, and updates the
         * optimal row height of the cell. See method
         * SheetModel.setSingleCellContents() for details.
         */
        function setSingleCellContents(address, value, attributes, options) {
            model.getUndoManager().enterUndoGroup(function () {
                var result = activeSheetModel.setSingleCellContents(address, value, attributes, options);
                if (result === '') {
                    updateOptimalRowHeight({ start: address, end: address });
                }
            });
        }

        /**
         * Fills a cell range with the passed values and formatting, and
         * updates the optimal row height of the affected cells. See method
         * SheetModel.setCellContents() for details.
         */
        function setCellContents(start, contents, options) {
            model.getUndoManager().enterUndoGroup(function () {
                var result = activeSheetModel.setCellContents(start, contents, options);
                if (result === '') {
                    updateOptimalRowHeight({ start: start, end: [start[0], start[1] + contents.length - 1] });
                }
            });
        }

        /**
         * Fills all cell ranges with the same value and formatting, and
         * updates the optimal row height of the affected cells. See method
         * SheetModel.fillCellRanges() for details.
         */
        function fillCellRanges(ranges, value, attributes, options) {
            model.getUndoManager().enterUndoGroup(function () {
                var result = activeSheetModel.fillCellRanges(ranges, value, attributes, options);
                switch (result) {
                case '':
                    updateOptimalRowHeight(ranges);
                    break;
                case 'overflow:cells':
                    self.yell({ type: 'info', message: MAX_CHANGE_CELLS_WARNING });
                    break;
                case 'overflow:cols':
                    self.yell({ type: 'info', message: MAX_CHANGE_COLS_WARNING });
                    break;
                case 'overflow:rows':
                    self.yell({ type: 'info', message: MAX_CHANGE_ROWS_WARNING });
                    break;
                }
            });
        }

        /**
         * Clears all cell ranges, and updates the optimal row height of the
         * affected cells. See method SheetModel.clearCellRanges() for details.
         */
        function clearCellRanges(ranges) {
            model.getUndoManager().enterUndoGroup(function () {
                var result = activeSheetModel.clearCellRanges(ranges);
                if (result === '') {
                    updateOptimalRowHeight(ranges);
                }
            });
        }

        /**
         * Handles changed view attributes in the active sheet. Shows a changed
         * zoom factor in the status label and redraws the view with the new
         * zoom factor, and triggers change events for the attributes.
         */
        function changeSheetViewAttributesHandler(event, attributes) {

            // changed selection: update selection data (cell contents, subtotals, etc.) and global appearance
            if ('selection' in attributes) {
                updateSelectionSettings();
                rootNode.toggleClass('drawing-selection', attributes.selection.drawings.length > 0);
                model.updateSelectionUserData();
            }

            // update header and grid panes, after split/freeze settings have changed
            if (Utils.hasProperty(attributes, /^(zoom$|split)/)) {
                initializePanes();
            }
        }

        /**
         * Handles changed column attributes in the active sheet. Updates the
         * size of frozen panes after the effective width of the columns has
         * changed.
         */
        function changeColumnsHandler(event, interval, attributes) {
            if (self.hasFrozenSplit() && ('column' in attributes) && Utils.hasProperty(attributes.column, /^(width|visible)$/)) {
                initializePanes();
            }
        }

        /**
         * Handles changed row attributes in the active sheet. Updates the size
         * of frozen panes after the effective height of the rows has changed.
         */
        function changeRowsHandler(event, interval, attributes) {
            if (self.hasFrozenSplit() && ('row' in attributes) && Utils.hasProperty(attributes.row, /^(height|visible)$/)) {
                initializePanes();
            }
        }

        /**
         * Prepares insertion of a sheet into the document.
         */
        function beforeInsertSheetHandler(event, sheet) {
            // active sheet will change, if a sheet will be inserted before it
            if (sheet <= activeSheet) {
                prepareSetActiveSheet();
            }
        }

        /**
         * Finalizes insertion of a sheet into the document.
         */
        function afterInsertSheetHandler(event, sheet, sheetModel) {

            // registers an event handler that will be called while the sheet is active
            function handleActiveSheetEvent(eventType, options) {

                var // the local event handler called before the event will be triggered
                    handlerFunc = Utils.getFunctionOption(options, 'handler'),
                    // the type of the event to be triggered
                    triggerType = Utils.getStringOption(options, 'trigger');

                // handle specified event of the event source object, do nothing if the sheet is not active
                sheetModel.on(eventType, function () {
                    if (sheetModel === activeSheetModel) {
                        if (_.isFunction(handlerFunc)) {
                            handlerFunc.apply(self, _.toArray(arguments));
                        }
                        if (_.isString(triggerType)) {
                            self.trigger.apply(self, [triggerType].concat(_.toArray(arguments).slice(1)));
                        }
                    }
                });
            }

            // listen to changes in the sheet, notify listeners of the view
            handleActiveSheetEvent('change:attributes', { trigger: 'change:sheet:attributes' });
            handleActiveSheetEvent('change:viewattributes', { handler: changeSheetViewAttributesHandler, trigger: 'change:sheet:viewattributes' });
            handleActiveSheetEvent('change:columns', { handler: changeColumnsHandler, trigger: 'change:columns' });
            handleActiveSheetEvent('insert:columns', { trigger: 'insert:columns' });
            handleActiveSheetEvent('delete:columns', { trigger: 'delete:columns' });
            handleActiveSheetEvent('change:rows', { handler: changeRowsHandler, trigger: 'change:rows' });
            handleActiveSheetEvent('delete:rows', { trigger: 'delete:rows' });
            handleActiveSheetEvent('insert:rows', { trigger: 'insert:rows' });
            handleActiveSheetEvent('insert:merged', { trigger: 'insert:merged' });
            handleActiveSheetEvent('delete:merged', { trigger: 'delete:merged' });
            handleActiveSheetEvent('change:cells', { trigger: 'change:cells' });
            handleActiveSheetEvent('change:usedarea', { trigger: 'change:usedarea' });
            handleActiveSheetEvent('insert:drawing', { trigger: 'insert:drawing' });
            handleActiveSheetEvent('delete:drawing', { trigger: 'delete:drawing' });
            handleActiveSheetEvent('change:drawing', { trigger: 'change:drawing' });

            // update selection settings debounced in case of mass operations, e.g. undo
            handleActiveSheetEvent('triggered', { handler: app.createDebouncedMethod($.noop, updateSelectionSettings, { delay: 20, maxDelay: 500 }) });

            // update the collection of visible sheets
            updateVisibleSheets();

            // update active sheet index
            if (sheet <= activeSheet) {
                setActiveSheet(activeSheet + 1);
            }

            // notify listeners
            self.trigger('change:sheets');
        }

        /**
         * Prepares deletion of a sheet from the document.
         */
        function beforeDeleteSheetHandler(event, sheet, sheetModel) {

            var // unique identifier of the sehet model
                uid = sheetModel.getUid();

            // Bug 31772: cancel cell edit mode to prevent generating additional operations
            if (activeSheetModel) {
                self.cancelCellEditMode();
            }

            // active sheet will change, if a sheet before will be deleted, or itself
            if (sheet <= activeSheet) {
                prepareSetActiveSheet();
            }

            // delete the rendering cache of the sheet
            destroyRenderCache(uid);
        }

        /**
         * Finalizes deletion of a sheet from the document.
         */
        function afterDeleteSheetHandler(event, sheet) {

            // update the collection of visible sheets
            updateVisibleSheets();

            // update active sheet index
            if (sheet < activeSheet) {
                // sheet before active sheet deleted: decrease index of active sheet
                setActiveSheet(activeSheet - 1);
            } else if (sheet === activeSheet) {
                // active sheet deleted: keep index of active sheet, unless it was the last sheet
                setActiveSheet(Math.min(activeSheet, model.getSheetCount() - 1));
            }

            // notify listeners
            self.trigger('change:sheets');
        }

        /**
         * Prepares moving a sheet inside the document.
         */
        function beforeMoveSheetHandler() {
            // Bug 31772: cancel cell edit mode to prevent generating additional operations
            if (activeSheetModel) {
                self.cancelCellEditMode();
            }
        }

        /**
         * Finalizes moving a sheet inside the document.
         */
        function afterMoveSheetHandler(event, from, to) {

            // update the collection of visible sheets
            updateVisibleSheets();

            if (activeSheet === from) {
                // active sheet has been moved directly
                prepareSetActiveSheet();
                setActiveSheet(to);
            } else if ((from < activeSheet) && (activeSheet <= to)) {
                // active sheet has been moved backwards indirectly
                prepareSetActiveSheet();
                setActiveSheet(activeSheet - 1);
            } else if ((to <= activeSheet) && (activeSheet < from)) {
                // active sheet has been moved forwards indirectly
                prepareSetActiveSheet();
                setActiveSheet(activeSheet + 1);
            }

            // notify listeners
            self.trigger('change:sheets');
        }

        /**
         * Handles a renamed sheet.
         */
        function renameSheetHandler() {

            // update the collection of visible sheets
            updateVisibleSheets();

            // notify listeners
            self.trigger('change:sheets');
        }

        /**
         * Updates the visible sheets collection, if the visibility of any
         * sheet in the document has been changed.
         */
        function changeSheetAttributesHandler(event, sheet, newAttributes, oldAttributes) {

            var // index of the active sheet in the (old) set of visible sheets
                activeVisibleSheet = self.getActiveSheet({ visible: true });

            // do nothing, if the visibility of the sheet has not been changed
            if (newAttributes.sheet.visible === oldAttributes.sheet.visible) { return; }

            // update the collection of visible sheets
            updateVisibleSheets();

            // activate another sheet, if the active sheet has been hidden
            if ((sheet === activeSheet) && !newAttributes.sheet.visible && (visibleSheets.length > 0)) {
                activeVisibleSheet = Utils.minMax(activeVisibleSheet, 0, visibleSheets.length - 1);
                self.setActiveSheet(activeVisibleSheet, { visible: true });
            }

            // notify listeners
            self.trigger('change:sheets');
        }

        /**
         * Shows a notification message, if renaming or copying a sheet has
         * failed. Returns a Boolean value reflecting the passed result error
         * code.
         *
         * @param {String} result
         *  The result code received from the document model after a sheet has
         *  been tried to rename or to copy.
         *
         * @returns {Boolean}
         *  Whether the passed result is NOT an error code (success).
         */
        function yellSheetNameError(result) {

            // show an appropriate notification for the result code
            switch (result) {
            case 'empty':
                self.yell({ type: 'info', message: gt('The sheet name must not be empty. Please enter a name.') });
                break;
            case 'invalid':
                self.yell({ type: 'info', message: gt('This sheet name contains invalid characters. Please ensure that the name does not contain / \\ * ? : [ or ], and does neither start nor end with an apostrophe.') });
                break;
            case 'used':
                self.yell({ type: 'info', message: gt('This sheet name is already used. Please enter another name.') });
                break;
            }

            return result.length === 0;
        }

        /**
         * Handles 'docs:update:cells' notifications. If the active sheet
         * contains changed cells, updates the visible cells in all grid panes.
         */
        function updateNotificationHandler(changedData) {

            // do nothing if no changes contained in the active sheet
            if (activeSheet in changedData) {

                // update selection settings in the active sheet
                updateSelectionSettings();

                // throw away search results if sheet is altered in any way
                if (self.isSearchActive() && !replaceAction) {
                    self.searchClear();
                }
            }
        }

        /**
         * Handles all tracking events for the split separators.
         */
        function splitTrackingHandler(event) {

            var // the event source node
                sourceNode = $(this),
                // whether column split tracking and/or row split tracking is active
                colSplit = sourceNode.hasClass('columns'),
                rowSplit = sourceNode.hasClass('rows'),
                // minimum and maximum position of split lines
                minLeft = self.getHeaderWidth(),
                maxLeft = rootNode.width() - Utils.SCROLLBAR_WIDTH - SPLIT_SIZE,
                minTop = self.getHeaderHeight(),
                maxTop = rootNode.height() - Utils.SCROLLBAR_HEIGHT - SPLIT_SIZE;

            // returns the X position of a split line according to the current event
            function getTrackingSplitLeft() {
                var offset = paneSideSettings.left.offset + paneSideSettings.left.size + (colSplit ? event.offsetX : 0);
                return (offset - MIN_PANE_SIZE < minLeft) ? minLeft : (offset + MIN_PANE_SIZE > maxLeft) ? maxLeft : offset;
            }

            // returns the Y position of a split line according to the current event
            function getTrackingSplitTop() {
                var offset = paneSideSettings.top.offset + paneSideSettings.top.size + (rowSplit ? event.offsetY : 0);
                return (offset - MIN_PANE_SIZE < minTop) ? minTop : (offset + MIN_PANE_SIZE > maxTop) ? maxTop : offset;
            }

            function updateNodeOffset(node, propName, offset, min, max) {
                node.toggleClass('collapsed', (offset <= min) || (offset >= max))
                    .add(centerSplitTrackingNode)
                    .css(propName, (offset - TRACKING_OFFSET) + 'px');
            }

            function finalizeTracking() {
                allSplitTrackingNodes.removeClass('tracking-active');
                self.grabFocus();
            }

            switch (event.type) {

            case 'tracking:start':
                colSplitTrackingNode.toggleClass('tracking-active', colSplit);
                rowSplitTrackingNode.toggleClass('tracking-active', rowSplit);
                break;

            case 'tracking:move':
                if (colSplit) { updateNodeOffset(colSplitTrackingNode, 'left', getTrackingSplitLeft(), minLeft, maxLeft); }
                if (rowSplit) { updateNodeOffset(rowSplitTrackingNode, 'top', getTrackingSplitTop(), minTop, maxTop); }
                break;

            case 'tracking:end':
                var splitWidth = null, splitHeight = null;
                if (colSplit) {
                    splitWidth = getTrackingSplitLeft();
                    splitWidth = ((minLeft < splitWidth) && (splitWidth < maxLeft)) ? activeSheetModel.convertPixelToHmm(splitWidth - self.getHeaderWidth()) : 0;
                } else {
                    splitWidth = activeSheetModel.getSplitWidthHmm();
                }
                if (rowSplit) {
                    splitHeight = getTrackingSplitTop();
                    splitHeight = ((minTop < splitHeight) || (splitHeight < maxTop)) ? activeSheetModel.convertPixelToHmm(splitHeight - self.getHeaderHeight()) : 0;
                } else {
                    splitHeight = activeSheetModel.getSplitHeightHmm();
                }
                activeSheetModel.setDynamicSplit(splitWidth, splitHeight);
                finalizeTracking();
                break;

            case 'tracking:cancel':
                initializePanes();
                finalizeTracking();
                break;
            }
        }

        /**
         * Handles 'keydown' events from the view root node.
         */
        function keyDownHandler(event) {

            // do nothing while in-place edit mode is active
            if (self.isCellEditMode()) { return; }

            // special handling for drawing selection
            if (self.hasDrawingSelection()) {

                // delete selected drawings
                if (KeyCodes.matchKeyCode(event, 'DELETE') || KeyCodes.matchKeyCode(event, 'BACKSPACE')) {
                    if (self.requireEditMode()) { self.deleteDrawings(); }
                    return false;
                }

                return;
            }

            // enter cell edit mode with current cell contents
            if (KeyCodes.matchKeyCode(event, 'F2')) {
                self.enterCellEditMode();
                return false;
            }

            // enter cell edit mode with empty text area
            if (KeyCodes.matchKeyCode(event, 'BACKSPACE')) {
                self.enterCellEditMode({ text: '' });
                return false;
            }

            // clear contents (not formatting) of all sells in the selection
            if (KeyCodes.matchKeyCode(event, 'DELETE')) {
                if (self.requireEditMode()) { self.fillCellRanges(null); }
                return false;
            }

            // selects a drawing, if there is at least one
            if (KeyCodes.matchKeyCode(event, 'F4', { shift: true })) {
                if (drawingCollection.getModelCount() > 0) {
                    self.selectDrawing([0]);
                    self.getActiveGridPane().scrollToDrawingFrame([0]);
                    return false;
                }
            }
        }

        /**
         * Handles 'keypress' events from the view root node. Starts the cell
         * in-place edit mode on-the-fly for valid Unicode characters.
         */
        function keyPressHandler(event) {

            var // the initial text to be inserted into the text area
                initialText = null;

            // do not handle 'keypress' events bubbled up from the active in-place text
            // area, or if drawings are selected
            if (self.isCellEditMode() || self.hasDrawingSelection()) { return; }

            // ignore key events where either CTRL/META or ALT is pressed, ignore
            // SPACE keys with any control keys (used as shortcuts for column/row selection)
            if (((event.charCode > 32) && Utils.boolEq(event.ctrlKey || event.metaKey, event.altKey)) || ((event.charCode === 32) && !KeyCodes.hasModifierKeys(event))) {

                // build string from character code
                initialText = String.fromCharCode(event.charCode);

                // percentage number format: add percent sign to number-like first character
                if ((activeCellSettings.format.cat === 'percent') && (/^[-0-9]$/.test(initialText) || (initialText === app.getDecimalSeparator()))) {
                    initialText += '%';
                }

                // start cell edit mode, and drop the key event
                self.enterCellEditMode({ quick: true, text: initialText, pos: 1 });
                return false;
            }
        }

        /**
         * configure global search for OX Spreadsheet
         */
        function initSearch() {

            // helper variables
            var win = app.getWindow(),
                controller = app.getController(),
                // the ox search panel
                searchDiv = win.nodes.search,
                searchForm = searchDiv.find('form'),
                searchInput = searchForm.find('input[name="query"]'),
                searchButton = searchForm.find('button[data-action="search"]'),
                // additional controls for search & replace
                nextButton = $('<button class="btn btn-default margin-right search-next" style="border-radius:0 4px 4px 0;vertical-align:top;" tabindex="1">' + Forms.createIconMarkup('fa-chevron-right') + '</button>'),
                previousButton = $('<button class="btn btn-default search-prev" style="border-radius:4px 0 0 4px;vertical-align:top;" tabindex="1">' + Forms.createIconMarkup('fa-chevron-left') + '</button>'),
                replaceInput = $('<input type="text" class="form-control" name="replace" tabindex="1">').attr('placeholder', gt('Replace with ...')),
                replaceButton = $('<button class="btn btn-default" style="border-radius:0;" tabindex="1">').text(gt('Replace')),
                replaceAllButton = $('<button class="btn btn-default" tabindex="1">').text(/*#. in search&replace: replace all occurrences */ gt('Replace all')),
                clearReplaceInputIcon = $(Forms.createIconMarkup('fa-times', { classes: 'clear-query' })),
                replaceInputGroup = $('<span class="input-group-btn">'),
                replaceQueryContainer =  $('<div class="search-query-container replace-query-container input-group" style="width:400px;display:inline-table;vertical-align:top;">');

            replaceInputGroup.append(replaceButton, replaceAllButton);
            replaceQueryContainer.append(replaceInput, clearReplaceInputIcon, replaceInputGroup);

            // add next/prev search hit,replace input, replace and replace all buttons
            searchForm.find('.search-query-container').css({ display: 'inline-table'})
                .after(previousButton, nextButton, replaceQueryContainer);

            // init replacement
            win.search.replacement = '';
            win.search.replacementInput = replaceInput;

            // Handles the 'search' event from the search panel.
            function searchHandler() {
                // do a new search if its not done yet or  if search query has changed.
                if (_.isUndefined(searchLayoutData.sheets) || activeSearchQuery !== win.search.query) {
                    controller.executeItem('document/search/search', win.search.query);
                } else {
                    // if there are search results, iterate them with the enter key.
                    self.searchIterate();
                }
            }

            // when search bar is closed
            function searchCloseHandler() {
                self.searchClear({ clearInputs: true });
            }

            // Handles tab and esc key events
            function keyHandler(event) {
                var items, focus, index, down;

                if (event.keyCode === KeyCodes.TAB) {
                    event.preventDefault();

                    items = Forms.findFocusableNodes(this);
                    focus = $(document.activeElement);
                    down = ((event.keyCode === KeyCodes.TAB) && !event.shiftKey);

                    index = (items.index(focus) || 0);
                    if (!down) {
                        index--;
                    } else {
                        index++;
                    }

                    if (index >= items.length) {
                        index = 0;
                    } else if (index < 0) {
                        index = items.length - 1;
                    }
                    items[index].focus();
                    return false;
                }
            }

            // Handles the 'change' event of the search input.
            // Stops the propagation of the 'change' event to all other attached handlers
            // if the focus stays inside the search panel.
            // With that moving the focus out of the search input e.g. via tab
            // doesn't trigger the search and so navigation to other search panel controls
            // is possible.
            function searchInputChangeHandler(event) {
                event.stopImmediatePropagation();
                // if we stop the event propagation we need to update the query here
                win.search.query = win.search.getQuery();
                return false;
            }

            function editmodeChangeHandler(event, editMode) {
                replaceButton.toggleClass('disabled', !editMode);
                replaceAllButton.toggleClass('disabled', !editMode);
            }

            // set the tool tips
            Forms.setToolTip(searchInput, gt('Find text'), 'bottom');
            Forms.setToolTip(searchInput.siblings('i'), gt('Clear'), 'bottom');
            Forms.setToolTip(searchForm.find('button[data-action=search]'), gt('Start search'), 'bottom');
            Forms.setToolTip(previousButton, gt('Select previous search result'), 'bottom');
            Forms.setToolTip(nextButton, gt('Select next search result'), 'bottom');
            Forms.setToolTip(replaceInput, gt('Replacement text'), 'bottom');
            Forms.setToolTip(replaceButton, gt('Replace selected search result and select the next result'), 'bottom');
            Forms.setToolTip(replaceAllButton, gt('Replace all search results'), 'bottom');

            // event handling
            self.listenTo(searchInput, 'keydown', function (event) {
                if (KeyCodes.matchKeyCode(event, 'ENTER')) {
                    // prevent that enter key causing lost of focus
                    event.preventDefault();
                    // do the search
                    searchHandler();
                }
            });
            self.listenTo(searchButton, 'click', searchHandler);
            self.listenTo(previousButton, 'click', function () { controller.executeItem('document/search/previousResult'); });
            self.listenTo(nextButton, 'click', function () { controller.executeItem('document/search/nextResult'); });
            self.listenTo(replaceButton, 'click', function () { controller.executeItem('document/search/replaceSelected'); });
            self.listenTo(replaceAllButton, 'click', function () { controller.executeItem('document/search/replaceAll'); });
            self.listenTo(searchDiv, 'keydown', keyHandler);
            self.listenTo(replaceInput, 'change', function () { win.search.replacement = Utils.cleanString(this.value || ''); });
            self.listenTo(clearReplaceInputIcon, 'click', function () { replaceInput.val(''); win.search.replacement = ''; });

            // TODO: We should make the search able to handle tab between multiple controls in the search bar,
            // registering as first handler to the 'change' event is just an interim solution.
            Utils.bindAsFirstHandler(searchInput, 'change', searchInputChangeHandler, self);

            // subscribe the events handlers above
            self.listenTo(win, 'search:close', searchCloseHandler);

            // handle edit mode change situation
            self.listenTo(model, 'change:editmode', editmodeChangeHandler);
        }

        /**
         * Initialization after construction.
         */
        function initHandler() {

            // store reference to document model and the style sheet containers
            model = app.getModel();
            documentStyles = model.getDocumentStyles();
            fontCollection = documentStyles.getFontCollection();
            styleCollection = documentStyles.getStyleCollection('cell');
            numberFormatter = model.getNumberFormatter();

            // insert the root node into the application pane
            self.insertContentNode(rootNode);

            // update view when document edit mode changes
            model.on('change:editmode', function (event, editMode) {
                rootNode.toggleClass('edit-mode', editMode);
            });

            // insert resize overlay node first (will increase its z-index while active)
            rootNode.append(resizeOverlayNode);

            // insert the pane nodes into the DOM
            _.each(gridPanes, function (gridPane) { rootNode.append(gridPane.getNode()); });
            _.each(headerPanes, function (headerPane) { rootNode.append(headerPane.getNode()); });
            rootNode.append(cornerPane.getNode());

            // append the split lines and split tracking nodes
            rootNode.append(colSplitLineNode, rowSplitLineNode, colSplitTrackingNode, rowSplitTrackingNode, centerSplitTrackingNode);

            // enable tracking and register tracking event handlers
            allSplitTrackingNodes.enableTracking();
            allSplitTrackingNodes.on('tracking:start tracking:move tracking:end tracking:cancel', splitTrackingHandler);

            // handle generic key events
            rootNode.on({ keydown: keyDownHandler, keypress: keyPressHandler });

            // handle sheet collection events
            model.on({
                'insert:sheet:before': beforeInsertSheetHandler,
                'insert:sheet:after': afterInsertSheetHandler,
                'delete:sheet:before': beforeDeleteSheetHandler,
                'delete:sheet:after': afterDeleteSheetHandler,
                'move:sheet:before': beforeMoveSheetHandler,
                'move:sheet:after': afterMoveSheetHandler,
                'rename:sheet': renameSheetHandler,
                'change:sheet:attributes': changeSheetAttributesHandler
            });

            // application notifies changed contents/results of cells
            app.on('docs:update:cells', updateNotificationHandler);

            // activate a sheet in the view after the document has been imported (also if import has failed)
            app.onImport(function () {

                var // index of the active sheet from document attributes
                    activeSheet = documentStyles.getDocumentAttributes().activeSheet,
                    // array index into the visibleSheets array
                    arrayIndex = Math.max(0, Math.min(visibleSheets.length - 1, _.sortedIndex(visibleSheets, { sheet: activeSheet }, 'sheet')));

                // initialize sheet view settings from the imported sheet attributes
                model.iterateSheetModels(function (sheetModel) {
                    sheetModel.initializeViewAttributes();
                });

                // activate a sheet (this causes initial painting)
                self.setActiveSheet(arrayIndex, { visible: true });
            });

            // create the status pane
            self.addPane(statusPane = new StatusPane(app));

            // repaint all grid and header panes on layout changes
            self.on('refresh:layout', initializePanes);

            // update locked state of active sheet in the DOM
            self.on('change:activesheet change:sheet:attributes', function () {
                rootNode.toggleClass('sheet-locked', activeSheetModel.isLocked());
            });

            // trigger a 'change:selection' event for convenience
            self.on('change:sheet:viewattributes', function (event, attributes) {
                if ('selection' in attributes) {
                    self.trigger('change:selection', attributes.selection);
                }
            });

            // update CSS marker class for cell in-place edit mode
            self.on('celledit:enter celledit:leave', function () {
                rootNode.toggleClass('cell-edit', self.isCellEditMode());
            });

            // show error messages if leaviong cell edit mode fails
            self.on('celledit:reject', function (event, cause) {
                var message = null;
                switch (cause) {
                case 'protected':
                    message = PROTECTED_CELLS_WARNING;
                    break;
                case 'invalid':
                    message = gt('The formula contains an error.');
                    break;
                case 'array':
                    message = gt('Parts of a matrix formula cannot be changed.');
                    break;
                }
                if (message) {
                    self.yell({ type: 'info', message: message });
                } else {
                    Utils.warn('SpreadsheetView.initHandler(): cell edit mode: invalid error code: "' + cause + '"');
                }
            });

            // process events triggered by header panes
            _.each(headerPanes, function (headerPane, paneSide) {

                var columns = PaneUtils.isColumnSide(paneSide),
                    offsetAttr = columns ? 'left' : 'top',
                    sizeAttr = columns ? 'width' : 'height',
                    showResizeTimer = null;

                function showResizerOverlayNode(offset, size) {
                    resizeOverlayNode.attr('data-orientation', columns ? 'columns' : 'rows');
                    updateResizerOverlayNode(offset, size);
                    // Delay displaying the overlay nodes, otherwise they cover
                    // the resizer drag nodes of the header pane which will
                    // interfere with double click detection.
                    showResizeTimer = app.executeDelayed(function () {
                        rootNode.addClass('tracking-active');
                    }, { delay: 200 });
                }

                function updateResizerOverlayNode(offset, size) {
                    var relativeOffset = paneSideSettings[paneSide].offset + offset - headerPane.getVisiblePosition().offset;
                    resizeOverlayNode.toggleClass('collapsed', size === 0);
                    resizeOverlayNode.find('>.leading').css(sizeAttr, (Math.max(0, relativeOffset) + 10) + 'px');
                    resizeOverlayNode.find('>.trailing').css(offsetAttr, (relativeOffset + size) + 'px');
                }

                function hideResizerOverlayNode() {
                    if (showResizeTimer) {
                        showResizeTimer.abort();
                        showResizeTimer = null;
                    }
                    rootNode.removeClass('tracking-active');
                    resizeOverlayNode.find('>.leading').css(sizeAttr, '');
                    resizeOverlayNode.find('>.trailing').css(offsetAttr, '');
                }

                // visualize resize tracking from header panes
                headerPane.on({
                    'resize:start': function (event, offset, size) {
                        if (self.leaveCellEditMode('auto', { validate: true })) {
                            showResizerOverlayNode(offset, size);
                        } else {
                            event.preventDefault();
                        }
                    },
                    'resize:move': function (event, offset, size) {
                        updateResizerOverlayNode(offset, size);
                    },
                    'resize:end': function () {
                        hideResizerOverlayNode();
                    }
                });
            });

            // initialize selection settings, used until first real update
            importSelectionSettings({});
            activeCellSettings = _.copy(CellCollection.DEFAULT_CELL_DATA, true);
            activeCellSettings.explicit = {};
            activeCellSettings.attributes = styleCollection.getStyleSheetAttributes(styleCollection.getDefaultStyleId());

            // init global search for spreadsheet
            initSearch();
        }

        /**
         * Additional debug initialization after construction.
         */
        function initDebugHandler(operationsPane, clipboardPane) {

            // create the output nodes in the operations pane
            operationsPane
                .addDebugInfoNode('version', { header: 'application', tooltip: 'Calc Engine version' })
                .addDebugInfoHeader('selection', { tooltip: 'Current sheet selection' })
                .addDebugInfoNode('sheet', { tooltip: 'Active (visible) sheet' })
                .addDebugInfoNode('ranges', { tooltip: 'Selected cell ranges' })
                .addDebugInfoNode('draw', { tooltip: 'Selected drawing objects' })
                .addDebugInfoNode('auto', { tooltip: 'Auto-fill range' })
                .addDebugInfoHeader('formatting', { tooltip: 'Formatting of the active cell' })
                .addDebugInfoNode('cell', { tooltip: 'Explicit cell attributes of active cell' })
                .addDebugInfoNode('char', { tooltip: 'Explicit character attributes of active cell' })
                .addDebugInfoNode('style', { tooltip: 'Cell style sheet of active cell' })
                .addDebugInfoNode('format', { tooltip: 'Number format settings of active cell' });

            // log the version of the Calc Engine
            app.onImport(function () {
                operationsPane.setDebugInfoText('application', 'version', app.getEngineVersion());
            });

            // log information about the active sheet
            self.on('change:activesheet change:sheets change:usedarea', function () {
                var used = (cellCollection && (cellCollection.getUsedCols() > 0) && (cellCollection.getUsedRows() > 0)) ? SheetUtils.getRangeName({ start: [0, 0], end: cellCollection.getLastUsedCell() }) : '<empty>';
                operationsPane.setDebugInfoText('selection', 'sheet', 'index=' + activeSheet + ', name="' + self.getSheetName() + '", used=' + used);
            });

            // log all selection events
            self.on('change:sheet:viewattributes', function (event, attributes) {
                if ('selection' in attributes) {
                    var selection = attributes.selection;
                    operationsPane
                        .setDebugInfoText('selection', 'ranges', 'count=' + selection.ranges.length + ', ranges=' + SheetUtils.getRangesName(selection.ranges) + ', active=' + selection.activeRange + ', cell=' + SheetUtils.getCellName(selection.activeCell) + (selection.originCell ? (', origin=' + SheetUtils.getCellName(selection.originCell)) : ''))
                        .setDebugInfoText('selection', 'draw', 'count=' + selection.drawings.length + ', frames=' + ((selection.drawings.length > 0) ? ('[' + (selection.drawings).map(JSON.stringify).join() + ']') : '<none>'));
                }
                if ('autoFillData' in attributes) {
                    var autoFill = attributes.autoFillData;
                    operationsPane.setDebugInfoText('selection', 'auto', autoFill ? ('border=' + autoFill.border + ', count=' + autoFill.count) : '<none>');
                }
            });

            // log explicit formatting of the active cell
            self.on('change:layoutdata', function () {
                var attributes = activeCellSettings.explicit;

                var cellAttrs = attributes.cell;
                if (cellAttrs && cellAttrs.numberFormat && cellAttrs.numberFormat.id && (!cellAttrs.numberFormat.code || !cellAttrs.numberFormat.code - length)) {
                    cellAttrs = _.copy(cellAttrs, true);
                    cellAttrs.numberFormat.code = app.getModel().getNumberFormatter().getBuiltInFormatCode(cellAttrs.numberFormat.id) || '';
                }

                function stringify(attrs) { return JSON.stringify(attrs || {}).replace(/^\{(.*)\}$/, '$1').replace(/"(\w+)":/g, '$1:').replace(/ /g, '\xb7'); }
                operationsPane
                    .setDebugInfoText('formatting', 'cell', stringify(cellAttrs))
                    .setDebugInfoText('formatting', 'char', stringify(attributes.character))
                    .setDebugInfoText('formatting', 'style', _.isString(attributes.styleId) ? ('id="' + attributes.styleId + '", name="' + styleCollection.getName(attributes.styleId) + '"') : '')
                    .setDebugInfoText('formatting', 'format', stringify(activeCellSettings.format));
            });

            // process debug events triggered by grid panes
            _.each(gridPanes, function (gridPane) {
                gridPane.on('debug:clipboard', function (event, content) {
                    clipboardPane.setClipboardContent(content);
                });
            });
        }

        /**
         * Initialization after importing the document. Creates all tool boxes
         * in the side pane and overlay pane. Needed to be executed after
         * import, to be able to hide specific GUI elements depending on the
         * file type.
         */
        function initGuiHandler(viewMenuGroup) {

            // the 'Format' tool bar
            self.createToolBarTab('format', { label: Controls.FORMAT_HEADER_LABEL, visibleKey: 'document/editable/cell' });

            // create and prepare all 'format'-toolbars
            var fontToolBar             = self.createToolBar('format', {priority: 1, prepareShrink: true, icon: 'fa-font', tooltip: Controls.FONT_STYLES_LABEL }),
                fontStyleToolBar        = self.createToolBar('format', {priority: 2, prepareShrink: true, icon: 'docs-font-bold', caretTooltip: gt('More font styles'), splitBtn: { key: 'character/bold', options: Controls.BOLD_BUTTON_OPTIONS }}),
                painterToolBar          = self.createToolBar('format', {priority: 3}),
                colorToolBar            = self.createToolBar('format', {priority: 4, prepareShrink: true, icon: 'docs-color', tooltip: gt('Colors') }),
                cellToolBar             = self.createToolBar('format', {priority: 5, prepareShrink: true, icon: 'fa-align-center', tooltip: Controls.ALIGNMENT_HEADER_LABEL }),
                numberformatToolBar     = self.createToolBar('format', {priority: 6, hideable: true, prepareShrink: true, icon: 'docs-percent', tooltip: Controls.NUMBERFORMAT_HEADER_LABEL }),
                cellborderToolBar       = self.createToolBar('format', {priority: 7, hideable: true, prepareShrink: true, icon: 'docs-cell-style', tooltip: Controls.CELL_BORDER_LABEL }),
                cellstyleToolBar        = self.createToolBar('format', {priority: 8, hideable: true});

            // font
            fontToolBar
                .addGroup('character/fontname', new Controls.FontFamilyPicker(app))
                .addGap()
                .addGroup('character/fontsize', new Controls.FontHeightPicker());

            // font styles
            fontStyleToolBar
                .addGroup('character/bold', new Button(Controls.BOLD_BUTTON_OPTIONS))
                .addGroup('character/italic', new Button(Controls.ITALIC_BUTTON_OPTIONS))
                .addGroup('character/underline', new Button(Controls.UNDERLINE_BUTTON_OPTIONS))
                .addSeparator({classes: 'noVerticalSeparator'})
                .addGroup('character/format', new ComponentMenuButton(app, { icon: 'docs-font-format', tooltip: gt('More font styles'), dropDownVersion: {visible: false}})
                    .addGroup('character/strike',    new Button(Controls.STRIKEOUT_BUTTON_OPTIONS))
                    .addSeparator()
                    .addGroup('cell/reset',   new Button(Controls.CLEAR_FORMAT_BUTTON_OPTIONS))
                )
                .addGroup('character/strike', new Button(Utils.extendOptions(Controls.STRIKEOUT_BUTTON_OPTIONS, {dropDownVersion: {visible: true}})))
                .addSeparator({classes: 'hidden'})
                .addGroup('cell/reset', new Button(Utils.extendOptions(Controls.CLEAR_FORMAT_BUTTON_OPTIONS, {dropDownVersion: {visible: true}})));

            // painter
            painterToolBar
                .addGroup('cell/painter', new Button({ icon: 'docs-format-painter', tooltip: Controls.FORMAT_PAINTER_LABEL, toggle: true, dropDownVersion: { label: Controls.FORMAT_PAINTER_LABEL } }));

            // colors
            colorToolBar
                .addGroup('character/color', new Controls.TextColorPicker(app))
                .addGroup('cell/fillcolor', new Controls.FillColorPicker(app));

            // alignment
            cellToolBar
                .addGroup('cell/alignhor',  new Controls.CellHAlignmentPicker())
                .addGroup('cell/alignvert', new Controls.CellVAlignmentPicker())
                .addGroup('cell/linebreak', new Button({ icon: 'docs-insert-linebreak', tooltip: /*#. checkbox: automatic word wrapping (multiple text lines) in spreadsheet cells */ gt('Automatic word wrap'), toggle: true, dropDownVersion: { label: gt('Automatic word wrap') } }))
                .addGroup('cell/merge',     new Controls.MergePicker(app, {dropDownVersion: { label: gt('Merge or unmerge cells') }}));

            // number format
            numberformatToolBar
                .addGroup('cell/numberformat/category', new Controls.FormatCategoryGroup(app))
                .addGroup('cell/numberformat/category', new Controls.FormatCategoryPicker(app))
                .addGroup('cell/numberformat/code', new Controls.FormatCodePicker(app));

            // borders
            if (app.isOOXML()) {
                cellborderToolBar
                    .addGroup('cell/border/mode', new Controls.CellBorderPicker(app,                {dropDownVersion: { label: Controls.CELL_BORDERS_LABEL }}))
                    .addGroup('cell/border/style/preset', new Controls.BorderPresetStylePicker(app, {dropDownVersion: { label: Controls.BORDER_STYLE_LABEL }}))
                    .addGroup('cell/border/color', new Controls.CellBorderColorPicker(app,          {dropDownVersion: { label: Controls.BORDER_COLOR_LABEL }}));
            } else {
                cellborderToolBar
                    .addGroup('cell/border/mode', new Controls.CellBorderPicker(app,        {dropDownVersion: { label: Controls.CELL_BORDERS_LABEL }}))
                    .addGroup('cell/border/style', new Controls.BorderLineStylePicker(      {dropDownVersion: { label: Controls.BORDER_STYLE_LABEL }}))
                    .addGroup('cell/border/width', new Controls.BorderLineWidthPicker(      {dropDownVersion: { label: Controls.BORDER_WIDTH_LABEL }}))
                    .addGroup('cell/border/color', new Controls.CellBorderColorPicker(app,  {dropDownVersion: { label: Controls.BORDER_COLOR_LABEL }}));
            }

            // cell styles
            cellstyleToolBar
                .addGroup('cell/stylesheet', new Controls.CellStylePicker(app));





            // the 'Insert' tool bar
            self.createToolBarTab('insert', { label: Controls.INSERT_HEADER_LABEL, visibleKey: 'document/editable/cell' });

            self.createToolBar('insert')
                .addGroup('cell/autoformula', new Button({icon: 'docs-auto-sum', label: gt('Sum'), tooltip: /*#. automatically create a SUM function for selected cells */ gt('Sum automatically'), value: 'SUM', smallerVersion: { css: { width: 35 }, hideLabel: true } }))
                .addSeparator()
                .addGroup('character/hyperlink/dialog', new Button(Controls.HYPERLINK_BUTTON_OPTIONS))
                .addSeparator()
                .addGroup('image/insert/dialog', new Button(Controls.INSERT_IMAGE_BUTTON_OPTIONS))
                .addGap()
                .addGroup('chart/insert', new Controls.ChartTypePicker(app, Controls.INSERT_CHART_BUTTON_OPTIONS));





            // the 'Rows/Columns' tool bar
            self.createToolBarTab('rowscols', { label: /* menu title: insert/delete/resize rows and columns in the sheet */ gt.pgettext('menu-title', 'Rows/Columns'), visibleKey: 'document/editable/cell' });

            self.createToolBar('rowscols')
                .addGroup('row/insert', new Button({ icon: 'docs-table-insert-row', tooltip: gt('Insert row') }))
                .addGroup('row/delete', new Button({ icon: 'docs-table-delete-row', tooltip: gt('Delete selected rows') }))
                .addGroup('row/height/active', new Label({ icon: 'docs-table-resize-row', tooltip: gt('Row height'), smallerVersion: {hide: true} }))
                .addGroup('row/height/active', new UnitField(Utils.extendOptions(DEFAULT_UNIT_FIELD_OPTIONS, { tooltip: gt('Row height'), max: SheetUtils.MAX_ROW_HEIGHT, smallerVersion: {hide: true} })))
                .addSeparator()
                .addGroup('column/insert', new Button({ icon: 'docs-table-insert-column', tooltip: gt('Insert column') }))
                .addGroup('column/delete', new Button({ icon: 'docs-table-delete-column', tooltip: gt('Delete selected columns') }))
                .addGroup('column/width/active', new Label({ icon: 'docs-table-resize-column', tooltip: gt('Column width'), smallerVersion: {hide: true} }))
                .addGroup('column/width/active', new UnitField(Utils.extendOptions(DEFAULT_UNIT_FIELD_OPTIONS, { tooltip: gt('Column width'), max: SheetUtils.MAX_COLUMN_WIDTH, smallerVersion: {hide: true} })));



            // the 'Drawing' tool bar
            self.createToolBarTab('drawing', { visibleKey: 'document/editable/drawing', labelKey: 'drawing/type/label' });

            // general drawing objects
            self.createToolBar('drawing')
                .addGroup('drawing/delete', new Button(Controls.DELETE_DRAWING_BUTTON_OPTIONS));



            // the 'Chart' (Drawing) tool bar
            var draw_one =      self.createToolBar('drawing', { visibleKey: 'drawing/chart', priority: 1 }),
                draw_two =      self.createToolBar('drawing', { visibleKey: 'drawing/chart', priority: 2, hideable: true }),
                draw_three =    self.createToolBar('drawing', { visibleKey: 'drawing/chart', priority: 3, hideable: true }),
                draw_four =     self.createToolBar('drawing', { visibleKey: 'drawing/chart', priority: 4, hideable: true });

            draw_one
                .addGroup('drawing/charttype', new Controls.ChartTypePicker(app))
                .addGap()
                .addGroup(null, new Controls.ChartLayerMenuButton(app, /*#. menu-title: menu-title of the chart-modify-menu */ gt.pgettext('menu-title', 'Labels & Axes'), { label: /*#. menu-title: button-label to open the chart-modify-menu */ gt.pgettext('menu-title', 'Labels'), tooltip: /*#. menu-tooltip: button-label to open the chart-modify-menu */ gt.pgettext('menu-title', 'Settings for labels and axes'), focusableNodes: $('.clipboard'), anchorSelection: '.drawing.selected', autoClose: false }))
                .addSeparator()
                .addGroup('drawing/chartcolorset', new Controls.ChartColorSetPicker(app))
                .addGroup('drawing/chartstyleset', new Controls.ChartStyleSetPicker(app));

            draw_two
                .addGroup(null, new ComponentMenuButton(app, Controls.CHART_DATA_POINTS_BUTTON_OPTIONS)
                    .addGroup('drawing/chartdatalabel', new CheckBox(Controls.CHART_SHOW_POINT_LABELS_BUTTON_OPTIONS))
                    .addGroup('drawing/chartvarycolor', new CheckBox(Controls.CHART_VARY_POINT_COLORS_BUTTON_OPTIONS))
                );

            draw_three
                .addGroup('drawing/chartlegend/pos', new Controls.ChartLegendPicker());

            draw_four
                .addGroup(null, new ComponentMenuButton(app, { label: /*#. menu title: options to modify the data source of a chart object */ gt.pgettext('chart-source', 'Data source') })
                    .addGroup('drawing/chartsource', new CheckBox({ label: /*#. change source data for a chart object in a spreadsheet */ gt.pgettext('chart-source', 'Edit data references') }))
                    .addGroup('drawing/chartexchange', new Button({ icon: 'fa-none', label: /*#. switch orientation of data series in rectangular source data of a chart object */ gt.pgettext('chart-source', 'Switch rows and columns') }))
                );







            // the 'View' drop-down menu

            viewMenuGroup
                .addHeaderLabel(Controls.ZOOM_LABEL)
                .addGroup('view/zoom/dec', new Button(Controls.ZOOMOUT_BUTTON_OPTIONS), { inline: true, sticky: true })
                .addGroup('view/zoom/inc', new Button(Controls.ZOOMIN_BUTTON_OPTIONS), { inline: true, sticky: true })
                .addGroup('view/zoom', new Controls.PercentageLabel(), { inline: true })
                .addHeaderLabel(/*#. menu title: settings for splits and frozen columns/rows in a spreadsheet */ gt.pgettext('sheet-split', 'Split and freeze'))
                .addGroup('view/split/dynamic', new CheckBox({label: /*#. checkbox: split a spreadsheet into 2 or 4 different parts that can be scrolled independently */ gt.pgettext('sheet-split', 'Split sheet'), tooltip: gt.pgettext('sheet-split', 'Split the sheet above and left of the current cursor position') }))
                .addGroup('view/split/frozen', new ComponentMenuCheckBox(app, {label: /*#. checkbox: split a spreadsheet into 2 or 4 different parts, the leading (left/upper) parts are frozen and cannot be scrolled */ gt.pgettext('sheet-split', 'Freeze sheet'), tooltip: gt.pgettext('sheet-split', 'Freeze the rows above and the columns left of the cursor') })
                    .addGroup('view/split/frozen/fixed', new Button({ label: gt('Freeze first row'), tooltip: gt('Freeze the first visible row'), value: { cols: 0, rows: 1 } }))
                    .addGroup('view/split/frozen/fixed', new Button({ label: gt('Freeze first column'), tooltip: gt('Freeze the first visible column'), value: { cols: 1, rows: 0 } }))
                    .addGroup('view/split/frozen/fixed', new Button({ label: gt('Freeze first row and column'), tooltip: gt('Freeze the first visible row and the first visible column'), value: { cols: 1, rows: 1 } }))
                )
                .addHeaderLabel(Controls.OPTIONS_LABEL)
                // note: we do not set aria role to 'menuitemcheckbox' or 'button' due to Safari just working correctly with 'checkox'. CheckBox constructor defaults aria role to 'checkbox'
                .addGroup('view/toolbars/show', new CheckBox(Controls.SHOW_TOOLBARS_CHECKBOX_OPTIONS))
                .addGroup('document/users', new CheckBox(Controls.SHOW_COLLABORATORS_CHECKBOX_OPTIONS))
                .addGroup('view/grid/show', new CheckBox({label: /*#. check box label: show/hide cell grid in the sheet */ gt('Show grid lines'), tooltip: gt('Show or hide the cell grid lines in the current sheet') }))
                .addGroup('view/statuspane/show', new CheckBox({label: /*#. check box label: show/hide the sheet tabs at the bottom border */ gt('Show sheet tabs'), tooltip: gt('Show or hide the sheet tabs below the sheet area') }));

        }

        /**
         * Additional initialization of debug GUI after importing the document.
         */
        function initDebugGuiHandler(viewMenuGroup, actionMenuGroup) {

            function requestUpdate(type) {
                switch (type) {
                case 'view':
                    updateNotificationHandler({ changed: { sheets: Utils.makeSimpleObject(activeSheet, { cells: [model.getSheetRange()] } ) } });
                    break;
                case 'cells':
                    updateNotificationHandler({ changed: { sheets: Utils.makeSimpleObject(activeSheet, { cells: self.getSelectedRanges() } ) } });
                    break;
                case 'selection':
                    requestSelectionUpdate(self.getSelection());
                    break;
                }
            }

            function insertDrawing() {
                var generator = model.createOperationsGenerator(),
                    index = drawingCollection.getModelCount(),
                    selection = _.extend(self.getSelection(), { drawings: [] });
                _.each(self.getSelectedRanges(), function (range) {
                    var attrs = { drawing: {
                        name: 'Diagram ' + (index + 1),
                        description: 'Very, very, very, very, very, very, very, very, very long description for this diagram.',
                        startCol: range.start[0],
                        startColOffset: 100,
                        startRow: range.start[1],
                        startRowOffset: 100,
                        endCol: range.end[0],
                        endColOffset: 2400,
                        endRow: range.end[1],
                        endRowOffset: 400
                    } };
                    generator.generateDrawingOperation(Operations.INSERT_DRAWING, activeSheet, [index], { type: 'diagram', attrs: attrs });
                    selection.drawings.push([index]);
                    index += 1;
                });
                model.applyOperations(generator.getOperations());
                activeSheetModel.setViewAttribute('selection', selection);
            }

            // additional controller items for the debug controls
            app.getController().registerDefinitions({
                'debug/view/update': {
                    parent: 'debug/enabled',
                    set: requestUpdate
                },
                'debug/insert/drawing': {
                    parent: ['debug/enabled', 'document/editable'],
                    set: insertDrawing
                }
            });

            // add debug actions to the drop-down menu
            actionMenuGroup
                .addHeaderLabel(_.noI18n('View update'))
                .addGroup('debug/view/update', new Button({ label: _.noI18n('Request full view update'),              value: 'view' }))
                .addGroup('debug/view/update', new Button({ label: _.noI18n('Request view update of selected cells'), value: 'cells' }))
                .addGroup('debug/view/update', new Button({ label: _.noI18n('Request update of selection data'),      value: 'selection' }))
                .addHeaderLabel(_.noI18n('Drawings'))
                .addGroup('debug/insert/drawing', new Button({ label: _.noI18n('Insert a generic drawing object') }));
        }

        /**
         * Moves the browser focus into the active sheet pane.
         */
        function grabFocusHandler() {
            // prevent JS errors when importing the document fails, and the view does not have activated a sheet yet
            if (activeSheetModel) {
                self.getActiveGridPane().grabFocus();
            }
        }

        /**
         * Refreshes the visibility, position, and size of all header panes,
         * grid panes, and tracking nodes.
         */
        function initializePanes() {

            var // the size of the header corner node (and thus of the row/column header nodes)
                headerWidth = 0, headerHeight = 0,
                // whether frozen split mode is active
                frozenSplit = self.hasFrozenSplit(),
                // whether dynamic split mode is really active
                dynamicSplit = !frozenSplit && self.hasSplit(),
                // the size of the split lines
                splitLineSize = frozenSplit ? FREEZE_SIZE : SPLIT_SIZE,
                // whether the split lines are visible
                colSplit = false, rowSplit = false,
                // start position of the split lines
                splitLineLeft = activeSheetModel.getSplitWidth(),
                splitLineTop = activeSheetModel.getSplitHeight();

            // expand dynamic splits to minimum split size
            if (dynamicSplit) {
                if (splitLineLeft > 0) { splitLineLeft = Math.max(splitLineLeft, MIN_PANE_SIZE); }
                if (splitLineTop > 0) { splitLineTop = Math.max(splitLineTop, MIN_PANE_SIZE); }
            }

            // whether the left and top panes are visible
            paneSideSettings.left.visible = (dynamicSplit || frozenSplit) && (splitLineLeft > 0);
            paneSideSettings.top.visible = (dynamicSplit || frozenSplit) && (splitLineTop > 0);

            // calculate current size of header nodes
            // TODO: use current maximum row index visible in the panes
            cornerPane.initializePaneLayout(model.getMaxRow());
            headerWidth = self.getHeaderWidth();
            headerHeight = self.getHeaderHeight();

            // calculate inner width of left panes
            if (paneSideSettings.left.visible) {
                paneSideSettings.left.offset = headerWidth;
                paneSideSettings.left.size = splitLineLeft;
                paneSideSettings.left.hiddenSize = frozenSplit ? colCollection.convertScrollAnchorToPixel(activeSheetModel.getViewAttribute('anchorLeft')) : 0;
            } else {
                _.extend(paneSideSettings.left, { offset: 0, size: 0, hiddenSize: 0 });
            }

            // calculate inner height of top panes
            if (paneSideSettings.top.visible) {
                paneSideSettings.top.offset = headerHeight;
                paneSideSettings.top.size = splitLineTop;
                paneSideSettings.top.hiddenSize = frozenSplit ? rowCollection.convertScrollAnchorToPixel(activeSheetModel.getViewAttribute('anchorTop')) : 0;
            } else {
                _.extend(paneSideSettings.top, { offset: 0, size: 0, hiddenSize: 0 });
            }

            // calculate effective position of split lines
            splitLineLeft = paneSideSettings.left.offset + paneSideSettings.left.size;
            splitLineTop = paneSideSettings.top.offset + paneSideSettings.top.size;

            // determine whether right and bottom panes are visible (must have enough room in split and frozen mode)
            paneSideSettings.right.visible = !paneSideSettings.left.visible || (splitLineLeft + splitLineSize + MIN_PANE_SIZE + Utils.SCROLLBAR_WIDTH <= rootNode.width());
            paneSideSettings.bottom.visible = !paneSideSettings.top.visible || (splitLineTop + splitLineSize + MIN_PANE_SIZE + Utils.SCROLLBAR_HEIGHT <= rootNode.height());

            // visibility of the split lines
            colSplit = paneSideSettings.left.visible && paneSideSettings.right.visible;
            rowSplit = paneSideSettings.top.visible && paneSideSettings.bottom.visible;

            // calculate the resulting grid pane positions and sizes
            if (paneSideSettings.right.visible) {
                paneSideSettings.right.offset = colSplit ? (splitLineLeft + splitLineSize) : headerWidth;
                paneSideSettings.right.size = rootNode.width() - paneSideSettings.right.offset;
                paneSideSettings.right.hiddenSize = frozenSplit ? (paneSideSettings.left.hiddenSize + paneSideSettings.left.size) : 0;
            } else if (paneSideSettings.left.visible) {
                paneSideSettings.left.size = rootNode.width() - headerWidth;
            }
            if (paneSideSettings.bottom.visible) {
                paneSideSettings.bottom.offset = rowSplit ? (splitLineTop + splitLineSize) : headerHeight;
                paneSideSettings.bottom.size = rootNode.height() - paneSideSettings.bottom.offset;
                paneSideSettings.bottom.hiddenSize = frozenSplit ? (paneSideSettings.top.hiddenSize + paneSideSettings.top.size) : 0;
            } else if (paneSideSettings.top.visible) {
                paneSideSettings.top.size = rootNode.height() - headerHeight;
            }

            // set frozen mode (left/top panes are not scrollable in frozen mode in their
            // own direction, e.g. left frozen panes are not scrollable to left/right)
            paneSideSettings.left.frozen = frozenSplit;
            paneSideSettings.right.frozen = false;
            paneSideSettings.top.frozen = frozenSplit;
            paneSideSettings.bottom.frozen = false;

            // set up scroll bar visibility in the opposite direction (e.g. left/right
            // scroll bars of top panes are hidden, if bottom panes are visible)
            paneSideSettings.left.showOppositeScroll = !paneSideSettings.right.visible;
            paneSideSettings.right.showOppositeScroll = true;
            paneSideSettings.top.showOppositeScroll = !paneSideSettings.bottom.visible;
            paneSideSettings.bottom.showOppositeScroll = true;

            // initialize the header panes
            _.each(headerPanes, function (headerPane, paneSide) {
                headerPane.initializePaneLayout(paneSideSettings[paneSide]);
            });

            // initialize the grid panes
            _.each(gridPanes, function (gridPane, panePos) {
                gridPane.initializePaneLayout(paneSideSettings[PaneUtils.getColPaneSide(panePos)], paneSideSettings[PaneUtils.getRowPaneSide(panePos)]);
            });

            // visibility and position of the split lines
            colSplitLineNode.toggle(colSplit).css({ left: splitLineLeft, width: splitLineSize });
            rowSplitLineNode.toggle(rowSplit).css({ top: splitLineTop, height: splitLineSize });

            // visibility and position of the split tracking nodes
            colSplitTrackingNode.toggle(dynamicSplit && colSplit).css({ left: splitLineLeft - TRACKING_OFFSET });
            rowSplitTrackingNode.toggle(dynamicSplit && rowSplit).css({ top: splitLineTop - TRACKING_OFFSET });
            centerSplitTrackingNode.toggle(dynamicSplit && colSplit && rowSplit).css({ left: splitLineLeft - TRACKING_OFFSET, top: splitLineTop - TRACKING_OFFSET });
        }

        // public methods -----------------------------------------------------

        /**
         * Returns the specified grid pane.
         *
         * @param {String} panePos
         *  The position identifier of the grid pane.
         *
         * @returns {GridPane}
         *  The grid pane instance at the specified pane position.
         */
        this.getGridPane = function (panePos) {
            return gridPanes[panePos];
        };

        /**
         * Returns the specified grid pane, if it is visible. Otherwise,
         * returns the nearest visible grid pane.
         *
         * @param {String} panePos
         *  The position identifier of the preferred grid pane. If only one
         *  grid pane is visible, it will be returned regardless of this
         *  parameter. If one of the pane sides contained in the passed pane
         *  position is not visible, returns the grid pane from the other
         *  visible pane side (for example, if requesting the top-left pane
         *  while only bottom panes are visible, returns the bottom-left pane).
         *
         * @returns {GridPane}
         *  The grid pane instance at the specified pane position.
         */
        this.getVisibleGridPane = function (panePos) {

            // jump to other column pane, if the pane side is hidden
            if (!paneSideSettings[PaneUtils.getColPaneSide(panePos)].visible) {
                panePos = PaneUtils.getNextColPanePos(panePos);
            }

            // jump to other row pane, if the pane side is hidden
            if (!paneSideSettings[PaneUtils.getRowPaneSide(panePos)].visible) {
                panePos = PaneUtils.getNextRowPanePos(panePos);
            }

            // now, panePos points to a visible grid pane (at least one pane is always visible)
            return gridPanes[panePos];
        };

        /**
         * Returns the grid pane that is currently focused.
         *
         * @returns {GridPane}
         *  The focused grid pane instance.
         */
        this.getActiveGridPane = function () {
            return this.getVisibleGridPane(activeSheetModel.getViewAttribute('activePane'));
        };

        /**
         * Invokes the passed iterator function for all grid panes contained in
         * this view.
         *
         * @param {Function} iterator
         *  The iterator function invoked for all grid pane instances. Receives
         *  the following parameters:
         *  (1) {GridPane} gridPane
         *      The current grid pane instance.
         *  (2) {String} panePos
         *      The position identifier of the grid pane.
         *  If the iterator returns the Utils.BREAK object, the iteration
         *  process will be stopped immediately.
         *
         * @param {Object} [options]
         *  Additional optional parameters:
         *  @param {Object} [options.context]
         *      If specified, the iterator will be called with this context
         *      (the symbol 'this' will be bound to the context inside the
         *      iterator function).
         *
         * @returns {Utils.BREAK|Undefined}
         *  A reference to the Utils.BREAK object, if the iterator has returned
         *  Utils.BREAK to stop the iteration process, otherwise undefined.
         */
        this.iterateGridPanes = function (iterator, options) {
            return _.any(gridPanes, function (gridPane, panePos) {
                return iterator.call(this, gridPane, panePos) === Utils.BREAK;
            }, Utils.getOption(options, 'context')) ? Utils.BREAK : undefined;
        };

        /**
         * Returns the specified header pane.
         *
         * @param {String} paneSide
         *  The identifier of the pane side ('left', 'right', 'top', 'bottom').
         *
         * @returns {HeaderPane}
         *  The header pane instance at the specified pane side.
         */
        this.getHeaderPane = function (paneSide) {
            return headerPanes[paneSide];
        };

        /**
         * Returns the horizontal header pane (left or right) for the specified
         * pane position.
         *
         * @param {String} panePos
         *  The position identifier of the grid pane.
         *
         * @returns {HeaderPane}
         *  The horizontal header pane instance at the corresponding pane side.
         */
        this.getColHeaderPane = function (panePos) {
            return headerPanes[PaneUtils.getColPaneSide(panePos)];
        };

        /**
         * Returns the vertical header pane (top or bottom) for the specified
         * pane position.
         *
         * @param {String} panePos
         *  The position identifier of the grid pane.
         *
         * @returns {HeaderPane}
         *  The vertical header pane instance at the corresponding pane side.
         */
        this.getRowHeaderPane = function (panePos) {
            return headerPanes[PaneUtils.getRowPaneSide(panePos)];
        };

        /**
         * Returns the first visible header pane in the specified direction
         * (for example, for columns returns the left header pane, if it is
         * visible, otherwise the right header pane).
         *
         * @param {Boolean} columns
         *  If set to true, returns the first visible column header pane (left
         *  or right); otherwise returns the first visible row header pane (top
         *  or bottom).
         *
         * @returns {HeaderPane}
         *  The first visible header pane instance in the specified direction.
         */
        this.getVisibleHeaderPane = function (columns) {
            return columns ?
                (paneSideSettings.left.visible ? headerPanes.left : headerPanes.right) :
                (paneSideSettings.top.visible ? headerPanes.top : headerPanes.bottom);
        };

        /**
         * Returns a header pane associated to the grid pane that is currently
         * focused.
         *
         * @param {Boolean} columns
         *  If set to true, returns the column header pane (left or right);
         *  otherwise returns the row header pane (top or bottom).
         *
         * @returns {HeaderPane}
         *  A header pane instance associated to the grid pane that is
         *  currently focused.
         */
        this.getActiveHeaderPane = function (columns) {
            var panePos = activeSheetModel.getViewAttribute('activePane');
            return columns ? this.getColHeaderPane(panePos) : this.getRowHeaderPane(panePos);
        };

        /**
         * Invokes the passed iterator function for all header panes contained
         * in this view.
         *
         * @param {Function} iterator
         *  The iterator function invoked for all header pane instances.
         *  Receives the following parameters:
         *  (1) {HeaderPane} headerPane
         *      The current header pane instance.
         *  (2) {String} paneSide
         *      The identifier of the pane side ('left', 'top', etc.).
         *  If the iterator returns the Utils.BREAK object, the iteration
         *  process will be stopped immediately.
         *
         * @param {Object} [options]
         *  Additional optional parameters:
         *  @param {Object} [options.context]
         *      If specified, the iterator will be called with this context
         *      (the symbol 'this' will be bound to the context inside the
         *      iterator function).
         *
         * @returns {Utils.BREAK|Undefined}
         *  A reference to the Utils.BREAK object, if the iterator has returned
         *  Utils.BREAK to stop the iteration process, otherwise undefined.
         */
        this.iterateHeaderPanes = function (iterator, options) {
            return _.any(headerPanes, function (headerPane, paneSide) {
                return iterator.call(this, headerPane, paneSide) === Utils.BREAK;
            }, Utils.getOption(options, 'context')) ? Utils.BREAK : undefined;
        };

        /**
         * Returns the top-left corner pane.
         *
         * @returns {CornerPane}
         *  The corner pane instance.
         */
        this.getCornerPane = function () {
            return cornerPane;
        };

        /**
         * Returns the current width of the row header panes.
         *
         * @returns {Number}
         *  The current width of the row header panes, in pixels.
         */
        this.getHeaderWidth = function () {
            return cornerPane.getNode().outerWidth();
        };

        /**
         * Returns the current height of the column header panes.
         *
         * @returns {Number}
         *  The current height of the column header panes, in pixels.
         */
        this.getHeaderHeight = function () {
            return cornerPane.getNode().outerHeight();
        };

        /**
         * Returns the status pane containing the sheet tabs, and the subtotal
         * label.
         */
        this.getStatusPane = function () {
            return statusPane;
        };

        /**
         * Collects several update requests sends a single deferred server
         * request.
         *
         * @param {Object} requestProps
         *  The properties to be inserted into the current request data object.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.requestUpdate = (function () {

            var // the data object collecting all grid pane requests
                requestData = null,
                // the last running request
                runningDef = null,
                // the start time of the initial request (without other requests already running)
                startTime = 0,
                // counter for running requests (next free index)
                nextRequestIndex = 0,
                // minimum request index valid for evaluation (ignore requests after switching to another sheet)
                requiredRequestIndex = 0;

            // direct callback: register a new sheet rectangle to be updated in a grid pane
            function registerUpdateRequest(requestProps) {

                // do not request anything in error mode
                if (app.isInQuit() || (app.getState() === 'error')) { return self; }

                // initialize request data, if missing or for another sheet
                if (!requestData || (requestData.sheet !== activeSheet)) {
                    requestData = { sheet: activeSheet, locale: Config.LOCALE };
                }

                // extend the array of cell ranges
                if (_.isObject(requestProps.cells)) {
                    requestProps.cells = _.getArray(requestProps.cells);
                    if (_.isArray(requestData.cells)) {
                        requestProps.cells = requestData.cells.concat(requestProps.cells);
                    }
                }

                // insert the passed request data
                _.extend(requestData, requestProps);

                return self;
            }

            // deferred callback: send single server request and notify all waiting grid panes
            function executeUpdateRequest() {

                var // local reference to the current request dats
                    localRequestData = requestData,
                    // current request index
                    localRequestIndex = nextRequestIndex,
                    // start time of this request
                    localStartTime = _.now(),
                    // the new server request
                    def = null;

                if (!localRequestData || app.isInQuit() || (app.getState() === 'error')) { return; }
                requestData = null;

                // take start time, if no other request is running already
                if (!runningDef) {
                    startTime = _.now();
                }

                // restrict requested ranges to visible area of all current grid panes
                if (_.isArray(localRequestData.cells)) {
                    Utils.error('SpreadsheetView.executeUpdateRequest(): cell data requested via view update');
                }

                // immediately delete the request data (new updates may be requested in the meantime)
                localRequestIndex = nextRequestIndex;
                nextRequestIndex += 1;
                Utils.info('SpreadsheetView.executeUpdateRequest(): sending update request #' + (localRequestIndex + 1) + ':', localRequestData);

                // wait for pending actions (bug 30044, must wait for the
                // 'insertSheet' operation before requesting data for the sheet)
                def = runningDef = app.saveChanges().then(function () {
                    return app.sendFilterRequest({
                        method: 'POST',
                        params: {
                            action: 'updateview',
                            requestdata: JSON.stringify(localRequestData)
                        }
                    });
                });

                def.fail(function () {
                    Utils.error('SpreadsheetView.executeUpdateRequest(): update request #' + (localRequestIndex + 1) + ' failed');
                })
                .done(function (layoutData) {

                    Utils.info('SpreadsheetView.executeUpdateRequest(): update response #' + (localRequestIndex + 1) + ' received (' + (_.now() - localStartTime) + 'ms):', layoutData);
                    // check validity of response object
                    if (!_.isObject(layoutData) || !_.isObject(layoutData.sheet)) {
                        Utils.error('SpreadsheetView.executeUpdateRequest(): missing required layout data');
                        return;
                    }

                    // ignore this response if the active sheet has changed in the meantime
                    if (localRequestIndex < requiredRequestIndex) { return; }

                    // if this update request is triggered by the search function
                    if (_.isObject(localRequestData.search) && !_.isObject(layoutData.found)) {
                        // be silent if this is a search result refresh request.
                        // a search result refresh request is fired if a local/remote user changes the document while searching.
                        if (!localRequestData.search.refresh) {
                            self.yell({
                                type: 'info',
                                message: /*#. Search&Replace: used when the searched text has not been found */ gt('There are no cells matching "%1$s".', localRequestData.search.searchText)
                            });
                        }
                    }

                    // store search results
                    if (_.isObject(layoutData.found)) {
                        // save search result
                        searchLayoutData = _.copy(layoutData.found, true);
                        // jump to first search result only if this is a new search.
                        if (!replaceAction) {
                            // iterate initially to first result
                            self.searchIterate();
                        }
                    }

                    // copy layout data for the current selection
                    if (_.isObject(layoutData.selection)) {
                        importSelectionSettings(layoutData.selection);
                    }

                    // Check if this is the last request. Do not notify listeners about response
                    // data of previous requests, but collect the response data and all change
                    // flags of multiple requests, and notify listeners after last response.
                    if (def === runningDef) {
                        runningDef = null;
                        self.trigger('change:layoutdata');
                    } else if (_.now() - startTime >= 3000) {
                        startTime = _.now();
                        self.trigger('change:layoutdata');
                    }
                });
            }

            // delete the current cached request data when the active sheet changes
            self.on('before:activesheet', function () {
                requestData = null;
                requiredRequestIndex = nextRequestIndex;
            });

            // create and return the debounced method SpreadsheetView.requestUpdate()
            return app.createDebouncedMethod(registerUpdateRequest, executeUpdateRequest, { delay: 100, maxDelay: 250 });

        }()); // SpreadsheetView.requestUpdate()

        /**
         * Tries to initiate an edit action. if the document is in read-only
         * mode, a notification will be shown to the user.
         *
         * @returns {Boolean}
         *  Whether modifying the document is posssible.
         */
        this.requireEditMode = function () {
            var editMode = model.getEditMode();
            if (!editMode) { app.rejectEditAttempt(); }
            return editMode;
        };

        /**
         * Tries to initiate modifying the active cell in the active sheet. If
         * the document is in read-only mode, or the active sheet and th active
         * cell are locked, a notification will be shown to the user.
         *
         * @returns {Boolean}
         *  Whether modifying the current active cell is posssible.
         */
        this.requireEditableActiveCell = function () {
            if (!this.requireEditMode()) { return false; }
            var locked = this.isActiveCellLocked();
            if (locked) { this.yell({ type: 'info', message: PROTECTED_CELLS_WARNING }); }
            return !locked;
        };

        /**
         * Tries to initiate modifying all cells in the current selection of
         * the active sheet. If the document is in read-only mode, or the
         * active sheet is locked and the selection contains locked cells, a
         * notification will be shown to the user.
         *
         * @returns {Boolean}
         *  Whether modifying the current selection is posssible.
         */
        this.requireEditableSelection = function () {
            if (!this.requireEditMode()) { return false; }
            var locked = this.isSelectionLocked();
            if (locked) { this.yell({ type: 'info', message: PROTECTED_CELLS_WARNING }); }
            return !locked;
        };

        /**
         * Returns the current status text label.
         *
         * @returns {String}
         *  The current status text.
         */
        this.getStatusLabel = function () {
            return statusText;
        };

        /**
         * Sets the passed text as current status text label, shown in the
         * status pane at the bottom border of the application pane.
         *
         * @param {String} text
         *  The new status text.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.setStatusLabel = function (text) {
            if (statusText !== text) {
                statusText = text;
                app.getController().update();
            }
            return this;
        };

        /**
         * Returns the current zoom factor of the active sheet.
         *
         * @returns {Number}
         *  The current zoom factor, as floating-point number. The value 1
         *  represents a zoom of 100%.
         */
        this.getZoom = function () {
            return activeSheetModel.getViewAttribute('zoom');
        };

        /**
         * Changes the current zoom factor of all sheets in the document.
         *
         * @param {Number} zoom
         *  The new zoom factor, as floating-point number. The value 1
         *  represents a zoom of 100%.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.setZoom = function (zoom) {
            activeSheetModel.setViewAttribute('zoom', zoom);
            return this;
        };

        /**
         * Decreases the zoom factor of the active sheet in the document.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.decZoom = function () {

            var // current zoom factor
                currZoom = this.getZoom(),
                // find last entry in ZOOM_FACTORS with a factor less than current zoom
                prevZoom = Utils.findLast(ZOOM_FACTORS, function (zoom) { return zoom < currZoom; });

            // set new zoom at active sheet
            if (_.isNumber(prevZoom)) { this.setZoom(prevZoom); }
            return this;
        };

        /**
         * Increases the zoom factor of the active sheet in the document.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.incZoom = function () {

            var // current zoom factor
                currZoom = this.getZoom(),
                // find first entry in ZOOM_FACTORS with a factor greater than current zoom
                nextZoom = _.find(ZOOM_FACTORS, function (zoom) { return zoom > currZoom; });

            // set new zoom at active sheet
            if (_.isNumber(nextZoom)) { this.setZoom(nextZoom); }
            return this;
        };

        /**
         * Returns the effective zoom factor to be used for the active sheet,
         * according to the value of the 'zoom' view attribute, and the type of
         * the current device. On touch devices, the effective zoom factor will
         * be increased for better usability.
         *
         * @returns {Number}
         *  The effective zoom factor for the active sheet.
         */
        this.getEffectiveZoom = function () {
            return activeSheetModel.getEffectiveZoom();
        };

        /**
         * Calculates the effective font size for the passed font size,
         * according to the current zoom factor.
         *
         * @param {Number} fontSize
         *  The original font size, in points.
         *
         * @returns {Number}
         *  The effective font size, in points.
         */
        this.getEffectiveFontSize = function (fontSize) {
            return activeSheetModel.getEffectiveFontSize(fontSize);
        };

        /**
         * Calculates the effective line height to be used in spreadsheet cells
         * for the passed character attributes, according to the current zoom
         * factor of this sheet.
         *
         * @param {Object} charAttributes
         *  Character formatting attributes influencing the line height. See
         *  SheetModel.getEffectiveLineHeight() for details.
         *
         * @returns {Number}
         *  The line height for the passed character attributes, in pixels.
         */
        this.getEffectiveLineHeight = function (charAttributes) {
            return activeSheetModel.getEffectiveLineHeight(charAttributes);
        };

        /**
         * Returns the effective horizontal padding between cell grid lines and
         * the text contents of the cell for the current zoom factor.
         *
         * @returns {Number}
         *  The effective horizontal cell content padding, in points.
         */
        this.getEffectiveCellPadding = function () {
            return activeSheetModel.getEffectiveCellPadding();
        };

        /**
         * Returns the effective grid color of the active sheet. The automatic
         * color will be replaced with black.
         *
         * @param {Object} autoColor
         *  The color to be returned as replacement for the automatic color.
         *
         * @returns {Object}
         *  The effective grid color of the active sheet.
         */
        this.getEffectiveGridColor = function (autoColor) {
            return activeSheetModel ? activeSheetModel.getEffectiveGridColor(autoColor) : autoColor;
        };

        /**
         * Returns the effective font size for the current zoom factor to be
         * used for the column and row labels in all header panes.
         *
         * @returns {Number}
         *  The effective font size for all header panes, in points.
         */
        this.getHeaderFontSize = function () {
            return Utils.minMax(Utils.round(5 * this.getEffectiveZoom() + 6, 0.1), 8, 30);
        };

        // cells and ranges ---------------------------------------------------

        /**
         * Returns whether a range consists of a single cell, or whether it is
         * a merged range.
         *
         * @param {Object} range
         *  A logical range address.
         *
         * @returns {Boolean}
         *  Whether the cell is the only (merged) cell in the range.
         */
        this.isSingleCellInRange = function (range) {
            return (SheetUtils.getCellCount(range) === 1) || mergeCollection.isMergedRange(range);
        };

        /**
         * Returns the address of a visible cell located as close as possible
         * to the passed cell address.
         *
         * @param {Number[]} address
         *  The logical address of the cell.
         *
         * @param {String} method
         *  Specifies how to look for another cell, if the specified cell is
         *  hidden. The following lookup methods are supported:
         *  - 'exact':
         *      Does not look for other columns/rows. If the specified cell is
         *      hidden, returns null.
         *  - 'next':
         *      Looks for a visible column and/or row following the cell.
         *  - 'prev':
         *      Looks for a visible column and/or row preceding the cell.
         *  - 'nextPrev':
         *      First, looks for visible column and/or row following the cell.
         *      If there are none available, looks for a visible column and/or
         *      row preceding the cell. It may happen that a preceding column,
         *      but a following row will be found, and vice versa.
         *  - 'prevNext':
         *      First, looks for visible column and/or row preceding the cell.
         *      If there are none available, looks for a visible column and/or
         *      row preceding the cell. It may happen that a preceding column,
         *      but a following row will be found, and vice versa.
         *
         * @returns {Number[]|Null}
         *  The address of a visible cell near the passed cell address; or null
         *  if all columns or all rows are hidden.
         */
        this.getVisibleCell = function (address, method) {

            var // collection entries of visible column/row
                colEntry = colCollection.getVisibleEntry(address[0], method),
                rowEntry = rowCollection.getVisibleEntry(address[1], method);

            return (colEntry && rowEntry) ? [colEntry.index, rowEntry.index] : null;
        };

        /**
         * Returns the subtotal results in the current selection.
         *
         * @returns {Object}
         *  The current subtotal results in the current selection, in the
         *  properties 'sum', 'min', 'max', 'numbers', 'cells', and 'average'.
         */
        this.getSubtotals = function () {
            return _.clone(selectionSettings.subtotals);
        };

        /**
         * Returns whether the active sheet, and the current active cell in the
         * selection is locked.
         *
         * @returns {Boolean}
         *  Whether the active cell in the current selection is locked.
         */
        this.isActiveCellLocked = function () {
            return !activeCellSettings.attributes.cell.unlocked && activeSheetModel.isLocked();
        };

        /**
         * Returns whether the active sheet is locked, and the current
         * selection in the active sheet contains at least one locked cell.
         *
         * @returns {Boolean}
         *  Whether the cells in the current selection are locked.
         */
        this.isSelectionLocked = function () {
            return selectionSettings.locked && activeSheetModel.isLocked();
        };

        /**
         * Returns whether the passed ranges are editable, whereas editable
         * means that no locked cells are found in the ranges, and the sheet
         * where these ranges are, is not protected.
         *
         * @param {Object|Array} ranges
         *  The logical address of a single range, or an array of logical
         *  range addresses whose cells will be iterated.
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved if the
         *  passed ranges are editable, and rejected if they are not.
         */
        this.areRangesEditable = function (ranges) {

            var // the Deferred object of the AJAX request
                request = null,
                // whether all cells in all ranges are editable (not locked)
                rangesEditable = true,
                // the number of cells visited locally
                visitedCells = 0;

            // return resolved Deferred object and quit if sheet is not locked
            if (!activeSheetModel.isLocked()) { return $.when(); }

            // prevent processing any cell twice
            ranges = SheetUtils.getUnifiedRanges(ranges);

            // visit all cells locally, if they are covered by the cell collection
            if (cellCollection.containsRanges(ranges)) {

                // visit only the existing cells in the collection (performance),
                // but include all cells in all hidden columns and rows
                cellCollection.iterateCellsInRanges(ranges, function (cellData) {
                    visitedCells += 1;
                    if (!cellData.attributes.cell.unlocked) {
                        rangesEditable = false;
                        return Utils.BREAK; // locked cell found: exit loop
                    }
                }, { type: 'existing', hidden: 'all' });

                // if 'rangesEditable' is false, a locked cell has been found already
                if (!rangesEditable) { return $.Deferred().reject(); }

                // if the ranges contain undefined cells not visited in the iterator above,
                // check the default attributes of empty cells
                if (visitedCells < Utils.getSum(ranges, SheetUtils.getCellCount)) {
                    if (!cellCollection.getDefaultAttributes().cell.unlocked) {
                        return $.Deferred().reject();
                    }
                }

                // all cells are editable
                return $.when();
            }

            // ranges are not covered by the cell collections, send server request
            // to detect the locked state of the specified cells
            request = app.sendFilterRequest({
                method: 'POST',
                params: {
                    action: 'updateview',
                    requestdata: JSON.stringify({
                        sheet: activeSheet,
                        locale: Config.LOCALE,
                        selection: { ranges: ranges, active: ranges[0].start }
                    })
                }
            });

            // reject Deferred object if selection contains a locked cell
            return request.then(function (response) {
                return response.selection.locked ? $.Deferred().reject() : null;
            });
        };

        // split settings -----------------------------------------------------

        /**
         * Returns whether the view split mode is enabled for the active sheet,
         * regardless of the specific split mode (dynamic or frozen mode).
         *
         * @returns {Boolean}
         *  Whether the view split mode is enabled for the active sheet.
         */
        this.hasSplit = function () {
            return activeSheetModel.hasSplit();
        };

        /**
         * Returns the current view split mode of the active sheet.
         *
         * @returns {String}
         *  The current view split mode. The empty string indicates that the
         *  view is not split at all. Otherwise, one of the following values
         *  will be returned:
         *  - 'split': Dynamic split mode (movable split lines).
         *  - 'frozen': Frozen split mode (a fixed number of columns and/or
         *      rows are frozen, split lines are not movable). When leaving
         *      frozen split mode, the split lines will be hidden.
         *  - 'frozenSplit': Frozen split mode (same behavior as 'frozen'
         *      mode). When leaving frozen split mode, the split lines remain
         *      visible and become movable (dynamic 'split' mode).
         */
        this.getSplitMode = function () {
            return activeSheetModel.getSplitMode();
        };

        /**
         * Returns whether the view is currently split, and dynamic split mode
         * is activated. Returns also true, if the split view is currently
         * frozen, but will thaw to dynamic split mode.
         *
         * @returns {Boolean}
         *  Whether dynamic split mode is activated (split modes 'split' or
         *  'frozenSplit').
         */
        this.hasDynamicSplit = function () {
            return activeSheetModel.hasDynamicSplit();
        };

        /**
         * Returns whether the view is currently split, and the frozen split
         * mode is activated.
         *
         * @returns {Boolean}
         *  Whether frozen split mode is activated (split modes 'frozen' or
         *  'frozenSplit').
         */
        this.hasFrozenSplit = function () {
            return activeSheetModel.hasFrozenSplit();
        };

        /**
         * Toggles the dynamic view split mode according to the current view
         * split mode, and the current selection. Enabling dynamic split mode
         * in a frozen split view will thaw the frozen panes. Otherwise,
         * enabling the dynamic split mode will split the view at the position
         * left of and above the active cell. Disabling the dynamic split mode
         * will remove the split also if the view is in frozen split mode.
         *
         * @param {Boolean} state
         *  The new state of the dynamic view split mode.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.setDynamicSplit = function (state) {

            var // whether the view is currently split
                hasSplit = this.hasSplit(),
                // address of the active cell
                address = this.getActiveCell(),
                // the position of the active cell in the sheet
                cellRectangle = null,
                // the visible area of the active grid pane that will be split
                visibleRectangle = null,
                // the new split positions
                splitWidth = 0, splitHeight = 0,
                // additional scroll anchor attributes
                attributes = null;

            // disable dynamic or frozen split mode
            if (!state && hasSplit) {
                activeSheetModel.clearSplit();
            }

            // convert frozen split to dynamic split
            else if (state && this.hasFrozenSplit()) {
                activeSheetModel.setDynamicSplit(activeSheetModel.getSplitWidthHmm(), activeSheetModel.getSplitHeightHmm());
            }

            // set dynamic split at active cell
            else if (state && !hasSplit) {

                // additional scroll anchor attributes
                attributes = {};

                // get position of active cell and visible area of the grid pane
                cellRectangle = this.getCellRectangle(address, { expandMerged: true });
                visibleRectangle = this.getActiveGridPane().getVisibleRectangle();

                // do not split vertically, if the active cell is shown at the left border
                splitWidth = cellRectangle.left - visibleRectangle.left;
                if ((-cellRectangle.width < splitWidth) && (splitWidth < MIN_PANE_SIZE)) {
                    splitWidth = 0;
                }

                // do not split horizontally, if the active cell is shown at the top border
                splitHeight = cellRectangle.top - visibleRectangle.top;
                if ((-cellRectangle.height < splitHeight) && (splitHeight < MIN_PANE_SIZE)) {
                    splitHeight = 0;
                }

                // split in the middle of the grid pane, if active cell is outside the visible area
                if (((splitWidth === 0) && (splitHeight === 0)) ||
                    (splitWidth < 0) || (splitWidth > visibleRectangle.width - MIN_PANE_SIZE) ||
                    (splitHeight < 0) || (splitHeight > visibleRectangle.height - MIN_PANE_SIZE)
                ) {
                    splitWidth = Math.floor(visibleRectangle.width / 2);
                    splitHeight = Math.floor(visibleRectangle.height / 2);
                    attributes.anchorRight = colCollection.getScrollAnchorByOffset(visibleRectangle.left + splitWidth, { pixel: true });
                    attributes.anchorBottom = rowCollection.getScrollAnchorByOffset(visibleRectangle.top + splitHeight, { pixel: true });
                    attributes.activePane = 'topLeft';
                } else {
                    if (splitWidth > 0) { attributes.anchorRight = { index: address[0], ratio: 0 }; }
                    if (splitHeight > 0) { attributes.anchorBottom = { index: address[1], ratio: 0 }; }
                }

                // activate the dynamic split view
                splitWidth = activeSheetModel.convertPixelToHmm(splitWidth);
                splitHeight = activeSheetModel.convertPixelToHmm(splitHeight);
                activeSheetModel.setDynamicSplit(splitWidth, splitHeight, attributes);
            }

            return this;
        };

        /**
         * Toggles the frozen view split mode according to the current view
         * split mode, and the current selection. Enabling frozen split mode
         * in a dynamic split view will freeze the panes with their current
         * size (split mode 'frozenSplit'). Otherwise, enabling the frozen
         * split mode will split the view at the position left of and above the
         * active cell. Disabling the frozen split mode will either return to
         * dynamic split mode (if split mode was 'frozenSplit'), or will remove
         * the frozen split at all (if split mode was 'frozen').
         *
         * @param {Boolean} state
         *  The new state of the frozen view split mode.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.setFrozenSplit = function (state) {

            var // whether the view is currently split
                hasSplit = this.hasSplit(),
                // address of the active cell
                address = this.getActiveCell(),
                // the current scroll position
                colAnchor = null, rowAnchor = null,
                // the new frozen column and row interval
                colInterval = null, rowInterval = null;

            // convert frozen split to dynamic split
            if (!state && hasSplit && (this.getSplitMode() === 'frozenSplit')) {
                activeSheetModel.setDynamicSplit(activeSheetModel.getSplitWidthHmm(), activeSheetModel.getSplitHeightHmm());
            }

            // disable split mode completely
            else if (!state && hasSplit) {
                activeSheetModel.clearSplit();
            }

            // convert dynamic split to frozen split
            else if (state && hasSplit && !this.hasFrozenSplit()) {
                activeSheetModel.setFrozenSplit(activeSheetModel.getSplitColInterval(), activeSheetModel.getSplitRowInterval());
            }

            // enable frozen split mode
            else if (state && !hasSplit) {

                // calculate frozen column interval (must result in at least one frozen column)
                colAnchor = activeSheetModel.getViewAttribute('anchorRight');
                colInterval = { first: (colAnchor.ratio < 0.5) ? colAnchor.index : (colAnchor.index + 1), last: address[0] - 1 };
                if (colInterval.first > colInterval.last) { colInterval = null; }

                // calculate frozen row interval (must result in at least one frozen row)
                rowAnchor = activeSheetModel.getViewAttribute('anchorBottom');
                rowInterval = { first: (rowAnchor.ratio < 0.5) ? rowAnchor.index : (rowAnchor.index + 1), last: address[1] - 1 };
                if (rowInterval.first > rowInterval.last) { rowInterval = null; }

                // activate the frozen split view
                activeSheetModel.setFrozenSplit(colInterval, rowInterval);
            }

            return this;
        };

        /**
         * Freezes the passed number of columns and rows in the active sheet,
         * regardless of the current split settings.
         *
         * @param {Number} cols
         *  The number of columns to freeze. The value zero will not freeze any
         *  column.
         *
         * @param {Number} rows
         *  The number of rows to freeze. The value zero will not freeze any
         *  row.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.setFixedFrozenSplit = function (cols, rows) {

            var // the scroll anchors for the first visible column/row
                colAnchor = activeSheetModel.getViewAttribute(activeSheetModel.hasColSplit() ? 'anchorLeft' : 'anchorRight'),
                rowAnchor = activeSheetModel.getViewAttribute(activeSheetModel.hasRowSplit() ? 'anchorTop' : 'anchorBottom'),
                // the first visible column and row in the split view
                col = ((colAnchor.index < model.getMaxCol()) && (colAnchor.ratio > 0.5)) ? (colAnchor.index + 1) : colAnchor.index,
                row = ((rowAnchor.index < model.getMaxRow()) && (rowAnchor.ratio > 0.5)) ? (rowAnchor.index + 1) : rowAnchor.index;

            // reduce start indexes to be able to freeze the passed number of columns/rows
            col = Math.max(0, Math.min(col, model.getMaxCol() - cols + 1));
            row = Math.max(0, Math.min(row, model.getMaxRow() - rows + 1));

            // activate the frozen split view
            activeSheetModel.setFrozenSplit((cols > 0) ? { first: col, last: col + cols - 1 } : null, (rows > 0) ? { first: row, last: row + rows - 1 } : null);

            return this;
        };

        // sheet operations ---------------------------------------------------

        /**
         * Returns whether the passed sheet type is supported by this
         * application.
         *
         * @param {String} sheetType
         *  A sheet type dentifier, as used in the 'insertSheet' operation.
         *
         * @returns {Boolean}
         *  Whether the passed sheet type is supported by this application.
         */
        this.isSheetTypeSupported = function (sheetType) {
            return (/^(worksheet)$/).test(sheetType);
        };

        /**
         * Inserts a new sheet in the spreadsheet document, and activates it in
         * the view. In case of an error, shows a notification message.
         *
         * @returns {Boolean}
         *  Whether the new sheet has been inserted successfully.
         */
        this.insertSheet = function () {

            var // try to insert a new sheet
                result = model.insertSheet(activeSheet + 1, model.generateUnusedSheetName());

            // insert the new sheet into the document, and activate the new sheet
            if (result.length === 0) {
                this.activateNextSheet();
                return true;
            }

            // show notification message on error
            return yellSheetNameError(result);
        };

        /**
         * Deletes the active sheet from the spreadsheet document, and
         * activates another sheet in the view. Does nothing, if the document
         * contains a single sheet only.
         *
         * @returns {Boolean}
         *  Whether the sheet has been deleted successfully.
         */
        this.deleteSheet = function () {
            // handlers of the delete events will update active sheet
            return model.deleteSheet(activeSheet);
        };

        /**
         * Moves the active sheet in the spreadsheet document to a new
         * position.
         *
         * @param {Number} toSheet
         *  The new zero-based index of the sheet.
         *
         * @returns {Boolean}
         *  Whether moving the sheet was successful.
         */
        this.moveSheet = function (toSheet) {
            // TODO: activate the sheet at its new position
            return model.moveSheet(activeSheet, toSheet);
        };

        /**
         * Copies the active sheet and inserts the copy with the provided sheet
         * name behind the active sheet. In case of an error, shows a
         * notification message.
         *
         * @param {String} sheetName
         *  The name of the new sheet.
         *
         * @returns {Boolean}
         *  Whether the sheet has been copied successfully.
         */
        this.copySheet = function (sheetName) {

            var // try to copy the sheet
                result = model.copySheet(activeSheet, activeSheet + 1, sheetName);

            // activate the new sheet
            if (result.length === 0) {
                this.activateNextSheet();
                return true;
            }

            // show notification message on error
            return yellSheetNameError(result);
        };

        /**
         * Returns the name of the active sheet.
         *
         * @returns {String}
         *  The name of the active sheet.
         */
        this.getSheetName = function () {
            return model.getSheetName(activeSheet);
        };

        /**
         * Renames the active sheet in the spreadsheet document, if possible.
         * In case of an error, shows a notification message.
         *
         * @param {String} sheetName
         *  The new name of the active sheet.
         *
         * @returns {Boolean}
         *  Whether the sheet has been renamed successfully.
         */
        this.setSheetName = function (sheetName) {

            var // try to rename the sheet
                result = null;

            // leave cell edit mode before renaming the sheet (sheet name may be used in formulas)
            if (!this.leaveCellEditMode('auto', { validate: true })) {
                return false;
            }

            // try to rename the sheet
            result = model.setSheetName(activeSheet, sheetName);

            // show notification message on error
            return yellSheetNameError(result);
        };

        /**
         * Returns the merged formatting attributes of the active sheet.
         *
         * @returns {Object}
         *  The merged attribute set of the active sheet.
         */
        this.getSheetAttributes = function () {
            return activeSheetModel.getMergedAttributes();
        };

        /**
         * Changes the formatting attributes of the active sheet.
         *
         * @param {Object} attributes
         *  An (incomplete) attribute set with the new attributes for the
         *  active sheet.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.setSheetAttributes = function (attributes) {

            var // the generator for the operation
                generator = null;

            // leave cell edit mode before hiding the sheet
            if (_.isObject(attributes.sheet) && (attributes.sheet.visible === false)) {
                if (!this.leaveCellEditMode('auto', { validate: true })) {
                    return this;
                }
            }

            generator = model.createOperationsGenerator();
            generator.generateSheetOperation(Operations.SET_SHEET_ATTRIBUTES, activeSheet, { attrs: attributes });
            model.applyOperations(generator.getOperations());
            return this;
        };

        /**
         * Returns the number of visible sheets in this spreadsheet view.
         *
         * @returns {Number}
         *  The number of visible sheets in the document.
         */
        this.getVisibleSheetCount = function () {
            return visibleSheets.length;
        };

        /**
         * Returns the number of hidden sheets in this spreadsheet view that
         * can be made visible.
         *
         * @returns {Number}
         *  The number of hidden sheets in the document. Does NOT include the
         *  sheets with unsupported types that will always be hidden.
         */
        this.getHiddenSheetCount = function () {
            return hiddenSheets.length;
        };

        /**
         * Makes all hidden sheets in the view visible.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.showAllSheets = function () {

            var // the generator for the operation
                generator = model.createOperationsGenerator(),
                // the attribute set to be inserted into the operations
                operationOptions = { attrs: { sheet: { visible: true } } };

            _.each(hiddenSheets, function (sheetInfo) {
                generator.generateSheetOperation(Operations.SET_SHEET_ATTRIBUTES, sheetInfo.sheet, operationOptions);
            });
            model.applyOperations(generator.getOperations());
            return this;
        };

        /**
         * Invokes the passed iterator function for all visible sheet contained
         * in this view.
         *
         * @param {Function} iterator
         *  The iterator function invoked for all visible sheets. Receives the
         *  following parameters:
         *  (1) {SheetModel} sheetModel
         *      The current sheet model instance.
         *  (2) {Number} index
         *      The zero-based sheet index in the collection of all visible
         *      sheets.
         *  (3) {String} sheetName
         *      The name of the sheet.
         *  (4) {Number} sheet
         *      The zero-based sheet index in the document model.
         *  If the iterator returns the Utils.BREAK object, the iteration
         *  process will be stopped immediately.
         *
         * @param {Object} [options]
         *  Additional optional parameters:
         *  @param {Object} [options.context]
         *      If specified, the iterator will be called with this context
         *      (the symbol 'this' will be bound to the context inside the
         *      iterator function).
         *  @param {Boolean} [options.reverse=false]
         *      If set to true, the sheets will be visited in reversed order.
         *
         * @returns {Utils.BREAK|Undefined}
         *  A reference to the Utils.BREAK object, if the iterator has returned
         *  Utils.BREAK to stop the iteration process, otherwise undefined.
         */
        this.iterateVisibleSheets = function (iterator, options) {
            return Utils.iterateArray(visibleSheets, function (sheetInfo, index) {
                return iterator.call(this, model.getSheetModel(sheetInfo.sheet), index, sheetInfo.name, sheetInfo.sheet);
            }, options);
        };

        /**
         * Returns the zero-based index of the active sheet currently displayed
         * in this spreadsheet view. Optionally, the index of the active sheet
         * inside the collection of all visible sheets can be returned.
         *
         * @param {Object} [options]
         *  Additional optional parameters:
         *  @param {Boolean} [options.visible=false]
         *      If set to true, the index of the active sheet inside the
         *      collection of all visible sheets will be returned instead of
         *      the physical sheet index.
         *
         * @returns {Number}
         *  The zero-based index of the active sheet, depending on the passed
         *  options. Will return -1, if the index in the collection of visible
         *  sheets is requested, but the active sheet is currently hidden.
         */
        this.getActiveSheet = function (options) {
            // look up the index of the active sheet in the collection of visible sheets,
            // or return the plain sheet index of the active sheet
            return Utils.getBooleanOption(options, 'visible', false) ? getVisibleIndex(activeSheet) : activeSheet;
        };

        /**
         * Activates a sheet in this spreadsheet view instance.
         *
         * @param {Number} index
         *  The zero-based index of the new active sheet. If the specified
         *  sheet is currently hidden, it will not be activated.
         *
         * @param {Object} [options]
         *  Additional optional parameters:
         *  @param {Boolean} [options.visible=false]
         *      If set to true, the passed sheet index will be interpreted as
         *      index into the collection of all visible sheets, instead of the
         *      physical sheet index.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.setActiveSheet = function (sheet, options) {

            var // the model of the new active sheet
                sheetModel = null;

            // get sheet index from the collection of visible sheets
            if (Utils.getBooleanOption(options, 'visible', false)) {
                sheet = ((0 <= sheet) && (sheet < visibleSheets.length)) ? visibleSheets[sheet].sheet : -1;
            }

            // the model of the new active sheet
            sheetModel = model.getSheetModel(sheet);

            // do nothing, if active sheet does not change, or if the sheet is hidden
            if (sheetModel && (sheetModel !== activeSheetModel) && (getVisibleIndex(sheet) >= 0)) {
                prepareSetActiveSheet();
                setActiveSheet(sheet);
            }

            return this;
        };

        /**
         * Activates the preceding visible sheet.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.activatePreviousSheet = function () {
            return this.setActiveSheet(this.getActiveSheet({ visible: true }) - 1, { visible: true });
        };

        /**
         * Activates the next visible sheet.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.activateNextSheet = function () {
            return this.setActiveSheet(this.getActiveSheet({ visible: true }) + 1, { visible: true });
        };

        /**
         * Activates a sheet for the import preview, while the document import
         * is still running.
         *
         * @internal
         *  Called from the preview handler of the application. MUST NOT be
         *  called directly.
         *
         * @param {Number} index
         *  The zero-based index of the initial active sheet.
         *
         * @returns {Boolean}
         *  Whether a sheet has been activated successfully.
         */
        this.activatePreviewSheet = function (sheet) {

            var // the sheet model
                sheetModel = model.getSheetModel(sheet);

            // no preview, if the sheet does not exist
            if (sheetModel) {
                // initialize view settings (scroll position, selection, zoom, split)
                sheetModel.initializeViewAttributes();
                // try to activate the preview sheet (may fail, e.g. for hidden sheets)
                this.setActiveSheet(sheet);
            }

            // return whether the sheet has been activated successfully
            return activeSheet >= 0;
        };

        /**
         * Returns the model instance of the active sheet.
         *
         * @returns {SheetModel}
         *  The model instance of the active sheet.
         */
        this.getSheetModel = function () {
            return activeSheetModel;
        };

        /**
         * Returns the rendering cache for the active sheet model.
         *
         * @returns {RenderCache}
         *  The rendering cache for the active sheet model.
         */
        this.getRenderCache = function () {
            return activeRenderCache;
        };

        /**
         * Returns the cell collection instance of the active sheet.
         *
         * @returns {CellCollection}
         *  The cell collection instance of the active sheet.
         */
        this.getCellCollection = function () {
            return activeSheetModel.getCellCollection();
        };

        /**
         * Returns the column collection instance of the active sheet.
         *
         * @returns {ColRowCollection}
         *  The column collection instance of the active sheet.
         */
        this.getColCollection = function () {
            return activeSheetModel.getColCollection();
        };

        /**
         * Returns the row collection instance of the active sheet.
         *
         * @returns {ColRowCollection}
         *  The row collection instance of the active sheet.
         */
        this.getRowCollection = function () {
            return activeSheetModel.getRowCollection();
        };

        /**
         * Returns the merge collection instance of the active sheet.
         *
         * @returns {MergeCollection}
         *  The merge collection instance of the active sheet.
         */
        this.getMergeCollection = function () {
            return activeSheetModel.getMergeCollection();
        };

        /**
         * Returns the validation collection instance of the active sheet.
         *
         * @returns {ValidationCollection}
         *  The validation collection instance of the active sheet.
         */
        this.getValidationCollection = function () {
            return activeSheetModel.getValidationCollection();
        };

        /**
         * Returns the drawing collection instance of the active sheet.
         *
         * @returns {SheetDrawingCollection}
         *  The drawing collection instance of the active sheet.
         */
        this.getDrawingCollection = function () {
            return activeSheetModel.getDrawingCollection();
        };

        /**
         * Converts the passed length in pixels to a length in 1/100 of
         * millimeters, according to the current sheet zoom factor.
         *
         * @param {Number} length
         *  The length in pixels.
         *
         * @returns {Number}
         *  The converted length in 1/100 of millimeters.
         */
        this.convertPixelToHmm = function (length) {
            return activeSheetModel.convertPixelToHmm(length);
        };

        /**
         * Converts the passed length in 1/100 of millimeters to a length in
         * pixels, according to the current sheet zoom factor.
         *
         * @param {Number} length
         *  The length in 1/100 of millimeters.
         *
         * @returns {Number}
         *  The converted length in pixels.
         */
        this.convertHmmToPixel = function (length) {
            return activeSheetModel.convertHmmToPixel(length);
        };

        /**
         * Returns the absolute position of the passed cell range in the active
         * sheet, in 1/100 of millimeters, as well as in pixels according to
         * the current sheet zoom factor.
         *
         * @param {Object} range
         *  The address of the cell range in the active sheet.
         *
         * @returns {Object}
         *  The absolute position and size of the range in the sheet. See
         *  SheetModel.getRangeRectangle() for details.
         */
        this.getRangeRectangle = function (range) {
            return activeSheetModel.getRangeRectangle(range);
        };

        /**
         * Returns the absolute position of the specified cell in the active
         * sheet, in 1/100 of millimeters, as well as in pixels according to
         * the current sheet zoom factor.
         *
         * @param {Number[]} address
         *  The address of the cell in the active sheet.
         *
         * @param {Object} [options]
         *  Optional parameters. See method SheetModel.getCellRectangle() for
         *  details.
         *
         * @returns {Object}
         *  The absolute position and size of the cell in the active sheet. See
         *  method SheetModel.getCellRectangle() for details.
         */
        this.getCellRectangle = function (address, options) {
            return activeSheetModel.getCellRectangle(address, options);
        };

        /**
         * Shrinks the passed range address to the visible area, if the leading
         * or trailing columns/rows of the range are hidden.
         *
         * @param {Object} range
         *  The address of the original cell range.
         *
         * @returns {Object|Null}
         *  The range address of the visible area of the passed cell range; or
         *  null, if the range is completely hidden. The first and last column,
         *  and the first and last row of the range are visible, but some inner
         *  columns/rows in the range may be hidden.
         */
        this.getVisibleRange = function (range) {
            return activeSheetModel.getVisibleRange(range);
        };

        /**
         * Returns whether the active sheet is locked. In locked sheets it is
         * not possible to insert, delete, or modify columns and rows, to edit
         * protected cells, or to manipulate drawing objects.
         *
         * @returns {Boolean}
         *  Whether the active sheet is locked.
         */
        this.isSheetLocked = function () {
            return activeSheetModel.isLocked();
        };

        /**
         * Returns the values of all view attribute of the active sheet.
         *
         * @returns {Object}
         *  A cloned map containing all view attribute values of the active
         *  sheet.
         */
        this.getSheetViewAttributes = function () {
            return activeSheetModel.getViewAttributes();
        };

        /**
         * Returns the value of a specific view attribute of the active sheet.
         *
         * @param {String} name
         *  The name of the view attribute.
         *
         * @returns {Any}
         *  A clone of the attribute value.
         */
        this.getSheetViewAttribute = function (name) {
            return activeSheetModel.getViewAttribute(name);
        };

        /**
         * Changes specific view attributes of the active sheet.
         *
         * @param {Object} attributes
         *  A map with all view attributes to be changed.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.setSheetViewAttributes = function (attributes) {
            activeSheetModel.setViewAttributes(attributes);
            return this;
        };

        /**
         * Changes a specific view attribute of the active sheet.
         *
         * @param {String} name
         *  The name of the view attribute.
         *
         * @param {Any} value
         *  The new value for the view attribute.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.setSheetViewAttribute = function (name, value) {
            activeSheetModel.setViewAttribute(name, value);
            return this;
        };

        /**
         * Generates and sends operations for all view attributes that have
         * been changed while the document was open.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.sendChangedViewAttributes = function () {

            // Only send changed view attributes when user has edit rights, and has modified
            // the document locally, or if 'important' sheet view settings have been changed.
            // This is needed to prevent creating new versions of the document after selecting
            // cells or scrolling around, but without editing anything.
            if (!model.getEditMode()) { return; }

            var // whether the document is modified, and view settings can be sent
                modified = app.isLocallyModified();

            // detect if any sheet contains 'important' changed view attributes
            if (!modified) {
                model.iterateSheetModels(function (sheetModel) {
                    if (sheetModel.hasChangedViewAttributes()) {
                        modified = true;
                        return Utils.BREAK;
                    }
                });
            }

            // send the changed view attributes
            if (modified) {
                model.getUndoManager().enterUndoGroup(function () {

                    // send 'setDocumentAttributes' operation for the active sheet index
                    if (activeSheet !== documentStyles.getDocumentAttributes().activeSheet) {
                        model.applyOperations({ name: Operations.SET_DOCUMENT_ATTRIBUTES, attrs: { document: { activeSheet: activeSheet } } });
                    }

                    // send operations for all changed sheet view attributes
                    model.iterateSheetModels(function (sheetModel) {
                        sheetModel.sendChangedViewAttributes();
                    });
                });
            }

            return this;
        };

        // column operations --------------------------------------------------

        /**
         * Returns whether additional columns can be inserted into the sheet,
         * based on the current selection.
         *
         * @returns {Boolean}
         *  Whether additional columns can be inserted, according to the
         *  current selection and the number of used columns in the sheet.
         */
        this.canInsertColumns = function () {

            var // the column intervals in the current selection
                colIntervals = SheetUtils.getColIntervals(this.getSelectedRanges());

            return activeSheetModel.canInsertColumns(colIntervals);
        };

        /**
         * Inserts new columns into the active sheet, according to the current
         * selection.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.insertColumns = function () {

            var // the column intervals in the current selection
                colIntervals = SheetUtils.getColIntervals(this.getSelectedRanges());

            activeSheetModel.insertColumns(colIntervals);
            return this;
        };

        /**
         * Returns whether the selected columns can be deleted from the sheet.
         *
         * @returns {Boolean}
         *  Whether the selected columns can be deleted.
         */
        this.canDeleteColumns = function () {

            var // the column intervals in the current selection
                colIntervals = SheetUtils.getColIntervals(this.getSelectedRanges());

            return activeSheetModel.canDeleteColumns(colIntervals);
        };

        /**
         * Deletes existing columns from the active sheet, according to the
         * current selection.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.deleteColumns = function () {

            var // the column intervals in the current selection
                colIntervals = SheetUtils.getColIntervals(this.getSelectedRanges());

            activeSheetModel.deleteColumns(colIntervals);
            return this;
        };

        /**
         * Returns the mixed formatting attributes of all columns in the active
         * sheet covered by the current selection.
         *
         * @param {Boolean} [previousCol]
         *  Should the previous column attributes to be returned?
         *
         * @returns {Object}
         *  A complete attribute set containing the mixed attributes of all
         *  selected columns. All attributes that could not be resolved
         *  unambiguously will be set to the value null.
         */
        this.getColumnAttributes = function (previousCol) {

            var // the column intervals in the current selection
                colIntervals = SheetUtils.getColIntervals(this.getSelectedRanges());

            if (previousCol) {
                colIntervals = _.map(colIntervals, function (colInterval) {
                    return { first: colInterval.first - 1, last: colInterval.last - 1 };
                });
            }

            return activeSheetModel.getColumnAttributes(colIntervals);
        };

        /**
         * Changes the specified column attributes in the active sheet,
         * according to the current selection.
         *
         * @param {Object} attributes
         *  The (incomplete) column attributes, as simple object (NOT mapped as
         *  a 'column' sub-object).
         *
         * @param {Boolean} [previousCol]
         *  Should the previous column to be changed?
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.setColumnAttributes = function (attributes, previousCol) {

            var // the column intervals in the current selection
                colIntervals = SheetUtils.getColIntervals(this.getSelectedRanges());

            if (previousCol) {
                colIntervals = _.map(colIntervals, function (colInterval) {
                    return { first: colInterval.first -1, last: colInterval.last - 1 };
                });
            }

            if (activeSheetModel.setColumnAttributes(colIntervals, attributes) === 'overflow') {
                self.yell({ type: 'info', message: MAX_CHANGE_COLS_WARNING });
            }
            return this;
        };

        /**
         * Returns whether there were columns that can be make visible.
         *  Either in the current selected range(s)
         *  or before the one column (or multiple single columns) which ist/were selected
         *
         * @returns {Boolean}
         *  Whether the selected columns can be make visible.
         */
        this.canShowColumns = function () {

            var currentAttributes = this.getColumnAttributes(),
                currentVisible = currentAttributes.visible,
                prevExists = false,
                prevAttributes = this.getColumnAttributes(true),
                prevVisible = null;

            if (_.isObject(prevAttributes)) {
                prevExists = true;
                prevVisible = prevAttributes.visible;
            }

            return (!currentVisible || (prevExists && !prevVisible) )?true:false;
        };

        /**
         * Make hidden Columns visible
         *  Either all hidden columns in the current selected range(s)
         *  or if only one column is (or multiple single columns are) selected, the previous Column
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.showColumns = function () {

            var currentAttributes = this.getColumnAttributes(),
                currentVisible = currentAttributes.visible,
                prevAttributes = this.getColumnAttributes(true),
                prevVisible = null;

            if (_.isObject(prevAttributes)) {
                prevVisible = prevAttributes.visible;
            }

            return this.setColumnAttributes({ visible: true }, currentVisible);
        };

        /**
         * Changes the width of columns in the active sheet, according to the
         * current selection.
         *
         * @param {Number} width
         *  The column width in 1/100 mm. If this value is less than 1, the
         *  columns will be hidden, and their original width will not be
         *  changed.
         *
         * @param {Object} [options]
         *  Additional optional parameters:
         *  @param {Boolean} [options.custom=true]
         *      The new value of the 'customWidth' column attribute.
         *  @param {Number} [options.target]
         *      The target column to be changed. If specified, the column width
         *      will be set to that column only. If the current selection
         *      contains any ranges covering this column completely, the column
         *      width will be set to all columns contained in these ranges.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.setColumnWidth = function (width, options) {

            var // the target column to be modified
                targetCol = Utils.getIntegerOption(options, 'target'),
                // the column intervals to be modified
                colIntervals = getSelectedIntervals(true, { target: targetCol });

            if (activeSheetModel.setColumnWidth(colIntervals, width, options) === 'overflow') {
                self.yell({ type: 'info', message: MAX_CHANGE_COLS_WARNING });
            }
            return this;
        };

        /**
         * Sets optimal column width based on the content of cells.
         *
         * @param {Number} [targetCol]
         *  The target column to be changed. If the current selection contains
         *  any ranges covering this column completely, the optimal column
         *  width will be set to all columns contained in these ranges.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.setOptimalColumnWidth = function (targetCol) {

            var // the column intervals to be modified (only columns covered by the grid panes)
                colIntervals = getSelectedIntervals(true, { target: targetCol, visible: true });

            setOptimalColumnWidth(colIntervals);
            return this;
        };

        /**
         * Returns the column attributes of the column the active cell is
         * currently located in.
         *
         * @returns {Object}
         *  The column attribute map of the active cell.
         */
        this.getActiveColumnAttributes = function () {
            var activeCol = this.getActiveCell()[0];
            return colCollection.getEntry(activeCol).attributes.column;
        };

        // row operations -----------------------------------------------------

        /**
         * Returns whether additional rows scan be inserted into the sheet,
         * based on the current selection.
         *
         * @returns {Boolean}
         *  Whether additional rows can be inserted, according to the current
         *  selection and the number of used rows in the sheet.
         */
        this.canInsertRows = function () {

            var // the row intervals in the current selection
                rowIntervals = SheetUtils.getRowIntervals(this.getSelectedRanges());

            return activeSheetModel.canInsertRows(rowIntervals);
        };

        /**
         * Inserts new rows into the active sheet, according to the current
         * selection.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.insertRows = function () {

            var // the row intervals in the current selection
                rowIntervals = SheetUtils.getRowIntervals(this.getSelectedRanges());

            activeSheetModel.insertRows(rowIntervals);
            return this;
        };

        /**
         * Returns whether the selected rows can be deleted from the sheet.
         *
         * @returns {Boolean}
         *  Whether the selected rows can be deleted.
         */
        this.canDeleteRows = function () {

            var // the row intervals in the current selection
                rowIntervals = SheetUtils.getRowIntervals(this.getSelectedRanges());

            return activeSheetModel.canDeleteRows(rowIntervals);
        };

        /**
         * Deletes existing rows from the active sheet, according to the
         * current selection.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.deleteRows = function () {

            var // the row intervals in the current selection
                rowIntervals = SheetUtils.getRowIntervals(this.getSelectedRanges());

            activeSheetModel.deleteRows(rowIntervals);
            return this;
        };

        /**
         * Returns the mixed formatting attributes of all rows in the active
         * sheet covered by the current selection.
         *
         * @param {Boolean} [previousRow]
         *  Should the previous row attributes to be returned?
         *
         * @returns {Object}
         *  A complete attribute set containing the mixed attributes of all
         *  selected rows. All attributes that could not be resolved
         *  unambiguously will be set to the value null.
         */
        this.getRowAttributes = function (previousRow) {

            var // the row intervals in the current selection
                rowIntervals = SheetUtils.getRowIntervals(this.getSelectedRanges());

            if (previousRow) {
                rowIntervals = _.map(rowIntervals, function (rowInterval) {
                    return { first: rowInterval.first - 1, last: rowInterval.last - 1 };
                });
            }

            return activeSheetModel.getRowAttributes(rowIntervals);
        };

        /**
         * Changes the specified row attributes in the active sheet, according
         * to the current selection..
         *
         * @param {Object} attributes
         *  The (incomplete) row attributes, as simple object (NOT mapped as a
         *  'row' sub-object).
         *
         * @param {Boolean} [previousRow]
         *  Should the previous row to be changed?
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.setRowAttributes = function (attributes, previousRow) {

            var // the row intervals in the current selection
                rowIntervals = SheetUtils.getRowIntervals(this.getSelectedRanges());

            if (previousRow) {
                rowIntervals = _.map(rowIntervals, function (rowInterval) {
                    return { first: rowInterval.first - 1, last: rowInterval.last - 1 };
                });
            }

            if (activeSheetModel.setRowAttributes(rowIntervals, attributes) === 'overflow') {
                self.yell({ type: 'info', message: MAX_CHANGE_ROWS_WARNING });
            }
            return this;
        };

        /**
         * Returns whether there were rows that can be make visible.
         *  Either in the current selected range(s)
         *  or before the one row (or multiple single rows) which ist/were selected
         *
         * @returns {Boolean}
         *  Whether the selected rows can be make visible.
         */
        this.canShowRows = function () {

            var currentAttributes = this.getRowAttributes(),
                currentVisible = currentAttributes.visible,
                prevExists = false,
                prevAttributes = this.getRowAttributes(true),
                prevVisible = null;

            if (_.isObject(prevAttributes)) {
                prevExists = true;
                prevVisible = prevAttributes.visible;
            }

            return (!currentVisible || (prevExists && !prevVisible) )?true:false;
        };

        /**
         * Make hidden Rows visible
         *  Either all hidden rows in the current selected range(s)
         *  or if only one column is (or multiple single rows are) selected, the previous Row
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.showRows = function () {

            var currentAttributes = this.getRowAttributes(),
                currentVisible = currentAttributes.visible,
                prevAttributes = this.getRowAttributes(true),
                prevVisible = null;

            if (_.isObject(prevAttributes)) {
                prevVisible = prevAttributes.visible;
            }

            return this.setRowAttributes({ visible: true }, currentVisible);
        };

        /**
         * Changes the height of rows in the active sheet, according to the
         * current selection.
         *
         * @param {Number} height
         *  The row height in 1/100 mm. If this value is less than 1, the rows
         *  will be hidden, and their original height will not be changed.
         *
         * @param {Object} [options]
         *  Additional optional parameters:
         *  @param {Boolean} [options.custom=true]
         *      The new value of the 'customHeight' row attribute.
         *  @param {Number} [options.target]
         *      The target row to be changed. If specified, the row height will
         *      be set to that row only. If the current selection contains any
         *      ranges covering this row completely, the row height will be set
         *      to all rows contained in these ranges.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.setRowHeight = function (height, options) {

            var // the target row to be modified
                targetRow = Utils.getIntegerOption(options, 'target'),
                // the row intervals to be modified
                rowIntervals = getSelectedIntervals(false, { target: targetRow });

            if (activeSheetModel.setRowHeight(rowIntervals, height, options) === 'overflow') {
                self.yell({ type: 'info', message: MAX_CHANGE_ROWS_WARNING });
            }
            return this;
        };

        /**
         * Sets optimal row height based on the content of cells.
         *
         * @param {Number} [targetRow]
         *  The target row to be changed. If the current selection contains any
         *  ranges covering this row completely, the optimal row height will be
         *  set to all rows contained in these ranges.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.setOptimalRowHeight = function (targetRow) {

            var // the row intervals to be modified (only rows covered by the grid panes)
                rowIntervals = getSelectedIntervals(false, { target: targetRow, visible: true });

            setOptimalRowHeight(rowIntervals);
            return this;
        };

        /**
         * Returns the row attributes of the row the active cell is currently
         * located in.
         *
         * @returns {Object}
         *  The row attribute map of the active cell.
         */
        this.getActiveRowAttributes = function () {
            var activeRow = this.getActiveCell()[1];
            return rowCollection.getEntry(activeRow).attributes.row;
        };

        // cell operations ----------------------------------------------------

        /**
         * Returns the value and formatting of the current active cell.
         *
         * @returns {Object}
         *  The cell value and formatting, in the following properties:
         *  - {String} display
         *      The formatted display string.
         *  - {Null|Number|Boolean|String} result
         *      The typed result value of the cell, or the formula result.
         *  - {String|Null} formula
         *      The formula definition, if the current cell is a formula cell.
         *  - {Object} attributes
         *      All formatting attributes of the cell, mapped by cell attribute
         *      family ('cell' and 'character'), and the identifier of the cell
         *      style sheet in the 'styleId' string property.
         *  - {Object} explicit
         *      The explicit formatting attributes of the cell, mapped by cell
         *      attribute family ('cell' and 'character'), and optionally the
         *      identifier of the cell style sheet.
         */
        this.getCellContents = function () {
            return _.clone(activeCellSettings);
        };

        /**
         * Returns the merged formatting attributes of the current active cell.
         *
         * @returns {Object}
         *  All formatting attributes of the active cell, mapped by cell
         *  attribute family ('cell' and 'character'), and the identifier of
         *  the cell style sheet in the 'styleId' string property. If the cell
         *  in-place edit mode is currently active, the edit attributes added
         *  there will be included too.
         */
        this.getCellAttributes = function () {

            var // the merged attributes of the active cell
                cellAttributes = _.copy(activeCellSettings.attributes, true),
                // additional formatting attributes from cell edit mode
                editAttributes = this.getCellEditAttributes();

            if (_.isObject(editAttributes)) {
                _.extend(cellAttributes.cell, editAttributes.cell);
                _.extend(cellAttributes.character, editAttributes.character);
            }

            return cellAttributes;
        };

        /**
         * Sets the contents of the active cell, and updates the view according
         * to the new cell value and formatting.
         *
         * @param {String} [value]
         *  The new value for the active cell. If omitted, only the cell
         *  formatting will be changed.
         *
         * @param {Object} [attributes]
         *  A cell attribute set, optionally containing the string property
         *  'styleId', and attribute value maps (name/value pairs), keyed by
         *  the supported attribute families 'cell' and 'character'. Omitted
         *  attributes will not be changed. Attributes set to the value null
         *  will be removed from the cell.
         *
         * @param {Object} [options]
         *  Additional optional parameters:
         *  @param {Boolean} [options.parse=false]
         *      If set to true, the 'parse' property will be inserted into the
         *      operation, set to the name of the current UI language.
         *  @param {Boolean} [options.clear=false]
         *      If set to true, all existing explicit attributes will be
         *      removed from the cell while applying the new attributes.
         *  @param {Any} [options.result]
         *      The result of a formula, as calculated by the local formula
         *      interpreter.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.setCellContents = function (value, attributes, options) {

            // store character attributes while in-place cell edit mode is active
            if (this.isCellEditMode()) {
                return this.setCellEditAttributes(attributes);
            }

            // do nothing if active cell is locked
            if (this.requireEditableActiveCell()) {
                setSingleCellContents(this.getActiveCell(), value, attributes, options);
            }

            return this;
        };

        /**
         * Fills all cell ranges in the current selection with the same value
         * and formatting, and updates the view according to the new cell value
         * and formatting.
         *
         * @param {String} [value]
         *  The new value for all cells in the current selection. If omitted,
         *  only the cell formatting will be changed.
         *
         * @param {Object} [attributes]
         *  A cell attribute set, optionally containing the string property
         *  'styleId', and attribute value maps (name/value pairs), keyed by
         *  the supported attribute families 'cell' and 'character'. Omitted
         *  attributes will not be changed. Attributes set to the value null
         *  will be removed from the cell.
         *
         * @param {Object} [options]
         *  Additional optional parameters:
         *  @param {Boolean} [options.parse=false]
         *      If set to true, the 'parse' property will be inserted into the
         *      operations, set to the name of the current UI language.
         *  @param {Boolean} [options.clear=false]
         *      If set to true, all existing explicit attributes will be
         *      removed from the selection while applying the new attributes.
         *  @param {String} [options.cat]
         *      The number format category the format code contained in the
         *      passed formatting attributes belongs to.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.fillCellRanges = function (value, attributes, options) {

            // store character attributes while in-place cell edit mode is active
            if (this.isCellEditMode()) {
                this.setCellEditAttributes(attributes);
            }

            // do nothing if current selection contains locked cells
            else if (this.requireEditableSelection()) {
                fillCellRanges(this.getSelectedRanges(), value, attributes, options);
            }

            return this;
        };

        /**
         * Deletes the values and formatting attributes from all cell ranges in
         * the current selection.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.clearCellRanges = function () {

            // do nothing if current selection contains locked cells
            if (this.requireEditableSelection()) {
                clearCellRanges(this.getSelectedRanges());
            }

            return this;
        };

        /**
         * Changes a single attribute of type 'cell' for all cells in the
         * current selection.
         *
         * @param {String} name
         *  The name of the cell attribute.
         *
         * @param {Any} value
         *  The new value of the cell attribute. If set to the value null, all
         *  attributes set explicitly will be removed from the cells.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.setCellAttribute = function (name, value) {
            return this.fillCellRanges(undefined, { cell: Utils.makeSimpleObject(name, value) });
        };

        /**
         * Changes a single attribute of type 'character' for all cells in the
         * current selection.
         *
         * @param {String} name
         *  The name of the character attribute.
         *
         * @param {Any} value
         *  The new value of the character attribute. If set to the value null,
         *  the current explicit attribute will be removed from the cells.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.setCharacterAttribute = function (name, value) {
            return this.fillCellRanges(undefined, { character: Utils.makeSimpleObject(name, value) });
        };

        /**
         * removes all explicit formatting attributes from all cells in the
         * current selection.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.clearAttributes = function () {
            return this.fillCellRanges(undefined, undefined, { clear: true });
        };

        /**
         * Returns the number format category of the active cell.
         *
         * @returns {String}
         *  The current number format category.
         */
        this.getNumberFormatCategory = function () {
            return activeCellSettings.format.cat;
        };

        /**
         * Sets selected number format category for the current active cell in
         * the current selection, and implicitly set the first format code of
         * the predefined template of this category to the cell.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.setNumberFormatCategory = function (category) {
            var formatCode = numberFormatter.getCategoryDefaultCode(category);
            if ((category !== 'custom') && _.isString(formatCode)) {
                this.fillCellRanges(undefined, { cell: { numberFormat: { code: formatCode } } }, { cat: category });
            }
            return this;
        };

        /**
         * Returns the format code of the active cell.
         *
         * @returns {String}
         *  The current number format code.
         */
        this.getNumberFormatCode = function () {
            var formatCode = activeCellSettings.attributes.cell.numberFormat.code;
            return _.isString(formatCode) ? formatCode : numberFormatter.getCategoryDefaultCode('standard');
        };

        /**
         * Returns the format code of the active cell.
         *
         * @param {String} formatCode
         *  The number format code to be set to the current selection.
         */
        this.setNumberFormatCode = function (formatCode) {
            // do nothing if custom fallback button is clicked
            if (formatCode !== Controls.CUSTOM_FORMAT_VALUE) {
                this.setCellAttribute('numberFormat', { code: formatCode });
            }
            return this;
        };

        /**
         * Returns the border mode of the current selection.
         *
         * @returns {Object}
         *  The border mode representing the visibility of the borders in all
         *  ranges of the current selection. See 'MixedBorder.getBorderMode()'
         *  for more details.
         */
        this.getBorderMode = function () {
            return MixedBorder.getBorderMode(selectionSettings.borders);
        };

        /**
         * Changes the visibility of the borders in all ranges of the current
         * selection.
         *
         * @param {Object} borderMode
         *  The border mode representing the visibility of the borders in all
         *  ranges of the current selection. See 'MixedBorder.getBorderMode()'
         *  for more details.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.setBorderMode = function (borderMode) {

            // quit if current selection contains locked cells
            if (this.requireEditableSelection()) {

                // create and apply the operations
                var borderAttributes = MixedBorder.getBorderAttributes(borderMode, selectionSettings.borders, DEFAULT_SINGLE_BORDER);
                activeSheetModel.setBorderAttributes(self.getSelectedRanges(), borderAttributes);

                // bug 34021: immediately update all selection settings to prevent flickering UI elements
                updateSelectionSettings({ direct: true });
            }

            return this;
        };

        /**
         * Returns the merged border attributes of the current selection.
         *
         * @returns {Object}
         *  An attribute map containing the merged borders of all available
         *  border attributes ('borderTop', 'borderLeft', etc.) in the current
         *  selection. Each border attribute value contains the properties
         *  'style', 'width', and 'color'. If the border property is equal in
         *  all affected cells, it will be set as property value in the border
         *  object, otherwise the property will be set to the value null.
         */
        this.getBorderAttributes = function () {
            return _.copy(selectionSettings.borders, true);
        };

        /**
         * Changes the specified border properties (line style, line color, or
         * line width) of all visible cell borders in the current selection.
         *
         * @param {Border} border
         *  A border attribute, may be incomplete. All properties contained in
         *  this object will be set for all visible borders in the current
         *  selection. Omitted border properties will not be changed.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.changeVisibleBorders = function (border) {

            // quit if current selection contains locked cells
            if (this.requireEditableSelection()) {

                // create and apply the operations
                activeSheetModel.changeVisibleBorders(this.getSelectedRanges(), border);

                // bug 34021: immediately update all selection settings to prevent flickering UI elements
                updateSelectionSettings({ direct: true });
            }

            return this;
        };

        /**
         * Fills or clears a cell range according to the position and contents
         * of the cell range currently selected (the 'auto-fill' feature).
         *
         * @param {String} border
         *  Which border of the selected range will be modified by the
         *  operation. Allowed values are 'left', 'right', 'top', or 'bottom'.
         *
         * @param {Number} count
         *  The number of columns/rows to extend the selected range with
         *  (positive values), or to shrink the selected range (negative
         *  values).
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved when the
         *  auto-fill target range is editable, and the auto-fill operation has
         *  been applied. Otherwise, the promise will be rejected.
         */
        this.autoFill = function (border, count) {

            var // all cell ranges selected in the sheet
                ranges = this.getSelectedRanges(),
                // whether to expand/shrink the leading or trailing border
                leading = /^(left|top)$/.test(border),
                // whether to expand/delete columns or rows
                columns = /^(left|right)$/.test(border),
                // the array index in the cell address
                addrIndex = columns ? 0 : 1,
                // the target range for auto-fill
                targetRange = _.copy(ranges[0], true);

            function checkEditable(range) {
                return self.areRangesEditable(range).fail(function () {
                    self.yell({ type: 'info', message: PROTECTED_CELLS_WARNING });
                });
            }

            // do nothing if called in a multi-selection
            if (ranges.length !== 1) { return $.Deferred().reject(); }

            // fill cells outside the selected range
            if (count > 0) {

                // build the target range (adjacent to the selected range)
                if (leading) {
                    targetRange.start[addrIndex] = ranges[0].start[addrIndex] - count;
                    targetRange.end[addrIndex] = ranges[0].start[addrIndex] - 1;
                } else {
                    targetRange.start[addrIndex] = ranges[0].end[addrIndex] + 1;
                    targetRange.end[addrIndex] = ranges[0].end[addrIndex] + count;
                }

                // only the target range must be editable (not the source cells)
                return checkEditable(targetRange)
                    .done(function () {
                        model.getUndoManager().enterUndoGroup(function () {
                            // apply the 'autoFill' operation
                            activeSheetModel.autoFill(ranges[0], border, count);
                            // update automatic row height in the target range
                            updateOptimalRowHeight(targetRange);
                        });
                    })
                    .always(function () {
                        // expand the selected range
                        self.changeActiveRange(SheetUtils.getBoundingRange(ranges[0], targetRange));
                    });
            }

            // delete cells inside the selected range
            if (count < 0) {

                // build the target range (part of the selected range, may be the entire selected range)
                if (leading) {
                    targetRange.end[addrIndex] = targetRange.start[addrIndex] - count - 1;
                } else {
                    targetRange.start[addrIndex] = targetRange.end[addrIndex] + count + 1;
                }

                // the target range must be editable (not the entire selected range)
                return checkEditable(targetRange)
                    .done(function () {
                        model.getUndoManager().enterUndoGroup(function () {
                            // apply the 'clearCellRange' operation
                            activeSheetModel.clearCellRanges(targetRange);
                            // update automatic row height in the deleted range
                            updateOptimalRowHeight(targetRange);
                        });
                    })
                    .always(function () {
                        // do not change selection after deleting the entire range
                        if (_.isEqual(ranges[0], targetRange)) { return; }
                        // select the unmodified part of the original range
                        targetRange = _.copy(ranges[0], true);
                        if (leading) {
                            targetRange.start[addrIndex] -= count;
                        } else {
                            targetRange.end[addrIndex] += count;
                        }
                        self.changeActiveRange(targetRange);
                    });
            }

            // do nothing if no cells will be filled or cleared
            return $.when();
        };

        /**
         * Inserts one or more formulas calculating subtotal results into or
         * next to the current selection.
         *
         * @param {String} funcName
         *  The name of the subtotal function to be inserted into the formulas.
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved when the
         *  server request used to query the subtotal formulas has returned.
         */
        this.insertAutoFormula = function (value) {

            var // the current selection
                selection = self.getSelection(),
                // the selected range
                range = selection.ranges[0],
                // whether the range consists of a single column
                singleCol = range.start[0] === range.end[0],
                // whether the range consists of a single row
                singleRow = range.start[1] === range.end[1],
                // the translated upper-case function name
                funcName = app.getLocalizedFunctionName(value);

            // returns the correct formula expression for the specified cell range
            function buildFormula(sourceRange) {
                return '=' + funcName + '(' + SheetUtils.getRangeName(sourceRange) + ')';
            }

            // returns a promise that resolves if all cells in the passed range are editable
            function ensureEditable(editRange) {
                return self.areRangesEditable(editRange).fail(function () {
                    self.yell({ type: 'info', message: PROTECTED_CELLS_WARNING });
                });
            }

            // sends a server request and returns the promise that resolves with the cell content array
            function sendFormulaRequest() {
                Utils.info('SpreadsheetView.insertAutoFormula(): requesting formulas for range ' + SheetUtils.getRangeName(range));
                return app.sendFilterRequest({
                    method: 'POST',
                    params: {
                        action: 'query',
                        requestdata: JSON.stringify({
                            type: 'autoFormula',
                            func: funcName,
                            sheet: activeSheet,
                            start: range.start,
                            end: range.end,
                            ref: selection.activeCell
                        })
                    },
                    resultFilter: function (data) {
                        return Utils.getArrayOption(data, 'contents', undefined, true);
                    }
                })
                .done(function (response) {
                    Utils.info('SpreadsheetView.insertAutoFormula(): request succeeded:', response);
                })
                .fail(function (response) {
                    Utils.error('SpreadshetView.insertAutoFormula(): request failed:', response);
                });
            }

            // selects the specified range, and keeps the current active cell
            function selectRange(start, end) {
                self.selectRange({ start: start, end: end}, { active: selection.activeCell });
            }

            // multiple selection not supported
            if (selection.ranges.length !== 1) {
                return $.Deferred().reject();
            }

            // single cell is selected
            if (singleCol && singleRow) {

                // query CalcEngine for a formula string and start in-place edit mode
                return sendFormulaRequest().done(function (contents) {
                    self.enterCellEditMode({ text: contents[0].formula });
                });
            }

            // multiple columns and rows are selected: fill formulas into the row below the range
            if (!singleCol && !singleRow) {

                // there must be space below the range
                if (range.end[1] >= model.getMaxRow()) { return $.Deferred().reject(); }

                var // fill formulas into the cells below the selected range
                    targetRange = { start: [range.start[0], range.end[1] + 1], end: [range.end[0], range.end[1] + 1] };

                // check that the range is completely unlocked, fill it with the formula
                return ensureEditable(targetRange).done(function () {
                    var sourceRange = { start: range.start, end: [range.start[0], range.end[1]] };
                    fillCellRanges(targetRange, buildFormula(sourceRange), undefined, { ref: targetRange.start, parse: true });
                    selectRange(range.start, targetRange.end);
                });
            }

            // single column or single row (with multiple cells): send server request if range is empty
            if (cellCollection.areRangesBlank(range)) {

                // check cell count first
                if (SheetUtils.getCellCount(range) > SheetUtils.MAX_FILL_CELL_COUNT) {
                    this.yell({ type: 'info', message: MAX_CHANGE_CELLS_WARNING });
                    return $.Deferred().reject();
                }

                // send server request, fill cells with one operation
                return sendFormulaRequest().done(function (contents) {

                    // convert the contents array according to the 'setCellContents' operation
                    contents = _.map(contents, function (cellData) {
                        var result = { value: cellData.formula };
                        if ('attrs' in cellData) { result.attrs = cellData.attrs; }
                        // put every array element into a single-element array for column selection
                        return singleCol ? [result] : result;
                    });

                    // apply the operation (changing selection not necessary)
                    setCellContents(range.start, singleRow ? [contents] : contents, { parse: true });
                });
            }

            // range contains any values: insert single formula into first empty cell
            return (function () {

                var // the cell address used to find the first empty cell
                    address = _.clone(range.end),
                    // modified array index in a cell address
                    addrIndex = singleRow ? 0 : 1,
                    // maximum index in the cell address (restrict loop to 1000 iterations)
                    maxIndex = Math.min(model.getMaxIndex(singleRow), address[addrIndex] + 1000);

                // JSHint requires to defined the function outside the while loop
                function createFormulaCell() {
                    var sourceRange = _.copy(range, true);
                    sourceRange.end[addrIndex] = Math.min(address[addrIndex] - 1, range.end[addrIndex]);
                    setSingleCellContents(address, buildFormula(sourceRange), undefined, { parse: true });
                    selectRange(range.start, address);
                }

                // find the first empty cell, starting at the last cell in the selected range
                while (address[addrIndex] <= maxIndex) {
                    if (cellCollection.isBlankCell(address)) {
                        return ensureEditable({ start: address, end: address }).done(createFormulaCell);
                    }
                    address[addrIndex] += 1;
                }

                return $.Deferred().reject();
            }());
        };

        /**
         * Merges or unmerges all cells of the current selection.
         *
         * @param {String} type
         *  The merge type. Must be one of:
         *  - 'merge': Merges entire ranges.
         *  - 'horizontal': Merges the single rows in all ranges.
         *  - 'vertical': Merges the single columns in all ranges.
         *  - 'unmerge': Removes all merged ranges covered by the selection.
         *  - 'toggle': Will be mapped to 'merge', if the selection does not
         *      cover any merged ranges, otherwise will be mapped to 'unmerge'.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.mergeRanges = function (type) {

            var // the selected ranges
                ranges = this.getSelectedRanges(),
                // the result of the operation
                result = null;

            // quit if current selection contains locked cells
            if (!this.requireEditableSelection()) { return this; }

            // toggle merge state if specified
            if (type === 'toggle') {
                type = mergeCollection.rangesOverlapAnyMergedRange(ranges) ? 'unmerge' : 'merge';
            }

            // send the 'mergeCells' operations
            result = activeSheetModel.mergeRanges(ranges, type);

            // yell errors
            switch (result) {
            case 'overlap':
                this.yell({
                    type: 'info',
                    message:
                        //#. Warning text: trying to merge multiple cell ranges which overlap each other
                        gt('Overlapping ranges cannot be merged.')
                });
                break;
            case 'overflow':
                this.yell({
                    type: 'info',
                    message:
                        //#. Warning text: trying to merge too many cell ranges at the same time
                        //#. %1$d is the maximum number of cell ranges that can be merged at a time
                        //#, c-format
                        gt('It is not possible to merge more than %1$d ranges at the same time.', _.noI18n(SheetUtils.MAX_MERGED_RANGES_COUNT))
                });
                break;
            }

            return this;
        };

        // drawing operations -------------------------------------------------

        /**
         * Inserts a new drawing object into the active sheet, and selects it
         * afterwards.
         *
         * @param {String} type
         *  The type identifier of the drawing object.
         *
         * @param {Object} attributes
         *  Initial attributes for the drawing object, especially the anchor
         *  attributes specifying the position of the drawing in the sheet.
         *
         * @param {Function} [generatorCallback]
         *  A callback function that will be invoked after the initial
         *  'insertDrawing' operation has been generated. Allows to create
         *  additional operations for the drawing object, before all these
         *  operations will be applied at once. Receives the following
         *  parameters:
         *  (1) {SpreadsheetOperationsGenerator} generator
         *      The operations generator (already containing the initial
         *      'insertDrawing' operation).
         *  (2) {Number} sheet
         *      The zero-based index of the active sheet that will contain the
         *      drawing object.
         *  (3) {Array} position
         *      The logical position of the new drawing object in the sheet.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.insertDrawing = function (type, attributes, generatorCallback) {

            var // the logical position of the new drawing object
                position = [drawingCollection.getModelCount()],
                // create the new drawing object
                result = activeSheetModel.insertDrawing(position, type, attributes, generatorCallback);

            // select the new drawing object
            if (result.length === 0) {
                this.selectDrawing(position);
            } else {
                this.removeDrawingSelection();
            }

            return this;
        };

        /**
         * Deletes all drawings currently selected in the active sheet.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.deleteDrawings = function () {

            if (activeSheetModel.isLocked()) {
                self.yell({ type: 'info', message: gt('Objects on a protected sheet cannot be deleted.') });
                return this;
            }

            var // the selected drawing objects
                positions = this.getSelectedDrawings();

            // back to cell selection, delete all drawings previously selected
            this.removeDrawingSelection();
            activeSheetModel.deleteDrawings(positions);

            return this;
        };

        /**
         * Setting drawing attributes.
         *
         * @param {Object} [attributes]
         *  A drawing attribute set, keyed by the supported attribute family
         *  'drawing'. Omitted attributes will not be changed. Attributes set
         *  to the value null will be removed from the drawing.
         *
         * @param {Array} [pos]
         *  An array with at least one logical drawing address. If not
         *  specified, the selected drawings are used.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.setDrawingAttributes = function (attributes, pos) {

            var // the selected drawing objects
                drawings = pos || this.getSelectedDrawings(),
                // the generator for the operations
                generator = model.createOperationsGenerator();

            // send the 'setDrawingAttributes' operations
            _.each(drawings, function (position) {
                generator.generateDrawingOperation(Operations.SET_DRAWING_ATTRIBUTES, activeSheet, position, { attrs: attributes });
            });
            model.applyOperations(generator.getOperations());

            return this;
        };

        // search operations --------------------------------------------------

        /**
         *  Triggers a search: Build a search object and trigger a requestUpdate with it.
         *
         *  @param {String} searchQuery
         *  The string to be searched.
         */
        this.search = function (searchQuery, options) {

            // quit if search bar is maybe opened but user are searching yet
            if (searchQuery.length === 0) {return; }

            // reset current search result pointer and range pointer before every search
            searchResultPointer = -1;
            activeSearchQuery = searchQuery;
            self.searchClear();

            var isRefresh = Utils.getBooleanOption(options, 'refresh', false);

            // build request props
            var requestProps = {};

            // The search starts from the selected cell per default.
            // This way, user will always get the closest result hit to his/her eye focus.
            var currentActiveCell = self.getActiveCell();

            requestProps.search = {
                searchText : searchQuery,
                start: currentActiveCell,
                limit: MAX_SEARCH_RESULT_COUNT,
                refresh: isRefresh
            };

            // request the search results with an view update
            self.requestUpdate(requestProps);
        };

        /**
         * Iterate search result items.
         *
         * @param {String} direction
         * The iterate direction. 2 values possible: 'next' or 'prev'
         */
        this.searchIterate = function (direction) {

            // if there are no results or if user wants to search a new query, do a search refresh implicitely and quit
            if (!_.isObject(searchLayoutData.sheets) || activeSearchQuery !== app.getWindow().search.query) {
                if (app.getState() !== 'sending') self.search(app.getWindow().search.query);
                return;
            }

            // if direction is not passed, set next as default
            direction = direction || 'next';

            var currentSheetResult = searchLayoutData.sheets[activeSheet];

            // do nothing, if user switched to other sheets after search.
            if (_.isUndefined(currentSheetResult)) return;

            // get result array size
            var searchResultSize = currentSheetResult.length;

            // init next iteration address
            var nextHitAddress = [];

            // set ring structure for next and prev actions
            switch (direction)
            {
            case 'prev':
                if (searchResultPointer > 0) {
                    searchResultPointer--;
                } else {
                    searchResultPointer = searchResultSize - 1;
                }
                break;
            case 'next':
                if (searchResultPointer < (searchResultSize - 1)) {
                    searchResultPointer++;
                } else {
                    searchResultPointer = 0;
                }
                break;
            default:
                return;
            }

            // get next iteration item
            nextHitAddress = currentSheetResult[searchResultPointer];

            // go to the grid area and select that cell
            this.getActiveGridPane().scrollToCell(nextHitAddress);

            // and finally, go to that next iteration cell
            this.selectCell(nextHitAddress);

        };

        /**
         * Replaces the value of the current (single) selection with the replacement value.
         *
         * @param {Object} option.all = false
         * option.all = true -> do a replace all.
         */
        this.searchReplace = function (option) {

            if (activeSheetModel.isLocked()) {
                self.yell({ type: 'info', message: gt('You cannot use this command on a protected sheet.') });
                return;
            }

            // get fresh search query and replacement
            var win = app.getWindow(),
                searchQuery = win.search.getQuery(),
                replacementText = win.search.replacement;

            // Similar like native indexOf, but only for cell addresses.
            function indexOfAddress(addressArray, address) {
                var resultIndex = -1;
                _.each(addressArray, function (elem, index) {
                    if (address[0] === elem[0] && address[1] === elem[1]) {
                        resultIndex = index;
                    }
                });
                return resultIndex;
            }

            // 'fly' to cell: scroll and select desired cell.
            function goToCell(address) {
                if (!model.isValidAddress(address)) return;
                self.getActiveGridPane().scrollToCell(address);
                self.selectCell(address);
            }

            // do nothing and return immediately if search query is empty
            if (searchQuery === '') {
                self.yell({ type: 'info', message: gt('Nothing to replace.') });
                return;
            }

            // defaults to false-> a single replacement
            var replaceAll = Utils.getBooleanOption(option, 'all', false);

            // initiate a replace call to calcengine.
            if (replaceAll) {
                var
                def = null,
                localStartTime = _.now(),
                localRequestData = {
                    sheet: activeSheet,
                    searchText: searchQuery,
                    replaceText: replacementText
                };

                def = app.sendFilterRequest({
                    method: 'POST',
                    params: {
                        action: 'replaceAll',
                        requestdata: JSON.stringify(localRequestData)
                    }
                });
                def.fail(function () {
                    Utils.error('SpreadsheetView.executeReplace(): replace request # failed');
                })
                .done(function (result) {
                    Utils.info('SpreadsheetView.executeReplace(): replace response # received (' + (_.now() - localStartTime) + 'ms)', result);
                    var replacedCellsCount = 0;
                    var changedRanges = [];

                    if (_.isEmpty(result.changed.sheets)) {
                        self.yell({ type: 'info', message: gt('Nothing to replace.') });
                        return;
                    }

                    changedRanges = result.changed.sheets[activeSheet].cells;

                    _.each(changedRanges, function (elem) {
                        if (_.isUndefined(elem.end)) elem.end = elem.start;
                        replacedCellsCount = replacedCellsCount + SheetUtils.getCellCount(elem);
                    });

                    var countLabel = gt.format(
                        //#. %1$d is the number of rows in an oversized text table
                        //#, c-format
                        gt.ngettext('Successfully replaced %1$d cell.', 'Successfully replaced %1$d cells.', replacedCellsCount),
                        _.noI18n(replacedCellsCount)
                    );

                    self.yell({ type: 'info', message: countLabel });
                });

            } else { // single replacement
                var activeResult = searchLayoutData.sheets;

                // do a initial search if not done yet, and additionally everytime user changes the query.
                // Don't search while app is sending data to prevent invalid old results.
                if (_.isUndefined(activeResult) || activeSearchQuery !== app.getWindow().search.query) {
                    if (app.getState() !== 'sending') self.search(searchQuery);
                    return;
                }

                // fly to the search results if user are not on result cells
                if (indexOfAddress(activeResult[activeSheet], self.getActiveCell()) === -1) {
                    goToCell(activeResult[activeSheet][0]);
                }

                var currentCellContent = self.getCellContents(),
                    oldCellValue = '',
                    newCellValue = '';

                // generate replacement value:
                // if the current search result hit is a formula cell, check equality and replace the formula.
                // TODO handle number format special replacements like Date, Currency etc.
                if (_.isString(currentCellContent.formula)) {
                    newCellValue = currentCellContent.formula.replace(new RegExp(searchQuery, 'gi'), replacementText);
                    oldCellValue = currentCellContent.formula;
                } else { // if cell is a normal value cell, replace the display property.
                    newCellValue = currentCellContent.display.replace(new RegExp(searchQuery, 'gi'), replacementText);
                    oldCellValue = currentCellContent.display;
                }

                // and finally only do single replace operation if we have a new value
                if (oldCellValue !== newCellValue) {
                    replaceAction = true;
                    self.setCellContents(newCellValue, undefined, { parse: true });
                    var activeResultArray = activeResult[activeSheet],
                        activeCell = self.getActiveCell(),
                        activeCellIndex = 0;
                    // take out the cell to be replaced out of the result array, and save its index to
                    // do a immediate replacement at that address.
                    activeResult[activeSheet] = _.reject(activeResultArray, function (element, index) {
                        if (element[0] === activeCell[0] && element[1] === activeCell[1]) {
                            activeCellIndex = index;
                            return true;
                        }
                        return false;
                    });

                    var nextCell = activeResult[activeSheet][activeCellIndex];
                    if (!_.isUndefined(nextCell)) {
                        // iterate to next cell in the search result
                        goToCell(nextCell);
                    } else {
                        //we're at the end of the results array. Check if we still have results to iterate.
                        if (activeResult[activeSheet].length === 0) {
                            searchLayoutData.sheets = undefined;
                            self.yell({ type: 'info', message: gt('Nothing else to replace.') });
                        } else {
                            // iterate back at the beginning of the search result
                            goToCell(activeResult[activeSheet][0]);
                        }
                    }
                }
            }
        };

        /**
         * Clears local search result array and optionally search/replace input fields.
         *
         * @param {Object} [options.clearInputs=false] - Clears search/replace input fields if set to true.
         *
         */
        this.searchClear = function (options) {
            searchLayoutData = [];
            var win = app.getWindow();
            if (Utils.getBooleanOption(options, 'clearInputs', false)) {
                win.search.clear();
                win.search.replacementInput.val('');
            }
            replaceAction = false;
        };

        /**
         * Returns whether the format painter is currently active.
         *
         * @returns {Boolean}
         *  Whether the format painter is currently active.
         */
        this.isFormatPainterActive = function () {
            return _.isObject(formatPainterAttrs);
        };

        /**
         * Activates or deactivates the format painter.
         *
         * @param {Boolean} state
         *  Whether to activate or deactivate the format painter.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.activateFormatPainter = function (state) {

            // callback function used to apply the foramtting to the selected range
            function applyFormatPainter(selection) {
                if (_.isObject(selection)) {
                    self.setCellSelection(selection);
                    self.fillCellRanges(undefined, formatPainterAttrs, { clear: true });
                }
                formatPainterAttrs = null;
            }

            if (state) {
                // enter custom selection mode
                formatPainterAttrs = _.copy(this.getCellContents().explicit, true);
                this.enterCustomSelectionMode(applyFormatPainter, {
                    selection: this.getActiveCellAsSelection(),
                    statusLabel: Controls.FORMAT_PAINTER_LABEL
                });
            } else {
                // cancel custom selection mode
                this.leaveCustomSelectionMode();
            }

            return this;
        };

        /**
         * Creates a client clipboard ID and resets the server clipboard ID.
         *
         * @returns {String}
         *  The client clipboard ID.
         */
        this.createClientClipboardId = function () {
            clipboardId = {client: Clipboard.createClientClipboardId(), server: null};
            return clipboardId.client;
        };

        /**
         * Returns the server clipboard ID if the client clipboard ID matches
         * the given one, otherwise null.
         *
         * @param {String} clientClipboardId
         *  The client clipboard ID.
         *
         * @returns {String|Null}
         *  The corresponding server clipboard ID.
         */
        this.getServerClipboardId = function (clientClipboardId) {
            return (clipboardId.client === clientClipboardId) ? clipboardId.server : null;
        };

        /**
         * Initiates the server side clipboard copy action.
         *
         * @param {Object} selection
         *  The current selection.
         *
         * @returns {jQuery.Promise}
         *  The Promise of the copy action request.
         */
        this.clipboardServerCopy = function (selection) {

            var activeRange = selection.ranges[selection.activeRange],
                localRequestData = {
                    sheet: activeSheet,
                    selection: { sheet: activeSheet, start: activeRange.start, end: activeRange.end }
                };

            return app.sendFilterRequest({
                method: 'POST',
                params: {
                    action: 'copy',
                    requestdata: JSON.stringify(localRequestData)
                }
            })
            .done(function (result) {
                // store server clipboard id
                clipboardId.server = result.clipboardHandle;
                Utils.info('clipboard - copy - response received - server id: ' + result.clipboardHandle);
            })
            .fail(function () {
                Utils.error('clipboard - copy - request failed');
            });
        };

        /**
         * Initiates the server side clipboard paste action.
         * To avoid that operations are applied during the clipboard, the following steps are performed:
         * - enter busy state
         * - send paste request
         * - wait for paste request's  response
         * - wait for 'docs:update' event
         * - leave busy state again
         *
         * @param {Object} selection
         *  The current selection.
         *
         * @param {String} serverClipboardId
         *  The server clipboard id that identifies the data on the server side.
         *
         * @returns {jQuery.Promise}
         *  The Promise of the paste action request.
         */
        this.clipboardServerPaste = function (selection, serverClipboardId) {

            var activeRange = selection.ranges[selection.activeRange],
                localRequestData = {
                    sheet: activeSheet,
                    selection: { sheet: activeSheet, start: activeRange.start, end: activeRange.end },
                    clipboardHandle: serverClipboardId
                };

            /**
             * Returns a promise of a Deferred object that will be resolved
             * if a 'docs:update' event that results from a clipboard paste
             * is recieved within 60 seconds timeout.
             * Otherwise returns a promise of a rejected Deferred object.
             *
             * @returns {jQuery.Promise}
             *  The promise of a Deferred object.
             */
            function waitForPasteDocsUpdate() {
                var // the timer for the failure timeout
                    timer = null,
                    def = $.Deferred();


                // handles 'docs:update' notifications.
                function updateHandler(data) {
                    // check if the 'docs:update' was initiated by the paste request
                    if (data && data.pasted) {
                        def.resolve();
                        Utils.log('clipboard - matching docs:update recieved');
                    }
                }

                app.on('docs:update', updateHandler);

                timer = app.executeDelayed(function () {
                    def.reject();
                }, { delay: 60000 });

                def.fail(function () {
                    Utils.log('clipboard - the docs:update initiated by paste timed out');
                    app.rejectEditAttempt('clipboardTimeout');
                })
                .always(function () {
                    timer.abort();
                    app.off('docs:update', updateHandler);
                });

                return def.promise();
            }


            self.enterBusy();

            return app.sendFilterRequest({
                method: 'POST',
                params: {
                    action: 'paste',
                    requestdata: JSON.stringify(localRequestData)
                }
            })
            .then(waitForPasteDocsUpdate)
            .done(function () {
                Utils.info('clipboard - paste - succeeded');
            })
            .fail(function () {
                Utils.error('clipboard - paste - failed');
            })
            .always(function () {
                // leave busy state
                self.leaveBusy();
            });
        };

        // initialization -----------------------------------------------------

        // marker for touch devices and browser types
        Utils.addDeviceMarkers(rootNode);

        // create the header pane instances, initialize settings for pane sides
        _.each(['left', 'right', 'top', 'bottom'], function (paneSide) {
            paneSideSettings[paneSide] = _.clone(DEFAULT_PANE_SIDE_SETTINGS);
            headerPanes[paneSide] = new HeaderPane(app, paneSide);
        });

        // create the cell collections and grid pane instances
        _.each(['topLeft', 'topRight', 'bottomLeft', 'bottomRight'], function (panePos) {
            gridPanes[panePos] = new GridPane(app, panePos);
        });

        // create the corner pane instance
        cornerPane = new CornerPane(app);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            _.each(renderCaches, function (data, uid) { destroyRenderCache(uid); });
            _.invoke(gridPanes, 'destroy');
            _.invoke(headerPanes, 'destroy');
            cornerPane.destroy();
            gridPanes = headerPanes = cornerPane = statusPane = renderCaches = null;
            colSplitLineNode = rowSplitLineNode = resizeOverlayNode = null;
            colSplitTrackingNode = rowSplitTrackingNode = centerSplitTrackingNode = allSplitTrackingNodes = null;
            model = documentStyles = fontCollection = styleCollection = numberFormatter = null;
            activeSheetModel = activeRenderCache = null;
            cellCollection = colCollection = rowCollection = mergeCollection = drawingCollection = null;
        });

    } // class SpreadsheetView

    // exports ================================================================

    // derive this class from class EditView
    return EditView.extend({ constructor: SpreadsheetView });

});
