/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/formula/reference',
    ['io.ox/office/tk/utils',
     'io.ox/office/spreadsheet/utils/sheetutils',
     'io.ox/office/spreadsheet/model/formula/tokenutils'
    ], function (Utils, SheetUtils, TokenUtils) {

    'use strict';

    var // shortcut for the map of error code literals
        ErrorCodes = SheetUtils.ErrorCodes;

    // class Reference ========================================================

    /**
     * Representation of a reference literal: an unresolved list of cell range
     * addresses including their sheet indexes.
     *
     * @constructor
     *
     * @param {Object|Array} [ranges]
     *  The cell range, or an array of cell ranges to be inserted into this
     *  instance. Each cell range must contain the following properties:
     *  - {Number} sheet1
     *      The zero-based index of the first sheet containing the range.
     *  - {Number} sheet2
     *      The zero-based index of the last sheet containing the range.
     *  - {Number[]} start
     *      The address of the top-left cell in the range.
     *  - {Number[]} end
     *      The address of the bottom-right cell in the range.
     */
    function Reference(ranges) {

        // private methods ----------------------------------------------------

        /**
         * Throws a CIRCULAR_ERROR error code, if the passed range contains the
         * specified reference cell.
         */
        function checkCircularReference(range, refSheet, refAddress) {
            if ((range.sheet1 <= refSheet) && (refSheet <= range.sheet2) && SheetUtils.rangeContainsCell(range, refAddress)) {
                throw TokenUtils.CIRCULAR_ERROR;
            }
        }

        // public methods -----------------------------------------------------

        /**
         * Returns the number of ranges contained in this reference.
         *
         * @returns {Number}
         *  The number of ranges contained in this reference.
         */
        this.getLength = function () {
            return ranges.length;
        };

        /**
         * Returns the number of all cells contained in all ranges of this
         * reference.
         *
         * @returns {Number}
         *  The number of all cells contained in all ranges of this reference.
         */
        this.getCellCount = function () {
            return Utils.getSum(ranges, function (range) {
                return (range.sheet2 - range.sheet1 + 1) * SheetUtils.getCellCount(range);
            });
        };

        /**
         * Returns all ranges contained in this reference.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.sameSheet=false]
         *      If set to true, all ranges must refer to the same single sheet.
         *
         * @returns {Array}
         *  All ranges contained in this reference, if available.
         *
         * @throws {ErrorCode}
         *  - The #NULL! error code, if the reference is empty.
         *  - The #VALUE! error code, if the option 'sameSheet' is set, and the
         *      ranges in this reference refer to different sheets, or if at
         *      least one range refers to multiple sheets.
         */
        this.getRanges = function (options) {

            // reference must not be empty
            if (ranges.length === 0) { throw ErrorCodes.NULL; }

            // check that all ranges refer to the same single sheet
            if (Utils.getBooleanOption(options, 'sameSheet', false)) {

                // check that first range refers to a single sheet
                if (ranges[0].sheet1 !== ranges[0].sheet2) { throw ErrorCodes.VALUE; }

                // check sheet indexes of the following ranges
                for (var index = 1, length = ranges.length, sheet = ranges[0].sheet1; index < length; index += 1) {
                    if ((ranges[index].sheet1 !== sheet) || (ranges[index].sheet2 !== sheet)) {
                        throw ErrorCodes.VALUE;
                    }
                }
            }

            return _.clone(ranges);
        };

        /**
         * Returns the cell range of this reference, but only if it contains
         * exactly one range pointing to a single sheet.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {ErrorCode} [options.errorCode=ErrorCodes.REF]
         *      The error code to be thrown if the reference contains more than
         *      one range. Defaults to the #REF! error code.
         *  @param {Boolean} [options.multiSheet=false]
         *      If set to true, the range may refer to multiple sheets.
         *
         * @returns {Object}
         *  The range object, if available.
         *
         * @throws {ErrorCode}
         *  - The #NULL! error code, if the reference is empty.
         *  - The specified error code, if the reference contains more than one
         *      range.
         *  - The #VALUE! error code, if the range in this reference refers to
         *      multiple sheets, and the option 'multiSheet'. has not been set.
         */
        this.getSingleRange = function (options) {

            // must not be empty
            if (ranges.length === 0) {
                throw ErrorCodes.NULL;
            }

            // must be a single range in a single sheet
            if (ranges.length > 1) {
                throw Utils.getOption(options, 'errorCode', ErrorCodes.REF);
            }

            // must point to a single sheet, unless multiple sheets are allowed
            if (!Utils.getBooleanOption(options, 'multiSheet', false) && (ranges[0].sheet1 !== ranges[0].sheet2)) {
                throw ErrorCodes.VALUE;
            }

            return ranges[0];
        };

        /**
         * Extracts a single cell range from this reference, intended to be
         * resolved to the cell contents in the range. The reference must
         * contain exactly one range address pointing to a single sheet, and
         * the specified reference cell must not be part of that range.
         *
         * @param {Number} refSheet
         *  The zero-based index of the reference sheet.
         *
         * @param {Number[]} refAddress
         *  The address of the reference cell.
         *
         * @returns {Object}
         *  The range object, if available.
         *
         * @throws {ErrorCode}
         *  - The #NULL! error code, if the reference is empty.
         *  - The #VALUE! error code, if the reference contains more than one
         *      range, or if the range refers to multiple sheets.
         *  - The CIRCULAR_ERROR error code, if a range is available, but the
         *      reference cell is part of that range (circular reference).
         */
        this.getValueRange = function (refSheet, refAddress) {

            var // resolve to a single range (resolving to value throws #VALUE! on multiple ranges)
                range = this.getSingleRange({ errorCode: ErrorCodes.VALUE });

            // detect circular references
            checkCircularReference(range, refSheet, refAddress);
            return range;
        };

        /**
         * Invokes the passed iterator for all cell ranges in this reference,
         * intended to be resolved to the cell contents in the ranges.
         *
         * @param {Number} refSheet
         *  The zero-based index of the reference sheet.
         *
         * @param {Number[]} refAddress
         *  The address of the reference cell.
         *
         * @param {Function} iterator
         *  The iterator callback function invoked for every range. Receives
         *  the current range as first parameter, and its index as second
         *  parameter. May return Utils.BREAK to stop iteration immediately
         *  without causing the formula interpretation to fail.
         *
         * @param {Object} [context]
         *  The calling context for the iterator function.
         *
         * @returns {Utils.BREAK|Undefined}
         *  The result of the iterator.
         *
         * @throws {ErrorCode}
         *  - The #NULL! error code, if the reference is empty.
         *  - The CIRCULAR_ERROR error code, if the reference cell is part of
         *      any of the ranges in this reference (circular reference).
         */
        this.iterateValueRanges = function (refSheet, refAddress, iterator, context) {

            // must not be empty
            if (ranges.length === 0) { throw ErrorCodes.NULL; }

            // call passed iterator for all ranges
            return Utils.iterateArray(ranges, function (range, index) {
                checkCircularReference(range, refSheet, refAddress);
                return iterator.call(context, range, index);
            });
        };

        /**
         * Returns the string representation of this reference for debug
         * logging.
         */
        this.toString = function () {
            return _.map(ranges, function (range) {
                return range.sheet1 + ':' + range.sheet2 + '!' + SheetUtils.getRangeName(range);
            }).join(',');
        };

        // initialization -----------------------------------------------------

        // convert ranges to an array
        ranges = _.isArray(ranges) ? _.clone(ranges) : _.isObject(ranges) ? [ranges] : [];

    } // class Reference

    // exports ================================================================

    return Reference;

});
