/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Stefan Eckert <stefan.eckert@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/drawing/manualcolorchart',
        ['io.ox/office/tk/utils',
         'io.ox/office/drawinglayer/view/chartstyleutil'], function (Utils, ChartStyleUtil) {

    'use strict';

    var BACKGROUNDTRANSFORMATION = ChartStyleUtil.getBackgroundTransformation();

    var STYLESET = ChartStyleUtil.getStyleSet();

    var COLORSET = ChartStyleUtil.getColorSet();

    function ManualColorChart(app, chartmodel, data) {
        var self = this;
        data.colorSet = null;

        // private methods ------------------------------------------------------------

        function getColorOfPattern(meth, index, schemeColors, variations) {
            var variationIndex = null;
            var colorIndex = null;
            if (meth === 'cycle') {
                variationIndex = index % variations.length;
                colorIndex = index % schemeColors.length;
            } else {
                Utils.warn('cant handle color meth:' + meth);
                return null;
            }

            var color = schemeColors[colorIndex];
            var variation = variations[variationIndex];

            var tmpColor = {};
            tmpColor.type = color.type;
            tmpColor.value = color.value;
            if (variation) {
                tmpColor.transformations = variation.transformations;
            }
            return tmpColor;
        }

        function getColor(index, type) {
            var attributes = chartmodel.getMergedAttributes();

            var colors = attributes.chart.chartColors;

            var result = null;

            if (colors) {
                //modern color pattern
                result = getColorOfPattern(colors.meth, index, colors.schemeClr, colors.variation);
            } else if (attributes.chart.chartStyleId) {
                var styleId = attributes.chart.chartStyleId - 1;
                var colorSet = COLORSET[styleId % 8];

                var res = getColorOfPattern('cycle', index, colorSet.colors, colorSet.variations);
                result = {
                    color: res,
                    type: type
                };
            }

            return result;
        }

        function getStyleSet(index, color) {
            var attributes = chartmodel.getMergedAttributes();

            var result = color;

            if (attributes.chart.chartStyleId) {
                var styleId = attributes.chart.chartStyleId - 1;
                var styleSet = STYLESET[(styleId / 8)|0];

                var dataSeries = data.data[index];
                dataSeries.bevelEnabled = styleSet.bevelEnabled;
            }

            if (index === 0) {
                data.cssBackgroundColor = null;
                if (attributes.chart.chartStyleId && result) {

                    var styleId = attributes.chart.chartStyleId - 1;
                    var styleSet = STYLESET[(styleId / 8)|0];

                    var cells = app.getModel().getStyleCollection('cell');

                    var givenStyle = styleSet.bg;
                    if (!givenStyle) {
                        givenStyle = {type: result.color.type, value: result.color.value, transformations: BACKGROUNDTRANSFORMATION};

                    }

                    var bg = cells.getCssColor(givenStyle, 'fill');
                    data.cssBackgroundColor = bg;

                }
            }
        }

        function getVaryColor() {
            if (self.isVaryColorEnabled()) {
                var attributes = chartmodel.getMergedAttributes();
                return attributes.chart.varyColors;
            }
            return false;
        }

        function accessColor(attrs) {
            var type = chartmodel.getMergedAttributes().chart.type;
            if (type.indexOf('line') >= 0 || type.indexOf('scatter') >= 0) {
                return attrs.line;
            } else {
                return attrs.fill;
            }
        }

        function applyColor(cjsPoint, fill){
            var cells = app.getModel().getStyleCollection('cell');

            var cssColor = cells.getCssColor(fill.color, 'fill');
            if (fill.type !== 'solid') {
                cjsPoint.color = 'transparent';
            } else {
                cjsPoint.color = cssColor;
            }
            cjsPoint.markerColor = cssColor;
        }

        // public functions ------------------------------------------------------------

        this.isVaryColorEnabled = function () {
            if (data.data.length === 1) {
                return true;
            }
            var type = chartmodel.getChartType().split(' ')[0];
            if (type === 'pie' || type === 'donut') {
                return true;
            }
            return false;
        };

        this.handleColor = function (attrs, index) {
            var vary = getVaryColor();
            var fill = accessColor(attrs);
            var dataSeries = data.data[index];
            var attributes = chartmodel.getMergedAttributes();

            _(dataSeries.dataPoints).each(function (dataPoint) {
                dataPoint.color = null;
                dataPoint.markerColor = null;
            });

            if (fill.color.type === 'auto') {
                if (vary) {
                    dataSeries.color = null;
                    _(dataSeries.dataPoints).each(function (dataPoint, pi) {
                        var colorHolder = getColor(pi, fill.type);
                        if (pi === 0) {
                            getStyleSet(0, colorHolder);
                        }
                        applyColor(dataPoint, colorHolder);
                    });
                } else {
                    fill = getColor(index, fill.type);
                }
            }

            if (fill.color.type !== 'auto') {
                applyColor(dataSeries, fill);
            }

            if (!vary) {
                getStyleSet(index, fill);
            }
            if (index === 0) {
                var bg = attributes.fill.color;
                if (bg && bg.type !== 'auto') {
                    var cells = app.getModel().getStyleCollection('cell');
                    data.cssBackgroundColor = cells.getCssColor(bg, 'fill');
                }
            }
        };

        this.getBackgroundColor = function (chartStyleId) {
            var styleId = chartStyleId - 1;
            var index = (styleId / 8)|0;
            var styleSet = STYLESET[index];

            var givenStyle = styleSet.bg;
            if (!givenStyle) {
                var colorSet = COLORSET[styleId % 8];
                var color = getColorOfPattern('cycle', index, colorSet.colors, colorSet.variations);
                givenStyle = {type: color.type, value: color.value, transformations: BACKGROUNDTRANSFORMATION};
            }
            if (!givenStyle.fallbackValue) {
                var cells = app.getModel().getStyleCollection('cell');
                givenStyle.fallbackValue = cells.getCssColor(givenStyle, 'fill');
            }
            return givenStyle;
        };

    } // class ManualColorChart

    // exports ================================================================

    return _.makeExtendable(ManualColorChart);

});
