/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Carsten Driesner <carsten.driesner@open-xchange.com>
 * @author Stefan Eckert <stefan.eckert@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/drawing/chartmodel',
        ['io.ox/office/tk/utils',
         'io.ox/office/editframework/utils/color',
         'io.ox/office/drawinglayer/model/drawingmodel',
         'io.ox/office/drawinglayer/view/chartstyleutil',
         'io.ox/office/drawinglayer/view/chartformatter',
         'io.ox/office/spreadsheet/utils/sheetutils',
         'io.ox/office/spreadsheet/model/operations',
         'io.ox/office/spreadsheet/model/formula/tokenarray',
         'io.ox/office/spreadsheet/model/drawing/drawingmodelmixin',
         'io.ox/office/spreadsheet/model/drawing/manualcolorchart',
         'io.ox/office/spreadsheet/model/drawing/axismodel',
         'io.ox/office/spreadsheet/model/drawing/dataseriesmodel',
         'io.ox/office/spreadsheet/model/drawing/legendmodel',
         'io.ox/office/spreadsheet/model/drawing/axistitlemodel'
        ], function (Utils, Color, DrawingModel, ChartStyleUtil, ChartFormatter, SheetUtils, Operations, TokenArray, DrawingModelMixin, ManualColor, AxisModel, DataSeriesModel, LegendModel, AxisTitleModel) {

    'use strict';

    var TYPES = {
        'column standard' : 'column',
        'column clustered' : 'column',
        'column stacked' : 'stackedColumn',
        'column percentStacked' : 'stackedColumn100',
        'bar standard' : 'bar',
        'bar clustered' : 'bar',
        'bar stacked' : 'stackedBar',
        'bar percentStacked' : 'stackedBar100',
        'area standard' : 'area',
        'area stacked' : 'stackedArea',
        'area percentStacked' : 'stackedArea100',
        'line standard' : 'line',
        'line clustered' : 'line',
        'line stacked' : 'line',
        'line percentStacked' : 'spline',
        'line standard curved' : 'spline',
        'line clustered curved' : 'spline',
        'line stacked curved' : 'spline',
        'line percentStacked curved' : 'spline',
        'scatter standard' : 'line',
        'scatter standard curved' : 'spline',
        'bubble standard' : 'bubble',
        'pie standard' : 'pie',
        'donut standard' : 'doughnut'
    };

    var MARKER_TYPES = {
        'circle': 'circle',
        'dot': 'circle',
        'square': 'square',
        'triangle' : 'triangle',
        'x' : 'cross',
        'none': 'none'
    };
    var MARKER_LIST = ['circle', 'square', 'triangle', 'cross'];

    // class SheetChartModel ==================================================

    /**
     * A class that implements a chart model to be inserted into a sheet of a
     * spreadsheet document.
     *
     * Additionally to the events triggered by the base class DrawingModel,
     * instances of this class trigger the following 'change:drawing' events:
     * - 'insert:series'
     *      After a new data series has been inserted into this chart. Event
     *      handlers receive the logical index of the new data series.
     * - 'delete:series'
     *      After a new data series has been deleted from this chart. Event
     *      handlers receive the last logical index of the data series.
     *
     * @constructor
     *
     * @extends DrawingModel
     * @extends DrawingModelMixin
     *
     * @param {SpreadsheetApplication} app
     *  The application containing this chart model.
     *
     * @param {SheetModel} sheetModel
     *  The model instance of the sheet that contains this chart object.
     *
     * @param {Object} [initAttributes]
     *  Initial formatting attribute set for this chart model.
     */
    function SheetChartModel(app, sheetModel, initAttributes) {

        var self = this;

        var docModel = app.getModel();

        var controllerToken = null;

        var data = {
            culture:  'en',
//            zoomEnabled: true,
            bg: 'white',
            backgroundColor: 'transparent',
            axisX: {stripLines: []},
            axisY: {stripLines: []},
            axisZ: {stripLines: []},
            creditHref: '',
            creditText: '',
            title: { text: ''},
            legend: {},
            data: []
        };


        data.axisY.stripLines[0] = {
            value: 0,
            color: 'black',
            label : '',
            zero: true
        };

        var indexLabelAttrs = {
            fontSize: 10,
            color: {type: 'auto'},
            fontName: 'Arial'
        };

        var colorHandler = new ManualColor(app, self, data);
        var chartFormatter = new ChartFormatter(self);

        // base constructors --------------------------------------------------

        DrawingModel.call(this, app, 'chart', initAttributes, { additionalFamilies: ['chart', 'fill'] });
        DrawingModelMixin.call(this, app, sheetModel);


        this.getApp = function() {
            return app;
        };

        /**
         * this is only called by the Drawingframe to initialize the REAL CanvasJS-chart
         */
        this.getModelData = function () {
            return data;
        };

        var axes = {
            x : new AxisModel(self, 'x', 'y'),
            y : new AxisModel(self, 'y', 'x'),
            z : new AxisModel(self, 'z', 'x')
        };

        var mainName = null;
        var mainTitleModel = new AxisTitleModel(self);

        var legendModel = new LegendModel(this, {}, data.legend);


        // private methods ----------------------------------------------------

        function colRowChangeHandler(event, sheet, interval) {
            switch (event.type) {
            case 'insert:columns':
                checkDataInsideInterval(sheet, interval, true, true);
                break;
            case 'delete:columns':
                checkDataInsideInterval(sheet, interval, false, true);
                break;
            case 'insert:rows':
                checkDataInsideInterval(sheet, interval, true, false);
                break;
            case 'delete:rows':
                checkDataInsideInterval(sheet, interval, false, false);
                break;
            }
        }

        function checkDataInsideInterval(sheetModelIndex, interval, insert, columns) {
            var res = null;

            for (var i = 0; i < data.data.length; i++) {
                var dataSeries = data.data[i];
                var model = dataSeries.model;

                var change = model.checkRangeInsideInterval(sheetModelIndex, interval, insert, columns);
                if (change) {
                    dataSeries.needUpdate = true;
                    res = true;
                }
            }
            return res;
        }

        /**
         * Handles 'docs:update:cells' notifications. If this chart refers to
         * changed cells, the respective data series will be updated.
         */
        function updateNotificationHandler(changedData, allChangedRanges) {

            var // whether any data series overlaps with the changed ranges
                changed = false;

            // check passed ranges if they overlap with the source ranges of the data series
            _(data.data).each(function (dataSeries) {
                if (dataSeries.needUpdate) {
                    changed = true;
                } else if (dataSeries.model.rangesOverlap(allChangedRanges)) {
                    changed = true;
                    dataSeries.needUpdate = true;
                }
            });

            if (changed) { self.refresh(); }
        }

        /**
         * Handles 'change:cells' events triggered by the document model (from
         * any sheet), and refreshes this chart model debounced, if it refers
         * to any of the changed cells.
         */
        var changeCellsHandler = (function () {

            var // whether any data series overlaps with the changed ranges
                changed = false;

            function eventHandler(event, sheet, ranges) {

                // chart is already dirty, no need to check the ranges
                if (changed) { return; }

                // insert sheet index into the ranges (required by series model)
                ranges = _.map(ranges, function (range) {
                    return _.extend({ sheet: sheet }, range);
                });

                // check passed ranges if they overlap with the source ranges of the data series
                _.each(data.data, function (dataSeries) {
                    if (dataSeries.needUpdate) {
                        changed = true;
                    } else if (dataSeries.model.rangesOverlap(ranges)) {
                        dataSeries.needUpdate = changed = true;
                    }
                });
            }

            function refreshChart() {
                if (changed && self && !self.destroyed) {
                    changed = false;
                    self.refresh();
                }
            }

            // wait quite a long time before refreshing the chart, cell ranges
            // may be queried in a background loop
            return app.createDebouncedMethod(eventHandler, refreshChart, { delay: 500 });
        }());

        function changeAttributes(evt, mergedAttributes, oldMergedAttributes) {
            var mc = mergedAttributes.chart;
            var oc = oldMergedAttributes.chart;
            var mf = mergedAttributes.fill;
            var of = oldMergedAttributes.fill;

            if (!_.isEqual(mc, oc) || !_.isEqual(mf, of)) {

                updateChartType();

                changeDrawing(evt, 'series');
            }
        }

        function isXYType() {
            var attributes = self.getMergedAttributes();
            return attributes.chart.type.indexOf('bubble') === 0 || attributes.chart.type.indexOf('scatter') === 0;
        }


        function changeDrawing(evt, subtype) {
            var lAttrs = legendModel.getMergedAttributes();
            var attributes = self.getMergedAttributes();

            var cjst = getCJSType();

            _(data.data).each(function (dataSeries, index) {
                if(dataSeries.needUpdate){
                    return;
                }
                if (subtype === 'series' || !index) {
                    colorHandler.handleColor(dataSeries.model.getMergedAttributes(), index);
                }
                if (cjst === 'pie' || cjst === 'doughnut') {
                    dataSeries.startAngle = attributes.chart.rotation - 90;
                }
                if (lAttrs.legend.pos === 'off') {
                    dataSeries.showInLegend = false;
                } else {
                    dataSeries.showInLegend = true;
                    if (cjst === 'pie' || cjst === 'doughnut') {
                        if (index > 0) {
                            //we only see the first series
                            dataSeries.showInLegend = false;
                        } else if (dataSeries.dataPoints.length > 50) {
                            //canvasjs has rendering problems with to big legend-sizes
                            dataSeries.showInLegend = false;
                        }
                    }
                }

                ChartStyleUtil.handleCharacterProps(self, indexLabelAttrs, dataSeries, 'indexLabel');

                if (attributes.chart.dataLabel) {
                    dataSeries.indexLabel = '{name}';
                    dataSeries.indexLabelLineColor = null;
                    dataSeries.indexLabelPlacement = 'inside';
                } else {
                    dataSeries.indexLabel = ' ';
                    dataSeries.indexLabelLineColor = 'transparent';
                    dataSeries.indexLabelPlacement = null;
                }

                if (cjst === 'line' || cjst === 'spline') {
                    dataSeries.legendMarkerType = dataSeries.marker;
                } else {
                    dataSeries.legendMarkerType = 'square';
                }

                _.each(dataSeries.dataPoints, function (dataPoint) {
                    if (cjst === 'line' || cjst === 'spline') {
                        dataPoint.markerType = dataSeries.marker;
                    } else {
                        dataPoint.markerType = null;
                    }
                });
            });

            chartFormatter.format();

            var mnt = mainTitleModel.getMergedAttributes();
            data.title.text = mnt.text.link[0] || '';
            if (!data.title.text && data.data.length === 1) {
                data.title.text = mainName || '';
            }
            if (data.title.text) {
                ChartStyleUtil.handleCharacterProps(self, mnt.character, data.title);
            }

            legendModel.refreshInfo();
            refreshAxis();
        }

        function getCJSType() {
            var type = self.getChartType();
            var res = TYPES[type];

            if (!res) {
                Utils.warn('no correct charttype found', type);
                res = 'column';
            }

            if (res === 'line' && self.getMergedAttributes().chart.curved === true) {
                res = 'spline';
            }
            return res;
        }

        function updateChartType() {
            var cjs = getCJSType();
            for (var i = 0; i < data.data.length; i++) {
                var series = data.data[i];
                series.type = cjs;
            }
            self.trigger('change:drawing');
        }

        // public methods -----------------------------------------------------

        this.setMainName = function(name) {
            mainName = name;
        };

        /**
         * is called by the DrawingController for a selected chart
         */
        this.getChartType = function () {
            var attributes = self.getMergedAttributes();

            var type = attributes.chart.type;
            type = type.replace('2d', '').replace('3d', '');

            var stacking = attributes.chart.stacking;
            if (stacking.length) {
                type = type + ' ' + stacking;
            }
            var curved = attributes.chart.curved;
            if (curved && /(line|scatter)/.test(type)) {
                type = type + ' curved';
            }
            return type;
        };

        this.getChartTypeForGui = function() {
            var chartType = this.getChartType();
            return chartType.replace('clustered', 'standard');
        };

        /**
         * @param operation
         */
        this.deleteDataSeries = function (operation) {
            var removed = data.data.splice(operation.series, 1);
            removed[0].model.destroy();
            this.trigger('change:drawing');
            return true;
        };

        /**
         * refresh checks if something is changed in the series,
         * if true it calls the 'updateview'-request to the server
         */
        this.refresh = app.createDebouncedMethod($.noop, function () {

            if (!app.isImportFinished()) {
                return;
            }

            var // all ranges to be querried from server, as array of arrays of range addresses
                sourceRanges = null,
                // maps series indexes and source link types to array elements in 'sourceRanges'
                collectInfo = null;

            // debounced method: chart may have been deleted in the meantime
            if (!self || self.destroyed) { return; }

            _(data.data).each(function (seriesData, series) {
                if (!seriesData.needUpdate) { return; }

                if (!sourceRanges) {
                    sourceRanges = [];
                    collectInfo = [];
                }

                var info = { series: series, indexes: {} };


                seriesData.model.iterateTokens(function (token, name) {
                    var entryRanges = token.resolveRangeList();
                    if (_.isArray(entryRanges)) {
                        info.indexes[name] = sourceRanges.length;
                        sourceRanges.push(entryRanges);
                    }
                });
                collectInfo.push(info);

                seriesData.needUpdate = false;
            });

            if (!_.isArray(sourceRanges)) {
                changeDrawing();
                return;
            }

            if (sourceRanges.length > 0) {

                var // query cell contents for the data series
                    request = docModel.queryCellContents(sourceRanges, { attributes: true });

                // do not call the handler function, if the chart has been deleted in the meantime
                self.listenTo(request, 'done', function (allContents) {

                    _(collectInfo).each(function (info) {

                        // returns the cell contents for the specified source link type
                        function getCellContents(key) {
                            return _.isNumber(info.indexes[key]) ? allContents[info.indexes[key]] : [];
                        }

                        // series values must exist
                        var valuesSource = getCellContents('values');
                        if (valuesSource.length === 0) { return; }

                        // all other source data may be missing
                        var titleSource = getCellContents('title');
                        var namesSource = getCellContents('names');
                        var bubblesSource = getCellContents('bubbles');

                        chartFormatter.update(info.series, valuesSource, titleSource, namesSource, bubblesSource);

                    });
                    refreshAxis();
                    self.trigger('change:drawing', 'series');
                });
            } else {
                _(data.data).each(function (seriesData) {
                    seriesData.needUpdate = false;
                });
                self.trigger('change:drawing', 'series');
            }
        }, { delay: 50, mayDelay: 250 });

        /**
         * saves the Operation-Series in the model, but fetches the correct Data only if its visible
         * via trigger the Events 'change:drawing' and 'insert:series' and mark the current series that ists changed
         * @param operation
         *  @param {Object} [options.attrs]
         *   @param {Object} [options.attrs.series]
         *    the content of the series could be a formula {String}, or explicit Values {Array}.
         *    the values can be taken directy. the formula calls a request to server, the next time the chart visible
         *    @param {String|Array} [options.attrs.series.title]
         *    @param {String|Array} [options.attrs.series.names]
         *    @param {String|Array} [options.attrs.series.bubbles]
         *    @param {String|Array} [options.attrs.series.values]
         *
         * @returns {Boolean}
         */
        this.insertDataSeries = function (operation) {
            var dataSeries = {
                type: getCJSType()
            };
            data.data.splice(operation.series, 0, dataSeries);

            var series = operation.attrs.series;

            dataSeries.marker = MARKER_TYPES[series.marker];
            if (!dataSeries.marker) {
                var index = operation.series % MARKER_LIST.length;
                dataSeries.marker = MARKER_LIST[index];
            }

            var model = new DataSeriesModel(app, sheetModel, operation.attrs);

            dataSeries.model = model;

            dataSeries.needUpdate = true;

            dataSeries.dataPoints = [];
            dataSeries.name = '';

            this.trigger('change:drawing');

            if (controllerToken) {
                updateTokenArrays();
            }

            return true;
        };

        this.setAxisAttributes = function (operation) {
            var axisModel = this.getAxisModel(operation.axis);

            axisModel.setAttributes(operation.attrs);
            self.trigger('change:drawing');
            return true;
        };

        this.setGridlineAttributes = function (operation) {
            var gridModel = this.getAxisModel(operation.axis).getGrid();
            gridModel.setAttributes(operation.attrs);
            self.trigger('change:drawing');
            return true;
        };

        this.setTitleAttributes = function (operation) {
            if (operation.axis === 'main') {
                mainTitleModel.setAttributes(operation.attrs);
            } else {
                var titleModel = this.getAxisModel(operation.axis).getTitle();
                titleModel.setAttributes(operation.attrs);
            }
            self.trigger('change:drawing');
            return true;
        };

        this.setLegendAttributes = function (operation) {
            legendModel.setAttributes(operation.attrs);
            self.trigger('change:drawing', 'series');
        };

        this.getLegendModel = function () {
            return legendModel;
        };

        this.getAxisModel = function (axis) {
            return axes[axis];
        };

        this.getTitleModel = function (type) {
            if (type === 'main') {
                return mainTitleModel;
            } else {
                return this.getAxisModel(type).getTitle();
            }
        };

        this.setChartDataSeriesAttributes = function (operation) {
            var index = operation.series;
            var attrs = operation.attrs;

            var dataSeries = data.data[index];
            dataSeries.model.setAttributes(attrs);
            dataSeries.needUpdate = true;
            updateTokenArrays();
            self.trigger('change:drawing', 'series');
            return true;
        };

        /**
         *  Adds formula tokens to the given containing the source ranges of all data
         *  series of this chart object.
         *
         *  @param {Array} array where tokenarrays are pushed in
         */
        this.getTokenArrays = function (res) {
            updateTokenArrays();
            res.push(controllerToken);
        };

        function updateTokenArrays() {
            if (!controllerToken) {
                controllerToken = new TokenArray(app);
            }

            var tmpList = {};
            _.each(data.data, function (dataSerie) {
                dataSerie.model.iterateTokens(function (tkn) {
                    var formula = tkn.getFormula();
                    if (formula.length) {
                        tmpList[formula] = true;
                    }
                });
            });

            var form = _(tmpList).keys().join(',');
            controllerToken.parseFormula(form);
        }

        function refreshAxis() {
            if (!data.data.length) {
                return;
            }
            axes.x.refreshInfo();
            axes.y.refreshInfo();
        }

        function getStyleId() {
            return self.getMergedAttributes().chart.chartStyleId;
        }

        function clearSeriesColor(generator, position) {
            _(data.data).each(function (dataSeries, index) {
                var at = dataSeries.model.getMergedAttributes();
                if (at.fill.color.type !== 'auto' || at.line.color.type !== 'auto') {
                    generator.generateDrawingOperation(Operations.SET_CHART_DATASERIES_ATTRIBUTES, position[0], position[1], { series: index, attrs: { fill: { color: Color.AUTO }, line: { color: Color.AUTO } } });
                }
            });
        }

        function createBackgroundColorOp(styleId) {
            var drAttrs = { chart: {chartStyleId: styleId } };

            var bgColor = colorHandler.getBackgroundColor(styleId);
            if (bgColor) {
                drAttrs.fill = { color: bgColor };
            }
            return drAttrs;
        }

        this.isXYType = isXYType;

        /**
         * return true if the chart has only one series or is type of bubble or donut (because there you see only the first series)
         * @returns {Boolean}
         */
        this.isVaryColorEnabled = function () {
            return colorHandler.isVaryColorEnabled();
        };

        this.isAxesEnabled = function () {
            var type = getCJSType();
            return type !== 'pie' && type !== 'doughnut';
        };

        this.isDataLabelEnabled = function () {
            var type = getCJSType();
            return type !== 'bubble';
        };

        /**
         * it generates a unique-color-ID out of the internal chartStyleId
         * for describing the color patterns
         * return null, if one or more series have a unique fill color
         * @returns {String|null}
         */
        this.getColorSet = function () {
            var indiv = _(data.data).find(function (dataSeries) {
                var at = dataSeries.model.getMergedAttributes();
                if (at.fill.color.type !== 'auto') {
                    return true;
                }
            });

            if (indiv) {
                return null;
            }

            var styleId = getStyleId() - 1;
            var colorSet = styleId % 8;
            return 'cs' + colorSet;
        };

        /**
         * it generates a unique-style-ID out of the internal chartStyleId
         * for describing the background and the highlight of the chart
         * it ignores unique fill colors of the series
         * @returns {String}
         */
        this.getStyleSet = function () {
            var styleId = getStyleId() - 1;
            var styleSet = (styleId / 8) | 0;

            if (styleSet < 4) {
                return 'ss0';
            } else {
                return 'ss' + styleSet;
            }
        };

        /**
         * updates the interal chartStyleId by the unique ID
         * deletes all series colors if existing
         * @param colorSet
         */
        this.changeColorSet = function (colorSet) {
            var styleId = getStyleId() - 1;
            var csId = colorSet.replace('cs', '') | 0;
            var styleSet = (styleId / 8) | 0;

            styleId = styleSet * 8 + csId + 1;


            var position = self.getPosition();
            var generator = docModel.createOperationsGenerator();

            clearSeriesColor(generator, position);

            var drAttrs = createBackgroundColorOp(styleId);
            generator.generateDrawingOperation(Operations.SET_DRAWING_ATTRIBUTES, position[0], position[1], { attrs: drAttrs });
            docModel.applyOperations(generator.getOperations());

        };

        /**
         * updates the interal chartStyleId by the unique ID
         * does not touch the series colors
         * @param colorSet
         */
        this.changeStyleSet = function (styleSet) {
            var styleId = getStyleId() - 1;
            var ssId = styleSet.replace('ss', '') | 0;
            var colorSet = styleId % 8;

            styleId = ssId * 8 + colorSet + 1;

            var position = self.getPosition();

            var generator = docModel.createOperationsGenerator();

            var drAttrs = createBackgroundColorOp(styleId);
            generator.generateDrawingOperation(Operations.SET_DRAWING_ATTRIBUTES, position[0], position[1], { attrs: drAttrs });
            docModel.applyOperations(generator.getOperations());

        };

        /**
         * changes the varycolors flag
         * deletes all series colors if existing, so the colors only orient on the patterns
         * @param colorSet
         */
        this.changeVaryColors = function (state) {
            var position = self.getPosition();
            var generator = docModel.createOperationsGenerator();

            clearSeriesColor(generator, position);

            generator.generateDrawingOperation(Operations.SET_DRAWING_ATTRIBUTES, position[0], position[1], { attrs: { chart: {varyColors: state} } });

            docModel.applyOperations(generator.getOperations());
        };

        this.getSeriesCount = function () {
            return data.data.length;
        };

        /**
         * must be called for having correct behavior mit "varyColors" in combination with more than one series
         */
        this.firstInit = function () {
            updateChartType();
        };

        /**
         * must be called for having correct behavior mit "new CanvasJS.Chart" is called, because of a crazy bug inside there
         */
        this.resetData = function() {
            data.axisX._oldOptions = null;
            data.axisY._oldOptions = null;
        };

        /**
         * iterates all sources and calculates min and max positions,
         * direction of the series-values and the sheetindex
         *
         * @returns {object}
         * [res.warn] reutrns if sources are not on the same sheet,
         * or if they have different direction
         * (these are symptoms that the sources have no definite relations)
         * [res.range.start] start has the smallest position of all sources
         * [res.range.end] start has the biggest position of all sources
         * [res.axis] detected main-axis of the source-values
         * [res.sheet] sheet-index of the sources
         *
         */
        this.getExchangeInfo = function () {
            var start = [docModel.getMaxCol() + 1, docModel.getMaxRow() + 1];
            var end = [-1, -1];

            var mainAxis = null;
            var sheet = null;
            var warn = null;

            _.each(data.data, function (dataSeries) {
                dataSeries.model.iterateTokens(function (token, name) {
                    var rangeList = token.resolveRangeList();
                    if (!rangeList) {
                        return;
                    }
                    var range = rangeList[0];
                    if (!range) {
                        return;
                    }
                    if (sheet === null) {
                        sheet = range.sheet;
                    } else if (sheet !== range.sheet) {
                        warn = 'sheets';
                    }
                    if (name === 'values') {
                        var axis = null;
                        var col = SheetUtils.getColCount(range);
                        var row = SheetUtils.getRowCount(range);
                        if (col > row) {
                            axis = 0;
                        } else if (row > 1) {
                            axis = 1;
                        }
                        if (_.isNumber(axis)) {
                            if (mainAxis === null) {
                                mainAxis = axis;
                            } else if (mainAxis !== axis) {
                                warn = 'directions';
                            }
                        }
                    }
                    _.times(2, function (ax) {
                        start[ax] = Math.min(start[ax], range.start[ax]);
                        end[ax] = Math.max(end[ax], range.end[ax]);
                    });
                });
            });
            if (warn) {
                return {warn: warn};
            } else if (sheet === null) {
                return {warn: 'nodata'};
            } else {
                return {range: {start: start, end: end}, axis: mainAxis, sheet: sheet};
            }
        };

        // initialization -----------------------------------------------------

        this.on('change:attributes', changeAttributes);
        this.on('change:drawing', changeDrawing);
        this.listenTo(app, 'docs:update:cells', updateNotificationHandler);
        this.listenTo(docModel, 'insert:columns delete:columns insert:rows delete:rows', colRowChangeHandler);
        this.listenTo(docModel, 'change:cells', changeCellsHandler);

		this.listenTo(app.getImportPromise(), 'done', function() {
		    app.executeDelayed(function() {
		        //small hack for fastcalc-loading
		        var position = self.getPosition();
		        var view = app.getView();

		        if (position[0] === view.getActiveSheet()) {
		            self.refresh();
		        }
            });
        });

        // constructor called from BaseObject.clone()
        this.registerCloneConstructor(function (newSheetModel, newSheet) {

            var cloneData = { series: [], axes: axes, oldSheet: sheetModel.getIndex(), newSheet: newSheet, legend: legendModel, title: mainTitleModel },
                tokenArray = new TokenArray(app, sheetModel);

            _.each(data.data, function (dataSeries) {
                var series = {};

                dataSeries.model.iterateTokens(function (token, name) {
                    tokenArray.parseFormula(token.getFormula());
                    tokenArray.relocateSheet(cloneData.oldSheet, newSheet);
                    series[name] = tokenArray.getFormula();
                });
                var nsp = dataSeries.model.getExplicitAttributes();
                nsp.series = series;
                cloneData.series.push(nsp);
            });
            tokenArray.destroy();
            tokenArray = null;

            // pass private data as hidden arguments to the constructor
            return new SheetChartModel(app, newSheetModel, this.getExplicitAttributes(), cloneData);
        });

        // clone private data passed as hidden argument to the c'tor
        (function (args) {
            var cloneData = args[SheetChartModel.length];
            if (_.isObject(cloneData)) {
                app.executeDelayed(function () {
                    if (!self || self.destroyed) { return; }
                    _.each(cloneData.axes, function (axisModel, axisType) {
                        self.setAxisAttributes({ axis: axisType, attrs: axisModel.getExplicitAttributes() });
                        var axGrid = axisModel.getGrid();
                        if(axGrid){
                            self.setGridlineAttributes({ axis: axisType, attrs: axGrid.getExplicitAttributes() });
                        }
                        var axTitle = axisModel.getTitle();
                        if (axTitle) {
                            self.setTitleAttributes({ axis: axisType, attrs: axTitle.getExplicitAttributes() });
                        }
                    });
                    _.each(cloneData.series, function (series, index) {
                        self.insertDataSeries({ series: index, attrs: series });
                    });
                    self.setTitleAttributes({ axis: 'main', attrs: cloneData.title.getExplicitAttributes() });
                    self.setLegendAttributes({ attrs: cloneData.legend.getExplicitAttributes() });
                });
            }
        }(arguments));

        // destroy all class members on destruction
        this.registerDestructor(function () {
            _(axes).invoke('destroy');
            _.chain(data.data).pluck('model').invoke('destroy');
            mainTitleModel.destroy();
            legendModel.destroy();

            mainTitleModel = legendModel = docModel = self = data = axes = null;

        });

    } // class SheetChartModel

    // exports ================================================================

    // derive this class from class DrawingModel
    return DrawingModel.extend({ constructor: SheetChartModel });

});
