/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/editframework/view/edittoolpane',
    ['io.ox/office/tk/utils',
     'io.ox/office/tk/forms',
     'io.ox/office/baseframework/view/toolpane',
     'io.ox/office/baseframework/view/toolbar',
     'io.ox/office/baseframework/view/basecontrols'
    ], function (Utils, Forms, ToolPane, ToolBar, BaseControls) {

    'use strict';

    // class EditToolPane =====================================================

    /**
     * Represents the main tool pane in OX Document applications shown below
     * the top-level view pane, containing the currently active tool bar, and
     * the 'View' drop-down menu.
     *
     * @constructor
     *
     * @extends ToolPane
     *
     * @param {EditApplication} app
     *  The application containing this tool pane.
     *
     * @param {ToolBarTabCollection} toolBarTabs
     *  The collection of all registered tool bar tabs.
     */
    function EditToolPane(app, toolBarTabs, initOptions) {

        var // self reference
            self = this,

            // the tool bars, mapped by tab identifier
            toolBarsMap = {},

            // the wrapper nodes for the center tool bar area
            centerWrapperNode = null,
            centerAreaNode = null,

            // state of "paneLayoutHandler" (to prevent multiple calls)
            reLayout = false;

        // base constructor ---------------------------------------------------

        ToolPane.call(this, app, Utils.extendOptions({
            position: 'top',
            classes: 'edit-pane standard-design',
            cursorNavigate: true
        }, initOptions));

        // private methods ----------------------------------------------------

        /**
         * Shows or hides the tool bars associated to the specified tab
         * identifier, according to the own visibility of the tool bars if
         * bound to a controller item.
         */
        function toggleToolBars(tabId, state) {
            if (_.isString(tabId)) {
                _.each(toolBarsMap[tabId], function (toolBarInfo) {
                    if (_.isObject(toolBarInfo.shrinkedToolBar)) {
                        toolBarInfo.toolBar.toggle(state && toolBarInfo.visible);

                        if (toolBarInfo.shrinked && toolBarInfo.shrinkedMenu.getComponent() && toolBarInfo.shrinkedMenu.getComponent().hasVisibleGroups()) {
                            toolBarInfo.shrinkedToolBar.toggle(state && toolBarInfo.visible);
                        }else {
                            toolBarInfo.shrinkedToolBar.hide();
                        }

                    } else {
                        toolBarInfo.toolBar.toggle(state && toolBarInfo.visible);
                    }
                });
            }
        }



        /**
         * Updates the layout of tool bars according to available space in the
         * tool pane.
         */
        function paneLayoutHandler() {
            // check if another process/trigger is running this already.
            // cancel in this case, to prevent flashing controls
            if (reLayout) { return; }
            reLayout = true;

            var // pane width
                paneWidth = centerWrapperNode.width(),
                // toolbars width
                toolbarsWidth = centerAreaNode.outerWidth(),
                // overflow size of the center tool bar area
                overflowWidth = toolbarsWidth - paneWidth,
                // all toolbars of the active tab
                toolbars = _.sortBy(toolBarsMap[toolBarTabs.getActiveTabId()], _.property('priority')),
                // shrinkable toolbars
                shrinkableToolbars = _.filter(toolbars, function(bar){ return _.has(bar, 'shrinkedToolBar'); }),
                // get the number of shrinked toolbars
                countShrinked = _.filter(toolbars, function(bar){ return bar.shrinked; }).length,
                // hideable toolbars
                hideableToolbars = _.filter(toolbars, function(bar){ return (bar.hideable === true); }),
                // get the number of hidden toolbars
                countHidden = _.filter(toolbars, function(bar){ return bar.hidden; }).length,
                // stop acting
                forceStop = false;


            // catch new widths
            function resetDimensions(){
                // go on, if the node isn't visible
                if (centerAreaNode.outerWidth() === 0) { return false; }

                // pane width
                paneWidth = centerWrapperNode.width();
                // toolbars width
                toolbarsWidth = centerAreaNode.outerWidth();
                // overflow size of the center tool bar area
                return toolbarsWidth - paneWidth;
            }

            /**
             * add/remove given class to/from toolbar
             *
             * @param {String} addClass
             *  class which should be add/remove
             *
             * @param {Boolean} visible
             *  whether add or remove the classes
             */
            function toolbarClass(addClass, visible){
                var node = app.getView().getToolPane().getNode();
                    node.toggleClass(addClass, visible);
            }

            /**
             * add/remove class "vertical-list" to/from popup
             *
             * @param {Integer} i
             *  which of the shrinkable toolbars
             *
             * @param {Boolean} state
             *  whether add or remove the classes
             */
            function makeVerticalList(i, state){
                var popupWrapper = shrinkableToolbars[i].shrinkedMenu.getNode();
                    popupWrapper.toggleClass('vertical-list drop-down', state);
            }

            /**
             * moves the toolbar from pane to drop-down or reversed
             *
             * @param {Integer} i
             *  which of the shrinkable toolbars
             *
             * @param {String} type
             *  whether shrink or unshrink = move to default pane or to small drop-down
             */
            function moveToolBar(i, type){
                if (type === 'shrink' && self.hasViewComponent(shrinkableToolbars[i].toolBar)) {
                    // remove component from pane
                    self.removeViewComponent(shrinkableToolbars[i].toolBar);
                    // add component to shrinked (drop-down) menu
                    shrinkableToolbars[i].shrinkedMenu.addComponent(shrinkableToolbars[i].toolBar);

                    // go through all groups
                    shrinkableToolbars[i].toolBar.iterateGroups(function(group){
                        // and, if exists a menu
                        if (_.has(group, 'getMenu')) {
                            // add the components to the focusable nodes (to prevent closing the popup)
                            shrinkableToolbars[i].shrinkedMenu.registerFocusableNodes(group.getMenu().getContentNode());
                        }
                        // show Controls, which should only be shown in dropdown-version
                        group.dropDownVisible(true);
                    });

                    // maximize all groups
                    shrinkableToolbars[i].toolBar.iterateGroups(function(group){
                        group.deactivateSmallVersion();
                    });

                    // add class for vertical design
                    makeVerticalList(i, true);
                    return true;



                } else if(type === 'unshrink' && !self.hasViewComponent(shrinkableToolbars[i].toolBar)) {
                    // add Component to default pane
                    self.addViewComponent(shrinkableToolbars[i].shrinkedMenu.getComponent(), { insertBefore: shrinkableToolbars[i].shrinkedToolBar });
                    // remove component from shrinked drop-down menu
                    shrinkableToolbars[i].shrinkedMenu.removeComponent(false);

                    // minify all groups
                    Utils.iterateArray(shrinkableToolbars, function(toolbar){
                        if (toolbar.shrinked === false) {
                            toolbar.toolBar.iterateGroups(function(group){
                                group.activateSmallVersion();
                                // hide Controls, which should only be shown in dropdown-version
                                group.dropDownVisible(false);
                            });
                        }
                    });

                    // remove vertical-design
                    makeVerticalList(i, false);
                    return true;

                } else {
                    return false;
                }
            }


            app.getView().lockPaneLayout(function(){
                // returns whether the toolbar(s) are too big for the pane
                function tooBig(){
                    overflowWidth = resetDimensions();
                    return (overflowWidth > 0);
                }
                // returns whether the toolbar(s) fit in the pane
                function fitIn(){
                    overflowWidth = resetDimensions();
                    return (overflowWidth < 0);
                }
                // returns the sum of all shrinked toolbars of this pane
                function getCountShrinked(){
                    countShrinked = _.filter(toolBarsMap[toolBarTabs.getActiveTabId()], function(bar){ return bar.shrinked; }).length;
                    return countShrinked;
                }
                // returns the sum of all hidden toolbars of this pane
                function getCountHidden(){
                    countHidden = _.filter(toolBarsMap[toolBarTabs.getActiveTabId()], function(bar){ return bar.hidden; }).length;
                    return countHidden;
                }

                // Controls not fit in the pane
                if (overflowWidth > 0) {
                    // shrink the space between the controls
                    toolbarClass('small-distance', true);

                    // activate small versions of groups
                    if ( tooBig() ) {
                        Utils.iterateArray(toolbars, function(toolbar){
                            toolbar.toolBar.iterateGroups(function(group){
                                group.activateSmallVersion();
                            });
                            if (fitIn()) { return Utils.BREAK; }
                        }, {reverse: true});

                        // shrink complete toolbar to dropdown-menu
                        if (tooBig()) {
                            // iterate over all toolbars (reversed)
                            Utils.iterateArray(shrinkableToolbars, function(toolbar, i){
                                if (toolbar.shrinked) { return; }

                                if (i===(shrinkableToolbars.length-1) || (i < (shrinkableToolbars.length-1) && shrinkableToolbars[(i+1)].shrinked)) {
                                    if (moveToolBar(i, 'shrink')) {
                                        toolbar.shrinked = true;
                                        if (fitIn()) { return Utils.BREAK; }
                                    }
                                }
                            }, {reverse: true});

                            // hide toolbars (last step to fit in the given area)
                            if (tooBig()) {
                                // iterate backwards through all hideable toolbars
                                Utils.iterateArray(hideableToolbars, function(toolbar, i){
                                    if (toolbar.hidden) { return; }

                                    if (i===(hideableToolbars.length-1) || (i < (hideableToolbars.length-1) && hideableToolbars[(i+1)].hidden)) {
                                        var toolBarNode = (_.has(toolbar, 'shrinkedToolBar'))? toolbar.shrinkedToolBar.getNode():toolbar.toolBar.getNode();
                                            toolBarNode.hide();

                                        toolbar.hidden = true;
                                        if (fitIn()) { return Utils.BREAK; }
                                    }
                                }, {reverse: true});
                            }
                        }
                    }


                // Controls have free space in the pane
                } else {

                    // show hidden toolbars
                    Utils.iterateArray(hideableToolbars, function(toolbar, i){
                        if (!toolbar.hidden) { return; }

                        if (i===0 || (i > 0 && !hideableToolbars[(i-1)].hidden)) {
                            var toolbarNode = (_.has(toolbar, 'shrinkedToolBar'))?toolbar.shrinkedToolBar.getNode():toolbar.toolBar.getNode();

                            toolbarNode.css('display','');
                            toolbar.hidden = false;

                            // if it was too much, revert to last state
                            if (tooBig()) {
                                toolbarNode.hide();
                                toolbar.hidden = true;
                                forceStop = true;
                                return Utils.BREAK;
                            }
                        }
                    });

                    if (fitIn() && forceStop === false && getCountHidden() === 0) {

                        // unshrink toolbars
                        Utils.iterateArray(shrinkableToolbars, function(toolbar, i){
                            if (!toolbar.shrinked) { return; }

                            // toolbar have to be shrinked
                            // and the toolbar before have to be unshrinked (or do not exists "===0")
                            if (i===0 || (i > 0 && !shrinkableToolbars[(i-1)].shrinked)) {
                                if (moveToolBar(i, 'unshrink')) {
                                    toolbar.shrinked = false;
                                    overflowWidth = resetDimensions();
                                    forceStop = true;

                                    // if "overflow" minus "shrinked-toolbar" positive
                                    // reshrink the toolbar
                                    if ((overflowWidth - toolbar.shrinkedToolBar.getNode().outerWidth()) > 0) {
                                        if (moveToolBar(i, 'shrink')) {
                                            toolbar.shrinked = true;
                                        }
                                        overflowWidth = resetDimensions();
                                        return Utils.BREAK;
                                    }
                                }
                            }
                        });

                        // maximize groups
                        if (fitIn() && forceStop === false && getCountShrinked() === 0) {
                            // iterate through all groups of all toolbars
                            Utils.iterateArray(toolbars, function(toolbar){
                                toolbar.toolBar.iterateGroups(function(group){
                                    group.deactivateSmallVersion();
                                });

                                // if it was too much, revert to last state
                                if (tooBig()) {
                                    toolbar.toolBar.iterateGroups(function(group){
                                        group.activateSmallVersion();
                                    });
                                    forceStop = true;
                                    return Utils.BREAK;
                                }
                            });


                            // if the overflow now is present
                            if (fitIn() && forceStop === false) {
                                // reset the spaces between the controls to default padding (13px)
                                toolbarClass('small-distance', false);

                                // if it was too much, revert to last state
                                if (tooBig()) {
                                    // shrink the space again
                                    toolbarClass('small-distance', true);
                                }
                            }
                        }
                    }
                }
            });

            // check visibility of toolbars
            toggleToolBars(toolBarTabs.getActiveTabId(), true);

            // reset status (actually not in layout-progess)
            reLayout = false;
        }


        /**
         * Updates the visibility of the tool bars according to the active tab.
         */
        function tabActivateHandler(event, newTabId, oldTabId) {
            app.getView().lockPaneLayout(function () {
                toggleToolBars(oldTabId, false);
                if (_.isString(newTabId)) { self.show(); }
                toggleToolBars(newTabId, true);
            });
        }

        /**
         * Updates the visibility of the active tool bars that are bound to the
         * enabled state of a controller item.
         */
        function controllerChangeHandler(event, changedItems) {
            _.each(toolBarsMap, function (toolBarInfos) {
                _.each(toolBarInfos, function (toolBarInfo) {
                    var key = toolBarInfo.visibleKey,
                        enabled = (_.isString(key) && (key in changedItems)) ? changedItems[key].enabled : null;
                    if (_.isBoolean(enabled)) {
                        toolBarInfo.visible = enabled;
                    }
                });
            });
            toggleToolBars(toolBarTabs.getActiveTabId(), true);
        }

        // methods ------------------------------------------------------------

        /**
         * Creates a new tool bar in this tool pane.
         *
         * @param {String} tabId
         *  The identifier of the tab button the tool bar will be associated
         *  with.
         *
         * @param {Object} [options]
         *  Optional parameters. Supports all options supported by the ToolBar
         *  class constructor. Additionally, the following options are
         *  supported:
         *  @param {String} [options.visibleKey]
         *      The key of the controller item that controls the visibility of
         *      the tool bar while it is active according to the specified tab
         *      identifier. The visibility will be bound to the 'enabled' state
         *      of the respective controller item. If omitted, the tool bar
         *      will always be visible.
         *  @param {Boolean} [options.withShrink=false]
         *      Prepare a (smaller) dropdown menu where all the controls will
         *      be moved to, if there isn't enough free space for the normal
         *      toolbar.
         *      Special logic for mobile-ui.
         *
         * @returns {ToolBar}
         *  The new tool bar instance.
         */
        this.createToolBar = function (tabId, options) {

            var // the new tool bar instance
                toolBar = new ToolBar(app, options),
                // create shrinked toolbar
                withShrink = Utils.getBooleanOption(options, 'prepareShrink', false),
                // shrinked menu
                smenu = null,
                // shrinked object (includes the toolbar and the menu)
                shrinkObj = {},
                // use SplitButton
                splitBtn = Utils.getOption(options, 'splitBtn', false),
                // controller key for split-button
                controllerKey = null;

            // prepare shrinked toolbar
            if (withShrink) {
                // without split button in toolbar
                if (splitBtn === false) {
                    smenu = new BaseControls.ComponentMenuButton(app, {
                        tooltip:    Utils.getStringOption(options, 'tooltip', ''),
                        label:      Utils.getStringOption(options, 'label', ''),
                        icon:       Utils.getStringOption(options, 'icon', ''),
                        classes:    Utils.getStringOption(options, 'classes', '')
                    });

                // with split button in toolbar
                } else {
                    smenu = new BaseControls.ComponentMenuSplitButton(app, Utils.extendOptions({
                        tooltip:        Utils.getStringOption(options, 'tooltip', ''),
                        caretTooltip:   Utils.getStringOption(options, 'caretTooltip', ''),
                        label:          Utils.getStringOption(options, 'label', ''),
                        icon:           Utils.getStringOption(options, 'icon', ''),
                        classes:        Utils.getStringOption(options, 'classes', '')
                    }, Utils.getOption(options.splitBtn, 'options')));

                    controllerKey = Utils.getStringOption(options.splitBtn, 'key');
                }

                shrinkObj.shrinkedMenu = smenu.getMenu();
                shrinkObj.shrinkedToolBar = new ToolBar(app).addGroup(controllerKey, smenu).hide();

                this.addViewComponent(shrinkObj.shrinkedToolBar);
            }

            // set to hidden state initially
            toolBar.hide();

            // insert the tool bar into this pane
            (toolBarsMap[tabId] || (toolBarsMap[tabId] = [])).push(_.extend({
                toolBar: toolBar,                                                   // normal toolbar
                visibleKey: Utils.getStringOption(options, 'visibleKey'),           // toolbar visible at
                visible: true,                                                      // toolbar visibility
                priority: Utils.getIntegerOption(options, 'priority', null),        // toolbar priority (1 = high prio, 10 = low)
                hideable: Utils.getBooleanOption(options, 'hideable', null),        // is toolbar hideable
                hidden: false,                                                      // is toolbar hidden
                shrinked: false                                                     // is toolbar shrinked
            }, shrinkObj));
            this.addViewComponent(toolBar);

            return toolBar;
        };

        // initialization -----------------------------------------------------

        // update layout of tool bars according to available space
        this.on('pane:layout', paneLayoutHandler);

        // update visibility of the tool bars
        this.listenTo(toolBarTabs, 'tab:activate', tabActivateHandler);
        this.listenTo(app.getController(), 'change:items', controllerChangeHandler);

        // activate a tool bar tab, if the tool pane becomes visible
        this.on('pane:show', function () {
            if (!toolBarTabs.getActiveTabId()) {
                toolBarTabs.activateFirstTab();
            }
        });

        // get the container nodes for the center tool bar area
        centerWrapperNode = this.getAreaWrapperNode('center');
        centerAreaNode = centerWrapperNode.find('>.area-container');

        // destroy all class members on destruction
        this.registerDestructor(function () {
            app = toolBarTabs = self = toolBarsMap = centerWrapperNode = centerAreaNode = null;
        });

    } // class ToolPane

    // exports ================================================================

    // derive this class from class ToolPane
    return ToolPane.extend({ constructor: EditToolPane });

});
