/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Mario Schroeder <mario.schroeder@open-xchange.com>
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/editframework/view/clipboardpane',
    ['io.ox/office/tk/utils',
     'io.ox/office/tk/keycodes',
     'io.ox/office/baseframework/view/pane'
    ], function (Utils, KeyCodes, Pane) {

    'use strict';

    // class ClipboardPane ====================================================

    /**
     * A view pane attached to the bottom border of the application view that
     * displays the HTML mark-up contained in the browser clipboard.
     *
     * @constructor
     *
     * @extends Pane
     */
    function ClipboardPane(app) {

        var // self reference
            self = this,

            // the DOM node containing the clipboard data
            contentNode = null;

        // base constructor ---------------------------------------------------

        Pane.call(this, app, { position: 'bottom', classes: 'clipboard-pane', enableContextMenu: true });

        // private methods ----------------------------------------------------

        function createOperationsMarkup(operations) {

            var markup = '<table><tbody><tr><th></th><th>Name</th><th>Parameters</th>';

            // add operations
            _.each(operations, function (operation, index) {
                var name = Utils.escapeHTML(operation.name || '');
                operation = _.clone(operation);
                delete operation.name;
                delete operation.osn;
                delete operation.opl;
                operation = JSON.stringify(operation).replace(/^\{(.*)\}$/, '$1').replace(/"(\w+)":/g, '$1:').replace(/ /g, '\xb7');
                markup += '<tr><td>' + index + '</td><td>' + name + '</td><td>' + Utils.escapeHTML(operation) + '</td>';
            });

            markup += '</tbody></table>';
            return markup;
        }

        // methods ------------------------------------------------------------

        this.setClipboardContent = function (content) {

            var // the mark-up to be inserted into the clipboard node
                markup = null;

            if ((_.isArray(content)) && (content.length > 0)) {
                markup = createOperationsMarkup(content);
            } else if ((content instanceof $) && (content.contents().length > 0)) {
                markup = '<span>' + Utils.escapeHTML(content.html()) + '</span>';
            } else if ((_.isString(content)) && (content.length > 0)) {
                markup = '<span>' + Utils.escapeHTML(content) + '</span>';
            } else {
                markup = '<span>Clipboard contains no data</span>';
            }

            // clear old and set new clipboard content
            contentNode[0].innerHTML = markup;

            return this;
        };

        // initialization -----------------------------------------------------

        // initialize and insert the clipboard data node
        contentNode = $('<div tabindex="1"><span>Clipboard debug window</span></div>');
        this.getNode().append(contentNode);

        // keyboard shortcuts
        contentNode.on('keydown', function (event) {
            if (KeyCodes.matchKeyCode(event, 'A', { ctrlOrMeta: true })) {
                var docRange = window.document.createRange();
                docRange.setStart(event.delegateTarget, 0);
                docRange.setEnd(event.delegateTarget, 1);
                window.getSelection().removeAllRanges();
                window.getSelection().addRange(docRange);
                return false;
            }
            if (KeyCodes.matchKeyCode(event, 'DELETE')) {
                contentNode.empty();
                return false;
            }
            if (event.keyCode === KeyCodes.ESCAPE) {
                app.getView().grabFocus();
                return false;
            }
        });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            contentNode.remove();
            self = contentNode = null;
        });

    } // class ClipboardPane

    // exports ================================================================

    // derive this class from class Pane
    return Pane.extend({ constructor: ClipboardPane });

});
