/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Carsten Driesner <carsten.driesner@open-xchange.com>
 */

define('io.ox/office/editframework/utils/color', ['io.ox/office/tk/utils'], function (Utils) {

    'use strict';

    var // maps the lower-case names of all preset colors to the real mixed-case names and RGB color values
        PRESET_COLORS = (function () {
            var map = {};
            _.each({
                aliceBlue: 'f0f8ff',         antiqueWhite: 'faebd7',      aqua: '00ffff',                  aquamarine: '7fffd4',
                azure: 'f0ffff',             beige: 'f5f5dc',             bisque: 'ffe4c4',                black: '000000',
                blanchedAlmond: 'ffebcd',    blue: '0000ff',              blueViolet: '8a2be2',            brown: 'a52a2a',
                burlyWood: 'deb887',         cadetBlue: '5f9ea0',         chartreuse: '7fff00',            chocolate: 'd2691e',
                coral: 'ff7f50',             cornflowerBlue: '6495ed',    cornsilk: 'fff8dc',              crimson: 'dc143c',
                cyan: '00ffff',              darkBlue: '00008b',          darkCyan: '008b8b',              darkGoldenrod: 'b8860b',
                darkGray: 'a9a9a9',          darkGreen: '006400',         darkKhaki: 'bdb76b',             darkMagenta: '8b008b',
                darkOliveGreen: '556b2f',    darkOrange: 'ff8c00',        darkOrchid: '9932cc',            darkRed: '8b0000',
                darkSalmon: 'e9967a',        darkSeaGreen: '8fbc8f',      darkSlateBlue: '483d8b',         darkSlateGray: '2f4f4f',
                darkTurquoise: '00ced1',     darkViolet: '9400d3',        deepPink: 'ff1493',              deepSkyBlue: '00bfff',
                dimGray: '696969',           dodgerBlue: '1e90ff',        firebrick: 'b22222',             floralWhite: 'fffaf0',
                forestGreen: '228b22',       fuchsia: 'ff00ff',           gainsboro: 'dcdcdc',             ghostWhite: 'f8f8ff',
                gold: 'ffd700',              goldenrod: 'daa520',         gray: '808080',                  green: '008000',
                greenYellow: 'adff2f',       honeydew: 'f0fff0',          hotPink: 'ff69b4',               indianRed : 'cd5c5c',
                indigo: '4b0082',            ivory: 'fffff0',             khaki: 'f0e68c',                 lavender: 'e6e6fa',
                lavenderBlush: 'fff0f5',     lawnGreen: '7cfc00',         lemonChiffon: 'fffacd',          lightBlue: 'add8e6',
                lightCoral: 'f08080',        lightCyan: 'e0ffff',         lightGoldenrodYellow: 'fafad2',  lightGray: 'd3d3d3',
                lightGreen: '90ee90',        lightPink: 'ffb6c1',         lightSalmon: 'ffa07a',           lightSeagreen: '20b2aa',
                lightSkyBlue: '87cefa',      lightSlateGray: '778899',    lightSteelBlue: 'b0c4de',        lightYellow: 'ffffe0',
                lime: '00ff00',              limeGreen: '32cd32',         linen: 'faf0e6',                 magenta: 'ff00ff',
                maroon: '800000',            mediumAquamarine: '66cdaa',  mediumBlue: '0000cd',            mediumOrchid: 'ba55d3',
                mediumPurple: '9370d8',      mediumSeagreen: '3cb371',    mediumSlateBlue: '7b68ee',       mediumSpringGreen: '00fa9a',
                mediumTurquoise: '48d1cc',   mediumVioletred: 'c71585',   midnightBlue: '191970',          mintCream: 'f5fffa',
                mistyRose: 'ffe4e1',         moccasin: 'ffe4b5',          navajoWhite: 'ffdead',           navy: '000080',
                oldLace: 'fdf5e6',           olive: '808000',             oliveDrab: '6b8e23',             orange: 'ffa500',
                orangeRed: 'ff4500',         orchid: 'da70d6',            paleGoldenrod: 'eee8aa',         paleGreen: '98fb98',
                paleTurquoise: 'afeeee',     paleVioletRed: 'd87093',     papayaWhip: 'ffefd5',            peachPuff: 'ffdab9',
                peru: 'cd853f',              pink: 'ffc0cb',              plum: 'dda0dd',                  powderBlue: 'b0e0e6',
                purple: '800080',            red: 'ff0000',               rosyBrown: 'bc8f8f',             royalBlue: '4169e1',
                saddleBrown: '8b4513',       salmon: 'fa8072',            sandyBrown: 'f4a460',            seaGreen: '2e8b57',
                seaShell: 'fff5ee',          sienna: 'a0522d',            silver: 'c0c0c0',                skyBlue: '87ceeb',
                slateBlue: '6a5acd',         slateGray: '708090',         snow: 'fffafa',                  springGreen: '00ff7f',
                steelBlue: '4682b4',         tan: 'd2b48c',               teal: '008080',                  thistle: 'd8bfd8',
                tomato: 'ff6347',            turquoise: '40e0d0',         violet: 'ee82ee',                wheat: 'f5deb3',
                white: 'ffffff',             whiteSmoke: 'f5f5f5',        yellow: 'ffff00',                yellowGreen: '9acd32'
            }, function (value, key) {
                map[key.toLowerCase()] = { name: key, rgb: value };
            });
            return map;
        })(),

        SYSTEM_COLORS = (function () {
            var map = {};
            _.each([
                'ActiveBorder', 'ActiveCaption', 'AppWorkspace', 'Background',
                'ButtonFace', 'ButtonHighlight', 'ButtonShadow',
                'ButtonText', 'CaptionText', 'GrayText', 'Highlight',
                'HighlightText', 'InactiveBorder', 'InactiveCaption',
                'InactiveCaptionText', 'InfoBackground', 'InfoText',
                'Menu', 'MenuText', 'Scrollbar', 'threeDDarkShadow',
                'ThreeDFace', 'ThreeDHighlight', 'ThreeDLightShadow',
                'ThreeDShadow', 'Window', 'WindowFrame', 'WindowText'
            ], function (key) {
                var name = key.substring(0, 1).toLowerCase() + key.substring(1);
                map[key.toLowerCase()] = {
                    name: name
                };
            });
            return map;
        }()),

        // representation of color black in CRGB color model
        CRGB_BLACK = { r: 0, g: 0, b: 0, a: 1 },

        // representation of full transparency in CRGB color model
        CRGB_TRANSPARENT = { r: 0, g: 0, b: 0, a: 0 },

        TRANSFORM_HANDLERS = {
            /**
             * Darkens the passed CRGB color value with the provided shade value.
             *
             * @param {Object} crgb
             *  The CRGB color value.
             *
             * @param {String} shade
             *  An integer value in the interval [0, 100000] to darken the provided
             *  color. A value of 0 will change the color to black, a value of 100000
             *  will not modify the color, values in between will darken the color
             *  accordingly.
             *
             * @returns {Object}
             *  The darkened CRGB color Object.
             */
            shade: function (crgb, shade) {
                var hsl = convertCRGBToHSL(crgb);
                shade = Utils.minMax(shade / 100000, 0, 1);
                hsl.l *= shade;
                return convertHSLToCRGB(hsl);
            },
            /**
             * Lightens the passed CRGB color value with the provided tint value.
             *
             * @param {Object} crgb
             *  The CRGB color value.
             *
             * @param {Number} tint
             *  An integer value in the interval [0, 100000] to lighten the provided
             *  color. A value of 0 will change the color to white, a value of 100000
             *  will not modify the color, values in between will lighten the color
             *  accordingly.
             *
             * @returns {Object}
             *  The lightened CRGB color Object.
             */
            tint: function (crgb, tint) {
                var hsl = convertCRGBToHSL(crgb);
                tint = Utils.minMax(tint / 100000, 0, 1);
                hsl.l = (hsl.l - 1) * tint + 1;
                return convertHSLToCRGB(hsl);
            },
            /**
             * Shift the passed CRGB Luminance with the provided lumOff value.
             *
             * @param {Object} crgb
             *  The CRGB color value.
             *
             * @param {Number} lumOff
             *  An integer value in the interval [-100000, 100000] to shift the provided
             *  luminance. The value 0 leaves the channel unmodified.
             *
             * @returns {Object}
             *  The CRGB color Object value restricted between 0 and 1
             */
            lumOff: function (crgb, lumOff) {
                var hsl = convertCRGBToHSL(crgb);
                hsl.l = Utils.minMax(hsl.l + lumOff / 100000, 0, 1);
                return convertHSLToCRGB(hsl);
            },
            /**
             * Modulates the passed CRGB luminance with the provided lumMod value.
             *
             * @param {Object} crgb
             *  The CRGB color value.
             *
             * @param {Number} lumMod
             *  An integer value in the interval [0, *] to modulate the provided
             *  luminance. The value 100000 leaves the channel unmodified.
             *
             * @returns {Object}
             *  The CRGB color Object value restricted between 0 and 1
             */
            lumMod: function (crgb, lumMod) {
                var hsl = convertCRGBToHSL(crgb);
                hsl.l = Utils.minMax(hsl.l * lumMod / 100000, 0, 1);
                return convertHSLToCRGB(hsl);
            },

            /**
             * Modulates the passed CRGB Saturation with the provided satMod value.
             *
             * @param {Object} crgb
             *  The CRGB color value.
             *
             * @param {Number} satMod
             *  An integer value in the interval [0, *] to modulate the provided
             *  saturation. The value 0 leaves the channel unmodified.
             *
             * @returns {Object}
             *  The CRGB color Object value restricted between 0 and 1
             */
            satMod: function (crgb, satMod) {
                var hsl = convertCRGBToHSL(crgb);
                hsl.s = Utils.minMax(hsl.s * satMod / 100000, 0, 1);
                return convertHSLToCRGB(hsl);
            },

            /**
             * Set the passed CRGB Alpha with the provided alpha value.
             *
             * @param {Object} crgb
             *  The CRGB color value.
             *
             * @param {Number} alpha
             *  An integer value in the interval [0, 100000] to set the provided
             *  alpha.
             *
             * @returns {Object}
             *  The CRGB color Object value restricted between 0 and 1
             */
            alpha: function (crgb, alpha) {
                crgb.a = Utils.minMax(alpha / 100000, 0, 1);
                return crgb;
            },

            /**
             * Modulates the alpha channel of the passed CRGB color.
             *
             * @param {Object} crgb
             *  The CRGB color value.
             *
             * @param {Number} alphaMod
             *  An integer value to modulate the alpha channel.
             *
             * @returns {Object}
             *  The CRGB color object.
             */
            alphaMod: function (crgb, alphaMod) {
                crgb.a = Utils.mixMax(crgb.a * alphaMod / 100000, 0, 1);
                return crgb;
            }
        };

    // private static functions ===============================================

    /**
     * Decreases the gamma correction of the passed CRGB channel value.
     *
     * @param {Number} channel
     *  The value of a color channel (CRGB color model), between 0 and 1.
     *
     * @returns {Number}
     *  The new value of the color channel with applied gamma correction.
     */
    function gamma(channel) {
        return Math.pow(channel, 1 / 2.3);
    }

    /**
     * Increases the gamma correction of the passed CRGB channel value.
     *
     * @param {Number} channel
     *  The value of a color channel (CRGB color model), between 0 and 1.
     *
     * @returns {Number}
     *  The new value of the color channel with applied gamma correction.
     */
    function invGamma(channel) {
        return Math.pow(channel, 2.3);
    }

    /**
     * Converts a hexadecimal RGB color value to a CRGB color descriptor.
     *
     * @param {String} rgb
     *  The hexadecimal RGB color (6 digits).
     *
     * @returns {Object}
     *  The CRGB color object with applied gamma correction, in the following
     *  properties:
     *  - {Number} r
     *      The red channel (0 to 1).
     *  - {Number} g
     *      The green channel (0 to 1).
     *  - {Number} b
     *      The blue channel (0 to 1).
     *  - {Number} a
     *      The alpha channel (transparency, set to 1 for full opaqueness).
     */
    function convertRGBToCRGB(rgb) {

        // convert passed string to number
        rgb = parseInt(rgb, 16);

        // return the CRGB color descriptor with gamma correction
        return {
            r: invGamma((rgb & 0xFF0000) / 0xFF0000),
            g: invGamma((rgb & 0xFF00) / 0xFF00),
            b: invGamma((rgb & 0xFF) / 0xFF),
            a: 1
        };
    }

    /**
     * Converts a CRGB color object to a HSL color object.
     *
     * @param {Object} crgb
     *  The CRGB color object to be converted to the HSL color model, with the
     *  properties 'r', 'g', 'b', and 'a' (each from 0 to 1).
     *
     * @returns {Object}
     *  The HSL color object, with the following properties:
     *  - {Number} h
     *      The color hue value (0 to 1).
     *  - {Number} s
     *      The color saturation (0 to 1).
     *  - {Number} l
     *      The color luminance (0 to 1).
     *  - {Number} a
     *      The alpha channel (0 to 1).
     */
    function convertCRGBToHSL(crgb) {

        var min = Math.min(crgb.r, crgb.g, crgb.b),
            max = Math.max(crgb.r, crgb.g, crgb.b),
            l = (min + max) / 2,
            s = (min === max) ? 0 : (l < 0.5) ? (max - min) / (max + min) : (max - min) / (2.0 - max - min),
            h = 0;

        if (s !== 0) {
            h = (max === crgb.r) ? ((crgb.g - crgb.b) / (max - min)) : (max === crgb.g) ? (2.0 + (crgb.b - crgb.r) / (max - min)) : (4.0 + (crgb.r - crgb.g) / (max - min));
            h = ((h < 0) ? (h + 6) : h) / 6;
        }

        return { h: h, s: s, l: l, a: crgb.a };
    }

    /**
     * Converts a HSL color object to a CRGB color object.
     *
     * @param {Object} hsl
     *  The HSL color object to be converted to the CRGB color model, with the
     *  properties 'h', 's', 'l', and 'a' (each from 0 to 1).
     *
     * @returns {Object}
     *  The CRGB color object (see method convertRGBToCRGB() for details).
     */
    function convertHSLToCRGB(hsl) {

        var r, g, b;

        if (hsl.s === 0) {
            r = g = b = hsl.l;
        } else {

            var t1 = (hsl.l < 0.5) ? (hsl.l * (1 + hsl.s)) : (hsl.l + (1 - hsl.l) * hsl.s),
                t2 = 2 * hsl.l - t1;

            var tr = hsl.h + 1 / 3;
            tr = (tr < 0) ? (tr + 1) : (tr > 1) ? (tr - 1) : tr;

            var tg = hsl.h;
            tg = (tg < 0) ? (tg + 1) : (tg > 1) ? (tg - 1) : tg;

            var tb = hsl.h - 1 / 3;
            tb = (tb < 0) ? (tb + 1) : (tb > 1) ? (tb - 1) : tb;

            var convChannel = function conv(ch) {
                if ((6 * ch) < 1)
                    return t2 + (t1 - t2) * 6 * ch;
                else if ((2 * ch) < 1)
                    return t1;
                else if ((3 * ch) < 2)
                    return t2 + (t1 - t2) * ((2 / 3) - ch) * 6;
                else
                    return t2;
            };

            r = convChannel(tr);
            g = convChannel(tg);
            b = convChannel(tb);
        }

        return { r: r, g: g, b: b, a: hsl.a };
    }

    /**
     * Applies the passed color transformations to a CRGB color object.
     *
     * @param {Object} crgb
     *  The CRGB color object to be transformed, with the properties 'r', 'g',
     *  'b', and 'a' (each from 0 to 1).
     *
     * @param {Array} [transformations]
     *  The color transformations.
     *
     * @returns {Object}
     *  The resulting CRGB color object.
     */
    function transformCRGBColor(crgb, transformations) {
        crgb = _.clone(crgb);
        _.each(transformations, function (transformation) {
            var tHandler = TRANSFORM_HANDLERS[transformation.type];
            if (!tHandler) {
                Utils.warn('Color.transformCRGBColor(): unknown color transformation: ' + transformation.type);
            } else {
                crgb = tHandler(crgb, transformation.value);
            }
        });
        return crgb;
    }

    /**
     * Converts SystemColor Name to its rgb color value
     *
     * @param {String} sysColor as name
     *
     * @returns {String}
     * the correct rgb hex string.
     */
    function convertSysColorToRGB(sysColor) {
        var lower = sysColor.toLowerCase();
        var entry = SYSTEM_COLORS[lower];
        if (entry) {
            if (!entry.rgb) {
                var div = $('div');
                div.css('color', lower);
                entry.rgb = div.css('color');
            }
            return entry.rgb;
        }
        Utils.warn('System Color "' + sysColor + '" is not defined in Color.SYSTEM_COLORS');
        return '9E9E9E';
    }

    /**
     * Converts the passed color attribute object to a CRGB color object.
     *
     * @param {Object} color
     *  The color object as used in operations.
     *
     * @param {String|Object} autoColor
     *  Additional information needed to resolve the automatic color. Can be a
     *  string for predefined automatic colors, or an explicit color object
     *  with a value other than the automatic color. Predefined identifiers are
     *  - 'text': for text colors (mapped to to black),
     *  - 'line': for line colors (e.g. borders, mapped to black),
     *  - 'fill': for fill colors (e.g. table cells, mapped to transparent).
     *
     * @param {Theme} theme
     *  The theme object used to map scheme color names to color values.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Number} [options.alphaMod=1]
     *      If specified, a factor for the opacity that will be applied to the
     *      resulting ARGB color, regardless of the type of the original color.
     *
     * @returns {Object}
     *  The CRGB color object, with the following properties:
     *  - {Number} r
     *      The value of the red channel (CRGB color model, 0 to 1).
     *  - {Number} g
     *      The value of the green channel (CRGB color model, 0 to 1).
     *  - {Number} b
     *      The value of the blue channel (CRGB color model, 0 to 1).
     *  - {Number} a
     *      The value of the alpha channel (0 to 1).
     */
    function resolveToCRGBColor(color, autoColor, theme, options) {

        var // the color type (specifies the meaning of the color value)
            type = Utils.getStringOption(color, 'type', 'auto'),
            // the transformation array
            transformations = Utils.getArrayOption(color, 'transformations'),
            // the resulting CRGB color object
            crgb = null;

        // resolve automatic color with explicitly passed 'autoColor'
        if ((type === 'auto') && _.isObject(autoColor)) {
            if (_.isArray(transformations)) {
                autoColor = _.clone(autoColor);
                autoColor.transformations = _.isArray(autoColor.transformations) ? (autoColor.transformations.concat(transformations)) : transformations;
            }
            return resolveToCRGBColor(autoColor, 'fill', theme, options);
        }

        switch (type) {

        // predefined system UI colors without fixed RGB values
        case 'system':
            crgb = transformCRGBColor(convertRGBToCRGB(convertSysColorToRGB(color.value)), transformations);
            break;

        // predefined colors with fixed RGB values
        case 'preset':
            var presetColor = PRESET_COLORS[color.value.toLowerCase()];
            crgb = transformCRGBColor(presetColor ? convertRGBToCRGB(presetColor.rgb) : CRGB_BLACK, transformations);
            break;

        // CRGB color model (each channel from 0 to 100000 with gamma correction)
        case 'crgb':
            crgb = { r: Math.minMax(color.value.r / 100000, 0, 1), g: Math.minMax(color.value.g / 100000, 0, 1), b: Math.minMax(color.value.b / 100000, 0, 1), a: 1 };
            crgb = transformCRGBColor(crgb, transformations);
            break;

        // HSL color model (hue, saturation, luminance)
        case 'hsl':
            var hsl = { h: Math.minMax(color.value.h / 360000, 0, 1), s: Math.minMax(color.value.s / 100000, 0, 1), l: Math.minMax(color.value.l / 100000, 0, 1), a: 1 };
            crgb = transformCRGBColor(convertHSLToCRGB(hsl), transformations);
            break;

        // simple hexadecimal RGB (without gamma correction)
        case 'rgb':
            crgb = transformCRGBColor(convertRGBToCRGB(color.value), transformations);
            break;

        // scheme color
        case 'scheme':
            if (theme) {
                if (color.value === 'style') {
                    Utils.warn('Color.getColorDetails(): scheme color type "style" not implemented');
                    crgb = transformCRGBColor(CRGB_BLACK, transformations);
                } else {
                    crgb = transformCRGBColor(convertRGBToCRGB(theme.getSchemeColor(color.value)), transformations);
                }
            } else if (_.isString(color.fallbackValue)) {
                crgb = convertRGBToCRGB(color.fallbackValue);
            } else {
                crgb = transformCRGBColor(CRGB_BLACK, transformations);
            }
            break;

        // automatic color, effective color depends on passed autoColor parameter
        case 'auto':
            // resolve automatic color by context type
            switch (autoColor) {
            case 'text':
            case 'line':
                crgb = CRGB_BLACK;
                break;
            case 'fill':
                // transparent: keep colorDesc empty (see below)
                break;
            default:
                Utils.error('Color.getColorDetails(): unknown context type "' + autoColor + '" for auto color');
                crgb = CRGB_BLACK;
            }
            break;

        default:
            crgb = _.isString(color.fallbackValue) ? convertRGBToCRGB(color.fallbackValue) : null;
            Utils.error('Color.getColorDetails(): unknown color type "' + type + '"');
        }

        if (_.isObject(crgb)) {
            crgb = _.clone(crgb);
            // add final modifier for alpha channel
            crgb.a = Utils.minMax(crgb.a * Utils.getNumberOption(options, 'alphaMod', 1, 0), 0, 1);
        } else {
            // descriptor for a fully transparent color
            crgb = _.clone(CRGB_TRANSPARENT);
        }

        return crgb;
    }

    // static class Color =====================================================

    /**
     * Predefined color objects.
     */
    var Color = {
            BLACK: { type: 'rgb', value: '000000' },
            WHITE: { type: 'rgb', value: 'FFFFFF' },
            RED:   { type: 'rgb', value: 'FF0000' },
            AUTO:  { type: 'auto' }
        };

    /**
     * Converts the passed color attribute object to a color descriptor with
     * details about the resulting RGB color.
     *
     * @param {Object} color
     *  The color object as used in operations.
     *
     * @param {String|Object} autoColor
     *  Additional information needed to resolve the automatic color. Can be a
     *  string for predefined automatic colors, or an explicit color object
     *  with a value other than the automatic color. Predefined identifiers are
     *  - 'text': for text colors (mapped to to black),
     *  - 'line': for line colors (e.g. borders, mapped to black),
     *  - 'fill': for fill colors (e.g. table cells, mapped to transparent).
     *
     * @param {Theme} theme
     *  The theme object used to map scheme color names to color values.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Number} [options.alphaMod=1]
     *      If specified, a factor for the opacity that will be applied to the
     *      resulting ARGB color, regardless of the type of the original color.
     *
     * @returns {Object}
     *  The color descriptor constructed from the passed color object, with the
     *  following properties:
     *  - {Number} r
     *      The value of the red channel (CRGB color model, 0 to 1).
     *  - {Number} g
     *      The value of the green channel (CRGB color model, 0 to 1).
     *  - {Number} b
     *      The value of the blue channel (CRGB color model, 0 to 1).
     *  - {Number} h
     *      The value of the color hue (HSL color model, 0 to 1).
     *  - {Number} s
     *      The value of the color saturation (HSL color model, 0 to 1).
     *  - {Number} l
     *      The value of the color liminance (HSL color model, 0 to 1).
     *  - {Number} r8
     *      The value of the red channel (8-bit RGB color model, 0 to 255).
     *  - {Number} g8
     *      The value of the green channel (8-bit RGB color model, 0 to 255).
     *  - {Number} b8
     *      The value of the blue channel (8-bit RGB color model, 0 to 255).
     *  - {Number} y
     *      The luma value (weighted lightness) of the color (0 to 1).
     *  - {Number} a
     *      The value of the alpha channel (0 to 1).
     *  - {String} css
     *      The CSS color representation.
     */
    Color.getColorDetails = function (color, autoColor, theme, options) {

        var // the CRGB color object for the passed color
            colorDesc = resolveToCRGBColor(color, autoColor, theme, options),
            // alpha channel, as integral percentage
            alpha = Math.round(colorDesc.a * 100);

        // add HSL properties to the resulting descriptor
        _.extend(colorDesc, convertCRGBToHSL(colorDesc));

        // convert CRGB channels to 8-bit RGB
        colorDesc.r8 = Math.round(gamma(colorDesc.r) * 255);
        colorDesc.g8 = Math.round(gamma(colorDesc.g) * 255);
        colorDesc.b8 = Math.round(gamma(colorDesc.b) * 255);

        // add luma value
        colorDesc.y = (0.299 * colorDesc.r8 + 0.587 * colorDesc.g8 + 0.114 * colorDesc.b8) / 255;

        // add CSS color representation ('transparent' for full transparence, RGB hex
        // code for opaque colors, or 'rgba' function style for transparent colors)
        colorDesc.css = (alpha === 0) ? 'transparent' : (alpha === 100) ?
            ('#' + ('00000' + (colorDesc.r8 * 0x10000 + colorDesc.g8 * 0x100 + colorDesc.b8).toString(16)).substr(-6)) :
            'rgba(' + colorDesc.r8 + ',' + colorDesc.g8 + ',' + colorDesc.b8 + ',' + (alpha / 100) + ')';

        return colorDesc;
    };

    /**
     * Converts the passed attribute color object to a CSS color value.
     *
     * @param {Object} color
     *  The color object as used in operations.
     *
     * @param {String|Object} autoColor
     *  Additional information needed to resolve the automatic color. See
     *  method Color.getColorDetails() for details about this parameter.
     *
     * @param {Theme} theme
     *  The theme object used to map scheme color names to color values.
     *
     * @param {Object} [options]
     *  Optional parameters. See method Color.getColorDetails() for details
     *  about this parameter.
     *
     * @returns {String}
     *  The CSS color value converted from the passed color object.
     */
    Color.getCssColor = function (color, autoColor, theme, options) {
        return Color.getColorDetails(color, autoColor, theme, options).css;
    };

    /**
     * Determine if provided color has theme based color value
     *
     * @param color
     * A color object
     *
     * @returns {Boolean}
     * true if color is theme based otherwise false
     */
    Color.isThemeColor = function (color) {
        var type = Utils.getStringOption(color, 'type', 'none');
        return (type === 'scheme');
    };

    /**
     * Determine if provided color has type auto
     *
     * @param color
     * A color object
     *
     * @returns {Boolean}
     * true if color has type auto otherwise false
     */
    Color.isAutoColor = function (color) {
        var type = Utils.getStringOption(color, 'type', 'auto');
        return (type === 'auto');
    };

    /**
     * Returns whether the passed color resolves to full transparency.
     *
     * @param {Object} color
     *  The color object.
     *
     * @param {String} context
     *  The context needed to resolve the color type 'auto'.
     *
     * @returns {Boolean}
     *  Whether the passed color represents full transparency.
     */
    Color.isTransparentColor = function (color, context) {
        return Color.getColorDetails(color, context).a === 0;
    };

    /**
     * Determine if the provided color is dark or light
     *
     * @param {String} rgbColor
     * The rgb color as hex value string (RRGGBB) with/without trailing #
     *
     * @returns {Boolean}
     * True if the color is dark otherwise false.
     */
    Color.isDark = function (rgbColor) {
        if (rgbColor && (6 <= rgbColor.length) && (rgbColor.length <= 7)) {
            var hexString = rgbColor.length === 6 ? rgbColor : rgbColor.substring(1);
            var rgbColorValue = parseInt(hexString, 16);
            var luminance = ((((rgbColorValue & 0xFF0000) >> 16) * 5036060) +
                             (((rgbColorValue & 0xFF00) >> 8) * 9886846) +
                              ((rgbColorValue & 0xFF) * 1920103)) / (1 << 24);

            return luminance <= 60;
        }

        return false;
    };

    /**
     * Comparing two color values. These are equal if:
     * - both color parameter are Color objects
     * - both have the same 'type'
     * - both have no property 'value' or both have a property 'value' and the value of 'value' is equal
     * - both have no property 'transformations' or both have a property 'transformations' and the value of 'transformations' is equal
     *
     * @param {Color} color1
     * The first color to compare
     *
     * @param {Color} color2
     * The second color to compare
     *
     * @returns {Boolean}
     * Whether the two colors are equal.
     */
    Color.isEqual = function (color1, color2) {
        return (color1 &&
                color2 &&
                ('type' in color1) &&
                ('type' in color2) &&
                (color1.type === color2.type) &&
                ((!('value' in color1) && (!('value' in color2))) || (('value' in color1) && ('value' in color2) && (color1.value === color2.value))) &&
                ((!('transformations' in color1) && (!('transformations' in color2))) || (('transformations' in color1) && ('transformations' in color2) && _.isEqual(color1.transformations, color2.transformations))));
    };

    /**
     * Converts a CSS color string into a Color object.
     * Accepts the following CSS colors as input:
     *  - predefined color names e.g. 'red'
     *  - hex  colors e.g. #RRGGBB
     *  - rgb  colors e.g. rgb(R, G, B)
     *  - rgba colors e.g. rgba(R, G, B, A)
     * Returns a Color of type auto if a rgba color defines a transparent alpha.
     *
     * @param {String} cssColor
     *  The CSS color string.
     *
     * @returns {Object|null}
     *  A Color object or null.
     */
    Color.convertCssColorToRgbColor = function (cssColor) {

        var parts;

        if (!_.isString(cssColor)) { return null; }

        // CSS keyword 'transparent'
        if (cssColor === 'transparent') { return Color.AUTO; }

        // predefined color names
        var preset = PRESET_COLORS[cssColor.toLowerCase()];
        if (preset) {
            //Presets not supported
            //return { type: 'preset', value: preset.name };
            return { type: 'rgb', value: preset.rgb };
        }

        // 6-digit hex color
        if (/^#[0-9A-F]{6}$/i.test(cssColor)) {
            return { type: 'rgb', value: cssColor.slice(1).toUpperCase() };
        }

        // 3-digit hex color
        if (/^#[0-9A-F]{3}$/i.test(cssColor)) {
            var c = cssColor.toUpperCase();
            return { type: 'rgb', value: c[1] + c[1] + c[2] + c[2] + c[3] + c[3] };
        }

        // rgb(a) color
        parts = cssColor.match(/^rgba?\(\s*(\d+)\s*,\s*(\d+)\s*,\s*(\d+)(?:\s*,\s*(1|0|0?\.\d+))?\s*\)$/);
        if (!_.isArray(parts)) { return null; }

        // transparent rgba color
        if (parseFloat(parts[4]) === 0) { return Color.AUTO; }

        for (var i = 1; i <= 3; ++i) {
            parts[i] = Math.min(255, parseInt(parts[i], 10)).toString(16).toUpperCase();
            if (parts[i].length === 1) { parts[i] = '0' + parts[i]; }
        }

        return { type: 'rgb', value: parts[1] + parts[2] + parts[3] };
    };

    /**
     * Appends a new transformation to the passed color in-place.
     *
     * @param {Object} color
     *  A color object that will be modified in-place.
     *
     * @param {String} type
     *  The name of a color transformation.
     *
     * @param {Number} [value]
     *  The value for the color transformation. Can be omitted, if the
     *  specified transformations does not need a value (e.g. 'inv').
     */
    Color.appendTransformation = function (color, type, value) {
        var transformation = { type: type };
        if (_.isNumber(value)) { transformation.value = value; }
        (color.transformations || (color.transformations = [])).push(transformation);
    };

    // exports ================================================================

    return Color;

});
