/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/editframework/utils/attributeutils', ['io.ox/office/tk/utils'], function (Utils) {

    'use strict';

    // static class AttributeUtils ============================================

    var AttributeUtils = {};

    // methods ----------------------------------------------------------------

    /**
     * Returns whether the passed string is the name of a attribute that is
     * registered in the passed attribute definition map.
     *
     * @param {Object} definitions
     *  The attribute definitions map to be checked for an attribute.
     *
     * @param {String} name
     *  The attribute name to be checked.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Boolean} [options.special=false]
     *      If set to true, the check will include 'special' attributes too
     *      (attributes marked with the 'special' flag in the attribute
     *      definitions denoting an attribute for internal use only, not for
     *      usage in document operations). Otherwise, special attributes will
     *      not be recognized by this method.
     *
     * @returns {Boolean}
     *  Whether the attribute is registered in the attribute definitions map.
     */
    AttributeUtils.isRegisteredAttribute = function (definitions, name, options) {

        var // whether to include special attributes
            special = Utils.getBooleanOption(options, 'special', false);

        return (name in definitions) && (special || (definitions[name].special !== true));
    };

    /**
     * Returns the attribute set containing all explicit attributes stored in
     * the passed DOM element.
     *
     * @param {HTMLElement|jQuery} node
     *  The DOM element whose explicit attribute set will be returned. If this
     *  object is a jQuery collection, uses the first node it contains.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {String} [options.family]
     *      If specified, extracts the attributes of a specific attribute
     *      family from the attribute set. Otherwise, returns the entire
     *      attribute set object with all attributes mapped by their family.
     *  @param {Boolean} [options.direct=false]
     *      If set to true, the returned attribute set will be a reference to
     *      the original map stored in the attribute, which MUST NOT be
     *      modified. By default, a deep clone of the attribute set that can be
     *      modified freely will be returned.
     *
     * @returns {Object}
     *  The (incomplete) attribute set if existing, otherwise an empty object.
     */
    AttributeUtils.getExplicitAttributes = function (node, options) {

        var // the original and complete attribute map
            attributes = $(node).data('attributes'),
            // the attribute family to be extracted from the complete map
            family = Utils.getStringOption(options, 'family'),
            // whether to return the resulting object without cloning
            direct = Utils.getBooleanOption(options, 'direct', false);

        // reduce to selected family
        if (_.isObject(attributes) && _.isString(family)) {
            attributes = (family in attributes) ? attributes[family] : undefined;
        }

        // return attributes directly or as a deep clone
        return _.isObject(attributes) ? (direct ? attributes : _.copy(attributes, true)) : {};
    };

    /**
     * Returns the identifier of the style sheet referred by the passed
     * element.
     *
     * @param {HTMLElement|jQuery} node
     *  The DOM element whose style sheet identifier will be returned. If this
     *  object is a jQuery collection, uses the first DOM node it contains.
     *
     * @returns {String|Null}
     *  The style sheet identifier at the passed element.
     */
    AttributeUtils.getElementStyleId = function (node) {
        var styleId = AttributeUtils.getExplicitAttributes(node, { direct: true }).styleId;
        return _.isString(styleId) ? styleId : null;
    };

    /**
     * Returns whether the passed elements contain equal formatting attributes.
     *
     * @param {HTMLElement|jQuery} node1
     *  The first DOM element whose formatting attributes will be compared with
     *  the attributes of the other passed DOM element. If this object is a
     *  jQuery collection, uses the first DOM node it contains.
     *
     * @param {HTMLElement|jQuery} node2
     *  The second DOM element whose formatting attributes will be compared
     *  with the attributes of the other passed DOM element. If this object is
     *  a jQuery collection, uses the first DOM node it contains.
     *
     * @returns {Boolean}
     *  Whether both DOM elements contain equal explicit formatting attributes.
     */
    AttributeUtils.hasEqualElementAttributes = function (node1, node2) {
        return _.isEqual(
            AttributeUtils.getExplicitAttributes(node1, { direct: true }),
            AttributeUtils.getExplicitAttributes(node2, { direct: true })
        );
    };

    // exports ================================================================

    return AttributeUtils;

});
