/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Stefan Eckert <stefan.eckert@open-xchange.com>
 */

define('io.ox/office/drawinglayer/view/charttypepicker',
    ['io.ox/office/tk/utils',
     'io.ox/office/tk/forms',
     'io.ox/office/baseframework/view/basecontrols',
     'io.ox/office/drawinglayer/lib/canvasjs.min',
     'gettext!io.ox/office/drawing'
    ], function (Utils, Forms, BaseControls, CanvasJS, gt) {

    'use strict';

    var // class name shortcuts
        RadioList = BaseControls.RadioList,

        // all chart type that can appear in the chart type picker control
        CHART_TYPES = {
            'column standard':         { cat: 1, title: /*#. Chart type: vertical bars (clustered) */                                                                     gt.pgettext('chart-type', 'Column'),           type: 'column2d',  stacking: 'standard',       cjs: 'column' },
            'column stacked':          { cat: 1, title: /*#. Chart type: vertical bars (stacked from bottom to top) */                                                    gt.pgettext('chart-type', 'Column (stacked)'), type: 'column2d',  stacking: 'stacked',        cjs: 'stackedColumn' },
            'column percentStacked':   { cat: 1, title: /*#. Chart type: vertical bars (stacked from bottom to top with percent scaling) */                               gt.pgettext('chart-type', 'Column (percent)'), type: 'column2d',  stacking: 'percentStacked', cjs: 'stackedColumn100' },
            'bar standard':            { cat: 1, title: /*#. Chart type: horizontal bars (clustered) */                                                                   gt.pgettext('chart-type', 'Bar'),              type: 'bar2d',     stacking: 'standard',       cjs: 'bar' },
            'bar stacked':             { cat: 1, title: /*#. Chart type: horizontal bars (stacked from left to right) */                                                  gt.pgettext('chart-type', 'Bar (stacked)'),    type: 'bar2d',     stacking: 'stacked',        cjs: 'stackedBar' },
            'bar percentStacked':      { cat: 1, title: /*#. Chart type: horizontal bars (stacked from left to right with percent scaling) */                             gt.pgettext('chart-type', 'Bar (percent)'),    type: 'bar2d',     stacking: 'percentStacked', cjs: 'stackedBar100' },
            'line standard':           { cat: 2, title: /*#. Chart type: data points connected with lines */                                                              gt.pgettext('chart-type', 'Line'),             type: 'line2d',    stacking: 'standard',       cjs: 'line' },
            'line standard curved':    { cat: 2, title: /*#. Chart type: data points connected with curved lines */                                                       gt.pgettext('chart-type', 'Line (curved)'),    type: 'line2d',    stacking: 'standard',       cjs: 'spline', curved: true },
            'scatter standard':        { cat: 2, title: /*#. Chart type: data points with free X/Y coordinates connected with lines */                                    gt.pgettext('chart-type', 'Scatter'),          type: 'scatter2d', stacking: 'standard',       cjs: 'line' },
            'scatter standard curved': { cat: 2, title: /*#. Chart type: data points with free X/Y coordinates connected with curved lines */                             gt.pgettext('chart-type', 'Scatter (curved)'), type: 'scatter2d', stacking: 'standard',       cjs: 'spline', curved: true },
            'bubble standard':         { cat: 3, title: /*#. Chart type: data points with free X/Y coordinates drawn as circles/bubbles */                                gt.pgettext('chart-type', 'Bubble'),           type: 'bubble2d',  stacking: 'standard',       cjs: 'bubble' },
            'pie standard':            { cat: 3, title: /*#. Chart type: single pie */                                                                                    gt.pgettext('chart-type', 'Pie'),              type: 'pie2d',     stacking: 'standard',       cjs: 'pie', varyColors: true },
            'donut standard':          { cat: 3, title: /*#. Chart type: one or multiple concentric circles */                                                            gt.pgettext('chart-type', 'Donut'),            type: 'donut2d',   stacking: 'standard',       cjs: 'doughnut', varyColors: true },
            'area standard':           { cat: 3, title: /*#. Chart type: filled areas between X axis and data points (areas overlay each other) */                        gt.pgettext('chart-type', 'Area'),             type: 'area2d',    stacking: 'standard',       cjs: 'area' },
            'area stacked':            { cat: 3, title: /*#. Chart type: filled areas between X axis and data points (stacked from bottom to top) */                      gt.pgettext('chart-type', 'Area (stacked)'),   type: 'area2d',    stacking: 'stacked',        cjs: 'stackedArea' },
            'area percentStacked':     { cat: 3, title: /*#. Chart type: filled areas between X axis and data points (stacked from bottom to top with percent scaling) */ gt.pgettext('chart-type', 'Area (percent)'),   type: 'area2d',    stacking: 'percentStacked', cjs: 'stackedArea100' }
        };

    // set missing Boolean properties explicitly to false
    _.each(CHART_TYPES, function(CHART_TYPE) {
        CHART_TYPE.curved = CHART_TYPE.curved || false;
    });

    // class ChartTypePicker ==================================================

    /**
     * ChartTypePicker generates preview Images for all assigned Chart-Types
     * With CanvasJS it uses the same Drawer like the Chart in the Sheets
     * Colors are Accents 1 till 4 and the preview are refresh if the theme is changed
     */
    function ChartTypePicker(app, initOptions) {

        var // self reference
            self = this,

            // data-model for CanvasJS to drop all changed value before drawing
            data = {},

            scatterPoints = [2.5, 4, 1.5, 3.5, 2, 2.5];

        // base constructor ---------------------------------------------------

        RadioList.call(this, Utils.extendOptions({
            width: 150,
            icon: 'fa-bar-chart-o',
            tooltip: gt('Chart type'),
            updateCaptionMode: 'label',
            itemDesign: 'grid',
            gridColumns: 6,
            smallerVersion: {
                css: {
                    width: 50
                },
                hideLabel: true
            }
        }, initOptions));

        // private methods ----------------------------------------------------

        /**
         * hook on create items, only called once per type,
         * sets the button text and fill the chain-data for deferred rendering
         */
        function createItemHandler(event, buttonNode) {
            buttonNode.addClass('mini-caption').css({ width: 100, height: 100, textAlign: 'center' });
        }

        /**
         * called initial and on theme-change
         * creates CanvasJS structure
         * and draws deferred all types in the chain to images
         */
        var updatePreviewCharts = app.createDebouncedMethod($.noop, function () {

            var // the style collection of the document
                documentStyles = app.getModel().getDocumentStyles(),
                // unique identifier used by CanvasJS
                id = _.uniqueId('io-ox-office-chart-dummy-'),
                // container node for the CanvasJS chart
                containerNode = $('<div id="' + id + '" style="position:fixed;visibility:hidden;left:-100px;top:-100px;width:100px;height:100px;">').appendTo('body'),
                // the CanvasJS renderer
                chart = new CanvasJS.Chart(id, data),
                // the <canvas> node from the renderer
                canvasNode = containerNode.find('canvas');

            if (canvasNode.length === 0) {
                Utils.warn('ChartTypePicker.updatePreviewCharts(): canvas element not found');
                return;
            }

            app.iterateArraySliced(self.getOptionButtons(), function (buttonNode) {

                var id = Forms.getButtonValue(buttonNode),
                    options = $(buttonNode).data('options'),
                    cjs = options.data.cjs,
                    single = /^(pie|doughnut)$/.test(cjs),
                    alt = Utils.getStringOption(options, 'label', ''),
                    markup = '';

                _.each(data.data, function (dataSeries, index) {
                    dataSeries.type = cjs;
                    var cl = { type: 'scheme', value: 'accent' + (index + 1) };
                    dataSeries.color = documentStyles.getCssColor(cl, 'fill');
                });

                _.each(data.data[0].dataPoints, function (dataPoint, index) {
                    if (single) {
                        var cl = { type: 'scheme', value: 'accent' + (index + 1) };
                        dataPoint.color = documentStyles.getCssColor(cl, 'fill');
                    } else {
                        dataPoint.color = null;
                    }
                });

                _.each(data.data[1].dataPoints, function (dataPoint, index) {
                    if (id.indexOf('scatter') === 0) {
                        dataPoint.x = scatterPoints[index];
                    } else {
                        dataPoint.x = index;
                    }
                });

                chart.render();

                // safely remove all images from the button element
                app.destroyImageNodes(buttonNode);

                // create a new image element and insert it into the button
                markup = '<img src="' + canvasNode[0].toDataURL() + '" style="width:100px;height:100px;margin-left:-11px;"';
                if (alt.length > 0) { markup += ' alt="' + Utils.escapeHTML(alt) + '"'; }
                markup += '>';
                $(buttonNode).append(markup);

            }).done(function () {
                containerNode.remove();
            });

        }, { delay: 50 });

        // initialization -----------------------------------------------------

        // initializes the data-model with disabled axes and random-values
        (function initData() {
            data.axisX = {};
            data.axisY = {};
            data.creditHref = '';
            data.creditText = '';
            data.title = { text: '' };
            data.data = [];


            var pointList = [[6, 4, 5, 3, 4, 1], [1, 2, 3, 4, 6, 7]];

            _.each(pointList, function (points, series) {
                var dataPoints = [];
                data.data[series] = { markerSize: 2, indexLabelFontColor: 'transparent', indexLabelLineColor: 'transparent', showInLegend: false, dataPoints: dataPoints, sourcePoints: {}};
                _.each(points, function (point, index) {
                    dataPoints.push({ label: 'bla ' + (index + 1), y: point, z: (2 + Math.random() * 100)|0});
                });
            });

            data.axisX.labelFontColor = data.axisX.lineColor = data.axisX.tickColor = data.axisX.gridColor = 'transparent';
            data.axisX.labelFontSize = 1;
            data.axisY.labelFontColor = data.axisY.lineColor = data.axisY.tickColor = data.axisY.gridColor = 'transparent';
            data.axisY.labelFontSize = 1;
        })();

        this.getMenuNode().addClass('chart-type-picker');

        // create the (empty) option buttons
        self.getMenu().on('create:item', createItemHandler);
        _.each(CHART_TYPES, function (chartType, id) {
            self.createOptionButton(id, { section: chartType.cat, label: chartType.title, tooltip: chartType.title, data: chartType });
        });

        // lazy initialization of the charts before first usage of the menu
        this.getMenu().one('popup:beforeshow', updatePreviewCharts);

        // refresh preview charts in the drop-down menu after changed theme (color scheme)
        this.listenTo(app.getImportPromise(), 'done', function () {
            self.listenTo(app.getModel().getThemeCollection(), 'triggered', updatePreviewCharts);
        });

    } // class ChartTypePicker

    // constants --------------------------------------------------------------

    ChartTypePicker.CHART_TYPES = CHART_TYPES;

    // exports ================================================================

    return RadioList.extend({ constructor: ChartTypePicker });

});
