/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/baseframework/view/basecontrols',
    ['io.ox/office/tk/utils',
     'io.ox/office/tk/keycodes',
     'io.ox/office/tk/forms',
     'io.ox/office/tk/popup/basemenu',
     'io.ox/office/tk/control/group',
     'io.ox/office/tk/control/label',
     'io.ox/office/tk/control/button',
     'io.ox/office/tk/control/checkbox',
     'io.ox/office/tk/control/radiogroup',
     'io.ox/office/tk/control/radiolist',
     'io.ox/office/tk/control/textfield',
     'io.ox/office/tk/control/unitfield',
     'io.ox/office/tk/control/combofield',
     'io.ox/office/tk/control/spinfield',
     'io.ox/office/tk/control/picture',
     'io.ox/office/tk/control/captionmixin',
     'io.ox/office/tk/control/widthmixin',
     'io.ox/office/tk/control/menumixin',
     'io.ox/office/baseframework/view/component',
     'gettext!io.ox/office/main'
    ], function (Utils, KeyCodes, Forms, BaseMenu, Group, Label, Button, CheckBox, RadioGroup, RadioList, TextField, UnitField, ComboField, SpinField, Picture, CaptionMixin, WidthMixin, MenuMixin, Component, gt) {

    'use strict';

    // static class BaseControls ==============================================

    /**
     * Provides different classes for GUI controls. Collects all standard
     * controls defined in the toolkit, and adds more sophisticated controls.
     */
    var BaseControls = {
            Group: Group,
            Label: Label,
            Button: Button,
            CheckBox: CheckBox,
            RadioGroup: RadioGroup,
            RadioList: RadioList,
            TextField: TextField,
            UnitField: UnitField,
            ComboField: ComboField,
            SpinField: SpinField,
            Picture: Picture
        };

    // constants --------------------------------------------------------------

    /**
     * A standard header label for a menu with view options.
     *
     * @constant
     */
    BaseControls.VIEW_LABEL = /*#. menu title: view settings (zoom etc.) for a document */ gt.pgettext('menu-title', 'View');

    /**
     * A standard header label for a menu with various user actions.
     *
     * @constant
     */
    BaseControls.ACTIONS_LABEL = /*#. menu title: various user actions */ gt.pgettext('menu-title', 'Actions');

    /**
     * A standard header label for a menu with more options or functions.
     *
     * @constant
     */
    BaseControls.MORE_LABEL = /*#. menu title: additional options or actions */ gt.pgettext('menu-title', 'More');

    /**
     * A standard header label for options in a drop-down menu etc.
     *
     * @constant
     */
    BaseControls.OPTIONS_LABEL = gt.pgettext('menu-title', 'Options');

    /**
     * A standard header label for a zoom menu.
     *
     * @constant
     */
    BaseControls.ZOOM_LABEL = gt.pgettext('menu-title', 'Zoom');

    /**
     * Standard options for the 'Close' button.
     *
     * @constant
     */
    BaseControls.QUIT_BUTTON_OPTIONS = { icon: 'fa-times', tooltip: gt('Close document') };

    /**
     * Standard options for the 'Hide side panel' button.
     *
     * @constant
     */
    BaseControls.HIDE_SIDEPANE_BUTTON_OPTIONS = { icon: 'docs-hide-sidepane', tooltip: gt('Hide side panel'), value: false };

    /**
     * Standard options for the 'Show side panel' button.
     *
     * @constant
     */
    BaseControls.SHOW_SIDEPANE_BUTTON_OPTIONS = { icon: 'docs-show-sidepane', tooltip: gt('Show side panel'), value: true };

    /**
     * Standard options for the 'Zoom out' button.
     *
     * @constant
     */
    BaseControls.ZOOMOUT_BUTTON_OPTIONS = { icon: 'docs-zoom-out', tooltip: gt('Zoom out') };

    /**
     * Standard options for the 'Zoom in' button.
     *
     * @constant
     */
    BaseControls.ZOOMIN_BUTTON_OPTIONS = { icon: 'docs-zoom-in', tooltip: gt('Zoom in') };

    // class DynamicLabel =====================================================

    /**
     * A label control that updates its text dynamically via its current
     * internal value.
     *
     * @constructor
     *
     * @extends Label
     *
     * @param {Object} [initOptions]
     *  Optional parameters:
     *  @param {Function} [initOptions.captionResolver]
     *      A callback function that receives the control value, and returns a
     *      string to be shown as label, or an options object containing label
     *      and/or icon options. Will be called in the context of this label
     *      instance. If omitted, the current value will be used. If it is an
     *      object, it will be used as caption options object, if it is null,
     *      the label text will be cleared, otherwise it will be converted to a
     *      string.
     * @param {Boolean} [initOptions.setToolTip=false]
     *      If this is 'true', set the given label as ToolTip
     */
    BaseControls.DynamicLabel = Label.extend({ constructor: function (initOptions) {

        var // self reference
            self = this,

            // resolves the value to a display string, or a caption options object
            captionResolver = Utils.getFunctionOption(initOptions, 'captionResolver'),

            // set the label as ToolTip
            setToolTip = Utils.getOption(initOptions, 'setToolTip', false);

        // base constructor ---------------------------------------------------

        Label.call(this, initOptions);

        // private methods ----------------------------------------------------

        /**
         * Updates the label text.
         */
        function updateHandler(value) {
            var label = _.isFunction(captionResolver) ? captionResolver.call(self, value) : value;
            if (_.isObject(label)) {
                self.setCaption(label);
            } else {
                self.setLabel(_.isNull(label) ? '' : label);
                if (setToolTip) {
                    self.setToolTip(label);
                }
            }
        }

        // initialization -----------------------------------------------------

        // additional options for an ARIA live region for dynamic labels
        this.getLabelNode().attr({
            role: 'log',
            'aria-live': 'assertive',
            'aria-relevant': 'additions',
            'aria-atomic': true,
            'aria-readonly': true
        });

        // update label text according to current control value
        this.registerUpdateHandler(updateHandler);

        // destroy all class members
        this.registerDestructor(function () {
            initOptions = self = captionResolver = null;
        });

    }}); // class DynamicLabel

    // class PercentageLabel ==================================================

    /**
     * A label control that shows a number as integral percentage.
     *
     * @constructor
     *
     * @extends BaseControls.DynamicLabel
     *
     * @param {Object} [initOptions]
     *  Optional parameters. Supports all options supported by the base class
     *  DynamicLabel, except for the option 'captionResolver'.
     */
    BaseControls.PercentageLabel = BaseControls.DynamicLabel.extend({ constructor: function (initOptions) {

        // base constructor ---------------------------------------------------

        BaseControls.DynamicLabel.call(this, Utils.extendOptions({
            minWidth: 90,
            style: 'text-align:right;'
        }, initOptions, {
            captionResolver: captionResolver
        }));

        // private methods ----------------------------------------------------

        /**
         * Converts the passed number to the percentage label text.
         */
        function captionResolver(value) {
            return (
                //#. a text label showing any number as integral percentage, e.g. "50%"
                //#. %1$d is the value, converted to percent
                //#. the trailing percent sign must remain in the text
                //#, c-format
                gt('%1$d%', _.noI18n(Math.round(value * 100)))
            );
        }

        // initialization -----------------------------------------------------

        // destroy all class members
        this.registerDestructor(function () {
            initOptions = null;
        });

    }}); // class PercentageLabel

    // class StatusLabel ======================================================

    /**
     * A status label with special appearance and fade-out animation.
     *
     * The method StatusLabel.setValue() accepts an options map containing the
     * same options as described for the 'options' constructor parameter,
     * allowing to override the default options of the status label.
     *
     * @constructor
     *
     * @extends BaseControls.DynamicLabel
     *
     * @param {Object} [initOptions]
     *  Optional parameters:
     *  @param {String} [initOptions.type='info']
     *      The label type ('success', 'warning', 'error', or 'info').
     *  @param {Boolean} [initOptions.fadeOut=false]
     *      Whether to fade out the label automatically after a short delay.
     *  @param {Number} [initOptions.delay=0]
     *      The delay time after the status label will be actually updated.
     *  @param {Number} [initOptions.changed=false]
     *      If set to true, a status label currently faded out will only be
     *      shown if its label has really changed (and not with every update
     *      call with the same label).
     */
    BaseControls.StatusLabel = BaseControls.DynamicLabel.extend({ constructor: function (app, initOptions) {

        var // self reference
            self = this,

            // the new type of the label (colors)
            type = Utils.getStringOption(initOptions, 'type', 'info'),

            // whether to fade out the label automatically
            fadeOut = Utils.getBooleanOption(initOptions, 'fadeOut', false),

            // delay time after the label will be updated
            delay = Utils.getIntegerOption(initOptions, 'delay', 0, 0),

            // whether to show the label only after the value has changed
            changed = Utils.getBooleanOption(initOptions, 'changed', false),

            // current initial delay timer before the state of the label will be changed
            initialTimer = null,

            // current animation delay timer (jQuery.delay() does not work well with jQuery.stop())
            animationTimer = null;

        // base constructor ---------------------------------------------------

        BaseControls.DynamicLabel.call(this, { classes: 'status-label' });

        // private methods ----------------------------------------------------

        /**
         * Stops the running jQuery fade animation and removes all explicit CSS
         * attributes from the group node.
         */
        function stopNodeAnimation() {
            self.getNode().stop(true).css({ display: '', opacity: '' });
        }

        /**
         * Callback called from the Label.setValue() method.
         */
        function updateHandler(caption, options) {

            // do nothing if the label will not be updated when the text does not change
            if (Utils.getBooleanOption(options, 'changed', changed) && (self.getLabel() === caption)) {
                return;
            }

            if (initialTimer) { initialTimer.abort(); }
            initialTimer = app.executeDelayed(function () {

                // stop running fade-out and remove CSS attributes added by jQuery
                if (animationTimer) { animationTimer.abort(); }
                stopNodeAnimation();

                // update the status label
                if (_.isString(caption) && (caption.length > 0)) {
                    self.show().getNode().attr('data-type', Utils.getStringOption(options, 'type', type));
                    if (Utils.getBooleanOption(options, 'fadeOut', fadeOut)) {
                        animationTimer = app.executeDelayed(function () {
                            self.getNode().fadeOut(function () { self.hide(); });
                        }, { delay: 2000 });
                    }
                } else {
                    self.hide();
                }

            }, { delay: Utils.getIntegerOption(options, 'delay', delay, 0) });
        }

        // initialization -----------------------------------------------------

        this.registerUpdateHandler(updateHandler).setValue(null);

        // destroy all class members
        this.registerDestructor(function () {
            stopNodeAnimation();
        });

    }}); // class StatusLabel

    // class ComponentMenu ====================================================

    /**
     * A pop-up menu element that contains a complete self-contained instance
     * of a view component (class Component) with independent controls bound to
     * different controller items.
     *
     * @constructor
     *
     * @extends BaseMenu
     *
     * @param {BaseApplication} app
     *  The application containing this pop-up menu.
     *
     * @param {Object} [initOptions]
     *  Optional parameters. Supports all options also supported by the base
     *  class BaseMenu.
     */
    BaseControls.ComponentMenu = BaseMenu.extend({ constructor: function (app, initOptions) {

        var // self reference
            self = this,

            // the view component in the drop-down menu
            component = new Component(app);

        // base constructor ---------------------------------------------------

        BaseMenu.call(this, initOptions);

        // private methods ----------------------------------------------------

        /**
         * Inserts the passed group into the menu component.
         */
        function groupInserter(group, options) {

            var // the root node of the passed group
                groupNode = group.getNode(),
                // the target node for the passed group
                targetNode = component.getNode(),
                // the last child in the menu component
                lastNode = targetNode.children().last();

            if (Utils.getBooleanOption(options, 'inline', false)) {
                if (lastNode.is('.inline-container')) {
                    targetNode = lastNode;
                } else {
                    targetNode = $('<div class="inline-container">').appendTo(targetNode);
                    if (lastNode.is('.group')) { targetNode.append(lastNode); }
                }
            }

            // insert the group node
            targetNode.append(groupNode);
        }

        // methods ------------------------------------------------------------

        /**
         * Appends the passed control group to the pop-up menu and binds it to
         * the specified controller item.
         *
         * @param {String|Null} key
         *  The key of a controller item associated to the control group. The
         *  visible state, the enabled state, and the current value of the
         *  group will be bound to the states and value of the controller item.
         *  If the group triggers a 'group:change' event, the controller item
         *  will be executed. The key can be set to the value null to insert an
         *  unbound group into this pop-up menu.
         *
         * @param {Group} group
         *  The control group object to be inserted into the pop-up menu.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.inline=false]
         *      Inserts the group node into the same menu row as the preceding
         *      inline group. If this group is the first inline group, it will
         *      be inserted into a new menu row.
         *  @param {Boolean} [options.sticky=false]
         *      If set to true, this pop-up menu will not be closed after the
         *      inserted group has been activated.
         *
         * @returns {ComponentMenu}
         *  A reference to this instance.
         */
        this.addGroup = function (key, group, options) {

            // set the root node of the embedded view component as focus target node for sticky groups
            if (Utils.getBooleanOption(options, 'sticky', false)) {
                options = _.extend({ focusTarget: this.getContentNode() }, options);
                delete options.sticky;
            }

            // do not close this menu, if a sub menu from the component is opened
            if (_.isFunction(group.getMenuNode)) {
                this.registerFocusableNodes(group.getMenuNode());
            }

            // insert the passed group
            component.addGroup(key, group, options);
            return this;
        };

        /**
         * Appends a separator line to the pop-up menu.
         *
         * @returns {ComponentMenu}
         *  A reference to this instance.
         */
        this.addSeparator = function () {
            component.getNode().append('<div class="separator-line">');
            return this;
        };

        /**
         * Appends a header label to the pop-up menu. If the menu is not empty
         * anymore, automatically adds a separator line above the header label.
         *
         * @param {String} label
         *  The label text.
         *
         * @returns {ComponentMenu}
         *  A reference to this instance.
         */
        this.addHeaderLabel = function (label, options) {

            var // whether to add a separator line
                separator = Utils.getBooleanOption(options, 'separator', true);

            if (separator && component.getNode()[0].hasChildNodes()) {
                this.addSeparator();
            }

            component.getNode().append('<div class="header-label" aria-hidden="true">' + Utils.escapeHTML(label) + '</div>');
            return this;
        };

        this.addComponent = function (newComponent) {
            component = newComponent;
            this.clearContents();
            this.appendContentNodes(component.getNode());
        };

        this.getComponent = function () {
            return component;
        };

        this.removeComponent = function (clearContents) {
            component = null;

            if (clearContents !== false) {
                this.clearContents();
            }
        };

        // initialization -----------------------------------------------------

        // do not allow to clear the menu manually
        // delete this.clearContents;

        // insert the root node of the view component into the drop-down menu
        this.appendContentNodes(component.getNode());

        // insert new groups into the component
        component.registerGroupInserter(groupInserter);

        // hide pop-up menu when application is not active
        app.getWindow().on('beforehide', function () { self.hide(); });

        // resize the pop-up node when the contents of the view component have been changed
        component.on('component:layout', function () { self.refreshImmediately(); });

        // additional cursor key shortcuts for focus navigation
        Forms.enableCursorFocusNavigation(this.getNode(), { homeEnd: true });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            if (!_.isNull(component)) { component.destroy(); }
            app = initOptions = self = component = null;
        });

    }}); // class ComponentMenu

    // class ComponentMenuMixin ===============================================

    /**
     * A mix-in class for control groups that adds a drop-down button with a
     * complete self-contained view component instance with completely
     * independent controls in its drop-down menu.
     *
     * Note: This is a mix-in class supposed to extend an existing instance of
     * the class Group or one of its derived classes. Expects the symbol 'this'
     * to be bound to an initialized instance of Group (this mix-in constructor
     * MUST be called after the Group constructor).
     *
     * @constructor
     *
     * @extends MenuMixin
     *
     * @param {BaseApplication} app
     *  The application containing this control.
     *
     * @param {Object} [initOptions]
     *  Optional parameters. Supports all options also supported by the base
     *  class MenuMixin (visibility of caret icon), and the class
     *  BaseControls.ComponentMenu (settings for the drop-down menu).
     */
    BaseControls.ComponentMenuMixin = MenuMixin.extend({ constructor: function (app, initOptions) {

        var // self reference
            self = this,

            // the pop-up menu instance
            menu = new BaseControls.ComponentMenu(app, Utils.extendOptions(initOptions, { anchor: this.getNode() }));

        // base constructors --------------------------------------------------

        MenuMixin.call(this, menu, initOptions);

        // private methods ----------------------------------------------------

        /**
         * Sets the minimum width of the menu to the width of the group.
         */
        function popupBeforeShowHandler() {
            menu.getContentNode().children().css({ minWidth: self.getNode().outerWidth() });
        }

        // methods ------------------------------------------------------------

        // export methods of the menu
        _.each(['addGroup', 'addSeparator', 'addHeaderLabel'], function (methodName) {
            this[methodName] = function () {
                var result = menu[methodName].apply(menu, arguments);
                return (result === menu) ? this : result;
            };
        }, this);

        // initialization -----------------------------------------------------

        // register event handlers
        this.listenTo(menu, 'popup:beforeshow', popupBeforeShowHandler);

        // destroy all class members
        this.registerDestructor(function () {
            app = initOptions = self = menu = null;
        });

    }}); // class ComponentMenuMixin

    // class ComponentMenuButton ==============================================

    /**
     * A generic drop-down button control that shows a complete self-contained
     * view component instance with completely independent controls in its
     * drop-down menu. See class ComponentMenuSplitButton for a similar
     * drop-down control with an additional split button control.
     *
     * @constructor
     *
     * @extends Group
     * @extends ComponentMenuMixin
     * @extends CaptionMixin
     * @extends WidthMixin
     *
     * @param {BaseApplication} app
     *  The application containing this control.
     *
     * @param {Object} [initOptions]
     *  Optional parameters. Supports all options also supported by the base
     *  class Group, the mix-in class WidthMixin, and the mix-in class
     *  ComponentMenuMixin.
     */
    BaseControls.ComponentMenuButton = Group.extend({ constructor: function (app, initOptions) {

        var // the drop-down button
            menuButton = $(Forms.createButtonMarkup(_.extend({ tooltip: Utils.getStringOption(initOptions, 'label', null, true), attributes: { 'aria-haspopup': true } }, initOptions)));

        // base constructors --------------------------------------------------

        Group.call(this, initOptions);
        CaptionMixin.call(this, menuButton, initOptions);
        WidthMixin.call(this, menuButton, initOptions);
        BaseControls.ComponentMenuMixin.call(this, app, Utils.extendOptions(initOptions, { button: menuButton }));

        // initialization -----------------------------------------------------

        // add the drop-down button to the group
        this.addChildNodes(menuButton);

        // destroy all class members
        this.registerDestructor(function () {
            app = initOptions = menuButton = null;
        });

    }}); // class ComponentMenuButton

    // class ComponentMenuSplitButton =========================================

    /**
     * A button control with a combined drop-down menu containing a complete
     * self-contained view component instance with completely independent
     * controls. See class ComponentMenuButton for a similar simple drop-down
     * control, but without additional split button control.
     *
     * @constructor
     *
     * @extends Button
     * @extends ComponentMenuMixin
     *
     * @param {BaseApplication} app
     *  The application containing this control.
     *
     * @param {Object} [initOptions]
     *  Optional parameters. Supports all options also supported by the base
     *  class Button, and the mix-in class ComponentMenuMixin.
     */
    BaseControls.ComponentMenuSplitButton = Button.extend({ constructor: function (app, initOptions) {

        var // the drop-down button
            menuButton = $(Forms.createButtonMarkup({ tooltip: Utils.getStringOption(initOptions, 'caretTooltip', Utils.getStringOption(initOptions, 'tooltip', '')) }));

        // base constructors --------------------------------------------------

        Button.call(this, initOptions);
        BaseControls.ComponentMenuMixin.call(this, app, Utils.extendOptions(initOptions, { button: menuButton }));

        // initialization -----------------------------------------------------

        // add the drop-down button to the group
        this.addChildNodes(menuButton);

        // destroy all class members
        this.registerDestructor(function () {
            app = initOptions = menuButton = null;
        });

    }}); // class ComponentMenuSplitButton

    // class ComponentMenuCheckBox ============================================

    /**
     * A check box control with a combined drop-down menu containing a complete
     * self-contained view component instance with completely independent
     * controls.
     *
     * @constructor
     *
     * @extends CheckBox
     * @extends ComponentMenuMixin
     *
     * @param {BaseApplication} app
     *  The application containing this control.
     *
     * @param {Object} [initOptions]
     *  Optional parameters. Supports all options also supported by the base
     *  class CheckBox, and the mix-in class ComponentMenuMixin.
     */
    BaseControls.ComponentMenuCheckBox = CheckBox.extend({ constructor: function (app, initOptions) {

        var // the drop-down button
            menuButton = $(Forms.createButtonMarkup({ tooltip: Utils.getOption(initOptions, 'tooltip', '') }));

        // base constructors --------------------------------------------------

        CheckBox.call(this, initOptions);
        BaseControls.ComponentMenuMixin.call(this, app, Utils.extendOptions(initOptions, { button: menuButton }));

        // initialization -----------------------------------------------------

        // add the drop-down button to the group
        this.addChildNodes(menuButton);

        // destroy all class members
        this.registerDestructor(function () {
            app = initOptions = menuButton = null;
        });

    }}); // class ComponentMenuCheckBox

    // class ContextMenu ======================================================

    /**
     * An instance of this class represents a context menu bound to a specific
     * DOM element. Right-clicking that DOM element, or pressing specific keys
     * (CONTEXT key) or key combinations (e.g. Shift+F10) while the element is
     * focused or contains the browser focus will show this context menu.
     *
     * @constructor
     *
     * @extends ComponentMenu
     *
     * @param {BaseApplication} app
     *  The application containing this context menu.
     *
     * @param {HTMLObject|jQuery} sourceNode
     *  The DOM element used as event source for context events (right mouse
     *  clicks, or 'contextMenu' events for specific keyboard shortcuts). If
     *  this object is a jQuery collection, uses the first DOM element it
     *  contains.
     *
     * @param {Object} [initOptions]
     *  Initial options for this context menu.
     *  @param {Function} [initOptions.prepareShowHandler]
     *      A callback function that will be invoked before showing the context
     *      menu after a user action (right mouse click, or keyboard). Receives
     *      the original event caused opening the context menu (as jQuery.Event
     *      instance). May return Utils.BREAK to suppress the context menu. May
     *      return a specific anchor position (a rectangle object) for the menu
     *      node. Otherwise, the position of the source context event will be
     *      used as anchor position of the context menu.
     *  @param {String|String[]} [initOptions.enableKey]
     *      The key of a controller item, or an array of controller item keys,
     *      that specify whether the context menu is enabled. Uses the enabled
     *      state of the controller items registered for the specified keys.
     *      All specified items must be enabled to enable the context menu. If
     *      omitted, the context menu will always be enabled.
     *  @param {Boolean} [initOptions.mouseDown=false]
     *      If set to true, the context menu will be shown instantly after the
     *      'mousedown' event for the right mouse button. If omitted or set to
     *      false, the context menu will be opened after the 'mouseup' and its
     *      related 'contextmenu' event.
     */
    BaseControls.ContextMenu = BaseControls.ComponentMenu.extend({ constructor: function (app, sourceNode, initOptions) {

        var // self reference
            self = this,

            // preprocesses browser events before showing the context menu
            prepareShowHandler = Utils.getFunctionOption(initOptions, 'prepareShowHandler', $.noop),

            // controller keys specifying whether the context menu is enabled
            enableKeys = Utils.getOption(initOptions, 'enableKey'),

            // whether to show the context menu instantly after the 'mousedown' event
            mouseDown = Utils.getBooleanOption(initOptions, 'mouseDown', false),

            // the position to open the context menu (initialized from mouse/keyboard events)
            anchorPosition = null;

        // base constructor ---------------------------------------------------

        BaseControls.ComponentMenu.call(this, app, { anchor: getAnchorPosition, anchorPadding: 0 });

        // methods ------------------------------------------------------------

        /**
         * Returns the target position of the context menu.
         */
        function getAnchorPosition() {
            return anchorPosition;
        }

        /**
         * Invokes the callback function passed to the constructor, and shows
         * the context menu at the correct position.
         */
        function showContextMenu(event) {
            var position = prepareShowHandler.call(self, event);
            if ((position !== Utils.BREAK) && self.isEnabled()) {
                anchorPosition = _.isObject(position) ? position : { left: event.pageX, top: event.pageY, width: 1, height: 1 };
                self.hide().show();
            }
        }

        /**
         * Handles 'mousedown' events of the source node. Shows the context
         * menu, if the option 'mouseDown' has been passed to the constructor.
         */
        function mouseDownHandler(event) {
            if (mouseDown && (event.button === 2)) {
                showContextMenu(event);
                event.preventDefault();
            }
        }

        /**
         * Handles 'contextmenu' events of the source node, triggered after
         * right-clicking the anchor node, or pressing specific keys that cause
         * showing a context menu.
         */
        function contextMenuHandler(event) {
            showContextMenu(event);
            event.preventDefault();
        }

        /**
         * Handles 'change:items' events of the controller. Hides the context
         * menu if any of the controller items specified with the constructor
         * option 'enableKey' has been disabled.
         */
        function controllerChangeHandler(event, changedItems) {

            // nothing to do, if the context menu is not visible
            if (!self.isVisible()) { return; }

            var // whether to hide the context menu due to changed controller item states
                hideMenu = _.any(enableKeys, function (enableKey) {
                    return (enableKey in changedItems) && (changedItems[enableKey].enabled === false);
                });

            if (hideMenu) {
                self.hide();
                app.getView().grabFocus();
            }
        }

        // methods ------------------------------------------------------------

        /**
         * Returns whether this context menu is currently enabled (whether it
         * will be shown after right-clicking the anchor node, or using the
         * keyboard).
         *
         * @returns {Boolean}
         *  Whether this context menu is currently enabled.
         */
        this.isEnabled = function () {
            var controller = app.getController();
            return _.all(enableKeys, _.bind(controller.isItemEnabled, controller));
        };

        // initialization -----------------------------------------------------

        // listen to relevant context events on the source node
        sourceNode.on({ mousedown: mouseDownHandler, contextmenu: contextMenuHandler });

        // always use an array of parent keys (as strings)
        enableKeys =
            _.isArray(enableKeys) ? _.filter(enableKeys, _.isString) :
            _.isString(enableKeys) ? [enableKeys] :
            null;

        // hide displayed context menu, if controller item changes to disabled state
        if (_.isArray(enableKeys) && (enableKeys.length > 0)) {
            this.listenTo(app.getController(), 'change:items', controllerChangeHandler);
        }

        // destroy all class members on destruction
        this.registerDestructor(function () {
            app = sourceNode = null;
        });

    }}); // class ContextMenu

    // class LayerMenu ========================================================

    /**
     * An instance of this class represents a layer menu.
     *
     * Instances of this class trigger the following events:
     * - 'layermenu:userclose'
     *      If this menu is closed manually by the user.
     *
     * @constructor
     *
     * @extends ComponentMenu
     *
     * @param {BaseApplication} app
     *  The application containing this context menu.
     *
     * @param {String} layerTitle
     *  The title on top of the menu(-layer)
     *
     * @param {Object} [initOptions]
     *  Optional parameters. Supports all options supported by the base class
     *  ComponentMenu. Additionally, the following options are supported:
     *  @param {String} [initOptions.anchorSelection]
     *      The node to which the menu should snap on open
     *  @param {Boolean} [initOptions.draggable]
     *      Make the menu draggable on true (default).
     *  @param {Boolean} [initOptions.closer]
     *      Whether this menu should show a closer button in its header.
     *  @param {Function} [initOptions.positioning]
     *      Custom positioning function of this layer menu. If set, this function
     *      will override the default positioning function (setLayerMenuPosition).
     */
    BaseControls.LayerMenu = BaseControls.ComponentMenu.extend({ constructor: function (app, layerTitle, initOptions) {

        var // self reference
            self = this,
            // anchor to which the menu snap on
            anchorSelection = Utils.getOption(initOptions, 'anchorSelection', null),
            // container node, where this menu will be appended into
            rootContainerNode = Utils.getOption(initOptions, 'rootContainerNode', null),
            // menu draggable or not
            draggable = Utils.getBooleanOption(initOptions, 'draggable', true),
            // whether this menu should show a closer button in the header
            closer = Utils.getBooleanOption(initOptions, 'closer', false),
            // custom positioning function of the layer menu
            setPosition = Utils.getFunctionOption(initOptions, 'positioning', null),
            // drag-start-position
            layerX = 0, layerY = 0,
            // scroll bars (?)
            scrollHor = false, scrollVer = false,
            // available Windows Sizes
            availableSizes = null,
            // reset the menu-position
            resetPosition = true,
            // selected anchorNode-position
            nodePosition = null,
            // menu-position
            layerPosition = null,
            // menu-Node
            menuNode = null;

        // base constructor ---------------------------------------------------

        BaseControls.ComponentMenu.call(this, app, Utils.extendOptions(initOptions, { autoLayout: false }));
        menuNode = this.getNode();

        // private methods ----------------------------------------------------

        /**
         * fixed the start-position at dragging-start
         */
        function moveStartHandler(event) {
            event.preventDefault();
            var nodeOffset = rootContainerNode ? Utils.getChildNodePositionInNode(rootContainerNode, menuNode) : $(menuNode).offset();
            layerY = nodeOffset.top;
            layerX = nodeOffset.left;
        }

        /**
         * Handler for dragging the menu
         */
        function moveHandler(event) {
            event.preventDefault();
            self.setNodePosition({
                top: (layerY + event.offsetY),
                left: (layerX + event.offsetX)
            });
        }

        /**
         * Decides on which position the menu should be opened
         * relative to the drawing
         *
         * @returns {String}
         * ['top','right','bottom','left','center']
         */
        function choosePosition() {
            var returnPosition = null;
            if (nodePosition.right > layerPosition.width) {
                returnPosition = 'right';
            } else if (nodePosition.left > layerPosition.width) {
                returnPosition = 'left';
            } else if (nodePosition.bottom > layerPosition.height) {
                returnPosition = 'bottom';
            } else {
                returnPosition = 'top';
            }
            return returnPosition;
        }

        /**
         * Calculate the top/left-Position of the menu
         *
         * @param {String} [choosedPosition]
         *  the Position (from 'choosePosition') to which the menu should snap
         *
         * @returns {Object}
         *  - {Number} cssTop
         *      Pixel from top
         *  - {Number} cssRight
         *      Always empty
         *  - {Number} cssBottom
         *      Always empty
         *  - {Number} cssLeft
         *      Pixel from left
         */
        function getCssProps(choosedPosition) {
            var cssTop, cssLeft, cssBottom, cssRight,
                snapTo = null,
                margin = 5;

            if ((nodePosition.bottom <= 0) || (layerPosition.height >= (nodePosition.height+nodePosition.bottom-margin))){
                snapTo = 'bottom';
            }

            switch (choosedPosition) {
            case 'top':
                cssTop      = (nodePosition.top - layerPosition.height - margin);
                cssLeft     = (nodePosition.left);
                break;
            case 'right':
                cssTop      = (snapTo === 'bottom') ? '' : nodePosition.top;
                cssBottom   = (snapTo === 'bottom') ? 0 : '';
                cssLeft     = (nodePosition.left + nodePosition.width + margin);
                break;
            case 'bottom':
                cssTop      = (nodePosition.top + nodePosition.height + margin);
                cssLeft     = nodePosition.left;
                break;
            case 'left':
                cssTop      = (snapTo === 'bottom') ? '' : nodePosition.top;
                cssBottom   = (snapTo === 'bottom') ? 0 : '';
                cssLeft     = (nodePosition.left - layerPosition.width - margin);
                break;
            }

            return {
                top: cssTop,
                right: cssRight,
                bottom: cssBottom,
                left: cssLeft
            };
        }

        /**
         * Make the given Node draggable
         *
         * @param {String} [classSelector]
         *  The css-selector for the element that should be draggable
         */
        function makeDraggable(classSelector) {
            var ele = $(menuNode).find('.' + classSelector);

            ele.enableTracking();
            ele.on('tracking:start', moveStartHandler);
            ele.on('tracking:move', moveHandler);
        }

        function windowResizeHandler() {
            self.positionResetHandler();
            self.setLayerMenuPosition();
        }

        // public methods -----------------------------------------------------

        /**
         * Helper to reset the Position of the menu only in definite situations
         */
        this.positionResetHandler = function () {
            // set new availableSize on resizing the window
            availableSizes = Utils.getNodePositionInPage(app.getView().getAppPaneNode());
            // allow to reset position of menu
            resetPosition = true;
            return this;
        };

        /**
         * Set the new position of the menu
         */
        this.setLayerMenuPosition = setPosition || function () {

            var menu = menuNode,
                ApplicationWindow = rootContainerNode;

            // on first call, save the layer-position
            if (layerPosition === null && self.isVisible()) {
                layerPosition = Utils.getChildNodePositionInNode($(ApplicationWindow), $(menu));
            }

            // If 'resetPosition' is "true" and if the menu is visible
            if (resetPosition && self.isVisible()) {
                resetPosition = false;

                // check if the anchor-node exists, otherwise hide the menu
                if ($(anchorSelection).length === 0) {
                    self.hide();
                    return;
                }

                // if custom root container node is specified, get positions relative from it, otherwhise from <body>
                nodePosition = ApplicationWindow ? Utils.getChildNodePositionInNode(ApplicationWindow, $(anchorSelection))
                    : Utils.getNodePositionInPage($(anchorSelection));

                var choosedPosition = choosePosition();

                var cssProps = getCssProps(choosedPosition);

                scrollHor = layerPosition.width > availableSizes.width;
                scrollVer = layerPosition.height > (availableSizes.height + availableSizes.bottom);
                cssProps.overflowX = scrollHor ? 'scroll' : 'hidden';
                cssProps.overflowY = scrollVer ? 'scroll' : 'hidden';

                cssProps.width = (scrollHor) ? Math.min(availableSizes.width, layerPosition.width + Utils.SCROLLBAR_WIDTH):'';
                cssProps.height = (scrollVer) ? Math.min(availableSizes.height, layerPosition.height + Utils.SCROLLBAR_HEIGHT):'';

                self.setNodePosition(cssProps);
            }
            return this;
        };

        // initialization -----------------------------------------------------

        // add css-class to style the menu
        menuNode.addClass('layer-menu');

        // add da header-bar to the popup-container (inter alia to drag the menu)
        var viewComponent = menuNode.find('.view-component:eq(0)'),
            dragHead = $('<div>').addClass('dragHead');

        // add closer icon if desired
        if (closer) {
            var closerNode = $('<a>').addClass('layer-menu-closer'),
                closerIcon = $(Forms.createIconMarkup('fa-times'));
            closerNode.on('click', function () {
                self.trigger('layermenu:userclose');
                self.hide();
            });
            closerNode.append(closerIcon);
            dragHead.append(closerNode);
        }

        dragHead.append($('<div>').addClass('header-label dragme').text(layerTitle));
        dragHead.append($('<div>').addClass('clear'));

        viewComponent.append(dragHead);

        // in case, make it draggable
        if (draggable) {
            makeDraggable('dragme');
        }

        // on popup:show, set the menu-position
        this.on('popup:show', _.bind(this.positionResetHandler, this));
        this.on('popup:show', _.bind(this.setLayerMenuPosition, this));

        this.on('popup:show', function () {
            // on window:resize reset the menu-position
            $(window).on('resize', windowResizeHandler);
        });
        this.on('popup:hide', function () {
            // off window:resize reset the menu-position
            $(window).off('resize', windowResizeHandler);
            // return focus to application view
            app.getView().grabFocus();
        });

        // destructor
        this.registerDestructor(function () {
            this.hide();
            menuNode.remove();
            self = null;
        });

    }}); // class LayerMenu

    // class LayerMenuButton ==================================================

    /**
     * A button control with a combined LayerMenu containing a complete
     * self-contained view component instance with completely independent
     * controls.
     *
     * @constructor
     *
     * @extends Group
     *
     * @param {BaseApplication} app
     *  The application containing this context menu.
     *
     * @param {String} layerTitle
     *  The title on top of the menu(-layer)
     *
     * @param {Object} [initOptions]
     *  Supports all options of the LayerMenu class.
     *
     */
	BaseControls.LayerMenuButton = Group.extend({ constructor: function (app, layerTitle, initOptions) {

        var // the layerMenu-class
            menu = null,
            // node of the edit-pane-button
            menuButtonNode = $(Forms.createButtonMarkup(initOptions)),

            anchor = Utils.getOption(initOptions, 'anchorSelection', menuButtonNode);

        // base constructor ---------------------------------------------------

        Group.call(this, initOptions);
        CaptionMixin.call(this, menuButtonNode);
        WidthMixin.call(this, menuButtonNode, initOptions);
        menu = new BaseControls.LayerMenu(app, layerTitle, Utils.extendOptions(initOptions, {anchorSelection: anchor}));
        MenuMixin.call(this, menu, Utils.extendOptions(initOptions, { button: menuButtonNode, caret: false, tabNavigate: true }));

        // initialization -----------------------------------------------------

        // Add Button to Menu
        this.addChildNodes(menuButtonNode);

        menu.on('popup:show', function () {
            menu.grabFocus();
        });

        // destroy all class members
        this.registerDestructor(function () {
            app = initOptions = menuButtonNode = menu = null;
        });

    }}); // class LayerMenuButton

    // exports ================================================================

    return BaseControls;

});
