/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/baseframework/app/basecontroller',
    ['io.ox/office/tk/utils',
     'io.ox/office/tk/keycodes',
     'io.ox/office/tk/dialogs',
     'io.ox/office/tk/object/triggerobject',
     'io.ox/office/tk/object/baseobject'
    ], function (Utils, KeyCodes, Dialogs, TriggerObject, BaseObject) {

    'use strict';

    // class BaseController ===================================================

    /**
     * A controller contains a collection of items, consisting of unique key
     * and value, and providing arbitrary getter and setter methods for their
     * values.
     *
     * Triggers the following events:
     * - 'change:items'
     *      After the current state (value and/or enabled state) of at least
     *      one controller item has been changed after the controller has
     *      updated itself or the method BaseController.update() has been
     *      invoked. Event listeners receive data about the changed items in a
     *      map with item identifiers as keys, and objects containing the
     *      optional properties 'value' and 'enable' as values.
     *
     * @constructor
     *
     * @extends TriggerObject
     *
     * @param {BaseApplication} app
     *  The application that has created this controller instance.
     *
     * @param {Object} [initOptions]
     *  Optional parameters:
     *  @param {Number} [initOptions.updateDelay=50]
     *      The delay for the debounced Controller.update() method, in
     *      milliseconds. Must not be less than 50.
     */
    function BaseController(app, initOptions) {

        var // self reference
            self = this,

            // all the little controller items
            items = {

                'app/quit': {
                    // quit in a timeout (otherwise this controller becomes invalid while still running)
                    set: function () { _.defer(function () { app.quit(); }); }
                },

                // disabled during import, enabled afterwards
                'app/imported': {
                    enable: function () { return app.isImportFinished(); }
                },

                'document/download': {
                    // enabled in read-only mode, disabled during import
                    parent: 'app/imported',
                    set: function () { return app.downloadDocument(); }
                },

                'document/print': {
                    // enabled in read-only mode, disabled during import
                    parent: 'app/imported',
                    set: function () { return app.print(); },
                    shortcut: { keyCode: 'P', ctrlOrMeta: true }
                },

                'document/sendmail': {
                    // enabled in read-only mode, disabled during import
                    parent: 'app/imported',
                    set: function () { return app.sendMail(); }
                }
            },

            // the delay time for the debounced BaseController.update() method
            // Bug 30479: must NOT be 0, to be able give some space for deferred
            // focus handling in form controls, for example in text fields which
            // want to commit their value before the controller will reset it.
            updateDelay = Utils.getIntegerOption(initOptions, 'updateDelay', 50, 50),

            // cached item states, mapped by item keys
            stateCache = {},

            // number of item setters currently running (recursion counter)
            runningSetters = 0,

            // shortcut definitions for 'keydown', mapped by key code (for performance)
            keyShortcuts = {},

            // shortcut definitions for 'keypress', mapped by char code (for performance)
            charShortcuts = {},

            // dummy item singleton, used e.g. for undefined parent items
            dummyItem = null;

        // class Item ---------------------------------------------------------

        var Item = BaseObject.extend({ constructor: function (key, definition) {

            var // parent items whose values/states are needed to resolve the own value/state
                parentKeys = Utils.getOption(definition, 'parent'),
                // handler for enabled state (default to function that returns true)
                enableHandler = Utils.getFunctionOption(definition, 'enable', _.constant(true)),
                // handler for value getter (default to identity to forward first parent value)
                getHandler = Utils.getFunctionOption(definition, 'get', _.identity),
                // handler for value setter
                setHandler = Utils.getFunctionOption(definition, 'set'),
                // behavior for returning browser focus to application
                itemPreserveFocus = Utils.getBooleanOption(definition, 'preserveFocus', false),
                // custom focus target node or callback handler
                itemFocusTarget = Utils.getOption(definition, 'focusTarget');

            BaseObject.call(this);

            // private methods  - - - - - - - - - - - - - - - - - - - - - - - -

            /**
             * Returns the state descriptor of the specified parent item.
             */
            function getParentState(parentKey) {
                if (parentKey in items) { return items[parentKey].getState(); }
                Utils.warn('BaseController.Item.getParentState(): item "' + key + '" refers to unknown parent "' + parentKey + '"');
                return dummyItem.getState();
            }

            // methods  - - - - - - - - - - - - - - - - - - - - - - - - - - - -

            /**
             * Returns the unique key of this item.
             *
             * @returns {String}
             *  The unique key this item has been registered with.
             */
            this.getKey = function () {
                return key;
            };

            /**
             * Returns a raw descriptor for the current state of this item.
             *
             * @returns {Object}
             *  A state descriptor with the following properties:
             *  - {Boolean|Null} parentsEnabled
             *      The enabled state of all parents of this item. The value
             *      null represents the undetermined state (no 'enable'
             *      callback handlers found at the parents, or no parents
             *      defined).
             *  - {Boolean|Null} enabled
             *      The enabled state of this item. The value null represents
             *      the undetermined state (no 'enable' callback handlers
             *      found).
             *  - {Array} parentValues
             *      The values (results of all getter callbacks) of the parent
             *      items.
             *  - {Array} value
             *      The value (result of the getter callback) of the item.
             */
            this.getState = function () {

                // return cached state
                if (key in stateCache) { return stateCache[key]; }

                var // the states of all parent items
                    parentStates = _.map(parentKeys, getParentState),
                    // determine whether all parent items are enabled
                    parentsEnabled = _.all(parentStates, function (state) { return state.enabled; }),
                    // all parent values, as array
                    parentValues = _.pluck(parentStates, 'value'),
                    // create the state descriptor of this item
                    itemState = {
                        parentStates: parentStates,
                        parentsEnabled: parentsEnabled,
                        parentValues: parentValues,
                        enabled: false,
                        value: undefined
                    };

                // insert the state descriptor into the cache before resolving enabled state and value
                stateCache[key] = itemState;

                // resolve and store the value first (enable callback may want to query the own value)
                itemState.value = getHandler.apply(this, parentValues);

                // resolve and store the enabled state
                itemState.enabled = parentsEnabled && (enableHandler.apply(this, parentValues) === true);

                // return the state descriptor
                return itemState;
            };

            /**
             * Returns whether all parent items are enabled.
             *
             * @returns {Boolean}
             *  False, if at least one parent item is disabled, otherwise true.
             */
            this.areParentsEnabled = function () {
                return this.getState().parentsEnabled;
            };

            /**
             * Returns whether this item is effectively enabled.
             *
             * @returns {Boolean}
             *  False, if at least one parent item is disabled, or if the
             *  enable handler of this item has returned false; otherwise true.
             */
            this.isEnabled = function () {
                return this.getState().enabled;
            };

            /**
             * Returns the current value of the specified parent item.
             *
             * @param {Number} [index=0]
             *  The index of the parent item (if multiple parent items have
             *  been specified for this item).
             */
            this.getParentValue = function (index) {
                var parentState = this.getState().parentStates[_.isNumber(index) ? index : 0];
                return parentState ? parentState.value : undefined;
            };

            /**
             * Returns the current value of this item.
             */
            this.getValue = function () {
                return this.getState().value;
            };

            /**
             * Executes the setter function of this item (passing in the new
             * value), and moves the browser focus back to the application
             * pane.
             *
             * @param {Any} value
             *  The new value of the item.
             *
             * @param {Object} [options]
             *  Optional parameters. See method BaseController.executeItem()
             *  for details.
             *
             * @returns {jQuery.Promise}
             *  The Promise of a Deferred object that will be resolved or
             *  rejected according to the result of the item setter function.
             *  If the item is disabled, a rejected Promise will be returned.
             *  If the item setter returns a Promise, this Promise will be
             *  returned. Otherwise, the return value of the item setter
             *  function will be wrapped and returned in a resolved Promise.
             */
            this.execute = function (value, options) {

                var // the resulting promise
                    promise = null,
                    // whether the setter runs asynchronous code
                    pending = false,
                    // focus options after executing the item
                    focusOptions = _.extend({}, options);

                // mix item focus options and passed focus options
                focusOptions.preserveFocus = itemPreserveFocus || Utils.getBooleanOption(options, 'preserveFocus', false);
                focusOptions.focusTarget = Utils.getOption(options, 'focusTarget', itemFocusTarget);

                // bind focus target callback function to the item instance
                if (_.isFunction(focusOptions.focusTarget)) {
                    focusOptions.focusTarget = _.bind(focusOptions.focusTarget, this);
                }

                // do nothing if item is disabled (but accept undetermined state)
                if (this.isEnabled()) {

                    // execute the set handler
                    runningSetters += 1;
                    if (_.isFunction(setHandler)) {
                        promise = setHandler.call(this, value);
                    }

                    // convert result of the setter callback to a promise
                    promise = $.when(promise);
                    promise.always(function () {
                        runningSetters -= 1;
                    });

                    // show busy screen if setter is running asynchronously
                    // (but not for modal dialogs which have their own busy blocker)
                    pending = promise.state() === 'pending';
                    if (pending && !Dialogs.isModalDialogOpen()) {
                        app.getView().enterBusy({ delay: 500 });
                    }
                } else {
                    // item is disabled
                    promise = $.Deferred().reject();
                }

                // focus back to application
                if (!focusOptions.preserveFocus) {
                    promise.always(function () {
                        if (pending) {
                            // execute in a timeout, needed for dialogs which are closed *after* resolve/reject
                            app.executeDelayed(_.bind(self.grabFocus, self, focusOptions));
                        } else {
                            self.grabFocus(focusOptions);
                        }
                    });
                }

                // post-processing after the setter is finished
                Utils.logSelenium('time=' + _.now(), 'key=' + key, 'state=' + promise.state());
                if (pending) {
                    promise.always(function () {
                        app.getView().leaveBusy();
                        self.update();
                        Utils.logSelenium('time=' + _.now(), 'key=' + key, 'state=' + promise.state());
                    });
                } else {
                    self.update();
                }

                return promise;
            };

            // initialization - - - - - - - - - - - - - - - - - - - - - - - - -

            // always use an array of parent keys (as strings)
            parentKeys =
                _.isArray(parentKeys) ? _.filter(parentKeys, _.isString) :
                _.isString(parentKeys) ? [parentKeys] :
                null;

            // destroy all class members on destruction
            this.registerDestructor(function () {
                definition = parentKeys = null;
                enableHandler = getHandler = setHandler = null;
            });

        }}); // class Item

        // base constructor ---------------------------------------------------

        TriggerObject.call(this);

        // private methods ----------------------------------------------------

        /**
         * Clears all cached item results.
         */
        function clearStateCache() {
            stateCache = {};
        }

        /**
         * Handles 'keydown' and 'keypress' events and calls the setter of this
         * item, if it contains a matching keyboard shortcut definition.
         *
         * @param {jQuery.Event} event
         *  The jQuery event object. If a matching shortcut definition has been
         *  found, propagation of the event will be stopped, and the browser
         *  default action will be suppressed.
         */
        function keyHandler(event) {

            var // whether to stop propagation and prevent the default action
                stopPropagation = false;

            // executes the item setter defined in the passed shortcut
            function executeItemForShortcut(shortcut) {

                var // the definition data of the shortcut
                    definition = shortcut.definition,
                    // the current value to be passed to the item setter
                    value = null;

                // check whether to actually process the key event according to the event target node
                if (_.isString(definition.selector) && ($(event.target).closest(definition.selector, app.getWindow().nodes.outer[0]).length === 0)) {
                    return;
                }

                // bug 30096: do not allow to execute items while a blocking operation is still running
                if (runningSetters === 0) {
                    // call value resolver function, or take constant value
                    value = _.isFunction(definition.value) ? definition.value(self.getItemValue(shortcut.key)) : definition.value;
                    self.executeItem(shortcut.key, value, { sourceType: 'keyboard' });
                }

                // stop event propagation unless specified otherwise
                if (!Utils.getBooleanOption(definition, 'propagate', false)) {
                    stopPropagation = true;
                }
            }

            switch (event.type) {
            case 'keydown':
                // process all shortcut definitions for the key code in the passed event
                if (event.keyCode in keyShortcuts) {
                    _.each(keyShortcuts[event.keyCode], function (shortcut) {
                        // check if the additional modifier keys match the shortcut definition
                        if (KeyCodes.matchModifierKeys(event, shortcut.definition)) {
                            executeItemForShortcut(shortcut);
                        }
                    });
                }
                break;
            case 'keypress':
                // process all shortcut definitions for the char code in the passed
                // event, but ignore 'keypress' events in text fields
                if ((event.charCode in charShortcuts) && !$(event.target).is('input,textarea')) {
                    _.each(charShortcuts[event.charCode], executeItemForShortcut);
                }
                break;
            }

            return stopPropagation ? false : undefined;
        }

        /**
         * Initializes the key handlers for all registered keyboard shortcuts.
         */
        function initializeKeyHandler() {

            var // the root node of the application window
                windowNode = app.getWindow().nodes.outer;

            // Bug 27528: IE does not allow to prevent default actions of some
            // special key events (especially CTRL+P which always opens the Print
            // dialog, regardless whether the keydown event has been canceled in
            // the bubbling phase). The 'official' hack is to bind a keydown event
            // handler with IE's ancient 'attachEvent()' method, and to suppress
            // the default action by changing the key code in the event object.
            // (add smiley-with-big-eyes here). Note that the 'addEventListener()'
            // method does NOT work for this hack.
            // Bug 27528: IE 11 does not support the attachEvent() method anymore
            // (hack for bug 27528 will fail).
            if (_.browser.IE && _.isFunction(windowNode[0].attachEvent)) {
                windowNode[0].attachEvent('onkeydown', function (event) {
                    var result = keyHandler(_.extend(event, { target: event.srcElement }));
                    // modify the key code to prevent the browser default action
                    if (result === false) { event.keyCode = 0; }
                    return result;
                });
            } else {
                windowNode.on('keydown', keyHandler);
            }

            // register 'keypress' event listener for shortcuts
            windowNode.on('keypress', keyHandler);

            // prevent that Ctrl+A shortcut selects the entire HTML document
            windowNode.on('keydown', function (event) {
                if (!$(event.target).is('input,textarea') && KeyCodes.matchKeyCode(event, 'A', { ctrlOrMeta: true })) {
                    return false;
                }
            });
        }

        // methods ------------------------------------------------------------

        /**
         * Adds the definition for a new item to this controller.
         *
         * @param {String} key
         *  The key of the new item.
         *
         * @param {Object} definition
         *  A map with callback functions defining the behavior of the item.
         *  All callback functions will be executed in the context of the Item
         *  class instance. The following attributes are supported:
         *  @param {String|String[]} [definition.parent]
         *      The key of an item that will be used to calculate intermediate
         *      results for the getter function and enabler function (see
         *      below), or an array of item keys used as parents for this item.
         *      The key feature of parent items is that if a controller enables
         *      or updates multiple items at once, the getter or enabler of the
         *      same parent item registered at multiple items will be executed
         *      exactly once before the first item getter or enabler is called,
         *      and its result will be cached and passed to all item getters or
         *      enablers that are using this parent item.
         *  @param {Function} [definition.enable]
         *      Predicate function returning true if the item is enabled, and
         *      false otherwise. If one or more parent items have been
         *      specified (see above) and at least one of them returned false,
         *      the item is in disabled state already, and this function will
         *      not be called anymore. If the item does not specify an enable
         *      handler, it defaults to the value true, so the enabled state is
         *      dependent on the parent items only.
         *  @param {Function} [definition.get]
         *      Getter function returning the current value of the item. Can be
         *      omitted for one-way action items (actions without a return
         *      value). If one ore more parent items have been specified (see
         *      above), the cached return values of their getters will be
         *      passed to this getter, in the same order as specified. May
         *      return null to indicate an ambiguous state. Defaults to a
         *      function that returns undefined; or, if parent items have been
         *      registered, that returns the value of the first parent.
         *  @param {Function} [definition.set]
         *      Setter function changing the value of an item to the first
         *      parameter of the setter. Can be omitted for read-only items.
         *      Defaults to an empty function. May return a Deferred object or
         *      a Promise to indicate that the setter executes asynchronous
         *      code. In this case, the application window will be blocked and
         *      a busy indicator will be shown while the setter is running.
         *  @param {Object|Array} [definition.shortcut]
         *      One or multiple keyboard shortcut definitions. If the window
         *      root node of the application receives a 'keydown' or 'keypress'
         *      event that matches a shortcut definition, the setter function
         *      of this item will be executed. Can be as single shortcut
         *      definition, or an array of shortcut definitions. Each
         *      definition object supports the following attributes:
         *      - {Number|String} [shortcut.charCode]
         *          If specified, the shortcut definition will be matched
         *          against 'keypress' events, and the value represents the
         *          numeric code point of a Unicode character, or the Unicode
         *          character as a string.
         *      - {Number|String} [shortcut.keyCode]
         *          If specified, the shortcut definition will be matched
         *          against 'keydown' events, and the value represents the
         *          numeric key code, or the upper-case name of a key code, as
         *          defined in the static class KeyCodes. Be careful with digit
         *          keys, for example, the number 9 matches the TAB key
         *          (KeyCodes.TAB), but the string '9' matches the digit '9'
         *          key (KeyCodes['9'] with the key code 57).
         *      - {Boolean|Null} [shortcut.shift=false]
         *          If set to true, the SHIFT key must be pressed when the
         *          'keydown' events is received. If set to false (or omitted),
         *          the SHIFT key must not be pressed. If set to null, the
         *          current state of the SHIFT key will be ignored. Has no
         *          effect when evaluating 'keypress' events.
         *      - {Boolean|Null} [shortcut.alt=false]
         *          If set to true, the ALT key must be pressed when the
         *          'keydown' events is received. If set to false (or omitted),
         *          the ALT key must not be pressed. If set to null, the
         *          current state of the ALT key will be ignored. Has no effect
         *          when evaluating 'keypress' events.
         *      - {Boolean|Null} [shortcut.ctrl=false]
         *          If set to true, the CTRL key must be pressed when the
         *          'keydown' events is received. If set to false (or omitted),
         *          the CTRL key must not be pressed. If set to null, the
         *          current state of the CTRL key will be ignored. Has no
         *          effect when evaluating 'keypress' events.
         *      - {Boolean|Null} [shortcut.meta=false]
         *          If set to true, the META key must be pressed when the
         *          'keydown' events is received. If set to false (or omitted),
         *          the META key must not be pressed. If set to null, the
         *          current state of the META key will be ignored. Has no
         *          effect when evaluating 'keypress' events.
         *      - {Boolean} [shortcut.altOrMeta=false]
         *          Convenience option that if set to true, matches if either
         *          the ALT key, or the META key are pressed. Has the same
         *          effect as defining two separate shortcuts, one with the
         *          'shortcut.alt' option set to true, and one with the
         *          'shortcut.meta' option set to true, while keeping the other
         *          option false. Must not be used in a shortcut definition
         *          where these options are set explicitly.
         *      - {Boolean} [shortcut.ctrlOrMeta=false]
         *          Convenience option that if set to true, matches if either
         *          the CTRL key, or the META key are pressed. Has the same
         *          effect as defining two separate shortcuts, one with the
         *          'shortcut.ctrl' option set to true, and one with the
         *          'shortcut.meta' option set to true, while keeping the other
         *          option false. Must not be used in a shortcut definition
         *          where these options are set explicitly.
         *      - {Any|Function} [shortcut.value]
         *          The value that will be passed to the setter function of
         *          this item. If multiple shortcuts are defined for an item,
         *          each shortcut definition may define its own value. If this
         *          option contains a function, it will receive the current
         *          value of the controller item as first parameter, and its
         *          return value will be passed to the setter function.
         *      - {Boolean} [shortcut.propagate=false]
         *          If set to true, the event will propagate up to the DOM root
         *          element, and the browser will execute its default action.
         *          If omitted or set to false, the event will be cancelled
         *          immediately after calling the setter function.
         *      - {String} [shortcut.selector]
         *          A CSS selector used to restrict the keyboard shortcut to
         *          specific DOM nodes in the application DOM. The target node
         *          of the keyboard event has to match the selector, or has to
         *          be a descendant node of a DOM node matching the selector.
         *          Otherwise, the event will be ignored completely, and the
         *          browser will execute its default action.
         *  @param {Boolean} [definition.preserveFocus=false]
         *      Whether to return the browser focus to the application pane
         *      after executing the setter function of this item (the default
         *      behavior). If set to true, the current browser focus will not
         *      be changed. Otherwise, the focus will return to the application
         *      pane after the setter function has finished (will wait for
         *      asynchronous setters before returning the focus).
         *  @param {HTMLElement|jQuery|Object|Function} [options.focusTarget]
         *      A DOM node that will receive the browser focus after the item
         *      has been executed, an object that provides a grabFocus()
         *      method, or a callback function that returns a focus target
         *      dynamically. Default focus target is the view. Has no effect,
         *      if the option 'preserveFocus' is set to true.
         *
         * @returns {BaseController}
         *  A reference to this controller instance.
         */
        this.registerDefinition = function (key, definition) {
            if (_.isString(key) && key && _.isObject(definition)) {
                items[key] = new Item(key, definition);
                if (_.isObject(definition.shortcut)) {
                    _.chain(definition.shortcut).getArray().each(function (shortcut) {
                        var keyCode = Utils.getOption(shortcut, 'keyCode'),
                            charCode = Utils.getOption(shortcut, 'charCode');
                        if (_.isString(keyCode)) {
                            keyCode = KeyCodes[keyCode] || 0;
                        }
                        if (_.isNumber(keyCode) && (keyCode > 0)) {
                            (keyShortcuts[keyCode] || (keyShortcuts[keyCode] = [])).push({ key: key, definition: shortcut });
                        }
                        if (_.isString(charCode)) {
                            charCode = charCode.charCodeAt(0);
                        }
                        if (_.isNumber(charCode) && (charCode > 0)) {
                            (charShortcuts[charCode] || (charShortcuts[charCode] = [])).push({ key: key, definition: shortcut });
                        }
                    });
                }
            }
            return this;
        };

        /**
         * Adds definitions for multiple items to this controller.
         *
         * @param {Object} definitions
         *  A map of key/definition pairs for all new items. Each item will be
         *  defined by calling the method BaseController.registerDefinition().
         *  See this method for more details.
         *
         * @returns {BaseController}
         *  A reference to this controller instance.
         */
        this.registerDefinitions = function (definitions) {
            _.each(definitions, function (definition, key) {
                this.registerDefinition(key, definition);
            }, this);
            return this;
        };

        /**
         * Refreshes the states and values of all registered items, and
         * triggers a 'change:items' event.
         *
         * @returns {BaseController}
         *  A reference to this controller.
         */
        this.update = app.createDebouncedMethod(function () { return self; }, function () {

            // collect states of all matching items in resultCache
            clearStateCache();

            if (_.browser.IE) {

                // early break out, 'getState' is only important for hud,
                // but freezing to long can kill 'realtime' and the browser itself,
                // so we cancel it after 1sec.
                // (more than 3sec could also mean 5sec. or more, because we have to check later)

                var startTime = _.now();
                _.find(items, function(item) {
                    item.getState();
                    if ((_.now() - startTime) > 3000) { return true; }
                });
            } else {
                _.invoke(items, 'getState');
            }

            // notify all listeners (with locked GUI refresh for performance,
            // view elements are listening to controller items to change their visibility)
            app.getView().lockPaneLayout(function () {
                self.trigger('change:items', stateCache);
            });

        }, { delay: updateDelay, maxDelay: updateDelay * 5 });

        /**
         * Returns the current enabled state of the specified item.
         *
         * @param {String} key
         *  The key of the item.
         *
         * @returns {Boolean}
         *  Whether the specified item exists and is enabled.
         */
        this.isItemEnabled = function (key) {
            // do not clear the cache if currently running in a set handler
            if (runningSetters === 0) { clearStateCache(); }
            return (key in items) && items[key].isEnabled();
        };

        /**
         * Returns the current value of the specified item.
         *
         * @param {String} key
         *  The key of the item.
         *
         * @returns {Any}
         *  The current value of the item, or undefined, if the item does not
         *  exist.
         */
        this.getItemValue = function (key) {
            // do not clear the cache if currently running in a set handler
            if (runningSetters === 0) { clearStateCache(); }
            return (key in items) ? items[key].getValue() : undefined;
        };

        /**
         * Triggers a controller item manually. Executes the setter function of
         * the item associated to the specified key, passing in the new value.
         *
         * @param {String} key
         *  The key of the item to be changed.
         *
         * @param {Any} value
         *  The new value of the item.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {String} [options.sourceType='custom']
         *      The cause of the item invocation. Can be one of the following
         *      types:
         *      - 'click': The item has been executed due to a button click, or
         *          another comparable GUI action such as a tap on a touch
         *          device.
         *      - 'keyboard': The item has been executed due to a standard
         *          keyboard event, e.g. pressing the ENTER key on buttons or
         *          text fields, pressing the SPACE key on buttons, or pressing
         *          the TAB key in modified text fields; or a keyboard shortcut
         *          as defined for the respective controller item.
         *      - 'custom': (default) Any other cause.
         *  @param {Boolean} [options.preserveFocus=false]
         *      If set to true, the browser focus will not be modified after
         *      executing the item, regardless of the focus mode configured for
         *      that item.
         *  @param {HTMLElement|jQuery|Object|Function} [options.focusTarget]
         *      A DOM node that will receive the browser focus after the item
         *      has been executed, an object that provides a grabFocus()
         *      method, or a callback function that returns a focus target
         *      dynamically. Overrides the default focus target defined for the
         *      executed item (option 'focusTarget' in the item definition).
         *      Default focus target is the view. Has no effect, if the option
         *      'preserveFocus' is set to true.
         *  @param {jQuery.Event} [options.changeEvent]
         *      Set by the framework when executing a controller item after
         *      activating a GUI control element. Represents the event object
         *      of the 'group:change' event as triggered by instances of the
         *      Group class.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved or rejected
         *  according to the result of the item setter function. If the item is
         *  disabled or does not exist, a rejected Promise will be returned. If
         *  the item setter returns a Promise, this Promise will be returned.
         *  Otherwise, the return value of the item setter function will be
         *  wrapped and returned in a resolved Promise.
         */
        this.executeItem = function (key, value, options) {

            // execute setter of existing item
            if (key in items) {
                // do not clear cache if called recursively from another setter
                if (runningSetters === 0) { clearStateCache(); }
                return items[key].execute(value, options);
            }

            // item does not exist: return focus to application
            this.grabFocus(options);
            return $.Deferred().reject();
        };

        /**
         * Moves the browser focus back to the application pane, or to a custom
         * focus target node, if specified in the passed options.
         *
         * @param {Object} [options]
         *  Optional parameters. See method BaseController.executeItem() for
         *  details.
         *
         * @returns {BaseController}
         *  A reference to this instance.
         */
        this.grabFocus = function (options) {

            var preserveFocus = Utils.getBooleanOption(options, 'preserveFocus', false),
                focusTarget = Utils.getOption(options, 'focusTarget'),
                changeEvent = Utils.getOption(options, 'changeEvent'),
                sourceNode = Utils.getOption(changeEvent, 'sourceNode');

            // keep current focus, if 'preserveFocus' is set
            // Bug 28214: Focus back to application if source GUI element is hidden
            // now. If the source GUI element is hidden now, IE has already changed
            // the 'document.activeElement' property, so it cannot be used to detect
            // visibility. Therefore, all group instances now insert the source DOM
            // node into the 'group:change' event.
            if (preserveFocus && (!sourceNode || $(sourceNode).is(':visible'))) {
                return this;
            }

            // resolve callback function to focus target
            if (_.isFunction(focusTarget)) {
                focusTarget = focusTarget.call(self, Utils.getStringOption(options, 'sourceType', 'custom'));
            }

            // move focus to specified target, or to the view
            if (_.isObject(focusTarget) && _.isFunction(focusTarget.grabFocus) && _.isFunction(focusTarget.isVisible) && focusTarget.isVisible()) {
                focusTarget.grabFocus();
            } else if (((focusTarget instanceof $) || (focusTarget instanceof HTMLElement)) && $(focusTarget).is(':visible')) {
                $(focusTarget).focus();
            } else {
                app.getView().grabFocus();
            }
            return this;
        };

        // initialization -----------------------------------------------------

        // dummy item singleton, used e.g. for undefined parent items
        dummyItem = new Item('\x00');

        // register item definitions
        this.registerDefinitions(items);

        // update once after import (successful and failed)
        app.onImport(function () {
            initializeKeyHandler();
            self.listenTo(app.getView(), 'refresh:layout', function () { self.update(); });
            self.update();
        });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            _.invoke(items, 'destroy');
            self = items = null;
        });

    } // class BaseController

    // exports ================================================================

    // derive this class from class TriggerObject
    return TriggerObject.extend({ constructor: BaseController });

});
