/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/tk/popup/basepopup',
    ['io.ox/office/tk/utils',
     'io.ox/office/tk/object/triggerobject'
    ], function (Utils, TriggerObject) {

    'use strict';

    // class BasePopup ========================================================

    /**
     * Wrapper class for a DOM node used as pop-up window, shown on top of the
     * application window, and relative to an arbitrary DOM node.
     *
     * Instances of this class trigger the following events:
     * - 'popup:beforeshow'
     *      Before the pop-up node will be shown.
     * - 'popup:show'
     *      After the pop-up node has been shown.
     * - 'popup:beforehide'
     *      Before the pop-up node will be hidden.
     * - 'popup:hide'
     *      After the pop-up node has been hidden.
     * - 'popup:beforelayout'
     *      Before the contents of the pop-up node, the absolute position of
     *      the anchor node, or the browser window size has changed, and the
     *      position and size of the pop-up node needs to be adjusted.
     * - 'popup:layout'
     *      After the contents of the pop-up node, the absolute position of the
     *      anchor node or the browser window size has changed, and the
     *      position and size of the pop-up node has been adjusted (in
     *      auto-layout mode).
     * - 'popup:beforeclear'
     *      Before the pop-up node will be cleared.
     *
     * @constructor
     *
     * @extends TriggerObject
     *
     * @param {Object} [initOptions]
     *  Initial options for the pop-up node. The following options are
     *  supported:
     *  @param {String} [initOptions.classes]
     *      Additional CSS classes that will be set at the pop-up root node.
     *  @param {Boolean} [initOptions.autoLayout=true]
     *      If set to true or omitted, the pop-up node will be positioned and
     *      sized automatically next to the anchor node. If the available space
     *      is not sufficient for the contents of the pop-up node, scroll bars
     *      will be displayed. If set to false, the pop-up node must be sized
     *      and positioned manually inside event handlers of the events
     *      'popup:beforelayout' and 'popup:layout'.
     *  @param {Object|HTMLElement|jQuery|Function} [initOptions.anchor]
     *      The specification of an anchor rectangle used to calculate the
     *      position and size of the pop-up node in automatic layout mode (see
     *      option 'autoLayout' above). The pop-up node will be positioned in a
     *      way so that it does not cover the anchor rectangle. Can be one of
     *      the following values:
     *      - A constant rectangle (an object with the properties 'left',
     *          'top', 'width', and 'height'). The position is interpreted
     *          absolutely in the entire HTML document.
     *      - The reference to a DOM element, or a jQuery collection wrapping a
     *          DOM element, whose position and size will be used as anchor
     *          rectangle. While displaying the pop-up node, its position will
     *          be updated dynamically in case the position and/or size of the
     *          anchor node changes, e.g. due to scrolling.
     *      - A callback function that will be invoked before displaying the
     *          pop-up node, and that must return any value described above.
     *  @param {Object|HTMLElement|jQuery|String|Function} [initOptions.border]
     *      The specification of an outer boundary rectangle used to calculate
     *      the position and size of the pop-up node in automatic layout mode
     *      (see option 'autoLayout' above). Used in combination with the
     *      anchor rectangle (see option 'anchor' above). The pop-up node will
     *      not cover the border rectangle, but will still be positioned
     *      according to the offset of the anchor rectangle (for example, if
     *      the pop-up node will be shown below the anchor rectangle,
     *      vertically it will not cover the border rectangle, and horizontally
     *      it will be aligned to the actual position of the anchor rectangle).
     *      Can be one of the following values:
     *      - A constant rectangle (an object with the properties 'left',
     *          'top', 'width', and 'height'). The position is interpreted
     *          absolutely in the entire HTML document.
     *      - The reference to a DOM element, or a jQuery collection wrapping a
     *          DOM element, whose position and size will be used as border
     *          rectangle. Usually, this is an ancestor of the anchor node.
     *          While displaying the pop-up node, its position will be updated
     *          dynamically in case the position and/or size of the border node
     *          changes, e.g. due to scrolling.
     *      - A jQuery string selector that will select a descendant element of
     *          the anchor node (see option 'anchor' above). Must not be used,
     *          if the anchor specification does not resolve to a DOM node.
     *      - A callback function that will be invoked before displaying the
     *          pop-up node, and that must return any value described above.
     *          Additionally, null or undefined can be returned to indicate to
     *          use the current anchor position instead (see option 'anchor').
     *  @param {String} [initOptions.position='bottom top']
     *      The preferred position of the pop-up node relative to the anchor
     *      position. Used in automatic layout mode only (see option
     *      'autoLayout' above). Must be a space-separated list with the tokens
     *      'left', 'right', 'top', and 'bottom'. The implementation picks the
     *      best position according to the available space around the anchor
     *      position. If more than one position can be used (without needing to
     *      shrink the pop-up node), the first position token in this list will
     *      be used.
     *  @param {String} [initOptions.align='leading']
     *      The preferred alignment of the pop-up node relative to the anchor
     *      position. The value 'leading' tries to align the left borders (if
     *      shown above or below) or top border (if shown left or right) of the
     *      pop-up node and the anchor position, the value 'trailing' tries to
     *      align the right/bottom borders, and the value 'center' tries to
     *      place the pop-up node centered to the anchor position. Used in
     *      automatic layout mode only (see option 'autoLayout' above).
     *  @param {String} [options.restrictSize='none']
     *      If set to 'width', the width of the content node will be restricted
     *      to the available horizontal space, and the contents may grow
     *      vertically. A vertical scroll bar will be shown, if the contents do
     *      not fit into the available space. If set to 'height', the height of
     *      the content node will be restricted to the available vertical
     *      space, and the contents may grow horizontally. A horizontal scroll
     *      bar will be shown, if the contents do not fit into the available
     *      space. Otherwise, the size of the content node will not be
     *      restricted.
     *  @param {Number} [initOptions.minSize=1]
     *      If specified, the minimum size to be reserved for the pop-up node,
     *      if either its width or height is restricted (see option
     *      'restrictSize' above). This option has no effect, if the size is
     *      not restricted.
     *  @param {Number} [initOptions.maxSize=0x7FFFFFFF]
     *      If specified, the maximum size to be reserved for the pop-up node,
     *      if either its width or height is restricted (see option
     *      'restrictSize' above). This option has no effect, if the size is
     *      not restricted.
     *  @param {Number} [initOptions.windowPadding=6]
     *      The minimum distance between pop-up node and the borders of the
     *      browser window.
     *  @param {Number} [initOptions.anchorPadding=1]
     *      The distance between pop-up node and anchor position (or border
     *      position if specified, see option 'border' above).
     *  @param {String} [initOptions.role]
     *      The ARIA role attribute that will be set at the content DOM node.
     *  @param {HTMLElement|jQuery} [initOptions.rootContainerNode]
     *      The DOM container node for the pop-up node. By default, the pop-up
     *      node will be appended into the <body> element of the page.
     *      ATTENTION: Currently, it is not possible to use automatic layout
     *      (see option 'autoLayout') when using a custom container node.
     *      Manual positioning and sizing of the pop-up node is necessary.
     */
    function BasePopup(initOptions) {

        var // self reference (the Group instance)
            self = this,

            // the root pop-up DOM node (attribute unselectable='on' is needed for IE to prevent focusing the node when clicking on a scroll bar)
            rootNode = $('<div class="io-ox-office-main popup-container" unselectable="on">'),

            // the content node inside the root node used to calculate the effective node size
            contentNode = $('<div class="popup-content">').appendTo(rootNode),

            // automatic position and size of the pop-up node
            autoLayout = Utils.getBooleanOption(initOptions, 'autoLayout', true),

            // the anchor position specification (rectangle, DOM node, or callback function)
            anchorSpec = Utils.getOption(initOptions, 'anchor', null),

            // the border position specification (rectangle, DOM node, selector, or callback function)
            borderSpec = Utils.getOption(initOptions, 'border', null),

            // preferred border of the group node for the drop-down menu
            position = Utils.getStringOption(initOptions, 'position', 'bottom top'),

            // alignment of the pop-up node relative to the anchor node
            align = Utils.getStringOption(initOptions, 'align', 'leading'),

            // whether to restrict the width or height of the content node
            restrictSize = Utils.getStringOption(initOptions, 'restrictSize', 'none'),

            // minimum size for the content node in restricted size mode
            minSize = Utils.getIntegerOption(initOptions, 'minSize', 1, 1),

            // maximum size for the content node in restricted size mode
            maxSize = Utils.getIntegerOption(initOptions, 'maxSize', 0x7FFFFFFF, minSize),

            // distance between pop-up node and the borders of the browser window
            windowPadding = Utils.getIntegerOption(initOptions, 'windowPadding', 6, 0),

            // distance between pop-up node and anchor node
            anchorPadding = Utils.getIntegerOption(initOptions, 'anchorPadding', 1),

            // aria role
            ariaRole = Utils.getStringOption(initOptions, 'role', ''),

            // the container node for this pop-up node
            rootContainerNode = $(Utils.getOption(initOptions, 'rootContainerNode', window.document.body)),

            // an interval timer to refresh the the pop-up node while it is visible
            refreshTimer = null,

            // the DOM element that was last focused
            lastActiveElement = null,

            // the width and height of the virtual keyboard on touch devices
            keyboardWidth = 0,
            keyboardHeight = 0,

            // last position and size of the anchor node
            lastAnchorPosition = null;

        // base constructor ---------------------------------------------------

        TriggerObject.call(this);

        // private methods ----------------------------------------------------

        /**
         * Returns current position of the passed specifier as contained in the
         * constructor options 'anchor' or 'border', relative to the document
         * page, or the visible area of the browser window.
         *
         * @param {Object|HTMLElement|jQuery} positionValue
         *  The position specification, as passed to the constructor options
         *  'anchor' or 'border'.
         *
         * @param {Boolean} [windowArea=false]
         *  If set to true, the position returned by this method is relative to
         *  the visible area of the browser window, instead of the entire
         *  document page.
         *
         * @returns {Object}
         *  The rectangle of the passed DOM node.
         */
        function resolvePosition(positionValue, windowArea) {

            var // the size of the entire document page
                pageSize = Utils.getPageSize(),
                // the visible area of the browser window
                visibleArea = windowArea ? self.getVisibleWindowArea() : null,
                // the resulting position
                position = null;

            // resolve position of DOM element or jQuery object
            if ((positionValue instanceof HTMLElement) || (positionValue instanceof $)) {
                if ($(positionValue).length > 0) {
                    position = Utils.getNodePositionInPage(positionValue);
                }
            } else if (_.isObject(positionValue)) {
                position = _.clone(positionValue);
                // add right/bottom distance to the rectangle
                position.right = pageSize.width - position.width - position.left;
                position.bottom = pageSize.height - position.height - position.top;
            }

            // restrict to visible area of the browser window
            if (position && visibleArea) {
                _.each(['left', 'top', 'right', 'bottom'], function (border) {
                    position[border] -= visibleArea[border];
                });
            }

            return position;
        }

        /**
         * Returns the current anchor position, relative to the document page,
         * or the visible area of the browser window.
         *
         * @param {Boolean} [windowArea=false]
         *  If set to true, the position returned by this method is relative to
         *  the visible area of the browser window, instead of the entire
         *  document page.
         *
         * @returns {Object|Null}
         *  The current anchor position, if the 'anchor' option has been set in
         *  the constructor options, otherwise null.
         */
        function resolveAnchorPosition(windowArea) {

            var // resolve callback function to result
                anchorValue = _.isFunction(anchorSpec) ? anchorSpec.call(self) : anchorSpec;

            return resolvePosition(anchorValue, windowArea);
        }

        /**
         * Returns the outer boundary position, relative to the document page,
         * or the visible area of the browser window. If no border position has
         * been specified, returns the anchor position instead.
         *
         * @param {Boolean} [windowArea=false]
         *  If set to true, the position returned by this method is relative to
         *  the visible area of the browser window, instead of the entire
         *  document page.
         *
         * @returns {Object|Null}
         *  The current border position, if the 'border' option has been set in
         *  the constructor options, otherwise the anchor position, as returned
         *  by the method getAnchorPosition().
         */
        function resolveBorderPosition(windowArea) {

            var // resolve callback function to result
                borderValue = _.isFunction(borderSpec) ? borderSpec.call(self) : borderSpec,
                // the anchor DOM node as specified by the 'anchor' option
                anchorNode = null,
                // the descendant DOM node of the anchor node as specified by the 'anchor' option
                borderNode = null,
                // the resulting position
                position = null;

            // use string selector to receive a descendant of the anchor node
            if (_.isString(borderValue)) {
                anchorNode = _.isFunction(anchorSpec) ? anchorSpec.call(self) : anchorSpec;
                if ((anchorNode instanceof HTMLElement) || (anchorNode instanceof $)) {
                    borderNode = $(anchorNode).find(borderValue);
                }
                borderValue = (borderNode && (borderNode.length > 0)) ? borderNode : null;
            }

            // try to resolve to a position
            position = resolvePosition(borderValue, windowArea);

            // fall-back to anchor position
            return position ? position : resolveAnchorPosition(windowArea);
        }

        /**
         * Initializes the size and position of the pop-up node in automatic
         * layout mode.
         */
        function setAutomaticPosition() {

            var // anchor position in the document page
                anchorPosition = resolveAnchorPosition(),
                // border position in the document page
                borderPosition = resolveBorderPosition(),
                // the sizes available at every side of the anchor node
                availableSizes = self.getAvailableSizes(),
                // size of the pop-up node contents
                contentWidth = 0, contentHeight = 0,
                // resulting width and height available for the menu node
                availableWidth = 0, availableHeight = 0,
                // whether scroll bars will be shown
                scrollHor = false, scrollVer = false,
                // new CSS properties of the pop-up node
                cssProps = {},
                // the preferred side of the anchor node used for the pop-up node
                preferredPos = null, maxRatio = 0,
                // the size of the entire document page
                pageSize = Utils.getPageSize();

            // restricts the size of the content node
            function updateContentNodeSize(size, useMin) {

                var // the exact node size (floating-points)
                    contentSize = null;

                switch (restrictSize) {
                case 'width':
                    contentNode.css({ minWidth: useMin ? minSize : '', maxWidth: (minSize <= size.width) ? Math.min(maxSize, size.width) : '', minHeight: '', maxHeight: '' });
                    break;
                case 'height':
                    contentNode.css({ minWidth: '', maxWidth: '', minHeight: useMin ? minSize : '', maxHeight: (minSize <= size.height) ? Math.min(maxSize, size.height) : '' });
                    break;
                default:
                    contentNode.css({ minWidth: '', maxWidth: '', minHeight: '', maxHeight: '' });
                }

                // Browsers usually report node sizes rounded to integers when using properties like
                // 'offsetWidth', 'scrollWidth', etc. (example: content width of a tooltip according
                // to its text contents is 123.45 pixels, but browsers report 123 pixels). When using
                // this rounded width at the outer node, the text will break into two text lines.
                contentSize = Utils.getExactNodeSize(contentNode);
                contentWidth = Math.ceil(contentSize.width);
                contentHeight = Math.ceil(contentSize.height);
            }

            // set all sizes to auto to get effective size of the pop-up node contents
            rootNode.css({ left: 0, top: '', right: '', bottom: '-100%', width: 'auto', height: 'auto' });
            contentNode.css({ width: 'auto', height: 'auto' });

            // calculate the ratio of the menu node being visible at every side of the group
            _.each(availableSizes, function (size) {
                updateContentNodeSize(size, true);
                size.ratio = ((contentWidth > 0) && (contentHeight > 0)) ? (Math.min(size.width / contentWidth, 1) * Math.min(size.height / contentHeight, 1)) : 0;
            });

            // prefer the best side that can take the menu node, in the given order
            _.any(position.split(/\s+/), function (pos) {
                var ratio = availableSizes[pos].ratio;
                if (ratio >= 1) {
                    preferredPos = pos;
                    return true;
                } else if (ratio > maxRatio) {
                    preferredPos = pos;
                    maxRatio = ratio;
                }
            });
            if (!preferredPos) { return; }

            // extract available width and height
            availableWidth = availableSizes[preferredPos].width;
            availableHeight = availableSizes[preferredPos].height;

            // fix size of content node to prevent any reformatting when shrinking the root node
            updateContentNodeSize(availableSizes[preferredPos], false);
            contentNode.css({ width: contentWidth, height: contentHeight });

            // first part of the position of the pop-up node (keep in visible area
            // of browser window, but keep outside of the border node)
            switch (preferredPos) {
            case 'top':
                cssProps.bottom = pageSize.height - borderPosition.top + anchorPadding;
                break;
            case 'bottom':
                cssProps.top = borderPosition.top + borderPosition.height + anchorPadding;
                break;
            case 'left':
                cssProps.right = pageSize.width - borderPosition.left + anchorPadding;
                break;
            case 'right':
                cssProps.left = borderPosition.left + borderPosition.width + anchorPadding;
                break;
            }

            // add scroll bars if available width or height is not sufficient
            scrollHor = contentWidth > availableWidth;
            scrollVer = contentHeight > availableHeight;
            cssProps.width = Math.min(availableWidth, contentWidth + (scrollVer ? Utils.SCROLLBAR_WIDTH : 0));
            cssProps.height = Math.min(availableHeight, contentHeight + (scrollHor ? Utils.SCROLLBAR_HEIGHT : 0));
            cssProps.overflowX = scrollHor ? 'scroll' : 'hidden';
            cssProps.overflowY = scrollVer ? 'scroll' : 'hidden';

            // second part of the position of the pop-up node (keep in visible area of browser window)
            if (Utils.isVerticalPosition(preferredPos)) {
                switch (align) {
                case 'trailing':
                    cssProps.left = Utils.minMax(anchorPosition.left + anchorPosition.width, borderPosition.left, borderPosition.left + borderPosition.width) - cssProps.width;
                    break;
                case 'center':
                    cssProps.left = Utils.minMax(anchorPosition.left + anchorPosition.width / 2, borderPosition.left, borderPosition.left + borderPosition.width) - cssProps.width / 2;
                    break;
                default:
                    cssProps.left = Utils.minMax(anchorPosition.left, borderPosition.left, borderPosition.left + borderPosition.width);
                }
            } else {
                switch (align) {
                case 'trailing':
                    cssProps.top = Utils.minMax(anchorPosition.top + anchorPosition.height, borderPosition.top, borderPosition.top + borderPosition.height) - cssProps.height;
                    break;
                case 'center':
                    cssProps.top = Utils.minMax(anchorPosition.top + anchorPosition.height / 2, borderPosition.top, borderPosition.top + borderPosition.height) - cssProps.height / 2;
                    break;
                default:
                    cssProps.top = Utils.minMax(anchorPosition.top, borderPosition.top, borderPosition.top + borderPosition.height);
                }
            }

            if (Utils.LayoutLogger.isLoggingActive()) {
                Utils.LayoutLogger.info('BasePopup.setAutomaticPosition()');
                Utils.LayoutLogger.log(' anchor: ' + JSON.stringify(anchorPosition).replace(/"/g, '').replace(/,/g, ', '));
                Utils.LayoutLogger.log(' border: ' + JSON.stringify(borderPosition).replace(/"/g, '').replace(/,/g, ', '));
                Utils.LayoutLogger.log(' window: ' + JSON.stringify(self.getVisibleWindowArea()).replace(/"/g, '').replace(/,/g, ', '));
                Utils.LayoutLogger.log(' avail:  ' + JSON.stringify(availableSizes).replace(/"/g, '').replace(/,/g, ', '));
                Utils.LayoutLogger.log(' prefer: ' + preferredPos);
                Utils.LayoutLogger.log(' result: ' + JSON.stringify(cssProps).replace(/"/g, '').replace(/,/g, ', '));
            }

            // apply final CSS formatting
            self.setNodePosition(cssProps);
        }

        /**
         * Checks and updates the position of the pop-up node, according to the
         * current position of the anchor node (a scrollable parent node may
         * have been scrolled). Additionally, hides the pop-up node
         * automatically when the anchor node becomes inaccessible (also
         * indirectly, e.g. when hiding any parent nodes).
         */
        function refreshNode(force) {

            var // whether to update the position of the pop-up node
                refreshPosition = !!force,
                // the position of the anchor node in the visible area of the window
                anchorPosition = null;

            // do nothing, if the pop-up node is not visible currently
            if (!self.isVisible()) { return; }

            // on touch devices, try to detect the size of the virtual keyboard
            if (Modernizr.touch && (lastActiveElement !== document.activeElement)) {
                lastActiveElement = document.activeElement;
                refreshPosition = true;
                // if a text field is focused, try to scroll the browser window
                // right/down to get the size of the virtual keyboard
                if ($(lastActiveElement).is('input,textarea')) {
                    var pageX = window.pageXOffset, pageY = window.pageYOffset;
                    window.scrollTo(10000, 10000);
                    // if the pageX- and pageY-Offset higher than the window-dimensions
                    // there is no keyboard detected. Switch keyboard width/height to "0"
                    keyboardWidth = (window.pageXOffset > window.innerWidth)?0:window.pageXOffset;
                    keyboardHeight = (window.pageYOffset > window.innerHeight)?0:window.pageYOffset;
                    window.scrollTo(pageX, pageY);
                } else {
                    keyboardWidth = keyboardHeight = 0;
                    // FIX for IOS 8 (Safari)
                    // IOS 8 let the body scroll up.
                    // Don't know why, there is no keyboard or something else.
                    // However, we scroll back to zero.
                    window.scrollTo(0,0);
                }
            }

            // check whether the position of the anchor node has changed
            // (for example, by scrolling a parent node of the anchor node)
            if (!refreshPosition) {
                anchorPosition = resolveAnchorPosition(true);
                refreshPosition = !_.isEqual(lastAnchorPosition, anchorPosition);
                lastAnchorPosition = anchorPosition;
            }

            // refresh the position of the pop-up node, notify all listeners
            if (refreshPosition) {
                self.trigger('popup:beforelayout');
                if (autoLayout) { setAutomaticPosition(); }
                self.trigger('popup:layout');
            }
        }

        /**
         * Handles 'resize' events of the browser window.
         */
        function windowResizeHandler() {
            refreshNode(false);
        }

        // public methods -----------------------------------------------------

        /**
         * Returns the position and size of the visible area in the browser
         * window, relative to the entire document page.
         *
         * @param {Number} [padding=0]
         *  Additional padding size that will be removed from the resulting
         *  visible area.
         */
        this.getVisibleWindowArea = function (padding) {

            var // the visible area of the browser window
                visibleArea = { left: window.pageXOffset, top: window.pageYOffset },
                // the size of the entire document page
                pageSize = Utils.getPageSize();

            padding = _.isNumber(padding) ? Math.max(padding, 0) : 0;
            visibleArea.left += padding;
            visibleArea.top += padding;

            // The properties window.inner(Width|Height) for their own are not
            // reliable when virtual keyboard on touch devices is visible. This
            // 'hack' works for applications with fixed page size.
            visibleArea.width = Math.max(Math.min(window.innerWidth, pageSize.width - keyboardWidth) - 2 * padding, 0);
            visibleArea.height = Math.max(Math.min(window.innerHeight, pageSize.height - keyboardHeight) - 2 * padding, 0);

            // add right/bottom distance to the result
            visibleArea.right = pageSize.width - visibleArea.width - visibleArea.left;
            visibleArea.bottom = pageSize.height - visibleArea.height - visibleArea.top;

            return visibleArea;
        };

        /**
         * Returns the available sizes for the pop-up node at every side of the
         * anchor node.
         *
         * @internal
         *  Intended to be used by sub classes of this class for custom
         *  rendering of the pop-up node.
         *
         * @returns {Object}
         *  The sizes (objects with 'width' and 'height' properties) available
         *  for the pop-up node, mapped by the border side names 'left',
         *  'right', 'top', and 'bottom'.
         */
        this.getAvailableSizes = function () {

            var // boundary position in the visible area of the browser window
                borderPosition = resolveBorderPosition(true),
                // visible area of the browser window, reduced by border padding
                availableArea = this.getVisibleWindowArea(windowPadding),

                // vertical space above, below, left of, and right of the anchor node
                totalPadding = windowPadding + anchorPadding,
                availableAbove = Utils.minMax(borderPosition.top - totalPadding, 0, availableArea.height),
                availableBelow = Utils.minMax(borderPosition.bottom - totalPadding, 0, availableArea.height),
                availableLeft = Utils.minMax(borderPosition.left - totalPadding, 0, availableArea.width),
                availableRight = Utils.minMax(borderPosition.right - totalPadding, 0, availableArea.width);

            return {
                top: { width: availableArea.width, height: availableAbove },
                bottom: { width: availableArea.width, height: availableBelow },
                left: { width: availableLeft, height: availableArea.height },
                right: { width: availableRight, height: availableArea.height }
            };
        };

        /**
         * Sets the position of the pop-up node according to the passed CSS
         * positioning attributes. The resulting position of the pop-up node
         * will be adjusted according to the current visible area of the
         * browser window.
         *
         * @internal
         *  Intended to be used by sub classes of this class for custom
         *  rendering of the pop-up node.
         *
         * @param {Object} cssProps
         *  The CSS positioning properties for the pop-up node, relative to the
         *  document page. Must contain one horizontal offset (either 'left' or
         *  'right'), and one vertical offset (either 'top' or 'bottom').
         *  Optionally, may contain the new size ('width' and/or 'height') of
         *  the pop-up node. May contain any other CSS properties.
         *
         * @returns {BasePopup}
         *  A reference to this instance.
         */
        this.setNodePosition = function (cssProps) {

            var // visible area of the browser window, reduced by border padding
                availableArea = this.getVisibleWindowArea(windowPadding),
                // the width and height of the pop-up node (may be missing in passed properties)
                nodeWidth = _.isNumber(cssProps.width) ? cssProps.width : rootNode.outerWidth(),
                nodeHeight = _.isNumber(cssProps.height) ? cssProps.height : rootNode.outerHeight();

            // restrict all existing positioning attributes to the visible area
            cssProps.left = _.isNumber(cssProps.left) ? Utils.minMax(cssProps.left, availableArea.left, availableArea.left + availableArea.width - nodeWidth) : '';
            cssProps.right = _.isNumber(cssProps.right) ? Utils.minMax(cssProps.right, availableArea.right, availableArea.right + availableArea.width - nodeWidth) : '';
            cssProps.top = _.isNumber(cssProps.top) ? Utils.minMax(cssProps.top, availableArea.top, availableArea.top + availableArea.height - nodeHeight) : '';
            cssProps.bottom = _.isNumber(cssProps.bottom) ? Utils.minMax(cssProps.bottom, availableArea.bottom, availableArea.bottom + availableArea.height - nodeHeight) : '';

            rootNode.css(cssProps);
            return this;
        };

        /**
         * Returns the root DOM node of this pop-up, as jQuery object.
         *
         * @returns {jQuery}
         *  The root DOM node of this pop-up.
         */
        this.getNode = function () {
            return rootNode;
        };

        /**
         * Returns whether the pop-up node is currently visible.
         *
         * @returns {Boolean}
         *  Whether the pop-up node is currently visible.
         */
        this.isVisible = function () {
            return rootNode.parent().length > 0;
        };

        /**
         * Shows the pop-up node.
         *
         * @returns {BasePopup}
         *  A reference to this instance.
         */
        this.show = function () {

            // do nothing if the pop-up node is already open
            if (this.isVisible()) { return this; }

            // show and position the pop-up node, notify all listeners
            this.trigger('popup:beforeshow');
            rootContainerNode.append(rootNode);
            lastActiveElement = lastAnchorPosition = null;
            refreshNode(true);
            this.trigger('popup:show');

            // 'show' event handles must not hide the menu
            if (!this.isVisible()) {
                Utils.error('BasePopup.show(): "show" event handler has hidden the pop-up menu');
                return this;
            }

            // start a timer that regularly refreshes the position of the pop-up node, and
            // hides the pop-up node automatically when the anchor node becomes inaccessible
            refreshTimer = window.setInterval(function () { refreshNode(false); }, Modernizr.touch ? 200 : 50);

            // enable window resize handler which recalculates position and size of the pop-up node
            $(window).on('resize', windowResizeHandler);

            return this;
        };

        /**
         * Hides the pop-up node.
         *
         * @returns {BasePopup}
         *  A reference to this instance.
         */
        this.hide = function () {

            // do nothing if the pop-up node is not visible
            if (!this.isVisible()) { return this; }

            // stop the automatic refresh timer
            window.clearInterval(refreshTimer);
            $(window).off('resize', windowResizeHandler);

            // hide the pop-up node, notify all listeners
            this.trigger('popup:beforehide');
            rootNode.detach();
            this.trigger('popup:hide');

            return this;
        };

        /**
         * Toggles the visibility of the pop-up node.
         *
         * @returns {BasePopup}
         *  A reference to this instance.
         */
        this.toggle = function () {
            return this.isVisible() ? this.hide() : this.show();
        };

        /**
         * Returns the content node of this pop-up node, as jQuery object. All
         * contents to be shown in the pop-up have to be inserted into this
         * content node.
         *
         * @returns {jQuery}
         *  The content node of this pop-up.
         */
        this.getContentNode = function () {
            return contentNode;
        };

        /**
         * Returns whether this pop-up node contains any nodes.
         *
         * @returns {Boolean}
         *  Whether this pop-up node contains any nodes.
         */
        this.hasContents = function () {
            return contentNode[0].childNodes.length > 0;
        };

        /**
         * Removes all contents from this pop-up node.
         *
         * @returns {BasePopup}
         *  A reference to this instance.
         */
        this.clearContents = function () {
            if (this.hasContents()) {
                this.trigger('popup:beforeclear');
                contentNode.empty();
                this.refresh();
            }
            return this;
        };

        /**
         * Appends the passed DOM elements to the contents of this pop-up node.
         *
         * @param {HTMLElement|jQuery|String} [nodes ...]
         *  The DOM nodes to be inserted into this pop-up node, as plain HTML
         *  element, as jQuery collection, or as HTML mark-up string.
         *
         * @returns {BasePopup}
         *  A reference to this instance.
         */
        this.appendContentNodes = function () {
            contentNode.append.apply(contentNode, arguments);
            this.refresh();
            return this;
        };

        /**
         * Inserts the passed HTML mark-up into the content node of this pop-up
         * (all existing contents will be removed).
         *
         * @param {String} markup
         *  The HTML mark-up to be inserted into this pop-up node, as string.
         *
         * @returns {BasePopup}
         *  A reference to this instance.
         */
        this.setContentMarkup = function (markup) {
            this.clearContents(); // clean up and trigger events
            contentNode[0].innerHTML = markup;
            this.refresh();
            return this;
        };

        /**
         * Updates the size and position of the pop-up node immediately.
         *
         * @returns {BasePopup}
         *  A reference to this instance.
         */
        this.refreshImmediately = function () {
            refreshNode(true);
            return this;
        };

        /**
         * Updates the size and position of the pop-up node debounced.
         */
        this.refresh = _.debounce(function () {
            if (self && !self.destroyed) { refreshNode(true); }
        });

        /**
         * Scrolls the passed child node into the visible area of this pop-up
         * node.
         *
         * @param {HTMLElement|jQuery} childNode
         *  The DOM element that will be made visible by scrolling this pop-up
         *  node. If this object is a jQuery collection, uses the first node it
         *  contains.
         *
         * @param {Object} [options]
         *  Optional parameters. Supports all options that are supported by the
         *  method Utils.scrollToChildNode().
         *
         * @returns {BasePopup}
         *  A reference to this instance.
         */
        this.scrollToChildNode = function (childNode, options) {
            if (this.isVisible()) {
                Utils.scrollToChildNode(rootNode, childNode, options);
            }
            return this;
        };

        /**
         * Sets the anchor specification of this pop up.
         * This function can be used if the anchor could not be specified during construction time.
         *
         * @param {Object|HTMLElement|jQuery|Function} anchor
         * Identical to the initOptions.anchor specification of this class.
         */
        this.setAnchorSpec = function (anchor) {
            anchorSpec = anchor || null;
        };

        // initialization -----------------------------------------------------

        // additional CSS classes passed to constructor
        rootNode.addClass(Utils.getStringOption(initOptions, 'classes', ''));

        // disable context menu
        rootNode.on('contextmenu', function (event) {
            if (!$(event.target).is('input,textarea')) { return false; }
        });

        // disable dragging of controls (otherwise, it is possible to drag buttons and other controls around)
        rootNode.on('contextmenu drop dragstart dragenter dragexit dragover dragleave', false);

        // log layout events
        Utils.LayoutLogger.logEvent(this, 'popup:beforeshow popup:show popup:beforehide popup:hide popup:beforelayout popup:layout popup:beforeclear');

        // ARIA role
        if (ariaRole.length > 0) {
            contentNode.attr('role', ariaRole);
        }

        // destroy all class members on destruction
        this.registerDestructor(function () {
            this.hide();
            rootNode.remove();
            self = rootNode = contentNode = anchorSpec = borderSpec = null;
        });

    } // class BasePopup

    // exports ================================================================

    // derive this class from class TriggerObject
    return TriggerObject.extend({ constructor: BasePopup });

});
