/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/tk/control/radiolist',
    ['io.ox/office/tk/utils',
     'io.ox/office/tk/forms',
     'io.ox/office/tk/control/group',
     'io.ox/office/tk/control/captionmixin',
     'io.ox/office/tk/control/widthmixin',
     'io.ox/office/tk/control/menumixin',
     'io.ox/office/tk/popup/listmenu'
    ], function (Utils, Forms, Group, CaptionMixin, WidthMixin, MenuMixin, ListMenu) {

    'use strict';

    // class RadioList ========================================================

    /**
     * Creates a drop-down list control used to hold a set of radio buttons.
     *
     * @constructor
     *
     * @extends Group
     * @extends CaptionMixin
     * @extends WidthMixin
     * @extends MenuMixin
     *
     * @param {Object} [initOptions]
     *  Optional parameters. Supports all options of the Group base class, of
     *  the MenuMixin mix-in class, the ListMenu class used for the drop-down
     *  menu, all formatting options for button elements supported by the
     *  method Forms.createButtonMarkup(), and all options of the mix-in class
     *  WidthMixin. Additionally, the following options are supported:
     *  @param {Boolean|Function} [initOptions.highlight=false]
     *      If set to true, the drop-down button will be highlighted, if an
     *      existing list item in the drop-down menu is active. The drop-down
     *      button will not be highlighted, if the value of this group is
     *      undetermined (value null), or if there is no option button present
     *      with the current value of this group. If set to false, the
     *      drop-down button will never be highlighted, even if a list item in
     *      the drop-down menu is active. If set to a function, it will be
     *      called every time after a list item will be activated. The
     *      drop-down button will be highlighted if this handler function
     *      returns true. Receives the value of the selected/activated list
     *      item as first parameter (also if this value does not correspond to
     *      any existing list item). Will be called in the context of this
     *      radio group instance.
     *  @param {Any} [initOptions.toggleValue]
     *      If set to a value different to undefined and null, the option
     *      button that is currently active can be clicked to be switched off.
     *      In that case, this radio group will activate the button associated
     *      to the value specified in this option, and the action handler will
     *      return this value instead of the value of the button that has been
     *      switched off.
     *  @param {Any} [initOptions.splitValue]
     *      If set to a value different to undefined and null, a separate push
     *      button will be inserted before the drop-down button, representing
     *      the value of this option.
     *  @param {String} [initOptions.dataValue]
     *      A string that will be inserted into the 'data-value' attribute of
     *      the split button node. If omitted, the JSON string representation
     *      of the 'splitValue' option will be used instead (all double-quote
     *      characters will be removed from the string though), unless the
     *      value is a function.
     *  @param {String} [initOptions.updateCaptionMode='all']
     *      Specifies how to update the caption of the drop-down button when a
     *      list item in the drop-down menu has been activated. Supports the
     *      keywords 'all' and 'none', or a space separated list containing the
     *      string tokens 'icon', 'iconStyle', 'label', and 'labelStyle'.
     *      - 'icon': copies the icon of the list item to the drop-down button.
     *      - 'iconStyle': copies the icon CSS formatting of the list item to
     *          the drop-down button.
     *      - 'label': copies the label of the list item to the drop-down
     *          button.
     *      - 'labelStyle': copies the label CSS formatting of the list item to
     *          the label of the drop-down button.
     *      The keyword 'all' is equivalent to the string 'icon iconStyle label
     *      labelStyle'. The keyword 'none' is equivalent to the empty string.
     *  @param {Object} [initOptions.dropDownVersion]
     *      If specified, the RadioList will known, how it should act, if it
     *      will shrinked in a dropdown-menu.
     *  @param {Object} [initOptions.smallerVersion]
     *      If specified, the RadioList will known, how it should act, if there
     *      will be not enough free place to show normal view
     */
    function RadioList(initOptions) {

        var // self reference
            self = this,

            // whether to highlight the drop-down menu button
            highlight = Utils.getBooleanOption(initOptions, 'highlight', false),

            // custom predicate callback for drop-down menu button highlighting
            highlightHandler = Utils.getFunctionOption(initOptions, 'highlight'),

            // fall-back value for toggle click
            toggleValue = Utils.getOption(initOptions, 'toggleValue'),

            // value of the split button
            splitValue = Utils.getOption(initOptions, 'splitValue'),

            // which parts of a list item caption will be copied to the menu button
            updateCaptionMode = Utils.getStringOption(initOptions, 'updateCaptionMode', 'all'),

            // separate split button in split mode
            splitButton = (!_.isUndefined(splitValue) && !_.isNull(splitValue)) ? $(Forms.createButtonMarkup(initOptions)) : $(),

            // the options for the drop-down button
            menuButtonOptions = ((splitButton.length === 0) ? initOptions : { tooltip: Utils.getOption(initOptions, 'tooltip', '') }),

            // the drop-down button
            menuButton = $(Forms.createButtonMarkup(menuButtonOptions)),

            // the target node for the caption (icon and label text)
            captionButton = (splitButton.length > 0) ? splitButton : menuButton,

            // the drop-down menu instance (must be created after Group base constructor!)
            menu = null,

            // smaller version of radiolist (to fit in smaller resolutions)
            smallerVersion = Utils.getOption(initOptions, 'smallerVersion', false),

            // saves the (old) styles in case of switching to small version
            savedStyles = {},

            // special options for a drop-down-version of this radiolist
            dropDownVersion = Utils.getObjectOption(initOptions, 'dropDownVersion');

        // base constructors --------------------------------------------------

        Group.call(this, initOptions);
        CaptionMixin.call(this, captionButton);
        WidthMixin.call(this, captionButton, initOptions);
        menu = new ListMenu(Utils.extendOptions(initOptions, { anchor: this.getNode() }));
        MenuMixin.call(this, menu, Utils.extendOptions(initOptions, { button: menuButton }));

        // private methods ----------------------------------------------------

        /**
         * Scrolls the drop-down menu to make the specified list item visible.
         */
        function scrollToListItem(buttonNode) {
            if ((buttonNode.length > 0) && self.isMenuVisible()) {
                menu.scrollToChildNode(buttonNode);
            }
        }

        /**
         * Sets the minimum width of the menu to the width of the group.
         */
        function popupBeforeShowHandler() {
            menu.getSectionNodes().css({ minWidth: self.getNode().outerWidth() });
        }

        /**
         * Scrolls to the selected list item if available.
         */
        function popupShowHandler() {
            scrollToListItem(menu.getSelectedItemNodes().first());
        }

        /**
         * Clean up data stored in button nodes, before the entire menu node
         * will be cleared.
         */
        function popupBeforeClearHandler() {
            menu.getItemNodes().data('options', null).off();
        }

        /**
         * Activates an option button in this radio group.
         *
         * @param {Any} value
         *  The value associated to the button to be activated. If set to null,
         *  does not activate any button (ambiguous state).
         */
        function itemUpdateHandler(value) {

            var // activate an option button
                selectedButtons = menu.selectMatchingItemNodes(value),
                // the options used to create the list item button
                buttonOptions = selectedButtons.first().data('options') || {},
                // the options used to set the caption of the drop-down menu button
                captionOptions = _.clone(self.getOptions()),
                // whether the drop-down button will be highlighted
                isHighlighted = false;

            // update visibility of the list items
            menu.getItemNodes().each(function () {
                var options = $(this).data('options'),
                    visible = Utils.getFunctionOption(options, 'visible');
                visible = _.isFunction(visible) ? visible.call(self, value) : Utils.getBooleanOption(options, 'visible', null);
                if (_.isBoolean(visible)) { Forms.showNodes(this, visible); }
            });

            // update visibility of the section nodes (hide sections without visible items)
            menu.getSectionNodes().each(function () {
                var visible = $(this).find(Forms.BUTTON_SELECTOR + Forms.VISIBLE_SELECTOR).length > 0;
                Forms.showNodes(this, visible);
            });

            // highlight the drop-down button (call custom handler, if available)
            if (_.isNull(value)) {
                isHighlighted = null;
            } else if (_.isFunction(highlightHandler)) {
                isHighlighted = highlightHandler.call(self, value);
            } else {
                isHighlighted = highlight && (selectedButtons.length > 0);
            }
            Forms.selectButtonNodes(menuButton.add(captionButton), isHighlighted);

            // update the caption of the drop-down menu button
            if (updateCaptionMode.length > 0) {
                _.each(updateCaptionMode, function (name) {
                    if ((name === 'label') && (selectedButtons.length > 0)) {
                        captionOptions.label = Forms.getCaptionText(selectedButtons);
                    } else if (name in buttonOptions) {
                        captionOptions[name] = buttonOptions[name];
                    }
                });
                self.setCaption(captionOptions);
            }

            // update the data-value attribute of the group from all selected items
            self.getNode().attr('data-value', selectedButtons.map(function () {
                return $(this).attr('data-value');
            }).get().join(','));

            // prepare version for drop-down-view of this element
            if (dropDownVersion) { createDropDownVersion(); }
        }

        /**
		 * Activates/Deactivates a smaller version of the radiolist
		 *
		 * @param {Boolean} value
         *  decides wether the radiolist should be small or wide
         */
        function smallCaptionButton(value){
            // show the small version
            if (value === true) {
                // set new css if exists
                if (_.isObject(smallerVersion.css)) {
                    if (_.isEmpty(savedStyles.css)) { savedStyles.css = captionButton.attr('style'); }
                    captionButton.css(smallerVersion.css);
                }

                // set new/other label
                if (!_.isEmpty(smallerVersion.label)) {
                    if (_.isEmpty(savedStyles.label)) { savedStyles.label = captionButton.find('> .caption > span').text(); }
                    captionButton.find('> .caption > span').text(smallerVersion.label);
                }

                // set new/other icon
                if (!_.isEmpty(smallerVersion.icon)) {
                    var icon = captionButton.find('> .caret-node > i');
                    savedStyles.icon = icon.attr('class');
                    icon.removeAttr('class');
                    icon.addClass(smallerVersion.icon);
                }

                // hide text of label (if set)
                if (smallerVersion.hideLabel) { captionButton.addClass('hideLabel'); }

            // show the default
            } else {
                // re-set the old (saved) styles
                if (_.has(savedStyles, 'css') && !_.isNull(savedStyles.css)) {
                    captionButton.removeAttr('style');
                    captionButton.attr('style',savedStyles.css);
                    savedStyles.css = null;
                }

                // re-set the old (saved) label
                if (_.has(savedStyles, 'label') && !_.isNull(savedStyles.label)) {
                    captionButton.find('> .caption > span').text(savedStyles.label);
                    savedStyles.label = null;
                }

                // re-set default icon
                if (_.has(savedStyles, 'icon') && !_.isNull(savedStyles.icon)) {
                    var icon = captionButton.find('> .caret-node > i');
                    icon.removeAttr('class');
                    icon.addClass(savedStyles.icon);
                    savedStyles.icon = null;
                }

                // show text of label (if set)
                if (smallerVersion.hideLabel) { captionButton.removeClass('hideLabel'); }
            }
        }

        /**
         * Modifies the captionbutton for the dropdown-view
         */
        function createDropDownVersion(){
            if (_.has(dropDownVersion, 'label') && captionButton.find('> .caption > .drop-down').length === 0) {
                captionButton.find('> .caption').append(
                    Forms.createSpanMarkup(dropDownVersion.label, {
                        classes: 'drop-down'
                    })
                );
            }
        }

        // methods ------------------------------------------------------------

        /**
         * Adds a new section to the drop-down menu, if it does not exist yet.
         *
         * @param {String} sectionId
         *  The unique identifier of the menu section.
         *
         * @param {Object} [options]
         *  Optional parameters. Supports all options that are supported by the
         *  method ListMenu.createSectionNode().
         *
         * @returns {RadioList}
         *  A reference to this instance.
         */
        this.createMenuSection = function (sectionId, options) {
            menu.createSectionNode(sectionId, options);
            return this;
        };

        /**
         * Returns all option buttons as jQuery collection.
         *
         * @returns {jQuery}
         *  The collection with all existing option buttons.
         */
        this.getOptionButtons = function () {
            return menu.getItemNodes();
        };

        /**
         * Adds a new option button to this radio list.
         *
         * @param {Any} value
         *  The unique value associated to the option button. MUST NOT be null.
         *
         * @param {Object} [options]
         *  Optional parameters. Supports all options supported by the method
         *  ListMenu.createItemNode(). Additionally, the following options are
         *  supported:
         *  @param {Boolean|Function} [options.visible=true]
         *      Whether the list item is visible. If set to a function, it will
         *      be invoked every time this radio list will be refreshed. It
         *      receives the current value of this control as passed to the
         *      update handler, and must return a Boolean value whether the
         *      list item will be visible.
         *
         * @returns {RadioList}
         *  A reference to this instance.
         */
        this.createOptionButton = function (value, options) {
            menu.createItemNode(value, options).data('options', options);
            return this;
        };

        /**
         * Removes an option button from this radio list.
         *
         * @param {Any} value
         *  The unique value associated to the option button. MUST NOT be null.
         *
         * @returns {RadioList}
         *  A reference to this instance.
         */
        this.deleteOptionButton = function (value) {
            menu.deleteItemNodes(value);
            return this;
        };

        /**
         * Overwrites the base-methods (group.js) to
         * activate/deactivate the small version of the button
         */
        this.activateSmallVersion = function(){
            if (_.isObject(smallerVersion)) {
                smallCaptionButton(true);
            }
        };
        this.deactivateSmallVersion = function(){
            if (_.isObject(smallerVersion)) {
                smallCaptionButton(false);
            }
        };

        // initialization -----------------------------------------------------

        // add the split button to the group
        if (splitButton.length > 0) {
            Forms.setButtonValue(splitButton, splitValue, initOptions);
            this.addChildNodes(splitButton);
            splitButton.on('click', function (event) {
                self.triggerChange(Forms.getButtonValue(this), { sourceEvent: event });
            });
            // convert ENTER and SPACE keys to click events
            Forms.setButtonKeyHandler(splitButton);
        }

        // add the drop-down button to the group
        this.addChildNodes(menuButton);

        // initialize caption update mode
        switch (updateCaptionMode) {
        case 'all':
            updateCaptionMode = ['icon', 'iconStyle', 'label', 'labelStyle'];
            break;
        case 'none':
            updateCaptionMode = [];
            break;
        default:
            updateCaptionMode = updateCaptionMode.split(/\s+/);
        }

        // highlight list entries
        this.registerUpdateHandler(itemUpdateHandler);

        // handle click events in the list
        menu.getNode().on('click', Forms.BUTTON_SELECTOR, function (event) {
            var toggleClick = Forms.isSelectedNode(this) && !_.isNull(toggleValue) && !_.isUndefined(toggleValue),
                value = toggleClick ? toggleValue : Forms.getButtonValue(this);
            self.triggerChange(value, { sourceEvent: event });
        });

        // register event handlers
        this.listenTo(menu, 'popup:beforeshow', popupBeforeShowHandler);
        this.listenTo(menu, 'popup:show', popupShowHandler);
        this.listenTo(menu, 'popup:beforeclear', popupBeforeClearHandler);
        this.listenTo(menu, 'create:item', function () { self.refresh(); });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            popupBeforeClearHandler(); // clean up data stored in button nodes

            initOptions = self = menuButton = splitButton = menuButtonOptions = highlightHandler = menu = null;
        });

    } // class RadioList

    // exports ================================================================

    // derive this class from class Group
    return Group.extend({ constructor: RadioList });

});
