/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/tk/control/radiogroup',
    ['io.ox/office/tk/utils',
     'io.ox/office/tk/forms',
     'io.ox/office/tk/control/group'
    ], function (Utils, Forms, Group) {

    'use strict';

    // class RadioGroup =======================================================

    /**
     * Creates a container element used to hold a set of radio buttons.
     *
     * @constructor
     *
     * @extends Group
     *
     * @param {Object} [initOptions]
     *  Optional parameters. Supports all options of the Group base class.
     *  Additionally, the following options are supported:
     *  @param {Any} [options.toggleValue]
     *      If set to a value different to null or undefined, the option button
     *      that is currently selected can be clicked to be switched off. In
     *      that case, this radio group will activate the button associated to
     *      the value specified in this option (if existing), and the action
     *      handler will return this value instead of the value of the button
     *      that has been switched off.
     *  @param {Function} [options.matcher=_.isEqual]
     *      A comparison function that returns whether a button element in this
     *      group should be selected, deselected, or kept unmodified according
     *      to a specific value. Receives an arbitrary value as first
     *      parameter, and the value of the current button element as second
     *      parameter. The function must return the Boolean value true to
     *      select the respective button element, the Boolean value false to
     *      deselect the button element, or any other value to leave the button
     *      element unmodified. If omitted, uses _.isEqual() which compares
     *      arrays and objects deeply, and does not skip any button element.
     *  @param {Boolean} [initOptions.sorted=false]
     *      If set to true, the button elements will be sorted according to the
     *      custom sort order that has been specified via the 'sortIndex'
     *      options of each inserted button (see method
     *      RadioGroup.createOptionButton() for details).
     *  @param {Object} [initOptions.smallerVersion]
     *      If specified, the RadioGroup will known, how it should act, if
     *      there will be not enough free place to show normal view
     */
    function RadioGroup(initOptions) {

        var // self reference
            self = this,

            // fall-back value for toggle click
            toggleValue = Utils.getOption(initOptions, 'toggleValue'),

            // comparator for list item values
            matcher = Utils.getFunctionOption(initOptions, 'matcher'),

            // whether the buttons will be sorted
            sorted = Utils.getBooleanOption(initOptions, 'sorted', false),

            // smaller version of group (to fit in smaller resolutions)
            smallerVersion = Utils.getOption(initOptions, 'smallerVersion', false);

        // base constructor ---------------------------------------------------

        Group.call(this, initOptions);

        // private methods ----------------------------------------------------

        /**
         * Activates an option button in this radio group.
         *
         * @param {Any} value
         *  The value associated to the button to be activated. If set to null,
         *  does not activate any button (ambiguous state), but shows all
         *  buttons in the ambiguous state.
         */
        function updateHandler(value) {

            var // all existing option buttons
                optionButtons = self.getOptionButtons();

            // set all buttons to ambiguous state for null value
            if (_.isNull(value)) {
                Forms.selectButtonNodes(optionButtons, null);
                return;
            }

            // deselect all buttons, select all buttons matching the current value
            Forms.selectMatchingButtonNodes(optionButtons, value, matcher);
        }

        /**
         * Modifies the RadioGroup for the dropdown-view
         *
         * @param {jQuery} buttonNode
         *  The jQuery-Object of the button(-node)
         * @param {Object} dropDownVersion
         *  The following options are supported:
         *  @param {String} [dropDownVersion.label]
         *      If specified, the radioGroup gets a parent label
         */
        function createDropDownVersion(buttonNode, dropDownVersion){
            if (_.has(dropDownVersion, 'label')) {
                buttonNode.find('.caption').append(
                    Forms.createSpanMarkup(dropDownVersion.label, {
                        classes: 'drop-down'
                    })
                );
            }
        }

        /**
         * Activates/Deactivates a smaller version of the RadioGroup
         *
         * @param {Boolean} value
         *  decides wether the RadioGroup should be small or wide
         */
        function smallRadioGroup(value){
            // show the small version
            if (value === true) {
                // if a pendant exists
                if (_.has(smallerVersion, 'pendant')) {
                    // show pendant group
                    smallerVersion.pendant.show();
                    // hide default group
                    self.hide();
                }

            // show the default
            } else {
                // if pendant exists
                if (_.has(smallerVersion, 'pendant')) {
                    // hide pendant group
                    smallerVersion.pendant.hide();
                    // show default group
                    self.show();
                }
            }
        }

        // methods ------------------------------------------------------------

        /**
         * Returns all option buttons as jQuery collection.
         *
         * @returns {jQuery}
         *  The collection with all existing option buttons.
         */
        this.getOptionButtons = function () {
            return this.getNode().children('.option-button');
        };

        /**
         * Returns all option buttons that represent the specified value.
         *
         * @param {Any} value
         *  The value to filter the option buttons for.
         *
         * @returns {jQuery}
         *  The collection with all existing option buttons equal to the passed
         *  value.
         */
        this.findOptionButtons = function (value) {
            return Forms.filterButtonNodes(this.getOptionButtons(), value, matcher);
        };

        /**
         * Removes all option buttons from this radio group.
         *
         * @returns {RadioGroup}
         *  A reference to this instance.
         */
        this.clearOptionButtons = function () {
            this.getOptionButtons().data('options', null).remove();
            this.layout();
            return this;
        };

        /**
         * Adds a new option button to this radio group.
         *
         * @param {Any} value
         *  The unique value associated to the button. Must not be null.
         *
         * @param {Object} [options]
         *  Optional parameters. Supports all formatting options for button
         *  elements supported by the method Forms.createButtonMarkup().
         *  Additionally, the following options are supported:
         *  @param {String} [options.dataValue]
         *      A string that will be inserted into the 'data-value' attribute
         *      of the button node. If omitted, the JSON string representation
         *      of the 'value' parameter will be used instead (all double-quote
         *      characters will be removed from the string though), unless the
         *      value is a function.
         *  @param {Number|String} [options.sortIndex=0]
         *      Sorting index for the new button. All buttons will be sorted
         *      relatively to their index. Sorting will be stable, multiple
         *      buttons with the same index will remain in insertion order.
         *
         * @returns {RadioGroup}
         *  A reference to this instance.
         */
        this.createOptionButton = function (value, options) {

            var // the new button
                buttonNode = $(Forms.createButtonMarkup(options)).addClass('option-button').data('options', options),
                // all option buttons in this group
                optionButtons = null,
                // special options for a drop-down-version of this button
                dropDownVersion = Utils.getObjectOption(options, 'dropDownVersion');

            // add the button value, and insert the button node
            Forms.setButtonValue(buttonNode, value, options);
            this.addChildNodes(buttonNode);

            if (dropDownVersion) {
                createDropDownVersion(buttonNode, dropDownVersion);
            }

            // sort the buttons by their index
            if (sorted) {
                buttonNode.data('index', Utils.getOption(options, 'sortIndex', 0));
                optionButtons = this.getOptionButtons();
                optionButtons = _.sortBy(optionButtons.get(), function (node) { return $(node).data('index'); });
                self.getNode().append.apply(self.getNode(), optionButtons);
            }

            return this;
        };

        /**
         * Overwrites the base-methods (group.js) to
         * activate/deactivate the small version of the button
         */
        this.activateSmallVersion = function(){
            if (_.isObject(smallerVersion)) {
                smallRadioGroup(true);
            }
        };
        this.deactivateSmallVersion = function(){
            if (_.isObject(smallerVersion)) {
                smallRadioGroup(false);
            }
        };

        // initialization -----------------------------------------------------

        // update handler for selected state
        this.registerUpdateHandler(updateHandler);

        // option button click handler (convert ENTER and SPACE keys to click events)
        Forms.setButtonKeyHandler(this.getNode());
        this.getNode().on('click', Forms.BUTTON_SELECTOR + '.option-button', function (event) {
            var toggleClick = Forms.isSelectedNode(this) && !_.isNull(toggleValue) && !_.isUndefined(toggleValue),
                value = toggleClick ? toggleValue : Forms.getButtonValue(this);
            self.triggerChange(value, { sourceEvent: event });
        });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            this.getOptionButtons().data('options', null).remove();
            initOptions = self = matcher = null;
        });

    } // class RadioGroup

    // exports ================================================================

    // derive this class from class Group
    return Group.extend({ constructor: RadioGroup });

});
