/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/text/view/view',
    ['io.ox/core/date',
     'io.ox/contacts/api',
     'io.ox/office/tk/utils',
     'io.ox/office/tk/keycodes',
     'io.ox/office/tk/forms',
     'io.ox/office/baseframework/view/pane',
     'io.ox/office/editframework/view/editview',
     'io.ox/office/text/utils/config',
     'io.ox/office/text/view/controls',
     'io.ox/office/text/position',
     'io.ox/office/text/dom',
     'gettext!io.ox/office/text',
     'less!io.ox/office/text/view/style'
    ], function (CoreDate, ContactsAPI, Utils, KeyCodes, Forms, Pane, EditView, Config, Controls, Position, DOM, gt) {

    'use strict';

    var // class name shortcuts
        Button = Controls.Button,
        CheckBox = Controls.CheckBox,
        TextField = Controls.TextField,
        ComponentMenu = Controls.ComponentMenu,
        ComponentMenuButton = Controls.ComponentMenuButton,
        ComponentMenuSplitButton = Controls.ComponentMenuSplitButton,

    // predefined zoom factors
        ZOOM_FACTORS = [35, 50, 75, 100, 150, 200, 300, 400, 600, 800];

    if (Utils.SMALL_DEVICE) {
        ZOOM_FACTORS = [100, 150, 200];
    }

    // class TextView =========================================================

    /**
     * @constructor
     *
     * @extends EditView
     */
    function TextView(app) {

        var // self reference
            self = this,

            // the document model
            model = null,

            // the root node of the entire application pane
            appPaneNode = null,

            // the scrollable document content node
            contentRootNode = null,

            // the page node
            pageNode = null,

            //page content node
            pageContentNode = null,

            // scroll position of the application pane
            scrollPosition = { left: 0, top: 0 },

            // debug operations for replay
            replayOperations = null,

            // edit field for the debug HTML code for the page
            pageHtmlCode = null,

            // current zoom type (percentage or keyword)
            zoomType = 100,

            // the current effective zoom factor, in percent
            zoomFactor = 100,

            // outer width of page node, used for zoom calculations
            pageNodeWidth = 0,

            // outer height of page node, used for zoom calculations
            pageNodeHeight = 0,

            // change track popup and controls
            changeTrackPopup = null,
            changeTrackBadge = new Controls.ChangeTrackBadge(app, { classes: 'change-track-badge', backgroundMode: true}),
            changeTrackAcceptButton = new Button({ icon: 'fa-check', value: 'accept', tooltip: gt('Accept this change in the document') }),
            changeTrackRejectButton = new Button({ icon: 'fa-times', value: 'reject', tooltip: gt('Reject this change in the document') }),
            changeTrackShowButton = new Button({ icon: 'fa-eye', value: 'show', tooltip: gt('Show this change in the document') }),

            // change track popup timeout promise
            changeTrackPopupTimeout = null,

            // the top visible paragraph set during fast loading the document
            topVisibleParagraph = null;

            // TODO: Use the app-content node defined in baseview

        // base constructor ---------------------------------------------------

        EditView.call(this, app, {
            initHandler: initHandler,
            initDebugHandler: initDebugHandler,
            initGuiHandler: initGuiHandler,
            initDebugGuiHandler: initDebugGuiHandler,
            grabFocusHandler: grabFocusHandler,
            contentScrollable: true,
            contentMargin: 30,
            overlayMargin: { left: 8, right: Utils.SCROLLBAR_WIDTH + 8, bottom: Utils.SCROLLBAR_HEIGHT }
        });

        // private methods ----------------------------------------------------

        /**
         * Moves the browser focus focus to the editor root node.
         */
        function grabFocusHandler() {
            var node = model.getNode();

            if (Modernizr.touch && model.getEditMode() && $(document.activeElement).is(Forms.BUTTON_SELECTOR)) {
                // fix for Bug 34384 (fix for android and ios because of the annoying keyboard behavior)
                //
                // deferred focusing the page-node that the soft-keyboard does not pop up
                // only when a button was clicked (menu-dropdown or something like this)
                // not really sure about the operation but sure reproduceable
                app.executeDelayed(function () {node.focus(); });
            } else {
                node.focus();
            }
        }

        /**
         * Caches the current scroll position of the application pane.
         */
        function saveScrollPosition() {
            scrollPosition = { top: contentRootNode.scrollTop(), left: contentRootNode.scrollLeft() };
        }

        /**
         * Restores the scroll position of the application pane according to
         * the cached scroll position.
         */
        function restoreScrollPosition() {
            contentRootNode.scrollTop(scrollPosition.top).scrollLeft(scrollPosition.left);
        }

        /**
         * Scrolls the application pane to the focus position of the text
         * selection.
         */
        function scrollToSelection() {

            var // the boundaries of the text cursor
                boundRect = model.getSelection().getFocusPositionBoundRect();

            // make cursor position visible in application pane
            if (boundRect) {
                self.scrollToPageRectangle(boundRect);
            }
        }

        /**
         * Performance: Register in the selection, whether the current operation is an insertText
         * operation. In this case the debounced 'scrollToSelection' function can use the saved
         * DOM node instead of using Position.getDOMPosition in 'getPointForTextPosition'. This
         * makes the calculation of scrolling requirements faster.
         */
        function registerInsertText(options) {
            model.getSelection().registerInsertText(options);

            //fixing the problem with an upcoming soft-keyboard
            if (Modernizr.touch) { scrollToSelection(); }
        }

        /**
         * Configure the global 'Search and Replace' tool bar.
         */
        function configureSearchAndReplace() {

            var win = app.getWindow(),
                controller = app.getController(),
                lastKeyTab = false,
                // the ox search panel
                searchDiv = win.nodes.search,
                searchForm = searchDiv.find('form'),
                searchInput = searchForm.find('input[name="query"]'),
                // additional controls for search & replace
                nextButton = $('<button>')
                    .addClass('btn btn-default margin-right search-next')
                    .append(Forms.createIconMarkup('fa-chevron-right'))
                    .css({ borderRadius: '0 4px 4px 0', verticalAlign: 'top'})
                    .attr('tabindex', 1),
                previousButton = $('<button>')
                    .addClass('btn btn-default search-prev')
                    .append(Forms.createIconMarkup('fa-chevron-left'))
                    .css({ borderRadius: '4px 0 0 4px', verticalAlign: 'top'})
                    .attr('tabindex', 1),
                replaceInput = $('<input>', { type: 'text', name: 'replace' })
                    .attr({placeholder: gt('Replace with ...'), tabindex: 1 })
                    .addClass('form-control replace-query'),
                replaceButton = $('<button>')
                    .addClass('btn btn-default')
                    .css({'border-radius': '0'})
                    .attr('tabindex', 1)
                    .text(gt('Replace')),
                replaceAllButton = $('<button>')
                    .addClass('btn btn-default')
                    .attr('tabindex', 1)
                    .text(/*#. in search&replace: replace all occurrences */ gt('Replace all')),
                clearReplaceInputIcon = $(Forms.createIconMarkup('fa-times', { classes: 'clear-query' })),
                replaceInputGroup = $('<span>').addClass('input-group-btn'),
                replaceQueryContainer = $('<div>')
                    .addClass('search-query-container replace-query-container input-group')
                    .css({ width: '400px', display: 'inline-table', verticalAlign: 'top'});

            replaceInputGroup.append(replaceButton, replaceAllButton);
            replaceQueryContainer.append(replaceInput, clearReplaceInputIcon, replaceInputGroup);

            // add next/prev search hit,replace input, replace and replace all buttons
            searchForm.find('.search-query-container').css({ display: 'inline-table'}).after(previousButton, nextButton, replaceQueryContainer);

            // Handles the 'search' event from the search panel.
            function searchHandler(event, searchQuery) {
                controller.executeItem('document/search/highlight', searchQuery);
            }

            // Handles the 'search:close' event from the search panel.
            function searchCloseHandler() {
                // clear the input values
                searchInput.val('');
                replaceInput.val('');
                win.search.query = win.search.previous = '';
                model.removeHighlighting();
            }

            // Handles tab and esc key events
            function keyHandler(event) {
                var items, focus, index, down;

                if (event.keyCode === KeyCodes.TAB) {
                    lastKeyTab = true;
                    event.preventDefault();

                    items = Forms.findFocusableNodes(this);
                    focus = $(document.activeElement);
                    down = ((event.keyCode === KeyCodes.TAB) && !event.shiftKey);

                    index = (items.index(focus) || 0);
                    if (!down) {
                        index--;
                    } else {
                        index++;
                    }

                    if (index >= items.length) {
                        index = 0;
                    } else if (index < 0) {
                        index = items.length - 1;
                    }
                    items[index].focus();
                    return false;

                }

                lastKeyTab = false;
            }

            // Handles the 'change' event of the search input.
            // Stops the propagation of the 'change' event to all other attached handlers
            // if the focus stays inside the search panel.
            // With that moving the focus out of the search input e.g. via tab
            // doesn't trigger the search and so navigation to other search panel controls
            // is possible.
            function searchInputChangeHandler(event) {
                if (lastKeyTab) {
                    event.stopImmediatePropagation();
                    // if we stop the event propagation we need to update the query here
                    win.search.query = win.search.getQuery();
                    return false;
                }
            }

            function editmodeChangeHandler(event, editMode) {
                replaceButton.toggleClass('disabled', !editMode);
                replaceAllButton.toggleClass('disabled', !editMode);
            }

            // set the tool tips
            Forms.setToolTip(searchInput, gt('Find text'), 'bottom');
            Forms.setToolTip(searchInput.siblings('i'), gt('Clear'), 'bottom');
            Forms.setToolTip(searchForm.find('button[data-action=search]'), gt('Start search'), 'bottom');
            Forms.setToolTip(previousButton, gt('Select previous search result'), 'bottom');
            Forms.setToolTip(nextButton, gt('Select next search result'), 'bottom');
            Forms.setToolTip(replaceInput, gt('Replacement text'), 'bottom');
            Forms.setToolTip(replaceButton, gt('Replace selected search result and select the next result'), 'bottom');
            Forms.setToolTip(replaceAllButton, gt('Replace all search results'), 'bottom');


            // event handling
            self.listenTo(previousButton, 'click', function () { controller.executeItem('document/search/previousResult'); });
            self.listenTo(nextButton, 'click', function () { controller.executeItem('document/search/nextResult'); });
            self.listenTo(replaceButton, 'click', function () { controller.executeItem('document/search/replaceSelected'); });
            self.listenTo(replaceAllButton, 'click', function () { controller.executeItem('document/search/replaceAll'); });
            self.listenTo(searchForm, 'keydown', keyHandler);
            self.listenTo(replaceInput, 'change', function () { win.search.replacement = Utils.cleanString(this.value || ''); });
            self.listenTo(searchInput, 'keydown', function (event) { if (KeyCodes.matchKeyCode(event, 'ENTER')) { event.preventDefault(); } });
            self.listenTo(clearReplaceInputIcon, 'click', function () { replaceInput.val(''); win.search.replacement = ''; });

            // TODO: We should make the search able to handle tab between multiple controls in the search bar,
            // registering as first handler to the 'change' event is just an interim solution.
            Utils.bindAsFirstHandler(searchInput, 'change', searchInputChangeHandler, self);

            self.listenTo(win, 'search', searchHandler);
            self.listenTo(win, 'search:close', searchCloseHandler);

            self.listenTo(app.getModel(), 'change:editmode', editmodeChangeHandler);
        }

        /**
         * Gets the anchor (start) node of the given selection.
         *
         * @returns {Node | null}
         *  the anchor node of the given selection, otherwise null if its not found.
         */
        function getChangeTrackPopupAnchorNode() {
            if (!model || !model.getSelection()) { return null; }
            var selectionEndPoint = Position.getDOMPosition(model.getNode(), model.getSelection().getEndPosition());
            if (!selectionEndPoint) { return model.getNode(); }
            return selectionEndPoint.node.parentNode;
        }
        /**
         * Returns objects required for creation of the change track popup, containing:
         * - relevant change track info containing type, author, and date of the change,
         * - plus additional info containing info/accept/reject button texts.
         *
         * @returns {Object | Null}
         *  returns an object containing required meta information for the popup, and null if current selection is not a change
         *  tracked node (changeTrackInfo is null)
         *
         */
        function getChangeTrackPopupMeta() {
            var changeTrackInfo = model.getChangeTrack().getRelevantTrackingInfo();
            if (!changeTrackInfo) { return null; }
            var meta = null;
            if (changeTrackInfo.type === 'inserted') { meta = {actionText: gt('Inserted')}; }
            if (changeTrackInfo.type === 'removed') { meta =  {actionText: gt('Deleted')}; }
            if (changeTrackInfo.type === 'modified') { meta = {actionText: gt('Formatted')}; }
            meta.actionText += ': ' + Utils.capitalize(changeTrackInfo.nodeType);
            return _.extend(changeTrackInfo, meta);
        }

        /**
         * Custom Change track date formatter.
         *
         * @param {Number} timestamp
         *  timestamp in milliseconds
         *
         * @param {Object} options
         *  @param [options.toLocal = true]
         *  timestamp are converted to local time zone per default.
         *
         * @returns {String}
         *  returns the formatted date string
         */
        function formatChangeTrackDate(timestamp, options) {

            if (!_.isNumber(timestamp)) { return gt('Unknown'); }

            var toLocal = Utils.getBooleanOption(options, 'toLocal', true),
                DateConstructor = toLocal ? CoreDate.Local : CoreDate.UTC,
                now = new DateConstructor(),
                timeStampDate = new DateConstructor(timestamp);

            function getTimeString() { return timeStampDate.format(CoreDate.TIME); }

            function getDateString() {
                // get locale country default date format
                var formatString = CoreDate.locale.date;
                // hide year if the document is modified in this year
                if (timeStampDate.getYear() === now.getYear()) { formatString = CoreDate.DATE; }
                // show full day name if timestamp is still in the previous 7 days
                if (timestamp > (_.now() - CoreDate.WEEK)) { formatString = 'EEEE'; }
                return timeStampDate.format(formatString);
            }

            function isToday(date) {
                return date.getDate() === now.getDate() &&
                    date.getMonth() === now.getMonth() &&
                    date.getYear() === now.getYear();
            }

            function isYesterday(date) {
                return date.getDate() === (now.getDate() - 1) &&
                    date.getMonth() === now.getMonth() &&
                    date.getYear() === now.getYear();
            }

            // show yesterday and only time if the timestamp is from today
            if (isYesterday(timeStampDate)) { return gt('Yesterday'); }
            if (isToday(timeStampDate)) { return getTimeString(); }

            // return date string for all other cases
            return getDateString();
        }

        /**
         * Initiates and shows change track popup where applicable.
         *
         * @param {Object} options
         *  @param {jQuery Event} options.browserEvent
         *   The original jQuery Event object from a user selection. This object
         *   is used to filter out selections that are triggered by the keyboard.
         */
        function handleChangeTrackPopup(options) {

            // do not show popup for odf files (not supported yet)
            if (app.isODF()) { return; }

            // make sure old popup (if any) is hidden
            changeTrackPopup.hide();

            var // the event of the browser
                browserEvent = Utils.getObjectOption(options, 'browserEvent', null),
                // array of event types, that will trigger change track popup
                relevantEvents = ['mousedown', 'mouseup', 'touchstart', 'touchend'];

            // quit early if no listened events are found
            if (!browserEvent || !_.contains(relevantEvents, browserEvent.type)) {
                if (changeTrackPopupTimeout) { changeTrackPopupTimeout.abort(); }
                return;
            }

            // clear any existing proprietary change track selection (e.q. table column selections)
            model.getChangeTrack().clearChangeTrackSelection();

            // abort existing old timeout promise
            if (changeTrackPopupTimeout) { changeTrackPopupTimeout.abort(); }

            changeTrackPopupTimeout = app.executeDelayed(function () {

                // hyperlink popup has higher priority. Don't show change track popup if its visible.
                if (pageNode.find('.inline-popup.hyperlink').is(':visible')) { return; }

                // try show changetrackpopup directly if check spelling is off
                if (!model.isOnlineSpelling()) {
                    self.showChangeTrackPopup();
                }
                else {
                    // spell checker popups has priority, wait for spell checking promises
                    var onlineSpellingTimeoutPromise = model.getOnlineSpellingTimeout(),
                        spellReplacementPromise = model.getSpellReplacementPromise();

                    $.when(onlineSpellingTimeoutPromise, spellReplacementPromise).then(function () {
                        // check for presence of spell check popups (aka, selection has spelling errors)
                        var spellCheckPopup = model.getNode().find('.spell-replacement').filter(':visible');
                        if (spellCheckPopup.length === 0) {
                            self.showChangeTrackPopup();
                        }
                    });
                }
            }, { delay: 1000});

        }

        /**
         * Initializes the change track popup.
         */
        function initChangeTrackPopup() {
            // create and prepare controls of the popup
            changeTrackPopup = new ComponentMenu(app, {
                classes: 'change-track-popup f6-target',
                autoFocus: false,
                autoClose: false,
                autoLayout: false,
                rootContainerNode: pageNode
            });
            changeTrackAcceptButton.setLabel(gt('Accept'));
            changeTrackRejectButton.setLabel(gt('Reject'));
            changeTrackShowButton.setLabel(gt('Show'));
            changeTrackPopup
                .addGroup(null, changeTrackBadge)
                .addSeparator()
                .addGroup('acceptSelectedChangeTracking', changeTrackAcceptButton, { inline: true })
                .addGroup('rejectSelectedChangeTracking', changeTrackRejectButton, { inline: true })
                .addGroup('changetrackPopupShow', changeTrackShowButton, { inline: true });

            // disable inherited contenteditable attribute from page node to prevent text selection
            changeTrackPopup.getNode().attr('contenteditable', false);

            // manual popup sizing and positioning
            changeTrackPopup.on('popup:beforelayout', function () {
                var anchorNode = $(getChangeTrackPopupAnchorNode()),
                    anchorPosition = Utils.getChildNodePositionInNode(pageNode, anchorNode),
                    contentNode = this.getContentNode(),
                    contentSize = Utils.getExactNodeSize(contentNode),
                    zoom = self.getZoomFactor() / 100;
                // calculate positioning based on the anchor node position relative to the content root node
                anchorPosition.left = anchorPosition.left / zoom;
                anchorPosition.top = anchorPosition.top / zoom  + anchorNode.height();
                anchorPosition.width = contentSize.width / zoom;
                anchorPosition.height = contentSize.height / zoom;
                // apply positioning and dimensions
                this.getNode().css(anchorPosition);
            });
        }

        /**
         * Finding the first paragraph that is inside the visible area
         */
        function setFirstVisibleParagraph() {

            var // all paragraph elements in the document
                allParagraphs = model.getNode().find(DOM.PARAGRAPH_NODE_SELECTOR),
                // top offset for the scroll node
                topOffset = Utils.round(contentRootNode.offset().top, 1);

            // iterating over all collected change tracked nodes
            Utils.iterateArray(allParagraphs, function (paragraph) {

                if (Utils.round($(paragraph).offset().top, 1) > topOffset) {
                    topVisibleParagraph = paragraph;
                    return Utils.BREAK;
                }

            });
        }

        /**
         * Initialization after construction.
         */
        function initHandler() {

            var // deferred and debounced scroll handler, registering insertText operations immediately
                scrollToSelectionDebounced = app.createDebouncedMethod(registerInsertText, scrollToSelection, { delay: 50, maxDelay: 500 });

            // initialize other instance fields
            model = app.getModel();
            appPaneNode = self.getAppPaneNode();
            contentRootNode = self.getContentRootNode();

            // configure the OX search bar for OX Text search&replace
            configureSearchAndReplace();

            // insert the editor root node into the application pane
            self.insertContentNode(model.getNode());

            // Page node, used for zooming
            pageNode = model.getNode();

            //page content node, used for zooming on small devices
            pageContentNode = $(pageNode).children('div.pagecontent');

            // handle scroll position when hiding/showing the entire application
            self.listenTo(app.getWindow(), {
                beforehide: function () {
                    saveScrollPosition();
                    appPaneNode.css('opacity', 0);
                },
                show: function () {
                    restoreScrollPosition();
                    // restore visibility deferred, needed to restore IE scroll position first
                    app.executeDelayed(function () { appPaneNode.css('opacity', 1); });
                }
            });

            // disable drag&drop for the content root node to prevent dropping images which are
            // loaded by the browser
            self.listenTo(self.getContentRootNode(), 'drop dragstart dragenter dragexit dragover dragleave', false);

            // Registering scroll handler
            self.listenTo(self.getContentRootNode(), 'scroll', function () {
                // handler for updating change track markers in side bar after scrolling
                // -> the handler contains only functionality, if the document contains change tracks
                model.updateChangeTracksDebouncedScroll();
            });

            // handle selection change events
            self.listenTo(model, 'selection', function (event, selection, options) {
                // disable internal browser edit handles (browser may re-enable them at any time)
                self.disableBrowserEditControls();
                // scroll to text cursor
                scrollToSelectionDebounced(options);
                // do user data update
                var userData = {
                    selection : {
                        type: selection.getSelectionType(),
                        start: selection.getStartPosition(),
                        end: selection.getEndPosition()
                    }
                };
                if (app.getActiveClients() > 1) {
                    // fire and forget user data to server, only if we are in collaborative situation
                    app.updateUserData(userData);
                }
                // manage change track popups, but do not update, if 'keepChangeTrackPopup' is set to true
                if (!Utils.getBooleanOption(options, 'keepChangeTrackPopup', false)) {
                    handleChangeTrackPopup(options);
                }
            });

            // process page break event triggered by the model
            self.listenTo(model, 'pageBreak:before', function () {
                // scrolled downwards -> restore scroll position after inserting page breaks
                if (contentRootNode.scrollTop() > 0) { setFirstVisibleParagraph(); }
            });

            // process page break event triggered by the model
            self.listenTo(model, 'pageBreak:after', function () {
                if (!model.getSelection().isUndefinedSelection() && !model.getSelection().isTopPositionSelected()) {
                    // scrolling to an existing selection made by the user.
                    // 'topVisibleParagraph' should be used also in this case (user makes selection and then scrolls away
                    // without making further selection), but scrolling to 'topVisibleParagraph' does not work reliable,
                    // if there is a selection set.
                    scrollToSelection();
                } else if (topVisibleParagraph) {
                    // scrolling to the scroll position set by the user (if it was saved in 'pageBreak:before')
                    self.scrollToChildNode(topVisibleParagraph, { forceToTop: true });
                }
                topVisibleParagraph = null;
            });

            // store the values of page width and height with paddings, after loading is finished
            pageNodeWidth = pageNode.outerWidth();
            pageNodeHeight = pageNode.outerHeight();
            if (Utils.SMALL_DEVICE) {
                contentRootNode.addClass('draft-mode');
                pageContentNode.css('margin-bottom', (Math.max(screen.height, screen.width)/2) + 'px');
            }

            // initialize change track popup
            initChangeTrackPopup();

        }

        /**
         * Additional debug initialization after construction.
         */
        function initDebugHandler(operationsPane, clipboardPane) {

            var // pending operations
                pendingOperations = [],
                // delayed recording operations active
                delayedRecordingActive = false,
                // checking the operation length, so that operations removed from the
                // 'optimizeOperationHandler' will not collected twice in the recorder.
                oplCounter = 1;

            // collect operations
            function collectOperations(event, operations, external) {
                if (model.isRecordingOperations()) {
                    pendingOperations = pendingOperations.concat(_(operations).map(function (operation) {
                        return { operation: operation, external: external };
                    }));

                    if (!delayedRecordingActive) {
                        recordOperations();
                    }
                }
            }

            // record operations
            function recordOperations() {

                if (delayedRecordingActive) { return; }

                delayedRecordingActive = true;
                app.executeDelayed(function () {
                    var // full JSON string
                        fullJSONString = replayOperations.getFieldValue(),
                        // json part string to be appended
                        jsonPartString = '',
                        // current part to be processed
                        operations = pendingOperations.splice(0, 20),
                        // first entry
                        firstEntry = (fullJSONString.length === 0);

                    _(operations).each(function (entry) {

                        // skipping operations, that were 'optimized' in 'sendActions' in the
                        // optimizeOperationHandler. Otherwise 'merged operations' are still
                        // followed by the operations, that will not be sent to
                        // the server, because they were removed by the 'optimizeOperationHandler'.
                        var skipOperation = false,
                            opl = ('opl' in entry.operation) ? entry.operation.opl : '';

                        // checking if this operation was merged by the 'optimizeOperationHandler'. This can
                        // only happen for internal operations (task 29601)
                        if (! entry.external) {
                            if (_.isNumber(opl) && (opl > 1)) {
                                oplCounter = opl;
                            } else if (oplCounter > 1) {
                                oplCounter--;
                                skipOperation = true;
                            }
                        }

                        if (! skipOperation) {
                            if (!firstEntry) {
                                jsonPartString += ',';
                            }
                            jsonPartString += JSON.stringify(entry.operation);
                            firstEntry = false;
                        }
                    });

                    if (fullJSONString.length >= 2) {
                        if (fullJSONString.charAt(0) === '[') {
                            fullJSONString = fullJSONString.substr(1, fullJSONString.length - 2);
                        }
                    }
                    fullJSONString = '[' + fullJSONString + jsonPartString + ']';
                    replayOperations.setValue(fullJSONString);

                    delayedRecordingActive = false;
                    if (pendingOperations.length > 0) {
                        recordOperations();
                    }
                }, {delay: 50});
            }

            // create the output nodes in the operations pane
            operationsPane
                .addDebugInfoHeader('selection', { tooltip: 'Current selection' })
                .addDebugInfoNode('type', { tooltip: 'Type of the selection' })
                .addDebugInfoNode('start', { tooltip: 'Start position' })
                .addDebugInfoNode('end', { tooltip: 'End position' })
                .addDebugInfoNode('dir', { tooltip: 'Direction' })
                .addDebugInfoNode('attrs', { tooltip: 'Explicit attributes' });

            // log all selection events
            self.listenTo(model, 'selection', function (event, selection) {
                operationsPane
                    .setDebugInfoText('selection', 'type', selection.getSelectionType())
                    .setDebugInfoText('selection', 'start', selection.getStartPosition().join(', '))
                    .setDebugInfoText('selection', 'end', selection.getEndPosition().join(', '))
                    .setDebugInfoText('selection', 'dir', selection.isTextCursor() ? 'cursor' : selection.isBackwards() ? 'backwards' : 'forwards')
                    .setDebugInfoText('selection', 'attrs', operationsPane.isVisible() ? model.getExplicitAttributesString(selection.getStartPosition()) : '');
            });

            // operations to replay
            replayOperations = new TextField({ tooltip: _.noI18n('Paste operations here'), select: true });

            // page html code
            pageHtmlCode = new TextField({ tooltip: _.noI18n('Page html code'), select: true });

            // record operations after importing the document
            app.onImport(function () {
                self.listenTo(app.getModel(), 'operations:after', collectOperations);
            });

            // process debug events triggered by the model
            model.on('debug:clipboard', function (event, content) {
                clipboardPane.setClipboardContent(content);
            });
        }

        /**
         * Initialization after importing the document. Creates all tool boxes
         * in the side pane and overlay pane. Needed to be executed after
         * import, to be able to hide specific GUI elements depending on the
         * file type.
         */
        function initGuiHandler(viewMenuGroup) {

            var // ODF does not support inner borders at paragraphs -> special option only for this case
                displayInnerBorders = !app.isODF();


            // the 'Format' tool bar
            self.createToolBarTab('format', { label: Controls.FORMAT_HEADER_LABEL, visibleKey: 'document/editable/text' });

            var fontToolBar             = self.createToolBar('format', {priority: 1, prepareShrink: true, icon: 'fa-font', tooltip: Controls.FONT_HEADER_LABEL });
            var fontStyleToolBar        = self.createToolBar('format', {priority: 2, prepareShrink: true, icon: 'docs-font-bold', caretTooltip: gt('More font styles'), splitBtn: { key: 'character/bold', options: Controls.BOLD_BUTTON_OPTIONS } });
            var fontColorToolBar        = self.createToolBar('format', {priority: 3, prepareShrink: true, icon: 'docs-color', tooltip: gt('Colors') });
            var alignmentToolBar        = self.createToolBar('format', {priority: 4 });
            var fillBorderToolBar       = self.createToolBar('format', {priority: 4, hideable: true, prepareShrink: true, icon: 'docs-border-h', tooltip: gt('Paragraph formatting') });
            var paragraphStylesToolBar  = self.createToolBar('format', {priority: 5, hideable: true, icon: 'fa-paragraph', tooltip: Controls.PARAGRAPH_STYLES_LABEL });
            var listSettingsToolBar     = self.createToolBar('format', {priority: 5, prepareShrink: true, icon: 'docs-lists', tooltip: Controls.LIST_SETTINGS_LABEL });


            // font
            fontToolBar
                .addGroup('character/fontname', new Controls.FontFamilyPicker(app))
                .addGap()
                .addGroup('character/fontsize', new Controls.FontHeightPicker());

            // font styles
            fontStyleToolBar
                .addGroup('character/bold',      new Button(Controls.BOLD_BUTTON_OPTIONS))
                .addGroup('character/italic',    new Button(Controls.ITALIC_BUTTON_OPTIONS))
                .addGroup('character/underline', new Button(Controls.UNDERLINE_BUTTON_OPTIONS))
                .addSeparator({classes: 'noVerticalSeparator'})
                .addGroup('character/format', new ComponentMenuButton(app, {icon: 'docs-font-format', tooltip: gt('More font styles'), dropDownVersion: {visible: false}})
                    .addGroup('character/strike',    new Button(Controls.STRIKEOUT_BUTTON_OPTIONS))
                    .addSeparator()
                    .addGroup('character/vertalign', new Controls.EscapementGroup())
                    .addSeparator()
                    .addGroup('character/reset',     new Button(Controls.CLEAR_FORMAT_BUTTON_OPTIONS))
                )
                .addGroup('character/strike',    new Button(Utils.extendOptions(Controls.STRIKEOUT_BUTTON_OPTIONS, {dropDownVersion: {visible: true}})))
                .addSeparator({classes: 'hidden'})
                .addGroup('character/vertalign', new Controls.EscapementGroup({dropDownVersion: {visible: true}}))
                .addSeparator({classes: 'hidden'})
                .addGroup('character/reset',     new Button(Utils.extendOptions(Controls.CLEAR_FORMAT_BUTTON_OPTIONS, {dropDownVersion: {visible: true}})));

            // colors
            fontColorToolBar
                .addGroup('character/color',     new Controls.TextColorPicker(app))
                .addGroup('character/fillcolor', new Controls.FillColorPicker(app, {icon: 'docs-font-fill-color', tooltip: /*#. fill color behind single characters (instead of entire paragraph/cell) */ gt('Text highlight color') }));

            // alignment
            var dropdownAlignmentPicker = new Controls.ParagraphAlignmentPickerDropDown({dropDownVersion:{visible: true}}),
                normalAlignmentPicker = new Controls.ParagraphAlignmentPicker({smallerVersion: {pendant: dropdownAlignmentPicker}});
            alignmentToolBar
                .addGroup('paragraph/alignment', normalAlignmentPicker)
                .addGroup('paragraph/alignment', dropdownAlignmentPicker);

            // fill/border
            fillBorderToolBar
                .addGroup('paragraph/lineheight', new Controls.LineHeightPicker())
                .addGap()
                .addGroup('paragraph/fillcolor', new Controls.FillColorPicker(app, { icon: 'docs-para-fill-color', tooltip: gt('Paragraph fill color'), dropDownVersion: { label: gt('Paragraph fill color') } }))
                .addGap()
                .addGroup('paragraph/borders', new Controls.BorderPicker({ tooltip: gt('Paragraph borders'), showInsideHor: displayInnerBorders, dropDownVersion: { label: gt('Paragraph borders') } }));

            // paragraph styles
            paragraphStylesToolBar
                .addGroup('paragraph/stylesheet', new Controls.ParagraphStylePicker(app));

            // list settings
            listSettingsToolBar
                .addGroup('paragraph/list/bullet', new Controls.BulletListStylePicker(app))
                .addGap()
                .addGroup('paragraph/list/numbered', new Controls.NumberedListStylePicker(app))
                .addGap()
                .addGroup('paragraph/list/decindent', new Button({ icon: 'docs-list-dec-level', tooltip: /*#. indentation of lists (one list level up) */ gt('Demote one level'), dropDownVersion: { label: /*#. indentation of lists (one list level up) */ gt('Demote one level') } }))
                .addGroup('paragraph/list/incindent', new Button({ icon: 'docs-list-inc-level', tooltip: /*#. indentation of lists (one list level down) */ gt('Promote one level'), dropDownVersion: { label: /*#. indentation of lists (one list level down) */ gt('Promote one level') } }));


            // the 'Insert' tool bar
            self.createToolBarTab('insert', { label: Controls.INSERT_HEADER_LABEL, visibleKey: 'document/editable/text' });
            var insertToolBar = self.createToolBar('insert');

            insertToolBar
                .addGroup('table/insert', new Controls.TableSizePicker(app, {label: /*#. a table in a text document */ gt.pgettext('text-doc', 'Table'), maxCols: Config.MAX_TABLE_COLUMNS, maxRows: Config.MAX_TABLE_ROWS, smallerVersion: { css: { width: 50 }, hideLabel: true } }), { visibleKey: 'table/insert/available' })
                .addSeparator()
                .addGroup('image/insert/dialog', new Button(Controls.INSERT_IMAGE_BUTTON_OPTIONS))
                .addSeparator()
                .addGroup('character/hyperlink/dialog', new Button(Controls.HYPERLINK_BUTTON_OPTIONS))
                .addGroup('character/insert/tab',   new Button({ icon: 'docs-insert-tab',       label: gt('Tab stop'), tooltip: /*#. insert a horizontal tab stop into the text */ gt('Insert tab stop'), smallerVersion: {hideLabel: true} }))
                .addGroup('character/insert/break', new Button({ icon: 'docs-insert-linebreak', label: gt('Line break'),  tooltip: /*#. insert a manual line break into the text */ gt('Insert line break'), smallerVersion: {hideLabel: true} }))
                .addGroup('character/insert/pagebreak', new Button({ icon: 'docs-page-break',   label: gt('Page break'),  tooltip: /*#. insert a manual page break into the text */ gt('Insert page break'), smallerVersion: {hideLabel: true} }));

            // the 'Table' tool bar
            self.createToolBarTab('table', { label: Controls.TABLE_HEADER_LABEL, visibleKey: 'document/editable/table' });
            var tableToolBar = self.createToolBar('table');

            tableToolBar
                .addGroup('table/insert/row', new Button({ icon: 'docs-table-insert-row', tooltip: gt('Insert row') }))
                .addGroup('table/delete/row', new Button({ icon: 'docs-table-delete-row', tooltip: gt('Delete selected rows') }))
                .addGap()
                .addGroup('table/insert/column', new Button({ icon: 'docs-table-insert-column', tooltip: gt('Insert column') }))
                .addGroup('table/delete/column', new Button({ icon: 'docs-table-delete-column', tooltip: gt('Delete selected columns') }))
                .addGap();

            if (!app.isODF()) {
                tableToolBar
                    .addGroup('table/split', new Button({ icon: 'docs-table-split', tooltip: /*#. split current table */ gt('Split table') }));
            }

            if (!app.isODF()) {
                tableToolBar.addSeparator()
                    .addGroup('table/fillcolor', new Controls.FillColorPicker(app, { icon: 'docs-table-fill-color', tooltip: gt('Cell fill color') }))
                    .addGap()
                    .addGroup('table/cellborder', new Controls.BorderPicker({ tooltip: Controls.CELL_BORDERS_LABEL, showInsideHor: true, showInsideVert: true }));
                self.createToolBar('table', {priority: 1, hideable: true, classes: 'noSeparator'})
                    .addGroup('table/borderwidth', new Controls.BorderLineWidthPicker({ tooltip: gt('Cell border width') }));
            }

            if (!app.isODF()) {
                self.createToolBar('table', {priority: 2, hideable: true})
                    .addGroup('table/stylesheet', new Controls.TableStylePicker(app));
            }



            // the 'Drawing' tool bar
            self.createToolBarTab('drawing', { label: Controls.DRAWING_HEADER_LABEL, visibleKey: 'document/editable/drawing' });
            self.createToolBar('drawing')
                .addGroup('drawing/delete', new Button(Controls.DELETE_DRAWING_BUTTON_OPTIONS))
                .addGap()
                .addGroup('drawing/position', new Controls.DrawingPositionPicker());


            // the 'Review' tool bar
            self.createToolBarTab('review', { label: Controls.REVIEW_HEADER_LABEL, visibleKey: 'document/editable', sticky: true });

            // spell checker toolbar
            var review_one = self.createToolBar('review', { visibleKey: 'document/spelling/available', priority: 2, prepareShrink: true, icon: 'docs-online-spelling', caretTooltip: gt('More review actions'), splitBtn: { key: 'document/onlinespelling', options: {toggle:true, tooltip: gt('Check spelling')} } }),
                review_two = self.createToolBar('review');

            review_one
                .addGroup('document/onlinespelling', new Button({ icon: 'docs-online-spelling', tooltip: gt('Check spelling permanently'), toggle: true, dropDownVersion: { label: gt('Check spelling')} }))
                .addSeparator()
                .addGroup('character/language', new Controls.LanguagePicker({ width: 200 }));


            // change tracking toolbar (not supported for ODF files yet)
            if (!app.isODF()) {
                review_one
                    .addSeparator()
                    .addGroup('toggleChangeTracking',                   new CheckBox({ label: /*#. change tracking: toggle the change tracking feature on or off */gt('Track changes'), boxed: true, tooltip: gt('Turn change tracking on or off')}));

                review_two
                    .addGroup('acceptMoveSelectedChangeTracking',       new ComponentMenuSplitButton(app, { icon: 'fa-check', label:/*#. change tracking function: accept the current document change */ gt('Accept'), tooltip: gt('Accept the current change and select the next change'), smallerVersion:{ hideLabel: true}})
                        .addGroup('acceptMoveSelectedChangeTracking',   new Button({ label: /*#. change tracking: accept the current document change and jump to the next change */ gt('Accept and next'), tooltip: gt('Accept the current change and select the next change')}))
                        .addGroup('acceptSelectedChangeTracking',       new Button({ label: /*#. change tracking: accept the current document change */gt('Accept this change'), tooltip: gt('Accept this change in the document') }))
                        .addGroup('acceptChangeTracking',               new Button({ label: /*#. change tracking: accept all changes in the document */ gt('Accept all changes'), tooltip: gt('Accept all changes in the document') }))
                    )
                    .addGroup('rejectMoveSelectedChangeTracking',       new ComponentMenuSplitButton(app, { icon: 'fa-times', label: /*#. change tracking function: reject the current document change */ gt('Reject'), tooltip: gt('Reject the current change and select the next change'), smallerVersion:{ hideLabel: true}})
                        .addGroup('rejectMoveSelectedChangeTracking',   new Button({ label: /*#. change tracking: reject the current change and jump to the next change */ gt('Reject and next'), tooltip: gt('Reject the current change and select the next change')}))
                        .addGroup('rejectSelectedChangeTracking',       new Button({ label: /*#. change tracking: reject the current document change*/ gt('Reject this change'), tooltip: gt('Reject this change in the document')}))
                        .addGroup('rejectChangeTracking',               new Button({ label: /*#. change tracking: reject all changes in this document*/ gt('Reject all changes'), tooltip: gt('Reject all changes in the document')}))
                    )
                    .addGroup('selectPrevChangeTracking',               new Button({ icon: 'fa-chevron-left', label: /*#. change tracking: select the previous change */ gt('Previous'), tooltip: gt('Select the previous change in the document'), smallerVersion:{ hideLabel: true}}))
                    .addGroup('selectNextChangeTracking',               new Button({ icon: 'fa-chevron-right', label: /*#. change tracking: select the next change */ gt('Next'), tooltip: gt('Select the next change in the document'), smallerVersion:{ hideLabel: true}}));
            }




            // the 'View' drop-down menu
            viewMenuGroup
                .addHeaderLabel(Controls.ZOOM_LABEL)
                .addGroup('view/zoom/dec', new Button(Controls.ZOOMOUT_BUTTON_OPTIONS), { inline: true, sticky: true })
                .addGroup('view/zoom/inc', new Button(Controls.ZOOMIN_BUTTON_OPTIONS), { inline: true, sticky: true })
                .addGroup('view/zoom', new Controls.PercentageLabel(), { inline: true })
                .addGroup('view/zoom/type', new Button({ label: gt('Fit to screen width'), value: 'width' }))
                .addHeaderLabel(Controls.OPTIONS_LABEL)
                // note: we do not set aria role to 'menuitemcheckbox' or 'button' due to Safari just working correctly with 'checkox'. CheckBox constructor defaults aria role to 'checkbox'
                .addGroup('view/toolbars/show', new CheckBox(Controls.SHOW_TOOLBARS_CHECKBOX_OPTIONS))
                .addGroup('document/users', new CheckBox(Controls.SHOW_COLLABORATORS_CHECKBOX_OPTIONS));

            // event handling

            self.on('refresh:layout', function () {
                var // the app content node
                    appContentRootNode = self.getContentRootNode(),
                    // the page attributes contain the width of the page in hmm
                    pageAttribute = app.getModel().getStyleCollection('page').getElementAttributes(app.getModel().getNode()),
                    // whether the page padding need to be reduced
                    reducePagePadding = (pageAttribute.page.width > Utils.convertLengthToHmm(appContentRootNode.width(), 'px')),
                    // the remote selection object
                    remoteSelection = app.getModel().getRemoteSelection();

                // check, if there is sufficient space for left and right padding on the page
                appContentRootNode.toggleClass('small-device', reducePagePadding);

                // fix for #33829 - if padding is reduced, dependent variables have to be updated
                pageNodeWidth = pageNode.outerWidth();
                pageNodeHeight = pageNode.outerHeight();
                if (!Utils.SMALL_DEVICE) { self.recalculateDocumentMargin({ keepScrollPosition: true }); } // rotating device screen needs margin recalculated, except for < 7" devices

                // updating the change track side bar, if necessary
                model.updateChangeTracksDebounced();

                if (zoomType === 'width') {
                    app.executeDelayed(function () { self.setZoomType('width'); });
                }

                // clean and redraw remote selections
                remoteSelection.cleanCollaborativeSelections();
                _.each(app.getActiveUsers(), remoteSelection.renderCollaborativeSelection);
            });
        }

        /**
         * Additional initialization of debug GUI after importing the document.
         */
        function initDebugGuiHandler(viewMenuGroup) {

            viewMenuGroup
                .addGroup('debug/pagebreaks/toggle', new CheckBox({ label: _.noI18n('Show page breaks') }))
                .addGroup('debug/draftmode/toggle', new CheckBox({ label: _.noI18n('View in draft mode') }));

            self.createToolBar('debug')
                .addGroup('document/cut',   new Button({ icon: 'fa-cut',   tooltip: _.noI18n('Cut to clipboard') }))
                .addGroup('document/copy',  new Button({ icon: 'fa-copy',  tooltip: _.noI18n('Copy to clipboard') }))
                .addGroup('document/paste', new Button({ icon: 'fa-paste', tooltip: _.noI18n('Paste from clipboard') }));

            self.createToolBar('debug')
                .addGroup('debug/recordoperations', new Button({ icon: 'fa-play-circle-o', iconStyle: 'color:red;', toggle: true, tooltip: _.noI18n('Record operations')}))
                .addGroup('debug/replayoperations', new Button({ icon: 'docs-redo', tooltip: _.noI18n('Replay operations')}))
                .addGroup('debug/operations', replayOperations);

            self.createToolBar('debug')
                .addGroup('debug/getpagehtmlcode', new Button({ icon: 'docs-undo', tooltip: _.noI18n('Get page HTML mark-up')}))
                .addGroup('debug/setpagehtmlcode', new Button({ icon: 'docs-redo', tooltip: _.noI18n('Set page HTML mark-up')}))
                .addGroup('debug/pagehtmlcode', pageHtmlCode);

            self.createToolBar('debug')
                .addGroup('debug/uselocalstorage', new Button({ icon: 'fa-rocket', tooltip: _.noI18n('Use local storage'), toggle: true }))
                .addGroup('debug/useFastLoad', new Button({ icon: 'fa-fighter-jet', tooltip: _.noI18n('Use fast load'), toggle: true }))
                .addGap()
                .addGroup('format/spellchecking', new Button({ label: _.noI18n('ab'), tooltip: gt('Check spelling of selected text') }))
                .addGap()
                .addGroup('debug/accessrights', new Controls.AccessRightsPicker(app));
        }

        function calculateOptimalDocumentWidth() {
            var
                documentWidth = pageNode.outerWidth(),
                parentWidth = appPaneNode.width(),
                outerDocumentMargin = 3 + parseInt(appPaneNode.find('.app-content').css('margin-left'), 10) + parseInt(appPaneNode.find('.app-content').css('margin-right'), 10), // "3" because of the rounding issues
                optimalLevel = ((parentWidth * parentWidth) / (documentWidth * (parentWidth + Utils.SCROLLBAR_WIDTH + 4 + outerDocumentMargin))) * 100;

            return optimalLevel;
        }

        // methods ------------------------------------------------------------

        /**
         * Returns the position of the passed node in the entire scroll area of
         * the application pane.
         *
         * @param {HTMLElement|jQuery} node
         *  The DOM element whose dimensions will be calculated. Must be
         *  contained somewhere in the application pane. If this object is a
         *  jQuery collection, uses the first node it contains.
         *
         * @returns {Object}
         *  An object with numeric attributes representing the position and
         *  size of the passed node relative to the entire scroll area of the
         *  application pane. See method Utils.getChildNodePositionInNode() for
         *  details.
         */
        this.getChildNodePositionInNode = function (node) {
            return Utils.getChildNodePositionInNode(contentRootNode, node);
        };

        /**
         * Scrolls the application pane to make the passed node visible.
         *
         * @param {HTMLElement|jQuery} node
         *  The DOM element that will be made visible by scrolling the
         *  application pane. Must be contained somewhere in the application
         *  pane. If this object is a jQuery collection, uses the first node it
         *  contains.
         *
         * @param {Object} options
         *  @param {Boolean} [options.forceToTop=false]
         *   Whether the specified node shall be at the top border of the
         *   visible area of the scrolling node.
         *
         * @returns {TextView}
         *  A reference to this instance.
         */
        this.scrollToChildNode = function (node, options) {
            Utils.scrollToChildNode(contentRootNode, node, _.extend(options || {}, { padding: 15 }));
            return this;
        };

        /**
         * Scrolls the application pane to make the passed document page
         * rectangle visible.
         *
         * @param {Object} pageRect
         *  The page rectangle that will be made visible by scrolling the
         *  application pane. Must provide the properties 'left', 'top',
         *  'width', and 'height' in pixels. The properties 'left' and 'top'
         *  are interpreted relatively to the entire document page.
         *
         * @returns {TextView}
         *  A reference to this instance.
         */
        this.scrollToPageRectangle = function (pageRect) {
            var options = { padding: 15 };

            //fixing the problem with an upcoming soft-keyboard
            if (Modernizr.touch && model.getEditMode()) { options.regardSoftkeyboard = true; }

            Utils.scrollToPageRectangle(contentRootNode, pageRect, options);
            return this;
        };

        /**
         * Provides the debug operations as string for the debugging feature
         * replay operations.
         *
         * @returns {String}
         *  The content of the debug operations field, or null if debug mode is
         *  not active.
         */
        this.getDebugOperationsContent = function () {
            if (replayOperations) {
                return replayOperations.getFieldValue();
            }
            return null;
        };

        /**
         * Receiving the HTML mark-up of the edit field to insert it into the
         * document page.
         *
         * @returns {String}
         *  The content of the debug HTML mark-up field, or null if debug mode
         *  is not active, or the text field is empty.
         */
        this.getPageHtmlCode = function () {
            var value = pageHtmlCode ? pageHtmlCode.getFieldValue() : null;
            return (value === '') ? null : value;
        };

        /**
         * Setting html code into the debug edit field.
         *
         * @param {String} htmlCode
         *  The html code as string that will be displayed in
         *  the debug edit field.
         */
        this.setPageHtmlCode = function (htmlCode) {
            if (pageHtmlCode) {
                pageHtmlCode.setFieldValue(htmlCode);
            }
        };

        /**
         * Returns the current zoom type.
         *
         * @returns {Number|String}
         *  The current zoom type, either as fixed percentage, or as one of the
         *  keywords 'width' (page width is aligned to width of the visible
         *  area), or 'page' (page size is aligned to visible area).
         */
        this.getZoomType = function () {
            return zoomType;
        };

        /**
         * Returns the current effective zoom factor in percent.
         *
         * @returns {Number}
         *  The current zoom factor in percent.
         */
        this.getZoomFactor = function () {
            return zoomFactor;
        };

        /**
         * Returns the minimum zoom factor in percent.
         *
         * @returns {Number}
         *  The minimum zoom factor in percent.
         */
        this.getMinZoomFactor = function () {
            return ZOOM_FACTORS[0];
        };

        /**
         * Returns the maximum zoom factor in percent.
         *
         * @returns {Number}
         *  The maximum zoom factor in percent.
         */
        this.getMaxZoomFactor = function () {
            return _.last(ZOOM_FACTORS);
        };

        /**
         * Changes the current zoom settings.
         *
         * @param {Number|String} newZoomType
         *  The new zoom type. Either a fixed percentage, or one of the
         *  keywords 'width' (page width is aligned to width of the visible
         *  area), or 'page' (page size is aligned to visible area).
         *
         * @returns {TextView}
         *  A reference to this instance.
         */
        this.setZoomType = function (newZoomType) {
            var optimalLevel = calculateOptimalDocumentWidth();

            if (zoomType !== newZoomType && _.isNumber(newZoomType)) {
                if (zoomType < newZoomType) {
                    this.increaseZoomLevel(newZoomType);
                } else {
                    this.decreaseZoomLevel(newZoomType);
                }
            } else if (newZoomType === 'width') {
                if (Utils.SMALL_DEVICE || app.getModel().isDraftMode()) {
                    this.increaseZoomLevel(100);
                } else {
                    this.increaseZoomLevel(optimalLevel);
                }
            }
            zoomType = newZoomType;
            return this;
        };

        /**
         * Switches to 'fixed' zoom mode, and decreases the current zoom level
         * by one according to the current effective zoom factor, and updates
         * the view.
         *
         * @param {Number} newZoomFactor
         *  The new zoom type. If passed as argument, use this zoom factor,
         *  otherwise use one of predefined
         *
         * @returns {TextView}
         *  A reference to this instance.
         */
        this.decreaseZoomLevel = function (newZoomFactor) {

            var // find last entry in ZOOM_FACTORS with a factor less than current zoom
                prevZoomFactor = Utils.findLast(ZOOM_FACTORS, function (factor) { return factor < zoomFactor; }),
                zoomInsertion = newZoomFactor || 0,
                appContentRoot = $(appPaneNode).find('.app-content');

            if (zoomInsertion === 0) {
                zoomInsertion = _.isNumber(prevZoomFactor) ? prevZoomFactor : this.getMinZoomFactor();
            }
            zoomFactor = zoomInsertion;

            if (Utils.SMALL_DEVICE || app.getModel().isDraftMode()) {
                Utils.setCssAttributeWithPrefixes(pageContentNode, 'transform', 'scale(' +  (zoomInsertion / 100) + ')');
                Utils.setCssAttributeWithPrefixes(pageContentNode, 'transform-origin', 'top left');
                if (zoomInsertion < 100 && app.getModel().isDraftMode()) {
                    pageContentNode.css({'width' : '100%' });
                } else {
                    pageContentNode.css({'width' : ((100 / zoomInsertion) * 100) + '%', 'background-color' : '#fff' });
                }
                zoomType = zoomFactor;
            } else {

                Utils.setCssAttributeWithPrefixes(pageNode, 'transform', 'scale(' +  (zoomInsertion / 100) + ')');
                Utils.setCssAttributeWithPrefixes(pageNode, 'transform-origin', 'top');

                if (_.browser.Firefox) {
                    pageNode.css({'opacity': zoomInsertion > 100 ? 0.99 : 1});
                }

                this.recalculateDocumentMargin();
                appContentRoot.css({
                    'overflow': zoomInsertion > 99 ? '' : 'hidden'
                });

                // Use zoom as soon as all browsers support it.
                // pageNode.css({
                //    'zoom': zoomInsertion / 100
                // });

                zoomType = zoomFactor;
                scrollToSelection();
            }
            return this;
        };

        /**
         * Switches to 'fixed' zoom mode, and increases the current zoom level
         * by one according to the current effective zoom factor, and updates
         * the view.
         *
         * @param {Number} newZoomFactor
         *  The new zoom type. If passed as argument, use this zoom factor,
         *  otherwise use one of predefined
         *
         * @param {Boolean} option.clean
         *  Switching between draft and normal mode requires zoom cleanup
         *
         * @returns {TextView}
         *  A reference to this instance.
         */
        this.increaseZoomLevel = function (newZoomFactor, clean) {

            var // find first entry in ZOOM_FACTORS with a factor greater than current zoom
                nextZoomFactor = _(ZOOM_FACTORS).find(function (factor) { return factor > zoomFactor; }),
                zoomInsertion = newZoomFactor || 0,
                appContentRoot = $(appPaneNode).find('.app-content');

            if (zoomInsertion === 0) {
                zoomInsertion = _.isNumber(nextZoomFactor) ? nextZoomFactor : this.getMaxZoomFactor();
            }
            zoomFactor = zoomInsertion;

            if ((Utils.SMALL_DEVICE || app.getModel().isDraftMode()) && !clean) {
                Utils.setCssAttributeWithPrefixes(pageContentNode, 'transform', 'scale(' +  (zoomInsertion / 100) + ')');
                Utils.setCssAttributeWithPrefixes(pageContentNode, 'transform-origin', 'top left');
                pageContentNode.css({'width' : ((100 / zoomInsertion) * 100) + '%', 'background-color' : '#fff' });
                zoomType = zoomFactor;
            } else {

                Utils.setCssAttributeWithPrefixes(pageNode, 'transform', 'scale(' +  (zoomInsertion / 100) + ')');
                Utils.setCssAttributeWithPrefixes(pageNode, 'transform-origin', 'top');

                if (_.browser.Firefox) {
                    pageNode.css({'opacity': zoomInsertion > 100 ? 0.99 : 1});
                }

                this.recalculateDocumentMargin();
                appContentRoot.css({
                    'overflow': zoomInsertion > 99 ? '' : 'hidden'
                });

                // Use zoom as soon as all browsers support it.
                //  pageNode.css({
                //      'zoom': zoomInsertion / 100
                //  });

                zoomType = zoomFactor;
                scrollToSelection();

            }
            return this;
        };

        /**
         * Creates necessary margin values when using transform scale css property
         *
         */
        this.recalculateDocumentMargin = function (options) {
            var
                marginAddition = zoomFactor < 99 ? 3 : 0,
                recalculatedMargin = (pageNodeWidth * (zoomFactor / 100 - 1)) / 2 + marginAddition,
                // whether the (vertical) scroll position needs to be restored
                keepScrollPosition = Utils.getBooleanOption(options, 'keepScrollPosition', false);

            // #34735 page width and height values have to be updated
            pageNodeWidth = pageNode.outerWidth();
            pageNodeHeight = pageNode.outerHeight();

            if (_.browser.Chrome && zoomFactor > 100) {
                pageNode.css({
                    'margin':  marginAddition + 'px ' + recalculatedMargin + 'px ' + (30 / (zoomFactor / 100)) + 'px ' + recalculatedMargin + 'px'
                });
            } else {
                pageNode.css({
                    'margin':  marginAddition + 'px ' + recalculatedMargin + 'px ' + (pageNodeHeight * (zoomFactor / 100 - 1) + marginAddition) + 'px ' + recalculatedMargin + 'px'
                });
            }

            //extra special code for softkeyboard behavior, we nee more space to scroll to the end of the textfile
            if (Modernizr.touch) { pageNode.css('margin-bottom', (parseInt(pageNode.css('margin-bottom') + Math.max(screen.height, screen.width)/2)) + 'px'); }

            // in the end scroll to cursor position/selection (but not after event 'refresh:layout', that uses the option 'keepScrollPosition')
            if (!keepScrollPosition) { scrollToSelection(); }
        };

        /**
         * Rejects an attempt to edit the text document (e.g. due to reaching
         * the table size limits).
         *
         * @param {String} cause
         *  The cause of the rejection.
         *
         * @returns {TextView}
         *  A reference to this instance.
         */
        this.rejectEditTextAttempt = function (cause) {

            switch (cause) {
            case 'tablesizerow':
                this.yell({ type: 'info', message: gt('The table reached its maximum size. You cannot insert further rows.') });
                break;

            case 'tablesizecolumn':
                this.yell({ type: 'info', message: gt('The table reached its maximum size. You cannot insert further columns.') });
                break;
            }

            return this;
        };

        // returns the change track popup object
        this.getChangeTrackPopup = function () {
            return changeTrackPopup;
        };

        // show changes handler of the changetracking popup
        this.changeTrackPopupSelect = function () {
            model.getChangeTrack().showChangeTrackGroup();
            self.getChangeTrackPopup().show();
        };

        /**
         * Shows the change track popup consisting updated change track data.
         */
        this.showChangeTrackPopup = function () {
            var popupMeta = getChangeTrackPopupMeta();
            // create show popup only on change track nodes
            if (popupMeta) {
                // update popup anchor node
                changeTrackPopup.setAnchorSpec(getChangeTrackPopupAnchorNode());
                // update popup contents
                changeTrackBadge.update({
                    author: popupMeta.author || gt('Unknown'),
                    authorUserId: popupMeta.uid,
                    date: formatChangeTrackDate(Date.parse(popupMeta.date), { toLocal : false}),
                    action: popupMeta.actionText || gt('Unknown')
                });
                changeTrackPopup.show();
            }
        };

        // initialization -----------------------------------------------------

        // store the values of page width and height with paddings, after loading is finished
        app.onImportSuccess(function () {
            pageNodeWidth = pageNode.outerWidth();
            pageNodeHeight = pageNode.outerHeight();

            if (Utils.COMPACT_DEVICE && !Utils.SMALL_DEVICE) {
                app.executeDelayed(function () {
                    var optimalDocWidth = calculateOptimalDocumentWidth();
                    if (optimalDocWidth < 100) {
                        zoomType = 'width';
                    }
                });
            }
        });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            self = pageHtmlCode = replayOperations = model = null;
        });

    } // class TextView

    // exports ================================================================

    // derive this class from class EditView
    return EditView.extend({ constructor: TextView });

});
