/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define('io.ox/office/text/format/drawingstyles',
    ['io.ox/office/tk/utils',
     'io.ox/office/editframework/utils/border',
     'io.ox/office/drawinglayer/model/drawingstylecollection',
     'io.ox/office/drawinglayer/view/drawingframe',
     'io.ox/office/text/dom',
     'io.ox/office/text/drawingResize'
    ], function (Utils, Border, DrawingStyleCollection, DrawingFrame, DOM, DrawingResize) {

    'use strict';

    var // definitions for text-specific drawing attributes
        DEFINITIONS = {

            /**
             * Margin from top border of the drawing to text contents, in 1/100
             * of millimeters.
             */
            marginTop: { def: 0 },

            /**
             * Margin from bottom border of the drawing to text contents, in
             * 1/100 of millimeters.
             */
            marginBottom: { def: 0 },

            /**
             * Margin from left border of the drawing to text contents, in 1/100
             * of millimeters.
             */
            marginLeft: { def: 0 },

            /**
             * Margin from right border of the drawing to text contents, in
             * 1/100 of millimeters.
             */
            marginRight: { def: 0 },

            /**
             * Style, width and color of the left drawing border.
             */
            borderLeft: {
                def: Border.NONE,
                format: function (element, border) {
                    element.css('border-left', this.getCssBorder(border));
                }
            },

            /**
             * Style, width and color of the right drawing border.
             */
            borderRight: {
                def: Border.NONE,
                format: function (element, border) {
                    element.css('border-right', this.getCssBorder(border));
                }
            },

            /**
             * Style, width and color of the top drawing border.
             */
            borderTop: {
                def: Border.NONE,
                format: function (element, border) {
                    element.css('border-top', this.getCssBorder(border));
                }
            },

            /**
             * Style, width and color of the bottom drawing border.
             */
            borderBottom: {
                def: Border.NONE,
                format: function (element, border) {
                    element.css('border-bottom', this.getCssBorder(border));
                }
            },

            // object anchor --------------------------------------------------

            /**
             * Width of the drawing, as number in 1/100 of millimeters. The
             * value 0 will be interpreted as 100% width of the parent element.
             */
            width: {
                def: 0,
                format: function (element, width) {
                    if (width === 0) {
                        element.width('100%');
                    } else {
                        element.width(Utils.convertHmmToLength(width, 'px', 1));
                    }
                }
            },

            /**
             * Height of the drawing, as number in 1/100 of millimeters.
             */
            height: {
                def: 0,
                format: function (element, height) {
                    element.height(Utils.convertHmmToLength(height, 'px', 1));
                }
            },

            /**
             * If set to true, the drawing is rendered as inline element ('as
             * character'), otherwise it is anchored relative to another
             * element (page, paragraph, table cell, ...).
             */
            inline: { def: true },

            anchorHorBase: { def: 'margin' },

            anchorHorAlign: { def: 'left' },

            anchorHorOffset: { def: 0 },

            anchorVertBase: { def: 'margin' },

            anchorVertAlign: { def: 'top' },

            anchorVertOffset: { def: 0 },

            /**
             * Specifies how text floats around the drawing.
             * - 'none': Text does not float around the drawing.
             * - 'square': Text floats around the bounding box of the drawing.
             * - 'tight': Text aligns to the left/right outline of the drawing.
             * - 'through': Text aligns to the entire outline of the drawing.
             * - 'topAndBottom': Text floats above and below the drawing only.
             */
            textWrapMode: { def: 'none' },

            /**
             * Specifies on which side text floats around the drawing. Effective
             * only if the attribute 'textWrapMode' is either 'square',
             * 'tight', or 'through'.
             * - 'both': Text floats at the left and right side.
             * - 'left': Text floats at the left side of the drawing only.
             * - 'right': Text floats at the right side of the drawing only.
             * - 'largest': Text floats at the larger side of the drawing only.
             */
            textWrapSide: { def: 'both' }

        },

        // values for the 'textWrapMode' attribute allowing to wrap the text around the drawing
        WRAPPED_TEXT_VALUES = /^(square|tight|through)$/;

    // private global functions ===============================================

    /**
     * Returns whether the passed 'textWrapMode' attribute allows to wrap the
     * text around the drawing.
     */
    function isTextWrapped(textWrapMode) {
        return WRAPPED_TEXT_VALUES.test(textWrapMode);
    }

    // class TextDrawingStyles ================================================

    /**
     * Contains the style sheets for drawing formatting attributes. The CSS
     * formatting will be read from and written to drawing elements of any type.
     *
     * @constructor
     *
     * @extends DrawingStyleCollection
     *
     * @param {TextApplication} app
     *  The root application instance.
     *
     * @param {DocumentStyles} documentStyles
     *  Collection with the style containers of all style families.
     */
    function TextDrawingStyles(app, documentStyles) {

        // base constructor ---------------------------------------------------

        DrawingStyleCollection.call(this, app, documentStyles, { additionalFamilies: ['changes'], formatHandler: updateDrawingFormatting });

        // private methods ----------------------------------------------------

        /**
         * Repaints the selection boxes for the specified drawing node, if it
         * is selected. Updates the move and resize handling after the drawing
         * has changed between inline and floating mode.
         */
        function repaintDrawingSelection(drawing) {
            if (DrawingFrame.isSelected(drawing)) {
                DrawingFrame.clearSelection(drawing);
                DrawingResize.drawDrawingSelection(app, drawing);
            }
        }

        /**
         * Calculating the largest line height of a span inside a paragraph.
         *
         * @param {jQuery} paragraph
         *  The paragraph node whose spans will be investigated.
         *
         * @returns {Number}
         *  The largest line height of the spans in the paragraph in 1/100 mm.
         */
        function getMaximumLineHeightInParagraph(paragraph) {

            var // the largest value for the line height of a span in 1/100 mm
                maxCssLineHeight = 0;

            paragraph.children('span').each(function (index, span) {

                var lineHeight = Utils.convertCssLengthToHmm($(span).css('line-height'));

                if (lineHeight > maxCssLineHeight) {
                    maxCssLineHeight = lineHeight;
                }
            });

            return maxCssLineHeight;
        }

        /**
         * Modifying the offset for all floated drawings, that are positioned behind
         * the moved drawing.
         *
         * @param {jQuery} drawingNode
         *  The moved drawing node.
         *
         * @param {Number} addOffset
         *  The change of the offset node for the drawings, that are located behind the
         *  moved drawing node (in px).
         */
        function modifyFollowingOffsetNodes(drawingNode, addOffset) {

            var node = Utils.getDomNode(drawingNode),
                localDrawing = null;

            while (node) {
                if (DOM.isOffsetNode(node)) {
                    localDrawing = node.nextSibling;
                    $(node).height($(node).height() + addOffset);
                    $(node).remove();
                    $(node).insertBefore(localDrawing);
                }

                node = node.nextSibling;
            }
        }

        /**
         * Will be called for every drawing node whose attributes have been
         * changed. Repositions and reformats the drawing according to the
         * passed attributes.
         *
         * @param {jQuery} drawing
         *  The drawing node whose attributes have been changed, as jQuery
         *  object.
         *
         * @param {Object} mergedAttributes
         *  A map of attribute maps (name/value pairs), keyed by attribute
         *  family, containing the effective attribute values merged from style
         *  sheets and explicit attributes.
         */
        function updateDrawingFormatting(drawing, mergedAttributes) {

            var // the drawing attributes of the passed attribute map
                drawingAttributes = mergedAttributes.drawing,
                // the paragraph element containing the drawing node
                paragraph = drawing.parent(),
                // total width of the paragraph, in 1/100 mm
                paraWidth = Utils.convertLengthToHmm(paragraph.width(), 'px'),
                // preceding node used for vertical offset
                verticalOffsetNode = drawing.prev(DOM.OFFSET_NODE_SELECTOR),
                // current drawing width, in 1/100 mm
                drawingWidth = Utils.convertLengthToHmm(drawing.width(), 'px'),
                // offset from top/left/right margin of paragraph element, in 1/100 mm
                topOffset = 0, leftOffset = 0, rightOffset = 0,
                // margins to be applied at the drawing
                topMargin = 0, bottomMargin = 0, leftMargin = 0, rightMargin = 0,
                // text wrapping side (left/right/none)
                wrapMode = 'none',
                // the height of the offset node before moving the drawing
                oldVerticalOffsetNodeHeight = 0,
                // the height of the offset node after moving the drawing
                newVerticalOffsetNodeHeight = 0,
                // maximum line height in paragraph
                maxLineHeightInParagraph;

            // reducing width of drawing to width of paragraph, if no left or right cropping is enabled (Task 30982)
            if ((drawingWidth > paraWidth) && (! drawingAttributes.cropLeft) && (! drawingAttributes.cropRight)) {
                drawingWidth = paraWidth;
                drawing.width(Utils.convertHmmToCssLength(drawingWidth, 'px', 1));
            }

            // position
            if (drawingAttributes.inline) {

                // switch from floating to inline mode
                if (!drawing.hasClass('inline')) {

                    // remove leading node used for positioning
                    verticalOffsetNode.remove();

                    // TODO: Word uses fixed predefined margins in inline mode, we too?
                    drawing.removeClass('float left right').addClass('inline').css('margin', '0 1mm');
                    // ignore other attributes in inline mode

                    // repaint the selection, convert it to a non-movable selection
                    repaintDrawingSelection(drawing);
                }

            } else {

                // switch from inline to floating mode
                if (!drawing.hasClass('float')) {
                    drawing.removeClass('inline').addClass('float');
                    // repaint the selection, convert it to a movable selection
                    repaintDrawingSelection(drawing);
                }

                // calculate top offset (only if drawing is anchored to paragraph or to line)
                // anchor to line means the base line of the character position. so it is not correctly calculated
                // using the paragraph, but it is better to use it, than to ignore the value. the calculation is
                // nearly correct for characters in the first row inside the paragraph.
                if ((drawingAttributes.anchorVertBase === 'paragraph') || (drawingAttributes.anchorVertBase === 'line')) {
                    if (drawingAttributes.anchorVertAlign === 'offset') {
                        topOffset = Math.max(drawingAttributes.anchorVertOffset, 0);
                        // For an offset node it is necessary to subtract the height of the text position in
                        // the paragraph in which the drawing is located. The drawing cannot be located higher
                        // than its position allows it.
                        topOffset -= Utils.convertLengthToHmm(drawing.offset().top - drawing.parent().offset().top, 'px');
                        // adding the height of an already existing offset node
                        if (($(drawing).prev().length) && (DOM.isOffsetNode($(drawing).prev()))) {
                            topOffset += Utils.convertLengthToHmm($(drawing).prev().height(), 'px');
                        }

                        if (topOffset < 0) { topOffset = 0; }
                        // But this is also no final solution, because the drawing is shifted
                        // downwards, if text is included before the drawing position in the paragraph.
                    } else {
                        // TODO: automatic alignment (top/bottom/center/...)
                        topOffset = 0;
                    }
                }

                // calculate top offset (only if drawing is anchored to line)
                // if (drawingAttributes.anchorVertBase === 'line') {
                //     if (drawingAttributes.anchorVertAlign === 'offset') {
                //         topOffset = Math.max(drawingAttributes.anchorVertOffset, 0);
                //     } else {
                //         // TODO: automatic alignment (top/bottom/center/...)
                //         topOffset = 0;
                //     }
                // }

                // saving the old height of the vertical offset node
                if (verticalOffsetNode) {
                    oldVerticalOffsetNodeHeight = verticalOffsetNode.height();  // saving the height (in pixel)
                }

                // calculate top/bottom drawing margins
                topMargin = Utils.minMax(drawingAttributes.marginTop, 0, topOffset);
                bottomMargin = Math.max(drawingAttributes.marginBottom, 0);

                // add or remove leading offset node used for positioning
                // TODO: support for multiple drawings (also overlapping) per side
                topOffset -= topMargin;
                if (topOffset < 700) {
                    // offset less than 7mm: expand top margin to top of paragraph,
                    // otherwise the first text line overwrites the drawing
                    topMargin += topOffset;
                    // remove offset node
                    verticalOffsetNode.remove();
                } else {
                    // create offset node if not existing yet
                    if (verticalOffsetNode.length === 0) {
                        verticalOffsetNode = $('<div>', { contenteditable: false }).addClass('float offset').width(1).insertBefore(drawing);
                    }
                    // set height of the offset node
                    verticalOffsetNode.height(Utils.convertHmmToLength(topOffset, 'px', 1));
                    // Sometimes the browser ignores the change of height of the vertical offset node. This happens often, if there are only
                    // small movements of the drawing upwards. The browser does not ignore the vertical offset node, if it is removed from
                    // the dom and immediately inserted again before the drawing (28312).
                    verticalOffsetNode.remove();
                    verticalOffsetNode.insertBefore(drawing);
                    newVerticalOffsetNodeHeight = verticalOffsetNode.height();
                }

                // modifying the vertical offset for all following floated drawings in the same paragraph
                if (newVerticalOffsetNodeHeight !== oldVerticalOffsetNodeHeight) {
                    // the drawing was moved upwards -> all following floated drawings in the same paragraph need an increased offset node
                    modifyFollowingOffsetNodes(drawing, oldVerticalOffsetNodeHeight - newVerticalOffsetNodeHeight);
                }

                // calculate left/right offset (only if drawing is anchored to column)
                if (drawingAttributes.anchorHorBase === 'column') {
                    switch (drawingAttributes.anchorHorAlign) {
                    case 'center':
                        leftOffset = (paraWidth - drawingWidth) / 2;
                        break;
                    case 'right':
                        leftOffset = paraWidth - drawingWidth;
                        break;
                    case 'offset':
                        leftOffset = drawingAttributes.anchorHorOffset;
                        if (drawingAttributes.anchorHorOffset + drawingWidth > paraWidth) {
                            leftOffset = paraWidth - drawingWidth;
                        }
                        break;
                    default:
                        leftOffset = 0;
                    }
                } else if ((drawingAttributes.anchorHorBase === 'page') || (drawingAttributes.anchorHorBase === 'character')) {
                    // 'page' looks better if it is handled like column, not like the page node 'div.page'
                    // pageWidth = Utils.convertLengthToHmm(drawing.closest(DOM.PAGE_NODE_SELECTOR).width(), 'px');
                    // Also 'character' looks better, if it is handled like 'column' (33276). The value for 'anchorHorOffset'
                    // is not relative to the character, but to the paragraph.
                    switch (drawingAttributes.anchorHorAlign) {
                    case 'center':
                        leftOffset = (paraWidth - drawingWidth) / 2;
                        break;
                    case 'right':
                        leftOffset = paraWidth - drawingWidth;
                        break;
                    case 'offset':
                        leftOffset = drawingAttributes.anchorHorOffset;
                        break;
                    default:
                        leftOffset = 0;
                    }
                } else {
                    // TODO: other anchor bases (character/margins/...)
                    leftOffset = 0;
                }
                rightOffset = paraWidth - leftOffset - drawingWidth;

                // determine text wrapping side
                if (isTextWrapped(drawingAttributes.textWrapMode)) {
                    switch (drawingAttributes.textWrapSide) {
                    case 'left':
                        wrapMode = 'left';
                        break;
                    case 'right':
                        wrapMode = 'right';
                        break;
                    case 'both':
                    case 'largest':
                        // no support for 'wrap both sides' in CSS, default to 'largest'
                        wrapMode = (leftOffset > rightOffset) ? 'left' : 'right';
                        break;
                    default:
                        Utils.warn('updateDrawingFormatting(): invalid text wrap side: ' + drawingAttributes.textWrapSide);
                        wrapMode = 'none';
                    }
                } else {
                    // text does not float beside drawing
                    wrapMode = 'none';
                }

                // calculate left/right drawing margins
                switch (wrapMode) {

                case 'left':
                    // drawing floats at right paragraph margin
                    rightMargin = rightOffset;
                    leftMargin = Math.max(drawingAttributes.marginLeft, 0);
                    // if there is less than 6mm space available for text, occupy all space (no wrapping)
                    if (leftOffset - leftMargin < 600) { leftMargin = Math.max(leftOffset, 0); }
                    break;

                case 'right':
                    // drawing floats at left paragraph margin
                    leftMargin = leftOffset;
                    rightMargin = Math.max(drawingAttributes.marginRight, 0);
                    // if there is less than 6mm space available for text, occupy all space (no wrapping)
                    if (rightOffset - rightMargin < 600) { rightMargin = Math.max(rightOffset, 0); }
                    break;

                default:
                    // no wrapping: will be modeled by left-floated with large CSS margins
                    wrapMode = 'right';
                    leftMargin = leftOffset;
                    rightMargin = Math.max(rightOffset, 0);
                }

                // set text wrap mode to drawing and offset node
                drawing.add(verticalOffsetNode).removeClass('left right').addClass((wrapMode === 'left') ? 'right' : 'left');

                // in Webkit browser there is a bug, that a floated drawing can be moved above the surrounding text at the top
                // of the drawing. So if the drawing is not located in the first line of the paragraph (then it has an offset
                // node), the drawing should get a top margin, so that it does not overlap the text (fix for 28011).
                if (_.browser.WebKit && (topOffset >= 700)) {
                    // receiving the largest 'line-height' of the paragraph.
                    maxLineHeightInParagraph = getMaximumLineHeightInParagraph(paragraph);
                    topMargin += maxLineHeightInParagraph;
                    if (drawing.prev().hasClass('offset')) { // if margin is increased, decrease height of offset node for that value, so that page height doesn't get over max size
                        drawing.prev().height(drawing.prev().height() - Utils.convertHmmToLength(maxLineHeightInParagraph, 'px', 1 ));
                    }
                }

                // apply CSS formatting to drawing node
                drawing.css({
                    marginTop: Utils.convertHmmToCssLength(topMargin, 'px', 1),
                    marginBottom: Utils.convertHmmToCssLength(bottomMargin, 'px', 1),
                    marginLeft: Utils.convertHmmToCssLength(leftMargin, 'px', 1),
                    marginRight: Utils.convertHmmToCssLength(rightMargin, 'px', 1)
                });
            }

            // change track attribute handling
            app.getModel().getChangeTrack().updateChangeTrackAttributes(drawing, mergedAttributes);

            // set generic formatting to the drawing frame
            DrawingFrame.updateFormatting(app, drawing, mergedAttributes);
        }

        // initialization -----------------------------------------------------

        // register the attribute definitions for the style family
        documentStyles.registerAttributeDefinitions('drawing', DEFINITIONS);

    } // class TextDrawingStyles

    // static methods ---------------------------------------------------------

    /**
     * Returns the drawing position identifier representing the passed drawing
     * attributes.
     *
     * @param {Object} attributes
     *  A map with drawing attributes, as name/value pairs.
     *
     * @returns {String|Null}
     *  The GUI drawing position identifier; or null, if any of the passed
     *  attributes is ambiguous.
     */
    TextDrawingStyles.getPositionFromAttributes = function (attributes) {

        // inline mode overrules floating attributes
        if (_.isNull(attributes.inline)) { return null; }
        if (attributes.inline === true) { return 'inline'; }

        // filter unsupported position anchor modes
        if (!/^(column)$/.test(attributes.anchorHorBase) || !/^(paragraph|margin|line)$/.test(attributes.anchorVertBase)) { return null; }

        // filter ambiguous or unsupported horizontal alignment types
        if (!/^(left|center|right)$/.test(attributes.anchorHorAlign)) { return null; }

        // filter ambiguous text wrapping modes
        if (_.isNull(attributes.textWrapMode) || _.isNull(attributes.textWrapSide)) { return null; }

        // build the resulting drawing position
        return attributes.anchorHorAlign + ':' + (isTextWrapped(attributes.textWrapMode) ? attributes.textWrapSide : 'none');
    };

    /**
     * Returns the drawing attributes that are needed to represent the passed
     * GUI drawing position.
     *
     * @param {String} position
     *  The GUI drawing position.
     *
     * @param {Object} [origAttributes]
     *  The original attributes of the target drawing object.
     *
     * @param {Object}
     *  A map with drawing attributes, as name/value pairs.
     */
    TextDrawingStyles.getAttributesFromPosition = function (position, origAttributes) {

        var // extract alignment and text wrap mode
            matches = position.match(/^([a-z]+):([a-z]+)$/),
            // resulting drawing attributes
            drawingAttributes = {};

        // inline flag overrules all other attributes, no need to return any other (unfortunately it is still necessary for undo reasons)
        if (position === 'inline') {
            return { inline: true,
                     anchorHorBase: null,
                     anchorHorAlign: null,
                     anchorHorOffset: null,
                     anchorVertBase: null,
                     anchorVertAlign: null,
                     anchorVertOffset: null,
                     textWrapMode: null,
                     textWrapSide: null };
        }

        // check that passed position contains alignment and text wrapping mode
        if (!_.isArray(matches) || (matches.length < 3)) {
            Utils.warn('TextDrawingStyles.getAttributesFromPosition(): invalid drawing position: "' + position + '"');
            return {};
        }

        // set horizontal alignment
        _(drawingAttributes).extend({ inline: false, anchorHorBase: 'column', anchorHorAlign: matches[1], anchorHorOffset: 0 });

        // set text wrapping mode
        if (matches[2] === 'none') {
            _(drawingAttributes).extend({ textWrapMode: 'topAndBottom', textWrapSide: null });
        } else {
            _(drawingAttributes).extend({ textWrapMode: 'square', textWrapSide: matches[2] });
        }

        // reset vertical alignments for unsupported positions ('page' and 'margin')
        if (_.isObject(origAttributes) && /^(page|margin)$/.test(origAttributes.anchorVertBase)) {
            _(drawingAttributes).extend({ anchorVertBase: 'paragraph', anchorVertOffset: 0 });
        }

        return drawingAttributes;
    };

    // exports ================================================================

    // derive this class from class DrawingStyleCollection
    return DrawingStyleCollection.extend({ constructor: TextDrawingStyles });

});
