/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/render/rendercache',
    ['io.ox/office/tk/utils',
     'io.ox/office/tk/object/baseobject',
     'io.ox/office/tk/object/triggerobject',
     'io.ox/office/editframework/utils/color',
     'io.ox/office/spreadsheet/utils/sheetutils',
     'io.ox/office/spreadsheet/utils/paneutils',
     'io.ox/office/spreadsheet/model/cellcollection',
     'io.ox/office/spreadsheet/view/render/renderutils'
    ], function (Utils, BaseObject, TriggerObject, Color, SheetUtils, PaneUtils, CellCollection, RenderUtils) {

    'use strict';

    // private global functions ===============================================

    /**
     * Inserts CSS style properties into the passed cache entry. Used by the
     * column, row, and cell cache to calculate CSS style properties from the
     * current formatting attributes contained in the cache entry.
     *
     * @param {Object} cacheEntry
     *  (in/out) The cache entry to be extended with CSS style properties. Must
     *  contain the property 'attributes' with the current formatting
     *  attributes of the object represented by the cache entry. This method
     *  will insert the properties 'fill' and 'borders' into this object, but
     *  only if the respective attributes are visible (see method
     *  RenderCache.getCellEntry() for details about these properties).
     *
     * @param {DocumentStyles} documentStyles
     *  The document style container used to convert formatting attributes to
     *  CSS style properties.
     *
     * @param {Object} gridColor
     *  The color of the grid lines, used as automatic border color.
     */
    function insertStyleProperties(cacheEntry, documentStyles, gridColor) {

        var // the formatting attributes of type 'cell'
            cellAttrs = cacheEntry.attributes.cell;

        // resolves and inserts a border style object into the passed cache entry
        function insertBorderStyle(pos, attrName) {
            var borderStyle = RenderUtils.getBorderStyle(cellAttrs[attrName], documentStyles, gridColor);
            if (borderStyle) { cacheEntry.borders[pos] = borderStyle; }
        }

        // resolve fill style to color descriptor
        cacheEntry.fill = documentStyles.getColorDetails(cellAttrs.fillColor, 'fill');
        if (cacheEntry.fill.a === 0) { delete cacheEntry.fill; }

        // resolve style of all border lines
        cacheEntry.borders = {};
        insertBorderStyle('t', 'borderTop');
        insertBorderStyle('b', 'borderBottom');
        insertBorderStyle('l', 'borderLeft');
        insertBorderStyle('r', 'borderRight');
        if (_.isEmpty(cacheEntry.borders)) { delete cacheEntry.borders; }
    }

    // class ColRowCache ======================================================

    /**
     * A cache that stores settings for visible columns or rows.
     *
     * @constructor
     *
     * @extends BaseObject
     *
     * @param {SpreadsheetApplication} app
     *  The application that contains this cache instance.
     *
     * @param {SheetModel} sheetModel
     *  The model of the sheet this rendering cache is associated to.
     */
    var ColRowCache = BaseObject.extend({ constructor: function (app, sheetModel, columns) {

        var // the spreadsheet model and its containers
            model = app.getModel(),
            documentStyles = model.getDocumentStyles(),

            // the column/row collection represented by this cache
            collection = columns ? sheetModel.getColCollection() : sheetModel.getRowCollection(),

            // the registered column/row intervals, as interval arrays mapped by identifier
            intervalMap = {},

            // the unified intervals covered by this cache
            intervals = [],

            // cache entries for all visible columns/rows (as array, and mapped by index)
            entries = [],
            entryMap = {},

            // whether any of the entries contains a visible fill style
            hasFill = false,

            // whether any of the entries contains visible border styles
            hasBorders = false;

        // base constructor ---------------------------------------------------

        BaseObject.call(this);

        // private methods ----------------------------------------------------

        /**
         * Updates the intervals and cached entries, after the map of intervals
         * has been changed.
         */
        function updateCache() {

            var // grid color used for 'automatic' border colors
                gridColor = sheetModel.getEffectiveGridColor(Color.BLACK),
                // the previous cache entry
                prevEntry = null;

            // resolve interval map to unified interval array
            intervals = SheetUtils.getUnifiedIntervals(_.flatten(_.values(intervalMap), true)),

            // delete all cached data
            entries = [];
            entryMap = {};
            hasFill = hasBorders = false;

            // recreate the cached entries
            collection.iterateEntries(intervals, function (entry) {

                var // 'custom' flag always set for columns
                    customFormat = columns || entry.attributes.row.customFormat,
                    // the new cache entry
                    cacheEntry = {
                        index: entry.index,
                        offset: entry.offset,
                        size: entry.size,
                        attributes: entry.attributes,
                        custom: customFormat
                    };

                // insert fill/border style into the cache entry (only with 'customFormat' flag)
                if (customFormat) {
                    insertStyleProperties(cacheEntry, documentStyles, gridColor);
                }

                // insert the entry
                entries.push(cacheEntry);
                entryMap[entry.index] = cacheEntry;

                // update the global formatting flags
                if ('fill' in cacheEntry) { hasFill = true; }
                if ('borders' in cacheEntry) { hasBorders = true; }

                // double-link the entries
                cacheEntry.prev = prevEntry;
                cacheEntry.next = null;
                if (prevEntry) { prevEntry.next = cacheEntry; }
                prevEntry = cacheEntry;
            });
        }

        /**
         * Returns the first available cache entry located at or after the
         * specified column/row index.
         */
        function findFirstEntry(index) {
            // for performance, look into the map first (simple JS property access)
            return entryMap[index] || Utils.findFirst(entries, function (entry) {
                return index <= entry.index;
            }, { sorted: true });
        }

        // public methods -----------------------------------------------------

        /**
         * Returns the merged and unified index intervals currently covered by
         * this cache.
         *
         * @returns {Array}
         *  The merged and unified index intervals covered by this cache.
         */
        this.getIntervals = function () {
            return intervals;
        };

        /**
         * Registers the passed column/row interval, and updates all cached
         * entries.
         *
         * @param {String} id
         *  The identifier of the interval.
         *
         * @param {Object|Null} interval
         *  The new interval to be registered for the specified identifier. Can
         *  be set to null to clear the current interval.
         *
         * @returns {ColRowCache}
         *  A reference to this instance.
         */
        this.registerInterval = function (id, interval) {
            RenderUtils.log('ColRowCache.registerInterval(): id=' + id + ' old=' + (intervalMap[id] ? SheetUtils.getIntervalsName(intervalMap[id], columns) : 'null') + ' new=' + (interval ? SheetUtils.getIntervalName(interval, columns) : 'null'));

            // write the new interval into the map, or delete old interval from the map
            if (_.isObject(interval)) {
                intervalMap[id] = [_.clone(interval)];
            } else {
                delete intervalMap[id];
            }

            // update the cached entries
            updateCache();
            return this;
        };

        /**
         * Updates this cache, after columns or rows have been inserted into or
         * deleted from the sheet.
         *
         * @returns {Boolean}
         *  Whether the passed interval has caused an update of the cache. If
         *  the passed interval is located behind all intervals covered by this
         *  cache, false will be returned.
         */
        this.transformIntervals = function (interval, insert) {

            // do nothing, if the inserted/deleted interval is located after the bounding intervals
            if ((intervals.length === 0) || (_.last(intervals).last) < interval.first) { return false; }

            // transform the current bounding intervals
            _.each(intervalMap, function (boundIntervals, id) {
                intervalMap[id] = model.transformIntervals(boundIntervals, interval, insert, columns, 'split');
            });

            // update the cached entries
            updateCache();
            return true;
        };

        /**
         * Returns the cache entry with the passed index.
         *
         * @param {Number} index
         *  Zero-based column/row index.
         *
         * @returns {Object|Null}
         *  Settings for the specified column/row entry, if it is visible, with
         *  the following properties:
         *  - {Number} index
         *      The zero-based column/row index.
         *  - {Number} offset
         *      The position of the column/row in the entire sheet, in pixels.
         *  - {Number} size
         *      The size of the column/row, in pixels.
         *  - {Object} attributes
         *      The effective merged formatting attributes.
         *  - {Boolean} custom
         *      Whether the column/row has custom formatting attributes (always
         *      true for columns). Custom row formatting overrides custom
         *      column formatting.
         *  - {Object|Null} fill
         *      A descriptor for the fill color of all cells in the column/row,
         *      or null for transparent cells. See method
         *      Color.getColorDetails() for details about this object.
         *  - {Object} borders
         *      The styles of the borders of all cells in the column/row, as
         *      returned by the method RenderUtils.getBorderStyles().
         *  If the entry is not visible, the value null will be returned
         *  instead.
         */
        this.getEntry = function (index) {
            return (index in entryMap) ? entryMap[index] : null;
        };

        /**
         * Returns whether the column/row entry with the passed index is
         * currently visible.
         *
         * @param {Number} index
         *  Zero-based column/row index.
         *
         * @returns {Boolean}
         *  Whether the entry with the passed index is currently visible.
         */
        this.isEntryVisible = function (index) {
            return index in entryMap;
        };

        /**
         * Returns whether the passed interval contains at least one visible
         * column/row entry.
         *
         * @param {Object} interval
         *  The column/row interval.
         *
         * @returns {Boolean}
         *  Whether the passed interval is currently visible.
         */
        this.isIntervalVisible = function (interval) {
            // for performance, first check the existence of the passed indexes
            // in the cache (simple and fast JavaScript object property access)
            return (interval.first in entryMap) || (interval.last in entryMap) ||
                _.any(entryMap, function (entry) { return SheetUtils.intervalContainsIndex(interval, entry.index); });
        };

        /**
         * Invokes the passed iterator function for all visible entries covered
         * by the passed interval in this cache.
         *
         * @param {Object} interval
         *  The column/row interval whose visible entries will be visited.
         *
         * @param {Function} iterator
         *  The iterator function called for every visible entry in ascending
         *  order. Receives the descriptor for the column/row (see method
         *  ColRowCache.getEntry() for details about this descriptor) as first
         *  parameter.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Object} [options.context]
         *      The calling context for the passed iterator function.
         *
         * @returns {ColRowCache}
         *  A reference to this instance.
         */
        this.iterateEntries = function (interval, iterator, options) {
            var context = Utils.getOption(options, 'context');
            for (var entry = findFirstEntry(interval.first); entry && (entry.index <= interval.last); entry = entry.next) {
                iterator.call(context, entry);
            }
            return this;
        };

        /**
         * Returns whether any of the entries in this cache contains a visible
         * fill style.
         *
         * @returns {Boolean}
         *  Whether this cache contains a visible fill style.
         */
        this.hasFillStyle = function () {
            return hasFill;
        };

        /**
         * Returns whether any of the entries in this cache contains visible
         * border styles.
         *
         * @returns {Boolean}
         *  Whether this cache contains visible border styles.
         */
        this.hasBorderStyles = function () {
            return hasBorders;
        };

        // initialization -----------------------------------------------------

        // destroy all class members on destruction
        this.registerDestructor(function () {
            app = sheetModel = collection = null;
            model = documentStyles = null;
            intervalMap = intervals = entries = entryMap = null;
        });

    }}); // class ColRowCache

    // class RenderCache ======================================================

    /**
     * A cache that stores settings for the visible areas in a specific sheet.
     *
     * Instances of this class trigger the following events:
     *  - 'cache:update'
     *      After updating the cached cell entries due to cell changes from
     *      operations or view update notifications received from the server.
     *      Event listeners receive the following parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {Object} changedCells
     *          The cell descriptors of all changed or deleted cells, mapped by
     *          cell key. Deleted cache entries are represented by null values
     *          in this object.
     *  - 'cache:operation'
     *      After columns or rows have been inserted into, deleted from, or
     *      modified in the sheet associated to this cache. The cache tries to
     *      collect multiple column/row operations, and triggers an event with
     *      accumulated information about the sheet operations. Event listeners
     *      receive the following parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {Number|Null} col
     *          The index of the left-most column manipulated with a column
     *          operation. If null, the columns did not change (row operations
     *          only). Either the 'col' or the 'row' parameter will be a
     *          number.
     *      (3) {Number|Null} row
     *          The index of the top-most row manipulated with a row operation.
     *          If null, the rows did not change (column operations only).
     *          Either the 'col' or the 'row' parameter will be a number.
     *
     * @constructor
     *
     * @extends TriggerObject
     *
     * @param {SpreadsheetApplication} app
     *  The application that contains this cache instance.
     *
     * @param {SheetModel} sheetModel
     *  The model of the sheet this rendering cache is associated to.
     */
    function RenderCache(app, sheetModel) {

        var // self reference
            self = this,

            // the spreadsheet model and its containers
            model = app.getModel(),
            documentStyles = model.getDocumentStyles(),

            // the collections of the sheet
            cellCollection = sheetModel.getCellCollection(),
            colCollection = sheetModel.getColCollection(),
            rowCollection = sheetModel.getRowCollection(),
            mergeCollection = sheetModel.getMergeCollection(),

            // the unified bounding ranges covered by this cache
            boundRanges = [],
            totalRange = null,

            // caches for column and row settings
            colCache = new ColRowCache(app, sheetModel, true),
            rowCache = new ColRowCache(app, sheetModel, false),

            // all merged ranges covering the bounding ranges
            visibleMergedRanges = null,
            shrunkenMergedRanges = null,

            // column intervals of merged ranges in every row, mapped by row index
            mergedColIntervals = {},

            // all cells to be rendered, mapped by cell key
            cellEntries = {},

            // number of cell entries with visible fill/border styles
            fillEntries = 0,
            borderEntries = 0,

            // all pending cell ranges to be updated
            dirtyRanges = [],

            // whether the cell geometry have been invalidated
            dirtyGeometry = false;

        // base constructor ---------------------------------------------------

        TriggerObject.call(this);

        // private methods ----------------------------------------------------

        /**
         * Returns either the column cache, or the row cache, depending on the
         * passed flag.
         */
        function getColRowCache(columns) {
            return columns ? colCache : rowCache;
        }

        /**
         * Returns whether the cell with the passed address is visible.
         *
         * @param {Number[]} address
         *  The address of a cell.
         *
         * @returns {Boolean}
         *  Whether the passed cell is currently visible.
         */
        function isCellVisible(address) {
            return colCache.isEntryVisible(address[0]) && rowCache.isEntryVisible(address[1]);
        }

        /**
         * Returns whether the passed range overlaps the current bounding
         * ranges, and at least one of its cells is visible.
         *
         * @param {Object} range
         *  The address of a cell range.
         *
         * @returns {Boolean}
         *  Whether the passed range is currently visible.
         */
        function isRangeVisible(range) {
            return colCache.isIntervalVisible(SheetUtils.getColInterval(range)) && rowCache.isIntervalVisible(SheetUtils.getRowInterval(range));
        }

        /**
         * Returns whether the passed cell cache entry is visible in the
         * current bounding ranges (checks the merged range if available,
         * otherwise the cell address).
         *
         * @param {Object} cellData
         *  The cell descriptor.
         *
         * @returns {Boolean}
         *  Whether the passed cell entry is currently visible.
         */
        function isEntryVisible(cellData) {
            return cellData.mergedRange ? isRangeVisible(cellData.mergedRange) : isCellVisible(cellData.address);
        }

        /**
         * Updates the bounding ranges, after the column/row caches have been
         * modified.
         */
        function updateBoundingRanges() {
            boundRanges = SheetUtils.getUnifiedRanges(SheetUtils.makeRangesFromIntervals(colCache.getIntervals(), rowCache.getIntervals()));
            totalRange = SheetUtils.getBoundingRange(boundRanges);
            RenderUtils.log('boundRanges=' + SheetUtils.getRangesName(boundRanges));
        }

        /**
         * Deletes all cached settings for merged ranges.
         */
        function clearMergedRanges() {
            visibleMergedRanges = shrunkenMergedRanges = null;
            mergedColIntervals = {};
        }

        /**
         * Deletes all cached cell entries.
         */
        function clearCellEntries() {
            cellEntries = {};
            fillEntries = borderEntries = 0;
        }

        /**
         * Deletes the specified cell entry from the cache, and updates other
         * internal statistics about the existing cell entries.
         *
         * @param {String} mapKey
         *  The map key of the cell entry to be deleted.
         */
        function deleteCellEntry(mapKey) {
            var cellData = cellEntries[mapKey];
            if (cellData && cellData.fill) { fillEntries -= 1; }
            if (cellData && cellData.borders) { borderEntries -= 1; }
            delete cellEntries[mapKey];
        }

        /**
         * Inserts the specified cell entry into the cache, and updates other
         * internal statistics about the existing cell entries.
         *
         * @param {String} mapKey
         *  The map key of the cell entry to be inserted.
         *
         * @param {Object} cellData
         *  The new cell entry to be inserted into the cache.
         */
        function insertCellEntry(mapKey, cellData) {
            deleteCellEntry(mapKey);
            if (cellData.fill) { fillEntries += 1; }
            if (cellData.borders) { borderEntries += 1; }
            cellEntries[mapKey] = cellData;
        }

        /**
         * Calculates the position and size of the specified cell, and inserts
         * the data as 'rectangle' property into the cell entry.
         *
         * @param {Object} cellData
         *  (in/out) The cell descriptor (entry of this cache), that will be
         *  extended with a 'rectangle' property, if the cell is visible in the
         *  current bounding ranges (otherwise, the property will be deleted
         *  from the cell entry).
         *
         * @returns {Boolean}
         *  Whether the passed cell entry is visible (the cell entry contains a
         *  valid 'rectangle' property).
         */
        function insertCellRectangle(cellData) {

            var // the address of the cell
                address = cellData.address,
                // the merged range contained in the cell data
                mergedRange = cellData.mergedRange,
                // entries in the column and row cache
                colEntry = null,
                rowEntry = null,
                // the resulting rectangle
                rectangle = null;

            // delete the old rectangle
            delete cellData.rectangle;

            // resolve position of merged ranges directly (first/last column/row may be
            // hidden), resolve position of simple cells via the column and row caches
            if (mergedRange) {
                // skip merged ranges that have been scrolled outside the bounding ranges
                if (_.any(boundRanges, function (boundRange) { return SheetUtils.rangeOverlapsRange(mergedRange, boundRange); })) {
                    rectangle = sheetModel.getRangeRectangle(mergedRange);
                }
            } else if ((colEntry = colCache.getEntry(address[0])) && (rowEntry = rowCache.getEntry(address[1]))) {
                rectangle = { left: colEntry.offset, top: rowEntry.offset, width: colEntry.size, height: rowEntry.size };
            }

            // rectangle must exist and must be valid (non-zero size)
            if (!rectangle || (rectangle.width === 0) || (rectangle.height === 0)) { return false; }

            // insert the valid rectangle into the cell entry
            cellData.rectangle = rectangle;
            return true;
        }

        /**
         * Calculates the clipping and overflow settings for the specified cell
         * (the additional space at the left and right side of the specified
         * cell that can be used for visible overlapping text contents), and
         * inserts the data object as 'clipData' property to the cell entry.
         *
         * @param {Object} cellData
         *  (in/out) The cell descriptor (entry of this cache), that will be
         *  extended with a 'clipData' property, if the cell contains
         *  overflowing text (otherwise, the property will be deleted from the
         *  cell entry). The 'clipData' property is an object with the
         *  following properties:
         *  - {Number} clipData.clipLeft
         *      The additional space left of the cell that can be used to show
         *      the text contents.
         *  - {Number} clipData.clipRight
         *      The additional space right of the cell that can be used to show
         *      the text contents.
         *  - {Number} clipData.hiddenLeft
         *      Additional width left of the visible clip region (needed to
         *      balance centered text cells where some text is partly hidden).
         *  - {Number} clipData.hiddenRight
         *      Additional width right of the visible clip region (needed to
         *      balance centered text cells where some text is partly hidden).
         */
        function insertCellClipData(cellData) {

            // only cells with overflowing text contents, but not for merged
            // ranges (they will always be cropped at cell borders)
            if (!CellCollection.isOverflowText(cellData) || _.isObject(cellData.mergedRange)) {
                delete cellData.clipData;
                return;
            }

            var // the address of the passed cell
                address = cellData.address,
                // the effective CSS text alignment
                textAlign = PaneUtils.getCssTextAlignment(cellData),
                // no overflow left of cell needed, if cell is left-aligned or justified
                overflowLeft = /^(right|center)$/.test(textAlign),
                // no overflow right of cell needed, if cell is right-aligned or justified
                overflowRight = /^(left|center)$/.test(textAlign),
                // the nearest content cell left or right of the cell
                nextCellData = null,
                // the nearest merged column interval left or right of the cell
                nextMergedInterval = null,
                // column index of the empty cell next to the content cell or merged range
                clipCol = 0,
                // the resulting clipping data
                clipData = { clipLeft: 0, clipRight: 0, hiddenLeft: 0, hiddenRight: 0 };

            // shortcut for the method of the cell collection
            function findNearestContentCell(direction) {
                return cellCollection.findNearestContentCell(address, direction, { boundRange: totalRange });
            }

            // calculate clipping distance left and right of the cell (but not
            // for merged ranges, they will always be cropped at cell borders)
            if (overflowLeft || overflowRight) {

                // calculate left clipping size (width of empty cells between the preceding
                // content cell or merged range, and the cell passed to this method)
                if (overflowLeft) {
                    clipCol = totalRange.start[0];
                    if ((nextCellData = findNearestContentCell('left'))) {
                        clipCol = Math.max(clipCol, nextCellData.address[0] + 1);
                    }
                    if ((nextMergedInterval = self.findNextMergedColInterval(address, false))) {
                        clipCol = Math.max(clipCol, nextMergedInterval.last + 1);
                    }
                    if (clipCol < address[0]) {
                        clipData.clipLeft = colCollection.getIntervalPosition({ first: clipCol, last: address[0] - 1 }).size;
                    }
                }

                // calculate right clipping size (width of empty cells between the cell
                // passed to this method, and the following content cell or merged range)
                if (overflowRight) {
                    clipCol = totalRange.end[0];
                    if ((nextCellData = findNearestContentCell('right'))) {
                        clipCol = Math.min(clipCol, nextCellData.address[0] - 1);
                    }
                    if ((nextMergedInterval = self.findNextMergedColInterval(address, true))) {
                        clipCol = Math.min(clipCol, nextMergedInterval.first - 1);
                    }
                    if (address[0] < clipCol) {
                        clipData.clipRight = colCollection.getIntervalPosition({ first: address[0] + 1, last: clipCol }).size;
                    }
                }
            }

            // reserve 1 pixel at right border for grid lines
            clipData.clipRight -= 1;

            // calculate size of additional hidden space for the entire text contents
            if (overflowLeft) { clipData.hiddenLeft = 500000 - clipData.clipLeft; }
            if (overflowRight) { clipData.hiddenRight = 500000 - clipData.clipRight; }

            // insert new clipping data into the cell
            cellData.clipData = clipData;
        }

        /**
         * Updates the cache entries in the passed cell ranges.
         *
         * @param {Object|Array} ranges
         *  The address of a single cell range, or an array with cell range
         *  addresses.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.immediate=false]
         *      If set to true, the ranges will be updated immediately. By
         *      default, multiple cell ranges will be collected and updated
         *      debounced.
         *  @param {Number} [options.refreshCol]
         *      If specified, the index of the first dirty column after a sheet
         *      operation. Needed to update the clipping areas of preceding
         *      cells, if no dirty ranges are available, e.g. after deleting
         *      entire columns.
         */
        var updateCellRanges = (function () {

            var // first dirty column after sheet operations
                dirtyCol = null;

            // direct callback: registers dirty cell ranges in the 'dirtyRanges' array
            function registerCellRanges(ranges, options) {

                // update array of cached ranges
                var refreshCol = Utils.getIntegerOption(options, 'refreshCol');
                ranges = _.getArray(ranges);
                RenderUtils.log('RenderCache.registerCellRanges(): ranges=' + SheetUtils.getRangesName(ranges) + ' refreshCol=' + refreshCol);
                dirtyRanges = dirtyRanges.concat(ranges);
                if (_.isNumber(refreshCol)) {
                    dirtyCol = _.isNumber(dirtyCol) ? Math.min(dirtyCol, refreshCol) : refreshCol;
                }

                // invoke deferred callback immediately if specified
                if (Utils.getBooleanOption(options, 'immediate', false)) {
                    updateCellRanges();
                }
            }

            // deferred callback: updates the dirty cell ranges
            function updateCellRanges() {

                // early exit, if nothing left to be updated
                if ((dirtyRanges.length === 0) && !_.isNumber(dirtyCol)) { return; }

                var // the ranges to be updated (dirty ranges, restricted to the bounding ranges)
                    updateRanges = [],
                    // all merged ranges covering the updated ranges
                    mergedRanges = [],
                    // all merged ranges to be updated, mapped by their reference address
                    mergedRangesMap = {},
                    // all ranges that have been changed and need to be rendered
                    renderRanges = [],
                    // all cells that have been changed, mapped by cell key (map entry null for deleted cells)
                    changedCells = {},
                    // grid color used for 'automatic' border colors
                    gridColor = sheetModel.getEffectiveGridColor(Color.BLACK);

                // inserts the cell entry into the cell cache, and adds more properties to the cell entry
                function updateCellEntry(cellData, mergedRange) {

                    var // the cell address
                        address = cellData.address,
                        // the map key of the cell address
                        mapKey = SheetUtils.getCellKey(address);

                    // insert an entry into the map of changed cells, if an old cache entry
                    // exists, and remove the existing outdated entry from the cache
                    if (mapKey in cellEntries) {
                        deleteCellEntry(mapKey);
                        changedCells[mapKey] = null;
                    }

                    // resolve merged range, if not passed to the method
                    if (!mergedRange) {
                        mergedRange = SheetUtils.findFirstRange(mergedRanges, address);
                    }

                    // skip cells hidden by a merged range
                    if (mergedRange && !_.isEqual(mergedRange.start, address)) { return; }

                    // remove merged range from the map (only the remaining ranges with the
                    // reference cell outside the updated ranges need to be processed later)
                    delete mergedRangesMap[mapKey];

                    // add the merged range address to the cell data
                    cellData.mergedRange = mergedRange;

                    // add the absolute pixel position and size to the cell data
                    if (!insertCellRectangle(cellData)) { return; }

                    // insert defined cells into the cache of filled cells (also transparent
                    // cells which may need to redraw a default column or row background),
                    // and all merged ranges (also on undefined cells) which may span into
                    // the next columns/rows with own default background/border formatting)
                    if (cellData.exists || mergedRange) {

                        // calculate clipping regions for overflowing text cells
                        insertCellClipData(cellData);

                        // calculate all visible style properties (background and border lines)
                        insertStyleProperties(cellData, documentStyles, gridColor);

                        // insert entry into the cache
                        insertCellEntry(mapKey, cellData);
                        changedCells[mapKey] = cellData;
                    }
                }

                // restrict passed ranges to the bounding ranges
                _.each(boundRanges, function (boundRange) {
                    updateRanges = updateRanges.concat(SheetUtils.getIntersectionRanges(dirtyRanges, boundRange));
                });
                dirtyRanges = [];

                // unify the ranges
                updateRanges = SheetUtils.getUnifiedRanges(updateRanges);
                RenderUtils.log('dirtyRanges=' + SheetUtils.getRangesName(updateRanges));

                // shrink ranges to visible areas (remove leading/trailing hidden columns/rows)
                updateRanges = sheetModel.getVisibleRanges(updateRanges);

                // find merged ranges covering the ranges to be updated (reference address
                // of the merged range may be located outside the update ranges)
                _.each(self.getMergedRanges(), function (mergedRange) {
                    if (SheetUtils.rangesOverlapRanges(updateRanges, mergedRange)) {
                        mergedRanges.push(mergedRange);
                        mergedRangesMap[SheetUtils.getCellKey(mergedRange.start)] = mergedRange;
                    }
                });
                RenderUtils.log('mergedRanges=' + SheetUtils.getRangesName(mergedRanges));

                // delete all cell entries from the cache that are not visible anymore
                _.each(cellEntries, function (cellData, mapKey) {

                    // check whether a merged range has been deleted in the sheet
                    if (cellData.mergedRange && !(mapKey in mergedRangesMap) && SheetUtils.rangesOverlapRanges(updateRanges, cellData.mergedRange)) {
                        renderRanges.push(cellData.mergedRange);
                        delete cellData.mergedRange;
                    }

                    // delete invisible cell entries from the cache
                    if (!isEntryVisible(cellData)) {
                        deleteCellEntry(mapKey);
                        changedCells[mapKey] = null;
                    }
                });

                // update the cache entries of all cells inside the updated ranges
                RenderUtils.takeTime('update ' + Utils.getSum(updateRanges, SheetUtils.getCellCount) + ' cells in cache', function () {
                    cellCollection.iterateCellsInRanges(updateRanges, function (cellData) {
                        updateCellEntry(cellData);
                    });

                    // update remaining merged ranges (reference cell outside the updated ranges)
                    _.each(mergedRangesMap, function (mergedRange) {
                        updateCellEntry(cellCollection.getCellEntry(mergedRange.start), mergedRange);
                    });
                });

                // find the preceding and following overlapping text cells which need to be rendered
                // again (their clip region may change due to the new contents of the changed cells)
                RenderUtils.takeTime('find adjacent overlapping text cells', function () {

                    var // the addresses of all cells whose adjacent text cells will be
                        // searched (prevent double searches for the same cells)
                        leadingAddresses = {},
                        trailingAddresses = {},
                        // the column intervals covered by this cache
                        colIntervals = colCache.getIntervals(),
                        // resulting content cells whose clipping regions need to be updated
                        adjacentClipCells = {};

                    // adds an address entry to the passed map
                    function addAddress(addresses, address) {
                        addresses[SheetUtils.getCellKey(address)] = address;
                    }

                    // finds an additional cell outside the update ranges whose text clip region needs to be updated
                    function findAdjacentClipData(addresses, forward) {
                        _.each(addresses, function (address) {

                            var // the next available non-empty cell (TODO: search in own cache instead of cell collection?)
                                nextCellData = cellCollection.findNearestContentCell(address, forward ? 'right' : 'left', { boundRange: totalRange }),
                                // map key of the cell data object
                                nextCellKey = nextCellData ? SheetUtils.getCellKey(nextCellData.address) : null;

                            // register existing content cell with clipping data
                            if (nextCellKey && (nextCellKey in cellEntries) && cellEntries[nextCellKey].clipData) {
                                adjacentClipCells[nextCellKey] = cellEntries[nextCellKey];
                            }

                        });
                    }

                    // get the addresses of all cells at the left/right borders of the update ranges
                    _.each(updateRanges, function (range) {
                        rowCache.iterateEntries(SheetUtils.getRowInterval(range), function (rowEntry) {
                            addAddress(leadingAddresses, [range.start[0], rowEntry.index]);
                            addAddress(trailingAddresses, [range.end[0], rowEntry.index]);
                        });
                    });

                    // get the addresses of all cells in the dirty column
                    if (_.isNumber(dirtyCol) && (colIntervals.length > 0) && (colIntervals[0].first <= dirtyCol)) {
                        _.each(rowCache.getIntervals(), function (rowInterval) {
                            rowCache.iterateEntries(rowInterval, function (rowEntry) {
                                addAddress(leadingAddresses, [dirtyCol, rowEntry.index]);
                                if (dirtyCol > 0) { addAddress(trailingAddresses, [dirtyCol - 1, rowEntry.index]); }
                            });
                        });
                        dirtyCol = null;
                    }

                    // find all preceding and following clipped cells
                    findAdjacentClipData(leadingAddresses, false);
                    findAdjacentClipData(trailingAddresses, true);

                    // update the clipping data of adjacent content cells outside the updated ranges
                    RenderUtils.log('adjacentClipCells=', adjacentClipCells);
                    _.each(adjacentClipCells, function (cellData, mapKey) {
                        if ((mapKey in cellEntries) && !_.isObject(changedCells[mapKey])) {
                            insertCellClipData(cellData);
                            changedCells[mapKey] = cellData;
                        }
                    });
                });

                // notify listeners
                if ((renderRanges.length > 0) || !_.isEmpty(changedCells)) {
                    RenderUtils.takeTime('trigger "cache:update" event', function () {
                        RenderUtils.log('renderRanges=', renderRanges, 'changedCells=', changedCells);
                        self.trigger('cache:update', renderRanges, changedCells);
                    });
                }

                // fetch missing cells in cell collection from server (simply pass the complete
                // ranges, cell collection will fetch visible parts of unknown areas automatically)
                _.each(updateRanges, function (updateRange) {
                    cellCollection.fetchCellRange(updateRange, { visible: true, merged: true });
                });
            }

            // the debounced updateCellRanges() method to be returned from the local scope
            return app.createDebouncedMethod(registerCellRanges, RenderUtils.profileMethod('RenderCache.updateCellRanges()', updateCellRanges));
        }());

        /**
         * Recalculates the dirty properties of all cell entries in the cache.
         */
        function validateCellEntries() {

            // recalculate cell geometry (position, size, text clipping)
            if (dirtyGeometry) {
                dirtyGeometry = false;
                RenderUtils.takeTime('RenderCache.validateCellEntries(): recalculating cell geometry', function () {
                    _.each(cellEntries, function (cellData, mapKey) {

                        // update the rectangle, remove the cache entry, if the cell is not visible anymore
                        if (!insertCellRectangle(cellData)) {
                            deleteCellEntry(mapKey);
                        }

                        // update the text clipping data
                        insertCellClipData(cellData);
                    });
                });
            }
        }

        /**
         * Triggers a 'cache:operation' event with the passed settings. The
         * event will be triggered debounced, after the document model has
         * finished applying all operations (needed e.g. in complex undo or
         * redo operations).
         */
        var triggerCacheOperationEvent = (function () {

            var // the first dirty column/row index collected from multiple calls
                col = null,
                row = null,
                // whether the deferred part is waiting for operations
                waiting = false;

            // the actual triggerCacheOperationEvent() method returned from the local scope
            function triggerCacheOperationEvent(interval, columns) {

                // update column/row index in the refresh descriptor
                if (columns) {
                    col = _.isNumber(col) ? Math.min(col, interval.first) : interval.first;
                } else {
                    row = _.isNumber(row) ? Math.min(row, interval.first) : interval.first;
                }

                // nothing to do, if the method is already waiting for the operation promise
                if (waiting) { return; }

                // wait for the operation promise (operations may currently be applied asynchronously)
                waiting = true;
                self.listenTo(model.getOperationsPromise(), 'always', function () {
                    RenderUtils.takeTime('trigger "cache:operation" operation: col=' + col + ' row=' + row, function () {
                        waiting = false;
                        self.trigger('cache:operation', col, row);
                        col = row = null;
                    });
                });
            }

            return triggerCacheOperationEvent;
        }());

        /**
         * Updates this cache, after columns or rows have been inserted
         * into or deleted from the sheet.
         */
        var transformationHandler = RenderUtils.profileMethod('RenderCache.transformationHandler()', function (interval, insert, columns) {

            // always transform the dirty ranges that will cause debounced update
            // of the cell cache (may contain addresses located inside the operation
            // range although the column/row cache did not change)
            dirtyRanges = model.transformRanges(dirtyRanges, interval, insert, columns, 'split');

            // transform the bounding intervals in the column/row cache; do nothing else,
            // if the inserted/deleted interval is located after the bounding ranges
            if (!getColRowCache(columns).transformIntervals(interval, insert)) { return; }

            // transform the current bounding intervals, and update the bounding ranges
            updateBoundingRanges();

            // delete cached merged ranges (will be updated lazily)
            clearMergedRanges();

            // update cell entries in the rendering cache
            RenderUtils.takeTime('transform cached cells', function () {

                var // the old map with all visible cell data entries
                    oldEntries = cellEntries;

                // update all valid cell data entries, and copy them to the new map
                clearCellEntries();
                _.each(oldEntries, function (cellData) {

                    // transform merged range first, before transforming the cell address
                    // (start position of the transformed merged range will be used instead)
                    if (cellData.mergedRange) {
                        cellData.mergedRange = mergeCollection.transformRange(cellData.mergedRange, interval, insert, columns);
                        // merged range may be missing now although the cell remains valid (shortened to a 1x1 range)
                    }

                    // use start address of an existing merged range; otherwise transform cell address directly
                    cellData.address = cellData.mergedRange ?
                        _.clone(cellData.mergedRange.start) :
                        model.transformAddress(cellData.address, interval, insert, columns);

                    // insert updated cell data into the new map, if the cell is still visible (forget
                    // deleted cells, calculate new position and size of the cell, skip invisible cells)
                    if (cellData.address && insertCellRectangle(cellData)) {
                        insertCellEntry(SheetUtils.getCellKey(cellData.address), cellData);
                    }
                });

                // notify listeners
                triggerCacheOperationEvent(interval, columns);
            });
        });

        /**
         * Updates this cache, after columns or rows have been changed in the
         * sheet.
         */
        var changeEntriesHandler = RenderUtils.profileMethod('RenderCache.changeEntriesHandler()', function (interval, columns) {

            // do nothing, if the changed interval is located after the bounding ranges
            if (!totalRange || (SheetUtils.getInterval(totalRange, columns).last < interval.first)) {
                return;
            }

            // delete cached merged ranges (will be updated lazily)
            clearMergedRanges();

            // invalidate geometry in all existing cache entries, will be recalculated
            // lazily before accessing any cell entry in this cache
            dirtyGeometry = true;

            // update all cache entries in the changed interval (e.g. for showing hidden columns/rows)
            updateCellRanges(model.makeFullRange(interval, columns));

            // notify listeners
            triggerCacheOperationEvent(interval, columns);
        });

        /**
         * Updates this rendering cache, after cells have been changed in the
         * cell collection of the sheet.
         */
        function changeCellsHandler(event, ranges, options) {

            // delete cached merged ranges (will be updated lazily) (bug 35123)
            if (Utils.getBooleanOption(options, 'merge', false)) {
                clearMergedRanges();
            }

            // immediately update the rendering cache (not debounced), to prevent
            // 'flickering' old cell contents when typing into cells (bug 35009)
            updateCellRanges(ranges, { immediate: true });
        }

        /**
         * Updates this rendering cache, after the grid color of the sheet has
         * been changed (cell borders with automatic line color are using the
         * current sheet grid color).
         */
        function changeViewAttributesHandler(event, attributes) {
            if ('gridColor' in attributes) {
                // TODO: recalculate border colors
            }
        }

        // public methods -----------------------------------------------------

        /**
         * Invokes the passed iterator function for all visible columns or rows
         * covered by the passed interval.
         *
         * @param {Object} range
         *  The cell range whose visible columns or rows will be visited.
         *
         * @param {Boolean} columns
         *  Whether to visit the visible columns in the passed range (true), or
         *  the visible rows (false).
         *
         * @param {Function} iterator
         *  The iterator function invoked for every visible column or row.
         *  Receives a descriptor object for the current column or row (see
         *  method ColRowCache.getEntry() for details).
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Object} [options.context]
         *      The calling context for the passed iterator function.
         *
         * @returns {RenderCache}
         *  A reference to this instance.
         */
        this.iterateInterval = function (range, columns, iterator, options) {
            getColRowCache(columns).iterateEntries(SheetUtils.getInterval(range, columns), iterator, options);
            return this;
        };

        /**
         * Returns the cache entry with the passed index.
         *
         * @param {String} mapKey
         *  The address key of the cell.
         *
         * @returns {Object|Null}
         *  The cell descriptor of the specified cell, if it exists, with all
         *  properties contained in cell descriptors returned by the cell
         *  collection, and the following additional properties:
         *  - {Object|Null} mergedRange
         *      The address of the merged range, if this cell is its reference
         *      cell; otherwise null.
         *  - {Object} rectangle
         *      The pixel rectangle of the cell, or the entire merged range.
         *  - {Object|Null} fill
         *      A descriptor for the fill color of the cell, or null for
         *      transparent cells. See method Color.getColorDetails() for
         *      details about this object.
         *  - {Object} borders
         *      The styles of all cell borders, as returned by the method
         *      RenderUtils.getBorderStyles().
         *  If no cell descriptor exists of the specified cell; null will be
         *  returned instead.
         */
        this.getCellEntry = function (mapKey) {
            validateCellEntries();
            return (mapKey in cellEntries) ? cellEntries[mapKey] : null;
        };

        /**
         * Invokes the passed iterator function for all filled cells covered by
         * the passed cell range this cache.
         *
         * @param {Object} range
         *  The cell range whose visible cells will be visited.
         *
         * @param {Function} iterator
         *  The iterator function called for every visible cell, in no specific
         *  order. Receives the cell descriptor, and the unique address key of
         *  the cell.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.sorted=false]
         *      If set to true, the cells will be visited in order of their
         *      cell address (in case of merged ranges, by their reference
         *      address). Otherwise, the cells will be visited in no specific
         *      order.
         *  @param {Object} [options.context]
         *      The calling context for the passed iterator function.
         *
         * @returns {RenderCache}
         *  A reference to this instance.
         */
        this.iterateCells = function (range, iterator, options) {

            // returns whether the passed cell entry covers the current iteration range
            function entryOverlapsRange(cellData) {
                return cellData.mergedRange ? SheetUtils.rangeOverlapsRange(range, cellData.mergedRange) : SheetUtils.rangeContainsCell(range, cellData.address);
            }

            // update missing/dirty settings in the cell entries
            validateCellEntries();

            // visit cells in address order if specified
            if (Utils.getBooleanOption(options, 'sorted', false)) {

                var // all cell entries to be visited (covering the passed range)
                    visitEntries = _.filter(cellEntries, entryOverlapsRange);

                // sort by cell addresses
                visitEntries.sort(function (cellData1, cellData2) { return SheetUtils.compareCells(cellData1.address, cellData2.address); });

                // invoke iterator for all cells
                _.each(visitEntries, function (cellData, mapKey) {
                    iterator.call(this, cellData, mapKey);
                }, Utils.getOption(options, 'context'));

            } else {
                _.each(cellEntries, function (cellData, mapKey) {
                    if (entryOverlapsRange(cellData)) {
                        iterator.call(this, cellData, mapKey);
                    }
                }, Utils.getOption(options, 'context'));
            }

            return this;
        };

        /**
         * Returns all merged ranges located in the current bounding ranges.
         *
         * @returns {Array}
         *  The merged ranges in the current bounding ranges.
         */
        this.getMergedRanges = function () {

            // lazy update of the cached array of visible merged ranges
            if (!visibleMergedRanges) {
                // get all merged range overlapping with the bounding ranges
                visibleMergedRanges = mergeCollection.getMergedRanges(boundRanges);
                // filter by ranges with at least one visible cell
                visibleMergedRanges = _.filter(visibleMergedRanges, isRangeVisible);
            }

            return visibleMergedRanges;
        };

        /**
         * Returns all merged ranges located in the current bounding ranges,
         * that have a visible area of more than one cell, shrunken to their
         * visible area.
         *
         * @returns {Object}
         *  The shrunken merged ranges in the current bounding ranges, mapped
         *  by the address key of their reference cell.
         */
        this.getShrunkenMergedRanges = function () {

            // lazy update of the cached array of shrunken merged ranges
            if (!shrunkenMergedRanges) {

                shrunkenMergedRanges = {};
                _.each(this.getMergedRanges(), function (mergedRange) {

                    var // map key of the reference address
                        mapKey = SheetUtils.getCellKey(mergedRange.start);

                    // shrink to visible area, insert valid shrunken merged ranges (existing, and larger than 1x1) into the map
                    mergedRange = sheetModel.getVisibleRange(mergedRange);
                    if (mergedRange && ((mergedRange.start[0] < mergedRange.end[0]) || (mergedRange.start[1] < mergedRange.end[1]))) {
                        shrunkenMergedRanges[mapKey] = mergedRange;
                    }
                });
            }

            return shrunkenMergedRanges;
        };

        /**
         * Returns an array with the column intervals of all merged ranges
         * covering the specified row.
         *
         * @param {Number} row
         *  The zero-based row index.
         *
         * @returns {Array}
         *  The column intervals of all merged ranges covering the passed row.
         */
        this.getMergedColIntervals = function (row) {

            // lazy update of the cache entry
            if (!(row in mergedColIntervals) && rowCache.isEntryVisible(row)) {

                var // all merged ranges covering the passed row
                    mergedRanges = _.filter(this.getMergedRanges(), function (mergedRange) {
                        return SheetUtils.rangeContainsRow(mergedRange, row);
                    });

                // store the column intervals in the cache
                mergedColIntervals[row] = SheetUtils.getColIntervals(mergedRanges);
            }

            return mergedColIntervals[row];
        };

        /**
         * Returns the column interval of the next merged range in the row of
         * the specified cell, searching either left of or right of the cell.
         *
         * @param {Number[]} address
         *  The address of the cell.
         *
         * @param {Boolean} forward
         *  Whether to look for an interval right (true) of or left (false) of
         *  the passed address.
         *
         * @returns {Object|Null}
         *  The column interval of the next merged range in the row of the
         *  cell if existing, otherwise null.
         */
        this.findNextMergedColInterval = function (address, forward) {

            var // the column index of the address
                col = address[0],
                // the merged intervals in the row
                mergedIntervals = this.getMergedColIntervals(address[1]),
                // the first interval found in the array (use fast binary search)
                nextInterval = !mergedIntervals ? false : forward ?
                    Utils.findFirst(mergedIntervals, function (interval) { return col < interval.first; }, { sorted: true }) :
                    Utils.findLast(mergedIntervals, function (interval) { return interval.last < col; }, { sorted: true });

            return nextInterval ? nextInterval : null;
        };

        /**
         * Returns a matrix with rendering data for all cell borders in the
         * passed range.
         *
         * @param {Object} range
         *  The address of the cell range whose border matrix will be returned.
         *
         * @returns {Object|Null}
         *  A map with entries for all cells in all visible columns and rows
         *  covered by the passed range address (including all visible but
         *  empty cells, and all cells covered by merged ranges). The cell
         *  descriptors are mapped by the unique cell address key (column and
         *  row index, separated by a comma), and contains the following
         *  properties:
         *  - {Number} entry.col
         *      The zero-based column index of the cell.
         *  - {Number} entry.row
         *      The zero-based row index of the cell.
         *  - {Object} entry.link
         *      References to all existing adjacent matrix elements, in the
         *      following properties:
         *      - {Object} [entry.link.t]
         *          Reference to the matrix element above this entry.
         *      - {Object} [entry.link.b]
         *          Reference to the matrix element below this entry.
         *      - {Object} [entry.link.l]
         *          Reference to the matrix element left of this entry.
         *      - {Object} [entry.link.r]
         *          Reference to the matrix element right of this entry.
         *  - {Number} entry.x1
         *      The X coordinate of the left border of the cell, in pixels.
         *  - {Number} entry.y1
         *      The Y coordinate of the top border of the cell, in pixels.
         *  - {Number} entry.x2
         *      The X coordinate of the right border of the cell, in pixels
         *      (ignores merged ranges).
         *  - {Number} entry.y2
         *      The Y coordinate of the bottom border of the cell, in pixels
         *      (ignores merged ranges).
         *  - {Number} entry.width
         *      The width of the cell, in pixels (ignores merged ranges).
         *  - {Number} entry.height
         *      The height of the cell, in pixels (ignores merged ranges).
         *  - {Object} entry.borders
         *      Border styles for the cell (borders of merged ranges will be
         *      distributed to the respective matrix elements), as returned by
         *      the method RenderUtils.getBorderStyles().
         *  If there are no visible borders at all in the entire rendering
         *  cache, this method returns null instead.
         */
        this.getBorderMatrix = RenderUtils.profileMethod('RenderCache.getBorderMatrix()', function (range) {

            // fast check whether the caches contain any borders
            if (!colCache.hasBorderStyles() && !rowCache.hasBorderStyles() && (borderEntries === 0)) { return null; }

            // shrink to visible area (needed for oversized merged ranges to find the correct start address)
            range = sheetModel.getVisibleRange(range);
            if (!range) { return null; }

            var // resulting cell matrix, mapped by cell address key
                matrix = {},
                // visible parts of all merged ranges
                mergedRanges = this.getShrunkenMergedRanges(),
                // the row interval of the passed range
                rowInterval = SheetUtils.getRowInterval(range),
                // index of the previous column while iterating
                prevCol = -1;

            // returns a unique cell address key, prevent unneeded construction
            // of arrays for cell addresses as required by SheetUtils.getCellKey()
            function key(col, row) { return col + ',' + row; }

            // merges the border styles (deletes the weaker border style)
            function mergeBorders(borders1, pos1, borders2, pos2) {
                var border = RenderUtils.getStrongerBorder(borders1[pos1], borders2[pos2]);
                // store border in leading cell entry, always delete trailing entry
                if (border) { borders1[pos1] = border; } else { delete borders1[pos1]; }
                delete borders2[pos2];
            }

            // initialize the matrix from default column formatting (every cell gets a map entry)
            colCache.iterateEntries(SheetUtils.getColInterval(range), function (colEntry) {

                var // upper border entry in the current column
                    topEntry = null;

                // create a border entry for all rows
                rowCache.iterateEntries(rowInterval, function (rowEntry) {

                    var // the map entry left of the current cell
                        leftEntry = (prevCol < 0) ? null : matrix[key(prevCol, rowEntry.index)],
                        // the new matrix entry
                        thisEntry = {
                            col: colEntry.index,
                            row: rowEntry.index,
                            link: {},
                            x1: colEntry.offset,
                            x2: colEntry.offset + colEntry.size,
                            y1: rowEntry.offset,
                            y2: rowEntry.offset + rowEntry.size,
                            width: colEntry.size,
                            height: rowEntry.size
                        },
                        // border styles from column or row
                        borders = rowEntry.custom ? rowEntry.borders : colEntry.borders;

                    // insert the new matrix entry, and double-link all affected entries
                    matrix[key(thisEntry.col, thisEntry.row)] = thisEntry;
                    if (topEntry) { thisEntry.link.t = topEntry; topEntry.link.b = thisEntry; }
                    if (leftEntry) { thisEntry.link.l = leftEntry; leftEntry.link.r = thisEntry; }

                    // insert border styles into the entry (needs a clone to be able to
                    // manipulate the object later while resolving concurrent borders)
                    thisEntry.borders = borders ? _.clone(borders) : {};

                    // remember current entry for next cell in the column
                    topEntry = thisEntry;
                });

                prevCol = colEntry.index;
            });

            // insert border styles of all existing cells
            this.iterateCells(range, function (cellData) {

                var // the visible part of a merged range
                    mergedRange = cellData.mergedRange ? mergedRanges[SheetUtils.getCellKey(cellData.mergedRange.start)] : null,
                    // top-left matrix entry of a merged range
                    firstEntry = mergedRange ? matrix[key(Math.max(range.start[0], mergedRange.start[0]), Math.max(range.start[1], mergedRange.start[1]))] : null;

                // prepare all covered cells of a merged range
                if (firstEntry) {
                    // visit all rows of the merged range
                    for (var rowEntry = firstEntry; rowEntry && (rowEntry.row <= mergedRange.end[1]); rowEntry = rowEntry.link.b) {
                        // visit all columns of the merged range in the current row
                        for (var entry = rowEntry; entry && (entry.col <= mergedRange.end[0]); entry = entry.link.r) {
                            entry.borders = {};
                            if (cellData.borders) {
                                if (cellData.borders.t && (entry.row === mergedRange.start[1])) { entry.borders.t = cellData.borders.t; }
                                if (cellData.borders.b && (entry.row === mergedRange.end[1])) { entry.borders.b = cellData.borders.b; }
                                if (cellData.borders.l && (entry.col === mergedRange.start[0])) { entry.borders.l = cellData.borders.l; }
                                if (cellData.borders.r && (entry.col === mergedRange.end[0])) { entry.borders.r = cellData.borders.r; }
                            }
                        }
                    }
                } else {
                    var entry = matrix[key(cellData.address[0], cellData.address[1])];
                    entry.borders = cellData.borders ? _.clone(cellData.borders) : {};
                }
            });

            // process overlapping border lines
            _.each(matrix, function (entry) {
                // merge right borders of this entry with left border of right neighbor
                if (entry.link.r) { mergeBorders(entry.borders, 'r', entry.link.r.borders, 'l'); }
                // merge bottom borders of this entry with top border of bottom neighbor
                if (entry.link.b) { mergeBorders(entry.borders, 'b', entry.link.b.borders, 't'); }
            });

            return matrix;
        });

        /**
         * Updates a bounding interval of this rendering cache. The column and
         * row bounding intervals define the cell ranges covered by this cache.
         *
         * @param {String} id
         *  A unique identifier for the passed bounding range. With different
         *  identifiers it is possible to manage different column and row
         *  intervals, as used for example in a split view.
         *
         * @param {Boolean} columns
         *  Whether the passed interval is a column interval (true), or a row
         *  interval (false).
         *
         * @param {Object|Null} interval
         *  The new bounding interval associated to the passed identifier. Can
         *  be set to null to clear the current interval.
         *
         * @param {Object} [refresh]
         *  The refresh strategy for the columns/rows in the new interval, with
         *  the following properties:
         *  @param {Number} [refresh.index]
         *      If specified, the layout of the columns/rows starting from this
         *      column/row index has changed, e.g. after a sheet operation such
         *      as inserting/deleting rows or columns, or changing their format
         *      attributes (including size and visibility).
         *  @param {Boolean} [refresh.geometry=false]
         *      If set to true, the positions and sizes of all columns, rows,
         *      and cells need to be refreshed completely (e.g. while zooming).
         *  If this parameter is omitted, the visible interval has changed but
         *  the columns/rows in the new interval did not change.
         *
         * @returns {RenderCache}
         *  A reference to this instance.
         */
        this.updateBoundingInterval = RenderUtils.profileMethod('RenderCache.updateBoundingInterval()', function (id, columns, interval, refresh) {
            RenderUtils.log('id=' + id + ' interval=' + (interval ? SheetUtils.getIntervalName(interval, columns) : 'null') + ' refresh=' + JSON.stringify(refresh));

            var // the old bounding ranges
                oldBoundRanges = boundRanges,
                // the column index of the dirty area after a sheet operation
                refreshCol = columns ? Utils.getIntegerOption(refresh, 'index', null) : null;

            // set the interval in the column/row cache, and update the bounding ranges
            getColRowCache(columns).registerInterval(id, interval);
            updateBoundingRanges();

            // delete cached merged ranges (will be updated lazily)
            clearMergedRanges();

            // invalidate geometry in all existing cache entries, will be recalculated
            // lazily before accessing any cell entry in this cache
            if (Utils.getBooleanOption(refresh, 'geometry', false)) {
                dirtyGeometry = true;
            }

            // insert cell entries that became visible, but do not trigger a change event
            updateCellRanges(SheetUtils.getRemainingRanges(boundRanges, oldBoundRanges), { refreshCol: refreshCol });

            return this;
        });

        // initialization -----------------------------------------------------

        // update cell entries after inserting/deleting columns or rows in the sheet
        sheetModel.registerTransformationHandler(transformationHandler);

        // update cell entries after modifying columns or rows in the sheet
        this.listenTo(colCollection, 'change:entries', function (event, interval) { changeEntriesHandler(interval, true); });
        this.listenTo(rowCollection, 'change:entries', function (event, interval) { changeEntriesHandler(interval, false); });

        // update cell entries after any cells have changed in the collection
        this.listenTo(cellCollection, 'change:cells', changeCellsHandler);

        // update cell entries with borders after the grid color has changed
        this.listenTo(sheetModel, 'change:viewattributes', changeViewAttributesHandler);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            colCache.destroy();
            rowCache.destroy();
            sheetModel.unregisterTransformationHandler(transformationHandler);
            app = self = model = documentStyles = null;
            sheetModel = cellCollection = colCollection = rowCollection = mergeCollection = null;
            colCache = rowCache = cellEntries = null;
            visibleMergedRanges = shrunkenMergedRanges = mergedColIntervals = null;
        });

    } // class RenderCache

    // exports ================================================================

    // derive this class from class TriggerObject
    return TriggerObject.extend({ constructor: RenderCache });

});
