/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/gridpane',
    ['io.ox/office/tk/utils',
     'io.ox/office/tk/keycodes',
     'io.ox/office/tk/forms',
     'io.ox/office/tk/popup/listmenu',
     'io.ox/office/tk/object/triggerobject',
     'io.ox/office/drawinglayer/utils/drawingutils',
     'io.ox/office/drawinglayer/view/drawingframe',
     'io.ox/office/drawinglayer/view/imageutil',
     'io.ox/office/spreadsheet/utils/sheetutils',
     'io.ox/office/spreadsheet/utils/paneutils',
     'io.ox/office/spreadsheet/utils/clipboard',
     'io.ox/office/spreadsheet/view/render/renderutils',
     'io.ox/office/spreadsheet/view/render/gridrenderer',
     'io.ox/office/spreadsheet/view/mixin/gridtrackingmixin',
     'io.ox/office/spreadsheet/view/controls',
     'gettext!io.ox/office/spreadsheet'
    ], function (Utils, KeyCodes, Forms, ListMenu, TriggerObject, DrawingUtils, DrawingFrame, ImageUtil, SheetUtils, PaneUtils, Clipboard, RenderUtils, GridRenderer, GridTrackingMixin, Controls, gt) {

    'use strict';

    var // maximum size of an image in any direction, in 1/100 mm
        MAX_IMAGE_SIZE = 21000;

    // class GridPane =========================================================

    /**
     * Represents a single scrollable grid pane in the spreadsheet view. If the
     * view has been split or frozen at a specific cell position, the view will
     * consist of up to four panes (top-left, top-right, bottom-left, and
     * bottom-right pane).
     *
     * Instances of this class trigger the following events:
     * - 'change:scrollpos':
     *      After the scroll position of the grid pane has been changed.
     *      Listeners may want to update the position of additional DOM content
     *      inserted into the grid pane.
     * - 'select:start'
     *      After selection tracking has been started. Event handlers receive
     *      the following parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {Number[]} address
     *          The address of the start cell.
     *      (3) {String} mode
     *          The selection mode according to the keyboard modifier keys:
     *          - 'select': Standard selection without modifier keys.
     *          - 'append': Append new range to current selection (CTRL/META).
     *          - 'extend': Extend current active range (SHIFT).
     * - 'select:move'
     *      After the address of the tracked cell has changed while selection
     *      tracking is active. Event handlers receive the following
     *      parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {Number[]} address
     *          The address of the current cell.
     * - 'select:end'
     *      After selection tracking has been finished (either successfully or
     *      not). Event handlers receive the following parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {Number[]} address
     *          The address of the last cell.
     *      (3) {Boolean} success
     *          True if selecting has finished successfully, false if selecting
     *          has been canceled.
     * - 'select:change'
     *      While selection tracking on touch devices is active, and a
     *      selection range has been modified by tracking the resizer handles.
     *      Event handlers receive the following parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {Number} rangeIndex
     *          The array index of the modified range in the current selection.
     *      (3) {Object} range
     *          The new address of the specified range.
     *      (4) {Number[]} [activeCell]
     *          The address of the active cell. If omitted, the current
     *          active cell will not be changed, unless the new active range
     *          does not contain it anymore.
     *
     * @constructor
     *
     * @extends TriggerObject
     *
     * @param {SpreadsheetApplication} app
     *  The application that contains this grid pane.
     *
     * @param {String} panePos
     *  The position of this grid pane.
     */
    function GridPane(app, panePos) {

        var // self reference
            self = this,

            // column and row pane side identifiers
            colPaneSide = PaneUtils.getColPaneSide(panePos),
            rowPaneSide = PaneUtils.getRowPaneSide(panePos),

            // the spreadsheet model, and related collections
            model = null,
            documentStyles = null,

            // the spreadsheet view, and the header panes associated to the grid pane
            view = null,
            colHeaderPane = null,
            rowHeaderPane = null,

            // the rendering engine for cells and the grid lines
            gridRenderer = new GridRenderer(app),

            // the container node of this grid pane
            rootNode = $('<div class="abs grid-pane f6-target" data-pos="' + panePos + '">'),

            // the scrollable node of this pane (embed into rootNode to allow to hide scroll bars if needed)
            // (attribute unselectable='on' is needed for IE to prevent focusing the node when clicking on a scroll bar)
            scrollNode = $('<div class="grid-scroll" unselectable="on">').appendTo(rootNode),

            // the node controlling the size and position of the scroll bars
            scrollSizeNode = $('<div class="grid-scroll-size">').appendTo(scrollNode),

            // the root node for all layer nodes, positioned dynamically in the scrollable area
            layerRootNode = $('<div class="grid-layer-root noI18n">').appendTo(scrollNode),

            // the collaborative layer node for displaying cursors of other users etc
            collaborativeLayerNode = $('<div class="grid-layer collab-layer">'),

            // the selection layer (container for the selected ranges)
            selectionLayerNode = $('<div class="grid-layer selection-layer">'),

            // the drawing layer (container for the drawing frames)
            drawingLayerNode = $('<div class="grid-layer drawing-layer">'),

            // the range highlight layer (container for the highlighted ranges of formulas)
            highlightLayerNode = $('<div class="grid-layer highlight-layer">'),

            // the active range layer (container for the active range selected in cell edit mode)
            activeRangeLayerNode = $('<div class="grid-layer active-layer">'),

            // the clipboard node that carries the browser focus for copy/paste events
            clipboardNode = null,
            clipboardFocusMethod = null,

            // the target node for the browser focus (either clipboard node, or the text area in cell edit mode)
            focusTargetNode = null,

            // drop-down button and menu for auto-completion and list validation of active cell
            cellListMenu = null,

            // the ratio between (public) absolute scroll position, and internal scroll bars
            scrollLeftRatio = 1,
            scrollTopRatio = 1,

            // distance between cell A1 and the top-left visible cell of this pane
            hiddenWidth = 0,
            hiddenHeight = 0,

            // the cell range and absolute position covered by the layer nodes in the sheet area
            layerRange = null,
            layerRectangle = null;

        // base constructor ---------------------------------------------------

        TriggerObject.call(this);
        GridTrackingMixin.call(this, app, panePos, layerRootNode);

        // private methods ----------------------------------------------------

        /**
         * Registers an event handler at the specified instance that will only
         * be invoked when this grid pane is visible.
         */
        function registerEventHandler(source, type, handler) {
            self.listenTo(source, type, function () {
                if (layerRange) {
                    handler.apply(self, _.toArray(arguments));
                }
            });
        }

        /**
         * Refreshes the appearance of this grid pane according to the position
         * of the grid pane that is currently active.
         */
        function updateFocusDisplay() {

            var // which header pane is currently focused (while selection tracking is active)
                focusPaneSide = view.getSheetViewAttribute('activePaneSide'),
                // which grid pane is currently focused
                focusPanePos = view.getSheetViewAttribute('activePane'),
                // whether this grid pane appears focused
                focused = false;

            if (view.isCellEditMode()) {
                // never highlight selection in cell in-place edit mode
                focused = false;
            } else if (view.hasFrozenSplit()) {
                // always highlight all panes in frozen split mode
                focused = true;
            } else if (_.isString(focusPaneSide)) {
                // a header pane is active (selection tracking): focus this
                // grid pane if it is related to the active header pane
                focused = (colPaneSide === focusPaneSide) || (rowPaneSide === focusPaneSide);
            } else {
                // a single grid pane is focused
                focused = focusPanePos === panePos;
            }

            rootNode.toggleClass(Forms.FOCUSED_CLASS, focused);
        }

        /**
         * Updates the position of the auto-fill handle and the resize handles
         * for cell selection, when entire columns or rows are selected.
         */
        function updateResizeHandlePositions() {

            var // the auto-fill resizer handle node
                autoFillHandleNode = selectionLayerNode.find('.autofill.resizers>[data-pos]'),
                // the range node containing the auto-fill handle
                autoFillRangeNode = autoFillHandleNode.closest('.range');

            // adjust position of the auto-fill handle node for entire column or row selection
            switch (autoFillHandleNode.attr('data-pos')) {
            case 'r':
                autoFillHandleNode.css({ top: -(layerRootNode.position().top + autoFillRangeNode.position().top) });
                break;
            case 'b':
                autoFillHandleNode.css({ left: -(layerRootNode.position().left + autoFillRangeNode.position().left) });
                break;
            }

            // adjust position of the selection handle node for entire column or row selection
            selectionLayerNode.find('.select.resizers>[data-pos]').each(function () {

                var handleNode = $(this),
                    rangeNode = handleNode.closest('.range');

                switch (handleNode.attr('data-pos')) {
                case 'l':
                case 'r':
                    handleNode.css({ top: -(layerRootNode.position().top + rangeNode.position().top) + (scrollNode[0].clientHeight / 2) });
                    break;
                case 't':
                case 'b':
                    handleNode.css({ left: -(layerRootNode.position().left + rangeNode.position().left) + (scrollNode[0].clientWidth / 2) });
                    break;
                }
            });
        }

        /**
         * Handles changed size of the scroll area received from the column or
         * row header pane.
         */
        function updateScrollAreaSize() {

            var // the available size according to existence of scroll bars
                clientWidth = scrollNode[0].clientWidth,
                clientHeight = scrollNode[0].clientHeight,
                // difference of scroll area size between header pane and this grid pane
                widthCorrection = rootNode.width() - clientWidth,
                heightCorrection = rootNode.height() - clientHeight,
                // the new absolute width and height of the scrollable area
                scrollWidth = Math.max(0, colHeaderPane.getScrollSize() - widthCorrection),
                scrollHeight = Math.max(0, rowHeaderPane.getScrollSize() - heightCorrection),
                // the real size of the scroll size node (may be smaller due to browser limits)
                effectiveSize = null;

            // initialize the size of the scroll sizer node, the resulting size may be smaller
            effectiveSize = Utils.setContainerNodeSize(scrollSizeNode, scrollWidth, scrollHeight);

            // recalculate ratio between absolute and effective scroll area size
            scrollLeftRatio = (effectiveSize.width > clientWidth) ? ((scrollWidth - clientWidth) / (effectiveSize.width - clientWidth)) : 0;
            scrollTopRatio = (effectiveSize.height > clientHeight) ? ((scrollHeight - clientHeight) / (effectiveSize.height - clientHeight)) : 0;
        }

        /**
         * Handles changed scroll position received from the column or row
         * header pane.
         */
        function updateScrollPosition() {

            var // the new DOM scroll position
                scrollLeft = (scrollLeftRatio > 0) ? Math.round(colHeaderPane.getScrollPos() / scrollLeftRatio) : 0,
                scrollTop = (scrollTopRatio > 0) ? Math.round(rowHeaderPane.getScrollPos() / scrollTopRatio) : 0,
                // the effective DOM node offsets according to the scroll position
                offsetLeft = 0,
                offsetTop = 0;

            // Only set new DOM scroll position if it differs from current position. This test
            // MUST be done, otherwise Firefox scrolls extremely slow when using the mouse wheel!
            if (scrollLeft !== scrollNode[0].scrollLeft) { scrollNode[0].scrollLeft = scrollLeft; }
            if (scrollTop !== scrollNode[0].scrollTop) { scrollNode[0].scrollTop = scrollTop; }

            // calculate the effective offsets according to current scroll position
            offsetLeft = Math.max(-layerRectangle.width * 2, layerRectangle.left - colHeaderPane.getScrollPos() + scrollLeft - hiddenWidth);
            offsetTop = Math.max(-layerRectangle.height * 2, layerRectangle.top - rowHeaderPane.getScrollPos() + scrollTop - hiddenHeight);

            // workaround IE positioning limitations (not possible to set an
            // absolute left/top position of more than Utils.MAX_NODE_SIZE pixels)
            Utils.setPositionInContainerNode(scrollNode, layerRootNode, offsetLeft, offsetTop);

            // set size of the layer root node (should always be inside the node size limits)
            layerRootNode.css({ width: layerRectangle.width, height: layerRectangle.height });

            // update the position of resizer handles (auto fill, touch selection) for column/row selection
            updateResizeHandlePositions();

            // notify listeners
            self.trigger('change:scrollpos');
        }

        /**
         * Invokes the passed iterator function for all cell ranges that are
         * located inside the visible area of this grid pane. The iterator
         * function will receive additional data needed for rendering those
         * ranges in some way.
         *
         * @param {Object|Array}
         *  The logical address of a single cell range, or an array of cell
         *  range addresses.
         *
         * @param {Function} iterator
         *  The iterator function that will be invoked for each range that is
         *  inside the visible area of this grid pane. Receives
         *  the following parameters:
         *  (1) {Object} range
         *      The logical address of the cell range to be rendered.
         *  (2) {Number} index
         *      The array index of the current cell range.
         *  (3) {Object} rectangle
         *      The location of the current range, relative to the layer root
         *      node of this grid pane.
         *  If the iterator returns the Utils.BREAK object, the iteration
         *  process will be stopped immediately.
         *
         * @returns {Utils.BREAK|Undefined}
         *  A reference to the Utils.BREAK object, if the iterator has returned
         *  Utils.BREAK to stop the iteration process, otherwise undefined.
         */
        function iterateRangesForRendering(ranges, iterator) {

            var // the bounding rectangle for range nodes (prevent oversized DOM nodes)
                boundRectangle = { width: Utils.MAX_NODE_SIZE, height: Utils.MAX_NODE_SIZE };

            // get start position of the bounding rectangle for range nodes, to prevent oversized DOM nodes
            boundRectangle.left = Math.max(0, layerRectangle.left - (boundRectangle.width - layerRectangle.width) / 2);
            boundRectangle.top = Math.max(0, layerRectangle.top - (boundRectangle.height - layerRectangle.height) / 2);

            // restrict to ranges in the current cell area to prevent oversized DOM nodes
            // that would collapse to zero size, e.g. when entire columns or rows are selected
            // (for details, see the comments for the Utils.MAX_NODE_SIZE constant)
            return Utils.iterateArray(_.getArray(ranges), function (range, index) {

                var // position and size of the selection range, restricted to the bounding rectangle
                    rectangle = DrawingUtils.getIntersectionRectangle(view.getRangeRectangle(range), boundRectangle);

                // skip ranges completely outside the bounding rectangle
                if (!rectangle) { return; }

                // convert range position relative to layer node
                rectangle.left -= layerRectangle.left;
                rectangle.top -= layerRectangle.top;

                // invoke the iterator
                return iterator.call(self, range, index, rectangle);
            });
        }

        function enlargeLeft(rectangle, size) {
            rectangle.left -= size;
            rectangle.width += size;
        }

        function enlargeTop(rectangle, size) {
            rectangle.top -= size;
            rectangle.height += size;
        }

        function enlargeTopLeft(rectangle, size) {
            enlargeLeft(rectangle, size);
            enlargeTop(rectangle, size);
        }

        /**
         * Renders all visible selection ranges according to the current pane
         * layout data received from a view layout update notification.
         */
        function renderCellSelection() {

            var // the entire cell selection
                selection = view.getSelection(),
                // the range occupied by the active cell (will not be filled)
                activeRange = null,
                // the position of the active cell, extended to the merged range
                activeRectangle = null,
                // additional data for active auto-fill tracking
                autoFillData = view.getSheetViewAttribute('autoFillData'),
                // whether the active cell touches the borders of selection ranges
                activeBorders = { left: false, top: false, right: false, bottom: false },
                // hide left/top border of selection ranges, if first column/row is hidden
                firstColHidden = !view.getColCollection().isEntryVisible(0),
                firstRowHidden = !view.getRowCollection().isEntryVisible(0),
                // the HTML mark-up for all selection ranges but the active range
                rangesMarkup = '',
                // the HTML mark-up for the active selection range
                activeMarkup = '';

            // returns the mark-up of a resizer handler according to the type of the passed range
            function createResizerMarkup(range, leadingCorner) {
                var hPos = leadingCorner ? 'l' : 'r', vPos = leadingCorner ? 't' : 'b';
                return '<div data-pos="' + (model.isColRange(range) ? hPos : model.isRowRange(range) ? vPos : (vPos + hPos)) + '"></div>';
            }

            // show drop-down button for list validation
            function createCellListButton() {

                var // validation settings for the active cell
                    validationSettings = null,
                    // the button node, as jQuery object
                    cellListButton = null,
                    // inner size of the drop-down button (according to cell height)
                    buttonInnerSize = 0;

                // do not show in read-only mode, or when auto-fill is active
                if (!activeRectangle || !model.getEditMode() || _.isObject(autoFillData)) { return; }

                // check that the active cell is inside the row interval covered by this grid pane (needed to prevent
                // painting the upper part of the button, if cell is located in the top row of the lower part of a frozen split)
                if (!SheetUtils.intervalContainsIndex(rowHeaderPane.getAvailableInterval(), selection.activeCell[1])) { return; }

                // check that the active cell provides a validation drop-down list
                validationSettings = view.getValidationCollection().getValidationSettings(selection.activeCell);
                if (!_.isObject(validationSettings) || !/^(source|list)$/.test(validationSettings.attributes.type) || !validationSettings.attributes.showDropDown) { return; }

                // create the button node
                cellListButton = $('<div class="cell-dropdown-button skip-tracking" tabindex="-1">' + Forms.createIconMarkup('fa-caret-down') + '</div>'),

                // set the button position, and insert the button into the DOM
                buttonInnerSize = Utils.minMax(activeRectangle.height, 18, 24);
                cellListButton.css({
                    width: buttonInnerSize,
                    height: buttonInnerSize,
                    lineHeight: buttonInnerSize + 'px',
                    top: Math.max(activeRectangle.top + activeRectangle.height - buttonInnerSize - 1, hiddenHeight) - activeRectangle.top,
                    marginLeft: activeBorders.right ? 2 : 0
                });
                selectionLayerNode.find('.active-cell').append(cellListButton);
            }

            // add active flag to range object
            selection.ranges[selection.activeRange].active = true;

            // adjust single range for auto-fill tracking
            if ((selection.ranges.length === 1) && _.isObject(autoFillData)) {

                var // the selected cell range
                    firstRange = selection.ranges[0],
                    // whether to expand/shrink the leading or trailing border
                    leading = /^(left|top)$/.test(autoFillData.border),
                    // whether to expand/shrink columns or rows
                    columns = /^(left|right)$/.test(autoFillData.border),
                    // the array index in the cell address
                    addrIndex = columns ? 0 : 1;

                activeRange = _.copy(firstRange, true);
                if (autoFillData.count >= 0) {
                    // adjust range for auto-fill mode
                    if (leading) {
                        firstRange.start[addrIndex] -= autoFillData.count;
                    } else {
                        firstRange.end[addrIndex] += autoFillData.count;
                    }
                } else {
                    // adjust range for deletion mode
                    firstRange.collapsed = true;
                    if (-autoFillData.count < SheetUtils.getIndexCount(firstRange, columns)) {
                        // range partly covered
                        if (leading) {
                            activeRange.start[addrIndex] -= autoFillData.count;
                        } else {
                            activeRange.end[addrIndex] += autoFillData.count;
                        }
                    } else {
                        // remove active range for deletion of the entire range
                        activeRange = {};
                    }
                }

                // add tracking style effect
                firstRange.tracking = true;
            }

            // the range covered by the active cell (or any other range in auto-fill tracking mode)
            activeRange = _.isObject(activeRange) ? (_.isEmpty(activeRange) ? null : activeRange) :
                view.getMergeCollection().expandRangeToMergedRanges({ start: selection.activeCell, end: selection.activeCell });

            // convert active range to a rectangle relative to the layer root node
            if (_.isObject(activeRange)) {
                iterateRangesForRendering(activeRange, function (range, index, rectangle) {
                    activeRectangle = rectangle;
                });
            }

            // render all ranges that are visible in this grid pane
            iterateRangesForRendering(selection.ranges, function (range, index, rectangle) {

                var // position of active range, relative to current range
                    relActiveRect = null,
                    // mark-up for the current range
                    rangeMarkup = '';

                // enlarge rectangle by one pixel to the left and top, to get matching
                // border line positions in multi-selection
                enlargeTopLeft(rectangle, 1);

                // hide left/top border of the range, if the first column/row is hidden
                if (firstColHidden && (range.start[0] === 0)) {
                    enlargeLeft(rectangle, 5);
                }
                if (firstRowHidden && (range.start[1] === 0)) {
                    enlargeTop(rectangle, 5);
                }

                // generate the HTML mark-up for the range
                rangeMarkup = '<div class="range';
                if (range.active) { rangeMarkup += ' active'; }
                if (range.tracking) { rangeMarkup += ' tracking-active'; }
                if (range.collapsed) { rangeMarkup += ' collapsed'; }
                rangeMarkup += '" style="' + PaneUtils.getRectangleStyleMarkup(rectangle) + '" data-index="' + index + '">';

                // insert the semi-transparent fill elements (never cover the active range with the fill elements)
                if (activeRectangle && SheetUtils.rangeOverlapsRange(range, activeRange)) {

                    // enlarge rectangle by one pixel to the left and top, to get matching
                    // border line positions in multi-selection
                    enlargeTopLeft(activeRectangle, 1);

                    // initialize position of active cell, relative to selection range
                    relActiveRect = _.clone(activeRectangle);
                    relActiveRect.left -= rectangle.left;
                    relActiveRect.top -= rectangle.top;
                    relActiveRect.right = rectangle.width - relActiveRect.left - relActiveRect.width;
                    relActiveRect.bottom = rectangle.height - relActiveRect.top - relActiveRect.height;

                    // insert fill element above active cell
                    if (relActiveRect.top > 0) {
                        rangeMarkup += '<div class="fill" style="left:0;right:0;top:0;height:' + relActiveRect.top + 'px;"></div>';
                    }

                    // insert fill element left of active cell
                    if (relActiveRect.left > 0) {
                        rangeMarkup += '<div class="fill" style="left:0;width:' + relActiveRect.left + 'px;top:' + relActiveRect.top + 'px;height:' + relActiveRect.height + 'px;"></div>';
                    }

                    // insert fill element right of active cell
                    if (relActiveRect.right > 0) {
                        rangeMarkup += '<div class="fill" style="left:' + (relActiveRect.left + relActiveRect.width) + 'px;right:0;top:' + relActiveRect.top + 'px;height:' + relActiveRect.height + 'px;"></div>';
                    }

                    // insert fill element below active cell
                    if (relActiveRect.bottom > 0) {
                        rangeMarkup += '<div class="fill" style="left:0;right:0;top:' + (relActiveRect.top + relActiveRect.height) + 'px;bottom:0;"></div>';
                    }

                    // update border flags for the active cell
                    activeBorders.left = activeBorders.left || (range.start[0] === activeRange.start[0]);
                    activeBorders.top = activeBorders.top || (range.start[1] === activeRange.start[1]);
                    activeBorders.right = activeBorders.right || (range.end[0] === activeRange.end[0]);
                    activeBorders.bottom = activeBorders.bottom || (range.end[1] === activeRange.end[1]);

                } else {
                    rangeMarkup += '<div class="abs fill"></div>';
                }

                // generate the HTML mark-up for the selection range
                rangeMarkup += '<div class="border"></div>';

                // additional mark-up for single-range selection
                if (selection.ranges.length === 1) {
                    if (Modernizr.touch && !_.isObject(autoFillData)) {
                        // add resize handlers for selection on touch devices
                        rangeMarkup += '<div class="select resizers">' + createResizerMarkup(range, true) + createResizerMarkup(range, false) + '</div>';
                    } else {
                        // add the auto-fill handler in the bottom right corner of the selection
                        rangeMarkup += '<div class="autofill resizers">' + createResizerMarkup(range, false) + '</div>';
                    }
                }

                // close the range node
                rangeMarkup += '</div>';

                // and the generated mark-up to the appropriate mark-up list
                if (range.active) { activeMarkup += rangeMarkup; } else { rangesMarkup += rangeMarkup; }
            });

            // additions for the active cell (or active range in auto-fill)
            if (activeRectangle) {

                // add thin border for active cell on top of the selection
                rangesMarkup += '<div class="active-cell';
                _.each(activeBorders, function (isBorder, borderName) { if (!isBorder) { rangesMarkup += ' ' + borderName; } });
                rangesMarkup += '" style="' + PaneUtils.getRectangleStyleMarkup(activeRectangle) + '"></div>';
            }

            // insert entire HTML mark-up into the selection container node
            selectionLayerNode[0].innerHTML = rangesMarkup + activeMarkup;

            // update the visibility/position of additional DOM elements
            createCellListButton();
            updateResizeHandlePositions();
            view.updateHyperlinkPopup();
        }

        /**
         * Renders all visible highlighted cell ranges.
         */
        function renderHighlightedRanges() {

            var // all ranges currently highlighted
                highlightRanges = view.getSheetViewAttribute('highlightRanges'),
                // unified range identifiers
                rangeIds = _.chain(highlightRanges).pluck('id').unique().value(),
                // array index of the range currently tracked
                highlightIndex = view.getSheetViewAttribute('highlightIndex'),
                // the HTML mark-up for all highlight ranges
                markup = '';

            // render all ranges that are visible in this grid pane
            if (_.isArray(highlightRanges)) {
                iterateRangesForRendering(highlightRanges, function (range, index, rectangle) {

                    var // array index of the range identifier (multiple ranges may origin from the same defined name)
                        idIndex = rangeIds.indexOf(range.id),
                        // CSS class name for range color style
                        styleId = (idIndex % Utils.SCHEME_COLOR_COUNT) + 1;

                    // generate the HTML mark-up for the highlighted range
                    markup += '<div class="range ' + ((highlightIndex === index) ? ' tracking-active' : '') + '" style="' + PaneUtils.getRectangleStyleMarkup(rectangle) + '"';
                    markup += ' data-style="' + styleId + '" data-index="' + index + '"><div class="fill"></div>';
                    if (model.getEditMode() && Utils.getBooleanOption(range, 'draggable', false)) {
                        markup += '<div class="borders">';
                        markup += _.map('lrtb', function (pos) { return '<div data-pos="' + pos + '"></div>'; }).join('');
                        markup += '</div><div class="resizers">';
                        markup += _.map(['tl', 'tr', 'bl', 'br'], function (pos) { return '<div data-pos="' + pos + '"></div>'; }).join('');
                        markup += '</div>';
                    } else {
                        markup += '<div class="static-border"></div>';
                    }
                    markup += '</div>';
                });
            }

            // insert entire HTML mark-up into the container node
            highlightLayerNode[0].innerHTML = markup;
        }

        /**
         * Renders the activated cell ranges. These ranges are used for example
         * while selecting new ranges for a formula while in cell edit mode.
         */
        var renderActiveSelection = (function () {

            var // the timer for the wireframe animation
                timer = null;

            return function () {

                var // the active ranges
                    selection = view.getSheetViewAttribute('activeSelection'),
                    // the HTML mark-up for the range
                    markup = '';

                // generate the HTML mark-up for the ranges
                if (_.isObject(selection)) {
                    iterateRangesForRendering(selection.ranges, function (range, index, rectangle) {
                        markup += '<div class="range" style="' + PaneUtils.getRectangleStyleMarkup(rectangle) + '"><div class="borders">';
                        markup += _.map('lrtb', function (pos) { return '<div data-pos="' + pos + '"></div>'; }).join('');
                        markup += '</div></div>';
                    });
                    if (!timer) {
                        timer = app.repeatDelayed(function (index) {
                            activeRangeLayerNode.attr('data-frame', index % 6);
                        }, { delay: 100 });
                    }
                } else if (timer) {
                    timer.abort();
                    timer = null;
                }

                // insert entire HTML mark-up into the container node
                activeRangeLayerNode[0].innerHTML = markup;
            };
        }());

        /**
         * Renders collaborative user selections.
         */
        function renderCollaborativeSelections() {

            var // current user selections to render
                collabSelections = view.getSheetViewAttribute('collaborativeSelections'),
                // the style color index (different per user)
                styleIndex = 0,
                // the HTML mark-up for all user selections
                markup = '';

            // iterate all collaborating users
            _.each(collabSelections, function (collabSelection) {

                // ignore own cursor data
                if ((collabSelection.userId === app.getClientId()) || !collabSelection.selection) { return; }

                var // the escaped user name, ready to be inserted into HTML mark-up
                    userName = Utils.escapeHTML(collabSelection.userName);

                // render all ranges of the user
                iterateRangesForRendering(collabSelection.selection.ranges, function (range, index, rectangle) {

                    // adjust left and top border to fit the range to the grid lines
                    enlargeTopLeft(rectangle, 1);

                    // create the mark-up for the range
                    markup += '<div class="range" data-style="' + ((styleIndex % Utils.SCHEME_COLOR_COUNT) + 1) + '" data-username="' + userName + '"';
                    markup += ' style="' + PaneUtils.getRectangleStyleMarkup(rectangle) + '"><div class="borders">';
                    markup += _.map('lrtb', function (pos) { return '<div data-pos="' + pos + '"></div>'; }).join('');
                    markup += '</div></div>';
                });

                // new color for next user
                styleIndex += 1;
            });

            // insert entire HTML mark-up into the container node
            collaborativeLayerNode[0].innerHTML = markup;
        }

        /**
         * Updates the position of the passed drawing frame according to the
         * anchor attributes in the passed attribute set.
         */
        function updateDrawingFramePosition(drawingModel, drawingFrame) {

            var // the effective drawing rectangle (null if the drawing is hidden)
                rectangle = drawingModel.getRectangle();

            if (rectangle) {
                // convert position relative to layer node
                rectangle.left -= layerRectangle.left;
                rectangle.top -= layerRectangle.top;
                // show the drawing frame, and set its position and size
                drawingFrame.show().css(rectangle);
            } else {
                // hide the drawing frame if it is located in hidden columns or rows
                drawingFrame.hide();
            }
        }

        /**
         * Creates and inserts a new DOM drawing frame node that represents the
         * passed drawing model.
         */
        function createDrawingFrame(drawingModel, position) {

            var // the new drawing frame, as jQuery object
                drawingFrame = DrawingFrame.createDrawingFrame(drawingModel),
                // the merged drawing attributes
                attributes = drawingModel.getMergedAttributes(),
                // all existing drawing frames, as plain DOM collection
                drawingNodes = drawingLayerNode[0].childNodes,
                // the DOM child index (Z order) specified by the logical position
                index = position[0];

            // add the unique identifier of the drawing model for DOM look-up
            drawingFrame.attr('data-model-uid', drawingModel.getUid());

            // insert the drawing frame with the correct Z order
            if (drawingNodes.length === index) {
                drawingLayerNode.append(drawingFrame);
            } else {
                drawingFrame.insertBefore(drawingNodes[index]);
            }

            // update position and formatting
            updateDrawingFramePosition(drawingModel, drawingFrame);
            DrawingFrame.updateFormatting(app, drawingFrame, attributes);
        }

        /**
         * Updates the position and formatting of the drawing frame that
         * represents the passed drawing model.
         */
        function updateDrawingFrame(drawingModel) {

            var // the existing DOM drawing frame
                drawingFrame = self.getDrawingFrame(drawingModel.getUid());

            // update position and formatting of frames inside the visible area
            if (SheetUtils.rangeOverlapsRange(layerRange, drawingModel.getRange())) {
                updateDrawingFramePosition(drawingModel, drawingFrame);
                DrawingFrame.updateFormatting(app, drawingFrame, drawingModel.getMergedAttributes());
            } else {
                drawingFrame.hide();
            }
        }

        /**
         * Updates position and formatting of all drawing frames in the current
         * visible area.
         */
        function updateDrawingFrames() {
            view.getDrawingCollection().iterateModelsByPosition(updateDrawingFrame);
        }

        /**
         * Updates the selection of all drawing frames.
         */
        function renderDrawingSelection() {

            var // all drawing frames currently selected in the DOM (also embedded drawings)
                oldSelectedFrames = drawingLayerNode.find(DrawingFrame.NODE_SELECTOR + Forms.SELECTED_SELECTOR),
                // all existing drawing frames, as plain DOM collection
                drawingNodes = drawingLayerNode[0].childNodes,
                // disable tracking in read-only mode, or if sheet is protected
                editMode = model.getEditMode() && !view.isSheetLocked();

            // process all drawings to be selected
            _.each(view.getSelectedDrawings(), function (position) {

                var // the root drawing frame addressed by the current logical position
                    drawingFrame = drawingNodes[position[0]];

                // drawing frame may be missing, e.g. while remotely inserting new drawings
                // TODO: find embedded drawing frames
                if (drawingFrame && (position.length === 1)) {
                    DrawingFrame.drawSelection(drawingFrame, { movable: editMode, resizable: editMode });
                    oldSelectedFrames = oldSelectedFrames.not(drawingFrame);
                }
            });

            // remove selection border from all drawing frames not selected anymore
            oldSelectedFrames.each(function () { DrawingFrame.clearSelection(this); });
        }

        /**
         * Processes a new drawing object inserted into the drawing collection.
         */
        function insertDrawingHandler(event, drawingModel, position) {
            createDrawingFrame(drawingModel, position);
            renderDrawingSelection();
        }

        /**
         * Processes a drawing object removed from the drawing collection.
         */
        function deleteDrawingHandler(event, drawingModel) {
            self.getDrawingFrame(drawingModel.getUid()).remove();
            renderDrawingSelection();
        }

        /**
         * Processes a drawing object that has been changed in any way.
         */
        function changeDrawingHandler(event, drawingModel) {
            updateDrawingFrame(drawingModel);
        }

        /**
         * Updates the layer range and the DOM layer nodes. Removes DOM cell
         * nodes of all cells not visible anymore, and renders new cell nodes
         * now visible in this grid pane.
         *
         * @param {Object|Null} interval
         *  The new column/row interval to be inserted into the layer range. If
         *  set to null, the grid pane will be hidden, and all layer nodes will
         *  be cleared.
         *
         * @param {Boolean} columns
         *  Whether the modified layer range was caused by a change in the
         *  column header pane (true) or row header pane (false).
         *
         * @param {Object} [refresh]
         *  The refresh strategy for the columns/rows in the new interval, as
         *  passed in the respective 'change:interval' event of a header pane.
         *
         * @param {Object} [options]
         *  Additional options passed with the respective 'change:interval'
         *  event of a header pane.
         */
        var updateLayerInterval = (function () {

            var // the cached interval/refresh settings (needed to build the layer range from several events)
                colSettings = { interval: null, refresh: {} },
                rowSettings = { interval: null, refresh: {} };

            // direct callback: collect interval settings from column/row header pane
            function registerInterval(interval, columns, refresh) {

                var // store the passed column/row interval in the internal variable
                    settings = columns ? colSettings : rowSettings,
                    // the index of the first column/row that needs full refresh
                    index = Utils.getIntegerOption(refresh, 'index');

                RenderUtils.log('GridPane.registerInterval(): pos=' + panePos + ' interval=' + (interval ? SheetUtils.getIntervalName(interval, columns) : 'null') + ' refresh=' + JSON.stringify(refresh));

                // store the new interval, update refresh settings
                if (_.isObject(interval)) {
                    settings.interval = _.clone(interval);
                    if (_.isNumber(index)) {
                        settings.refresh.index = _.isNumber(settings.refresh.index) ? Math.min(settings.refresh.index, index) : index;
                    }
                    settings.refresh.geometry = settings.refresh.geometry || Utils.getBooleanOption(refresh, 'geometry', false);
                } else {
                    settings.interval = null;
                    settings.refresh = {};
                }
            }

            // deferred callback: calculate the new layer range
            function calculateLayerRange() {

                var // the old (current) layer range
                    oldLayerRange = layerRange;

                // calculate the new layer range
                layerRange = (colSettings.interval && rowSettings.interval) ?
                    SheetUtils.makeRangeFromIntervals(colSettings.interval, rowSettings.interval) : null;

                RenderUtils.log('GridPane.calculateLayerRange(): pos=' + panePos + ' layerRange=' + (layerRange ? SheetUtils.getRangeName(layerRange) : 'null') + ' colRefresh=' + JSON.stringify(colSettings.refresh) + ' rowRefresh=' + JSON.stringify(rowSettings.refresh));

                // update visible sheet contents according to the new layer range
                if (layerRange) {

                    // get the new layer rectangle in pixels
                    layerRectangle = view.getRangeRectangle(layerRange);

                    // update the scroll position (exact position of the layer root node in the DOM)
                    updateScrollPosition();

                    // update the layers of the cell and grid renderer
                    gridRenderer.updateLayerRange(layerRange, {
                        col: colSettings.refresh.index,
                        row: rowSettings.refresh.index,
                        geometry: colSettings.refresh.geometry || rowSettings.refresh.geometry
                    });

                    // update all own DOM layer nodes
                    renderCellSelection();
                    renderHighlightedRanges();
                    renderActiveSelection();
                    renderCollaborativeSelections();

                    // create all drawing frames if not done yet (e.g. enabled split in current sheet)
                    if (!drawingLayerNode[0].hasChildNodes()) {
                        view.getDrawingCollection().iterateModelsByPosition(createDrawingFrame);
                        renderDrawingSelection();
                    } else {
                        // repaint all visible drawing frames
                        updateDrawingFrames();
                    }
                }

                // do nothing, if the layer range was invalid before
                else if (oldLayerRange) {

                    layerRectangle = null;

                    // clear the layers of the cell and grid renderer
                    gridRenderer.clearLayerRange();

                    // safely destroy all image nodes to prevent memory problems on iPad
                    app.destroyImageNodes(drawingLayerNode.find('img'));

                    // clear all own DOM layers
                    collaborativeLayerNode.empty();
                    selectionLayerNode.empty();
                    drawingLayerNode.empty();
                    highlightLayerNode.empty();
                    activeRangeLayerNode.empty();
                }

                // reset cached refresh data (but keep the current interval)
                colSettings.refresh = {};
                rowSettings.refresh = {};
            }

            // debounced interval handling
            var calculateLayerRangeDebounced = app.createDebouncedMethod(registerInterval, calculateLayerRange);

            // the actual updateLayerInterval() method to be returned from the local scope
            function updateLayerInterval(interval, columns, refresh, options) {
                // refresh layer range directly depending on the passed option, but always if the interval is deleted
                if (!interval || Utils.getBooleanOption(options, 'direct', false)) {
                    registerInterval(interval, columns, refresh);
                    calculateLayerRange();
                } else {
                    calculateLayerRangeDebounced(interval, columns, refresh);
                }
            }

            return updateLayerInterval;
        }());

        /**
         * Updates the selection and highlight layers after specific sheet view
         * attributes have been changed.
         */
        function changeSheetViewAttributesHandler(event, attributes) {

            // render cell and drawing selection
            if (Utils.hasProperty(attributes, /^selection$|^split/)) {
                cellListMenu.hide();
                renderCellSelection();
                renderDrawingSelection();
            }

            // display collaborative user cursors
            if ('collaborativeSelections' in attributes) {
                renderCollaborativeSelections();
            }

            // refresh focus display
            if (Utils.hasProperty(attributes, /^activePane/)) {
                cellListMenu.hide();
                updateFocusDisplay();
            }

            // changed auto-fill tracking data
            if ('autoFillData' in attributes) {
                cellListMenu.hide();
                renderCellSelection();
            }

            // changed highlight ranges
            if (Utils.hasProperty(attributes, /^highlight/)) {
                renderHighlightedRanges();
            }

            // changed active ranges (used while selecting ranges in cell edit mode)
            if ('activeSelection' in attributes) {
                renderActiveSelection();
            }
        }

        /**
         * Prepares this grid pane for entering the in-place cell edit mode.
         */
        function cellEditEnterHandler() {
            cellListMenu.hide();
            // disable global F6 focus traveling into this grid pane while in-place cell edit mode is active in another grid pane
            rootNode.toggleClass('f6-target', self === view.getActiveGridPane());
            updateFocusDisplay();
        }

        /**
         * Prepares this grid pane for leaving the in-place cell edit mode.
         */
        function cellEditLeaveHandler() {
            rootNode.addClass('f6-target');
            updateFocusDisplay();
        }

        /**
         * Renders the cell grid, the selection, and the highlighted ranges,
         * after the cell geometry has been changed.
         */
        var changeCellLayoutHandler = app.createDebouncedMethod($.noop, function () {
            renderCellSelection();
            renderHighlightedRanges();
        });

        /**
         * Handles DOM 'scroll' events of the scrollable node.
         */
        function scrollHandler() {
            colHeaderPane.scrollTo(Math.round(scrollNode[0].scrollLeft * scrollLeftRatio));
            rowHeaderPane.scrollTo(Math.round(scrollNode[0].scrollTop * scrollTopRatio));
        }

        /**
         * Returns the drop-down button control attached to the active cell.
         */
        function getCellListButton() {
            return selectionLayerNode.find('.cell-dropdown-button');
        }

        /**
         * Shows the pop-up menu for list validation.
         */
        var cellListButtonClickHandler = (function () {

            var // the server request currently running
                runningRequest = null;

            // return the actual 'cellListButtonClickHandler()' method
            return function cellListButtonClickHandler() {

                // cancel old server request still running
                if (runningRequest) {
                    runningRequest.abort();
                    runningRequest = null;
                }

                // leave cell edit mode (do nothing, if formula validation fails)
                if (!view.leaveCellEditMode('auto', { validate: true })) { return; }

                // close menu and return, if it is visible
                if (cellListMenu.isVisible()) {
                    cellListMenu.hide();
                    self.grabFocus();
                    return;
                }

                // open and initialize the menu
                cellListMenu.clearContents().appendContentNodes($('<div style="height:26px;">').busy()).show();

                // receive source data from server
                runningRequest = view.getValidationCollection().queryListValues(view.getActiveCell());

                // fill list menu with values received from server
                runningRequest.done(function (values) {

                    // do nothing if the menu has been closed indirectly in the meantime (e.g. after selection change)
                    if (!cellListMenu.isVisible()) { return; }

                    // close the menu, if no data is available for the list
                    if (values.length === 0) {
                        cellListMenu.hide();
                        return;
                    }

                    // insert the data into the list
                    cellListMenu.clearContents();
                    _.each(values, function (value) {
                        cellListMenu.createItemNode(value, { label: _.noI18n(value) });
                    });
                });

                // reset the 'runningRequest' variable
                runningRequest.always(function () { runningRequest = null; });
            };
        }());

        /**
         * Inserts the value represented by the passed list item from the cell
         * pop-up menu into the cell.
         */
        function applyCellListValue(buttonNode) {

            var // the value of the selected list item
                itemValue = (buttonNode.length > 0) ? Forms.getButtonValue(buttonNode) : null;

            // always hide the pop-up menu
            cellListMenu.hide();

            // insert the value into the cell
            if (_.isString(itemValue)) { view.setCellContents(itemValue); }
        }

        /**
         * Updates the layout of the cell pop-up menu. Hides the menu if the
         * anchor cell is not visible anymore, otherwise adjusts the minimum
         * width of the menu node according to the width of the cell.
         */
        function cellListBeforeLayoutHandler() {

            var // the drop-down button control
                cellListButton = getCellListButton(),
                // the active cell attached to the drop-down button
                cellNode = cellListButton.parent();

            // do nothing if drop-down button is not visible at the moment
            if (cellNode.length === 0) { return; }

            // check the position of the cell node in the scrollable area
            if (!Utils.isChildNodeVisibleInNode(scrollNode, cellNode)) {
                cellListMenu.hide();
                return;
            }

            // update minimum width of the menu sections according to cell width (plus width of drop-down button)
            cellListMenu.getContentNode().children().css({ minWidth: cellListButton.offset().left + cellListButton.outerWidth() - cellNode.offset().left });
        }

        /**
         * Handles clicks in the cell pop-up menu.
         */
        function cellListClickHandler(event) {
            applyCellListValue($(event.currentTarget));
        }

        /**
         * Handles 'keydown' events from the root node.
         */
        function keyDownHandler(event) {

            // ESCAPE key (only one action per key event)
            if (event.keyCode === KeyCodes.ESCAPE) {

                // cancel current tracking, but do nothing else
                if ($.cancelTracking()) {
                    return false;
                }

                // cancel custom selection mode (before deselecting drawings)
                if (view.leaveCustomSelectionMode()) {
                    return false;
                }


                // drawing selection: back to cell selection
                if (view.hasDrawingSelection()) {
                    view.removeDrawingSelection();
                    return false;
                }

                // close cell pop-up menu
                if (cellListMenu.isVisible()) {
                    cellListMenu.hide();
                    return false;
                }

                // otherwise: let ESCAPE key bubble up
                return;
            }

            // show cell pop-up menu
            if (KeyCodes.matchKeyCode(event, 'DOWN_ARROW', { alt: true })) {
                var cellListButton = getCellListButton();
                if (!cellListMenu.isVisible() && (cellListButton.length > 0)) {
                    self.scrollToCell(view.getActiveCell());
                    cellListButton.click();
                }
                return false;
            }

            // handle keyboard shortcuts for opened cell pop-up menu
            if (cellListMenu.isVisible()) {
                var itemNode = cellListMenu.getSelectedItemNodes().first();

                // apply the selected value
                if (KeyCodes.matchKeyCode(event, 'TAB', { shift: null }) || KeyCodes.matchKeyCode(event, 'ENTER', { shift: null })) {
                    applyCellListValue(itemNode);
                    return; // let event bubble up for selection change
                }

                // change selection
                itemNode = cellListMenu.getItemNodeForKeyEvent(event, itemNode);
                if (itemNode.length > 0) {
                    cellListMenu.selectItemNode(itemNode, { scroll: true });
                    return false;
                }
            }
        }

        /**
         * Handles 'drop' events from the root node.
         */
        function dropHandler(event){

            function paintImage(imageHolder){
                var // create the image DOM node, wait for it to load
                    createImagePromise = app.createImageNode(ImageUtil.getFileUrl(app, imageHolder.url));

                // on success, create the drawing object in the document
                createImagePromise.done(function (imgNode) {

                    var // current width of the image, in 1/100 mm
                        width = Utils.convertLengthToHmm(imgNode[0].width, 'px'),
                        // current height of the image, in 1/100 mm
                        height = Utils.convertLengthToHmm(imgNode[0].height, 'px'),
                        // active cell as target position for the image
                        activeCell = view.getActiveCell();

                    // restrict size to fixed limit
                    if (width > height) {
                        if (width > MAX_IMAGE_SIZE) {
                            height = Math.round(height * MAX_IMAGE_SIZE / width);
                            width = MAX_IMAGE_SIZE;
                        }
                    } else {
                        if (height > MAX_IMAGE_SIZE) {
                            width = Math.round(width * MAX_IMAGE_SIZE / height);
                            height = MAX_IMAGE_SIZE;
                        }
                    }

                    view.insertDrawing('image', { drawing: {
                        anchorType: 'oneCell',
                        startCol: activeCell[0],
                        startRow: activeCell[1],
                        width: width,
                        height: height,
                        name: imageHolder.name,
                        imageUrl: imageHolder.url
                    }}, function(){
                        view.leaveBusy();
                    });

                    app.destroyImageNodes(imgNode);
                })
                .fail(function(){
                    view.leaveBusy();
                });
            }

            // image comes from local hard drive
            if (event.originalEvent.dataTransfer.files.length > 0) {
                _.each(event.originalEvent.dataTransfer.files, function(eventFile){

                    if (ImageUtil.hasFileImageExtension(eventFile.name)) {
                        view.enterBusy();

                        ImageUtil.insertFile(app, eventFile)
                        .done(function(imageHolder) {
                            paintImage(imageHolder);
                        })
                        .fail(function(){
                            view.leaveBusy();
                        });

                    } else {
                        view.leaveBusy();
                        view.yell({ type: 'info', message: gt('File type not supported.') });
                    }
                });

            // image comes from another browser-window
            } else if(event.originalEvent.dataTransfer.getData('Url').length > 0) {
                view.enterBusy();
                var url = event.originalEvent.dataTransfer.getData('Url');

                if (ImageUtil.hasUrlImageExtension(url)) {
                    ImageUtil.insertURL(app, url)
                    .done(function(imageHolder){
                        if (ImageUtil.hasUrlImageExtension(imageHolder.url)) {
                            paintImage(imageHolder);
                        }
                    })
                    .fail(function(){
                        view.leaveBusy();
                    });
                } else {
                    view.leaveBusy();
                    view.yell({ type: 'info', message: gt('File type not supported.') });
                }
            }

        }

        /**
         * Sets the browser focus to the clipboard node, and selects all its
         * contents.
         */
        function grabClipboardFocus() {

            var // the browser selection
                selection = window.getSelection(),
                // a browser selection range object
                docRange = null;

            // calls the native focus() method at the clipboard node
            function grabNativeFocus() {
                clipboardFocusMethod.call(clipboardNode[0]);
            }

            // bug 29948: prevent grabbing to clipboard node while in cell in-place edit mode
            if (view.isCellEditMode()) { return; }

            // on touch devices, selecting text nodes in a content-editable node will
            // always show the virtual keyboard
            // TODO: need to find a way to provide clipboard support
            if (Modernizr.touch) {
                grabNativeFocus();
                return;
            }

            // Bug 26283: Browsers are picky how to correctly focus and select text
            // in a node that is content-editable: Chrome and IE will change the
            // scroll position of the editable node, if it has been focused
            // explicitly while not having an existing browser selection (or even
            // if the selection is not completely visible). Furthermore, the node
            // will be focused automatically when setting the browser selection.
            // Firefox, on the other hand, wants to have the focus already set
            // before the browser selection can be changed, otherwise it may throw
            // exceptions. Additionally, changing the browser selection does NOT
            // automatically set the focus into the editable node.
            if (_.browser.Firefox) {
                grabNativeFocus();
            }

            // Clear the old browser selection.
            // Bug 28515, bug 28711: IE fails to clear the selection (and to modify
            // it afterwards), if it currently points to a DOM node that is not
            // visible anymore (e.g. the 'Show/hide side panel' button). Workaround
            // is to move focus to an editable DOM node which will cause IE to update
            // the browser selection object. The target container node cannot be used
            // for that, see comments above for bug 26283. Using another focusable
            // node (e.g. the body element) is not sufficient either. Interestingly,
            // even using the (editable) clipboard node does not work here. Setting
            // the new browser selection below will move the browser focus back to
            // the application pane.
            Utils.clearBrowserSelection();

            // set the browser selection
            try {
                docRange = window.document.createRange();
                docRange.setStart(clipboardNode[0], 0);
                docRange.setEnd(clipboardNode[0], clipboardNode[0].childNodes.length);
                selection.addRange(docRange);
            } catch (ex) {
                Utils.error('GridPane.grabClipboardFocus(): failed to select clipboard node: ' + ex);
            }
        }

        /**
         * Updates the grid pane after the document edit mode has changed.
         */
        function editModeHandler(event, editMode) {

            // nothing to do, if this grid pane is currently not visible
            if (!layerRange) { return; }

            // hide pop-up menus, if document changes to read-only mode
            if (!editMode && app.isImportFinished()) {
                cellListMenu.hide();
            }

            // redraw selection according to edit mode
            renderCellSelection();
            renderDrawingSelection();
            renderHighlightedRanges();
        }

        /**
         * Handles 'cut' events of the clipboard node.
         * Invokes the copyHandler, then deletes the values and clears
         * the attributes of the selected cells.
         */
        function cutHandler(event) {

            var // the clipboard event data
                clipboardData = event && event.originalEvent && event.originalEvent.clipboardData;

            // check for cell protection
            if (!view.requireEditableSelection()) {
                event.preventDefault();
                return false;
            }

            // invoke clipboard copy handler first
            copyHandler(event);

            if (clipboardData) {
                // prevent default cut handling for desktop browsers, but not for touch devices
                if (!Modernizr.touch) {
                    event.preventDefault();
                }

                // delete values and clear attributes of the selected cells
                view.fillCellRanges(null, undefined, { clear: true });

            } else {
                app.executeDelayed(function () {
                    // delete values and clear attributes of the selected cells
                    view.fillCellRanges(null, undefined, { clear: true });
                });
            }
        }

        /**
         * Handles 'copy' events of the clipboard node.
         * Creates a HTML table of the active range of the current selection
         * and generates the client clipboard id. Adds both to the system clipboard.
         * And invokes the Calcengine copy operation.
         */
        function copyHandler(event) {

            var // model of the active sheet
                sheetModel = view.getSheetModel(),
                // current selection in the active sheet
                selection = view.getSelection(),
                // the active cell range in the selection
                activeRange = selection.ranges[selection.activeRange],
                // the clipboard event data
                clipboardData = event && event.originalEvent && event.originalEvent.clipboardData,
                // the client clipboard id to identify the copy data when pasting
                clientClipboardId = view.createClientClipboardId(),
                // the html table string to export
                htmlTable,
                // the plain text to export
                plainText;

            // restrict the number of cells to copy at the same time
            if (SheetUtils.getCellCount(activeRange) > SheetUtils.MAX_FILL_CELL_COUNT) {
                view.yell({ type: 'info', message: gt('It is not possible to copy more than %1$d cells at the same time.', _.noI18n(SheetUtils.MAX_FILL_CELL_COUNT)) });
                event.preventDefault();
                return false;
            }

            // generate HTML table and plain text representation from the active range of the current selection
            htmlTable = Clipboard.getHTMLStringFromRange(app, sheetModel, activeRange, clientClipboardId);
            plainText = Clipboard.getPlainTextFromRange(sheetModel, activeRange);

            // initiate clipboard server copy call
            view.clipboardServerCopy(selection);
            Utils.log('clipboard - copy - client id: ' + clientClipboardId);

            // set clipboard debug pane content
            self.trigger('debug:clipboard', htmlTable);

            // if browser supports clipboard api add data to the event
            if (clipboardData) {
                // add plain text and html of the current browser selection
                clipboardData.setData('text/plain', plainText);
                clipboardData.setData('text/html', htmlTable);

                // prevent default copy handling for desktop browsers, but not for touch devices
                if (!Modernizr.touch) {
                    event.preventDefault();
                }
            } else {
                // add Excel XML name spaces for custom tag support
                $('html').attr({
                    'xmlns': 'http://www.w3.org/TR/REC-html40',
                    'xmlns:o': 'urn:schemas-microsoft-com:office:office',
                    'xmlns:x': 'urn:schemas-microsoft-com:office:excel'
                });

                // add content and focus
                app.destroyImageNodes(clipboardNode);
                clipboardNode.empty().append(htmlTable);
                grabClipboardFocus();

                // remove XML name spaces again
                app.executeDelayed(function () {
                    $('html').removeAttr('xmlns xmlns:o xmlns:x');
                });
            }
        }

        /**
         * Handles 'paste' events of the clipboard node.
         */
        function pasteHandler(event) {
            var // current selection in the active sheet
                selection = view.getSelection(),
                // the clipboard event data
                clipboardData = event && event.originalEvent && event.originalEvent.clipboardData,
                // the cleaned up html data
                htmlData,
                // the html data attached to the event
                htmlRawData,
                // the plain text data attached to the event
                textData;


            function pasteHtmlClipboard(html) {
                var // the client clipboard id to identify the copy data when pasting
                    clientClipboardId = Clipboard.getClientClipboardId(html),
                    // the server clipboard id to identify the copy data when pasting
                    serverClipboardId = view.getServerClipboardId(clientClipboardId);

                if (serverClipboardId) {
                    // set clipboard debug pane content
                    self.trigger('debug:clipboard', 'Server side paste with clipboard id: ' + serverClipboardId);
                    // server paste call
                    view.clipboardServerPaste(selection, serverClipboardId);
                    Utils.log('clipboard - server side paste');
                } else {
                    // parse HTML
                    handleParserResult(Clipboard.parseHTMLData(html, documentStyles));
                    Utils.log('clipboard - client side html paste');
                }
            }

            function pasteTextClipboard(text) {
                // set clipboard debug pane content
                self.trigger('debug:clipboard', text);
                // parse text
                handleParserResult(Clipboard.parseTextData(text));
                Utils.log('clipboard - client side text paste');
            }

            function handleParserResult(result) {

                var // the target range of the paste
                    pasteRange = Clipboard.createRangeFromParserResult(selection.activeCell, result),
                    // the number of cells to paste
                    pasteCellCount = SheetUtils.getCellCount(pasteRange);

                // restrict the number of cells to paste at the same time
                if (pasteCellCount > SheetUtils.MAX_FILL_CELL_COUNT) {
                    view.yell({ type: 'info', message: gt('It is not possible to paste more than %1$d cells at the same time.', _.noI18n(SheetUtils.MAX_FILL_CELL_COUNT)) });
                    return;
                }

                // check for protected cells
                view.areRangesEditable(pasteRange).done(function () {

                    // create a single undo action for all affected operations
                    model.getUndoManager().enterUndoGroup(function () {

                        var // model of the active sheet
                            sheetModel = view.getSheetModel(),
                            // the ranges to be merged before pasting
                            mergeRanges = [];

                        // unmerge cells if necessary
                        if ((pasteCellCount > 1) && sheetModel.getMergeCollection().rangesOverlapAnyMergedRange(pasteRange)) {
                            sheetModel.mergeRanges(pasteRange, 'unmerge');
                        }

                        // merge cells
                        _.each(result.mergeCollection, function (range) {
                            range = model.getCroppedMovedRange(range, selection.activeCell[0], selection.activeCell[1]);
                            if (range) { mergeRanges.push(range); }
                        });
                        if (mergeRanges.length > 0) {
                            sheetModel.mergeRanges(mergeRanges, 'merge');
                        }

                        // set cell contents
                        sheetModel.setCellContents(view.getActiveCell(), result.contents, { parse: true });
                    });
                })
                .fail(function () {
                    view.yell({ type: 'info', message: gt('Pasting into protected cells is not allowed.') });
                });
            }


            // check if we have edit rigths before pasting
            if (!model.getEditMode()) {
                event.preventDefault();
                app.rejectEditAttempt();
                return false;
            }

            htmlRawData = clipboardData && clipboardData.getData('text/html');
            htmlData = Utils.cleanUpHtml(htmlRawData);

            // check if clipboard event data contains html
            if (Clipboard.containsHtmlTable(htmlData)) {
                // prevent default paste handling for desktop browsers, but not for touch devices
                if (!Modernizr.touch) {
                    event.preventDefault();
                }

                // set clipboard debug pane content
                self.trigger('debug:clipboard', htmlRawData);
                // render the html to apply css styles provided by Excel or Calc
                app.destroyImageNodes(clipboardNode);
                clipboardNode.empty().append(htmlData);
                // parse and create operations
                pasteHtmlClipboard(clipboardNode);
                // clear clipboard
                clipboardNode.text('\xa0');

                return false;
            }

            // focus and select the clipboard node
            grabClipboardFocus();

            // check if clipboard event data contains plain text
            // to be used in case the clipboard div contains no html table to parse
            textData = (_.browser.IE) ? window.clipboardData && window.clipboardData.getData('text') : clipboardData && clipboardData.getData('text/plain');

            // read and parse pasted data
            app.executeDelayed(function () {
                if (Clipboard.containsHtmlTable(clipboardNode)) {
                    // set clipboard debug pane content
                    self.trigger('debug:clipboard', clipboardNode);
                    pasteHtmlClipboard(clipboardNode);
                } else if (textData) {
                    pasteTextClipboard(textData);
                }
                // clear clipboard
                clipboardNode.text('\xa0');
            });
        }

        // methods ------------------------------------------------------------

        /**
         * Returns the root DOM node of this grid pane.
         *
         * @returns {jQuery}
         *  The root node of this grid pane, as jQuery object.
         */
        this.getNode = function () {
            return rootNode;
        };

        /**
         * Returns whether the root node of this grid pane is currently focused
         * or contains a DOM node that is currently focused (e.g. the text area
         * node while edit mode is active).
         *
         * @returns {Boolean}
         *  Whether this grid pane is currently focused.
         */
        this.hasFocus = function () {
            return focusTargetNode.is(window.document.activeElement);
        };

        /**
         * Sets the browser focus into this grid pane.
         *
         * @returns {GridPane}
         *  A reference to this instance.
         */
        this.grabFocus = function () {
            focusTargetNode.focus();
            return this;
        };

        /**
         * Initializes the settings and layout of this grid pane.
         *
         * @param {Object} colSettings
         *  The view settings of the horizontal pane side (left or right).
         *  Supports the following properties:
         *  @param {Boolean} colSettings.visible
         *      Whether the column header pane is visible.
         *  @param {Number} colSettings.offset
         *      The absolute horizontal offset of the grid pane in the view
         *      root node, in pixels.
         *  @param {Number} colSettings.size
         *      The total outer width of the grid pane, in pixels.
         *  @param {Boolean} colSettings.frozen
         *      Whether the grid pane is frozen (not scrollable) horizontally.
         *  @param {Boolean} colSettings.showOppositeScroll
         *      Whether the vertical scroll bar will be visible or hidden
         *      outside the pane root node.
         *  @param {Number} colSettings.hiddenSize
         *      The total width of the hidden columns in front of the sheet in
         *      frozen view mode.
         *
         * @param {Object} rowSettings
         *  The view settings of the vertical pane side (top or bottom).
         *  Supports the following properties:
         *  @param {Boolean} rowSettings.visible
         *      Whether the row header pane is visible.
         *  @param {Number} rowSettings.offset
         *      The absolute vertical offset of the grid pane in the view root
         *      node, in pixels.
         *  @param {Number} rowSettings.size
         *      The total outer height of the grid pane, in pixels.
         *  @param {Boolean} rowSettings.frozen
         *      Whether the grid pane is frozen (not scrollable) vertically.
         *  @param {Boolean} rowSettings.showOppositeScroll
         *      Whether the horizontal scroll bar will be visible or hidden
         *      outside the pane root node.
         *  @param {Number} rowSettings.hiddenSize
         *      The total height of the hidden rows in front of the sheet in
         *      frozen view mode.
         *
         * @returns {GridPane}
         *  A reference to this instance.
         */
        this.initializePaneLayout = function (colSettings, rowSettings) {

            // initialize according to visibility
            if (colSettings.visible && rowSettings.visible) {

                // show pane root node and initialize auto-scrolling
                rootNode.show();
                layerRootNode.enableTracking(Utils.extendOptions(PaneUtils.DEFAULT_TRACKING_OPTIONS, {
                    autoScroll: colSettings.frozen ? (rowSettings.frozen ? false : 'vertical') : (rowSettings.frozen ? 'horizontal' : true),
                    borderNode: rootNode
                }));

                // position and size
                rootNode.css({ left: colSettings.offset, top: rowSettings.offset, width: colSettings.size, height: rowSettings.size });

                // initialize the scroll node containing the scroll bars
                scrollNode.css({
                    overflowX: colSettings.frozen ? 'hidden' : '',
                    overflowY: rowSettings.frozen ? 'hidden' : '',
                    bottom: (!colSettings.frozen && !rowSettings.showOppositeScroll) ? -Math.max(10, Utils.SCROLLBAR_HEIGHT) : 0,
                    right: (!rowSettings.frozen && !colSettings.showOppositeScroll) ? -Math.max(10, Utils.SCROLLBAR_WIDTH) : 0
                });

                // additional layout information for frozen splits
                hiddenWidth = colSettings.hiddenSize;
                hiddenHeight = rowSettings.hiddenSize;

                // update scroll area size annd CSS classes at root node for focus display
                updateScrollAreaSize();
                updateFocusDisplay();

            } else {

                // hide pane root node and deinitialize auto-scrolling
                rootNode.hide();
                layerRootNode.disableTracking();
            }

            return this;
        };

        /**
         * Returns whether this grid pane is currently visible (depending on
         * its position, and the view split settings of the active sheet).
         *
         * @returns {Boolean}
         *  Whether this grid pane is currently visible.
         */
        this.isVisible = function () {
            return _.isObject(layerRange);
        };

        /**
         * Returns the address of the layer cell range.
         *
         * @returns {Object|Null}
         *  The address of the layer cell range, if this grid pane is visible;
         *  otherwise null.
         */
        this.getLayerRange = function () {
            return _.copy(layerRange, true);
        };

        /**
         * Returns the offset and size of the entire sheet rectangle covered by
         * this grid pane, including the ranges around the visible area.
         *
         * @returns {Object|Null}
         *  The offset and size of the sheet area covered by this grid pane, in
         *  pixels, if this grid pane is visible; otherwise null.
         */
        this.getLayerRectangle = function () {
            return _.copy(layerRectangle, true);
        };

        /**
         * Returns the effective offset and size of the visible sheet area in
         * this grid pane.
         *
         * @returns {Object}
         *  The offset and size of the sheet area visible in this grid pane, in
         *  pixels.
         */
        this.getVisibleRectangle = function () {
            return {
                left: colHeaderPane.getVisiblePosition().offset,
                top: rowHeaderPane.getVisiblePosition().offset,
                width: scrollNode[0].clientWidth,
                height: scrollNode[0].clientHeight
            };
        };

        /**
         * Returns the logical address of the top-left cell visible in this
         * grid pane. The cell is considered visible, it it's column and row
         * are at least half visible.
         *
         * @returns {Number[]}
         *  The logical address of the top-left cell in this grid pane.
         */
        this.getTopLeftAddress = function () {
            return [colHeaderPane.getFirstVisibleIndex(), rowHeaderPane.getFirstVisibleIndex()];
        };

        /**
         * Returns the DOM cell nodes matching the specified selector.
         *
         * @param {String} selector
         *  The jQuery selector used to filter the existing cell nodes.
         *
         * @returns {jQuery}
         *  All matching cell nodes, as jQuery collection.
         */
        this.findCellNodes = function (selector) {
            return gridRenderer.findCellNodes(selector);
        };

        /**
         * Returns the DOM cell node at the specified logical address.
         *
         * @param {Number[]|String} address
         *  The address of the cell, or the unique cell key of the address.
         *
         * @returns {jQuery}
         *  The DOM node of the cell, as jQuery object (empty, if the cell node
         *  does not exist).
         */
        this.getCellNode = function (address) {
            return gridRenderer.getCellNode(address);
        };

        /**
         * Returns all DOM cell nodes in the specified column.
         *
         * @param {Number} col
         *  The zero-based column index.
         *
         * @returns {jQuery}
         *  The DOM nodes of all cells in the specified column, as jQuery
         *  collection.
         */
        this.getCellNodesInCol = function (col) {
            return gridRenderer.getCellNodesInCol(col);
        };

        /**
         * Returns all DOM cell nodes in the specified row.
         *
         * @param {Number} row
         *  The zero-based row index.
         *
         * @returns {jQuery}
         *  The DOM nodes of all cells in the specified row, as jQuery
         *  collection.
         */
        this.getCellNodesInRow = function (row) {
            return gridRenderer.getCellNodesInRow(row);
        };

        /**
         * Returns the DOM drawing frame node that represents the passed unique
         * drawing model identifier.
         *
         * @param {String} uid
         *  The unique identifier of the drawing model.
         *
         * @returns {jQuery}
         *  The DOM node of the drawing frame, as jQuery object (empty, if the
         *  drawing frame does not exist).
         */
        this.getDrawingFrame = function (uid) {
            return drawingLayerNode.find('>[data-model-uid="' + uid + '"]');
        };

        /**
         * Changes the scroll position of this grid pane relative to the
         * current scroll position.
         *
         * @param {Number} leftDiff
         *  The difference for the horizontal scroll position, in pixels.
         *
         * @param {Number} topDiff
         *  The difference for the vertical scroll position, in pixels.
         *
         * @returns {GridPane}
         *  A reference to this instance.
         */
        this.scrollRelative = function (leftDiff, topDiff) {
            colHeaderPane.scrollRelative(leftDiff);
            rowHeaderPane.scrollRelative(topDiff);
            return this;
        };

        /**
         * Scrolls this grid pane to make the specified cell visible.
         *
         * @param {Number[]} address
         *  The logical cell position to be made visible.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.forceTop=false]
         *      If set to true, the cell will always be scrolled into the upper
         *      border of the grid pane, also if it is already visible.
         *
         * @returns {GridPane}
         *  A reference to this instance.
         */
        this.scrollToCell = function (address, options) {
            var position = view.getCellRectangle(address, { expandMerged: true }),
                forceTop = Utils.getBooleanOption(options, 'forceTop', false);
            colHeaderPane.scrollToPosition(position.left, position.width);
            rowHeaderPane.scrollToPosition(position.top, position.height, { forceLeading: forceTop });
            return this;
        };

        /**
         * Scrolls this grid pane to make the specified DOM node visible.
         *
         * @param {HTMLElement|jQuery} node
         *  The DOM node to be made visible. If this object is a jQuery
         *  collection, uses the first node it contains. Must be a descendant
         *  node of the layer root node.
         *
         * @returns {GridPane}
         *  A reference to this instance.
         */
        this.scrollToNode = function (node) {

            var // the position of the node in the layer root node
                position = Utils.getChildNodePositionInNode(layerRootNode, node);

            colHeaderPane.scrollToPosition(position.left + layerRectangle.left, position.width);
            rowHeaderPane.scrollToPosition(position.top + layerRectangle.top, position.height);
            return this;
        };

        /**
         * Scrolls this grid pane to make the specified drawing frame visible.
         *
         * @param {Number[]} position
         *  The logical document position of the drawing frame.
         *
         * @returns {GridPane}
         *  A reference to this instance.
         */
        this.scrollToDrawingFrame = function (position) {
            // TODO: scroll to embedded drawing frames
            return this.scrollToNode(drawingLayerNode[0].childNodes[position[0]]);
        };

        // in-place cell edit mode --------------------------------------------

        /**
         * Prepares the cell in-place edit mode for this grid pane.
         *
         * @returns {GridPane}
         *  A reference to this instance.
         */
        this.initializeCellEditMode = function () {
            focusTargetNode = rootNode.find('textarea').focus();
            clipboardNode.detach();
            return this;
        };

        /**
         * Cleans up after cell in-place edit mode has ended in this grid pane.
         *
         * @returns {GridPane}
         *  A reference to this instance.
         */
        this.cleanupCellEditMode = function () {
            rootNode.append(clipboardNode);
            focusTargetNode = clipboardNode;
            grabClipboardFocus();
            return this;
        };

        // initialization -----------------------------------------------------

        // marker for touch devices and browser types
        Utils.addDeviceMarkers(rootNode);

        // insert all layer nodes into the layer root
        layerRootNode.append(
            gridRenderer.getNode(),
            collaborativeLayerNode,
            selectionLayerNode,
            drawingLayerNode,
            highlightLayerNode,
            activeRangeLayerNode
        );

        // initialize class members
        app.onInit(function () {

            // the spreadsheet model and view
            model = app.getModel();
            view = app.getView();

            // the style sheet container of the document
            documentStyles = model.getDocumentStyles();

            // references to column and row header pane for this grid pane
            colHeaderPane = view.getColHeaderPane(panePos);
            rowHeaderPane = view.getRowHeaderPane(panePos);

            // the clipboard node that carries the browser focus for copy/paste events
            clipboardNode = view.createClipboardNode().attr('tabindex', 1).text('\xa0').appendTo(rootNode);
            clipboardFocusMethod = clipboardNode[0].focus;

            // by default, focus the clipboard node
            focusTargetNode = clipboardNode;

            // show drop-down menu for active cell, suppress double-clicks for cell drop-down button
            // (bug 33773: do not register event handlers at cellListButton directly, layer nodes will
            // be cleared with jQuery's empty() method which removes all event handlers from child nodes)
            selectionLayerNode.on({ click: cellListButtonClickHandler, dblclick: false }, '.cell-dropdown-button');

            // drop-down menu for auto-completion and list validation of active cell
            cellListMenu = new ListMenu({ anchor: function () { return getCellListButton().parent(); }, anchorPadding: 2 });
            cellListMenu.registerFocusableNodes(clipboardNode).registerFocusableNodes(getCellListButton);

            // on touch devices, disable the content-editable mode of the clipboard node,
            // to prevent showing the virtual keyboard all the time
            if (Modernizr.touch) { clipboardNode.removeAttr('contenteditable'); }

            // listen to events from the view
            view.on({
                'celledit:enter': cellEditEnterHandler,
                'celledit:leave': cellEditLeaveHandler
            });

            // update DOM after interval changes of the column and row header pane
            colHeaderPane.on({
                'change:interval': function (event, interval, refresh, options) { updateLayerInterval(interval, true, refresh, options); },
                'hide:interval': function () { updateLayerInterval(null, true); }
            });
            rowHeaderPane.on({
                'change:interval': function (event, interval, refresh, options) { updateLayerInterval(interval, false, refresh, options); },
                'hide:interval': function () { updateLayerInterval(null, false); }
            });

            // update selection after the column, row, or cell geometry has changed
            registerEventHandler(view, 'insert:columns delete:columns change:columns insert:rows delete:rows change:rows insert:merged delete:merged', changeCellLayoutHandler);

            // update drawing layer node (only, if this grid pane is visible)
            registerEventHandler(view, 'insert:drawing', insertDrawingHandler);
            registerEventHandler(view, 'delete:drawing', deleteDrawingHandler);
            registerEventHandler(view, 'change:drawing', changeDrawingHandler);

            // update layers according to changed sheet view attributes (only, if this grid pane is visible)
            registerEventHandler(view, 'change:sheet:viewattributes', changeSheetViewAttributesHandler);

            // listen to DOM scroll events
            scrollNode.on('scroll', scrollHandler);

            // listen to changed scroll position/size from header panes
            registerEventHandler(colHeaderPane, 'change:scrollsize', updateScrollAreaSize);
            registerEventHandler(colHeaderPane, 'change:scrollpos', updateScrollPosition);
            registerEventHandler(rowHeaderPane, 'change:scrollsize', updateScrollAreaSize);
            registerEventHandler(rowHeaderPane, 'change:scrollpos', updateScrollPosition);

            // activate this pane on focus change
            rootNode.on('focusin', function () {
                // do not change active pane while in cell edit mode (range selection mode in formulas)
                if (!view.isCellEditMode()) {
                    view.setSheetViewAttribute('activePane', panePos);
                }
            });

            // handle key events
            rootNode.on('keydown', keyDownHandler);

            // handle dropped images
            rootNode.on('drop', dropHandler);

            // bug 31479: suppress double-clicks for drawings
            drawingLayerNode.on('dblclick', false);

            // cut, copy & paste events in clipboard node
            clipboardNode.on({
                cut: cutHandler,
                copy: copyHandler,
                paste: pasteHandler
            });

            // handle DOM focus requests at the clipboard node (select clipboard contents)
            clipboardNode[0].focus = grabClipboardFocus;

            // adjust minimum width of the cell pop-up menu
            cellListMenu.on({
                'popup:show': function () { self.grabFocus(); },
                'popup:beforelayout': cellListBeforeLayoutHandler
            });

            // handle click events in the cell pop-up menu
            cellListMenu.getNode().on('click tap', Forms.BUTTON_SELECTOR, cellListClickHandler);

            // hide pop-up menu when starting to select (including click on current cell with open menu)
            self.on('select:start', function () { cellListMenu.hide(); });

            // update grid pane when document edit mode changes
            model.on('change:editmode', editModeHandler);
        });

        // destroy all class members on destruction
        this.registerDestructor(function () {

            gridRenderer.destroy();
            cellListMenu.destroy();
            clipboardNode[0].focus = clipboardFocusMethod;

            app.destroyImageNodes(drawingLayerNode);
            app.destroyImageNodes(clipboardNode);

            model = documentStyles = null;
            view = colHeaderPane = rowHeaderPane = gridRenderer = null;
            rootNode = scrollNode = scrollSizeNode = layerRootNode = null;
            collaborativeLayerNode = selectionLayerNode = drawingLayerNode = highlightLayerNode = activeRangeLayerNode = null;
            clipboardNode = clipboardFocusMethod = cellListMenu = null;
        });

    } // class GridPane

    // exports ================================================================

    // derive this class from class TriggerObject
    return TriggerObject.extend({ constructor: GridPane });

});
